<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2015 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2015. All rights reserved".
     ********************************************************************************/

    /**
     * Helper class for working with missions
     */
    class MissionsUtil
    {
        /**
         * Renders string content for the mission description or the latest mission comment
         * if it exists.
         * @param Mission $mission
         * @return string
         */
        public static function renderDescriptionAndLatestForDisplayView(Mission $mission)
        {
            $url      = Yii::app()->createUrl('/missions/default/details', array('id' => $mission->id));
            $content  = $mission->description;
            $details  = ZurmoHtml::tag('span', array('class' => 'list-item-details'),
                                       DateTimeUtil::getTimeSinceDisplayContent($mission->latestDateTime));
            $link     = ZurmoHtml::link($content, $url);
            return $link . $details;
        }

        /**
         * @param Mission $mission
         * @param User $user
         */
        public static function markUserHasReadLatest(Mission $mission, User $user)
        {
            $mashableUtilRules  = MashableUtil::createMashableInboxRulesByModel('Mission');
            $hasReadLatest      = $mashableUtilRules->markUserAsHavingReadLatestModel($mission, $user);
            return $hasReadLatest;
        }

        /**
         * @param Mission $mission
         * @param User $user
         */
        public static function markUserHasUnreadLatest(Mission $mission, User $user)
        {
            $mashableUtilRules  = MashableUtil::createMashableInboxRulesByModel('Mission');
            $hasReadLatest      = $mashableUtilRules->markUserAsHavingUnreadLatestModel($mission, $user);
            return $hasReadLatest;
        }

        /**
         * @param Mission $mission
         * @param User $user
         * @return bool
         */
        public static function hasUserReadMissionLatest(Mission $mission, User $user)
        {
            $mashableUtilRules  = MashableUtil::createMashableInboxRulesByModel('Mission');
            $hasReadLatest      = $mashableUtilRules->hasUserReadLatest($mission, $user);
            return $hasReadLatest;
        }

        /**
         * @param Mission $mission
         */
        public static function markAllUserHasReadLatestExceptOwnerAndTakenBy(Mission $mission)
        {
            $users = User::getAll();
            foreach ($users as $user)
            {
                if ($user->getClassId('Item') !== $mission->owner->getClassId('Item') &&
                           $user->getClassId('Item') !== $mission->takenByUser->getClassId('Item') )
                {
                    static::markUserHasReadLatest($mission, $user);
                }
            }
        }

        /**
         * @param Mission $mission
         */
        public static function markAllUserHasUnreadLatest(Mission $mission)
        {
            $users = static::resolvePeopleToSendNotificationToOnNewMission($mission);
            foreach ($users as $user)
            {
                static::markUserHasUnreadLatest($mission, $user);
            }
        }

        /**
         * @param int $type
         * @return string
         * @throws NotSupportedException
         */
        public static function makeActiveActionElementType($type)
        {
            assert('$type == null || is_int($type)');
            if ($type == null)
            {
                $type = MissionsListConfigurationForm::LIST_TYPE_AVAILABLE;
            }
            if ($type == MissionsListConfigurationForm::LIST_TYPE_CREATED)
            {
                return 'MissionsCreatedLink';
            }
            elseif ($type == MissionsListConfigurationForm::LIST_TYPE_AVAILABLE)
            {
                return 'MissionsAvailableLink';
            }
            elseif ($type == MissionsListConfigurationForm::LIST_TYPE_MINE_TAKEN_BUT_NOT_ACCEPTED)
            {
                return 'MissionsMineTakenButNotAcceptedLink';
            }
            else
            {
                throw new NotSupportedException();
            }
        }

        /**
         * @param Mission $mission
         * @param int $type
         * @param int $pageSize
         * @return RedBeanModelDataProvider'
         */
        public static function makeDataProviderByType(Mission $mission, $type, $pageSize)
        {
            if ($type == null)
            {
                $type = MissionsListConfigurationForm::LIST_TYPE_AVAILABLE;
            }
            $searchAttributes = array();
            $metadataAdapter  = new MissionsSearchDataProviderMetadataAdapter(
                $mission,
                Yii::app()->user->userModel->id,
                $searchAttributes,
                $type
            );
            $dataProvider = RedBeanModelDataProviderUtil::makeDataProvider(
                $metadataAdapter->getAdaptedMetadata(),
                'Mission',
                'RedBeanModelDataProvider',
                'latestDateTime',
                true,
                $pageSize
            );
            return $dataProvider;
        }

        /**
         * Create and submit a notification when a status changes.
         * @param User $userToReceiveMessage
         * @param integer $missionId
         * @param string $messageContent
         */
        public static function makeAndSubmitStatusChangeNotificationMessage(User $userToReceiveMessage, $missionId, $messageContent)
        {
            assert('$userToReceiveMessage->id > 0');
            assert('is_int($missionId)');
            assert('is_string($messageContent)');
            $message                      = new NotificationMessage();
            $message->htmlContent         = $messageContent;
            $url                          = Yii::app()->createAbsoluteUrl('missions/default/details/',
                                                                array('id' => $missionId));
            $message->htmlContent        .= '-' . ZurmoHtml::link(Zurmo::t('Core', 'Click Here'), $url,
                                                                    array('target' => '_blank'));
            $message->textContent         = $messageContent;
            $message->textContent        .= Zurmo::t('MissionsModule', 'Check mission details in this link: ') . ShortUrlUtil::createShortUrl($url);
            $rules                        = new MissionStatusChangeNotificationRules();
            $rules->addUser($userToReceiveMessage);
            $rules->setAllowDuplicates(true);
            NotificationsUtil::submit($message, $rules);
        }

        /**
         * Create at most one notification for a user when there are new unread comments.
         * @param User $userToReceiveMessage
         * @param integer $missionId
         * @param string $messageContent
         */
        public static function makeAndSubmitNewCommentNotificationMessage(User $userToReceiveMessage)
        {
            assert('$userToReceiveMessage->id > 0');
            $message                      = new NotificationMessage();
            $url                          = Yii::app()->createAbsoluteUrl('missions/default/list/');
            $message->htmlContent         = ZurmoHtml::link(Zurmo::t('Core', 'Click Here'), $url,
                                                            array('target' => '_blank'));
            $rules                        = new MissionUnreadCommentNotificationRules();
            $rules->addUser($userToReceiveMessage);
            NotificationsUtil::submit($message, $rules);
        }

        /**
         * @param Mission $mission
         */
        public static function makeAndSubmitNewMissionNotificationMessage(Mission $mission)
        {
            $rules = new NewMissionNotificationRules();
            $rules->setModel($mission);
            $notificationMessage = new NotificationMessage();
            $notificationMessage->textContent = static::getTextContentForNotificationMessage($mission);
            $notificationMessage->htmlContent = static::getHtmlContentForNotificationMessage($mission);
            $peopleToSendNotification = static::resolvePeopleToSendNotificationToOnNewMission($mission);
            if (count($peopleToSendNotification) > 0)
            {
                foreach ($peopleToSendNotification as $person)
                {
                    $rules->addUser($person);
                }
                NotificationsUtil::submit($notificationMessage, $rules);
            }
        }

        /**
         * @param Mission $mission
         * @return string
         */
        protected static function getTextContentForNotificationMessage(Mission $mission)
        {
            $url           = CommentsUtil::getUrlToEmail($mission);
            $shortUrl      = ShortUrlUtil::createShortUrl($url);
            $textContent   = Zurmo::t('MissionsModule', 'Hello, {lineBreak}There is a new mission. ' .
                'Be the first one to start it and get this great reward: {reward}.' .
                '{lineBreak}{lineBreak} {url}',
                array('{lineBreak}' => "\n",
                    '{reward}'    => $mission->reward,
                    '{url}'       => $shortUrl
                ));
            return $textContent;
        }

        /**
         * @param Mission $mission
         * @return string
         */
        protected static function getHtmlContentForNotificationMessage(Mission $mission)
        {
            $url         = CommentsUtil::getUrlToEmail($mission);
            $htmlContent = Zurmo::t('MissionsModule', 'Hello, {lineBreak}There is a new {url}. ' .
                'Be the first one to start it and get this great reward: {reward}.',
                array('{lineBreak}'      => "<br/>",
                    '{strongStartTag}' => '<strong>',
                    '{strongEndTag}'   => '</strong>',
                    '{reward}'         => $mission->reward,
                    '{url}'            => ZurmoHtml::link($mission->getModelLabelByTypeAndLanguage(
                                                'SingularLowerCase'), $url, array('target' => '_blank'))
                ));
            return $htmlContent;
        }

        /**
         * @param Mission $mission
         * @return string
         */
        public static function getEmailSubject(Mission $mission)
        {
            return Zurmo::t('MissionsModule', 'New mission');
        }

        /**
         * @param Mission $mission
         * @return array
         */
        public static function resolvePeopleToSendNotificationToOnNewMission(Mission $mission)
        {
            $users = User::getAll();
            $people = array();
            foreach ($users as $user)
            {
                if ($user->getClassId('Item') != $mission->owner->getClassId('Item') &&
                    RightsUtil::canUserAccessModule('MissionsModule', $user)  && $user->isActive)
                {
                    $people[] = $user;
                }
            }
            return $people;
        }

        /**
         * Given a Mission and the User that created the new comment
         * return the people on the mission to send new notification to
         * @param Mission $mission
         * @param User $user
         * @return Array $peopleToSendNotification
         */
        public static function resolvePeopleToSendNotificationToOnNewComment(Mission $mission, User $user)
        {
            $usersToSendNotification = User::getAll();
            $peopleToSendNotification = array();
            foreach ($usersToSendNotification as $userToSendNotification)
            {
                if ($userToSendNotification->getClassId('Item') != $user->getClassId('Item') &&
                    RightsUtil::canUserAccessModule('MissionsModule', $userToSendNotification))
                {
                    if ($mission->takenByUser->id > 0)
                    {
                        if ($userToSendNotification->getClassId('Item') == $mission->owner->getClassId('Item') ||
                            $userToSendNotification->getClassId('Item') == $mission->takenByUser->getClassId('Item') )
                        {
                            $peopleToSendNotification[] = $userToSendNotification;
                        }
                    }
                    else
                    {
                        $peopleToSendNotification[] = $userToSendNotification;
                    }
                }
            }
            return $peopleToSendNotification;
        }
    }
?>