<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2013 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2013. All rights reserved".
     ********************************************************************************/

    require_once 'BaseUpgraderComponent.php';

    class UpgraderComponent extends BaseUpgraderComponent
    {
        /**
         * Tasks that should be executed before updating schema.
         * @param string $currentZurmoVersion
         *
         * @code
            <?php
                $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
                if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '0.6.80'))
                {
                    $metadata = AccountsModule::getMetadata();
                    if(!isset($metadata['global']['newElement']))
                    {
                        $metadata['global']['newElement'] = 'Some Content';
                        AccountsModule::setMetadata($metadata);
                    }
                    GeneralCache::forgetAll();
                }
            ?>
         * @endcode
         */
        public function processBeforeUpdateSchema()
        {
            parent::processBeforeUpdateSchema();
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '2.2.2'))
            {
                $this->prependAnyMixedAttributesToGlobalMetadata();
                $this->updateLengthMinRuleToOneForAccountAndContactAndMeetingAndTask();
            }
        }

        public function processAfterUpdateSchema()
        {
            parent::processAfterUpdateSchema();
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '2.2.0'))
            {
                $this->fixFilePolymorphicRelationForEmailTemplatesAndAutorespondersAndCampaigns();
                $this->clearMarketingListDetailsPagePortlets();
                $this->fixAutoresponderIntervals();
                $this->fixWorkflowIntervals();
                $this->moveSystemArchivedAndArchivedUnmatchedToEmailOwnerBox();
            }
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '2.2.3'))
            {
                $this->createOrUpdateBaseControlUser();
            }
        }

        protected function prependAnyMixedAttributesToGlobalMetadata()
        {
            $prependAnyMixedAttributesToGlobalMetadata = array(
                'AccountsMyListConfigView',
                'ContactsMyListConfigView',
                'TasksMyListConfigView',
                'OpportunitiesMyListConfigView',
                'LeadsMyListConfigView',
            );
            foreach ($prependAnyMixedAttributesToGlobalMetadata as $className)
            {
                $this->prependAnyMixedAttributesToGlobalMetadataByClassName($className);
            }
        }

        protected function prependAnyMixedAttributesToGlobalMetadataByClassName($className)
        {
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                $metadata = $className::getMetadata();
                if (!isset($metadata['global']['nonPlaceableAttributeNames']))
                {
                    $metadata['global']['nonPlaceableAttributeNames'] = array();
                }
                if (empty($metadata['global']['nonPlaceableAttributeNames']) ||
                    ($metadata['global']['nonPlaceableAttributeNames'][0] !== 'anyMixedAttributes'))
                {
                    array_unshift($metadata['global']['nonPlaceableAttributeNames'], 'anyMixedAttributes');
                }
                $className::setMetadata($metadata);
            }
        }

        protected function fixFilePolymorphicRelationForEmailTemplatesAndAutorespondersAndCampaigns()
        {
            $modelClassNames = array('Autoresponder', 'Campaign', 'EmailTemplate');
            foreach ($modelClassNames as $modelClassName)
            {
                $this->fixFilePolymorphicRelationForModelClassName($modelClassName);
            }
        }

        protected function fixFilePolymorphicRelationForModelClassName($modelClassName)
        {
            $relatedTableName   = RedBeanModel::getTableName($modelClassName);
            $columnName         = $relatedTableName . '_id';
            $fileModelTableName = RedBeanModel::getTableName('FileModel');
            $quote              = DatabaseCompatibilityUtil::getQuote();
            $updateQuery        = $this->getUpdateQueryForFileModelPolymorphicRelationFix($fileModelTableName,
                                                                                            $quote,
                                                                                            $columnName,
                                                                                            $relatedTableName);
            $alterQuery         = $this->getAlterQueryForFileModelPolymorphicRelationFix($fileModelTableName,
                                                                                            $quote,
                                                                                            $columnName);
            $query              = $updateQuery . $alterQuery;
            R::exec($query);
        }

        protected function getUpdateQueryForFileModelPolymorphicRelationFix($tableName,
                                                                                $quote,
                                                                                $columnName,
                                                                                $relatedTableName)
        {
            $query  = "update ${quote}${tableName}${quote} set ${quote}relatedmodel_id${quote} =";
            $query  .= " ${quote}${columnName}${quote},${quote}relatedmodel_type${quote} = ";
            $query  .= "'${relatedTableName}' where ${quote}${columnName}${quote} IS NOT NULL AND ";
            $query  .= "${quote}${columnName}${quote} != ''; ";
            return $query;
        }

        protected function getAlterQueryForFileModelPolymorphicRelationFix($tableName, $quote, $columnName)
        {
            $query  = "alter table ${quote}${tableName}${quote} drop ${quote}${columnName}${quote}";
            return $query;
        }

        protected function clearMarketingListDetailsPagePortlets()
        {
            $portletTableName   = RedBeanModel::getTableName('Portlet');
            $quote              = DatabaseCompatibilityUtil::getQuote();
            $layoutId           = 'MarketingListDetailsAndRelationsViewLeftBottomView';
            $query              = $this->getDeleteQueryForMarkeringListDetailsPagePortlets($portletTableName,
                                                                                            $quote,
                                                                                            $layoutId);
            R::exec($query);
        }

        protected function getDeleteQueryForMarkeringListDetailsPagePortlets($tableName, $quote, $layoutId)
        {
            $query  = "delete from ${quote}${tableName}${quote} where ";
            $query  .= "${quote}layoutid${quote} ='${layoutId}'";
            return $query;
        }

        protected function fixAutoresponderIntervals()
        {
            $tableName      = RedBeanModel::getTableName('Autoresponder');
            $quote          = DatabaseCompatibilityUtil::getQuote();
            $columnName     = 'secondsfromoperation';
            $records        = $this->getAutoresponderRecordsRequiringIntervalFix($tableName, $quote, $columnName);
            $updateQuery    = null;
            if ($records)
            {
                $updateQuery = $this->getUpdateQueryForAutoresponderRecordsOperationIntervalUpdate($records,
                                                                                                    $quote,
                                                                                                    $tableName);
            }
            $dropQuery      = $this->getAlterQueryForAutoresponder($tableName, $quote, $columnName);
            $query = $updateQuery . $dropQuery;
            R::exec($query);
        }

        protected function getAutoresponderRecordsRequiringIntervalFix($tableName, $quote, $columnName)
        {
            $query      = $this->getSelectQueryForAutoresponder($tableName, $quote, $columnName);
            $records    = R::$adapter->get($query);
            if (empty($records))
            {
                return false;
            }
            return $records;
        }

        protected function getSelectQueryForAutoresponder($tableName, $quote, $columnName)
        {
            $query    = "select ${quote}id${quote}, ${quote}${columnName}${quote} from ";
            $query    .= "${quote}${tableName}${quote} where ${quote}${columnName}${quote} is not null;";
            return $query;
        }

        protected function getAlterQueryForAutoresponder($tableName, $quote, $columnName)
        {
            $query  = "alter table ${quote}${tableName}${quote} drop ${quote}${columnName}${quote};";
            return $query;
        }

        protected function getUpdateQueryForAutoresponderRecordsOperationIntervalUpdate($records, $quote, $tableName)
        {
            $updateQuery    = null;
            foreach ($records as $record)
            {
                $query  = $this->getUpdateQueryForAutoresponderRecordOperationIntervalUpdate($record, $quote, $tableName);
                if ($query)
                {
                    $updateQuery    .= $query . PHP_EOL;
                }
            }
            return $updateQuery;
        }

        protected function getUpdateQueryForAutoresponderRecordOperationIntervalUpdate($record, $quote, $tableName)
        {
            $id                     = null;
            $secondsfromoperation   = null;
            $unit                   = null;
            $type                   = null;
            extract($record);
            $this->convertUnsignedSecondsToUnitAndTypeFormat($secondsfromoperation, $unit, $type);
            $updateQuery     = "update ${quote}${tableName}${quote} set ";
            $updateQuery    .= "${quote}fromoperationdurationinterval${quote} = ${unit}, ";
            $updateQuery    .= "${quote}fromoperationdurationtype${quote} = '${type}' where ";
            $updateQuery    .= "${quote}id${quote} = ${id};";
            return $updateQuery;
        }

        protected function fixWorkflowIntervals()
        {
            $this->fixSavedWorkflowIntervals();
        }

        protected function fixSavedWorkflowIntervals()
        {
            foreach (SavedWorkflow::getAll() as $workflow)
            {
                $serializedData             = $workflow->serializedData;
                $unserializedData           = unserialize($serializedData);
                $this->fixEmailMessageSendAfterDurationSecondsForSavedWorkflow($unserializedData);
                $this->fixActionsDynamicDateTimeForSavedWorkflow($unserializedData);
                $this->fixTimeTriggerForSavedWorkflow($unserializedData);
                $workflow->serializedData   = serialize($unserializedData);
                if (!$workflow->save())
                {
                    throw new FailedToSaveModelException("Unable to save workflow id #" . $workflow->id);
                }
            }
        }

        protected function fixEmailMessageSendAfterDurationSecondsForSavedWorkflow(& $unserializedData)
        {
            if (!isset($unserializedData['EmailMessages']) || !is_array($unserializedData['EmailMessages']))
            {
                return;
            }

            foreach ($unserializedData['EmailMessages'] as & $emailMessage)
            {
                $sendAfterDurationSeconds   = $emailMessage['sendAfterDurationSeconds'];
                unset($emailMessage['sendAfterDurationSeconds']);
                $sendAfterDurationInterval  = null;
                $sendAfterDurationType      = null;
                $this->convertUnsignedSecondsToUnitAndTypeFormat($sendAfterDurationSeconds,
                                                            $sendAfterDurationInterval,
                                                            $sendAfterDurationType);
                $emailMessage['sendAfterDurationInterval'] = $sendAfterDurationInterval;
                $emailMessage['sendAfterDurationType'] = $sendAfterDurationType;
            }
        }

        protected function fixTimeTriggerForSavedWorkflow(& $unserializedData)
        {
            if (!isset($unserializedData['TimeTrigger']) || !is_array($unserializedData['TimeTrigger']))
            {
                return;
            }
            $durationSeconds        = $unserializedData['TimeTrigger']['durationSeconds'];
            unset($unserializedData['TimeTrigger']['durationSeconds']);
            $durationInterval       = null;
            $durationType           = null;
            $durationSign           = null;
            $this->convertSignedSecondsToUnitAndTypeFormat($durationSeconds,
                                                            $durationInterval,
                                                            $durationType,
                                                            $durationSign);
            $unserializedData['TimeTrigger']['durationInterval']    = $durationInterval;
            $unserializedData['TimeTrigger']['durationType']        = $durationType;
            $unserializedData['TimeTrigger']['durationSign']        = $durationSign;
        }

        protected function fixActionsDynamicDateTimeForSavedWorkflow(& $unserializedData)
        {
            if (!isset($unserializedData['Actions']) || !is_array($unserializedData['Actions']))
            {
                return;
            }
            foreach ($unserializedData['Actions'] as & $action)
            {
                foreach ($action['ActionAttributes'] as & $attribute)
                {
                    $this->fixActionAttributesDateTimeForSavedWorkflow($attribute);
                }
            }
        }

        protected function fixActionAttributesDateTimeForSavedWorkflow(& $attribute)
        {
            if (strpos($attribute['type'], 'DynamicFromExistingDate') === 0 || // also covers DynamicFromExistingDateTime
                strpos($attribute['type'], 'DynamicFromTriggeredDate') === 0) // also covers DynamicFromTriggeredDateTime
            {
                $seconds = $attribute['value'];
                unset($attribute['value']);
                $durationInterval   = null;
                $durationType       = null;
                $durationSign       = null;
                $this->convertSignedSecondsToUnitAndTypeFormat($seconds, $durationInterval, $durationType, $durationSign);
                $attribute['durationInterval']  = $durationInterval;
                $attribute['durationType']      = $durationType;
                $attribute['durationSign']      = $durationSign;
            }
        }

        protected function convertSignedSecondsToUnitAndTypeFormat($seconds, & $unit, & $type, & $sign)
        {
            $sign       = TimeDurationUtil::DURATION_SIGN_POSITIVE;
            if ($seconds < 0)
            {
                $sign       = TimeDurationUtil::DURATION_SIGN_NEGATIVE;
                $seconds    *= -1;
            }
            $this->convertUnsignedSecondsToUnitAndTypeFormat($seconds, $unit, $type);
        }

        protected function convertUnsignedSecondsToUnitAndTypeFormat($seconds, & $unit, & $type)
        {
            $units = $this->getTypeToSecondsMapping();
            foreach ($units as $type => $secondsValue)
            {
                $unit = $seconds/$secondsValue;
                if (!is_float($unit))
                {
                    // if we were able to divide perfectly without any remainder then this is the type to go for.
                    // why do we have the above check instead of floor($seconds/$secondsValue)?
                    // so we can convert 45 Days to 45 Days and not to 1 Month
                    return;
                }
            }
            // For intervals less than 1 minute we default to 1 minute
            $unit = 1;
            $type = 'Minute';
        }

        protected function getTypeToSecondsMapping()
        {
            return array(
                TimeDurationUtil::DURATION_TYPE_YEAR     => 12*30*24*60*60,
                TimeDurationUtil::DURATION_TYPE_MONTH    => 30*24*60*60,
                TimeDurationUtil::DURATION_TYPE_WEEK     => 7*24*60*60,
                TimeDurationUtil::DURATION_TYPE_DAY      => 24*60*60,
                TimeDurationUtil::DURATION_TYPE_HOUR     => 60*60,
                TimeDurationUtil::DURATION_TYPE_MINUTE   => 60,
            );
        }

        protected function moveSystemArchivedAndArchivedUnmatchedToEmailOwnerBox()
        {
            $messages = $this->getAllSystemArchivedOrArchivedUnmatchedEmailMessages();
            if (count($messages) > 0)
            {
                $this->moveEmailMessagesToEmailOwnerBox($messages);
            }
        }

        protected function getAllSystemArchivedOrArchivedUnmatchedEmailMessages()
        {
            $box = EmailBox::resolveAndGetByName(EmailBox::NOTIFICATIONS_NAME);
            $messages = $this->getAllArchivedOrArchivedUnmatchedEmailMessagesForBoxId($box->id);
            return $messages;
        }

        protected function getAllArchivedOrArchivedUnmatchedEmailMessagesForBoxId($boxId)
        {
            $searchAttributeData = array();
            $searchAttributeData['clauses'] = array(
                1 => array(
                    'attributeName'        => 'folder',
                    'relatedAttributeName' => 'type',
                    'operatorType'         => 'equals',
                    'value'                => EmailFolder::TYPE_ARCHIVED,
                ),
                2 => array(
                    'attributeName'        => 'folder',
                    'relatedAttributeName' => 'type',
                    'operatorType'         => 'equals',
                    'value'                => EmailFolder::TYPE_ARCHIVED_UNMATCHED,
                ),
                3 => array(
                    'attributeName'             => 'folder',
                    'relatedModelData'          => array(
                        'attributeName'                 => 'emailBox',
                        'relatedAttributeName'          => 'id',
                        'operatorType'                  => 'equals',
                        'value'                         => $boxId
                    )
                ),
            );
            $searchAttributeData['structure'] = '((1 or 2) and 3)';
            $joinTablesAdapter = new RedBeanModelJoinTablesQueryAdapter('EmailMessage');
            $where = RedBeanModelDataProvider::makeWhere('EmailMessage', $searchAttributeData, $joinTablesAdapter);
            return EmailMessage::getSubset($joinTablesAdapter, null, null, $where, null);
        }

        protected function moveEmailMessagesToEmailOwnerBox($emailMessages)
        {
            $emailBoxes = array();
            foreach ($emailMessages as $emailMessage)
            {
                if ($this->shouldMoveEmailMessageToEmailOwnerBox($emailMessage))
                {
                    if(isset($emailBoxes[$emailMessage->owner->id]))
                    {
                        $box = $emailBoxes[$emailMessage->owner->id];
                    }
                    else
                    {
                        $emailBoxes[$emailMessage->owner->id] =  EmailBoxUtil::getDefaultEmailBoxByUser($emailMessage->owner);
                        $box = $emailBoxes[$emailMessage->owner->id];
                    }
                    $this->moveEmailMessageToEmailOwnerBox($emailMessage, $box);
                }
            }
        }

        protected function moveEmailMessageToEmailOwnerBox($emailMessage, EmailBox $box)
        {
            $folderType             = $emailMessage->folder->type;
            $folder                 = EmailFolder::getByBoxAndType($box, $folderType);
            $emailMessage->folder   = $folder;
            $quote      = DatabaseCompatibilityUtil::getQuote();
            $tableName  = RedBeanModel::getTableName('EmailMessage');
            $sql        = "update {$quote}{$tableName}${quote} set {$quote}folder_emailfolder_id{$quote}='{$folder->id}' where ";
            $sql        .= "{$quote}id{$quote}='{$emailMessage->id}'";
            R::exec($sql);
        }

        protected function shouldMoveEmailMessageToEmailOwnerBox($emailMessage)
        {
            return ($emailMessage->owner instanceof User);
        }

        protected function getModelClassNamesToPurgeGlobalMetadata()
        {
            // format: version => model class names to purge globalmetadata for.
            return array('2.2.0' => array(
                                        'AccountsForMixedModelsSearchListView',
                                        'Autoresponder',
                                        'AutoresponderEditAndDetailsView',
                                        'Campaign',
                                        'CampaignsForMarketingListRelatedListView',
                                        'ContactsForMixedModelsSearchListView',
                                        'EmailTemplate',
                                        'IframePortletConfigView',
                                        'ImportErrorsListView',
                                        'LeaderboardListView',
                                        'LeadsForMixedModelsSearchListView',
                                        'MarketingList',
                                        'MarketingListsForContactRelatedListView',
                                        'OpportunitiesForMixedModelsSearchListView',
                                        'UserConfigurationEditView',
                                        ));
        }

        protected function updateLengthMinRuleToOneForAccountAndContactAndMeetingAndTask()
        {
            $modelNamesWithAttributesToChangeMinFor = array(
                'Account'   => array('name'),
                'Contact'   => array('companyName'),
                'Meeting'   => array('name', 'location'),
                'Task'      => array('name'),
            );

            foreach ($modelNamesWithAttributesToChangeMinFor as $modelClassName => $attributeNames)
            {
                $this->updateLengthMinRuleToOneForModelClassAndAttributeNames($modelClassName, $attributeNames);
            }
        }

        protected function updateLengthMinRuleToOneForModelClassAndAttributeNames($modelClassName, $attributeNames)
        {
            if (GlobalMetadata::isClassMetadataSavedInDatabase($modelClassName))
            {
                $metadata = $modelClassName::getMetadata();
                foreach ($metadata[$modelClassName]['rules'] as $key => $rule)
                {
                    if (in_array($rule[0], $attributeNames) && $rule[1] == 'length')
                    {
                        $metadata[$modelClassName]['rules'][$key]['min'] = 1;
                    }
                }
                $modelClassName::setMetadata($metadata);
            }
        }

        protected function createOrUpdateBaseControlUser()
        {
            $systemUsername         = 'system';
            $baseControlUsername    = BaseControlUserConfigUtil::USERNAME;
            $this->renameUserIfExists($systemUsername, $baseControlUsername);
            BaseControlUserConfigUtil::getUserToRunAs();
        }

        protected function renameUserIfExists($oldUsername, $newUsername)
        {
            $quote      = DatabaseCompatibilityUtil::getQuote();
            $tableName  = RedBeanModel::getTableName('User');
            $sql        = "update {$quote}{$tableName}{$quote} set {$quote}username{$quote}='${newUsername}' where ";
            $sql        .= "{$quote}username{$quote}='{$oldUsername}'";
            R::exec($sql);
        }
    }
?>