<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2013 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2013. All rights reserved".
     ********************************************************************************/

    require_once 'BaseUpgraderComponent.php';

    class UpgraderComponent extends BaseUpgraderComponent
    {
        protected $_messageLogger = null;

        public function getMessageLogger() {
            return $this->_messageLogger;
        }

        public function setMessageLogger($messageLogger) {
            $this->_messageLogger = $messageLogger;
        }

        /**
         * Tasks to do before start processing config files.
         */
        public function processBeforeConfigFiles()
        {
            $this->messageLogger->addInfoMessage("Process before config files.");
            // Check if mcrypt is loaded or not
            if (!extension_loaded('mcrypt'))
            {
                $this->messageLogger->addErrorMessage("MCrypt is required PHP library. Please install mcrypt PHP extension, and start upgrade process again.");
                Yii::app()->end();
            }
            else
            {
                $this->messageLogger->addInfoMessage("MCrypt PHP library is installed.");
            }
        }

        /**
         * Process config files.
         * Problem here is that some config files might be modified, so there
         * can be some additional config options. We need to modify config
         * files(debug.php, debugTest.php, perInstance.php, perInstanceTest.php),
         * without loosing settings entered by user, so in most cases we will add
         * content at the end of those files where possible.
         * Should be used to files like debug.php, debugTest.php, perInstance.php, perInstanceTest.php,
         * which are not tracked by SCM tools, other files are updated automatically.
         *
         * @code
            <?php
                //Below is just sample code to add $myVariable to perInstance.php and perInstanceTest.php files.
                $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
                if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '0.6.80'))
                {
                    $perInstanceTestFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstanceTest.php';
                    if (is_file($perInstanceTestFile))
                    {
                        $perInstanceTestContent = file_get_contents($perInstanceTestFile);

                        $contentToAdd = "\$myVariable = 'aaa';\n";
                        $perInstanceTestContent = preg_replace('/\?\>/',
                                                               "\n" . $contentToAdd . "\n" . "?>",
                                                               $perInstanceTestContent);
                        file_put_contents($perInstanceTestFile, $perInstanceTestContent);
                    }
                    $perInstanceFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstance.php';
                    if (is_file($perInstanceFile))
                    {
                        $perInstanceContent = file_get_contents($perInstanceFile);

                        $contentToAdd = "\$myVariable = 'aaa';\n";
                        $perInstanceContent = preg_replace('/\?\>/',
                                                           "\n" . $contentToAdd . "\n" . "?>",
                                                           $perInstanceContent);
                        file_put_contents($perInstanceFile, $perInstanceContent);
                    }
                }
            ?>
         * @endcode
         */
        public function processConfigFiles($pathToConfigurationFolder)
        {
            // set ZURMO_PASSWORD_SALT
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '1.6.00'))
            {
                $perInstanceFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstance.php';
                if (is_file($perInstanceFile))
                {
                    $perInstanceContent = file_get_contents($perInstanceFile);

                    $zurmoPasswordSalt = substr(md5(microtime() * mt_rand()), 0, 15);
                    $contentToAdd = "    define('ZURMO_PASSWORD_SALT', '$zurmoPasswordSalt');\n";

                    $perInstanceContent = preg_replace('/\?\>/',
                        "\n" . $contentToAdd . "\n" . "?>",
                        $perInstanceContent);
                    file_put_contents($perInstanceFile, $perInstanceContent);
                }

                $perInstanceTestFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstanceTest.php';
                if (is_file($perInstanceTestFile))
                {
                    $perInstanceTestContent = file_get_contents($perInstanceTestFile);

                    $zurmoPasswordSalt = substr(md5(microtime() * mt_rand()), 0, 15);
                    $contentToAdd = "    define('ZURMO_PASSWORD_SALT', $zurmoPasswordSalt);\n";

                    $perInstanceTestContent = preg_replace('/\?\>/',
                        "\n" . $contentToAdd . "\n" . "?>",
                        $perInstanceTestContent);

                    file_put_contents($perInstanceTestFile, $perInstanceTestContent);
                }
            }
        }

        /**
         * Tasks after config files are processed.
         */
        public function processAfterConfigFiles()
        {

        }

        /**
         * Tasks to do before start processing files.
         */
        public function processBeforeFiles()
        {
        }

        /**
         * Process files
         * @param string $source
         * @param string $destination
         * @param array $configuration
         */
        public function processFiles($source, $destination, $configuration)
        {
            // Remove files that are marked for removal in manifest.php.
            if (isset($configuration['removedFiles']) && !empty($configuration['removedFiles']))
            {
                foreach ($configuration['removedFiles'] as $fileOrFolderToRemove)
                {
                    // Replace Lunux directory separators(from upgrade manifest file)
                    // with one used by system(DIRECTORY_SEPARATOR)
                    $fileOrFolderToRemove = str_replace(array('/', '\\'), DIRECTORY_SEPARATOR, $fileOrFolderToRemove);
                    $fileOrFolderToRemove = $destination . DIRECTORY_SEPARATOR . trim($fileOrFolderToRemove, DIRECTORY_SEPARATOR);
                    if (is_dir($fileOrFolderToRemove))
                    {
                        FileUtil::deleteDirectoryRecursive($fileOrFolderToRemove);
                    }
                    elseif (is_file($fileOrFolderToRemove))
                    {
                        unlink($fileOrFolderToRemove);
                    }
                }
            }
            // Copy new or modified files.
            FileUtil::copyRecursive($source, $destination);
        }

        /**
         * Tasks after files are processed.
         */
        public function processAfterFiles()
        {
        }

        /**
         * Tasks that should be executed before updating schema.
         * @param string $currentZurmoVersion
         *
         * @code
            <?php
                $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
                if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '0.6.80'))
                {
                    $metadata = AccountsModule::getMetadata();
                    if(!isset($metadata['global']['newElement']))
                    {
                        $metadata['global']['newElement'] = 'Some Content';
                        AccountsModule::setMetadata($metadata);
                    }
                    GeneralCache::forgetAll();
                }
            ?>
         * @endcode
         */
        public function processBeforeUpdateSchema()
        {
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '2.0.22'))
            {
                // Clear portlet table
                R::exec('delete from portlet');

                if (GlobalMetadata::isClassMetadataSavedInDatabase('AccountEditAndDetailsView'))
                {
                    $metadata = AccountEditAndDetailsView::getMetadata();
                    $search = array('type' => 'AccountDeleteLink');
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        unset($metadata['global']['toolbar']['elements'][$result['elementKey']]);
                        $metadata['global']['toolbar']['elements'] = array_values($metadata['global']['toolbar']['elements']);
                    }
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'CopyLink', 'renderType' => 'Details');
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'AccountDeleteLink', 'renderType' => 'Details');
                    AccountEditAndDetailsView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('Account'))
                {
                    $metadata = Account::getMetadata();
                    $metadata['Account']['relations']['products'] = array(RedBeanModel::HAS_MANY,  'Product');
                    Account::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('ContactEditAndDetailsView'))
                {
                    $metadata = ContactEditAndDetailsView::getMetadata();
                    $search = array('type' => 'ContactDeleteLink');
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        unset($metadata['global']['toolbar']['elements'][$result['elementKey']]);
                        $metadata['global']['toolbar']['elements'] = array_values($metadata['global']['toolbar']['elements']);
                    }
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'CopyLink', 'renderType' => 'Details');
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'ContactDeleteLink', 'renderType' => 'Details');
                    $metadata['global']['nonPlaceableAttributeNames'][] = 'googleWebTrackingId';
                    ContactEditAndDetailsView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('DashboardEditView'))
                {
                    R::exec('delete from globalmetadata where classname="DashboardEditView"');
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('Contact'))
                {
                    $metadata = Contact::getMetadata();

                    foreach ($metadata['Contact']['rules'] as $key => $rule)
                    {
                        if ($rule[0] == 'website' && $rule[1] == 'url')
                        {
                            $metadata['Contact']['rules'][$key]['defaultScheme'] = 'http';
                        }
                    }

                    $metadata['Contact']['members'][] = 'googleWebTrackingId';
                    $metadata['Contact']['relations']['products'] = array(RedBeanModel::HAS_MANY, 'Product');
                    $metadata['Contact']['rules'][] = array('googleWebTrackingId',    'type',    'type' => 'string');
                    Contact::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('EmailMessagesModule'))
                {
                    $metadata = EmailMessagesModule::getMetadata();
                    $metadata['global']['configureSubMenuItems'] = array(
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('EmailMessagesModule', 'Email SMTP Configuration')",
                            'descriptionLabel' => "eval:Zurmo::t('EmailMessagesModule', 'Manage Email SMTP Configuration')",
                            'route'            => '/emailMessages/default/configurationEditOutbound',
                            'right'            => EmailMessagesModule::RIGHT_ACCESS_CONFIGURATION,
                        ),
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('EmailMessagesModule', 'Email Archiving Configuration')",
                            'descriptionLabel' => "eval:Zurmo::t('EmailMessagesModule', 'Manage Email Archiving Configuration')",
                            'route'            => '/emailMessages/default/configurationEditImap',
                            'routeParams'      => array('type' => 1),
                            'right'            => EmailMessagesModule::RIGHT_ACCESS_CONFIGURATION,
                        ),
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('EmailMessagesModule', 'Bounce Configuration')",
                            'descriptionLabel' => "eval:Zurmo::t('EmailMessagesModule', 'Manage Bounce Configuration')",
                            'route'            => '/emailMessages/default/configurationEditImap',
                            'routeParams'      => array('type' => 2),
                            'right'            => EmailMessagesModule::RIGHT_ACCESS_CONFIGURATION,
                        ),
                    );
                    EmailMessagesModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('GroupEditAndDetailsView'))
                {
                    R::exec('delete from globalmetadata where classname="GroupEditAndDetailsView"');
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('GroupsModule'))
                {
                    $metadata = GroupsModule::getMetadata();
                    $metadata['global']['adminTabMenuItems'] = array(
                        array(
                            'label' => "eval:Zurmo::t('ZurmoModule', 'Groups')",
                            'url'   => array('/zurmo/group'),
                            'right' => GroupsModule::RIGHT_ACCESS_GROUPS,
                            'items' => array(
                                array(
                                    'label' => "eval:Zurmo::t('ZurmoModule', 'Create Group')",
                                    'url'   => array('/zurmo/group/create'),
                                    'right' => GroupsModule::RIGHT_CREATE_GROUPS
                                ),
                                array(
                                    'label' => "eval:Zurmo::t('ZurmoModule', 'Groups')",
                                    'url'   => array('/zurmo/group'),
                                    'right' => GroupsModule::RIGHT_ACCESS_GROUPS
                                ),
                            ),
                        ),
                    );
                    $metadata['global']['configureMenuItems'] = array(
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'Groups')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Manage Groups')",
                            'route'            => '/zurmo/group',
                            'right'            => GroupsModule::RIGHT_ACCESS_GROUPS,
                        ),
                    );
                    GroupsModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('ImportModule'))
                {
                    $metadata = ImportModule::getMetadata();
                    $metadata['global']['configureMenuItems'] = array(
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ImportModule', 'Import')",
                            'descriptionLabel' => "eval:Zurmo::t('ImportModule', 'Import data into Zurmo', LabelUtil::getTranslationParamsForAllModules())",
                            'route'            => '/import/default',
                            'right'            => ImportModule::RIGHT_ACCESS_IMPORT,
                        ),
                    );
                    ImportModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('LeadEditAndDetailsView'))
                {
                    $metadata = LeadEditAndDetailsView::getMetadata();
                    $search = array('type' => 'LeadDeleteLink');
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        unset($metadata['global']['toolbar']['elements'][$result['elementKey']]);
                        $metadata['global']['toolbar']['elements'] = array_values($metadata['global']['toolbar']['elements']);
                    }
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'CopyLink', 'renderType' => 'Details');
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'LeadDeleteLink', 'renderType' => 'Details');
                    LeadEditAndDetailsView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('MapsModule'))
                {
                    $metadata = MapsModule::getMetadata();
                    $metadata['global']['configureSubMenuItems'] = array(
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_PLUGINS,
                            'titleLabel'       => "eval:Zurmo::t('MapsModule', 'Maps')",
                            'descriptionLabel' => "eval:Zurmo::t('MapsModule', 'Manage Map Configuration')",
                            'route'            => '/maps/default/configurationView',
                            'right'            => MapsModule::RIGHT_ACCESS_MAPS_ADMINISTRATION,
                        ),
                    );
                    if (isset($metadata['global']['configureMenuItems']))
                    {
                        unset($metadata['global']['configureMenuItems']);
                    }
                    MapsModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('OpenTasksForAccountRelatedListView'))
                {
                    $metadata = OpenTasksForAccountRelatedListView::getMetadata();
                    $metadata['global']['rowMenu']['elements'] = array(
                        array('type' => 'EditLink'),
                        array('type' => 'CopyLink'),
                        array('type' => 'RelatedDeleteLink'),
                    );
                    OpenTasksForAccountRelatedListView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('OpenTasksForContactRelatedListView'))
                {
                    $metadata = OpenTasksForContactRelatedListView::getMetadata();
                    $metadata['global']['rowMenu']['elements'] = array(
                        array('type' => 'EditLink'),
                        array('type' => 'CopyLink'),
                        array('type' => 'RelatedDeleteLink'),
                    );
                    OpenTasksForContactRelatedListView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('OpenTasksForOpportunityRelatedListView'))
                {
                    $metadata = OpenTasksForOpportunityRelatedListView::getMetadata();
                    $metadata['global']['rowMenu']['elements'] = array(
                        array('type' => 'EditLink'),
                        array('type' => 'CopyLink'),
                        array('type' => 'RelatedDeleteLink'),
                    );
                    OpenTasksForOpportunityRelatedListView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('OpportunitiesMassEditView'))
                {
                    $metadata = OpportunitiesMassEditView::getMetadata();
                    $metadata['global']['nonPlaceableAttributeNames'][] = 'probability';
                    OpportunitiesMassEditView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('OpportunitiesModule'))
                {
                    $metadata = OpportunitiesModule::getMetadata();
                    $metadata['global']['stageToProbabilityMapping'] = array(
                        'Prospecting'   => 10,
                        'Qualification' => 25,
                        'Negotiating'   => 50,
                        'Verbal'        => 75,
                        'Closed Won'    => 100,
                        'Closed Lost'   => 0,
                    );
                    OpportunitiesModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('Opportunity'))
                {
                    $metadata = Opportunity::getMetadata();
                    foreach ($metadata['Opportunity']['rules'] as $key => $rule)
                    {
                        if ($rule[0] == 'stage' && $rule[1] == 'required')
                        {
                            unset($metadata['Opportunity']['rules'][$key]);
                        }
                    }
                    $metadata['Opportunity']['rules'] = array_values($metadata['Opportunity']['rules']);
                    $metadata['Opportunity']['rules'][] = array('probability', 'probability');
                    $metadata['Opportunity']['rules'][] = array('stage', 'required');
                    $metadata['Opportunity']['relations']['products'] = array(RedBeanModel::HAS_MANY, 'Product');
                    Opportunity::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('OpportunityEditAndDetailsView'))
                {
                    $metadata = OpportunityEditAndDetailsView::getMetadata();
                    $search = array('type' => 'OpportunityDeleteLink');
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        unset($metadata['global']['toolbar']['elements'][$result['elementKey']]);
                        $metadata['global']['toolbar']['elements'] = array_values($metadata['global']['toolbar']['elements']);
                    }
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'CopyLink', 'renderType' => 'Details');
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'OpportunityDeleteLink', 'renderType' => 'Details');

                    OpportunityEditAndDetailsView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('RolesModule'))
                {
                    $metadata = RolesModule::getMetadata();
                    $metadata['global']['adminTabMenuItems'] = array(
                        array(
                            'label' => "eval:Zurmo::t('ZurmoModule', 'Roles')",
                            'url'   => array('/zurmo/role'),
                            'right' => RolesModule::RIGHT_ACCESS_ROLES,
                            'items' => array(
                                array(
                                    'label' => "eval:Zurmo::t('ZurmoModule', 'Create Role')",
                                    'url'   => array('/zurmo/role/create'),
                                    'right' => RolesModule::RIGHT_CREATE_ROLES
                                ),
                                array(
                                    'label' => "eval:Zurmo::t('ZurmoModule', 'Roles')",
                                    'url'   => array('/zurmo/role'),
                                    'right' => RolesModule::RIGHT_ACCESS_ROLES
                                ),
                            ),
                        ),
                    );
                    $metadata['global']['configureMenuItems'] = array(
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'Roles')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Manage Roles')",
                            'route'            => '/zurmo/role',
                            'right'            => RolesModule::RIGHT_ACCESS_ROLES,
                        ),
                    );
                    RolesModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('TaskEditAndDetailsView'))
                {
                    $metadata = TaskEditAndDetailsView::getMetadata();
                    $search = array('type' => 'AuditEventsModalListLink');
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        unset($metadata['global']['toolbar']['elements'][$result['elementKey']]);
                        $metadata['global']['toolbar']['elements'] = array_values($metadata['global']['toolbar']['elements']);
                    }
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'CopyLink', 'renderType' => 'Details');
                    $metadata['global']['toolbar']['elements'][] = array('type' => 'AuditEventsModalListLink', 'renderType' => 'Details');

                    TaskEditAndDetailsView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('UsersModule'))
                {
                    $metadata = UsersModule::getMetadata();
                    $metadata['global']['adminTabMenuItems'] = array(
                        array(
                            'label' => "eval:Zurmo::t('UsersModule', 'Users')",
                            'url'   => array('/users/default'),
                            'right' => UsersModule::RIGHT_ACCESS_USERS,
                            'items' => array(
                                array(
                                    'label' => "eval:Zurmo::t('UsersModule', 'Create User')",
                                    'url'   => array('/users/default/create'),
                                    'right' => UsersModule::RIGHT_CREATE_USERS
                                ),
                                array(
                                    'label' => "eval:Zurmo::t('UsersModule', 'Users')",
                                    'url'   => array('/users/default'),
                                    'right' => UsersModule::RIGHT_ACCESS_USERS
                                ),
                            ),
                        ),
                    );
                    UsersModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('OpportunitiesModuleEditView'))
                {
                    $metadata = OpportunitiesModuleEditView::getMetadata();
                    $metadata['global']['panels'][0]['rows'][] =
                        array('cells' =>
                            array(
                                array(
                                    'elements' => array(
                                        array('attributeName' => 'stageToProbabilityMapping', 'type' => 'StageToProbabilityMapping'),
                                    ),
                                ),
                            ),
                        );
                    OpportunitiesModuleEditView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('OpportunityEditAndDetailsView'))
                {
                    $metadata = OpportunityEditAndDetailsView::getMetadata();
                    $search1 = array('attributeName' => 'probability');
                    $result1 = $this->findPositionOfElementInViewPanels($metadata, $search1);
                    $search2 = array('attributeName' => 'stage');
                    $result2 = $this->findPositionOfElementInViewPanels($metadata, $search2);
                    if ($result1 && $result2)
                    {
                        $temp = $metadata['global']['panels'][$result1['panelKey']]['rows'][$result1['rowKey']];
                        $metadata['global']['panels'][$result1['panelKey']]['rows'][$result1['rowKey']] = $metadata['global']['panels'][$result2['panelKey']]['rows'][$result2['rowKey']];
                        $metadata['global']['panels'][$result2['panelKey']]['rows'][$result2['rowKey']] = $temp;
                    }
                    OpportunityEditAndDetailsView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('ZurmoConfigurationEditAndDetailsView'))
                {
                    R::exec('delete from globalmetadata where classname="ZurmoConfigurationEditAndDetailsView"');
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('ZurmoModule'))
                {
                    $metadata = ZurmoModule::getMetadata();
                    $metadata['global']['configureMenuItems'] = array(
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'Global Configuration')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Manage Global Configuration')",
                            'route'            => '/zurmo/default/configurationEdit',
                            'right'            => ZurmoModule::RIGHT_ACCESS_GLOBAL_CONFIGURATION,
                        ),
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'Currency Configuration')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Manage Currency Configuration')",
                            'route'            => '/zurmo/currency/configurationList',
                            'right'            => ZurmoModule::RIGHT_ACCESS_CURRENCY_CONFIGURATION,
                        ),
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'Languages')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Manage Active Languages')",
                            'route'            => '/zurmo/language/configurationList',
                            'right'            => ZurmoModule::RIGHT_ACCESS_GLOBAL_CONFIGURATION,
                        ),
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'Developer Tools')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Access Developer Tools')",
                            'route'            => '/zurmo/development/',
                            'right'            => ZurmoModule::RIGHT_ACCESS_GLOBAL_CONFIGURATION,
                        ),
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'Authentication Configuration')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Manage Authentication Configuration')",
                            'route'            => '/zurmo/authentication/configurationEdit',
                            'right'            => ZurmoModule::RIGHT_ACCESS_GLOBAL_CONFIGURATION,
                        ),
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'Plugins')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Manage Plugins and Integrations')",
                            'route'            => '/zurmo/plugins/configurationEdit',
                            'right'            => ZurmoModule::RIGHT_ACCESS_GLOBAL_CONFIGURATION,
                        ),
                    );
                    $metadata['global']['headerMenuItems'] = array(
                        array(
                            'label'  => "eval:Zurmo::t('ZurmoModule', 'Settings')",
                            'url'    => array('/configuration'),
                            'right'  => ZurmoModule::RIGHT_ACCESS_ADMINISTRATION,
                            'order'  => 6,
                            'mobile' => false,
                        ),
                        array(
                            'label'  => "eval:Zurmo::t('ZurmoModule', 'Need Support?')",
                            'url'    => 'http://www.zurmo.com/needSupport.php',
                            'order'  => 9,
                            'mobile' => true,
                        ),
                        array(
                            'label'  => "eval:Zurmo::t('ZurmoModule', 'About Zurmo')",
                            'url'    => array('/zurmo/default/about'),
                            'order'  => 10,
                            'mobile' => true,
                        ),
                    );
                    $metadata['global']['configureSubMenuItems'] = array(
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_AUTHENTICATION,
                            'titleLabel'       => "eval:Zurmo::t('ZurmoModule', 'LDAP Configuration')",
                            'descriptionLabel' => "eval:Zurmo::t('ZurmoModule', 'Manage LDAP Authentication')",
                            'route'            => '/zurmo/ldap/configurationEditLdap',
                            'right'            => ZurmoModule::RIGHT_ACCESS_GLOBAL_CONFIGURATION,
                        ),
                    );

                    $metadata['global']['adminTabMenuItemsModuleOrdering'][] = 'contactWebForms';
                    $metadata['global']['tabMenuItemsModuleOrdering'][] = 'products';
                    ZurmoModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('WorkflowDetailsView'))
                {
                    R::exec('delete from globalmetadata where classname="WorkflowDetailsView"');
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('WorkflowsModule'))
                {
                    $metadata = WorkflowsModule::getMetadata();
                    $metadata['global']['adminTabMenuItems'] = array(
                        array(
                            'label' => "eval:Zurmo::t('WorkflowsModule', 'Workflows')",
                            'url'   => array('/workflows/default'),
                            'right' => WorkflowsModule::RIGHT_ACCESS_WORKFLOWS,
                        ),
                    );
                    $metadata['global']['configureMenuItems'] = array(
                        array(
                            'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                            'titleLabel'       => "eval:Zurmo::t('WorkflowsModule', 'Workflows')",
                            'descriptionLabel' => "eval:Zurmo::t('WorkflowsModule', 'Manage Workflows')",
                            'route'            => '/workflows/default',
                            'right'            => WorkflowsModule::RIGHT_CREATE_WORKFLOWS,
                        ),
                    );
                    WorkflowsModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('MarketingModule'))
                {
                    $metadata = MarketingModule::getMetadata();
                    $metadata['global']['tabMenuItems'] = array(
                        array(
                            'label'  => "eval:Zurmo::t('MarketingModule', 'Marketing')",
                            'url'    => array('/marketing/default/dashboardDetails'),
                            'mobile' => false,
                        ),
                    );
                    MarketingModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('ReportsModule'))
                {
                    $metadata = ReportsModule::getMetadata();
                    $metadata['global']['tabMenuItems'] = array(
                        array(
                            'label'  => "eval:Zurmo::t('ReportsModule', 'Reports')",
                            'url'    => array('/reports/default'),
                            'right'  => ReportsModule::RIGHT_ACCESS_REPORTS,
                            'mobile' => false,
                        ),
                    );
                    ReportsModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('Address'))
                {
                    R::exec('delete from globalmetadata where classname="Address"');
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('UserCreateView'))
                {
                    R::exec('delete from globalmetadata where classname="UserCreateView"');
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('UserEditView'))
                {
                    R::exec('delete from globalmetadata where classname="UserEditView"');
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('UserDetailsView'))
                {
                    R::exec('delete from globalmetadata where classname="UserDetailsView"');
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('UsersListView'))
                {
                    R::exec('delete from globalmetadata where classname="UsersListView"');
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('UsersSearchView'))
                {
                    R::exec('delete from globalmetadata where classname="UsersSearchView"');
                }
            }
        }

        /**
         * Update schema.
         */
        public function processUpdateSchema($messageLogger)
        {
            // We must update schema via command line, to ensure that newly
            // copied files are loaded into application correctly.
            InstallUtil::runAutoBuildFromUpdateSchemaCommand($messageLogger);
            ReadPermissionsOptimizationUtil::rebuild();
        }

        /**
         * Tasks to be executed after update schema
         */
        public function processAfterUpdateSchema()
        {
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '1.6.00'))
            {
                // Encrypt email passwords
                // Imap password
                $imapPassword = ZurmoConfigurationUtil::getByModuleName('EmailMessagesModule', 'imapPassword');
                if ($imapPassword != '' && $imapPassword != null)
                {
                    // Encrypt password
                    $imapPasswordEncrypted = ZurmoPasswordSecurityUtil::encrypt($imapPassword);
                    ZurmoConfigurationUtil::setByModuleName('EmailMessagesModule', 'imapPassword', $imapPasswordEncrypted);
                }

                // Outbound SMTP password
                $outboundPassword = ZurmoConfigurationUtil::getByModuleName('EmailMessagesModule', 'outboundPassword');
                if ($outboundPassword != '' && $outboundPassword != null)
                {
                    // Encrypt password
                    $outboundPasswordEncrypted = ZurmoPasswordSecurityUtil::encrypt($outboundPassword);
                    ZurmoConfigurationUtil::setByModuleName('EmailMessagesModule', 'outboundPassword', $outboundPasswordEncrypted);
                }

                // User SMTP passwords
                $emailAccounts = EmailAccount::getAll();
                if (is_array($emailAccounts) && count($emailAccounts) > 0)
                {
                    foreach ($emailAccounts as $emailAccount)
                    {
                        if ($emailAccount->outboundPassword != '' &&
                            $emailAccount->outboundPassword != null)
                        {
                            $emailAccount->save();
                        }
                    }
                }

                $emailBoxes = EmailBox::getAll();
                foreach ($emailBoxes as $emailBox)
                {
                    try
                    {
                        $folder = EmailFolder::getByBoxAndType($emailBox,  EmailFolder::getDefaultOutboxErrorName());
                    }
                    catch (NotFoundException $e)
                    {
                        $folder           = new EmailFolder();
                        $folder->name     = EmailFolder::getDefaultOutboxErrorName();
                        $folder->type     = EmailFolder::TYPE_OUTBOX_ERROR;
                        $folder->emailBox = $emailBox;
                        $emailBox->folders->add($folder);
                    }
                    catch (NotSupportedException $e)
                    {
                    }

                    try
                    {
                        $folder = EmailFolder::getByBoxAndType($emailBox,  EmailFolder::getDefaultOutboxFailureName());
                    }
                    catch (NotFoundException $e)
                    {
                        $folder           = new EmailFolder();
                        $folder->name     = EmailFolder::getDefaultOutboxFailureName();
                        $folder->type     = EmailFolder::TYPE_OUTBOX_FAILURE;
                        $folder->emailBox = $emailBox;
                        $emailBox->folders->add($folder);
                    }
                    catch (NotSupportedException $e)
                    {
                    }
                    $saved = $emailBox->save();
                }

                // Load default Demo Data for Products
                $productDefaultDataMaker = new ProductsDefaultDataMaker();
                $productDefaultDataMaker->make();
            }
        }

        /**
         * Clear assets and runtime folders.
         */
        public function clearAssetsAndRunTimeItems()
        {
            $pathToAssetsFolder = INSTANCE_ROOT . DIRECTORY_SEPARATOR . 'assets';
            FileUtil::deleteDirectoryRecursive($pathToAssetsFolder, false, array('index.html'));

            $pathToRuntimeFolder = Yii::app()->getRuntimePath();
            FileUtil::deleteDirectoryRecursive($pathToRuntimeFolder, false, array('index.html', 'upgrade'));
        }

        /**
         * Some final tasks can be added here.
         */
        public function processFinalTouches()
        {
        }

        /**
         * Determine if some task should be executed or not, depending on Zurmo version.
         * @param string $currentZurmoVersion
         * @param string $upgradeVersion - version in which change is added
         * @return boolean
         */
        protected function shouldRunTasksByVersion($currentZurmoVersion, $upgradeVersion)
        {
            $shouldRun = false;
            if (version_compare($currentZurmoVersion, $upgradeVersion, '<='))
            {
                $shouldRun = true;
            }
            return $shouldRun;
        }
    }
?>
