<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2013 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2013. All rights reserved".
     ********************************************************************************/

    class BaseUpgraderComponent extends CApplicationComponent
    {
        protected $_messageLogger = null;

        public function getMessageLogger() {
            return $this->_messageLogger;
        }

        public function setMessageLogger($messageLogger) {
            $this->_messageLogger = $messageLogger;
        }

        /**
         * Tasks to do before start processing config files.
         */
        public function processBeforeConfigFiles()
        {
            $this->messageLogger->addInfoMessage("Process before config files.");
        }

        /**
         * Process config files.
         * Problem here is that some config files might be modified, so there
         * can be some additional config options. We need to modify config
         * files(debug.php, debugTest.php, perInstance.php, perInstanceTest.php),
         * without loosing settings entered by user, so in most cases we will add
         * content at the end of those files where possible.
         * Should be used to files like debug.php, debugTest.php, perInstance.php, perInstanceTest.php,
         * which are not tracked by SCM tools, other files are updated automatically.
         *
         * @code
            <?php
                //Below is just sample code to add $myVariable to perInstance.php and perInstanceTest.php files.
                $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
                if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '0.6.80'))
                {
                    $perInstanceTestFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstanceTest.php';
                    if (is_file($perInstanceTestFile))
                    {
                        $perInstanceTestContent = file_get_contents($perInstanceTestFile);

                        $contentToAdd = "\$myVariable = 'aaa';\n";
                        $perInstanceTestContent = preg_replace('/\?\>/',
                                                               "\n" . $contentToAdd . "\n" . "?>",
                                                               $perInstanceTestContent);
                        file_put_contents($perInstanceTestFile, $perInstanceTestContent);
                    }
                    $perInstanceFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstance.php';
                    if (is_file($perInstanceFile))
                    {
                        $perInstanceContent = file_get_contents($perInstanceFile);

                        $contentToAdd = "\$myVariable = 'aaa';\n";
                        $perInstanceContent = preg_replace('/\?\>/',
                                                           "\n" . $contentToAdd . "\n" . "?>",
                                                           $perInstanceContent);
                        file_put_contents($perInstanceFile, $perInstanceContent);
                    }
                }
            ?>
         * @endcode
         */
        public function processConfigFiles($pathToConfigurationFolder)
        {
        }

        /**
         * Tasks after config files are processed.
         */
        public function processAfterConfigFiles()
        {

        }

        /**
         * Tasks to do before start processing files.
         */
        public function processBeforeFiles()
        {
        }

        /**
         * Process files
         * @param string $source
         * @param string $destination
         * @param array $configuration
         */
        public function processFiles($source, $destination, $configuration)
        {
            // Remove files that are marked for removal in manifest.php.
            if (isset($configuration['removedFiles']) && !empty($configuration['removedFiles']))
            {
                foreach ($configuration['removedFiles'] as $fileOrFolderToRemove)
                {
                    // Replace Lunux directory separators(from upgrade manifest file)
                    // with one used by system(DIRECTORY_SEPARATOR)
                    $fileOrFolderToRemove = str_replace('\\', DIRECTORY_SEPARATOR, $fileOrFolderToRemove);
                    $fileOrFolderToRemove = $destination . DIRECTORY_SEPARATOR . trim($fileOrFolderToRemove, DIRECTORY_SEPARATOR);
                    if (is_dir($fileOrFolderToRemove))
                    {
                        FileUtil::deleteDirectoryRecursive($fileOrFolderToRemove);
                    }
                    elseif (is_file($fileOrFolderToRemove))
                    {
                        unlink($fileOrFolderToRemove);
                    }
                }
            }
            // Copy new or modified files.
            FileUtil::copyRecoursive($source, $destination);
        }

        /**
         * Tasks after files are processed.
         */
        public function processAfterFiles()
        {
        }

        /**
         * Tasks that should be executed before updating schema.
         * @param string $currentZurmoVersion
         *
         * @code
            <?php
                $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
                if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '0.6.80'))
                {
                    $metadata = AccountsModule::getMetadata();
                    if(!isset($metadata['global']['newElement']))
                    {
                        $metadata['global']['newElement'] = 'Some Content';
                        AccountsModule::setMetadata($metadata);
                    }
                    GeneralCache::forgetAll();
                }
            ?>
         * @endcode
         */
        public function processBeforeUpdateSchema()
        {
        }

        /**
         * Update schema.
         */
        public function processUpdateSchema($messageLogger)
        {
            // We must update schema via command line, to ensure that newly
            // copied files are loaded into application correctly.
            InstallUtil::runAutoBuildFromUpdateSchemaCommand($messageLogger);
            ReadPermissionsOptimizationUtil::rebuild();
        }

        /**
         * Tasks to be executed after update schema
         */
        public function processAfterUpdateSchema()
        {
        }

        /**
         * Clear assets and runtime folders.
         */
        public function clearAssetsAndRunTimeItems()
        {
            $pathToAssetsFolder = INSTANCE_ROOT . DIRECTORY_SEPARATOR . 'assets';
            FileUtil::deleteDirectoryRecursive($pathToAssetsFolder, false, array('index.html'));

            $pathToRuntimeFolder = Yii::app()->getRuntimePath();
            FileUtil::deleteDirectoryRecursive($pathToRuntimeFolder, false, array('index.html', 'upgrade'));
        }

        /**
         * Some final tasks can be added here.
         */
        public function processFinalTouches()
        {
        }

        /**
         * Determine if some task should be executed or not, depending on Zurmo version.
         * @param string $currentZurmoVersion
         * @param string $upgradeVersion - version in which change is added
         * @return boolean
         */
        protected function shouldRunTasksByVersion($currentZurmoVersion, $upgradeVersion)
        {
            $shouldRun = false;
            if (version_compare($currentZurmoVersion, $upgradeVersion, '<='))
            {
                $shouldRun = true;
            }
            return $shouldRun;
        }

        /**
         * @param array $metadata
         * @param array $elementToFind
         * @return boolean|multitype:array
         */
        protected function findPositionOfElementInViewPanels($metadata, $elementToFind)
        {
            if (!isset($metadata['global']['panels']) || !is_array($metadata['global']['panels']))
            {
                return false;
            }

            foreach ($metadata['global']['panels'] as $panelKey => $panels)
            {
                if (!isset($panels['rows']) || !is_array($panels['rows']))
                {
                    continue;
                }
                foreach ($panels['rows'] as $rowKey => $rows)
                {
                    if (!isset($rows['cells']) || !is_array($rows['cells']))
                    {
                        continue;
                    }
                    foreach ($rows['cells'] as $cellKey => $cells)
                    {
                        if (!isset($cells['elements']) || !is_array($cells['elements']))
                        {
                            continue;
                        }
                        foreach ($cells['elements'] as $elementKey => $elements)
                        {
                            $found = false;
                            foreach ($elementToFind as $key => $value)
                            {
                                if (isset($elements[$key]) && $elements[$key] == $value)
                                {
                                    $found = true;
                                }
                                else
                                {
                                    $found = false;
                                }
                            }
                            if ($found)
                            {
                                $result = array(
                                    'panelKey'   => $panelKey,
                                    'rowKey'     => $rowKey,
                                    'cellKey'    => $cellKey,
                                    'elementKey' => $elementKey
                                );
                                return $result;
                            }
                        }
                    }
                }
            }
            return false;
        }

        /**
         * @param array $metadata
         * @param string $type
         * @param array $elementToFind
         * @throws NotSupportedException
         * @return boolean|multitype:array
         */
        protected function findPositionOfElementInViewToolbarsAndRowMenu($metadata, $type, $elementToFind)
        {
            if ($type !== 'toolbar' && $type !== 'rowMenu')
            {
                throw new NotSupportedException();
            }
            if (!isset($metadata['global'][$type]['elements']) || !is_array($metadata['global'][$type]['elements']))
            {
                return false;
            }

            foreach ($metadata['global'][$type]['elements'] as $elementKey => $elements)
            {
                $found = false;
                foreach ($elementToFind as $key => $value)
                {
                    if (isset($elements[$key]) && $elements[$key] == $value)
                    {
                        $found = true;
                    }
                    else
                    {
                        $found = false;
                    }
                }
                if ($found)
                {
                    $result = array(
                        'elementKey' => $elementKey
                    );
                    return $result;
                }
            }
            return false;
        }
    }
?>
