<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2017 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2017. All rights reserved".
     ********************************************************************************/

    class LeadsDefaultController extends ZurmoModuleController
    {
        public function filters()
        {
            return array_merge(parent::filters(),
                array(
                   array(
                        ZurmoBaseController::RIGHTS_FILTER_PATH . ' + convert, convertFinal, saveConvertedContact',
                        'moduleClassName' => 'LeadsModule',
                        'rightName' => LeadsModule::RIGHT_CONVERT_LEADS,
                   ),
                   array(
                        ZurmoBaseController::REQUIRED_ATTRIBUTES_FILTER_PATH . ' + create, edit',
                        'moduleClassName' => get_class($this->getModule()),
                        'viewClassName'   => 'LeadEditAndDetailsView',
                   ),
                   array(
                        ZurmoBaseController::REQUIRED_ATTRIBUTES_FILTER_PATH . ' + convert',
                        'moduleClassName' => 'AccountsModule',
                        'viewClassName'   => 'AccountConvertToView',
                   ),
                   array(
                        ZurmoBaseController::REQUIRED_ATTRIBUTES_FILTER_PATH . ' + convertFinal',
                        'moduleClassName' => 'OpportunitiesModule',
                        'viewClassName'   => 'OpportunityConvertToView',
                   ),
                   array(
                        ZurmoModuleController::ZERO_MODELS_CHECK_FILTER_PATH . ' + list, index',
                        'controller'                    => $this,
                        'stateMetadataAdapterClassName' => 'LeadsStateMetadataAdapter'
                   ),
               )
            );
        }

        public function actionList()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'listPageSize', get_class($this->getModule()));
            $contact                        = new Contact(false);
            $searchForm                     = new LeadsSearchForm($contact);
            $listAttributesSelector         = new ListAttributesSelector('LeadsListView', get_class($this->getModule()));
            $searchForm->setListAttributesSelector($listAttributesSelector);
            $dataProvider = $this->resolveSearchDataProvider(
                $searchForm,
                $pageSize,
                'LeadsStateMetadataAdapter',
                'LeadsSearchView'
            );
            if (isset($_GET['ajax']) && $_GET['ajax'] == 'list-view')
            {
                $mixedView = $this->makeListView(
                    $searchForm,
                    $dataProvider
                );
                $view = new LeadsPageView($mixedView);
            }
            else
            {
                $mixedView = $this->makeActionBarSearchAndListView($searchForm, $dataProvider,
                                                                    'SecuredActionBarForLeadsSearchAndListView');
                $view = new LeadsPageView(ZurmoDefaultViewUtil::
                                         makeStandardViewForCurrentUser($this, $mixedView));
            }
            echo $view->render();
        }

        public function actionDetails($id)
        {
            $contact = static::getModelAndCatchNotFoundAndDisplayError('Contact', intval($id));
            ControllerSecurityUtil::resolveAccessCanCurrentUserReadModel($contact);
            if (!LeadsUtil::isStateALead($contact->state))
            {
                $urlParams = array('/contacts/' . $this->getId() . '/details', 'id' => $contact->id);
                $this->redirect($urlParams);
            }
            else
            {
                AuditEvent::logAuditEvent('ZurmoModule', ZurmoModule::AUDIT_EVENT_ITEM_VIEWED, array(strval($contact), 'LeadsModule'), $contact);
                $getData                 = GetUtil::getData();
                $isKanbanBoardInRequest  = ArrayUtil::getArrayValue($getData, 'kanbanBoard');
                if ($isKanbanBoardInRequest == 0 || $isKanbanBoardInRequest == null || Yii::app()->userInterface->isMobile() === true)
                {
                    $breadCrumbView          = StickySearchUtil::resolveBreadCrumbViewForDetailsControllerAction($this, 'LeadsSearchView', $contact);
                    $detailsAndRelationsView = $this->makeDetailsAndRelationsView($contact, 'LeadsModule',
                                                                                  'LeadDetailsAndRelationsView',
                                                                                  Yii::app()->request->getRequestUri(),
                                                                                  $breadCrumbView);
                    $view = new LeadsPageView(ZurmoDefaultViewUtil::
                                             makeStandardViewForCurrentUser($this, $detailsAndRelationsView));
                }
                else
                {
                    $pageSize       = TasksForRelatedKanbanView::getDefaultPageSize();
                    $task           = new Task(false);
                    $searchForm     = new TasksForRelatedKanbanSearchForm($task, $contact);
                    $stickySearchKey = 'TasksForRelatedKanbanSearchView';
                    $dataProvider = $this->resolveSearchDataProvider(
                        $searchForm,
                        $pageSize,
                        null,
                        $stickySearchKey
                    );
                    $view = TasksUtil::resolveTaskKanbanViewForRelation($contact, $this->getModule()->getId(), $this,
                                                                        'TasksForLeadKanbanView', 'LeadsPageView',
                                                                        $searchForm, $dataProvider);
                }
                echo $view->render();
            }
        }

        /**
         * This method is called prior to creation of data provider in order to add
         * search metadata for related model. Used in actionDetails for Kanban view.
         */
        protected function resolveFilteredByMetadataBeforeMakingDataProvider($searchForm, & $metadata)
        {
            if ($searchForm instanceof TasksForRelatedKanbanSearchForm)
            {
                TasksUtil::resolveRelatedAdditionalSearchMetadata($searchForm, $metadata, 'activityItems');
            }
        }

        public function actionCreate()
        {
            $titleBarAndEditView = $this->makeEditAndDetailsView(
                    $this->attemptToSaveModelFromPost(new Contact()), 'Edit',
                    'LeadTitleBarAndEditAndDetailsView'
            );
            $view = new LeadsPageView(ZurmoDefaultViewUtil::
                                     makeStandardViewForCurrentUser($this, $titleBarAndEditView));
            echo $view->render();
        }

        public function actionEdit($id, $redirectUrl = null)
        {
            $contact = Contact::getById(intval($id));
            ControllerSecurityUtil::resolveAccessCanCurrentUserWriteModel($contact);
            if (!LeadsUtil::isStateALead($contact->state))
            {
                $urlParams = array('/contacts/' . $this->getId() . '/edit', 'id' => $contact->id);
                $this->redirect($urlParams);
            }
            else
            {
                $this->processEdit($contact, $redirectUrl);
            }
        }

        public function actionCopy($id)
        {
            $copyToContact  = new Contact();
            $postVariableName   = get_class($copyToContact);
            if (!isset($_POST[$postVariableName]))
            {
                $contact        = Contact::getById((int)$id);
                ControllerSecurityUtil::resolveAccessCanCurrentUserReadModel($contact);
                ZurmoCopyModelUtil::copy($contact, $copyToContact);
            }
            $this->processEdit($copyToContact);
        }

        protected function processEdit(Contact $contact, $redirectUrl = null)
        {
            $view = new LeadsPageView(ZurmoDefaultViewUtil::
                                     makeStandardViewForCurrentUser($this,
                                         $this->makeEditAndDetailsView(
                                            $this->attemptToSaveModelFromPost($contact, $redirectUrl), 'Edit',
                                                        'LeadTitleBarAndEditAndDetailsView')));
            echo $view->render();
        }

        /**
         * Action for displaying a mass edit form and also action when that form is first submitted.
         * When the form is submitted, in the event that the quantity of models to update is greater
         * than the pageSize, then once the pageSize quantity has been reached, the user will be
         * redirected to the makeMassEditProgressView.
         * In the mass edit progress view, a javascript refresh will take place that will call a refresh
         * action, usually massEditProgressSave.
         * If there is no need for a progress view, then a flash message will be added and the user will
         * be redirected to the list view for the model.  A flash message will appear providing information
         * on the updated records.
         * @see Controler->makeMassEditProgressView
         * @see Controller->processMassEdit
         * @see
         */
        public function actionMassEdit()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'massEditProgressPageSize');
            $activeAttributes = $this->resolveActiveAttributesFromMassEditPost();
            $contact = new Contact(false);
            $dataProvider = $this->getDataProviderByResolvingSelectAllFromGet(
                new LeadsSearchForm($contact),
                $pageSize,
                Yii::app()->user->userModel->id,
                'LeadsStateMetadataAdapter',
                'LeadsSearchView');
            $selectedRecordCount = static::getSelectedRecordCountByResolvingSelectAllFromGet($dataProvider);
            $contact = $this->processMassEdit(
                $pageSize,
                $activeAttributes,
                $selectedRecordCount,
                'LeadsPageView',
                $contact,
                Zurmo::t('LeadsModule', 'Leads'),
                $dataProvider
            );
            $massEditView = $this->makeMassEditView(
                $contact,
                $activeAttributes,
                $selectedRecordCount,
                Zurmo::t('LeadsModule', 'Leads')
            );
            $view = new LeadsPageView(ZurmoDefaultViewUtil::
                                     makeStandardViewForCurrentUser($this, $massEditView));
            echo $view->render();
        }

        /**
         * Action called in the event that the mass edit quantity is larger than the pageSize.
         * This action is called after the pageSize quantity has been updated and continues to be
         * called until the mass edit action is complete.  For example, if there are 20 records to update
         * and the pageSize is 5, then this action will be called 3 times.  The first 5 are updated when
         * the actionMassEdit is called upon the initial form submission.
         */
        public function actionMassEditProgressSave()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'massEditProgressPageSize');
            $contact = new Contact(false);
            $dataProvider = $this->getDataProviderByResolvingSelectAllFromGet(
                            new LeadsSearchForm($contact),
                            $pageSize,
                            Yii::app()->user->userModel->id,
                            'LeadsStateMetadataAdapter',
                            'LeadsSearchView');
            $this->processMassEditProgressSave(
                'Contact',
                $pageSize,
                Zurmo::t('LeadsModule', 'Leads'),
                $dataProvider
            );
        }

        /**
         * Action for displaying a mass delete form and also action when that form is first submitted.
         * When the form is submitted, in the event that the quantity of models to delete is greater
         * than the pageSize, then once the pageSize quantity has been reached, the user will be
         * redirected to the makeMassDeleteProgressView.
         * In the mass delete progress view, a javascript refresh will take place that will call a refresh
         * action, usually makeMassDeleteProgressView.
         * If there is no need for a progress view, then a flash message will be added and the user will
         * be redirected to the list view for the model.  A flash message will appear providing information
         * on the delete records.
         * @see Controler->makeMassDeleteProgressView
         * @see Controller->processMassDelete
         * @see
         */
        public function actionMassDelete()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'massDeleteProgressPageSize');
            $contact = new Contact(false);

            $activeAttributes = $this->resolveActiveAttributesFromMassDeletePost();
            $dataProvider = $this->getDataProviderByResolvingSelectAllFromGet(
                            new LeadsSearchForm($contact),
                            $pageSize,
                            Yii::app()->user->userModel->id,
                            'LeadsStateMetadataAdapter',
                            'LeadsSearchView');
            $selectedRecordCount = static::getSelectedRecordCountByResolvingSelectAllFromGet($dataProvider);
            $contact = $this->processMassDelete(
                $pageSize,
                $activeAttributes,
                $selectedRecordCount,
                'LeadsPageView',
                $contact,
                Zurmo::t('LeadsModule', 'Leads'),
                $dataProvider
            );
            $massDeleteView = $this->makeMassDeleteView(
                $contact,
                $activeAttributes,
                $selectedRecordCount,
                Zurmo::t('LeadsModule', 'Leads')
            );
            $view = new LeadsPageView(ZurmoDefaultViewUtil::
                                         makeStandardViewForCurrentUser($this, $massDeleteView));
            echo $view->render();
        }

        /**
         * Action called in the event that the mass delete quantity is larger than the pageSize.
         * This action is called after the pageSize quantity has been delted and continues to be
         * called until the mass delete action is complete.  For example, if there are 20 records to delete
         * and the pageSize is 5, then this action will be called 3 times.  The first 5 are updated when
         * the actionMassDelete is called upon the initial form submission.
         */
        public function actionMassDeleteProgress()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'massDeleteProgressPageSize');
            $contact = new Contact(false);
            $dataProvider = $this->getDataProviderByResolvingSelectAllFromGet(
                            new LeadsSearchForm($contact),
                            $pageSize,
                            Yii::app()->user->userModel->id,
                            'LeadsStateMetadataAdapter',
                            'LeadsSearchView');
            $this->processMassDeleteProgress(
                'Contact',
                $pageSize,
                Zurmo::t('LeadsModule', 'Leads'),
                $dataProvider
            );
        }

        public function actionConvert($id)
        {
            assert('!empty($id)');
            LeadsUtil::removeFromSession(LeadsUtil::LEAD_CONVERSION_ACCOUNT_DATA_SESSION_KEY);

            $contact                 = Contact::getById(intval($id));
            if (!LeadsUtil::isStateALead($contact->state))
            {
                $urlParams = array('/contacts/' . $this->getId() . '/details', 'id' => $contact->id);
                $this->redirect($urlParams);
            }
            $convertToAccountSetting = LeadsModule::getConvertToAccountSetting();
            $selectAccountForm       = new AccountSelectForm();
            $account                 = new Account();
            ControllerSecurityUtil::resolveAccessCanCurrentUserWriteModel($contact);

            $userCanAccessContacts = RightsUtil::canUserAccessModule('ContactsModule', Yii::app()->user->userModel);
            $userCanAccessAccounts = RightsUtil::canUserAccessModule('AccountsModule', Yii::app()->user->userModel);
            $userCanCreateAccount  = RightsUtil::doesUserHaveAllowByRightName('AccountsModule',
                                     AccountsModule::RIGHT_CREATE_ACCOUNTS, Yii::app()->user->userModel);
            LeadsControllerSecurityUtil::
                resolveCanUserProperlyConvertLead($userCanAccessContacts, $userCanAccessAccounts, $convertToAccountSetting);
            if (isset($_POST['AccountSelectForm']))
            {
                $selectAccountForm->setAttributes($_POST['AccountSelectForm']);
                if ($selectAccountForm->validate())
                {
                    $account = Account::getById(intval($selectAccountForm->accountId));
                    $accountPostData = array(
                        'accountId' => intval($selectAccountForm->accountId),
                        'SelectAccount' => true
                    );
                    LeadsUtil::storeIntoSession(LeadsUtil::LEAD_CONVERSION_ACCOUNT_DATA_SESSION_KEY, $accountPostData);
                    $this->redirect(array('/leads/default/convertFinal', 'id' => $contact->id));
                }
            }
            elseif (isset($_POST['Account']))
            {
                $account = LeadsUtil::attributesToAccountWithNoPostData($contact, $account, $_POST['Account']);
                $savedSuccessfully = false;
                $modelToStringValue = null;
                $postData = $_POST['Account'];
                $controllerUtil   = static::getZurmoControllerUtil();
                $account            = $controllerUtil->saveModelFromPost($postData, $account, $savedSuccessfully,
                                                                            $modelToStringValue, true);
                if (!$account->getErrors())
                {
                    $accountPostData = $postData;
                    $accountPostData['CreateAccount'] = true;
                    LeadsUtil::storeIntoSession(LeadsUtil::LEAD_CONVERSION_ACCOUNT_DATA_SESSION_KEY, $accountPostData);
                    $this->redirect(array('/leads/default/convertFinal', 'id' => $contact->id));
                }
            }
            elseif (isset($_POST['AccountSkip']) ||
                $convertToAccountSetting == LeadsModule::CONVERT_NO_ACCOUNT ||
                ($convertToAccountSetting == LeadsModule::CONVERT_ACCOUNT_NOT_REQUIRED && !$userCanAccessAccounts)
                )
            {
                $accountPostData = array('AccountSkip' => true);
                LeadsUtil::storeIntoSession(LeadsUtil::LEAD_CONVERSION_ACCOUNT_DATA_SESSION_KEY, $accountPostData);
                $this->redirect(array('/leads/default/convertFinal', 'id' => $contact->id));
            }
            else
            {
                $account = LeadsUtil::attributesToAccount($contact, $account);
                $account = Yii::app()->custom->resolveLeadToAccountCustomAttributesWithNoPostData($contact,
                                                                                        $account, array());
            }
            $progressBarAndStepsView = new LeadConversionStepsAndProgressBarForWizardView();
            $convertView = new LeadConvertView(
                $this->getId(),
                $this->getModule()->getId(),
                $contact->id,
                strval($contact),
                $selectAccountForm,
                $account,
                $convertToAccountSetting,
                $userCanCreateAccount
            );
            $view = new LeadsPageView(ZurmoDefaultViewUtil::
                                     makeTwoStandardViewsForCurrentUser($this, $progressBarAndStepsView, $convertView));
            echo $view->render();
        }

        public function actionConvertFinal($id)
        {
            assert('!empty($id)');

            $accountPostData = LeadsUtil::getFromSession(LeadsUtil::LEAD_CONVERSION_ACCOUNT_DATA_SESSION_KEY);
            if (empty($accountPostData))
            {
                $urlParams = array('/leads/' . $this->getId() . '/convert', 'id' => $id);
                $this->redirect($urlParams);
            }
            $contact                        = Contact::getById(intval($id));
            if (!LeadsUtil::isStateALead($contact->state))
            {
                $urlParams = array('/contacts/' . $this->getId() . '/details', 'id' => $contact->id);
                $this->redirect($urlParams);
            }
            $convertToAccountSetting        = LeadsModule::getConvertToAccountSetting();
            $convertToOpportunitySetting    = LeadsModule::getConvertToOpportunitySetting();
            $opportunity                    = new Opportunity();
            ControllerSecurityUtil::resolveAccessCanCurrentUserWriteModel($contact);

            $userCanAccessContacts = RightsUtil::canUserAccessModule('ContactsModule', Yii::app()->user->userModel);
            $userCanAccessAccounts = RightsUtil::canUserAccessModule('AccountsModule', Yii::app()->user->userModel);
            $userCanAccessOpportunities = RightsUtil::canUserAccessModule('OpportunitiesModule', Yii::app()->user->userModel);
            $userCanCreateOpportunity  = RightsUtil::doesUserHaveAllowByRightName('OpportunitiesModule',
                                     OpportunitiesModule::RIGHT_CREATE_OPPORTUNITIES, Yii::app()->user->userModel);
            LeadsControllerSecurityUtil::resolveCanUserProperlyConvertLead($userCanAccessContacts,
                                                            $userCanAccessAccounts, $convertToAccountSetting);
            LeadsControllerSecurityUtil::resolveCanUserProperlyConvertLeadFinalStep($userCanAccessContacts,
                                                            $userCanAccessOpportunities, $convertToOpportunitySetting);

            if (isset($_POST['Opportunity']))
            {
                $controllerUtil     = static::getZurmoControllerUtil();
                $savedSuccessfully  = false;
                $modelToStringValue = null;
                $postData           = $_POST['Opportunity'];
                $opportunity = Yii::app()->custom->resolveLeadToOpportunityCustomAttributesWithNoPostData($contact,
                                                                                            $opportunity, $postData);
                $opportunity        = $controllerUtil->saveModelFromPost($postData, $opportunity, $savedSuccessfully,
                                                                                            $modelToStringValue, false);
                if ($savedSuccessfully)
                {
                    $explicitReadWriteModelPermissions = ExplicitReadWriteModelPermissionsUtil::makeBySecurableItem($contact);
                    ExplicitReadWriteModelPermissionsUtil::resolveExplicitReadWriteModelPermissions($opportunity, $explicitReadWriteModelPermissions);
                    $account = LeadsUtil:: createAccountForLeadConversionFromAccountPostData($accountPostData, $contact, $controllerUtil);
                    $opportunity->account = $account;
                    if (!$opportunity->save())
                    {
                        throw new NotSupportedException();
                    }
                    LeadsUtil::removeFromSession(LeadsUtil::LEAD_CONVERSION_ACCOUNT_DATA_SESSION_KEY);
                    $this->actionSaveConvertedContact($contact, $account, $opportunity);
                }
            }
            elseif (isset($_POST['OpportunitySkip']) ||
                $convertToOpportunitySetting == LeadsModule::CONVERT_NO_OPPORTUNITY ||
                ($convertToOpportunitySetting == LeadsModule::CONVERT_OPPORTUNITY_NOT_REQUIRED && !$userCanAccessOpportunities)
                    )
            {
                $controllerUtil     = static::getZurmoControllerUtil();
                $account            = LeadsUtil::createAccountForLeadConversionFromAccountPostData($accountPostData,
                                                                                            $contact, $controllerUtil);
                LeadsUtil::removeFromSession(LeadsUtil::LEAD_CONVERSION_ACCOUNT_DATA_SESSION_KEY);
                $this->actionSaveConvertedContact($contact, $account, null);
            }
            else
            {
                $opportunity = Yii::app()->custom->resolveLeadToOpportunityCustomAttributesWithNoPostData(
                                                                            $contact, $opportunity, array());
            }

            $progressBarAndStepsView = new LeadConversionStepsAndProgressBarForWizardView(1);
            $convertView = new LeadConvertOpportunityView(
                $this->getId(),
                $this->getModule()->getId(),
                $contact->id,
                strval($contact),
                $opportunity,
                $convertToOpportunitySetting,
                $userCanCreateOpportunity
            );

            $view = new LeadsPageView(ZurmoDefaultViewUtil::
                                     makeTwoStandardViewsForCurrentUser($this, $progressBarAndStepsView, $convertView));
            echo $view->render();
        }

        protected function actionSaveConvertedContact($contact, $account = null, $opportunity = null)
        {
            $contact->account = $account;
            if ($opportunity !== null)
            {
                $contact->opportunities->add($opportunity);
            }
            $contact->state   = ContactsUtil::getStartingState();
            if ($contact->save())
            {
                Yii::app()->user->setFlash('notification',
                    Zurmo::t('LeadsModule', 'LeadsModuleSingularLabel successfully converted.',
                                           LabelUtil::getTranslationParamsForAllModules())
                );
                $this->redirect(array('/contacts/default/details', 'id' => $contact->id));
            }
            Yii::app()->user->setFlash('notification',
                Zurmo::t('LeadsModule', 'LeadsModuleSingularLabel was not converted. An error occurred.')
            );
            $this->redirect(array('default/details', 'id' => $contact->id));
            Yii::app()->end(0, false);
        }

        public function actionModalList()
        {
            $modalListLinkProvider = new SelectFromRelatedEditModalListLinkProvider(
                                            $_GET['modalTransferInformation']['sourceIdFieldId'],
                                            $_GET['modalTransferInformation']['sourceNameFieldId'],
                                            $_GET['modalTransferInformation']['modalId']
            );
            echo ModalSearchListControllerUtil::setAjaxModeAndRenderModalSearchList($this, $modalListLinkProvider,
                                                'LeadsStateMetadataAdapter');
        }

        public function actionDelete($id)
        {
            $contact = Contact::GetById(intval($id));
            ControllerSecurityUtil::resolveAccessCanCurrentUserDeleteModel($contact);
            if (!LeadsUtil::isStateALead($contact->state))
            {
                $urlParams = array('/contacts/' . $this->getId() . '/delete', 'id' => $contact->id);
                $this->redirect($urlParams);
            }
            else
            {
                $contact->delete();
                $this->redirect(array($this->getId() . '/index'));
            }
        }

        /**
         * Override to always add contact state filter on search results.
         */
        public function actionAutoComplete($term, $autoCompleteOptions = null)
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'autoCompleteListPageSize', get_class($this->getModule()));
            $autoCompleteResults = ContactAutoCompleteUtil::getByPartialName($term, $pageSize, 'LeadsStateMetadataAdapter', $autoCompleteOptions);
            echo CJSON::encode($autoCompleteResults);
        }

        protected function makeEditAndDetailsView($model, $renderType)
        {
            assert('$model != null');
            assert('$renderType == "Edit" || $renderType == "Details"');
            $editViewClassName = 'LeadEditAndDetailsView';
            return new $editViewClassName($renderType, $this->getId(), $this->getModule()->getId(), $model);
        }

        protected static function getSearchFormClassName()
        {
            return 'LeadsSearchForm';
        }

        public function actionExport()
        {
            $this->export('LeadsSearchView');
        }

        /**
         * Overriding to implement the dedupe action for new leads
         */
        public function actionSearchForDuplicateModels($attribute, $value)
        {
            assert('is_string($attribute)');
            assert('is_string($value)');
            $model          = new Contact();
            $depudeRules    = DedupeRulesFactory::createRulesByModel($model);
            $viewClassName  = $depudeRules->getDedupeViewClassName();
            $searchResult   = $depudeRules->searchForDuplicateModels($attribute, $value);
            if ($searchResult != null)
            {
                $summaryView    = new $viewClassName($this->id, $this->module->id, $model, $searchResult['matchedModels']);
                $content        = $summaryView->render();
                $message        = $searchResult['message'];
                echo CJSON::encode(array('content' => $content, 'message' => $message));
            }
        }

        /**
         * List view merge for leads
         */
        public function actionListViewMerge()
        {
            $this->processListViewMerge('Contact',
                                        'LeadsListDuplicateMergedModelForm',
                                        'LeadsMerged', 'LeadsPageView',
                                        '/leads/default/list');
        }

        public function actionMassSubscribe()
        {
            $this->triggerMassAction('Contact',
                static::getSearchFormClassName(),
                'LeadsPageView',
                LeadsModule::getModuleLabelByTypeAndLanguage('Plural'),
                'LeadsSearchView',
                'LeadsStateMetadataAdapter',
                false);
        }

        public function actionMassSubscribeProgress()
        {
            $this->triggerMassAction('Contact',
                static::getSearchFormClassName(),
                'LeadsPageView',
                LeadsModule::getModuleLabelByTypeAndLanguage('Plural'),
                'LeadsSearchView',
                'LeadsStateMetadataAdapter',
                false);
        }

        protected static function resolveTitleByMassActionId($actionId)
        {
            if (MassActionUtil::isMassSubscribeLikeAction($actionId))
            {
                return Zurmo::t('Core', 'Mass Subscribe');
            }
            return parent::resolveTitleByMassActionId($actionId);
        }

        protected static function applyGenericViewIdGenerationRules($actionId)
        {
            return (MassActionUtil::isMassSubscribeLikeAction($actionId) || parent::applyGenericViewIdGenerationRules($actionId));
        }

        protected static function processModelForMassSubscribe(& $model)
        {
            $marketingListMember            = Yii::app()->request->getPost('MarketingListMember');
            if ($marketingListMember['marketingList']['id'] > 0)
            {
                $marketingList = MarketingList::getById((int) $marketingListMember['marketingList']['id']);
                $marketingList->addNewMember(null, false, $model);
                return true;
            }
        }

        protected static function resolveMassSubscribeAlertMessage($postVariableName)
        {
            $marketingListMember = Yii::app()->request->getPost('MarketingListMember');
            if (isset($marketingListMember) && $marketingListMember['marketingList']['id'] == 0)
            {
                return Zurmo::t('LeadsModule', 'You must select a MarketingListsModuleSingularLabel',
                    LabelUtil::getTranslationParamsForAllModules());
            }
        }

        /**
         * Create comment via ajax for contact
         * @param type $id
         * @param string $uniquePageId
         */
        public function actionInlineCreateCommentFromAjax($id, $uniquePageId)
        {
            $comment       = new Comment();
            $redirectUrl   = Yii::app()->createUrl('/leads/default/inlineCreateCommentFromAjax',
                array('id'           => $id,
                      'uniquePageId' => $uniquePageId));
            $urlParameters = array('relatedModelId'           => (int)$id,
                                   'relatedModelClassName'    => 'Contact',
                                   'relatedModelRelationName' => 'comments',
                                   'redirectUrl'              => $redirectUrl); //After save, the url to go to.
            $uniquePageId  = 'CommentInlineEditForModelView';
            echo             ZurmoHtml::tag('h2', array(), Zurmo::t('CommentsModule', 'Add Comment'));
            $inlineView    = new CommentInlineEditView($comment, 'default', 'comments', 'inlineCreateSave',
                $urlParameters, $uniquePageId);
            $view          = new AjaxPageView($inlineView);
            echo $view->render();
        }

        /**
         * Add subscriber for contact
         * @param int $id
         */
        public function actionAddSubscriber($id)
        {
            $contact = Contact::getById((int)$id);
            $contact    = NotificationSubscriberUtil::processSubscriptionRequest($contact, Yii::app()->user->userModel);
            $content = NotificationSubscriberUtil::getSubscriberData($contact);
            $content .= NotificationSubscriberUtil::resolveSubscriptionLink($contact, 'detail-subscribe-model-link', 'detail-unsubscribe-model-link');
            echo $content;
        }

        /**
         * Remove subscriber for contact
         * @param int $id
         */
        public function actionRemoveSubscriber($id)
        {
            $contact = Contact::getById((int)$id);
            $contact    = NotificationSubscriberUtil::processUnsubscriptionRequest($contact, Yii::app()->user->userModel);
            $content = NotificationSubscriberUtil::getSubscriberData($contact);
            $content .= NotificationSubscriberUtil::resolveSubscriptionLink($contact, 'detail-subscribe-model-link', 'detail-unsubscribe-model-link');
            if ($content == null)
            {
                echo "";
            }
            else
            {
                echo $content;
            }
        }
    }
?>
