<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2017 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2017. All rights reserved".
     ********************************************************************************/

    class ContactsDefaultController extends ZurmoModuleController
    {
        public function filters()
        {
            $modelClassName   = $this->getModule()->getPrimaryModelName();
            $viewClassName    = $modelClassName . 'EditAndDetailsView';
            return array_merge(parent::filters(),
                array(
                    array(
                        ZurmoBaseController::REQUIRED_ATTRIBUTES_FILTER_PATH . ' + create, createFromRelation, edit',
                        'moduleClassName' => get_class($this->getModule()),
                        'viewClassName'   => $viewClassName,
                   ),
                    array(
                        ZurmoModuleController::ZERO_MODELS_CHECK_FILTER_PATH . ' + list, index',
                        'controller'                    => $this,
                        'stateMetadataAdapterClassName' => 'ContactsStateMetadataAdapter'
                   ),
               )
            );
        }

        public function actionList($searchFormClassName = null)
        {
            $pageSize                       = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                                              'listPageSize', get_class($this->getModule()));
            $contact                        = new Contact(false);

            if (isset($searchFormClassName) && class_exists($searchFormClassName))
            {
                $searchForm                     = new $searchFormClassName($contact);
                $stickySearchKey = $searchFormClassName; // We need to change this
            }
            else
            {
                $searchForm                     = new ContactsSearchForm($contact);
                $stickySearchKey = 'ContactsSearchView';
            }

            $listAttributesSelector         = new ListAttributesSelector('ContactsListView', get_class($this->getModule()));
            $searchForm->setListAttributesSelector($listAttributesSelector);
            $dataProvider = $this->resolveSearchDataProvider(
                $searchForm,
                $pageSize,
                'ContactsStateMetadataAdapter',
                $stickySearchKey
            );
            if (isset($_GET['ajax']) && $_GET['ajax'] == 'list-view')
            {
                $mixedView = $this->makeListView(
                    $searchForm,
                    $dataProvider
                );
                $view = new ContactsPageView($mixedView);
            }
            else
            {
                $mixedView = $this->makeActionBarSearchAndListView($searchForm, $dataProvider,
                                                                   'SecuredActionBarForContactsSearchAndListView');
                $view = new ContactsPageView(ZurmoDefaultViewUtil::
                                         makeStandardViewForCurrentUser($this, $mixedView));
            }
            echo $view->render();
        }

        public function actionDetails($id)
        {
            $contact = static::getModelAndCatchNotFoundAndDisplayError('Contact', intval($id));
                                    ControllerSecurityUtil::resolveAccessCanCurrentUserReadModel($contact);
            AuditEvent::logAuditEvent('ZurmoModule', ZurmoModule::AUDIT_EVENT_ITEM_VIEWED, array(strval($contact), 'ContactsModule'), $contact);
            if (KanbanUtil::isKanbanRequest() === false)
            {
                $breadCrumbView          = StickySearchUtil::resolveBreadCrumbViewForDetailsControllerAction($this, 'ContactsSearchView', $contact);
                $detailsAndRelationsView = $this->makeDetailsAndRelationsView($contact, 'ContactsModule',
                                                                              'ContactDetailsAndRelationsView',
                                                                              Yii::app()->request->getRequestUri(),
                                                                              $breadCrumbView);
                $view                    = new ContactsPageView(ZurmoDefaultViewUtil::
                                                                        makeStandardViewForCurrentUser($this, $detailsAndRelationsView));
            }
            else
            {
                $pageSize       = TasksForRelatedKanbanView::getDefaultPageSize();
                $task           = new Task(false);
                $searchForm     = new TasksForRelatedKanbanSearchForm($task, $contact);
                $stickySearchKey = 'TasksForRelatedKanbanSearchView';
                $dataProvider = $this->resolveSearchDataProvider(
                    $searchForm,
                    $pageSize,
                    null,
                    $stickySearchKey
                );
                $view = TasksUtil::resolveTaskKanbanViewForRelation($contact, $this->getModule()->getId(), $this,
                                                                    'TasksForContactKanbanView', 'ContactsPageView',
                                                                    $searchForm, $dataProvider);
            }
            echo $view->render();
        }

        /**
         * This method is called prior to creation of data provider in order to add
         * search metadata for related model. Used in actionDetails for Kanban view.
         */
        protected function resolveFilteredByMetadataBeforeMakingDataProvider($searchForm, & $metadata)
        {
            if ($searchForm instanceof TasksForRelatedKanbanSearchForm)
            {
                TasksUtil::resolveRelatedAdditionalSearchMetadata($searchForm, $metadata, 'activityItems');
            }
        }

        public function actionCreate()
        {
            $this->actionCreateByModel(new Contact());
        }

        public function actionCreateFromRelation($relationAttributeName, $relationModelId, $relationModuleId, $redirectUrl)
        {
            $contact             = $this->resolveNewModelByRelationInformation( new Contact(),
                                                                                $relationAttributeName,
                                                                                (int)$relationModelId,
                                                                                $relationModuleId);
            ContactsUtil::resolveAddressesFromRelatedAccount($contact);
            $this->actionCreateByModel($contact, $redirectUrl);
        }

        protected function actionCreateByModel(Contact $contact, $redirectUrl = null)
        {
            $titleBarAndEditView = $this->makeEditAndDetailsView(
                                            $this->attemptToSaveModelFromPost($contact, $redirectUrl), 'Edit');
            $view = new ContactsPageView(ZurmoDefaultViewUtil::
                                         makeStandardViewForCurrentUser($this, $titleBarAndEditView));
            echo $view->render();
        }

        public function actionEdit($id, $redirectUrl = null)
        {
            $contact = Contact::getById(intval($id));
            ControllerSecurityUtil::resolveAccessCanCurrentUserWriteModel($contact);
            $this->processEdit($contact, $redirectUrl);
        }

        public function actionCopy($id)
        {
            $copyToContact  = new Contact();
            $postVariableName   = get_class($copyToContact);
            if (!isset($_POST[$postVariableName]))
            {
                $contact        = Contact::getById((int)$id);
                ControllerSecurityUtil::resolveAccessCanCurrentUserReadModel($contact);
                ZurmoCopyModelUtil::copy($contact, $copyToContact);
            }
            $this->processEdit($copyToContact);
        }

        protected function processEdit(Contact $contact, $redirectUrl = null)
        {
            $view    = new ContactsPageView(ZurmoDefaultViewUtil::
                            makeStandardViewForCurrentUser($this,
                            $this->makeEditAndDetailsView(
                                $this->attemptToSaveModelFromPost($contact, $redirectUrl), 'Edit')));
            echo $view->render();
        }

        /**
         * Action for displaying a mass edit form and also action when that form is first submitted.
         * When the form is submitted, in the event that the quantity of models to update is greater
         * than the pageSize, then once the pageSize quantity has been reached, the user will be
         * redirected to the makeMassEditProgressView.
         * In the mass edit progress view, a javascript refresh will take place that will call a refresh
         * action, usually massEditProgressSave.
         * If there is no need for a progress view, then a flash message will be added and the user will
         * be redirected to the list view for the model.  A flash message will appear providing information
         * on the updated records.
         * @see Controler->makeMassEditProgressView
         * @see Controller->processMassEdit
         * @see
         */
        public function actionMassEdit()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'massEditProgressPageSize');
            $contact = new Contact(false);
            $activeAttributes = $this->resolveActiveAttributesFromMassEditPost();
            $dataProvider = $this->getDataProviderByResolvingSelectAllFromGet(
                            new ContactsSearchForm($contact),
                            $pageSize,
                            Yii::app()->user->userModel->id,
                            'ContactsStateMetadataAdapter',
                            'ContactsSearchView');
            $selectedRecordCount = static::getSelectedRecordCountByResolvingSelectAllFromGet($dataProvider);
            $contact = $this->processMassEdit(
                $pageSize,
                $activeAttributes,
                $selectedRecordCount,
                'ContactsPageView',
                $contact,
                ContactsModule::getModuleLabelByTypeAndLanguage('Plural'),
                $dataProvider
            );
            $massEditView = $this->makeMassEditView(
                $contact,
                $activeAttributes,
                $selectedRecordCount,
                ContactsModule::getModuleLabelByTypeAndLanguage('Plural')
            );
            $view = new ContactsPageView(ZurmoDefaultViewUtil::
                                         makeStandardViewForCurrentUser($this, $massEditView));
            echo $view->render();
        }

        /**
         * Action called in the event that the mass edit quantity is larger than the pageSize.
         * This action is called after the pageSize quantity has been updated and continues to be
         * called until the mass edit action is complete.  For example, if there are 20 records to update
         * and the pageSize is 5, then this action will be called 3 times.  The first 5 are updated when
         * the actionMassEdit is called upon the initial form submission.
         */
        public function actionMassEditProgressSave()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'massEditProgressPageSize');
            $contact = new Contact(false);
            $dataProvider = $this->getDataProviderByResolvingSelectAllFromGet(
                            new ContactsSearchForm($contact),
                            $pageSize,
                            Yii::app()->user->userModel->id,
                            'ContactsStateMetadataAdapter',
                            'ContactsSearchView');
            $this->processMassEditProgressSave(
                'Contact',
                $pageSize,
                ContactsModule::getModuleLabelByTypeAndLanguage('Plural'),
                $dataProvider
            );
        }

        /**
         * Action for displaying a mass delete form and also action when that form is first submitted.
         * When the form is submitted, in the event that the quantity of models to delete is greater
         * than the pageSize, then once the pageSize quantity has been reached, the user will be
         * redirected to the makeMassDeleteProgressView.
         * In the mass delete progress view, a javascript refresh will take place that will call a refresh
         * action, usually makeMassDeleteProgressView.
         * If there is no need for a progress view, then a flash message will be added and the user will
         * be redirected to the list view for the model.  A flash message will appear providing information
         * on the delete records.
         * @see Controler->makeMassDeleteProgressView
         * @see Controller->processMassDelete
         * @see
         */
        public function actionMassDelete()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'massDeleteProgressPageSize');
            $contact = new Contact(false);

            $activeAttributes = $this->resolveActiveAttributesFromMassDeletePost();
            $dataProvider = $this->getDataProviderByResolvingSelectAllFromGet(
                            new ContactsSearchForm($contact),
                            $pageSize,
                            Yii::app()->user->userModel->id,
                            'ContactsStateMetadataAdapter',
                            'ContactsSearchView');
            $selectedRecordCount = static::getSelectedRecordCountByResolvingSelectAllFromGet($dataProvider);
            $contact = $this->processMassDelete(
                $pageSize,
                $activeAttributes,
                $selectedRecordCount,
                'ContactsPageView',
                $contact,
                ContactsModule::getModuleLabelByTypeAndLanguage('Plural'),
                $dataProvider
            );
            $massDeleteView = $this->makeMassDeleteView(
                $contact,
                $activeAttributes,
                $selectedRecordCount,
                ContactsModule::getModuleLabelByTypeAndLanguage('Plural')
            );
            $view = new ContactsPageView(ZurmoDefaultViewUtil::
                                         makeStandardViewForCurrentUser($this, $massDeleteView));
            echo $view->render();
        }

        /**
         * Action called in the event that the mass delete quantity is larger than the pageSize.
         * This action is called after the pageSize quantity has been delted and continues to be
         * called until the mass delete action is complete.  For example, if there are 20 records to delete
         * and the pageSize is 5, then this action will be called 3 times.  The first 5 are updated when
         * the actionMassDelete is called upon the initial form submission.
         */
        public function actionMassDeleteProgress()
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'massDeleteProgressPageSize');
            $contact = new Contact(false);
            $dataProvider = $this->getDataProviderByResolvingSelectAllFromGet(
                            new ContactsSearchForm($contact),
                            $pageSize,
                            Yii::app()->user->userModel->id,
                            'ContactsStateMetadataAdapter',
                            'ContactsSearchView');
            $this->processMassDeleteProgress(
                'Contact',
                $pageSize,
                ContactsModule::getModuleLabelByTypeAndLanguage('Plural'),
                $dataProvider
            );
        }

        public function actionModalList()
        {
            $modalListLinkProvider = new SelectFromRelatedEditModalListLinkProvider(
                                            $_GET['modalTransferInformation']['sourceIdFieldId'],
                                            $_GET['modalTransferInformation']['sourceNameFieldId'],
                                            $_GET['modalTransferInformation']['modalId']
            );
            echo ModalSearchListControllerUtil::setAjaxModeAndRenderModalSearchList($this, $modalListLinkProvider,
                                                'ContactsStateMetadataAdapter');
        }

        public function actionDelete($id)
        {
            $contact = Contact::GetById(intval($id));
            ControllerSecurityUtil::resolveAccessCanCurrentUserDeleteModel($contact);
            $contact->delete();
            $this->redirect(array($this->getId() . '/index'));
        }

        /**
         * Override to provide a contact specific label for the modal page title.
         * @see ZurmoModuleController->actionSelectFromRelatedList()
         */
        public function actionSelectFromRelatedList($portletId,
                                                    $uniqueLayoutId,
                                                    $relationAttributeName,
                                                    $relationModelId,
                                                    $relationModuleId,
                                                    $stateMetadataAdapterClassName = null)
        {
            parent::actionSelectFromRelatedList($portletId,
                                                $uniqueLayoutId,
                                                $relationAttributeName,
                                                $relationModelId,
                                                $relationModuleId,
                                                'ContactsStateMetadataAdapter');
        }

        /**
         * Override to always add contact state filter on search results.
         */
        public function actionAutoComplete($term, $autoCompleteOptions = null)
        {
            $pageSize = Yii::app()->pagination->resolveActiveForCurrentUserByType(
                            'autoCompleteListPageSize', get_class($this->getModule()));
            $autoCompleteResults = ContactAutoCompleteUtil::getByPartialName($term, $pageSize,
                                                                'ContactsStateMetadataAdapter', $autoCompleteOptions);
            echo CJSON::encode($autoCompleteResults);
        }

        public function actionGetAccountAddressesToCopy($id)
        {
            $account = static::getModelAndCatchNotFoundAndDisplayError('Account', intval($id));
            ControllerSecurityUtil::resolveAccessCanCurrentUserReadModel($account);
            $addressData = array();
            foreach ($account->billingAddress->getAttributeNames() as $attribute)
            {
                $addressData['billingAddress_' . $attribute] = $account->billingAddress->{$attribute};
            }
            foreach ($account->shippingAddress->getAttributeNames() as $attribute)
            {
                $addressData['shippingAddress_' . $attribute] = $account->shippingAddress->{$attribute};
            }
            echo CJSON::encode($addressData);
        }

        public function actionGetAccountOfficePhoneAndFaxToCopy($id)
        {
            $account = static::getModelAndCatchNotFoundAndDisplayError('Account', intval($id));
            ControllerSecurityUtil::resolveAccessCanCurrentUserReadModel($account);

            $data = array();
            if ($account->officePhone != null)
            {
                $data['officePhone'] = $account->officePhone;
            }

            if ($account->officeFax != null)
            {
                $data['officeFax'] = $account->officeFax;
            }
            echo CJSON::encode($data);
        }

        protected static function getSearchFormClassName()
        {
            return 'ContactsSearchForm';
        }

        public function actionExport()
        {
            $this->export('ContactsSearchView');
        }

        public function actionMassSubscribe()
        {
            $this->triggerMassAction('Contact',
                static::getSearchFormClassName(),
                'ContactsPageView',
                Contact::getModelLabelByTypeAndLanguage('Plural'),
                'ContactsSearchView',
                'ContactsStateMetadataAdapter',
                false);
        }

        public function actionMassSubscribeProgress()
        {
            $this->triggerMassAction('Contact',
                static::getSearchFormClassName(),
                'ContactsPageView',
                Contact::getModelLabelByTypeAndLanguage('Plural'),
                'ContactsSearchView',
                'ContactsStateMetadataAdapter',
                false);
        }

        protected static function resolveTitleByMassActionId($actionId)
        {
            if (MassActionUtil::isMassSubscribeLikeAction($actionId))
            {
                return Zurmo::t('Core', 'Mass Subscribe');
            }
            return parent::resolveTitleByMassActionId($actionId);
        }

        protected static function applyGenericViewIdGenerationRules($actionId)
        {
            return (MassActionUtil::isMassSubscribeLikeAction($actionId) || parent::applyGenericViewIdGenerationRules($actionId));
        }

        protected static function processModelForMassSubscribe(& $model)
        {
            $marketingListMember            = Yii::app()->request->getPost('MarketingListMember');
            if ($marketingListMember['marketingList']['id'] > 0)
            {
                $marketingList = MarketingList::getById((int) $marketingListMember['marketingList']['id']);
                $marketingList->addNewMember(null, false, $model);
                return true;
            }
        }

        protected static function resolveMassSubscribeAlertMessage($postVariableName)
        {
            $marketingListMember = Yii::app()->request->getPost('MarketingListMember');
            if (isset($marketingListMember) && $marketingListMember['marketingList']['id'] == 0)
            {
                return Zurmo::t('ContactsModule', 'You must select a MarketingListsModuleSingularLabel',
                                                   LabelUtil::getTranslationParamsForAllModules());
            }
        }

        /**
         * List view merge for contacts
         */
        public function actionListViewMerge()
        {
            $this->processListViewMerge('Contact',
                                        'ContactsListDuplicateMergedModelForm',
                                        'ContactsMerged', 'ContactsPageView',
                                        '/contacts/default/list');
        }

        /**
         * Overriding to implement the dedupe action for new contacts
         */
        public function actionSearchForDuplicateModels($attribute, $value)
        {
            assert('is_string($attribute)');
            assert('is_string($value)');
            $model          = new Contact();
            $depudeRules    = DedupeRulesFactory::createRulesByModel($model);
            $viewClassName  = $depudeRules->getDedupeViewClassName();
            $searchResult   = $depudeRules->searchForDuplicateModels($attribute, $value);
            if ($searchResult != null)
            {
                $summaryView    = new $viewClassName($this->id, $this->module->id, $model, $searchResult['matchedModels']);
                $content        = $summaryView->render();
                $message        = $searchResult['message'];
                echo CJSON::encode(array('content' => $content, 'message' => $message));
            }
        }

        /**
         * Create comment via ajax for contact
         * @param type $id
         * @param string $uniquePageId
         */
        public function actionInlineCreateCommentFromAjax($id, $uniquePageId)
        {
            $comment       = new Comment();
            $redirectUrl   = Yii::app()->createUrl('/contacts/default/inlineCreateCommentFromAjax',
                array('id'           => $id,
                      'uniquePageId' => $uniquePageId));
            $urlParameters = array('relatedModelId'           => (int)$id,
                                   'relatedModelClassName'    => 'Contact',
                                   'relatedModelRelationName' => 'comments',
                                   'redirectUrl'              => $redirectUrl); //After save, the url to go to.
            $uniquePageId  = 'CommentInlineEditForModelView';
            echo             ZurmoHtml::tag('h2', array(), Zurmo::t('CommentsModule', 'Add Comment'));
            $inlineView    = new CommentInlineEditView($comment, 'default', 'comments', 'inlineCreateSave',
                $urlParameters, $uniquePageId);
            $view          = new AjaxPageView($inlineView);
            echo $view->render();
        }

        /**
         * Add subscriber for contact
         * @param int $id
         */
        public function actionAddSubscriber($id)
        {
            $contact = Contact::getById((int)$id);
            $contact    = NotificationSubscriberUtil::processSubscriptionRequest($contact, Yii::app()->user->userModel);
            $content = NotificationSubscriberUtil::getSubscriberData($contact);
            $content .= NotificationSubscriberUtil::resolveSubscriptionLink($contact, 'detail-subscribe-model-link', 'detail-unsubscribe-model-link');
            echo $content;
        }

        /**
         * Remove subscriber for contact
         * @param int $id
         */
        public function actionRemoveSubscriber($id)
        {
            $contact = Contact::getById((int)$id);
            $contact    = NotificationSubscriberUtil::processUnsubscriptionRequest($contact, Yii::app()->user->userModel);
            $content = NotificationSubscriberUtil::getSubscriberData($contact);
            $content .= NotificationSubscriberUtil::resolveSubscriptionLink($contact, 'detail-subscribe-model-link', 'detail-unsubscribe-model-link');
            if ($content == null)
            {
                echo "";
            }
            else
            {
                echo $content;
            }
        }
    }
?>
