<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2014 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2014. All rights reserved".
     ********************************************************************************/

    require_once 'BaseUpgraderComponent.php';
    class UpgraderComponent extends BaseUpgraderComponent
    {
        protected $unsubscribeMethod            = null;

        protected $manageSubscriptionsMethod    = null;

        public function processBeforeUpdateSchema()
        {
            parent::processBeforeUpdateSchema();
            if ($this->shouldRunTasksByVersion('2.8.0'))
            {
                // change hash length for user field to 60, min and max both
                $this->adjustHashLengthForUserToSixty();
                // change user password to use new hashing scheme
                $this->updateUserPasswordsAgainstNewScheme();
                // change global footer against new merge tags for unsubscribe and manage subscriptions url
                //getContentByType then change stuff and then setContentByType for both types
                $this->updateGlobalMarketingFootersForUnsubscribeAndManageSubscriptionsUrlMergeTags();
                // update existing models for new merge tags for unsubscribe and manage subscriptions url
                $this->updateModelsTextAndHtmlContentForUnsubscribeAndManageSubscriptionsUrlMergeTags();
                // patch up contact metadata for indexes
                $this->updateContactMetadataForIndexes();
                // patch ProductEditAndDetailsView
                $this->updateProductEditAndDetailsViewMetadata();
            }
        }

        protected function adjustHashLengthForUserToSixty()
        {
            if (GlobalMetadata::isClassMetadataSavedInDatabase('User'))
            {
                $this->getMessageLogger()->addInfoMessage('Updating User Metadata for hash(32 -> 60 chars)');
                $metadata   = User::getMetadata();
                $rules      = $metadata['User']['rules'];
                foreach ($rules as $key => $rule)
                {
                    if ($rule[0] == 'hash' && $rule[1] == 'length')
                    {
                        $rules[$key]['min'] = 60;
                        $rules[$key]['max'] = 60;
                    }
                }
                $metadata['User']['rules'] = $rules;
                unset($metadata['Person']);
                User::setMetadata($metadata);
                $this->getMessageLogger()->addInfoMessage('User Metadata updated');
            }
            // ok, so we have a chicken egg problem
            // we need to update user password before updateSchema runs so that some of the models do not cry out for
            // owner, createdByUser, or modifiedByUser having a hash less than 60 characters.
            // Problem? beforeUpdateSchema hash column is still 32 chars, even if we change passwords they will be
            // truncated to 32 chars on save. updateSchema would have fixed the schema for user table but we can't
            // wait till then, so we went ahead and refreshed User's table manually.
            $this->getMessageLogger()->addInfoMessage('Refreshing _user table');
            $schema         = RedBeanModelToTableSchemaAdapter::resolve('User', $this->_messageLogger);
            CreateOrUpdateExistingTableFromSchemaDefinitionArrayUtil::generateOrUpdateTableBySchemaDefinition($schema, $this->_messageLogger, false);
            $this->getMessageLogger()->addInfoMessage('_user table refreshed');
        }

        protected function updateUserPasswordsAgainstNewScheme()
        {
            $this->getMessageLogger()->addInfoMessage('Updating user password against new encryption scheme');
            $phpassHashObject   = User::resolvePhpassHashObject();
            $users              = User::getAll();
            foreach ($users as $user)
            {
                $this->getMessageLogger()->addInfoMessage('Updated user password for: ' . $user->username);
                $this->updateUserPasswordAgainstNewScheme($user, $phpassHashObject);
            }
            $this->getMessageLogger()->addInfoMessage('All passwords updated against new encryption scheme');
        }

        protected function updateUserPasswordAgainstNewScheme(User $user, $phpassHashObject)
        {
            $user->hash = $phpassHashObject->hashPassword($user->hash);
            $this->saveModel($user, false);
        }

        protected function updateGlobalMarketingFootersForUnsubscribeAndManageSubscriptionsUrlMergeTags()
        {
            $this->getMessageLogger()->addInfoMessage('Updating globalMarketingFooters for new merge tags');
            $this->getMessageLogger()->addInfoMessage('Updating globalMarketingFooterPlainText for new merge tags');
            $this->updateGlobalMarketingFooter(false);
            $this->getMessageLogger()->addInfoMessage('Updating globalMarketingFooterRichText for new merge tags');
            $this->updateGlobalMarketingFooter(true);
            $this->getMessageLogger()->addInfoMessage('globalMarketingFooters updated');
        }

        protected function updateGlobalMarketingFooter($isHtmlContent)
        {
            $content        = GlobalMarketingFooterUtil::getContentByType($isHtmlContent, false);
            if (!empty($content))
            {
                $content = $this->updateContentForUnsubscribeAndManageSubscriptionsUrlMergeTags($content, $isHtmlContent);
                GlobalMarketingFooterUtil::setContentByType($content, $isHtmlContent);
            }
        }

        protected function updateContentForUnsubscribeAndManageSubscriptionsUrlMergeTags($content, $isHtmlContent)
        {
            $search         = array('{{UNSUBSCRIBE_URL}}', '{{MANAGE_SUBSCRIPTIONS_URL}}');
            $replace        = array($this->resolveUnsubscribeMergeTagReplacement($isHtmlContent),
                                    $this->resolveManageSubscriptionMergeTagReplacement($isHtmlContent));
            $content        = str_replace($search, $replace, $content);
            return $content;
        }

        protected function updateModelsTextAndHtmlContentForUnsubscribeAndManageSubscriptionsUrlMergeTags()
        {
            // update existing models for new merge tags for unsubscribe and manage subscriptions url
            $modelTypes         = array('EmailTemplate', 'Autoresponder', 'Campaign');
            foreach ($modelTypes as $modelType)
            {
                $this->updateTextAndHtmlContentForUnsubscribeAndManageSubscriptionsUrlMergeTagsForModelType($modelType);
            }
        }

        protected function updateTextAndHtmlContentForUnsubscribeAndManageSubscriptionsUrlMergeTagsForModelType($modelType)
        {
            $this->getMessageLogger()->addInfoMessage("Updating $modelType instance against new merge tags");
            $models     = $modelType::getAll();
            foreach ($models as $model)
            {
                $this->updateModelTextAndHtmlContentForUnsubscribeAndManageSubscriptionsUrlMergeTags($model);
            }
            $this->getMessageLogger()->addInfoMessage("$modelType instances updated for new merge tags");
        }

        protected function updateModelTextAndHtmlContentForUnsubscribeAndManageSubscriptionsUrlMergeTags(RedBeanModel $model)
        {
            $model->textContent = $this->updateContentForUnsubscribeAndManageSubscriptionsUrlMergeTags($model->textContent, false);
            $model->htmlContent = $this->updateContentForUnsubscribeAndManageSubscriptionsUrlMergeTags($model->htmlContent, true);
            $this->saveModel($model, false);
        }

        protected function resolveUnsubscribeMergeTagReplacement($isHtmlContent)
        {
            return $this->resolveDefaultUnsubscribeUrlMergeTagContentMethod()->invokeArgs(null, array($isHtmlContent));
        }

        protected function resolveDefaultUnsubscribeUrlMergeTagContentMethod()
        {
            if (!isset($this->unsubscribeMethod))
            {
                $this->unsubscribeMethod    = $this->getGlobalMarketingFooterUtilMethod('resolveDefaultUnsubscribeUrlMergeTagContent');
            }
            return $this->unsubscribeMethod;
        }

        protected function resolveManageSubscriptionMergeTagReplacement($isHtmlContent)
        {
            return $this->resolveDefaultManageSubscriptionsUrlMergeTagContentMethod()->invokeArgs(null, array($isHtmlContent));
        }

        protected function resolveDefaultManageSubscriptionsUrlMergeTagContentMethod()
        {
            if (!isset($this->manageSubscriptionsMethod))
            {
                $this->manageSubscriptionsMethod  = $this->getGlobalMarketingFooterUtilMethod('resolveDefaultManageSubscriptionsUrlMergeTagContent');
            }
            return $this->manageSubscriptionsMethod;
        }

        protected function getGlobalMarketingFooterUtilMethod($methodName)
        {
            return static::getProtectedMethod('GlobalMarketingFooterUtil', $methodName);
        }

        protected function updateContactMetadataForIndexes()
        {
            // this array is almost same as the metadata array in models except it used hardcoded indexes on which
            // an element should be placed. If an element exists at this index already, it would be pushed to next index.
            // Indices at which a member, rule, or noAudit attribute is listed does not matter, we just use hardcoded
            // integer values to make them appear in same order in db as they do in files. I didn't mention relations
            // here explicitly, because relations change requested in $metadataChanges always overwrite the existing
            // relation against that key.
            $this->getMessageLogger()->addInfoMessage('Patching Contact metadata for indexes');
            $metadataChanges = array(
                'indexes' => array(
                    'person_id' => array(
                        'members' => array('person_id'),
                        'unique' => false),
                ),
            );
            $this->updateModelMetadataWithNewMetaDefinition('Contact', $metadataChanges);
            $this->getMessageLogger()->addInfoMessage('Contact metadata patched');
        }

        protected function updateProductEditAndDetailsViewMetadata()
        {
            $className = 'ProductEditAndDetailsView';
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                $metadata = $className::getMetadata();
                $elements = array(
                    array('type'  => 'SaveButton',        'renderType' => 'Edit'),
                    array('type'  => 'CancelLink',        'renderType' => 'Edit'),
                    array('type'  => 'EditLink',          'renderType' => 'Details'),
                    array('type'  => 'ProductDeleteLink', 'renderType' => 'Details'),
                    array('type'  => 'CopyLink',          'renderType' => 'Details'),
                );
                $metadata['global']['toolbar']['elements'] = $elements;
                $className::setMetadata($metadata);
            }
        }

        protected function getModelClassNamesToPurgeGlobalMetadata()
        {
            // format: version => model class names to purge globalmetadata for.
            $models = array(
                '2.8.0' => array(
                    'ContactWebForm',
                    'EmailTemplate',
                    'ExportModule',
                    'ExportListView',
                    'GroupsModule',
                    'ImportModule',
                    'MarketingConfigurationEditAndDetailsView',
                    'MissionEditView',
                    'Person',
                    'RolesModule',
                    'TaskCheckListItem',
                    'UsersModule',
                ),
            );
            $twoDotEightDotOneModified  = array(
                'Autoresponder',
                'Campaign',
                'EmailTemplate',
                'ExportItem',
                'ExportModuleEditView', // wasn't modified in 2.8.1, it was in 2.8.0 but someone missing in above list
                'ExportSearchView', // wasn't modified in 2.8.1, it was in 2.8.0 but someone missing in above list
                'ProjectsForMixedModelsSearchListView', // wasn't modified in 2.8.1, it was in 2.8.0 but someone missing in above list
                'ZurmoConfigurationEditAndDetailsView',
                'ZurmoSystemConfigurationEditAndDetailsView', // wasn't modified in 2.8.1, it was in 2.8.0 but someone missing in above list
            );
            $twoDotEightDotThreeModified  = array(
                'MarketingListMember',
            );
            if (Yii::app()->edition == 'Commercial')
            {
                $models['2.8.0'][] = 'QueueModel';
                $models['2.8.5'][] = 'QueueModel';
                $models['2.8.5'][] = 'UserGoogleAppsConfigurationEditAndDetailsView';
            }
            // re-purging metadata for 2.8.0 files just to be sure.
            // We should never need to do this but in this case lets just do it to avoid surprises.
            $models['2.8.1']  = CMap::mergeArray($models['2.8.0'], $twoDotEightDotOneModified);
            $models['2.8.3']  = CMap::mergeArray($models['2.8.1'], $twoDotEightDotThreeModified);
            return $models;
        }
    }
?>