<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2014 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2014. All rights reserved".
     ********************************************************************************/

    class BaseUpgraderComponent extends CApplicationComponent
    {
        protected $_messageLogger = null;

        protected $_zurmoVersionBeforeUpgrade = null;

        /**
         * Overriding inti to setup _zurmoVersionBeforeUpgrade
         */
        public function init()
        {
            parent::init();
            $this->_zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
        }

        /**
         * @return null|MessageLogger
         */
        public function getMessageLogger()
        {
            return $this->_messageLogger;
        }

        /**
         * @param $messageLogger
         */
        public function setMessageLogger($messageLogger)
        {
            $this->_messageLogger = $messageLogger;
        }

        /**
         * Tasks to do before start processing config files.
         */
        public function processBeforeConfigFiles()
        {
            $this->messageLogger->addInfoMessage("Process before config files.");
        }

        /**
         * Process config files.
         * Problem here is that some config files might be modified, so there
         * can be some additional config options. We need to modify config
         * files(debug.php, debugTest.php, perInstance.php, perInstanceTest.php),
         * without loosing settings entered by user, so in most cases we will add
         * content at the end of those files where possible.
         * Should be used to files like debug.php, debugTest.php, perInstance.php, perInstanceTest.php,
         * which are not tracked by SCM tools, other files are updated automatically.
         *
         * @code
            <?php
                //Below is just sample code to add $myVariable to perInstance.php and perInstanceTest.php files.
                if ($this->shouldRunTasksByVersion('0.6.80'))
                {
                    $perInstanceTestFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstanceTest.php';
                    if (is_file($perInstanceTestFile))
                    {
                        $perInstanceTestContent = file_get_contents($perInstanceTestFile);

                        $contentToAdd = "\$myVariable = 'aaa';\n";
                        $perInstanceTestContent = preg_replace('/\?\>/',
                                                               "\n" . $contentToAdd . "\n" . "?>",
                                                               $perInstanceTestContent);
                        file_put_contents($perInstanceTestFile, $perInstanceTestContent);
                    }
                    $perInstanceFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstance.php';
                    if (is_file($perInstanceFile))
                    {
                        $perInstanceContent = file_get_contents($perInstanceFile);

                        $contentToAdd = "\$myVariable = 'aaa';\n";
                        $perInstanceContent = preg_replace('/\?\>/',
                                                           "\n" . $contentToAdd . "\n" . "?>",
                                                           $perInstanceContent);
                        file_put_contents($perInstanceFile, $perInstanceContent);
                    }
                }
            ?>
         * @endcode
         */
        public function processConfigFiles($pathToConfigurationFolder)
        {
        }

        /**
         * Tasks after config files are processed.
         */
        public function processAfterConfigFiles()
        {
        }

        /**
         * Tasks to do before start processing files.
         */
        public function processBeforeFiles()
        {
        }

        /**
         * Process files
         * @param string $source
         * @param string $destination
         * @param array $configuration
         */
        public function processFiles($source, $destination, $configuration)
        {
            // Remove files that are marked for removal in manifest.php.
            if (isset($configuration['removedFiles']) && !empty($configuration['removedFiles']))
            {
                foreach ($configuration['removedFiles'] as $fileOrFolderToRemove)
                {
                    // Replace Lunux directory separators(from upgrade manifest file)
                    // with one used by system(DIRECTORY_SEPARATOR)
                    $fileOrFolderToRemove = str_replace(array('/', '\\'), DIRECTORY_SEPARATOR, $fileOrFolderToRemove);
                    $fileOrFolderToRemove = $destination . DIRECTORY_SEPARATOR . trim($fileOrFolderToRemove, DIRECTORY_SEPARATOR);
                    if (is_dir($fileOrFolderToRemove))
                    {
                        FileUtil::deleteDirectoryRecursive($fileOrFolderToRemove);
                    }
                    elseif (is_file($fileOrFolderToRemove))
                    {
                        unlink($fileOrFolderToRemove);
                    }
                }
            }
            // Copy new or modified files.
            FileUtil::copyRecursive($source, $destination);
        }

        /**
         * Tasks after files are processed.
         */
        public function processAfterFiles()
        {
        }

        /**
         * Tasks that should be executed before updating schema.
         * By default metadata for any classes specified in @see getModelClassNamesToPurgeGlobalMetadata
         * and default metadata for few specific keys for ZurmoModule is loaded into db.
         *
         * @code
            <?php
                if ($this->shouldRunTasksByVersion('0.6.80'))
                {
                    $metadata = AccountsModule::getMetadata();
                    if (!isset($metadata['global']['newElement']))
                    {
                        $metadata['global']['newElement'] = 'Some Content';
                        AccountsModule::setMetadata($metadata);
                    }
                    GeneralCache::forgetAll();
                }
            ?>
         * @endcode
         */
        public function processBeforeUpdateSchema()
        {
            $this->purgeGlobalMetadataForModelClassNamesNotEditableInDesigner();
            $this->resetDefaultDataForZurmoModule();
        }

        /**
         * Update schema.
         */
        public function processUpdateSchema($messageLogger)
        {
            // We must update schema via command line, to ensure that newly
            // copied files are loaded into application correctly.
            InstallUtil::runAutoBuildFromUpdateSchemaCommand($messageLogger);
            // do we want to overwrite existing read tables?
            ReadPermissionsOptimizationUtil::rebuild($this->doOverwriteExistingReadTables());
        }

        /**
         * Tasks to be executed after update schema
         */
        public function processAfterUpdateSchema()
        {
        }

        /**
         * Clear assets and runtime folders.
         */
        public function clearAssetsAndRunTimeItems()
        {
            $pathToAssetsFolder = INSTANCE_ROOT . DIRECTORY_SEPARATOR . 'assets';
            FileUtil::deleteDirectoryRecursive($pathToAssetsFolder, false, array('index.html'));

            $pathToRuntimeFolder = Yii::app()->getRuntimePath();
            FileUtil::deleteDirectoryRecursive($pathToRuntimeFolder, false, array('index.html', 'upgrade'));
        }

        /**
         * Some final tasks can be added here.
         */
        public function processFinalTouches()
        {
        }

        /**
         * Determine if some task should be executed or not, depending on Zurmo version.
         * @param string $upgradeVersion - version in which change is added
         * @param string $currentZurmoVersion
         * @return boolean
         */
        protected function shouldRunTasksByVersion($upgradeVersion, $currentZurmoVersion = null)
        {
            // Replace non-digit characters from beginning of file(because pro.2.6.1)
            $upgradeVersion = preg_replace("/^\D*/i", "", $upgradeVersion);
            if ($currentZurmoVersion)
            {
                $currentZurmoVersion = preg_replace("/^\D*/i", "", $this->_zurmoVersionBeforeUpgrade);
            }
            $shouldRun = false;
            if (version_compare($currentZurmoVersion, $upgradeVersion, '<='))
            {
                $shouldRun = true;
            }
            return $shouldRun;
        }

        /**
         * @param array $metadata
         * @param array $elementToFind
         * @return boolean|multitype:array
         */
        protected function findPositionOfElementInViewPanels($metadata, $elementToFind)
        {
            if (!isset($metadata['global']['panels']) || !is_array($metadata['global']['panels']))
            {
                return false;
            }

            foreach ($metadata['global']['panels'] as $panelKey => $panels)
            {
                if (!isset($panels['rows']) || !is_array($panels['rows']))
                {
                    continue;
                }
                foreach ($panels['rows'] as $rowKey => $rows)
                {
                    if (!isset($rows['cells']) || !is_array($rows['cells']))
                    {
                        continue;
                    }
                    foreach ($rows['cells'] as $cellKey => $cells)
                    {
                        if (!isset($cells['elements']) || !is_array($cells['elements']))
                        {
                            continue;
                        }
                        foreach ($cells['elements'] as $elementKey => $elements)
                        {
                            $found = false;
                            foreach ($elementToFind as $key => $value)
                            {
                                if (isset($elements[$key]) && $elements[$key] == $value)
                                {
                                    $found = true;
                                }
                                else
                                {
                                    $found = false;
                                }
                            }
                            if ($found)
                            {
                                $result = array(
                                    'panelKey'   => $panelKey,
                                    'rowKey'     => $rowKey,
                                    'cellKey'    => $cellKey,
                                    'elementKey' => $elementKey
                                );
                                return $result;
                            }
                        }
                    }
                }
            }
            return false;
        }

        /**
         * @param array $metadata
         * @param string $type
         * @param array $elementToFind
         * @throws Exception
         * @return boolean|multitype:array
         */
        protected function findPositionOfElementInViewToolbarsAndRowMenu($metadata, $type, $elementToFind)
        {
            if ($type !== 'toolbar' && $type !== 'rowMenu')
            {
                throw new Exception();
            }
            if (!isset($metadata['global'][$type]['elements']) || !is_array($metadata['global'][$type]['elements']))
            {
                return false;
            }

            foreach ($metadata['global'][$type]['elements'] as $elementKey => $elements)
            {
                $found = false;
                foreach ($elementToFind as $key => $value)
                {
                    if (isset($elements[$key]) && $elements[$key] == $value)
                    {
                        $found = true;
                    }
                    else
                    {
                        $found = false;
                    }
                }
                if ($found)
                {
                    $result = array(
                        'elementKey' => $elementKey
                    );
                    return $result;
                }
            }
            return false;
        }

        /**
         * Returns array of Classes for which metadata changed but we can safely purge the metadata saved in db
         * @return array
         */
        protected function getModelClassNamesToPurgeGlobalMetadata()
        {
            // format: version => model class names to purge globalmetadata for.
            return array();
        }

        /**
         * Purge global metadata for specific classes
         */
        protected function purgeGlobalMetadataForModelClassNamesNotEditableInDesigner()
        {
            $versionsWithModelClassNames = $this->getModelClassNamesToPurgeGlobalMetadata();
            foreach ($versionsWithModelClassNames as $version => $modelClassNames)
            {
                // only purge we were on a previous version than the one specific in $versionsWithModelClassNames
                if ($this->shouldRunTasksByVersion($version))
                {
                    $this->purgeGlobalMetadataForModelClassNames($modelClassNames);
                }
            }
        }

        /**
         * Purges global metadata for the class names provided
         * @param $modelClassNames
         */
        protected function purgeGlobalMetadataForModelClassNames(array $modelClassNames)
        {
            $purgeQueries = null;
            $tableName = 'globalmetadata'; //RedBeanModel::getTableName('GlobalMetadata')
            $quote = DatabaseCompatibilityUtil::getQuote();
            foreach ($modelClassNames as $modelClassName)
            {
                $purgeQueries .= $this->getPurgeQueryForGlobalMetadataForModelClassName($modelClassName, $tableName, $quote);
            }
            echo $purgeQueries;
            if (!empty($purgeQueries))
            {
                ZurmoRedBean::exec($purgeQueries);
            }
        }

        /**
         * Return query to purge globalmetata for provided modelclassname
         * @param $modelClassName
         * @param $tableName
         * @param $quote
         * @return string
         */
        protected function getPurgeQueryForGlobalMetadataForModelClassName($modelClassName, $tableName, $quote)
        {
            // if the class's metadata was saved in db then we return the query to purge its metadata, else null
            if (GlobalMetadata::isClassMetadataSavedInDatabase($modelClassName))
            {
                return "delete from ${quote}${tableName}${quote} where " .
                            "${quote}classname${quote}='${modelClassName}';" . PHP_EOL;
            }
        }

        /**
         * Reset default data for ZurmoModule's metadata for specific keys
         */
        protected function resetDefaultDataForZurmoModule()
        {
            // if ZurmoModule's metadata was saved in db, we want to reset few items to
            // their latest default values
            if (GlobalMetadata::isClassMetadataSavedInDatabase('ZurmoModule'))
            {
                $defaultKeys            = array('configureMenuItems',
                                                'headerMenuItems',
                                                'configureSubMenuItems',
                                                'adminTabMenuItemsModuleOrdering',
                                                'tabMenuItemsModuleOrdering');
                $metadataFromDatabase   = ZurmoModule::getMetadata();
                $defaultMetadata        = ZurmoModule::getDefaultMetadata();
                foreach ($defaultKeys as $defaultKey)
                {
                    $metadataFromDatabase['global'][$defaultKey] = $defaultMetadata['global'][$defaultKey];
                }
                ZurmoModule::setMetadata($metadataFromDatabase);
            }
        }

        /**
         * @return bool
         */
        protected function doOverwriteExistingReadTables()
        {
            return false;
        }

        /**
         * Delete lines from a file matching any of specific patterns. Matches are done using stripos.
         * @param $filePath
         * @param array $patterns
         * @throws Exception
         */
        protected function deleteLinesInFile($filePath, array $patterns)
        {
            $inputLines = file($filePath);
            if ($inputLines === false)
            {
                throw new Exception("Unable to open $filePath for reading");
            }
            $inputLinesCounts = count($inputLines);
            $outputLines = array();
            // we intentionally do not use foreach to allow loop to determine how much to skip.
            for ($index = 0; $index < $inputLinesCounts; $index++)
            {
                $addToOutput = !$this->deleteLineFromFile($inputLines, $patterns, $index);
                if ($addToOutput)
                {
                    $outputLines[] = $inputLines[$index];
                }
            }
            if (!file_put_contents($filePath, $outputLines))
            {
                throw new Exception("Unable to update $filePath");
            }
        }

        /**
         * Detemine if the line at current index matches of any of linesToBeDeleted patterns
         * @param array $inputLines
         * @param array $patternsToBeDeleted
         * @param $index
         * @return bool
         */
        protected function deleteLineFromFile(array $inputLines, array $patternsToBeDeleted, & $index)
        {
            // $index is sent to determine how much we want to skip, we could increment it in
            // currentLineMatchesLineToBeDeleted by count of lines to skip besides the current one
            $currentLine = $inputLines[$index];
            foreach ($patternsToBeDeleted as $pattern)
            {
                if ($this->currentLineMatchesLineToBeDeleted($currentLine, $pattern, $index))
                {
                    return true;
                }
            }
            return false;
        }

        /**
         * Return true|false depending on whether currentLine matched pattern
         * @param $currentLine
         * @param $pattern
         * @param $index
         * @return bool
         */
        protected function currentLineMatchesLineToBeDeleted($currentLine, $pattern, & $index)
        {
            return (stripos($currentLine, $pattern) !== false);
        }

        /**
         * Provided a model class name and metachanges array, apply changes to the metadata saved in db.
         * @param $className
         * @param array $metadataChanges
         */
        protected function updateModelMetadataWithNewMetaDefinition($className, array $metadataChanges)
        {
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                // list of keys we can handle change requested enclosed in metadataChanges array
                $validMetadataChangesKeys   = array('members', 'rules', 'relations', 'elements', 'noAudit');
                $metadata                   = $className::getMetadata();
                $ownMetadata                = $metadata[$className];
                // loop through valid keys one by one
                foreach ($validMetadataChangesKeys as $key)
                {
                    // this shouldn't happen, i mean why would i declared metadataChanges for member but no specifications?
                    // still, time has proven me wrong when date changes and i still decide to finish up some code.
                    if (!isset($metadataChanges[$key]))
                    {
                        continue;
                    }
                    // get the change requests for current key.
                    $metadataChangesForCurrentKey   = $metadataChanges[$key];
                    // initialized own metadata for that key to an empty array
                    $ownMetadataForCurrentKey       = array();
                    if (isset($ownMetadata[$key]))
                    {
                        // oh, wait, we found metadata for the key in question, lets grab it from there.
                        $ownMetadataForCurrentKey       = $ownMetadata[$key];
                    }
                    // looping through change requests one by one, $position tells us where
                    // to add and $newItem is what to add
                    foreach ($metadataChangesForCurrentKey as $position => $newItem)
                    {
                        // cool, add it into array.
                        $this->insertIntoArray($ownMetadataForCurrentKey, $newItem, $position);
                    }
                    // assigned it back to $ownMetadata, we could check if $ownMetadataForCurrentKey is not empty
                    // and only assigned it back then but not worth the effort. if it was empty, nature wanted it
                    // that way.
                    $ownMetadata[$key]              = $ownMetadataForCurrentKey;
                }
                // assigned ownMetadata back, we could check if ownMetadata was modified at all but again, not worth it.
                // plus, we know that it was modified, we never sent empty $metadataChanges
                $metadata[$className] = $ownMetadata;
                $className::setMetadata($metadata);
            }
        }

        /**
         * Push an element at a specific position in an array.
         * If its integer position, then push any element on provided position to next index and place provided
         * element at position.
         * If its string key, say its for relations, then just replace the old value
         * @param array $array
         * @param $element
         * @param $position
         */
        protected function insertIntoArray(array & $array, $element, $position)
        {
            if (is_int($position))
            {
                $array = CMap::mergeArray(array_slice($array, 0, $position), array($element), array_slice($array, $position));
            }
            else
            {
                $array[$position] = $element;
            }
        }
    }
?>