<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2013 Zurmo Inc.
     *
     * Zurmo is free software; you can redistribute it and/or modify it under
     * the terms of the GNU Affero General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * Zurmo is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
     * details.
     *
     * You should have received a copy of the GNU Affero General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 27 North Wacker Drive
     * Suite 370 Chicago, IL 60606. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in original and modified versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU Affero General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the Zurmo
     * logo and Zurmo copyright notice. If the display of the logo is not reasonably
     * feasible for technical reasons, the Appropriate Legal Notices must display the words
     * "Copyright Zurmo Inc. 2013. All rights reserved".
     ********************************************************************************/

    require_once 'BaseUpgraderComponent.php';

    // TODO: @Shoaibi: Critical: Add type hinting
    // TODO: @Shoaibi: Critical: find unused functions
    // TODO: @Shoaibi: Critical: Coding violations
    class UpgraderComponent extends BaseUpgraderComponent
    {
        /**
         * Process any config file changes
         * @param $pathToConfigurationFolder
         */
        public function processConfigFiles($pathToConfigurationFolder)
        {
            parent::processConfigFiles($pathToConfigurationFolder);
            // FORCE_NO_FREEZE and relevant configs are gone due to removal of no-freeze mode
            // from, almost, everywhere.
            $this->removeForceNoFreezeFromDebugConfigs($pathToConfigurationFolder);
        }

        /**
         * Carry out any metadata changes tasks
         */
        public function processBeforeUpdateSchema()
        {
            parent::processBeforeUpdateSchema();
            if ($this->shouldRunTasksByVersion('2.2.6'))
            {
                // this was added in 2.2.6, if user's version is 2.2.6 or 2.2.7 we don't want to re-add it.
                // Add the delete and Copy link introduced in 2.2.6
                $this->appendNewElementsToProductTemplateEditAndDetailsView();
            }
            if ($this->shouldRunTasksByVersion('2.5.0'))
            {
                // old starred tables were like: account_starred, new ones are accountstarred.
                // rename old ones
                $this->renameStarredTables();

                // rename model_id in *starred tables to relevant foreign key, say for accountstarred
                // it will rename model_id to account_id. This is useful because the new starred
                // table use different column names and we want to preserve the data while adhering to
                // new styles
                $this->renameModelIdToRelevantForeignKeysForStarredTables();

                // previously we didn't use any auto_increment column for _read tables and instead had a composite
                // key as primary key. In enw scheme we use composite key as candidate key and add a new
                // auto_increment field to be used as primary
                $this->updateReadTablesWithNewPrimaryKey();

                // globalSearchAttributeNames was added to Model's metadata
                $this->addGlobalSearchAttributeNamesToProductMetadata();
                $this->addGlobalSearchAttributeNamesToMeetingsModuleMetadata();
                $this->addGlobalSearchAttributeNamesToTasksModuleMetadata();

                // automaticProbabilityMappingDisabled was added to Model
                $this->addAutomaticProbabilityMappingDisabledForOpportunitiesModule();

                // shortcutsCreateMenuItems was added for Tasks Module
                $this->addShortcutsCreateMenuItemsForTasksModule();

                // Previously we only supported partial html template by not allowing html, head, title, body and
                // few other tags. In 2.5.0 we have added support for all those back and by default
                // templates are supposed to contain that. Its ok even if they don't as editing and saving them
                // from UI will add that(due to redactor's doing) but better do it manually
                $this->resolveHtmlContentForEmailTemplatesAndAutorespondersAndCampaigns();

                // Mininum length for name attribute was changed from 3 to 1 for few models
                $this->changeMinimumNameLengthToOneForOpportunityAndProductCatalogAndProductAndProductTemplate();

                // update the label for SaveButton
                $this->updateAccountConvertToViewMetadata();

                // We changed the toolbar and rowMenu elements for children of OpenTasksRelatedListView class
                $this->updateElementsAndRowMenuForOpenTasksRelatedListViewChildren();

                // Add the newly introduced latestActivityDateTime attribute to few models along with its rules
                // also add the new projects relation
                $this->updateAccountMetadataForLatestActivityDateTimeAttributeAndRelations();
                $this->updateContactMetadataForLatestActivityDateTimeAttributeAndRelations();

                // Add the newly introduced logged attribute for meeting along with rules
                $this->updateMeetingMetadataForLoggedAttribute();

                // Add the projects relation for opportunity model
                $this->updateOpportunityMetadataForProjectsRelation();

                // Add member and rules for status attribute and newly introduced relations
                $this->updateTaskMetadataForStatusAttributeAndNewRelations();
            }

            if ($this->shouldRunTasksByVersion('2.5.3'))
            {
                // Make Task status field required
                $this->updateTaskMetadataForStatusAttributeMakeRequired();
            }
        }

        /**
         * Carry out any tasks that need to be done after updateSchema, say renaming columns, migrating data...
         */
        public function processAfterUpdateSchema()
        {
            parent::processAfterUpdateSchema();
            if ($this->shouldRunTasksByVersion('2.5.0'))
            {
                // Status column needs to be either Completed or New depending on few criteria. By default it
                // would resolve to null, which would break few things.
                $this->fixTaskStatusForCompletedAndCompletedDateTime();

                // These models had duplicate FK, one with the link name and another without
                // With the new autobuild we favor the columns with the link name
                // so we need to migrate data from the other column(the one without link name)
                // and delete it
                $this->fixDuplicateForeignColumnsForContactFormEntryAndEmailMessage();

                // personOrAccount used to be a has_one, now its many_many. We need to migrate data
                // to the join table and then drop the extra column from table
                $this->fixPersonOrAccountForEmailMessageRecipientsAndSenders();

                // adding few gamification stuff which would need couple of views' metadata to be re-generated
                $this->refreshPortletsForGamification();

                // account_starred are now accountstarred, but they still have user_id
                // we need to migrate data to the newer basestarredmodel table and drop user_id
                $this->migrateStarredDataAndUpdateColumns();

                // naming convention of indices in legacy autobuild and autobuildv2 is different.
                // while we can keep the old indices in place as the autobuildv2 detects them and does not
                // create duplicate indices, we just as well go ahead and change those to the enw scheme.
                $this->updateIndexes();
            }
            if ($this->shouldRunTasksByVersion('2.5.3') && Yii::app()->edition == 'Commercial')
            {
                $this->setPerUserMetadataForGoogleAppsUpgrade();
                $this->saveGoogleAppsContactGroupName();
            }
        }

        /**
         * Remove lines related to FORCE_NO_FREEZE from debug.php and debugTest.php
         * @param $pathToConfigurationFolder
         */
        protected function removeForceNoFreezeFromDebugConfigs($pathToConfigurationFolder)
        {
            // FORCE_NO_FREEZE is no longer relevant now that we have removed unfrozen code from, almost, everywhere
            // do it for debug.php and debugTest.php
            $patternsToBeDeleted = array(
                '// Turn this on to use the application with the database unfrozen',
                'forceNoFreeze',
                'FORCE_NO_FREEZE',
            );
            $fileNames = array('debug.php', 'debugTest.php');
            foreach ($fileNames as $fileName)
            {
                $this->deletePatternFromFile($fileName, $pathToConfigurationFolder, $patternsToBeDeleted);
            }
        }

        /**
         * Take a filename, a path and an array of patterns that should be deleted from file's contents
         * @param $fileName
         * @param $pathToConfigurationFolder
         * @param array $patternsToBeDeleted
         */
        protected function deletePatternFromFile($fileName, $pathToConfigurationFolder, array $patternsToBeDeleted)
        {
            // generate the actual path
            $filePath = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . $fileName;
            if (file_exists($filePath))
            {
                $this->deleteLinesInFile($filePath, $patternsToBeDeleted);
            }
            // file does not exist? don't sweat it, it would be regenerated from new DIST files
        }

        /**
         * Return true/false depending on whether the current line should be deleted or not.
         * @param $currentLine
         * @param $pattern
         * @param $index
         * @return bool
         */
        protected function currentLineMatchesLineToBeDeleted($currentLine, $pattern, & $index)
        {
            if (stripos($pattern, '// Turn this on to use the application with the database unfrozen'))
            {
                // in the config file we have:
                // "// Turn this on to use the application with the database unfrozen"
                // and then on the next line we have:
                // "// Check it in as true"
                // We want to get rid of the later one too. We can't use it as a direct pattern because
                // exact same string is used elsewhere within same config so we would just increment
                // the index that loops through lines. Incrementing index would make loop skip the next line
                // from even checked if it should be added to output or not.
                ++$index;
            }
            return parent::currentLineMatchesLineToBeDeleted($currentLine, $pattern, $index);
        }

        /**
         * List Classes for which metadata changed but we can safely purge the metadata saved in db
         * @return array
         */
        protected function getModelClassNamesToPurgeGlobalMetadata()
        {
            // format: version => model class names to purge globalmetadata for.
            return array('2.5.0' => array(
                                            'Autoresponder',
                                            'BounceConfigurationEditAndDetailsView',
                                            'ByTimeWorkflowInQueuesSearchView',
                                            'Campaign',
                                            'CampaignDetailsView',
                                            'CampaignEditView',
                                            'ConfigurationModule',
                                            'ContactWebForm',
                                            'ContactWebFormEditAndDetailsView',
                                            'ContactWebFormEntry',
                                            'Conversation',
                                            'DesignerModule',
                                            'EmailArchivingConfigurationEditAndDetailsView',
                                            'EmailMessageEditAndDetailsView',
                                            'EmailTemplate',
                                            'HomeModule',
                                            'ImportModule',
                                            'MapConfigurationView',
                                            'MarketingList',
                                            'MarketingListDetailsView',
                                            'MarketingModule',
                                            'OpportunitiesModuleEditView',
                                            'Person',
                                            'ProductCategory',
                                            'ReportDetailsView',
                                            'ReportsModule',
                                            'WorkflowMessageInQueuesSearchView',
                                            'ZurmoConfigurationEditAndDetailsView'),
                           '2.5.3' => array(
                                            'ActionBarForUserEditAndDetailsView'),
                );
        }

        /**
         * Append Delete and CopyLink to ProductTemplateEditAndDetailsView
         */
        protected function appendNewElementsToProductTemplateEditAndDetailsView()
        {
            // if its saved in db, then update, else it would be generated from file anyway.
            if (GlobalMetadata::isClassMetadataSavedInDatabase('ProductTemplateEditAndDetailsView'))
            {
                $metadata = ProductTemplateEditAndDetailsView::getMetadata();
                $elements = $metadata['global']['toolbar']['elements'];
                $elements[] = array('type' => 'ProductTemplateDeleteLink',
                                                                        'renderType' => 'Details');
                $elements[] = array('type' => 'CopyLink', 'renderType' => 'Details');
                $metadata['global']['toolbar']['elements'] = $elements;
                ProductTemplateEditAndDetailsView::setMetadata($metadata);
            }
        }

        /**
         * Starred table names changed from account_starred to accountstarred. Take care of renaming these tables
         * before updateSchema takes place so we don't have account_starred from previous install and a blank
         * accountstarred generated by updateSchema
         */
        protected function renameStarredTables()
        {
            // we dynamically get the available _starred tables and rename only those.
            $this->dropPartiallyUpgradedStarredTables();
            $oldTableNames = $this->getStarredTableNames(true);
            $query = $this->buildRenameQueryForOldStarredTableNames($oldTableNames);
            ZurmoRedBean::exec($query);
        }

        /**
         * Drop accountstarred, contactstarred, conversationstarred, opportunitystarred from
         * last partial upgrade
         */
        protected function dropPartiallyUpgradedStarredTables()
        {
            $tableNames     = $this->getStarredTableNames(false);
            $this->dropTables($tableNames);
        }

        /**
         * Drop table(s).
         * @param $tableNames
         */
        protected function dropTables($tableNames)
        {
            if (empty($tableNames))
            {
                return;
            }
            if (!is_array($tableNames))
            {
                $tableNames = array($tableNames);
            }
            $tableNames = implode(',', $tableNames);
            $query      = $this->getDropTableQuery($tableNames);
            ZurmoRedBean::exec($query);
        }

        /**
         * Resolve query to drop a table.
         * @param $tableName
         * @param bool $ifExists
         * @return string
         */
        protected function getDropTableQuery($tableName, $ifExists = true)
        {
            $query      = 'DROP TABLE ';
            if ($ifExists)
            {
                $query  .= 'IF EXISTS ';
            }
            $query      .= $tableName;
            $query      .= ';';
            return $query;
        }

        /**
         * Get names of starred tables.
         * @param bool $OldStyle
         * @return array
         */
        protected function getStarredTableNames($OldStyle = false)
        {
            $pattern = $this->resolveStarredTableNamePattern($OldStyle);
            $tableNames = $this->getTableNamesLikePattern($pattern);
            if (!$OldStyle)
            {
                $tableNames = array_filter($tableNames, array($this, 'filterOldStyleStarredTableNames'));
            }
            return $tableNames;
        }

        /**
         * Return false if table name matched old starred table name pattern
         * @param $tableName
         * @return bool
         */
        protected function filterOldStyleStarredTableNames($tableName)
        {
            return (strpos($tableName, '_starred') == false);
        }

        /**
         * Get table names that match a certain pattern, case sensitive
         * @param $pattern
         * @return array
         */
        protected function getTableNamesLikePattern($pattern)
        {
            $query = $this->getTableNamesLikePatternQuery($pattern);
            $tableNames = ZurmoRedBean::getCol($query);
            return $tableNames;
        }

        /**
         * Get pattern used to starred tables
         * @param bool $OldStyle
         * @return string
         */
        protected function resolveStarredTableNamePattern($OldStyle = false)
        {
            $pattern = "starred";
            if ($OldStyle)
            {
                $pattern = '\_' . $pattern;
            }
            $pattern = '%' . $pattern;
            return $pattern;
        }

        /**
         * Get a query that would return table names matching a pattern, case sensitive
         * @param $pattern
         * @return string
         */
        protected function getTableNamesLikePatternQuery($pattern)
        {
            $query  = "show tables LIKE '${pattern}';";
            return $query;
        }

        /**
         * Given old style starred table names, build a query to rename those to the new style.
         * @param $oldTableNames
         * @return string
         */
        protected function buildRenameQueryForOldStarredTableNames($oldTableNames)
        {
            // generate one big query of rename table instead of multiple.
            $query = "rename table ";
            $tableQuery = array();
            foreach ($oldTableNames as $oldTableName)
            {
                // loop through tables, get new starred table name, append to the query array
                $newTableName = $this->resolveNewStarredTableNameFromOldTableName($oldTableName);
                $tableQuery[] = "$oldTableName to $newTableName";
            }
            // join query array on ,
            // at this point tableQuery would have something like "account_starred to accountstarred, contact_starred to contactstarred"
            $tableQuery = implode(', ', $tableQuery);
            // add the semicolon, not that it matters but i like it that way.
            $query .= $tableQuery . ';';
            return $query;
        }

        /**
         * Given old style starred table name, resolve it into new style and return string
         * @param $oldTableName
         * @return mixed
         */
        protected function resolveNewStarredTableNameFromOldTableName($oldTableName)
        {
            // not much, old style: account_starred, new style: accountstarred.
            // There weren't any names with double underscores so
            // i didn't did a more stricter replace
            return str_replace('_', '', $oldTableName);
        }

        /**
         * Given a new style starred table name, resolve it into old styled one
         * @param $newTableName
         * @return mixed
         */
        protected function resolveOldStarredTableNameFromNewTableName($newTableName)
        {
            // not much, old style: account_starred, new style: accountstarred
            $tableName = str_replace('starred', '_starred', $newTableName);
            return $tableName;
        }

        /**
         * Given new style starred table name, resolve the relevant model class.
         * @param $tableName
         * @return string
         */
        protected function resolveStarredModelClassNameFromNewStarredTableName($tableName)
        {
            // convert table name to old style. accountstarred becomes account_starred
            $oldTableName = $this->resolveOldStarredTableNameFromNewTableName($tableName);
            // now we camelize it and also capitilize first letter, this would give us AccountStarred
            $className    = StringUtil::camelize($oldTableName, true);
            return $className;
        }

        /**
         * Rename legacy model_id column to the new scheme. For accountstarred model_id becomes
         * account_id, for contactstarred it becomes contact_id and so on.
         */
        protected function renameModelIdToRelevantForeignKeysForStarredTables()
        {
            $starredTables = $this->getStarredTableNames(false);
            foreach ($starredTables as $starredTable)
            {
                $this->renameModelIdToRelevantForeignKeyForStarredTableName($starredTable);
            }
        }

        /**
         * Rename model_id for the given tableName to relevant foreign key's new name
         * @param $starredTableName
         */
        protected function renameModelIdToRelevantForeignKeyForStarredTableName($starredTableName)
        {
            // extract the relatedModelName, for accountstarred, this will be account
            $relatedModelName   = str_replace('starred', '', $starredTableName);
            // resolve starredClassName, for accountstarred, this will be AccountStarred
            $starredClassName   = $this->resolveStarredModelClassNameFromNewStarredTableName($starredTableName);
            // get the foreign key name
            $newForeignKey      = RedBeanModel::getForeignKeyName($starredClassName, $relatedModelName);
            // we know the old foreign key name already
            $oldForeignKey      = 'model_id';
            // its a foreign key, so int(11), unsigned and null. shouldn't be null perhaps but we allow all but PK
            // columns to be null in db and control inputs inside code. @see RedBeanModelMemberRulesToColumnAdapter.191
            $type               = 'int(11) unsigned null';
            $query              = $this->getRenameColumnQuery($starredTableName, $oldForeignKey, $newForeignKey, $type);
            ZurmoRedBean::exec($query);
        }

        /**
         * Provide tableName, oldColumnName, newColumnName and type, generate an alter query and return
         * @param $tableName
         * @param $oldColumnName
         * @param $newColumnName
         * @param $type
         * @return string
         */
        protected function getRenameColumnQuery($tableName, $oldColumnName, $newColumnName, $type)
        {
            $tableName  = ZurmoRedBean::$writer->safeTable($tableName);
            $query      = "alter table $tableName change $oldColumnName $newColumnName $type;";
            return $query;
        }

        /**
         * Migrate data from accountstarred.user_id to new starred models following
         * basestarredmodel scheme and drop user_id from starred tables, not from basestarredmodel of course.
         */
        protected function migrateStarredDataAndUpdateColumns()
        {
            $newStarredTableNames = $this->getStarredTableNames(false);
            // loop through them
            foreach ($newStarredTableNames as $newStarredTableName)
            {
                $this->migrateUserIdDataFromStarredToBaseStarredTable($newStarredTableName);
                $this->dropUserIdFromStarredTable($newStarredTableName);
            }
        }

        /**
         * Migrate data for user_id for provided starred table to basestarredmodel table
         * @param $newStarredTableName
         */
        protected function migrateUserIdDataFromStarredToBaseStarredTable($starredTableName)
        {
            $records            = ZurmoRedBean::findAll($starredTableName);
            foreach ($records as $record)
            {
                $this->migrateUserIdDataByTableNameAndRecord($starredTableName, $record);
            }
        }

        /**
         * Migrate specific starred record's user id to basestarredmodel table
         * @param $starredTableName
         * @param $record
         */
        protected function migrateUserIdDataByTableNameAndRecord($starredTableName, $record)
        {
            $baseStarredModelId = $this->copyUserIdFromStarredTableToBaseStarredModel($record->user_id);
            $this->updateStarredTableRecordBaseStarredModelIdKey($starredTableName, $record->id, $baseStarredModelId);
        }

        /**
         * Provided a userId, create a new basestarredmodel record and return newly created record's id
         * @param $userId
         * @return int
         * @throws FailedToSaveModelException
         */
        protected function copyUserIdFromStarredTableToBaseStarredModel($userId)
        {
            $baseStarredModel       = new BaseStarredModel();
            $baseStarredModel->user = User::getById($userId);
            if (!$baseStarredModel->save())
            {
                throw new FailedToSaveModelException("Unable to migrate userId data from starred table to basestarredmodel table");
            }
            return intval($baseStarredModel->id);
        }

        /**
         * Update current starred table record's basestarredmodel_id with the provided baseStarredModelId
         * @param $starredTableName
         * @param $recordId
         * @param $baseStarredModelId
         */
        protected function updateStarredTableRecordBaseStarredModelIdKey($starredTableName, $recordId, $baseStarredModelId)
        {
            $query      = $this->getUpdateStarredTableRecordBaseStarredModelIdKey($starredTableName,
                                                                                    $recordId,
                                                                                    $baseStarredModelId);
            ZurmoRedBean::exec($query);
        }

        /**
         * Generate query to update basestarredmodel_id for starredTableName's record with id of recordId
         * @param $starredTableName
         * @param $recordId
         * @param $baseStarredModelId
         * @return string
         */
        protected function getUpdateStarredTableRecordBaseStarredModelIdKey($starredTableName,
                                                                            $recordId,
                                                                            $baseStarredModelId)
        {
            $starredTableName   = ZurmoRedBean::$writer->safeTable($starredTableName);
            $query              = "update $starredTableName set basestarredmodel_id = $baseStarredModelId ";
            $query              .= "where id = $recordId;";
            return $query;
        }

        /**
         * Drop legacy user_id column from starred table
         * @param $newStarredTableName
         */
        protected function dropUserIdFromStarredTable($newStarredTableName)
        {
            $columnToBeDropped      = 'user_id';
            $newStarredTableName    = ZurmoRedBean::$writer->safeTable($newStarredTableName);
            $query                  = $this->getDropColumnQuery($newStarredTableName, $columnToBeDropped);
            ZurmoRedBean::exec($query);
        }

        /**
         * Resolve query to drop provided column from provided table
         * @param $safeTableName
         * @param $columnName
         * @return string
         */
        protected function getDropColumnQuery($safeTableName, $columnName)
        {
            $query  = "alter table $safeTableName drop $columnName;";
            return $query;
        }

        /**
         * Previous _read tables used to use a composite key as primary key. With 2.5.0 these tables
         * got an id auto_increment column which is used as primary key while the composite key
         * is left in place as candidate key
         */
        protected function updateReadTablesWithNewPrimaryKey()
        {
            $readTableNames = $this->getReadTableNames();
            foreach ($readTableNames as $readTableName)
            {
                $this->updateReadTableWithNewPrimaryKey($readTableName);
            }
        }

        /**
         * Change primary key for provided read table name
         * @param $readTableName
         */
        protected function updateReadTableWithNewPrimaryKey($readTableName)
        {
            $query  = $this->getUpdateReadTableWithNewPrimaryKeyQuery($readTableName);
            ZurmoRedBean::exec($query);
        }

        /**
         * Resolve a query to update primary key for provided read table name
         * @param $readTableName
         * @return string
         */
        protected function getUpdateReadTableWithNewPrimaryKeyQuery($readTableName)
        {
            // first, drop the previous primary key, we are dropping the index, not the column(s)
            $query = $this->getDropPrimaryKeyQuery($readTableName);
            // second, reset auto_increment count, this is just for aesthetics
            $query .= $this->getResetAutoIncrementQuery($readTableName);
            // third, add the id column and setup primary key on id column
            $query .= $this->getAddIdColumnQuery($readTableName, true);
            return $query;
        }

        /**
         * Return an array containing all _read table names
         * @return array
         */
        protected function getReadTableNames()
        {
            $pattern    = $this->resolveReadTableNamePattern();
            $tableNames = $this->getTableNamesLikePattern($pattern);
            return $tableNames;
        }

        /**
         * Return pattern tableNames must satisfy to be qualified as a _read table
         * @return string
         */
        protected function resolveReadTableNamePattern()
        {
            return '%_read';
        }

        /**
         * Resolve query to add id auto_increment column to provided table name, setting $withPrimaryKey
         * to true would also add query to setup primary key on newly added column.
         * @param $safeTableName
         * @param bool $withPrimaryKey
         * @return string
         */
        protected function getAddIdColumnQuery($safeTableName, $withPrimaryKey = true)
        {
            $query      = "alter table $safeTableName add `id` int(11) unsigned not null";
            $query      .= " comment '' auto_increment first";
            if ($withPrimaryKey)
            {
                $query .= ", add primary key (`id`)";
            }
            $query      .= ";";
            return $query;
        }

        /**
         * Return query to drop the primary key index from provided table name
         * @param $safeTableName
         * @return string
         */
        protected function getDropPrimaryKeyQuery($safeTableName)
        {
            $query  = $this->getDropColumnQuery($safeTableName, 'primary key');
            return $query;
        }

        /**
         * Return query to reset auto_increment count. Default reset value for auto_increment is 1, unless
         * otherwise provided as second argument
         * @param $safeTableName
         * @param int $newAutoIncrementValue
         * @return string
         */
        protected function getResetAutoIncrementQuery($safeTableName, $newAutoIncrementValue = 1)
        {
            $query      = "alter table $safeTableName AUTO_INCREMENT = $newAutoIncrementValue;";
            return $query;
        }

        /**
         * Resolve htmlContent field content to full/proper html content including html, head and body nodes
         */
        protected function resolveHtmlContentForEmailTemplatesAndAutorespondersAndCampaigns()
        {
            $classesWithHtmlContentAttribute = array('EmailTemplate', 'Autoresponder', 'Campaign');
            foreach ($classesWithHtmlContentAttribute as $classWithHtmlContentAttribute)
            {
                $this->resolveHtmlContentForClass($classWithHtmlContentAttribute);
            }
        }

        /**
         * Resolve htmlContent content to proper html containg html, head and body nodes for provided modelClassName
         * @param $modelClassName
         */
        protected function resolveHtmlContentForClass($modelClassName)
        {
            $models = $modelClassName::getAll();
            foreach ($models as $model)
            {
                $this->resolveHtmlContentForModel($model);
            }
        }

        /**
         * Resolve htmlContent content to proper html containg html, head and body nodes for provided model instance
         * @param $model
         * @throws FailedToSaveModelException
         */
        protected function resolveHtmlContentForModel( & $model)
        {
            if ($this->needsHtmlContentUpdate($model->htmlContent))
            {
                $this->populateProperHtmlContent($model);
                if (!$model->save())
                {
                    $class  = get_class($mode);
                    $id     = $model->id;
                    echo "ERR: Unable to update htmlContent for ${class}.${id}";
                }
            }
        }

        /**
         * Return boolean depending on if htmlContent needs to be padded with html, head and body nodes.
         * @param $htmlContent
         * @return bool
         */
        protected function needsHtmlContentUpdate($htmlContent)
        {
            $htmlContent = trim($htmlContent);
            // by default every htmlContent field will require padding update
            $needsUpdate = true;
            if (!empty($htmlContent))
            {
                $newTags = array('<html>', '<head>', '<body>');
                // if any of the tags about to padded are already in htmlContent then it doesn't needs padding update
                foreach ($newTags as $tag)
                {
                    if (strpos($htmlContent, $tag))
                    {
                        $needsUpdate = false;
                        break;
                    }
                }
            }
            return $needsUpdate;
        }

        /**
         * Sandwitch provided model's htmlContent in html, head and body nodes.
         * @param $model
         */
        protected function populateProperHtmlContent(& $model)
        {
            $htmlContent = <<<HTM
<html>
<head>
</head>
<body>
$model->htmlContent
</body>
</html>
HTM;
            $model->htmlContent = $htmlContent;
        }

        /**
         * Delete few portlet entries so the gamification dashboard appears correctly
         */
        protected function refreshPortletsForGamification()
        {
            $query = $this->getQueryToRefreshPortletsForGamification();
            ZurmoRedBean::exec($query);
        }

        /**
         * Resolve query to delete approperiate porlet entries to make gamification dashboard vsisible
         * @return string
         */
        protected function getQueryToRefreshPortletsForGamification()
        {
            $portletTable   = RedBeanModel::getTableName('Portlet');
            $portletTable   = ZurmoRedBean::$writer->safeTable($portletTable);
            $query          = "delete from ${portletTable} where layoutid = ";
            $query          .= "'UserDetailsAndRelationsViewLeftBottomView' or layoutid = ";
            $query          .= "'UserDetailsAndRelationsViewRightBottomView';";
            return $query;
        }

        /**
         * personOrAccount used to be has_one. In 2.5.0 its many_many.
         * Fix it for existing records, port any data and drop the legacy has_one column in relevant tables.
         */
        protected function fixPersonOrAccountForEmailMessageRecipientsAndSenders()
        {
            $modelClassNames = array('EmailMessageSender', 'EmailMessageRecipient');
            foreach ($modelClassNames as $className)
            {
                $this->patchPersonOrAccountForClass($className);
            }
        }

        /**
         * Patch personOrAccount change for provided model class name
         * @param $className
         */
        protected function patchPersonOrAccountForClass($className)
        {
            $tableName  = RedBeanModel::getTableName($className);
            $query      = $this->getPatchPersonOrAccountForTableNameQuery($tableName);
            ZurmoRedBean::exec($query);
        }

        /**
         * Return query to patch personOrAccount relation for provided tableName
         * @param $tableName
         * @return string
         */
        protected function getPatchPersonOrAccountForTableNameQuery($tableName)
        {
            $query  = $this->getMigratePersonOrAccountDataForTableNameQuery($tableName);
            $query  .= $this->getDropPersonOrAccountItemIdQuery($tableName);
            return $query;
        }

        /**
         * Return query to clone personaccount_item_id data from provided tableName to the many many table.
         * @param $tableName
         * @param string $personOrAccountId
         * @return string
         */
        protected function getMigratePersonOrAccountDataForTableNameQuery($tableName, $personOrAccountId = 'personoraccount_item_id')
        {
            $ownId              = $tableName . '_id';
            $itemTableName      = $tableName . '_item';
            $itemTableName      = ZurmoRedBean::$writer->safeTable($itemTableName);
            $safeTableName      = ZurmoRedBean::$writer->safeTable($tableName);
            $query              = "insert into $itemTableName($ownId, item_id) select id, $personOrAccountId from ";
            $query              .= "$safeTableName;";
            return $query;
        }

        /**
         * Return query to drop the legacy has_one column, personoraccount_id, in provided tableName
         * @param $tableName
         * @param string $personOrAccountId
         * @return string
         */
        protected function getDropPersonOrAccountItemIdQuery($tableName, $personOrAccountId = 'personoraccount_item_id')
        {
            $safeTableName  = ZurmoRedBean::$writer->safeTable($tableName);
            $query          = $this->getDropColumnQuery($safeTableName, $personOrAccountId);
            return $query;
        }

        /**
         * In legacy autobuild for few classes we have duplicate foreign keys, one with link/relation name and one without.
         * New autobuild favors foreign key columns with the prefix of relation/link name.
         * Remove the deprecated foreign key column after cloning data
         */
        protected function fixDuplicateForeignColumnsForContactFormEntryAndEmailMessage()
        {
            $contactWebFormEntryTableName   = RedBeanModel::getTableName('ContactWebFormEntry');
            $emailMessageTableName          = RedbeanModel::getTableName('EmailMessage');
            // mapping describing table names along with the column thats to be dropped and the column
            // we can backup value of dropped-column to.
            $mapping = array(
                $contactWebFormEntryTableName   => array('drop'     => 'contactwebform_id',
                                                        'copyTo'    => 'entries_contactwebform_id'),
                $emailMessageTableName          => array('drop'     => 'emailaccount_id',
                                                        'copyTo'    => 'account_emailaccount_id'),
            );
            foreach ($mapping as $tableName => $renameMapping)
            {
                $this->fixDuplicateForeignColumnsForTableName($tableName, $renameMapping['drop'], $renameMapping['copyTo']);
            }
        }

        /**
         * Clone data from cloneToBeDropped to columnToBackupValuesTo for provided tableName and drop columnToBeDropped.
         * @param $tableName
         * @param $columnToBeDropped
         * @param $columnToBackupValuesTo
         */
        protected function fixDuplicateForeignColumnsForTableName($tableName, $columnToBeDropped, $columnToBackupValuesTo)
        {
            $tableName  = ZurmoRedBean::$writer->safeTable($tableName);
            $query      = $this->getQueryToMigrateDuplicateColumnData($tableName, $columnToBeDropped, $columnToBackupValuesTo);
            $query      .= $this->getDropColumnQuery($tableName, $columnToBeDropped);
            ZurmoRedBean::exec($query);
        }

        /**
         * Return query to clone data from columnToBeDropped to columnToBackupValuesTo where required
         * @param $tableName
         * @param $columnToBeDropped
         * @param $columnToBackupValuesTo
         * @return string
         */
        protected function getQueryToMigrateDuplicateColumnData($tableName, $columnToBeDropped, $columnToBackupValuesTo)
        {
            $query  = "update $tableName set $columnToBackupValuesTo = $columnToBeDropped where ";
            $query  .= "$columnToBackupValuesTo is null and $columnToBeDropped is not null;";
            return $query;
        }

        /**
         * In 2.5.0 we change minimum length for name attribute for few class from 3 to 1.
         * Patch that in the metadata saved in database for those classes.
         */
        protected function changeMinimumNameLengthToOneForOpportunityAndProductCatalogAndProductAndProductTemplate()
        {
            $modelClassNames    = array('Opportunity', 'ProductCatalog', 'Product', 'ProductTemplate');
            foreach ($modelClassNames as $modelClassName)
            {
                $this->changeMinimumNameLengthToOneForClass($modelClassName);
            }
        }

        /**
         * Patch the in-db metadata for minimum length change for name attribute for provided modelClassName
         * @param $modelClassName
         */
        protected function changeMinimumNameLengthToOneForClass($modelClassName)
        {
            if (GlobalMetadata::isClassMetadataSavedInDatabase($modelClassName))
            {
                $metadata   = $modelClassName::getMetadata();
                $rules      = $metadata[$modelClassName]['rules'];
                foreach ($rules as $key => $rule)
                {
                    if ($rule[0] == 'name' && $rule[1] == 'length')
                    {
                        $rules[$key]['min'] = 1;
                    }
                }
                $metadata[$modelClassName]['rules'] = $rules;
                $modelClassName::setMetadata($metadata);
            }
        }

        /**
         * Add the newly introduced globalSearchAttributeNames to Product's in-db metadata
         */
        protected function addGlobalSearchAttributeNamesToProductMetadata()
        {
            $this->addGlobalSearchAttributeNamesToMetadataForClass('Product');
        }

        /**
         * Add the newly introduced globalSearchAttributeNames to TasksModule's in-db metadata
         */
        protected function addGlobalSearchAttributeNamesToTasksModuleMetadata()
        {
            $this->addGlobalSearchAttributeNamesToMetadataForClass('TasksModule');
        }

        /**
         * Add the newly introduced globalSearchAttributeNames to MeetingsModule's in-db metadata
         */
        protected function addGlobalSearchAttributeNamesToMeetingsModuleMetadata()
        {
            $this->addGlobalSearchAttributeNamesToMetadataForClass('MeetingsModule');
        }

        /**
         * Patch the in-db metadata for provided class name for globalSearchAttributeNames
         * @param $className
         */
        protected function addGlobalSearchAttributeNamesToMetadataForClass($className)
        {
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                $metadata   = $className::getMetadata();
                $key        = $className;
                if (is_subclass_of($className, 'Module'))
                {
                    // if its a module subclass then it wouldn't have $metadata[$className] but $metadata['global']
                    $key = 'global';
                }
                $metadata[$key]['globalSearchAttributeNames'] = array('name');
                $className::setMetadata($metadata);
            }
        }

        /**
         * Add the newly introduced automaticProbabilityMappingDisabled to OpportunitiesModule's in-db metadata
         */
        protected function addAutomaticProbabilityMappingDisabledForOpportunitiesModule()
        {
            $className = 'OpportunitiesModule';
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                $metadata = $className::getMetadata();
                $metadata['global']['automaticProbabilityMappingDisabled'] = false;
                $className::setMetadata($metadata);
            }
        }

        /**
         * Add the newly introduced shortcutsCreateMenuItems to TasksModule's in-db metadata
         */
        protected function addShortcutsCreateMenuItemsForTasksModule()
        {
            $className = 'TasksModule';
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                $metadata = $className::getMetadata();
                $metadata['global']['shortcutsCreateMenuItems'] = array(
                    array(
                        'label'  => "eval:Zurmo::t('TasksModule', 'TasksModuleSingularLabel', \$translationParams)",
                        'url'    => Yii::app()->createUrl('tasks/default/modalCreate',
                                                            array('modalId' => TasksUtil::getModalContainerId())),
                        'ajaxLinkOptions' => "TasksUtil::resolveAjaxOptionsForCreateMenuItem()",
                        'right'  => TasksModule::RIGHT_CREATE_TASKS,
                        'mobile' => true,
                    ),
                );
                $className::setMetadata($metadata);
            }
        }

        /**
         * Patch in-db AccountConvertToView's metadata for SaveButton's new label
         */
        protected function updateAccountConvertToViewMetadata()
        {
            $className = 'AccountConvertToView';
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                $metadata = $className::getMetadata();
                $elements = $metadata['global']['toolbar']['elements'];
                foreach ($elements as $key => $element)
                {
                    if ($element['type']  == 'SaveButton')
                    {
                        $elements[$key]['label'] = "eval:Zurmo::t('ZurmoModule', 'Complete Conversion')";
                    }
                }
                $metadata['global']['toolbar']['elements'] = $elements;
                $className::setMetadata($metadata);
            }
        }

        /**
         * Patch in-db metata for children of OpenTasksRelatedListView class to accommodate toolbar and rowMenu
         * element changes
         */
        protected function updateElementsAndRowMenuForOpenTasksRelatedListViewChildren()
        {
            // we know, for a fact, that these are the only classes that extend OpenTasksRelatedListView in 2.5.0
            $classNames    = array('OpenTasksForAccountRelatedListView',
                                    'OpenTasksForContactRelatedListView',
                                    'OpenTasksForOpportunityRelatedListView');
            foreach ($classNames as $className)
            {
                $this->updateElementsAndRowMenuForOpenTasksRelatedListViewChildrenClass($className);
            }
        }

        /**
         * Patch in-db metadata for provided className to accommodate modifcations to toolbar and rowMenu elements
         * @param $className
         */
        protected function updateElementsAndRowMenuForOpenTasksRelatedListViewChildrenClass($className)
        {
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                $metadata = $className::getMetadata();
                // lets start by patching toolbar elements first
                $toolbarElements = $metadata['global']['toolbar']['elements'];
                foreach ($toolbarElements as $key => $toolbarElement)
                {
                    if ($toolbarElement['type'] == 'CreateFromRelatedListLink')
                    {
                        // great, we found the only element we needed to patch, generate element definition.
                        $toolbarElements[$key] =  $this->getElementForOpenTasksRelatedListViewChildrenClasses(
                                                                                        'CreateFromRelatedModalLink',
                                                                                        'Create');
                    }
                }
                $metadata['global']['toolbar']['elements'] = $toolbarElements;
                // turn of rowMenu elements
                $rowMenuElements = $metadata['global']['rowMenu']['elements'];
                foreach ($rowMenuElements as $key => $rowMenuElement)
                {
                    if (in_array($rowMenuElement['type'], array('EditLink', 'CopyLink')))
                    {
                        // extract prefix, this could be Edit or Copy
                        $prefix                 = substr($rowMenuElement['type'], 0, 4);
                        // generate new type based on prefix
                        $newType                = $prefix . 'ModalLink';
                        // generate element definition using element type and renderType/prefix
                        $rowMenuElements[$key]  = $this->getElementForOpenTasksRelatedListViewChildrenClasses($newType,
                                                                                                            $prefix);
                    }
                }
                $metadata['global']['rowMenu']['elements'] = $rowMenuElements;
                $className::setMetadata($metadata);
            }
        }

        /**
         * Provided elementType and renderType, return an array containing element definition for toolbar or
         * rowMenu's element for children of OpenTasksRelatedListView
         * @param $elementType
         * @param $renderType
         * @return array
         */
        protected function getElementForOpenTasksRelatedListViewChildrenClasses($elementType, $renderType)
        {
            $element = array(
                'type'              => $elementType,
                'portletId'         => 'eval:$this->params["portletId"]',
                'routeModuleId'     => 'eval:$this->moduleId',
                'routeParameters'   => 'eval:$this->getCreateLinkRouteParameters()',
                'ajaxOptions'       => 'eval:TasksUtil::resolveAjaxOptionsForModalView("' . $renderType . '")',
                'uniqueLayoutId'    => 'eval:$this->uniqueLayoutId',
                'modalContainerId'  => 'eval:TasksUtil::getModalContainerId()'
            );
            return $element;
        }

        /**
         * Patch in-db Account's metadata for latestActivityDateTime attribute and Projects relationship.
         */
        protected function updateAccountMetadataForLatestActivityDateTimeAttributeAndRelations()
        {
            // this array is almost same as the metadata array in models except it used hardcoded indexes on which
            // an element should be placed. If an element exists at this index already, it would be pushed to next index.
            // Indices at which a member, rule, or noAudit attribute is listed does not matter, we just use hardcoded
            // integer values to make them appear in same order in db as they do in files. I didn't mention relations
            // here explicitly, because relations change requested in $metadataChanges always overwrite the existing
            // relation against that key.
            $metadataChanges = array(
                'members' => array(
                    3   => 'latestActivityDateTime',
                ),
                'rules' => array(
                    3   => array('latestActivityDateTime',  'readOnly'),
                    4   => array('latestActivityDateTime',  'type', 'type' => 'datetime'),
                ),
                'relations' => array(
                    'projects' => array(RedBeanModel::MANY_MANY, 'Project'),
                ),
                'elements' => array(
                    'latestActivityDateTime'  => 'DateTime',
                ),
                'noAudit' => array(
                    3   => 'latestActivityDateTime',
                ),
            );
            $this->updateModelMetadataWithNewMetaDefinition('Account', $metadataChanges);
        }

        /**
         * Patch in-db metadata of Contact for latestActivityDateTime and Projects relationship
         */
        protected function updateContactMetadataForLatestActivityDateTimeAttributeAndRelations()
        {
            // this array is almost same as the metadata array in models except it used hardcoded indexes on which
            // an element should be placed. If an element exists at this index already, it would be pushed to next index.
            // Indices at which a member, rule, or noAudit attribute is listed does not matter, we just use hardcoded
            // integer values to make them appear in same order in db as they do in files. I didn't mention relations
            // here explicitly, because relations change requested in $metadataChanges always overwrite the existing
            // relation against that key.
            $metadataChanges = array(
                'members' => array(
                    2   => 'latestActivityDateTime',
                ),
                'rules' => array(
                    3   => array('latestActivityDateTime',  'readOnly'),
                    4   => array('latestActivityDateTime',  'type', 'type' => 'datetime'),
                ),
                'relations' => array(
                    'projects' => array(RedBeanModel::MANY_MANY, 'Project'),
                ),
                'elements' => array(
                    'latestActivityDateTime'  => 'DateTime'
                ),
                'noAudit' => array(
                    1   => 'latestActivityDateTime',
                ),
            );
            $this->updateModelMetadataWithNewMetaDefinition('Contact', $metadataChanges);
        }

        /**
         * Patch in-db metadata of Meeting model for newly added logged attribute
         */
        protected function updateMeetingMetadataForLoggedAttribute()
        {
            // this array is almost same as the metadata array in models except it used hardcoded indexes on which
            // an element should be placed. If an element exists at this index already, it would be pushed to next index.
            // Indices at which a member, rule, or noAudit attribute is listed does not matter, we just use hardcoded
            // integer values to make them appear in same order in db as they do in files. I didn't mention relations
            // here explicitly, because relations change requested in $metadataChanges always overwrite the existing
            // relation against that key.
            $metadataChanges = array(
                'members' => array(
                    3   => 'logged',
                ),
                'rules' => array(
                    5   => array('logged', 'boolean'),
                ),
            );
            $this->updateModelMetadataWithNewMetaDefinition('Meeting', $metadataChanges);
        }

        /**
         * Patch in-db Opportunity metadata for Projects relationship
         */
        protected function updateOpportunityMetadataForProjectsRelation()
        {
            // this array is almost same as the metadata array in models except it used hardcoded indexes on which
            // an element should be placed. If an element exists at this index already, it would be pushed to next index.
            // Indices at which a member, rule, or noAudit attribute is listed does not matter, we just use hardcoded
            // integer values to make them appear in same order in db as they do in files. I didn't mention relations
            // here explicitly, because relations change requested in $metadataChanges always overwrite the existing
            // relation against that key.
            $metadataChanges = array(
                'relations' => array(
                    'projects' => array(RedBeanModel::MANY_MANY, 'Project'),
                ),
            );
            $this->updateModelMetadataWithNewMetaDefinition('Opportunity', $metadataChanges);
        }

        /**
         * Patch in-db Task metadata for status attribute, new relations and elements
         */
        protected function updateTaskMetadataForStatusAttributeAndNewRelations()
        {
            // this array is almost same as the metadata array in models except it used hardcoded indexes on which
            // an element should be placed. If an element exists at this index already, it would be pushed to next index.
            // Indices at which a member, rule, or noAudit attribute is listed does not matter, we just use hardcoded
            // integer values to make them appear in same order in db as they do in files. I didn't mention relations
            // here explicitly, because relations change requested in $metadataChanges always overwrite the existing
            // relation against that key.
            $metadataChanges = array(
                'members' => array(
                    5   => 'status',
                ),
                'rules' => array(
                    7   => array('status', 'type', 'type' => 'integer'),
                    8   => array('status', 'default', 'value' => Task::STATUS_NEW),
                ),
                'relations' => array(
                    'requestedByUser'           => array(RedBeanModel::HAS_ONE, 'User', RedBeanModel::NOT_OWNED,
                                                            RedBeanModel::LINK_TYPE_SPECIFIC, 'requestedByUser'),
                    'comments'                  => array(RedBeanModel::HAS_MANY, 'Comment', RedBeanModel::OWNED,
                                                            RedBeanModel::LINK_TYPE_POLYMORPHIC, 'relatedModel'),
                    'checkListItems'            => array(RedBeanModel::HAS_MANY, 'TaskCheckListItem', RedBeanModel::OWNED),
                    'notificationSubscribers'   => array(RedBeanModel::HAS_MANY, 'NotificationSubscriber', RedBeanModel::OWNED),
                    'files'                     => array(RedBeanModel::HAS_MANY, 'FileModel', RedBeanModel::OWNED,
                                                            RedBeanModel::LINK_TYPE_POLYMORPHIC, 'relatedModel'),
                    'project'                   => array(RedBeanModel::HAS_ONE, 'Project'),
                ),
                'elements' => array(
                    'requestedByUser'   => 'User',
                    'comment'           => 'Comment',
                    'checkListItem'     => 'TaskCheckListItem',
                    'files'             => 'Files',
                    'project'           => 'Project',
                    'status'            => 'TaskStatusDropDown'
                ),
            );
            $this->updateModelMetadataWithNewMetaDefinition('Task', $metadataChanges);
        }

        // Update Task metadata, make status member required
        protected function updateTaskMetadataForStatusAttributeMakeRequired()
        {
            $metadataChanges = array(
                'rules' => array(
                    9   => array('status', 'required'),
                ),
            );
            $this->updateModelMetadataWithNewMetaDefinition('Task', $metadataChanges);
        }

        /**
         * Provided a model class name and metachanges array, apply changes to the metadata saved in db.
         * @param $className
         * @param array $metadataChanges
         */
        protected function updateModelMetadataWithNewMetaDefinition($className, array $metadataChanges)
        {
            if (GlobalMetadata::isClassMetadataSavedInDatabase($className))
            {
                // list of keys we can handle change requested enclosed in metadataChanges array
                $validMetadataChangesKeys   = array('members', 'rules', 'relations', 'elements', 'noAudit');
                $metadata                   = $className::getMetadata();
                $ownMetadata                = $metadata[$className];
                // loop through valid keys one by one
                foreach ($validMetadataChangesKeys as $key)
                {
                    // this shouldn't happen, i mean why would i declared metadataChanges for member but no specifications?
                    // still, time has proven me wrong when date changes and i still decide to finish up some code.
                    if (!isset($metadataChanges[$key]))
                    {
                        continue;
                    }
                    // get the change requests for current key.
                    $metadataChangesForCurrentKey   = $metadataChanges[$key];
                    // initialized own metadata for that key to an empty array
                    $ownMetadataForCurrentKey       = array();
                    if (isset($ownMetadata[$key]))
                    {
                        // oh, wait, we found metadata for the key in question, lets grab it from there.
                        $ownMetadataForCurrentKey       = $ownMetadata[$key];
                    }
                    // looping through change requests one by one, $position tells us where
                    // to add and $newItem is what to add
                    foreach ($metadataChangesForCurrentKey as $position => $newItem)
                    {
                        // cool, add it into array.
                        $this->insertIntoArray($ownMetadataForCurrentKey, $newItem, $position);
                    }
                    // assigned it back to $ownMetadata, we could check if $ownMetadataForCurrentKey is not empty
                    // and only assigned it back then but not worth the effort. if it was empty, nature wanted it
                    // that way.
                    $ownMetadata[$key]              = $ownMetadataForCurrentKey;
                }
                // assigned ownMetadata back, we could check if ownMetadata was modified at all but again, not worth it.
                // plus, we know that it was modified, we never sent empty $metadataChanges
                $metadata[$className] = $ownMetadata;
                $className::setMetadata($metadata);
            }
        }

        /**
         * Push an element at a specific position in an array.
         * If its integer position, then push any element on provided position to next index and place provided
         * element at position.
         * If its string key, say its for relations, then just replace the old value
         * @param array $array
         * @param $element
         * @param $position
         */
        protected function insertIntoArray(array & $array, $element, $position)
        {
            if (is_int($position))
            {
                $array = CMap::mergeArray(array_slice($array, 0, $position), array($element), array_slice($array, $position));
            }
            else
            {
                $array[$position] = $element;
            }
        }

        /**
         * Projects were release bundled in 2.5.0 and they need for tasks to have verbose status.
         * Status column was added to Task model in 2..50 so here we set values for status table to some
         * sane alternates than just the default(NULL) database opts it to be.
         */
        protected function fixTaskStatusForCompletedAndCompletedDateTime()
        {
            $completedStatus    = Task::STATUS_COMPLETED;
            $tableName          = RedBeanModel::getTableName('Task');
            $tableName          = ZurmoRedBean::$writer->safeTable($tableName);
            $query              = $this->getTaskStatusSetToCompletedQuery($tableName, $completedStatus);
            $query              .= $this->getTaskStatusSetToNewQuery($tableName, $completedStatus);
            ZurmoRedBean::exec($query);
        }

        /**
         * Return query to update Task status to complete where required
         * @param $tableName
         * @param $completedStatus
         * @return string
         */
        protected function getTaskStatusSetToCompletedQuery($tableName, $completedStatus)
        {
            $query  = "update $tableName set status = $completedStatus where (completedDateTime is not null ";
            $query  .= "and completedDateTime != '0000-00-00 00:00:00') or completed = 1;";
            return $query;
        }

        /**
         * Return query to update Task status to new where required
         * @param $tableName
         * @param $completedStatus
         * @return string
         */
        protected function getTaskStatusSetToNewQuery($tableName, $completedStatus)
        {
            $newStatus  = Task::STATUS_NEW;
            $query      = "update $tableName set status = $newStatus where status is null or status != $completedStatus;";
            return $query;
        }

        /**
         * In the old Autobuild indexes weren't named morally correct, say like UQ_f97279e76d95d98f7433ff400fab94e189ee6052,
         * which they are in autobuildv2, kind of. We don't even need to do anything for indexes.
         * If you have old autobuild's scheme and you use upgradeSchema of autobuildv2
         * it would recognize old indexes(even if the name is different) and skip them.
         * This is just to make fresh install and upgraded versions bit more similar
         * in terms of db schema.
         * depending on size of db, this could take long...
         */
        protected function updateIndexes()
        {
            // Don't worry about all these index drops and creations
            $fixIndexesQuery = <<<QRY
ALTER TABLE `_group`
    DROP INDEX `UQ_f97279e76d95d98f7433ff400fab94e189ee6052`,
    ADD UNIQUE `unique_eman` (`name`);

ALTER TABLE `_group__user`
    DROP INDEX `UQ_8b7b9c47c851f14d46de32b2c5dd3ffd490b9319`,
    DROP INDEX `index_for__group__user__group_id`,
    DROP INDEX `index_for__group__user__user_id`,
    ADD UNIQUE `unique_di_resu__di_puorg_` (`_group_id`, `_user_id`),
    ADD INDEX `di_puorg_` (`_group_id`),
    ADD INDEX `di_resu_` (`_user_id`);

ALTER TABLE `_user`
    DROP INDEX `UQ_2f9f20ae60de87f7bdd974b52941c30e287c6eef`,
    ADD UNIQUE `unique_emanresu` (`username`);

ALTER TABLE `account_read`
    DROP INDEX `index_account_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `account_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `activity_item`
    DROP INDEX `UQ_3072cf7f6632136338312839309d6fb046214edc`,
    DROP INDEX `index_for_activity_item_activity_id`,
    DROP INDEX `index_for_activity_item_item_id`,
    ADD UNIQUE `unique_di_meti_di_ytivitca` (`activity_id`, `item_id`),
    ADD INDEX `di_ytivitca` (`activity_id`),
    ADD INDEX `di_meti` (`item_id`);

ALTER TABLE `campaign_read`
    DROP INDEX `index_campaign_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `campaign_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `contact_opportunity`
    DROP INDEX `UQ_49ec3f900018477a710f404b661c83de848a2192`,
    DROP INDEX `index_for_contact_opportunity_contact_id`,
    DROP INDEX `index_for_contact_opportunity_opportunity_id`,
    ADD UNIQUE `unique_di_ytinutroppo_di_tcatnoc` (`contact_id`, `opportunity_id`),
    ADD INDEX `di_tcatnoc` (`contact_id`),
    ADD INDEX `di_ytinutroppo` (`opportunity_id`);

ALTER TABLE `contact_read`
    DROP INDEX `index_contact_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `contact_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `contactwebform_read`
    DROP INDEX `index_contactwebform_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `contactwebform_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `conversation_item`
    DROP INDEX `UQ_f9926d7fcbce985516b367d10bb523785fbab2d7`,
    DROP INDEX `index_for_conversation_item_conversation_id`,
    DROP INDEX `index_for_conversation_item_item_id`,
    ADD UNIQUE `unique_di_meti_di_noitasrevnoc` (`conversation_id`, `item_id`),
    ADD INDEX `di_noitasrevnoc` (`conversation_id`),
    ADD INDEX `di_meti` (`item_id`);

ALTER TABLE `conversation_read`
    DROP INDEX `index_conversation_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `conversation_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `currency`
    DROP INDEX `UQ_4f5a32b86618fd9d6a870ffe890cf77a88669783`,
    ADD UNIQUE `unique_edoc` (`code`);

ALTER TABLE `customfielddata`
    DROP INDEX `UQ_f97279e76d95d98f7433ff400fab94e189ee6052`,
    ADD UNIQUE `unique_eman` (`name`);

ALTER TABLE `emailmessage_read`
    DROP INDEX `index_emailmessage_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `emailmessage_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `emailmessagerecipient`
    DROP INDEX `remailmessage_Index`,
    ADD INDEX `remailmessage` (`emailmessage_id`);

ALTER TABLE `emailtemplate_read`
    DROP INDEX `index_emailtemplate_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `emailtemplate_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `globalmetadata`
    DROP INDEX `UQ_6950932d5c0020179c0a175933c8d60ccab633ae`,
    ADD UNIQUE `unique_emaNssalc` (`classname`);

ALTER TABLE `marketinglist_read`
    DROP INDEX `index_marketinglist_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `marketinglist_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `meeting_read`
    DROP INDEX `index_meeting_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `meeting_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `messagesource`
    DROP INDEX `source_category_Index`,
    ADD UNIQUE `sourceCategory` (`category`, `source`(767));

ALTER TABLE `messagetranslation`
    DROP INDEX `source_language_translation_Index`,
    ADD UNIQUE `sourceLanguageTranslation` (`messagesource_id`, `language`, `translation`(767));

ALTER TABLE `mission_read`
    DROP INDEX `index_mission_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `mission_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `namedsecurableitem`
    DROP INDEX `UQ_f97279e76d95d98f7433ff400fab94e189ee6052`,
    ADD UNIQUE `unique_eman` (`name`);

ALTER TABLE `note_read`
    DROP INDEX `index_note_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `note_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `opportunity_read`
    DROP INDEX `index_opportunity_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `opportunity_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `product_productcategory`
    DROP INDEX `UQ_1847df0f79503c223534c3058299bc8e4db1c51e`,
    DROP INDEX `index_for_product_productcategory_product_id`,
    DROP INDEX `index_for_product_productcategory_productcategory_id`,
    ADD UNIQUE `unique_di_yrogetactcudorp_di_tcudorp` (`product_id`, `productcategory_id`),
    ADD INDEX `di_tcudorp` (`product_id`),
    ADD INDEX `di_yrogetactcudorp` (`productcategory_id`);

ALTER TABLE `product_read`
    DROP INDEX `index_product_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `product_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `productcatalog_productcategory`
    DROP INDEX `UQ_f6568fb4a89ba252a0046e46ae78caab1cf7b128`,
    DROP INDEX `index_for_productcatalog_productcategory_productcategory_id`,
    DROP INDEX `index_for_productcatalog_productcategory_productcatalog_id`,
    ADD UNIQUE `unique_di_yrogetactcudorp_di_golatactcudorp` (`productcatalog_id`, `productcategory_id`),
    ADD INDEX `di_golatactcudorp` (`productcatalog_id`),
    ADD INDEX `di_yrogetactcudorp` (`productcategory_id`);

ALTER TABLE `productcategory_producttemplate`
    DROP INDEX `UQ_ddf85331d64908459ca387f73c4133bee1cbab07`,
    DROP INDEX `index_for_productcategory_producttemplate_producttemplate_id`,
    DROP INDEX `index_for_productcategory_producttemplate_productcategory_id`,
    ADD UNIQUE `unique_di_etalpmettcudorp_di_yrogetactcudorp` (`productcategory_id`, `producttemplate_id`),
    ADD INDEX `di_yrogetactcudorp` (`productcategory_id`),
    ADD INDEX `di_etalpmettcudorp` (`producttemplate_id`);

ALTER TABLE `role`
    DROP INDEX `UQ_f97279e76d95d98f7433ff400fab94e189ee6052`,
    ADD UNIQUE `unique_eman` (`name`);

ALTER TABLE `savedreport_read`
    DROP INDEX `index_savedreport_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `savedreport_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `socialitem_read`
    DROP INDEX `index_socialitem_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `socialitem_read_securableitem_id` (`securableitem_id`);

ALTER TABLE `task_read`
    DROP INDEX `index_task_read_securable_item_id`,
    ADD UNIQUE `securableitem_id_munge_id` (`securableitem_id`, `munge_id`),
    ADD INDEX `task_read_securableitem_id` (`securableitem_id`);
QRY;
            ZurmoRedBean::exec($fixIndexesQuery);
        }

        /**
         * Set metadata for user so we will know that we need to update his GoogleApps contacts groups
         */
        protected function setPerUserMetadataForGoogleAppsUpgrade()
        {
            $users = User::getAll();
            foreach ($users as $user)
            {
                if($user->isSystemUser)
                {
                    continue;
                }
                try
                {
                    $googleAppsSyncInfo = GoogleAppsSyncInfo::getByUserId($user->id);
                    if ($googleAppsSyncInfo->contactsLastSyncZurmoToGoogleDateTime != null)
                    {
                        $perUserMetadata = new PerUserMetadata();
                        $perUserMetadata->className           = 'UpgradeGoogleApps';
                        $perUserMetadata->user                = $user;
                        $perUserMetadata->serializedMetadata  = serialize(array('runGoogleApps25xUpgrade' => 1));
                        $perUserMetadata->save();
                    }
                }
                catch (NotFoundException $e)
                {
                    // do nothing, just continue with next user
                }
            }
        }

        /**
         * Update GoogleApps contacts group name
         */
        protected function saveGoogleAppsContactGroupName()
        {
            if (Yii::app()->googleAppsHelper->isConfigured())
            {
                $googleAppsSettings = Yii::app()->googleAppsHelper->getGoogleAppsSettings();
                $googleAppsSettings['googleContactGroup'] = Yii::app()->label;
                Yii::app()->googleAppsHelper->setGoogleAppsSettings($googleAppsSettings);
            }
        }
    }
?>