<?php
    /*********************************************************************************
     * Zurmo is a customer relationship management program developed by
     * Zurmo, Inc. Copyright (C) 2012 Zurmo Inc.
     *
     * This program is free software; you can redistribute it and/or modify it under
     * the terms of the GNU General Public License version 3 as published by the
     * Free Software Foundation with the addition of the following permission added
     * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
     * IN WHICH THE COPYRIGHT IS OWNED BY ZURMO, ZURMO DISCLAIMS THE WARRANTY
     * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
     *
     * This program is distributed in the hope that it will be useful, but WITHOUT
     * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
     * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
     * details.
     *
     * You should have received a copy of the GNU General Public License along with
     * this program; if not, see http://www.gnu.org/licenses or write to the Free
     * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
     * 02110-1301 USA.
     *
     * You can contact Zurmo, Inc. with a mailing address at 113 McHenry Road Suite 207,
     * Buffalo Grove, IL 60089, USA. or at email address contact@zurmo.com.
     *
     * The interactive user interfaces in modified source and object code versions
     * of this program must display Appropriate Legal Notices, as required under
     * Section 5 of the GNU General Public License version 3.
     *
     * In accordance with Section 7(b) of the GNU General Public License version 3,
     * these Appropriate Legal Notices must retain the display of the "Powered by
     * Zurmo" logo. If the display of the logo is not reasonably feasible for
     * technical reasons, the Appropriate Legal Notices must display the words
     * "Powered by Zurmo".
     ********************************************************************************/

    require_once 'BaseUpgraderComponent.php';

    class UpgraderComponent extends BaseUpgraderComponent
    {
        protected $_messageLogger = null;

        public function getMessageLogger() {
            return $this->_messageLogger;
        }

        public function setMessageLogger($messageLogger) {
            $this->_messageLogger = $messageLogger;
        }

        /**
         * Tasks to do before start processing config files.
         */
        public function processBeforeConfigFiles()
        {
            $this->messageLogger->addInfoMessage("Process before config files.");
        }

        /**
         * Process config files.
         * Problem here is that some config files might be modified, so there
         * can be some additional config options. We need to modify config
         * files(debug.php, debugTest.php, perInstance.php, perInstanceTest.php),
         * without loosing settings entered by user, so in most cases we will add
         * content at the end of those files where possible.
         * Should be used to files like debug.php, debugTest.php, perInstance.php, perInstanceTest.php,
         * which are not tracked by SCM tools, other files are updated automatically.
         *
         * @code
            <?php
                //Below is just sample code to add $myVariable to perInstance.php and perInstanceTest.php files.
                $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
                if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '0.6.80'))
                {
                    $perInstanceTestFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstanceTest.php';
                    if (is_file($perInstanceTestFile))
                    {
                        $perInstanceTestContent = file_get_contents($perInstanceTestFile);

                        $contentToAdd = "\$myVariable = 'aaa';\n";
                        $perInstanceTestContent = preg_replace('/\?\>/',
                                                               "\n" . $contentToAdd . "\n" . "?>",
                                                               $perInstanceTestContent);
                        file_put_contents($perInstanceTestFile, $perInstanceTestContent);
                    }
                    $perInstanceFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstance.php';
                    if (is_file($perInstanceFile))
                    {
                        $perInstanceContent = file_get_contents($perInstanceFile);

                        $contentToAdd = "\$myVariable = 'aaa';\n";
                        $perInstanceContent = preg_replace('/\?\>/',
                                                           "\n" . $contentToAdd . "\n" . "?>",
                                                           $perInstanceContent);
                        file_put_contents($perInstanceFile, $perInstanceContent);
                    }
                }
            ?>
         * @endcode
         */
        public function processConfigFiles($pathToConfigurationFolder)
        {
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '1.1.20'))
            {
                $perInstanceFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstance.php';
                if (is_file($perInstanceFile))
                {
                    $perInstanceContent = file_get_contents($perInstanceFile);

                    $zurmoToken = substr(md5(microtime() * mt_rand()), 0, 15);
                    $contentToAdd = "    define('ZURMO_TOKEN', '$zurmoToken');\n";

                    $perInstanceContent = preg_replace('/\?\>/',
                                                       "\n" . $contentToAdd . "\n" . "?>",
                                                       $perInstanceContent);
                    file_put_contents($perInstanceFile, $perInstanceContent);
                }
                $perInstanceTestFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'perInstanceTest.php';
                if (is_file($perInstanceTestFile))
                {
                    $perInstanceTestContent = file_get_contents($perInstanceTestFile);

                    $zurmoToken = substr(md5(microtime() * mt_rand()), 0, 15);
                    $contentToAdd = "    define('ZURMO_TOKEN', $zurmoToken);\n";

                    $perInstanceTestContent = preg_replace('/\?\>/',
                                                       "\n" . $contentToAdd . "\n" . "?>",
                                                       $perInstanceTestContent);

                    file_put_contents($perInstanceTestFile, $perInstanceTestContent);
                }

                $debugFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'debug.php';
                if (is_file($debugFile))
                {
                    $debugContent = file_get_contents($debugFile);
                    $contentToAdd = "    // Turn this on to save RedBean queries into file. Turn it off in production.
    // Check it in as false!
    \$redBeanDebugLogToFileOn = false;\n";
                    $debugContent = preg_replace('/\/\/ Turn this on to see extra query data such as total count, duplicate count, and duplicate queries/',
                                                 "\n" . $contentToAdd . "\n" . "    // Turn this on to see extra query data such as total count, duplicate count, and duplicate queries",
                                                 $debugContent);
                    $contentToAdd = "    define('REDBEAN_DEBUG_TO_FILE',  \$redBeanDebugLogToFileOn);";
                    $debugContent = preg_replace('/define\(\'REDBEAN_DEBUG\',          \$redBeanDebugOn\);/',
                                                 "define('REDBEAN_DEBUG',          \$redBeanDebugOn);\n" . $contentToAdd,
                                                 $debugContent);

                    file_put_contents($debugFile, $debugContent);
                }
                $debugTestFile = $pathToConfigurationFolder . DIRECTORY_SEPARATOR . 'debugTest.php';
                if (is_file($debugTestFile))
                {
                    $debugTestContent = file_get_contents($debugTestFile);
                    $contentToAdd = "    // Turn this on to save RedBean queries into file. Turn it off in production.
    // Check it in as false!
    \$redBeanDebugLogToFileOn = false;\n";
                    $debugTestContent = preg_replace('/\/\/ Turn this on to see extra query data such as total count, duplicate count, and duplicate queries/',
                                                 "\n" . $contentToAdd . "\n" . "    // Turn this on to see extra query data such as total count, duplicate count, and duplicate queries",
                                                 $debugTestContent);
                    $contentToAdd = "    define('REDBEAN_DEBUG_TO_FILE',  \$redBeanDebugLogToFileOn);";
                    $debugTestContent = preg_replace('/define\(\'REDBEAN_DEBUG\',          \$redBeanDebugOn\);/',
                                                     "define('REDBEAN_DEBUG',          \$redBeanDebugOn);\n" . $contentToAdd,
                                                     $debugTestContent);
                    file_put_contents($debugTestFile, $debugTestContent);
                }
            }
        }

        /**
         * Process files
         * @param string $source
         * @param string $destination
         * @param array $configuration
         */
        public function processFiles($source, $destination, $configuration)
        {
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '1.1.20'))
            {
                // Remove files that are marked for removal in manifest.php.
                if (isset($configuration['removedFiles']) && !empty($configuration['removedFiles']))
                {
                    foreach ($configuration['removedFiles'] as $fileOrFolderToRemove)
                    {
                        // Replace Lunux directory separators(from upgrade manifest file)
                        // with one used by system(DIRECTORY_SEPARATOR)
                        $fileOrFolderToRemove = str_replace('/', DIRECTORY_SEPARATOR, $fileOrFolderToRemove);
                        $fileOrFolderToRemove = $destination . DIRECTORY_SEPARATOR . trim($fileOrFolderToRemove, DIRECTORY_SEPARATOR);
                        if (is_dir($fileOrFolderToRemove))
                        {
                            FileUtil::deleteDirectoryRecursive($fileOrFolderToRemove);
                        }
                        elseif (is_file($fileOrFolderToRemove))
                        {
                            unlink($fileOrFolderToRemove);
                        }
                    }
                }
                // Copy new or modified files.
                FileUtil::copyRecursive($source, $destination);
            }
        }

        /**
         * Tasks that should be executed before updating schema.
         * @param string $currentZurmoVersion
         *
         * @code
            <?php
                $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
                if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '0.6.80'))
                {
                    $metadata = AccountsModule::getMetadata();
                    if(!isset($metadata['global']['newElement']))
                    {
                        $metadata['global']['newElement'] = 'Some Content';
                        AccountsModule::setMetadata($metadata);
                    }
                    GeneralCache::forgetAll();
                }
            ?>
         * @endcode
         */
        public function processBeforeUpdateSchema()
        {
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '1.1.20'))
            {
                if (GlobalMetadata::isClassMetadataSavedInDatabase('AccountConvertToView'))
                {
                    $search = array('type' => 'SaveButton', 'label' => "eval:Yii::t('Default', 'Complete Conversion')");
                    $metadata = AccountConvertToView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['label'] = "eval:Zurmo::t('AccountsModule', 'Complete Conversion')";
                        AccountConvertToView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('CommentForSocialItemInlineEditView'))
                {
                    $search = array('type' => 'SaveButton', 'label' => 'eval:Yii::t("Default", "Comment")');
                    $metadata = CommentForSocialItemInlineEditView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['label'] = "eval:Zurmo::t('CommentsModule', 'Comment')";
                        CommentForSocialItemInlineEditView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('CommentInlineEditView'))
                {
                    $search = array('type' => 'SaveButton', 'label' => 'eval:Yii::t("Default", "Comment")');
                    $metadata = CommentInlineEditView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['label'] = "eval:Zurmo::t('CommentsModule', 'Comment')";
                        CommentInlineEditView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('ContactInlineCreateForArchivedEmailCreateView'))
                {
                    $search = array('type' => 'SaveButton');
                    $metadata = ContactInlineCreateForArchivedEmailCreateView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][] = $metadata['global']['toolbar']['elements'][$result['elementKey']];
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['type'] = "CancelLinkForEmailsMatchingList";
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['htmlOptions'] =
                            array(
                                'id' => 'eval:"createContactCancel" . $this->uniqueId',
                                'name' => 'eval:"createContactCancel" . $this->uniqueId',
                                'class' => 'eval:"createContactCancel"',
                            );
                        ContactInlineCreateForArchivedEmailCreateView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('EmailMessageDetailsView'))
                {
                    $metadata = EmailMessageDetailsView::getMetadata();
                    $search = array('attributeName' => 'sentDateTime', 'type' => 'EmailMessageSentDateTime');
                    if ($result = $this->findPositionOfElementInViewPanels($metadata, $search))
                    {
                        $metadata['global']['panels'][$result['panelKey']]['rows'][$result['rowKey']]['cells'][$result['cellKey']]['elements'][$result['elementKey']]['type'] = 'DateTime';

                        EmailMessageDetailsView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('JobLogDetailsView'))
                {
                    $search = array('type' => 'CancelToListLink', 'label' => "eval:Yii::t('Default', 'Return to Job Manager')");
                    $metadata = JobLogDetailsView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['label'] = "eval:Zurmo::t('JobsManagerModule', 'Return to Job Manager')";
                        JobLogDetailsView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('LeadInlineCreateForArchivedEmailCreateView'))
                {
                    $search = array('type' => 'SaveButton');
                    $metadata = LeadInlineCreateForArchivedEmailCreateView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][] = $metadata['global']['toolbar']['elements'][$result['elementKey']];
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['type'] = "CancelLinkForEmailsMatchingList";
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['htmlOptions'] =
                            array(
                                'id' => 'eval:"createLeadCancel" . $this->uniqueId',
                                'name' => 'eval:"createLeadCancel" . $this->uniqueId',
                                'class' => 'eval:"createLeadCancel"',
                            );
                        LeadInlineCreateForArchivedEmailCreateView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('MapConfigurationView'))
                {
                    $search = array('type' => 'ConfigurationLink', 'label' => "eval:Yii::t('Default', 'Cancel')");
                    $metadata = MapConfigurationView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['label'] = "eval:Zurmo::t('MapsModule', 'Cancel')";
                        MapConfigurationView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('SocialItemInlineEditView'))
                {
                    $search = array('type' => 'SaveButton', 'label' => 'eval:Yii::t("Default", "Comment")');
                    $metadata = SocialItemInlineEditView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['label'] = "eval:Zurmo::t('SocialItemsModule', 'Comment')";
                        SocialItemInlineEditView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('UsersListView'))
                {
                    $metadata = UsersListView::getMetadata();
                    $search = array('attributeName' => 'id', 'type' => 'Text');
                    if ($result = $this->findPositionOfElementInViewPanels($metadata, $search))
                    {
                        unset($metadata['global']['panels'][$result['panelKey']]['rows'][$result['rowKey']]);
                        $rows = array_values($metadata['global']['panels'][$result['panelKey']]['rows']);
                        unset($metadata['global']['panels'][$result['panelKey']]['rows']);
                        $metadata['global']['panels'][$result['panelKey']]['rows'] = $rows;

                        UsersListView::setMetadata($metadata);
                    }
                }
                if (GlobalMetadata::isClassMetadataSavedInDatabase('ZurmoConfigurationEditAndDetailsView'))
                {
                    $search = array('type' => 'SaveButton', 'label' => "eval:Yii::t('Default', 'Cancel')");
                    $metadata = ZurmoConfigurationEditAndDetailsView::getMetadata();
                    if($result = $this->findPositionOfElementInViewToolbarsAndRowMenu($metadata, 'toolbar', $search))
                    {
                        $metadata['global']['toolbar']['elements'][$result['elementKey']]['label'] = "eval:Zurmo::t('ZurmoModule', 'Cancel')";
                        ZurmoConfigurationEditAndDetailsView::setMetadata($metadata);
                    }

                    $metadata = ZurmoConfigurationEditAndDetailsView::getMetadata();
                    $search = array('attributeName' => 'applicationName', 'type' => 'Text');
                    if ($result = $this->findPositionOfElementInViewPanels($metadata, $search))
                    {
                        $row = array(
                            'cells' => array(
                                array(
                                    'elements' => array(
                                         array(
                                            'attributeName' => 'realtimeUpdatesEnabled',
                                            'type' => 'CheckBox'
                                         )
                                    )
                                )
                            )
                        );
                        $metadata['global']['panels'][$result['panelKey']]['rows'][] = $row;

                        ZurmoConfigurationEditAndDetailsView::setMetadata($metadata);
                    }
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('UsersSearchView'))
                {
                    $metadata = UsersSearchView::getMetadata();
                    $search = array('attributeName' => 'department', 'type' => 'Text');
                    if ($result = $this->findPositionOfElementInViewPanels($metadata, $search))
                    {
                        if ($metadata['global']['panels'][$result['panelKey']]['title'] == 'Advanced Search')
                        {
                            unset($metadata['global']['panels'][$result['panelKey']]['title']);
                            $metadata['global']['panels'][$result['panelKey']]['rows'] = array();
                            $metadata['global']['panels'][$result['panelKey']]['advancedSearchType'] = DynamicSearchView::ADVANCED_SEARCH_TYPE_DYNAMIC;
                            UsersSearchView::setMetadata($metadata);
                        }
                    }

                    if (!in_array("currency", $metadata['global']['nonPlaceableAttributeNames']))
                    {
                        $metadata['global']['nonPlaceableAttributeNames'][] = 'currency';
                    }
                    if (!in_array("timeZone", $metadata['global']['nonPlaceableAttributeNames']))
                    {
                        $metadata['global']['nonPlaceableAttributeNames'][] = 'timeZone';
                    }
                    if (!in_array("language", $metadata['global']['nonPlaceableAttributeNames']))
                    {
                        $metadata['global']['nonPlaceableAttributeNames'][] = 'language';
                    }
                    if (!in_array("serializedAvatarData", $metadata['global']['nonPlaceableAttributeNames']))
                    {
                        $metadata['global']['nonPlaceableAttributeNames'][] = 'serializedAvatarData';
                    }
                    UsersSearchView::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('ZurmoModule'))
                {
                    $metadata = ZurmoModule::getMetadata();
                    $metadata['global']['configureMenuItems'][] = array(
                        'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                        'titleLabel'       => 'Authentication Configuration',
                        'descriptionLabel' => 'Manage Authentication Configuration',
                        'route'            => '/zurmo/authentication/configurationEdit',
                        'right'            => ZurmoModule::RIGHT_ACCESS_GLOBAL_CONFIGURATION
                    );
                    $metadata['global']['configureSubMenuItems'][] = array(
                        'category'         => ZurmoModule::ADMINISTRATION_CATEGORY_GENERAL,
                        'titleLabel'       => 'Ldap Configuration',
                        'descriptionLabel' => 'Manage Ldap Authentication',
                        'route'            => '/zurmo/ldap/configurationEditLdap',
                        'right'            => ZurmoModule::RIGHT_ACCESS_GLOBAL_CONFIGURATION
                    );
                    ZurmoModule::setMetadata($metadata);
                }

                if (GlobalMetadata::isClassMetadataSavedInDatabase('UserConfigurationEditView'))
                {
                    $metadata = UserConfigurationEditView::getMetadata();
                    $search = array('attributeName' => 'listPageSize', 'type' => 'Integer');
                    if ($result = $this->findPositionOfElementInViewPanels($metadata, $search))
                    {
                        $row = array(
                            'cells' => array(
                                array(
                                    'elements' => array(
                                         array(
                                            'attributeName' => 'enableDesktopNotifications',
                                            'type' => 'DesktopNotification'
                                         )
                                    )
                                )
                            )
                        );
                        $metadata['global']['panels'][$result['panelKey']]['rows'][] = $row;
                    }
                    $row = array(
                            'cells' => array(
                                array(
                                    'elements' => array(
                                         array(
                                            'attributeName' => null,
                                            'type' => 'DerivedExplicitReadWritePermissionsUserConfig'
                                         )
                                    )
                                )
                            )
                        );
                    $metadata['global']['panels'][]['rows'][] = $row;
                    UserConfigurationEditView::setMetadata($metadata);
                }

                 if (GlobalMetadata::isClassMetadataSavedInDatabase('Conversation'))
                {
                    $metadata = Conversation::getMetadata();
                    if (!in_array('isClosed', $metadata['Conversation']['members']))
                    {
                        $metadata['Conversation']['members'][] = 'isClosed';
                        $rule = array('isClosed', 'boolean');
                        $metadata['Conversation']['rules'][] = $rule;
                        Conversation::setMetadata($metadata);
                    }
                }
            }
        }

        /**
         * Update schema.
         */
        public function processUpdateSchema($messageLogger)
        {
            // We must update schema via command line, to ensure that newly
            // copied files are loaded into application correctly.
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '1.1.20'))
            {
                InstallUtil::runAutoBuildFromUpdateSchemaCommand($messageLogger);
                ReadPermissionsOptimizationUtil::rebuild();
            }
        }

        /**
         * Tasks to be executed after update schema
         */
        public function processAfterUpdateSchema()
        {
            $zurmoVersionBeforeUpgrade = UpgradeUtil::getUpgradeState('zurmoVersionBeforeUpgrade');
            if ($this->shouldRunTasksByVersion($zurmoVersionBeforeUpgrade, '1.1.20'))
            {
                $allUsers = User::getAll();
                foreach ($allUsers as $user)
                {
                    $user->save();
                }
            }
        }

        /**
         * Clear assets and runtime folders.
         */
        public function clearAssetsAndRunTimeItems()
        {
            $pathToAssetsFolder = INSTANCE_ROOT . DIRECTORY_SEPARATOR . 'assets';
            FileUtil::deleteDirectoryRecursive($pathToAssetsFolder, false, array('index.html'));

            $pathToRuntimeFolder = Yii::app()->getRuntimePath();
            FileUtil::deleteDirectoryRecursive($pathToRuntimeFolder, false, array('index.html', 'upgrade'));
        }

        /**
         * Determine if some task should be executed or not, depending on Zurmo version.
         * @param string $currentZurmoVersion
         * @param string $upgradeVersion - version in which change is added
         * @return boolean
         */
        protected function shouldRunTasksByVersion($currentZurmoVersion, $upgradeVersion)
        {
            $shouldRun = false;
            if (version_compare($currentZurmoVersion, $upgradeVersion, '<'))
            {
                $shouldRun = true;
            }
            return $shouldRun;
        }
    }
?>
