<?php

/**
 * Routes.
 *
 * @copyright Zikula contributors (Zikula)
 * @license http://www.gnu.org/licenses/lgpl.html GNU Lesser General Public License
 * @author Zikula contributors <info@ziku.la>.
 * @see https://ziku.la
 * @version Generated by ModuleStudio 1.4.0 (https://modulestudio.de).
 */

declare(strict_types=1);

namespace Zikula\RoutesModule\Menu\Base;

use Knp\Menu\FactoryInterface;
use Knp\Menu\ItemInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Zikula\ExtensionsModule\Api\ApiInterface\VariableApiInterface;
use Zikula\UsersModule\Api\ApiInterface\CurrentUserApiInterface;
use Zikula\UsersModule\Constant as UsersConstant;
use Zikula\RoutesModule\Entity\RouteEntity;
use Zikula\RoutesModule\RoutesEvents;
use Zikula\RoutesModule\Event\ItemActionsMenuPostConfigurationEvent;
use Zikula\RoutesModule\Event\ItemActionsMenuPreConfigurationEvent;
use Zikula\RoutesModule\Event\ViewActionsMenuPostConfigurationEvent;
use Zikula\RoutesModule\Event\ViewActionsMenuPreConfigurationEvent;
use Zikula\RoutesModule\Helper\EntityDisplayHelper;
use Zikula\RoutesModule\Helper\ModelHelper;
use Zikula\RoutesModule\Helper\PermissionHelper;

/**
 * Menu builder base class.
 */
class AbstractMenuBuilder
{
    /**
     * @var FactoryInterface
     */
    protected $factory;
    
    /**
     * @var EventDispatcherInterface
     */
    protected $eventDispatcher;
    
    /**
     * @var RequestStack
     */
    protected $requestStack;
    
    /**
     * @var PermissionHelper
     */
    protected $permissionHelper;
    
    /**
     * @var EntityDisplayHelper
     */
    protected $entityDisplayHelper;
    
    /**
     * @var CurrentUserApiInterface
     */
    protected $currentUserApi;
    
    /**
     * @var VariableApiInterface
     */
    protected $variableApi;
    
    /**
     * @var ModelHelper
     */
    protected $modelHelper;
    
    public function __construct(
        FactoryInterface $factory,
        EventDispatcherInterface $eventDispatcher,
        RequestStack $requestStack,
        PermissionHelper $permissionHelper,
        EntityDisplayHelper $entityDisplayHelper,
        CurrentUserApiInterface $currentUserApi,
        VariableApiInterface $variableApi,
        ModelHelper $modelHelper
    ) {
        $this->factory = $factory;
        $this->eventDispatcher = $eventDispatcher;
        $this->requestStack = $requestStack;
        $this->permissionHelper = $permissionHelper;
        $this->entityDisplayHelper = $entityDisplayHelper;
        $this->currentUserApi = $currentUserApi;
        $this->variableApi = $variableApi;
        $this->modelHelper = $modelHelper;
    }
    
    /**
     * Builds the item actions menu.
     */
    public function createItemActionsMenu(array $options = []): ItemInterface
    {
        $menu = $this->factory->createItem('itemActions');
        if (!isset($options['entity'], $options['area'], $options['context'])) {
            return $menu;
        }
    
        $entity = $options['entity'];
        $routeArea = $options['area'];
        $context = $options['context'];
        $menu->setChildrenAttribute('class', 'nav item-actions');
    
        $this->eventDispatcher->dispatch(
            new ItemActionsMenuPreConfigurationEvent($this->factory, $menu, $options)
        );
    
        if ($entity instanceof RouteEntity) {
            $routePrefix = 'zikularoutesmodule_route_';
            
            if ('admin' === $routeArea) {
                $previewRouteParameters = $entity->createUrlArgs();
                $previewRouteParameters['preview'] = 1;
                $menu->addChild('Preview', [
                    'route' => $routePrefix . 'display',
                    'routeParameters' => $previewRouteParameters
                ])
                    ->setLinkAttribute('target', '_blank')
                    ->setLinkAttribute(
                        'title',
                        'Open preview page'
                    )
                    ->setAttribute('icon', 'fas fa-search-plus')
                ;
            }
            if ('display' !== $context) {
                $entityTitle = $this->entityDisplayHelper->getFormattedTitle($entity);
                $menu->addChild('Details', [
                    'route' => $routePrefix . $routeArea . 'display',
                    'routeParameters' => $entity->createUrlArgs()
                ])
                    ->setLinkAttribute(
                        'title',
                        str_replace('"', '', $entityTitle)
                    )
                    ->setAttribute('icon', 'fas fa-eye')
                ;
            }
            if ($this->permissionHelper->mayEdit($entity)) {
                $menu->addChild('Edit', [
                    'route' => $routePrefix . $routeArea . 'edit',
                    'routeParameters' => $entity->createUrlArgs()
                ])
                    ->setLinkAttribute(
                        'title',
                        'Edit this route'
                    )
                    ->setAttribute('icon', 'fas fa-edit')
                ;
                $menu->addChild('Reuse', [
                    'route' => $routePrefix . $routeArea . 'edit',
                    'routeParameters' => ['astemplate' => $entity->getKey()]
                ])
                    ->setLinkAttribute(
                        'title',
                        'Reuse for new route'
                    )
                    ->setAttribute('icon', 'fas fa-copy')
                ;
            }
            if ('display' === $context) {
                $menu->addChild('Routes list', [
                    'route' => $routePrefix . $routeArea . 'view'
                ])
                    ->setAttribute('icon', 'fas fa-reply')
                ;
            }
        }
    
        $this->eventDispatcher->dispatch(
            new ItemActionsMenuPostConfigurationEvent($this->factory, $menu, $options)
        );
    
        return $menu;
    }
    
    /**
     * Builds the view actions menu.
     */
    public function createViewActionsMenu(array $options = []): ItemInterface
    {
        $menu = $this->factory->createItem('viewActions');
        if (!isset($options['objectType'], $options['area'])) {
            return $menu;
        }
    
        $objectType = $options['objectType'];
        $routeArea = $options['area'];
        $menu->setChildrenAttribute('class', 'nav view-actions');
    
        $this->eventDispatcher->dispatch(
            new ViewActionsMenuPreConfigurationEvent($this->factory, $menu, $options)
        );
    
        $query = $this->requestStack->getMasterRequest()->query;
        $currentTemplate = $query->getAlnum('tpl', '');
        if ('route' === $objectType) {
            $routePrefix = 'zikularoutesmodule_route_';
            if (!in_array($currentTemplate, [])) {
                $canBeCreated = $this->modelHelper->canBeCreated($objectType);
                if ($canBeCreated) {
                    if ($this->permissionHelper->hasComponentPermission($objectType, ACCESS_EDIT)) {
                        $menu->addChild('Create route', [
                            'route' => $routePrefix . $routeArea . 'edit'
                        ])
                            ->setAttribute('icon', 'fas fa-plus')
                        ;
                    }
                }
                $routeParameters = $query->all();
                if (1 === $query->getInt('own')) {
                    $routeParameters['own'] = 1;
                } else {
                    unset($routeParameters['own']);
                }
                if (1 === $query->getInt('all')) {
                    unset($routeParameters['all']);
                    $menu->addChild('Back to paginated view', [
                        'route' => $routePrefix . $routeArea . 'view',
                        'routeParameters' => $routeParameters
                    ])
                        ->setAttribute('icon', 'fas fa-table')
                    ;
                } else {
                    $routeParameters['all'] = 1;
                    $menu->addChild('Show all entries', [
                        'route' => $routePrefix . $routeArea . 'view',
                        'routeParameters' => $routeParameters
                    ])
                        ->setAttribute('icon', 'fas fa-table')
                    ;
                }
                if ($this->permissionHelper->hasComponentPermission($objectType, ACCESS_EDIT)) {
                    $routeParameters = $query->all();
                    if (1 === $query->getInt('own')) {
                        unset($routeParameters['own']);
                        $menu->addChild('Show also entries from other users', [
                            'route' => $routePrefix . $routeArea . 'view',
                            'routeParameters' => $routeParameters
                        ])
                            ->setAttribute('icon', 'fas fa-users')
                        ;
                    } else {
                        $routeParameters['own'] = 1;
                        $menu->addChild('Show only own entries', [
                            'route' => $routePrefix . $routeArea . 'view',
                            'routeParameters' => $routeParameters
                        ])
                            ->setAttribute('icon', 'fas fa-user')
                        ;
                    }
                }
            }
        }
    
        $this->eventDispatcher->dispatch(
            new ViewActionsMenuPostConfigurationEvent($this->factory, $menu, $options)
        );
    
        return $menu;
    }
}
