<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Model;

use XLite\Core\MagicMethodsIntrospectionInterface;
use XLite\Core\Serialization\SerializableEntity;
use Includes\Utils\ConfigParser;

/**
 * Abstract entity
 */
abstract class AEntity extends \XLite\Base\SuperClass implements SerializableEntity
{
    /**
     * @var \XLite\Core\CommonCell
     */
    protected $_previous_state;

    /**
     * @return \XLite\Core\CommonCell
     */
    public function _getPreviousState()
    {
        if ($this->_previous_state === null) {
            $this->_previous_state  = new \XLite\Core\CommonCell();
        }

        return $this->_previous_state;
    }

    /**
     * Possible action by entity Repo
     */
    const ACTION_INSERT = 'insert';
    const ACTION_UPDATE = 'update';
    const ACTION_DELETE = 'delete';

    /**
     * Cache enabled flag (cache)
     *
     * @var array
     */
    protected static $cacheEnabled = array();

    /**
     * Constructor
     *
     * @param array $data Entity properties OPTIONAL
     */
    public function __construct(array $data = array())
    {
        parent::__construct();

        if (!empty($data)) {
            $this->map($data);
        }
    }

    /**
     * Map data to entity columns
     *
     * @param array $data Data
     *
     * @return \XLite\Model\AEntity
     */
    public function map(array $data)
    {
        foreach ($data as $key => $value) {
            // Map only existing properties with setter methods or direct
            $method = 'set' . \Includes\Utils\Converter::convertToPascalCase($key);

            if (method_exists($this, $method) || ($this instanceof MagicMethodsIntrospectionInterface && $this->hasMagicMethod($method))
            ) {
                // $method is assembled from 'set' + getMethodName()
                $this->$method($value);

            } else {
                $this->setterProperty($key, $value);
            }
        }

        return $this;
    }

    /**
     * Common getter
     *
     * @param string $name Property name
     *
     * @return mixed
     */
    public function __get($name)
    {
        // Accessor method name
        return $this->{'get' . \Includes\Utils\Converter::convertToPascalCase($name)}();
    }

    /**
     * Common setter
     *
     * @param string $name  Property name
     * @param mixed  $value Property value
     *
     * @return mixed
     */
    public function __set($name, $value)
    {
        // Mutator method name
        return $this->{'set' . \Includes\Utils\Converter::convertToPascalCase($name)}($value);
    }

    /**
     * Common isset
     *
     * @param string $name Property name
     *
     * @return boolean
     */
    public function __isset($name)
    {
        return !is_null($this->__get($name));
    }

    /**
     * Common unset
     *
     * @param string $name Property name
     *
     * @return void
     */
    public function __unset($name)
    {
        $this->__set($name, null);
    }

    /**
     * Get entity repository
     *
     * @return \XLite\Model\Repo\ARepo
     */
    public function getRepository()
    {
        return \XLite\Core\Database::getRepo(get_class($this));
    }

    /**
     * Check cache after entity persis or remove
     *
     * @return void
     */
    public function checkCache()
    {
        $class = get_class($this);

        if (!isset(static::$cacheEnabled[$class])) {
            $repo = $this->getRepository();

            static::$cacheEnabled[$class] = ($repo && is_subclass_of($repo, '\XLite\Model\Repo\ARepo'))
                ? $repo->hasCacheCells()
                : false;
        }

        if (static::$cacheEnabled[$class]) {
            $this->getRepository()->deleteCacheByEntity($this);
        }
    }

    /**
     * Detach static
     *
     * @return void
     */
    public function detach()
    {
        \XLite\Core\Database::getEM()->detach($this);
    }

    /**
     * Emulate the Doctrine autogenerated methods.
     * @todo: check usage, remove in 5.4
     *
     * @param string $method Method name
     * @param array  $args   Call arguments OPTIONAL
     *
     * @return mixed
     */
    public function __call($method, array $args = array())
    {
        $result = preg_match('/^(get|set)(\w+)$/Si', $method, $matches) && !empty($matches[2]);
        if ($result) {
            $property = \XLite\Core\Converter::convertFromCamelCase($matches[2]);
            $result = 'set' === $matches[1]
                ? $this->setterProperty($property, array_shift($args))
                : $this->getterProperty($property);
        } else {
            $result = null;
        }

        return $result;
    }

    /**
     * Return true if specified property exists
     *
     * @param string $name Property name
     *
     * @return boolean
     */
    public function isPropertyExists($name)
    {
        return property_exists($this, $name) || property_exists($this, \XLite\Core\Converter::convertFromCamelCase($name));
    }

    /**
     * Universal setter
     *
     * @param string $property
     * @param mixed  $value
     *
     * @return true|null Returns TRUE if the setting succeeds. NULL if the setting fails
     */
    public function setterProperty($property, $value)
    {
        $result = property_exists($this, $property);

        if ($result) {
            // Get property value
            $this->$property = $value;

        } else {
            // Log wrong access to property
            $this->logWrongPropertyAccess($property, false);
        }

        return $result ?: null;
    }

    /**
     * Universal getter
     *
     * @param string $property
     *
     * @return mixed|null Returns NULL if it is impossible to get the property
     */
    public function getterProperty($property)
    {
        $result = null;

        if (property_exists($this, $property)) {
            // Get property value
            $result = $this->$property;

        } else {
            // Log wrong access to property
            $this->logWrongPropertyAccess($property);
        }

        return $result;
    }

    /**
     * Log access to unknow property
     *
     * @param string  $property Property name
     * @param boolean $isGetter Flag: is called property getter (true) or setter (false) OPTIONAL
     *
     * @return void
     */
    protected function logWrongPropertyAccess($property, $isGetter = true)
    {
        if (LOG_DEBUG == ConfigParser::getOptions(['log_details', 'level'])) {
            \XLite\Logger::getInstance()->log(
                sprintf(
                    'Requested %s for unknown property: %s::%s',
                    $isGetter ? 'getter' : 'setter',
                    get_class($this),
                    $property
                ),
                LOG_ERR
            );
        }
    }

    /**
     * Check if entity is persistent
     *
     * @return boolean
     */
    public function isPersistent()
    {
        return (bool) $this->getUniqueIdentifier();
    }

    /**
     * Check if the entity is in the DETACHED state
     *
     * @return boolean
     */
    public function isDetached()
    {
        return \Doctrine\ORM\UnitOfWork::STATE_DETACHED === $this->getEntityState();
    }

    /**
     * Check if the entity is in the MANAGED state
     *
     * @return boolean
     */
    public function isManaged()
    {
        return \Doctrine\ORM\UnitOfWork::STATE_MANAGED === $this->getEntityState();
    }

    /**
     * The Entity state getter
     *
     * @return integer
     */
    protected function getEntityState()
    {
        return \XLite\Core\Database::getEM()->getUnitOfWork()->getEntityState($this);
    }

    /**
     * Get entity unique identifier name
     *
     * @return string
     */
    public function getUniqueIdentifierName()
    {
        return $this->getRepository()->getPrimaryKeyField();
    }

    /**
     * Get entity unique identifier value
     *
     * @return mixed
     */
    public function getUniqueIdentifier()
    {
        return $this->{'get' . \Includes\Utils\Converter::convertToPascalCase($this->getUniqueIdentifierName())}();
    }

    /**
     * Returns entity name
     *
     * @return string
     * @throws \Exception
     */
    public function getEntityName()
    {
        try {
            $entityName = \XLite\Core\Database::getEM()->getMetadataFactory()->getMetadataFor(get_class($this))->getName();
        } catch (\Doctrine\ORM\Mapping\MappingException $e) {
            throw new \Exception('Given object ' . get_class($this) . ' is not a Doctrine Entity. ');
        }

        return $entityName;
    }

    /**
     * Returns available metadata for an entity field to use in frontend
     *
     * @param  string  $property Entity property
     * @return array
     */
    public function getFieldMetadata($property)
    {
        return array(
            'data-model' => $this->getEntityName(),
            'data-identifier' => $this->getUniqueIdentifier(),
            'data-property' => $property,
        );
    }

    /**
     * Update entity
     *
     * @return boolean
     */
    public function update()
    {
        \XLite\Core\Database::getEM()->persist($this);
        \XLite\Core\Database::getEM()->flush();

        return true;
    }

    /**
     * Create entity
     *
     * @return boolean
     */
    public function create()
    {
        return $this->update();
    }

    /**
     * Delete entity
     *
     * @return boolean
     */
    public function delete()
    {
        \XLite\Core\Database::getEM()->remove($this);
        \XLite\Core\Database::getEM()->flush();
        \XLite\Core\Database::getEM()->clear();

        return true;
    }

    /**
     * Process files.
     * Returns array of error messages or empty array if no errors detected.
     *
     * @param string $field        Field
     * @param array  $data Data to save
     *
     * @return array
     */
    public function processFiles($field, array $data)
    {
        $errors = array();

        $entityProperties = $this->getRepository()->getEntityProperties();
        $entityProperties = $entityProperties[1][$field];
        if ($entityProperties && is_array($data)) {
            $value = $this->getterProperty($field);
            if ($entityProperties['many']) {
                foreach ($value as $file) {
                    if (isset($data[$file->getId()])) {
                        $errors[] = $this->processFile($file, $data[$file->getId()], $entityProperties);
                        unset($data[$file->getId()]);
                    }
                }
                foreach ($data as $k => $tempData) {
                    $errors[] = $this->processFile(null, $tempData, $entityProperties);
                }

            } else {
                $errors[] = $this->processFile($value, $data, $entityProperties);
            }
        }

        return array_filter($errors);
    }

    /**
     * Process single file.
     * Returns array(<text message>,<array with message params>) on error or null on success
     *
     * @param mixed $file       File
     * @param array $data       Data to save
     * @param array $properties Properties
     *
     * @return array|null
     */
    protected function processFile($file, $data, $properties)
    {
        $errorMessage = null;

        $temporaryFile = isset($data['temp_id'])
            ? \XLite\Core\Database::getRepo('\XLite\Model\TemporaryFile')->find($data['temp_id'])
            : null;

        if (isset($data['delete']) && $data['delete']) {
            if ($file) {
                if ($properties['many']) {
                    $this->{$properties['getter']}()->removeElement($file);

                } else {
                    $this->{$properties['setter']}(null);
                }
                \XLite\Core\Database::getEM()->remove($file);
            }

        } elseif ($temporaryFile) {

            $isNewFile = !$file;

            if ($isNewFile) {
                $file = new $properties['entityName'];
                $file->{$properties['mappedSetter']}($this);
                $this->{$properties['setter']}($file);
                \XLite\Core\Database::getEM()->persist($file);
            }

            if ($temporaryFile->isURL()) {
                $isSaved = $file->loadFromURL($temporaryFile->getPath(), false);

            } else {
                $isSaved = $file->loadFromLocalFile(
                    $temporaryFile->getStoragePath(),
                    pathinfo($temporaryFile->getPath(), \PATHINFO_FILENAME)
                    . '.' . pathinfo($temporaryFile->getPath(), \PATHINFO_EXTENSION)
                );
            }

            $file->postprocessByTemporary($temporaryFile);

            if ($isSaved === false) {
                if ($properties['many']) {
                    $this->{$properties['getter']}()->removeElement($file);

                } else {
                    $this->{$properties['setter']}(null);
                }

                if ($isNewFile) {
                    \XLite\Core\Database::getEM()->detach($file);
                }

                $errorMessage = $file->getLoadErrorMessage() ?: array('File can not be uploaded');

                $file = null;
            }
        }

        if ($file) {
            if (isset($data['alt'])) {
                $file->setterProperty('alt', $data['alt']);
            }

            if (isset($data['position']) && $data['position']) {
                $file->setterProperty('orderby', $data['position']);
            }

            if (
                !(isset($data['delete']) && $data['delete'])
                && ($file instanceof \XLite\Model\Base\Image)
                && !$file->isURL()
            ) {
                $file->prepareSizes();
            }
        }

        if ($temporaryFile) {
            \XLite\Core\Database::getEM()->remove($temporaryFile);
        }

        return $errorMessage;
    }

    /**
     * Clone
     *
     * @return static
     */
    public function cloneEntity()
    {
        $class = $this instanceof \Doctrine\ORM\Proxy\Proxy
            ? get_parent_class($this)
            : get_class($this);

        /** @var static $entity */
        $entity = new $class();

        $fields = $this->cloneFields();
        $map    = array();

        foreach (array_keys($fields) as $field) {
            $map[$field] = $this->$field;
        }

        return $entity->map($map);
    }

    /**
     * Returns cloned plain fields (defined by doctrine metadata). Doesn't clone unique identifier field.
     * 
     * @return array
     */
    protected function cloneFields()
    {
        $fields = $this->getFieldsDefinition();

        $id = $this->getUniqueIdentifierName();
        unset($fields[$id]);

        return $fields;
    }

    /**
     * Get model fields list
     *
     * @param string|null $class
     *
     * @return array
     */
    public function getFieldsDefinition($class = null)
    {
        if (!$class) {
            $class = $this instanceof \Doctrine\ORM\Proxy\Proxy
                ? get_parent_class($this)
                : get_class($this);
        }

        return \XLite\Core\Database::getEM()->getClassMetadata($class)->fieldMappings;
    }

    /**
     * Since Doctrine lifecycle callbacks do not allow to modify associations, we've added this method
     *
     * @param string $type Type of current operation
     *
     * @return void
     */
    public function prepareEntityBeforeCommit($type)
    {
    }

    /**
     * @inheritdoc
     */
    public function isSerializable()
    {
        return null !== $this->getUniqueIdentifier();
    }

    /**
     * @return string
     */
    public function __toString()
    {
        if (LOG_DEBUG == ConfigParser::getOptions(['log_details', 'level'])) {
            $result = serialize($this);
        } else {
            $result = 'EntityName: ' . $this->getEntityName() . ', '
                . 'UniqueIdentifier: ' . ($this->getUniqueIdentifier() ?: 'NULL');
        }

        return $result;
    }
}
