<?php
/**
 * ProfileExtender Plugin.
 *
 * @author Lincoln Russell <lincoln@vanillaforums.com>
 * @copyright 2009-2019 Vanilla Forums Inc.
 * @license http://www.opensource.org/licenses/gpl-2.0.php GNU GPL v2
 * @package ProfileExtender
 */

use Garden\Container\Container;
use Garden\EventManager;
use Garden\Schema\Schema;
use Garden\Web\Data;
use Vanilla\Attributes;
use Vanilla\OpenAPIBuilder;
use Vanilla\Web\APIExpandMiddleware;

/**
 * Plugin to add additional fields to user profiles.
 *
 * If the field name is an existing column on user table (e.g. Title, About, Location)
 * it will store there. Otherwise, it stores in UserMeta.
 *
 * @todo Option to show in discussions
 * @todo Sort order
 * @todo Lockable for Garden.Moderation.Manage
 * @todo Date fields
 * @todo Gender, birthday adding
 * @todo Dynamic magic field filtering/linking
 * @todo Dynamic validation rule
 */
class ProfileExtenderPlugin extends Gdn_Plugin {
    const FIELD_EXTENDED = "extended";

    /** @var array */
    public $MagicLabels = ['Twitter', 'Google', 'Facebook', 'LinkedIn', 'GitHub', 'Instagram', 'Website', 'Real Name'];

    /**
     * Available form field types in format Gdn_Type => DisplayName.
     */
    public $FormTypes = [
        'TextBox' => 'TextBox',
        'Dropdown' => 'Dropdown',
        'CheckBox' => 'Checkbox',
        'DateOfBirth' => 'Birthday',
    ];

    /**
     * Whitelist of allowed field properties.
     */
    public $FieldProperties = ['Name', 'Label', 'FormType', 'Required', 'Locked',
        'Options', 'Length', 'Sort', 'OnRegister', 'OnProfile', 'OnDiscussion', 'SalesForceID'];

    /**
     * Blacklist of disallowed field names.
     * Prevents accidental or malicious overwrite of sensitive fields.
     */
    public $ReservedNames = ['Name', 'Email', 'Password', 'HashMethod', 'Admin', 'Banned', 'Points',
        'Deleted', 'Verified', 'Attributes', 'Permissions', 'Preferences'];

    private const BUILTIN_FIELDS = ['Title', 'Location'];

    /** @var array */
    public $ProfileFields = [];

    /**
     * Hook in before content is rendered.
     *
     * @param mixed $sender
     */
    public function base_render_before($sender) {
        if ($sender->MasterView == 'admin') {
            $sender->addJsFile('profileextender.js', 'plugins/ProfileExtender');
        }
    }

    /**
     * Modify container rules.
     *
     * @param Container $dic
     */
    public function container_init(Container $dic): void {
        $dic->rule(APIExpandMiddleware::class)
            ->addCall(
                "addExpandField",
                [
                    self::FIELD_EXTENDED,
                    [
                        "firstInsertUser.extended" => "firstInsertUserID",
                        "insertUser.extended" => "insertUserID",
                        "lastInsertUser.extended" => "lastInsertUserID",
                        "lastPost.insertUser.extended" => "lastPost.insertUserID",
                        "lastUser.extended" => "lastUserID",
                        "updateUser.extended" => "updateUserID",
                        "user.extended" => "userID",
                        self::FIELD_EXTENDED => "userID",
                    ],
                    [$this, "getUserProfileValuesChecked"],
                ]
            );

        // Add the OpenAPI filter to set the field schema.
        $dic->rule(OpenAPIBuilder::class)
            ->addCall('addFilter', ['filter' => [$this, 'filterOpenApi']]);
    }

    /**
     * Change config settings based on whether Profile Extender fields duplicate built-in fields.
     */
    public function gdn_dispatcher_appStartup_handler() {
        // If profile extender fields replace built-in fields, they should be editable.
        $profileFields = $this->getProfileFields();
        $keys = $profileFields;
        array_walk($keys, function (&$item) {
            $item = $item['Name'];
        });
        if (in_array('Title', $keys, true)) {
            Gdn::config()->set('Garden.Profile.Titles', true, true, false);
        }
        if (in_array('Location', $keys, true)) {
            Gdn::config()->set('Garden.Profile.Locations', true, true, false);
        }
    }

    /**
     * Add the Dashboard menu item.
     *
     * @param Object $sender
     */
    public function base_getAppSettingsMenuItems_handler($sender) {
        $menu = &$sender->EventArguments['SideMenu'];
        $menu->addLink('Users', t('Profile Fields'), 'settings/profileextender', 'Garden.Settings.Manage');
    }

    /**
     * Add non-checkbox fields to registration forms.
     *
     * @param EntryController $sender
     */

    public function entryController_registerBeforePassword_handler($sender) {
        /* @var $form Gdn_Form */
        $form = $sender->Form;
        $isExistingUser = $form->getFormValue('ConnectingExistingUser', false);
        // Never show the Profile Extender fields when someone is reconnecting.
        if ($isExistingUser) {
            return;
        }
        $ProfileFields = $this->getProfileFields();
        $sender->RegistrationFields = [];
        foreach ($ProfileFields as $Name => $Field) {
            if (val('OnRegister', $Field) && val('FormType', $Field) != 'CheckBox') {
                $sender->RegistrationFields[$Name] = $Field;
            }
        }
        include $sender->fetchViewLocation('registrationfields', '', 'plugins/ProfileExtender');
    }

    /**
     * Add checkbox fields to registration forms.
     *
     * @param EntryController $sender
     * @throws Exception If there is an error in the form.
     */
    public function entryController_registerFormBeforeTerms_handler($sender) {
        /* @var $form Gdn_Form */

        $form = $sender->Form;
        $isExistingUser = $form->getFormValue('ConnectingExistingUser', false);
        // Never show the Profile Extender fields when someone is reconnecting.
        if ($isExistingUser) {
            return;
        }
        $ProfileFields = $this->getProfileFields();
        $sender->RegistrationFields = [];
        foreach ($ProfileFields as $Name => $Field) {
            if (val('OnRegister', $Field) && val('FormType', $Field) == 'CheckBox') {
                $sender->RegistrationFields[$Name] = $Field;
            }
        }
        include $sender->fetchViewLocation('registrationfields', '', 'plugins/ProfileExtender');
    }

    /**
     * Required fields on registration forms.
     *
     * @param EntryController $sender
     */
    public function entryController_registerValidation_handler($sender) {
        /* @var $form Gdn_Form */
        $form = $sender->Form;
        $isExistingUser = $form->getFormValue('ConnectingExistingUser', false);
        // Never show the Profile Extender fields when someone is reconnecting.
        if ($isExistingUser) {
            return;
        }
        // Require new fields
        $profileFields = $this->getProfileFields();
        foreach ($profileFields as $key => $field) {
            // Check both so you can't break register form by requiring omitted field
            $name = isset($field['Name']) ? $field['Name'] : (string) $key;
            if (val('Required', $field) && val('OnRegister', $field)) {
                $sender->UserModel->Validation->applyRule($name, 'Required', t('%s is required.', $field['Label']));
            }
        }

        // DateOfBirth zeroes => NULL
        if ('0-00-00' == $sender->Form->getFormValue('DateOfBirth')) {
            $sender->Form->setFormValue('DateOfBirth', null);
        }
    }

    /**
     * Special manipulations.
     */
    public function parseSpecialFields($fields = []) {
        if (!is_array($fields)) {
            return $fields;
        }

        foreach ($fields as $label => $value) {
            if ($value == '') {
                continue;
            }

            // Use plaintext for building these
            $value = Gdn_Format::text($value);

            switch ($label) {
                case 'Twitter':
                    $fields['Twitter'] = '@'.anchor($value, 'http://twitter.com/'.$value);
                    break;
                case 'Facebook':
                    $fields['Facebook'] = anchor($value, 'http://facebook.com/'.$value);
                    break;
                case 'LinkedIn':
                    $fields['LinkedIn'] = anchor($value, 'http://www.linkedin.com/in/'.$value);
                    break;
                case 'GitHub':
                    $fields['GitHub'] = anchor($value, 'https://github.com/'.$value);
                    break;
                case 'Google':
                    $fields['Google'] = anchor('Google+', $value, '', ['rel' => 'me']);
                    break;
                case 'Instagram':
                    $fields['Instagram'] = '@'.anchor($value, 'http://instagram.com/'.$value);
                    break;
                case 'Website':
                    $linkValue = (isUrl($value)) ? $value : 'http://'.$value;
                    $fields['Website'] = anchor($value, $linkValue);
                    break;
                case 'Real Name':
                    $fields['Real Name'] = wrap(htmlspecialchars($value), 'span', ['itemprop' => 'name']);
                    break;
            }
        }

        return $fields;
    }

    /**
     * Add fields to edit profile form.
     *
     * @param ProfileController $sender
     */
    public function profileController_editMyAccountAfter_handler($sender) {
        $this->profileFields($sender, true);
    }

    /**
     * Set the label on the Title and Location fields if there is a ProfileExtender field for them.
     *
     * @param ProfileExtender $sender
     */
    public function profileController_beforeEdit_handler($sender) {
        if (c('ProfileExtender.Fields.Title.Label')) {
            $sender->setData('_TitleLabel', c('ProfileExtender.Fields.Title.Label'));
            // Allow Title field to be a dropdown
            if (c('ProfileExtender.Fields.Title.FormType') === 'Dropdown') {
                $sender->setData('_TitleFormType', 'Dropdown');
                $titleArray = c('ProfileExtender.Fields.Title.Options');
                $titleArray = array_combine($titleArray, $titleArray);
                $sender->setData('_TitleOptions', $titleArray);
            }
        }
        if (c('ProfileExtender.Fields.Location.Label')) {
            $sender->setData('_LocationLabel', c('ProfileExtender.Fields.Location.Label'));
        }
    }

    /**
     * Add custom fields to discussions.
     */
    public function base_authorInfo_handler($sender, $args) {
        //echo ' '.wrapIf(htmlspecialchars(val('Department', $Args['Author'])), 'span', array('class' => 'MItem AuthorDepartment'));
        //echo ' '.wrapIf(htmlspecialchars(val('Organization', $Args['Author'])), 'span', array('class' => 'MItem AuthorOrganization'));
    }

    /**
     * Get custom profile fields.
     *
     * @param bool $stripBuiltinFields Whether to strip out built-in fields replicated in the profileExtender.
     * @return array
     */
    private function getProfileFields($stripBuiltinFields = false) {
        $fields = c('ProfileExtender.Fields', []);
        if (!is_array($fields)) {
            $fields = [];
        }

        // Data checks
        foreach ($fields as $k => $field) {
            $name = isset($field['Name']) ? $field['Name'] : $k;
            // Remove duplicated fields to let the Profile Controller use the default profile fields instead when editing.
            if (in_array($name, self::BUILTIN_FIELDS, true)) {
                if ($stripBuiltinFields) {
                    unset($fields[$k]);
                }
            }

            // Require an array for each field
            if (!is_array($field) || strlen($name) < 1) {
                unset($fields[$k]);
            }

            // Verify field form type
            if (!isset($field['FormType'])) {
                $fields[$k]['FormType'] = 'TextBox';
            } elseif (!array_key_exists($field['FormType'], $this->FormTypes)) {
                unset($this->ProfileFields[$name]);
            } elseif ($fields[$k]['FormType'] == 'DateOfBirth') {
                // Special case for birthday field
                $fields[$k]['FormType'] = 'Date';
                $fields[$k]['Label'] = t('Birthday');
            }
        }

        return $fields;
    }

    /**
     * Get data for a single profile field.
     *
     * @param $name
     * @return array|null
     */
    private function getProfileField($name) {
        $fields = $this->getProfileFields();
        foreach ($fields as $field) {
            if (isset($field['Name']) && $name === $field['Name']) {
                if (!isset($field['FormType'])) {
                    $field['FormType'] = 'TextBox';
                }
                return $field;
            }
        }
        return null;
    }

    /**
     * Display custom profile fields on form.
     *
     * @param Object $Sender
     * @param bool $stripBuiltinFields Whether to strip out built-in fields replicated in the profileExtender.
     * @access private
     */
    private function profileFields($Sender, $stripBuiltinFields = false) {
        $userID = $Sender->Form->getValue('UserID');

        /** @var EventManager $eventManager */
        $eventManager = Gdn::getContainer()->get(EventManager::class);
        // Retrieve user's existing profile fields
        $this->ProfileFields = $this->getProfileFields($stripBuiltinFields);
        $this->ProfileFields = $eventManager->fireFilter("modifyProfileFields", $this->ProfileFields);
        // Get user-specific data
        $userProfileValues = $this->getUserProfileValues([$userID]);
        $this->UserFields = $userProfileValues[$userID] ?? [];

        $this->fireEvent('beforeGetProfileFields');
        // Fill in user data on form
        foreach ($this->UserFields as $Field => $Value) {
            $Sender->Form->setValue($Field, $Value);
        }

        include_once $Sender->fetchViewLocation('profilefields', '', 'plugins/ProfileExtender');
    }

    /**
     * Settings page.
     */
    public function settingsController_profileExtender_create($sender) {
        $sender->permission('Garden.Settings.Manage');
        // Detect if we need to upgrade settings
        if (!c('ProfileExtender.Fields')) {
            $this->setup();
        }

        // Set data
        $data = $this->getProfileFields();
        $sender->setData('ExtendedFields', $data);

        $sender->setHighlightRoute('settings/profileextender');
        $sender->setData('Title', t('Profile Fields'));
        $sender->render('settings', '', 'plugins/ProfileExtender');
    }

    /**
     * Add/edit a field.
     *
     * @param SettingsController $sender
     * @param array $args
     */
    public function settingsController_profileFieldAddEdit_create($sender, $args) {
        $sender->permission('Garden.Settings.Manage');
        $sender->setData('Title', t('Add Profile Field'));

        if ($sender->Form->authenticatedPostBack()) {
            // Get whitelisted properties
            $formPostValues = $sender->Form->formValues();
            foreach ($formPostValues as $key => $value) {
                if (!in_array($key, $this->FieldProperties)) {
                    unset ($formPostValues[$key]);
                }
            }

            // Make Options an array
            if ($options = val('Options', $formPostValues)) {
                $options = explode("\n", preg_replace('/[^\w\s()-]/u', '', $options));
                if (count($options) < 2) {
                    $sender->Form->addError('Must have at least 2 options.', 'Options');
                }
                setValue('Options', $formPostValues, $options);
            }

            // Check label
            if (val('FormType', $formPostValues) == 'DateOfBirth') {
                setValue('Label', $formPostValues, 'DateOfBirth');
            }
            if (!val('Label', $formPostValues)) {
                $sender->Form->addError('Label is required.', 'Label');
            }

            // Check form type
            if (!array_key_exists(val('FormType', $formPostValues), $this->FormTypes)) {
                $sender->Form->addError('Invalid form type.', 'FormType');
            }

            // Merge updated data into config
            $fields = $this->getProfileFields();
            if (!$name = val('Name', $formPostValues)) {
                // Make unique name from label for new fields
                if (unicodeRegexSupport()) {
                    $regex = '/[^\pL\pN]/u';
                } else {
                    $regex = '/[^a-z\d]/i';
                }
                // Make unique slug
                $name = $testSlug = substr(preg_replace($regex, '', val('Label', $formPostValues)), 0, 50);
                $i = 1;

                // Fallback in case the name is empty
                if (empty($name)) {
                    $name = $testSlug = md5($field);
                }
                $keys = $fields;
                array_walk($keys, function (&$item) {
                    $item = $item['Name'];
                });
                while (in_array($name, $keys) || in_array($name, $this->ReservedNames)) {
                    $name = $testSlug.$i++;
                }
            }

            // Save if no errors
            if (!$sender->Form->errorCount()) {
                $data = (array) Gdn::config('ProfileExtender.Fields');
                $formPostValues = (array)$formPostValues;
                $key = null;
                foreach ($data as $k => &$field) {
                    if (isset($field['Name']) && $name === $field['Name']) {
                        $formPostValues = array_merge((array)$field, $formPostValues);
                        $key = $k;
                    }
                }

                if (!isset($formPostValues['Name'])) {
                    $formPostValues['Name'] = $name;
                }

                if (is_null($key)) {
                    $data = array_filter($data);
                    $key = count($data);
                }

                Gdn::config()->saveToConfig('ProfileExtender.Fields.' . $key, $formPostValues);
                $sender->setRedirectTo('/settings/profileextender');
            }
        } elseif (isset($args[0])) {
            // Editing
            $data = $this->getProfileField($args[0]);
            if (isset($data['Options']) && is_array($data['Options'])) {
                $data['Options'] = implode("\n", $data['Options']);
            }
            $sender->Form->setData($data);
            $sender->Form->addHidden('Name', $args[0]);
            $sender->setData('Title', t('Edit Profile Field'));
        }

        $currentFields = $this->getProfileFields();
        $formTypes = $this->FormTypes;

        /**
         * We only allow one DateOfBirth field, since it is a special case.  Remove it as an option if we already
         * have one, unless we're editing the one instance we're allowing.
         */
        if (array_key_exists('DateOfBirth', $currentFields) && $sender->Form->getValue('FormType') != 'DateOfBirth') {
            unset($formTypes['DateOfBirth']);
        }

        $sender->setData('FormTypes', $formTypes);
        $sender->setData('CurrentFields', $currentFields);
        $sender->fireEvent('beforeProfileExtenderAddEditRender');

        $sender->render('addedit', '', 'plugins/ProfileExtender');
    }

    /**
     * Delete a field.
     *
     * @param SettingsController $sender
     * @param array $args
     */
    public function settingsController_profileFieldDelete_create($sender, $args) {
        $sender->permission('Garden.Settings.Manage');
        $sender->setData('Title', 'Delete Field');
        if (isset($args[0])) {
            if ($sender->Form->authenticatedPostBack()) {
                $fields = $this->getProfileFields();
                foreach ($fields as $key => $field) {
                    if (isset($field['Name']) && $field['Name'] === $args[0]) {
                        unset($fields[$key]);
                    }
                }
                $fields = array_values($fields);
                Gdn::config()->set('ProfileExtender.Fields', $fields);
                $sender->setRedirectTo('/settings/profileextender');
            } else {
                $sender->setData('Field', $this->getProfileField($args[0]));
            }
        }
        $sender->render('delete', '', 'plugins/ProfileExtender');
    }

    /**
     * Display custom fields on Edit User form.
     */
    public function userController_afterFormInputs_handler($sender) {
        echo '<ul>';
        $this->profileFields($sender);
        echo '</ul>';
    }

    /**
     * Reorder ProfileFields according to the sequence in config.
     *
     * @param array $profileFields
     * @return array
     */
    public function reorderProfileFields(array $profileFields): array {
        $orderedFields = $this->getProfileFields();
        $orderedFieldsNamesAsKey = array_column($orderedFields, "Name");

        //the new array will have the right order
        $reordered = [];
        foreach ($orderedFieldsNamesAsKey as $name) {
            if (array_key_exists($name, $profileFields)) {
                $reordered[$name] = $profileFields[$name];
            }
        }

        //if the user has fields and they are not in config, we still need to include them
        $leftovers = array_diff_key($profileFields, $reordered);

        return array_merge($reordered, $leftovers);
    }

    /**
     * Display custom fields on Profile.
     *
     * @param UserInfoModule $Sender
     */
    public function userInfoModule_onBasicInfo_handler($Sender) {
        if ($Sender->User->Banned) {
            return;
        }

        try {
            // Get the custom fields
            $ProfileFields = Gdn::userModel()->getMeta($Sender->User->UserID, 'Profile.%', 'Profile.');

            $ProfileFields = $this->reorderProfileFields($ProfileFields);

            Gdn::controller()->setData('ExtendedFields', $ProfileFields);

            // Get allowed GDN_User fields.
            $Blacklist = array_combine($this->ReservedNames, $this->ReservedNames);
            $NativeFields = array_diff_key((array)$Sender->User, $Blacklist);

            // Combine custom fields (GDN_UserMeta) with GDN_User fields.
            // This is OK because we're blacklisting our $ReservedNames AND whitelisting $AllFields below.
            $ProfileFields = array_merge($ProfileFields, $NativeFields);

            // Import from CustomProfileFields if available
            if (!count($ProfileFields) && is_object($Sender->User) && c('Plugins.CustomProfileFields.SuggestedFields', false)) {
                $ProfileFields = Gdn::userModel()->getAttribute($Sender->User->UserID, 'CustomProfileFields', false);
                if ($ProfileFields) {
                    // Migrate to UserMeta & delete original
                    Gdn::userModel()->setMeta($Sender->User->UserID, $ProfileFields, 'Profile.');
                    Gdn::userModel()->saveAttribute($Sender->User->UserID, 'CustomProfileFields', false);
                }
            }

            // Send them off for magic formatting
            $ProfileFields = $this->parseSpecialFields($ProfileFields);

            // Get all field data, error check
            $AllFields = $this->getProfileFields();
            if (!is_array($AllFields) || !is_array($ProfileFields)) {
                return;
            }

            // DateOfBirth is special case that core won't handle
            // Hack it in here instead
            if (c('ProfileExtender.Fields.DateOfBirth.OnProfile')) {
                // Do not use Gdn_Format::Date because it shifts to local timezone
                $BirthdayStamp = Gdn_Format::toTimestamp($Sender->User->DateOfBirth);
                if ($BirthdayStamp) {
                    $ProfileFields['DateOfBirth'] = date(t('Birthday Format', 'F j, Y'), $BirthdayStamp);
                    $AllFields['DateOfBirth'] = ['Label' => t('Birthday'), 'OnProfile' => true];
                }
            }

            // CheckBox fields should display as "Yes" or "No"
            foreach ($AllFields as $name => $data) {
                if ($data['FormType'] === 'CheckBox') {
                    $ProfileFields[$name] = $ProfileFields[$name] == "1" ? t('Profile.Yes', 'Yes') : t('Profile.No', 'No');
                }
            }

            // Display all non-hidden fields
            require_once Gdn::controller()->fetchViewLocation('helper_functions', '', 'plugins/ProfileExtender', true, false);
            extendedProfileFields($ProfileFields, $AllFields, $this->MagicLabels);
        } catch (Exception $ex) {
            // No errors
        }
    }

    /**
     * Validate profile extender fields before saving the user.
     *
     * @param \UserModel $sender
     * @param array $args
     */
    public function userModel_beforeSaveValidation_handler(\UserModel $sender, array $args) {
        $allowedFields = $this->getProfileFields();
        foreach ($allowedFields as $key => $value) {
            $checkField = isset($args['FormPostValues'][$key]) && isset($value['Required']);
            $invalid = $checkField && $value['Required'] == 1 && trim($args['FormPostValues'][$key]) === "";
            if ($invalid) {
                $sender->Validation->addValidationResult($key, sprintf(t('%s is required.'), $key));
            }
        }
    }

    /**
     * Save custom profile fields when saving the user.
     *
     * @param \UserModel $sender
     * @param array $args
     */
    public function userModel_afterSave_handler(\UserModel $sender, array $args) {
        $this->updateUserFields($args['UserID'], $args['FormPostValues']);
    }

    /**
     * Save custom profile fields on registration.
     *
     * @param \UserModel $sender
     * @param array $args
     */
    public function userModel_afterInsertUser_handler(\UserModel $sender, array $args) {
        if (!empty($args['RegisteringUser'])) {
            $this->updateUserFields($args['InsertUserID'], $args['RegisteringUser']);
        }
    }

    /**
     * Update user with new profile fields.
     *
     * @param int $userID The user ID to update.
     * @param array $fields Key/value pairs of fields to update.
     */
    public function updateUserFields($userID, $fields) {
        // Confirm we have submitted form values
        if (is_array($fields)) {
            // Retrieve whitelist & user column list
            $allowedFields = array_column(
                $this->getProfileFields(),
                null,
                'Name'
            );
            $columns = Gdn::sql()->fetchColumns('User');

            foreach ($fields as $name => $field) {
                // Whitelist.
                if (!isset($allowedFields[$name])) {
                    unset($fields[$name]);
                    continue;
                }
                // Don't allow duplicates on User table
                if (in_array($name, $columns)) {
                    unset($fields[$name]);
                }

                // Allowed checkboxes should be 1 or 0.
                if ($allowedFields[$name]['FormType'] === 'CheckBox') {
                    $fields[$name] = $field == true ? 1 : 0;
                }
            }

            // Update UserMeta if any made it through.
            if (count($fields)) {
                Gdn::userModel()->setMeta($userID, $fields, 'Profile.');
            }
        }
    }

    /**
     * Get the profile extender fields for a single user.
     *
     * @param int $userID
     * @return array
     */
    public function getUserFields(int $userID): array {
        $values = $this->getUserProfileValues([$userID]);
        return $values[$userID] ?? [];
    }

    /**
     * Endpoint to export basic user data along with all custom fields into CSV.
     *
     * @param UtilityController $sender
     */
    public function utilityController_exportProfiles_create($sender) {
        // Clear our ability to do this.
        $sender->permission('Garden.Settings.Manage');
        if (Gdn::userModel()->pastUserMegaThreshold()) {
            throw new Gdn_UserException('You have too many users to export automatically.');
        }

        // Determine profile fields we need to add.
        $fields = $this->getProfileFields();
        $columnNames = [
            'Name', 'Email', 'Joined', 'Last Seen', 'LastIPAddress', 'Discussions', 'Comments', 'Visits', 'Points',
            'InviteUserID', 'InvitedByName', 'Location', 'Roles'
        ];

        // Set up our basic query.
        Gdn::sql()
            ->select([
                'u.Name',
                'u.Email',
                'u.DateInserted',
                'u.DateLastActive',
                'inet6_ntoa(u.LastIPAddress)',
                'u.CountDiscussions',
                'u.CountComments',
                'u.CountVisits',
                'u.Points',
                'u.InviteUserID',
                'u2.Name as InvitedByName',
                'u.Location',
                'group_concat(r.Name) as Roles',
            ])
            ->from('User u')
            ->leftJoin('User u2', 'u.InviteUserID = u2.UserID and u.InviteUserID is not null')
            ->join('UserRole ur', 'u.UserID = ur.UserID')
            ->join('Role r', 'r.RoleID = ur.RoleID')
            ->where('u.Deleted', 0)
            ->where('u.Admin <', 2)
            ->groupBy('u.UserID');

        if (val('DateOfBirth', $fields)) {
            $columnNames[] = 'Birthday';
            Gdn::sql()->select('u.DateOfBirth');
            unset($fields['DateOfBirth']);
        }

        if (Gdn::addonManager()->isEnabled('Ranks', \Vanilla\Addon::TYPE_ADDON)) {
            $columnNames[] = 'Rank';
            Gdn::sql()
                ->select('ra.Name as Rank')
                ->leftJoin('Rank ra', 'ra.RankID = u.RankID');
        }

        $lowerCaseColumnNames =  array_map('strtolower', $columnNames);
        $i = 0;
        foreach ($fields as $fieldData) {
            $slugName = $fieldData['Name'];
            // Don't overwrite data if there's already a column with the same name.
            if (in_array(strtolower($slugName), $lowerCaseColumnNames)) {
                continue;
            }
            // Add this field to the output
            $columnNames[] = val('Label', $fieldData, $slugName);

            // Add this field to the query.
            $quoted = Gdn::sql()->quote("Profile.$slugName");
            Gdn::sql()
                ->join('UserMeta a' . $i, "u.UserID = a$i.UserID and a$i.Name = $quoted", 'left')
                ->select('a' . $i . '.Value', '', $slugName);
            $i++;
        }

        // Get our user data.
        $users = Gdn::sql()->get()->resultArray();

        // Serve a CSV of the results.
        exportCSV($columnNames, $users);
        die();

        // Useful for query debug.
        // $sender->render('blank');
    }

    /**
     * Import from CustomProfileFields or upgrade from ProfileExtender 2.0.
     */
    public function setup() {
        if ($fields = c('Plugins.ProfileExtender.ProfileFields', c('Plugins.CustomProfileFields.SuggestedFields'))) {
            // Get defaults
            $hidden = c('Plugins.ProfileExtender.HideFields', c('Plugins.CustomProfileFields.HideFields'));
            $onRegister = c('Plugins.ProfileExtender.RegistrationFields');
            $length = c('Plugins.ProfileExtender.TextMaxLength', c('Plugins.CustomProfileFields.ValueLength'));

            // Convert to arrays
            $fields = array_filter((array)explode(',', $fields));
            $hidden = array_filter((array)explode(',', $hidden));
            $onRegister = array_filter((array)explode(',', $onRegister));

            // Assign new data structure
            $newData = [];
            foreach ($fields as $field) {
                if (unicodeRegexSupport()) {
                    $regex = '/[^\pL\pN]/u';
                } else {
                    $regex = '/[^a-z\d]/i';
                }
                // Make unique slug
                $name = $testSlug = preg_replace($regex, '', $field);
                $i = 1;

                // Fallback in case the name is empty
                if (empty($name)) {
                    $name = $testSlug = md5($field);
                }

                while (array_key_exists($name, $newData) || in_array($name, $this->ReservedNames)) {
                    $name = $testSlug.$i++;
                }

                // Convert
                $newData[] = [
                    'Label' => $field,
                    'Name' => $name,
                    'Length' => $length,
                    'FormType' => 'TextBox',
                    'OnProfile' => (in_array($field, $hidden)) ? 0 : 1,
                    'OnRegister' => (in_array($field, $onRegister)) ? 1 : 0,
                    'OnDiscussion' => 0,
                    'Required' => 0,
                    'Locked' => 0,
                    'Sort' => 0
                ];
            }
            Gdn::config()->saveToConfig('ProfileExtender.Fields', $newData);
        }
    }

    /**
     * Setup structure on update
     */
    public function structure() {
        $profileFields = $this->getProfileFields();
        $updateRequired = false;
        foreach ($profileFields as $k => &$field) {
            if (is_string($k)) {
                $field['Name'] = $k;
                $updateRequired = true;
            }
        }
        if ($updateRequired) {
            Gdn::config()->saveToConfig('ProfileExtender.Fields', array_values($profileFields));
        }
    }

    /**
     * Get the extended values associated with a user.
     *
     * @param int[] $userIDs
     */
    public function getUserProfileValues(array $userIDs): array {
        $result = Gdn::userModel()->getMeta($userIDs, "Profile.%", "Profile.");
        $eventManager = Gdn::getContainer()->get(EventManager::class);
        $result = $eventManager->fireFilter("modifyUserFields", $result);
        return $result;
    }

    /**
     * Get the extended values, but make sure they are defined and cast them to their correct types.
     *
     * @param array $userIDs
     * @return array
     */
    public function getUserProfileValuesChecked(array $userIDs): array {
        $values = $this->getUserProfileValues($userIDs);
        $fields = array_column($this->getProfileFields(), null, 'Name');
        $utc = new DateTimeZone('UTC');
        foreach ($values as $id => &$row) {
            $row = new Attributes(array_intersect_key($row, $fields));
            foreach ($row as $key => &$value) {
                switch ($fields[$key]['FormType'] ?? 'TextBox') {
                    case 'CheckBox':
                        $value = (bool)$value;
                        break;
                    case 'DateOfBirth':
                        try {
                            $value = new DateTimeImmutable($value, $utc);
                        } catch (\Exception $ex) {
                            $value = null;
                        }
                }
            }
        }
        return $values;
    }

    /**
     * Return the schema object that represents the API fields.
     *
     * This schema is used as the input schema for the `PATCH /users/:id/extended` endpoint and for field expansion schema.
     *
     * @param string|null $schemaType
     * @returns Schema
     */
    private function getProfileSchema(?string $schemaType = ''): Schema {
        $fields = array_column($this->getProfileFields(true), null, 'Name');

        // Dynamically build the schema based on the fields and data types.
        $schemaArray = [];

        foreach ($fields as $field) {
            $name = $field['Name'];
            $types = ['CheckBox' => 'b', "Date" => 'dt'];
            $dataType = $types[$field['FormType']] ?? 's';

            if ($field['FormType'] === 'Dropdown') {
                $schemaArray["{$name}:{$dataType}?"] = ['enum' => $field['Options']];
            } else {
                $schemaArray[] = "{$name}:{$dataType}?";
            }
        }

        $schema = Schema::parse($schemaArray);

        return $schema;
    }

    /**
     * The `PATCH /users/:id/extended` endpoint.
     *
     * @param UsersApiController $usersApi
     * @param int $id
     * @param array $body
     * @return \Garden\Web\Data
     */
    public function usersApiController_patch_extended(UsersApiController $usersApi, int $id, array $body): \Garden\Web\Data {
        $userID = $usersApi->getSession()->UserID;
        $userModel = new UserModel();
        if ($id !== $userID) {
            $usersApi->permission('Garden.Users.Edit');
        }
        $in = $this->getProfileSchema('in');
        $out = $this->getProfileSchema('out');
        $body = $in->validate($body, true);
        $schemaArray = $in->getSchemaArray();
        // Special handling of the DateOfBirth field, which lives in the User table.
        $dateOfBirth = isset($schemaArray['properties']['DateOfBirth']);
        if ($dateOfBirth && isset($body['DateOfBirth'])) {
            $dob = $body['DateOfBirth']->format('Y-m-d');
            $userModel->save(['UserID' => $id, 'DateOfBirth' => $dob]);
        }
        $this->updateUserFields($id, $body);
        $row = $this->getUserProfileValuesChecked([$id]);
        // More special handling of DateOfBirth.
        if ($dateOfBirth) {
            $user = $userModel->getId($id);
            $row[$id]->DateOfBirth = $user->DateOfBirth;
        }
        $result = $out->validate($row[$id]);
        $result = new Data($result);
        return $result;
    }

    /**
     * Augment the generated OpenAPI schema with the profile extender fields.
     *
     * Since the profile extender fields are defined at runtime we have to add them to the OpenAPI schema dynamically.
     *
     * @param array $openApi
     */
    public function filterOpenApi(array &$openApi): void {
        $schema = $this->getProfileSchema('out');

        \Vanilla\Utility\ArrayUtils::setByPath(
            'components.schemas.ExtendedUserFields.properties',
            $openApi,
            $schema->jsonSerialize()['properties']
        );
    }
}

// 2.0 used these config settings; the first 3 were a comma-separated list of field names.
//'Plugins.ProfileExtender.ProfileFields' => array('Control' => 'TextBox', 'Options' => array('MultiLine' => TRUE)),
//'Plugins.ProfileExtender.RegistrationFields' => array('Control' => 'TextBox', 'Options' => array('MultiLine' => TRUE)),
//'Plugins.ProfileExtender.HideFields' => array('Control' => 'TextBox', 'Options' => array('MultiLine' => TRUE)),
//'Plugins.ProfileExtender.TextMaxLength' => array('Control' => 'TextBox'),
