<?php
/**
 * KKART-API endpoint handler.
 *
 * This handles API related functionality in Kkart.
 * - kkart-api endpoint - Commonly used by Payment gateways for callbacks.
 * - Legacy REST API - Deprecated in 2.6.0. @see class-kkart-legacy-api.php
 * - WP REST API - The main REST API in Kkart which is built on top of the WP REST API.
 *
 * @package Kkart\RestApi
 * @since   2.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * KKART_API class.
 */
class KKART_API extends KKART_Legacy_API {

	/**
	 * Init the API by setting up action and filter hooks.
	 */
	public function init() {
		parent::init();
		add_action( 'init', array( $this, 'add_endpoint' ), 0 );
		add_filter( 'query_vars', array( $this, 'add_query_vars' ), 0 );
		add_action( 'parse_request', array( $this, 'handle_api_requests' ), 0 );
		add_action( 'rest_api_init', array( $this, 'register_wp_admin_settings' ) );
	}

	/**
	 * Get the version of the REST API package being ran. Since API package was merged into core, this now follows KKART version.
	 *
	 * @since 3.7.0
	 * @return string|null
	 */
	public function get_rest_api_package_version() {
		if ( ! $this->is_rest_api_loaded() ) {
			return null;
		}
		if ( method_exists( \Automattic\Kkart\RestApi\Server::class, 'get_path' ) ) {
			$path = \Automattic\Kkart\RestApi\Server::get_path();
			if ( 0 === strpos( $path, __DIR__ ) ) {
				// We are loading API from included version.
				return KKART()->version;
			}
		}
		// We are loading API from external plugin.
		return \Automattic\Kkart\RestApi\Package::get_version();
	}

	/**
	 * Get the version of the REST API package being ran.
	 *
	 * @since 3.7.0
	 * @return string
	 */
	public function get_rest_api_package_path() {
		if ( ! $this->is_rest_api_loaded() ) {
			return null;
		}
		if ( method_exists( \Automattic\Kkart\RestApi\Server::class, 'get_path' ) ) {
			// We are loading API from included version.
			return \Automattic\Kkart\RestApi\Server::get_path();
		}
		// We are loading API from external plugin.
		return \Automattic\Kkart\RestApi\Package::get_path();
	}

	/**
	 * Return if the rest API classes were already loaded.
	 *
	 * @since 3.7.0
	 * @return boolean
	 */
	protected function is_rest_api_loaded() {
		return class_exists( '\Automattic\Kkart\RestApi\Server', false );
	}

	/**
	 * Get data from a Kkart API endpoint.
	 *
	 * @since 3.7.0
	 * @param string $endpoint Endpoint.
	 * @param array  $params Params to passwith request.
	 * @return array|\WP_Error
	 */
	public function get_endpoint_data( $endpoint, $params = array() ) {
		if ( ! $this->is_rest_api_loaded() ) {
			return new WP_Error( 'rest_api_unavailable', __( 'The Rest API is unavailable.', 'kkart' ) );
		}
		$request = new \WP_REST_Request( 'GET', $endpoint );
		if ( $params ) {
			$request->set_query_params( $params );
		}
		$response = rest_do_request( $request );
		$server   = rest_get_server();
		$json     = wp_json_encode( $server->response_to_data( $response, false ) );
		return json_decode( $json, true );
	}

	/**
	 * Add new query vars.
	 *
	 * @since 2.0
	 * @param array $vars Query vars.
	 * @return string[]
	 */
	public function add_query_vars( $vars ) {
		$vars   = parent::add_query_vars( $vars );
		$vars[] = 'kkart-api';
		return $vars;
	}

	/**
	 * KKART API for payment gateway IPNs, etc.
	 *
	 * @since 2.0
	 */
	public static function add_endpoint() {
		parent::add_endpoint();
		add_rewrite_endpoint( 'kkart-api', EP_ALL );
	}

	/**
	 * API request - Trigger any API requests.
	 *
	 * @since   2.0
	 * @version 2.4
	 */
	public function handle_api_requests() {
		global $wp;

		if ( ! empty( $_GET['kkart-api'] ) ) { // WPCS: input var okay, CSRF ok.
			$wp->query_vars['kkart-api'] = sanitize_key( wp_unslash( $_GET['kkart-api'] ) ); // WPCS: input var okay, CSRF ok.
		}

		// kkart-api endpoint requests.
		if ( ! empty( $wp->query_vars['kkart-api'] ) ) {

			// Buffer, we won't want any output here.
			ob_start();

			// No cache headers.
			kkart_nocache_headers();

			// Clean the API request.
			$api_request = strtolower( kkart_clean( $wp->query_vars['kkart-api'] ) );

			// Make sure gateways are available for request.
			KKART()->payment_gateways();

			// Trigger generic action before request hook.
			do_action( 'kkart_api_request', $api_request );

			// Is there actually something hooked into this API request? If not trigger 400 - Bad request.
			status_header( has_action( 'kkart_api_' . $api_request ) ? 200 : 400 );

			// Trigger an action which plugins can hook into to fulfill the request.
			do_action( 'kkart_api_' . $api_request );

			// Done, clear buffer and exit.
			ob_end_clean();
			die( '-1' );
		}
	}

	/**
	 * Register KKART settings from WP-API to the REST API.
	 *
	 * @since  3.0.0
	 */
	public function register_wp_admin_settings() {
		$pages = KKART_Admin_Settings::get_settings_pages();
		foreach ( $pages as $page ) {
			new KKART_Register_WP_Admin_Settings( $page, 'page' );
		}

		$emails = KKART_Emails::instance();
		foreach ( $emails->get_emails() as $email ) {
			new KKART_Register_WP_Admin_Settings( $email, 'email' );
		}
	}
}
