/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { Fragment } from '@wordpress/element';
import { InspectorControls } from '@wordpress/block-editor';
import {
	Placeholder,
	Disabled,
	PanelBody,
	ToggleControl,
	Button,
} from '@wordpress/components';
import { PRODUCT_COUNT } from '@kkart/block-settings';
import { getAdminLink } from '@kkart/settings';
import HeadingToolbar from '@kkart/editor-components/heading-toolbar';
import BlockTitle from '@kkart/editor-components/block-title';
import ToggleButtonControl from '@kkart/editor-components/toggle-button-control';
import { Icon, bill, external } from '@kkart/icons';

/**
 * Internal dependencies
 */
import Block from './block.js';
import './editor.scss';

export default function ( { attributes, setAttributes } ) {
	const {
		className,
		heading,
		headingLevel,
		showInputFields,
		showFilterButton,
	} = attributes;

	const getInspectorControls = () => {
		return (
			<InspectorControls key="inspector">
				<PanelBody
					title={ __(
						'Block Settings',
						'kkart'
					) }
				>
					<ToggleButtonControl
						label={ __(
							'Price Range',
							'kkart'
						) }
						value={ showInputFields ? 'editable' : 'text' }
						options={ [
							{
								label: __(
									'Editable',
									'kkart'
								),
								value: 'editable',
							},
							{
								label: __(
									'Text',
									'kkart'
								),
								value: 'text',
							},
						] }
						onChange={ ( value ) =>
							setAttributes( {
								showInputFields: value === 'editable',
							} )
						}
					/>
					<ToggleControl
						label={ __(
							'Filter button',
							'kkart'
						) }
						help={
							showFilterButton
								? __(
										'Products will only update when the button is pressed.',
										'kkart'
								  )
								: __(
										'Products will update when the slider is moved.',
										'kkart'
								  )
						}
						checked={ showFilterButton }
						onChange={ () =>
							setAttributes( {
								showFilterButton: ! showFilterButton,
							} )
						}
					/>
					<p>
						{ __(
							'Heading Level',
							'kkart'
						) }
					</p>
					<HeadingToolbar
						isCollapsed={ false }
						minLevel={ 2 }
						maxLevel={ 7 }
						selectedLevel={ headingLevel }
						onChange={ ( newLevel ) =>
							setAttributes( { headingLevel: newLevel } )
						}
					/>
				</PanelBody>
			</InspectorControls>
		);
	};

	const noProductsPlaceholder = () => (
		<Placeholder
			className="kkart-block-price-slider"
			icon={ <Icon srcElement={ bill } /> }
			label={ __(
				'Filter Products by Price',
				'kkart'
			) }
			instructions={ __(
				'Display a slider to filter products in your store by price.',
				'kkart'
			) }
		>
			<p>
				{ __(
					"Products with prices are needed for filtering by price. You haven't created any products yet.",
					'kkart'
				) }
			</p>
			<Button
				className="kkart-block-price-slider__add-product-button"
				isDefault
				isLarge
				href={ getAdminLink( 'post-new.php?post_type=product' ) }
			>
				{ __( 'Add new product', 'kkart' ) +
					' ' }
				<Icon srcElement={ external } />
			</Button>
			<Button
				className="kkart-block-price-slider__read_more_button"
				isTertiary
				href="https://docs.kkart.com/document/managing-products/"
			>
				{ __( 'Learn more', 'kkart' ) }
			</Button>
		</Placeholder>
	);

	return (
		<Fragment>
			{ PRODUCT_COUNT === 0 ? (
				noProductsPlaceholder()
			) : (
				<div className={ className }>
					{ getInspectorControls() }
					<BlockTitle
						headingLevel={ headingLevel }
						heading={ heading }
						onChange={ ( value ) =>
							setAttributes( { heading: value } )
						}
					/>
					<Disabled>
						<Block attributes={ attributes } isEditor={ true } />
					</Disabled>
				</div>
			) }
		</Fragment>
	);
}
