<?php
/**
 * Load assets
 *
 * @package Kkart\Admin
 * @version 3.7.0
 */

use Automattic\Jetpack\Constants;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'KKART_Admin_Assets', false ) ) :

	/**
	 * KKART_Admin_Assets Class.
	 */
	class KKART_Admin_Assets {

		/**
		 * Hook in tabs.
		 */
		public function __construct() {
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_styles' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		}

		/**
		 * Enqueue styles.
		 */
		public function admin_styles() {
			global $wp_scripts;

			$version   = Constants::get_constant( 'KKART_VERSION' );
			$screen    = get_current_screen();
			$screen_id = $screen ? $screen->id : '';

			// Register admin styles.
			wp_register_style( 'kkart_admin_menu_styles', KKART()->plugin_url() . '/assets/css/menu.css', array(), $version );
			wp_register_style( 'kkart_admin_styles', KKART()->plugin_url() . '/assets/css/admin.css', array(), $version );
			wp_register_style( 'jquery-ui-style', KKART()->plugin_url() . '/assets/css/jquery-ui/jquery-ui.min.css', array(), $version );
			wp_register_style( 'kkart_admin_dashboard_styles', KKART()->plugin_url() . '/assets/css/dashboard.css', array(), $version );
			wp_register_style( 'kkart_admin_print_reports_styles', KKART()->plugin_url() . '/assets/css/reports-print.css', array(), $version, 'print' );
			wp_register_style( 'kkart_admin_marketplace_styles', KKART()->plugin_url() . '/assets/css/marketplace-suggestions.css', array(), $version );
			wp_register_style( 'kkart_admin_privacy_styles', KKART()->plugin_url() . '/assets/css/privacy.css', array(), $version );

			// Add RTL support for admin styles.
			wp_style_add_data( 'kkart_admin_menu_styles', 'rtl', 'replace' );
			wp_style_add_data( 'kkart_admin_styles', 'rtl', 'replace' );
			wp_style_add_data( 'kkart_admin_dashboard_styles', 'rtl', 'replace' );
			wp_style_add_data( 'kkart_admin_print_reports_styles', 'rtl', 'replace' );
			wp_style_add_data( 'kkart_admin_marketplace_styles', 'rtl', 'replace' );
			wp_style_add_data( 'kkart_admin_privacy_styles', 'rtl', 'replace' );

			// Sitewide menu CSS.
			wp_enqueue_style( 'kkart_admin_menu_styles' );

			// Admin styles for KKART pages only.
			if ( in_array( $screen_id, kkart_get_screen_ids() ) ) {
				wp_enqueue_style( 'kkart_admin_styles' );
				wp_enqueue_style( 'jquery-ui-style' );
				wp_enqueue_style( 'wp-color-picker' );
			}

			if ( in_array( $screen_id, array( 'dashboard' ) ) ) {
				wp_enqueue_style( 'kkart_admin_dashboard_styles' );
			}

			if ( in_array( $screen_id, array( KKART_SCREEN_PREFIX.'_page_kkart-reports', 'toplevel_page_kkart-reports' ) ) ) {
				wp_enqueue_style( 'kkart_admin_print_reports_styles' );
			}

			// Privacy Policy Guide css for back-compat.
			if ( isset( $_GET['wp-privacy-policy-guide'] ) || in_array( $screen_id, array( 'privacy-policy-guide' ) ) ) {
				wp_enqueue_style( 'kkart_admin_privacy_styles' );
			}

			// @deprecated 2.3.
			if ( has_action( 'kkart_admin_css' ) ) {
				do_action( 'kkart_admin_css' );
				kkart_deprecated_function( 'The kkart_admin_css action', '2.3', 'admin_enqueue_scripts' );
			}

			if ( KKART_Marketplace_Suggestions::show_suggestions_for_screen( $screen_id ) ) {
				wp_enqueue_style( 'kkart_admin_marketplace_styles' );
			}
		}


		/**
		 * Enqueue scripts.
		 */
		public function admin_scripts() {
			global $wp_query, $post;

			$screen       = get_current_screen();
			$screen_id    = $screen ? $screen->id : '';
			$kkart_screen_id = KKART_SCREEN_PREFIX;
			$suffix       = Constants::is_true( 'SCRIPT_DEBUG' ) ? '' : '.min';
			$version      = Constants::get_constant( 'KKART_VERSION' );

			// Register scripts.
			wp_register_script( 'kkart_admin', KKART()->plugin_url() . '/assets/js/admin/kkart_admin' . $suffix . '.js', array( 'jquery', 'jquery-blockui', 'jquery-ui-sortable', 'jquery-ui-widget', 'jquery-ui-core', 'jquery-tiptip' ), $version );
			wp_register_script( 'jquery-blockui', KKART()->plugin_url() . '/assets/js/jquery-blockui/jquery.blockUI' . $suffix . '.js', array( 'jquery' ), '2.70', true );
			wp_register_script( 'jquery-tiptip', KKART()->plugin_url() . '/assets/js/jquery-tiptip/jquery.tipTip' . $suffix . '.js', array( 'jquery' ), $version, true );
			wp_register_script( 'round', KKART()->plugin_url() . '/assets/js/round/round' . $suffix . '.js', array( 'jquery' ), $version );
			wp_register_script( 'kkart-admin-meta-boxes', KKART()->plugin_url() . '/assets/js/admin/meta-boxes' . $suffix . '.js', array( 'jquery', 'jquery-ui-datepicker', 'jquery-ui-sortable', 'accounting', 'round', 'kkart-enhanced-select', 'plupload-all', 'stupidtable', 'jquery-tiptip' ), $version );
			wp_register_script( 'zeroclipboard', KKART()->plugin_url() . '/assets/js/zeroclipboard/jquery.zeroclipboard' . $suffix . '.js', array( 'jquery' ), $version );
			wp_register_script( 'qrcode', KKART()->plugin_url() . '/assets/js/jquery-qrcode/jquery.qrcode' . $suffix . '.js', array( 'jquery' ), $version );
			wp_register_script( 'stupidtable', KKART()->plugin_url() . '/assets/js/stupidtable/stupidtable' . $suffix . '.js', array( 'jquery' ), $version );
			wp_register_script( 'serializejson', KKART()->plugin_url() . '/assets/js/jquery-serializejson/jquery.serializejson' . $suffix . '.js', array( 'jquery' ), '2.8.1' );
			wp_register_script( 'flot', KKART()->plugin_url() . '/assets/js/jquery-flot/jquery.flot' . $suffix . '.js', array( 'jquery' ), $version );
			wp_register_script( 'flot-resize', KKART()->plugin_url() . '/assets/js/jquery-flot/jquery.flot.resize' . $suffix . '.js', array( 'jquery', 'flot' ), $version );
			wp_register_script( 'flot-time', KKART()->plugin_url() . '/assets/js/jquery-flot/jquery.flot.time' . $suffix . '.js', array( 'jquery', 'flot' ), $version );
			wp_register_script( 'flot-pie', KKART()->plugin_url() . '/assets/js/jquery-flot/jquery.flot.pie' . $suffix . '.js', array( 'jquery', 'flot' ), $version );
			wp_register_script( 'flot-stack', KKART()->plugin_url() . '/assets/js/jquery-flot/jquery.flot.stack' . $suffix . '.js', array( 'jquery', 'flot' ), $version );
			wp_register_script( 'kkart-settings-tax', KKART()->plugin_url() . '/assets/js/admin/settings-views-html-settings-tax' . $suffix . '.js', array( 'jquery', 'wp-util', 'underscore', 'backbone', 'jquery-blockui' ), $version );
			wp_register_script( 'kkart-backbone-modal', KKART()->plugin_url() . '/assets/js/admin/backbone-modal' . $suffix . '.js', array( 'underscore', 'backbone', 'wp-util' ), $version );
			wp_register_script( 'kkart-shipping-zones', KKART()->plugin_url() . '/assets/js/admin/kkart-shipping-zones' . $suffix . '.js', array( 'jquery', 'wp-util', 'underscore', 'backbone', 'jquery-ui-sortable', 'kkart-enhanced-select', 'kkart-backbone-modal' ), $version );
			wp_register_script( 'kkart-shipping-zone-methods', KKART()->plugin_url() . '/assets/js/admin/kkart-shipping-zone-methods' . $suffix . '.js', array( 'jquery', 'wp-util', 'underscore', 'backbone', 'jquery-ui-sortable', 'kkart-backbone-modal' ), $version );
			wp_register_script( 'kkart-shipping-classes', KKART()->plugin_url() . '/assets/js/admin/kkart-shipping-classes' . $suffix . '.js', array( 'jquery', 'wp-util', 'underscore', 'backbone' ), $version );
			wp_register_script( 'kkart-clipboard', KKART()->plugin_url() . '/assets/js/admin/kkart-clipboard' . $suffix . '.js', array( 'jquery' ), $version );
			wp_register_script( 'select2', KKART()->plugin_url() . '/assets/js/select2/select2.full' . $suffix . '.js', array( 'jquery' ), '4.0.3' );
			wp_register_script( 'selectWoo', KKART()->plugin_url() . '/assets/js/selectWoo/selectWoo.full' . $suffix . '.js', array( 'jquery' ), '1.0.6' );
			wp_register_script( 'kkart-enhanced-select', KKART()->plugin_url() . '/assets/js/admin/kkart-enhanced-select' . $suffix . '.js', array( 'jquery', 'selectWoo' ), $version );
			wp_register_script( 'js-cookie', KKART()->plugin_url() . '/assets/js/js-cookie/js.cookie' . $suffix . '.js', array(), '2.1.4', true );

			wp_localize_script(
				'kkart-enhanced-select',
				'kkart_enhanced_select_params',
				array(
					'i18n_no_matches'           => _x( 'No matches found', 'enhanced select', 'kkart' ),
					'i18n_ajax_error'           => _x( 'Loading failed', 'enhanced select', 'kkart' ),
					'i18n_input_too_short_1'    => _x( 'Please enter 1 or more characters', 'enhanced select', 'kkart' ),
					'i18n_input_too_short_n'    => _x( 'Please enter %qty% or more characters', 'enhanced select', 'kkart' ),
					'i18n_input_too_long_1'     => _x( 'Please delete 1 character', 'enhanced select', 'kkart' ),
					'i18n_input_too_long_n'     => _x( 'Please delete %qty% characters', 'enhanced select', 'kkart' ),
					'i18n_selection_too_long_1' => _x( 'You can only select 1 item', 'enhanced select', 'kkart' ),
					'i18n_selection_too_long_n' => _x( 'You can only select %qty% items', 'enhanced select', 'kkart' ),
					'i18n_load_more'            => _x( 'Loading more results&hellip;', 'enhanced select', 'kkart' ),
					'i18n_searching'            => _x( 'Searching&hellip;', 'enhanced select', 'kkart' ),
					'ajax_url'                  => admin_url( 'admin-ajax.php' ),
					'search_products_nonce'     => wp_create_nonce( 'search-products' ),
					'search_customers_nonce'    => wp_create_nonce( 'search-customers' ),
					'search_categories_nonce'   => wp_create_nonce( 'search-categories' ),
				)
			);

			wp_register_script( 'accounting', KKART()->plugin_url() . '/assets/js/accounting/accounting' . $suffix . '.js', array( 'jquery' ), '0.4.2' );
			wp_localize_script(
				'accounting',
				'accounting_params',
				array(
					'mon_decimal_point' => kkart_get_price_decimal_separator(),
				)
			);

			wp_register_script( 'kkart-orders', KKART()->plugin_url() . '/assets/js/admin/kkart-orders' . $suffix . '.js', array( 'jquery', 'wp-util', 'underscore', 'backbone', 'jquery-blockui' ), $version );
			wp_localize_script(
				'kkart-orders',
				'kkart_orders_params',
				array(
					'ajax_url'      => admin_url( 'admin-ajax.php' ),
					'preview_nonce' => wp_create_nonce( 'kkart-preview-order' ),
				)
			);

			// Kkart admin pages.
			if ( in_array( $screen_id, kkart_get_screen_ids() ) ) {
				wp_enqueue_script( 'iris' );
				wp_enqueue_script( 'kkart_admin' );
				wp_enqueue_script( 'kkart-enhanced-select' );
				wp_enqueue_script( 'jquery-ui-sortable' );
				wp_enqueue_script( 'jquery-ui-autocomplete' );

				$locale  = localeconv();
				$decimal = isset( $locale['decimal_point'] ) ? $locale['decimal_point'] : '.';

				$params = array(
					/* translators: %s: decimal */
					'i18n_decimal_error'                => sprintf( __( 'Please enter with one decimal point (%s) without thousand separators.', 'kkart' ), $decimal ),
					/* translators: %s: price decimal separator */
					'i18n_mon_decimal_error'            => sprintf( __( 'Please enter with one monetary decimal point (%s) without thousand separators and currency symbols.', 'kkart' ), kkart_get_price_decimal_separator() ),
					'i18n_country_iso_error'            => __( 'Please enter in country code with two capital letters.', 'kkart' ),
					'i18n_sale_less_than_regular_error' => __( 'Please enter in a value less than the regular price.', 'kkart' ),
					'i18n_delete_product_notice'        => __( 'This product has produced sales and may be linked to existing orders. Are you sure you want to delete it?', 'kkart' ),
					'i18n_remove_personal_data_notice'  => __( 'This action cannot be reversed. Are you sure you wish to erase personal data from the selected orders?', 'kkart' ),
					'decimal_point'                     => $decimal,
					'mon_decimal_point'                 => kkart_get_price_decimal_separator(),
					'ajax_url'                          => admin_url( 'admin-ajax.php' ),
					'strings'                           => array(
						'import_products' => __( 'Import', 'kkart' ),
						'export_products' => __( 'Export', 'kkart' ),
					),
					'nonces'                            => array(
						'gateway_toggle' => wp_create_nonce( 'kkart-toggle-payment-gateway-enabled' ),
					),
					'urls'                              => array(
						'import_products' => current_user_can( 'import' ) ? esc_url_raw( admin_url( 'edit.php?post_type=product&page=product_importer' ) ) : null,
						'export_products' => current_user_can( 'export' ) ? esc_url_raw( admin_url( 'edit.php?post_type=product&page=product_exporter' ) ) : null,
					),
				);

				wp_localize_script( 'kkart_admin', 'kkart_admin', $params );
			}

			// Edit product category pages.
			if ( in_array( $screen_id, array( 'edit-product_cat' ) ) ) {
				wp_enqueue_media();
			}

			// Products.
			if ( in_array( $screen_id, array( 'edit-product' ) ) ) {
				wp_enqueue_script( 'kkart_quick-edit', KKART()->plugin_url() . '/assets/js/admin/quick-edit' . $suffix . '.js', array( 'jquery', 'kkart_admin' ), $version );

				$params = array(
					'strings' => array(
						'allow_reviews' => esc_js( __( 'Enable reviews', 'kkart' ) ),
					),
				);

				wp_localize_script( 'kkart_quick-edit', 'kkart_quick_edit', $params );
			}

			// Meta boxes.
			if ( in_array( $screen_id, array( 'product', 'edit-product' ) ) ) {
				wp_enqueue_media();
				wp_register_script( 'kkart-admin-product-meta-boxes', KKART()->plugin_url() . '/assets/js/admin/meta-boxes-product' . $suffix . '.js', array( 'kkart-admin-meta-boxes', 'media-models' ), $version );
				wp_register_script( 'kkart-admin-variation-meta-boxes', KKART()->plugin_url() . '/assets/js/admin/meta-boxes-product-variation' . $suffix . '.js', array( 'kkart-admin-meta-boxes', 'serializejson', 'media-models' ), $version );

				wp_enqueue_script( 'kkart-admin-product-meta-boxes' );
				wp_enqueue_script( 'kkart-admin-variation-meta-boxes' );

				$params = array(
					'post_id'                             => isset( $post->ID ) ? $post->ID : '',
					'plugin_url'                          => KKART()->plugin_url(),
					'ajax_url'                            => admin_url( 'admin-ajax.php' ),
					'kkart_placeholder_img_src'     => kkart_placeholder_img_src(),
					'add_variation_nonce'                 => wp_create_nonce( 'add-variation' ),
					'link_variation_nonce'                => wp_create_nonce( 'link-variations' ),
					'delete_variations_nonce'             => wp_create_nonce( 'delete-variations' ),
					'load_variations_nonce'               => wp_create_nonce( 'load-variations' ),
					'save_variations_nonce'               => wp_create_nonce( 'save-variations' ),
					'bulk_edit_variations_nonce'          => wp_create_nonce( 'bulk-edit-variations' ),
					/* translators: %d: Number of variations */
					'i18n_link_all_variations'            => esc_js( sprintf( __( 'Are you sure you want to link all variations? This will create a new variation for each and every possible combination of variation attributes (max %d per run).', 'kkart' ), Constants::is_defined( 'KKART_MAX_LINKED_VARIATIONS' ) ? Constants::get_constant( 'KKART_MAX_LINKED_VARIATIONS' ) : 50 ) ),
					'i18n_enter_a_value'                  => esc_js( __( 'Enter a value', 'kkart' ) ),
					'i18n_enter_menu_order'               => esc_js( __( 'Variation menu order (determines position in the list of variations)', 'kkart' ) ),
					'i18n_enter_a_value_fixed_or_percent' => esc_js( __( 'Enter a value (fixed or %)', 'kkart' ) ),
					'i18n_delete_all_variations'          => esc_js( __( 'Are you sure you want to delete all variations? This cannot be undone.', 'kkart' ) ),
					'i18n_last_warning'                   => esc_js( __( 'Last warning, are you sure?', 'kkart' ) ),
					'i18n_choose_image'                   => esc_js( __( 'Choose an image', 'kkart' ) ),
					'i18n_set_image'                      => esc_js( __( 'Set variation image', 'kkart' ) ),
					'i18n_variation_added'                => esc_js( __( 'variation added', 'kkart' ) ),
					'i18n_variations_added'               => esc_js( __( 'variations added', 'kkart' ) ),
					'i18n_no_variations_added'            => esc_js( __( 'No variations added', 'kkart' ) ),
					'i18n_remove_variation'               => esc_js( __( 'Are you sure you want to remove this variation?', 'kkart' ) ),
					'i18n_scheduled_sale_start'           => esc_js( __( 'Sale start date (YYYY-MM-DD format or leave blank)', 'kkart' ) ),
					'i18n_scheduled_sale_end'             => esc_js( __( 'Sale end date (YYYY-MM-DD format or leave blank)', 'kkart' ) ),
					'i18n_edited_variations'              => esc_js( __( 'Save changes before changing page?', 'kkart' ) ),
					'i18n_variation_count_single'         => esc_js( __( '%qty% variation', 'kkart' ) ),
					'i18n_variation_count_plural'         => esc_js( __( '%qty% variations', 'kkart' ) ),
					'variations_per_page'                 => absint( apply_filters( 'kkart_admin_meta_boxes_variations_per_page', 15 ) ),
				);

				wp_localize_script( 'kkart-admin-variation-meta-boxes', 'kkart_admin_meta_boxes_variations', $params );
			}
			if ( in_array( str_replace( 'edit-', '', $screen_id ), kkart_get_order_types( 'order-meta-boxes' ) ) ) {
				$default_location = kkart_get_customer_default_location();

				wp_enqueue_script( 'kkart-admin-order-meta-boxes', KKART()->plugin_url() . '/assets/js/admin/meta-boxes-order' . $suffix . '.js', array( 'kkart-admin-meta-boxes', 'kkart-backbone-modal', 'selectWoo', 'kkart-clipboard' ), $version );
				wp_localize_script(
					'kkart-admin-order-meta-boxes',
					'kkart_admin_meta_boxes_order',
					array(
						'countries'              => wp_json_encode( array_merge( KKART()->countries->get_allowed_country_states(), KKART()->countries->get_shipping_country_states() ) ),
						'i18n_select_state_text' => esc_attr__( 'Select an option&hellip;', 'kkart' ),
						'default_country'        => isset( $default_location['country'] ) ? $default_location['country'] : '',
						'default_state'          => isset( $default_location['state'] ) ? $default_location['state'] : '',
						'placeholder_name'       => esc_attr__( 'Name (required)', 'kkart' ),
						'placeholder_value'      => esc_attr__( 'Value (required)', 'kkart' ),
					)
				);
			}
			if ( in_array( $screen_id, array( 'shop_coupon', 'edit-shop_coupon' ) ) ) {
				wp_enqueue_script( 'kkart-admin-coupon-meta-boxes', KKART()->plugin_url() . '/assets/js/admin/meta-boxes-coupon' . $suffix . '.js', array( 'kkart-admin-meta-boxes' ), $version );
				wp_localize_script(
					'kkart-admin-coupon-meta-boxes',
					'kkart_admin_meta_boxes_coupon',
					array(
						'generate_button_text' => esc_html__( 'Generate coupon code', 'kkart' ),
						'characters'           => apply_filters( 'kkart_coupon_code_generator_characters', 'ABCDEFGHJKMNPQRSTUVWXYZ23456789' ),
						'char_length'          => apply_filters( 'kkart_coupon_code_generator_character_length', 8 ),
						'prefix'               => apply_filters( 'kkart_coupon_code_generator_prefix', '' ),
						'suffix'               => apply_filters( 'kkart_coupon_code_generator_suffix', '' ),
					)
				);
			}
			if ( in_array( str_replace( 'edit-', '', $screen_id ), array_merge( array( 'shop_coupon', 'product' ), kkart_get_order_types( 'order-meta-boxes' ) ) ) ) {
				$post_id            = isset( $post->ID ) ? $post->ID : '';
				$currency           = '';
				$remove_item_notice = __( 'Are you sure you want to remove the selected items?', 'kkart' );

				if ( $post_id && in_array( get_post_type( $post_id ), kkart_get_order_types( 'order-meta-boxes' ) ) ) {
					$order = kkart_get_order( $post_id );
					if ( $order ) {
						$currency = $order->get_currency();

						if ( ! $order->has_status( array( 'pending', 'failed', 'cancelled' ) ) ) {
							$remove_item_notice = $remove_item_notice . ' ' . __( "You may need to manually restore the item's stock.", 'kkart' );
						}
					}
				}

				$params = array(
					'remove_item_notice'            => $remove_item_notice,
					'i18n_select_items'             => __( 'Please select some items.', 'kkart' ),
					'i18n_do_refund'                => __( 'Are you sure you wish to process this refund? This action cannot be undone.', 'kkart' ),
					'i18n_delete_refund'            => __( 'Are you sure you wish to delete this refund? This action cannot be undone.', 'kkart' ),
					'i18n_delete_tax'               => __( 'Are you sure you wish to delete this tax column? This action cannot be undone.', 'kkart' ),
					'remove_item_meta'              => __( 'Remove this item meta?', 'kkart' ),
					'remove_attribute'              => __( 'Remove this attribute?', 'kkart' ),
					'name_label'                    => __( 'Name', 'kkart' ),
					'remove_label'                  => __( 'Remove', 'kkart' ),
					'click_to_toggle'               => __( 'Click to toggle', 'kkart' ),
					'values_label'                  => __( 'Value(s)', 'kkart' ),
					'text_attribute_tip'            => __( 'Enter some text, or some attributes by pipe (|) separating values.', 'kkart' ),
					'visible_label'                 => __( 'Visible on the product page', 'kkart' ),
					'used_for_variations_label'     => __( 'Used for variations', 'kkart' ),
					'new_attribute_prompt'          => __( 'Enter a name for the new attribute term:', 'kkart' ),
					'calc_totals'                   => __( 'Recalculate totals? This will calculate taxes based on the customers country (or the store base country) and update totals.', 'kkart' ),
					'copy_billing'                  => __( 'Copy billing information to shipping information? This will remove any currently entered shipping information.', 'kkart' ),
					'load_billing'                  => __( "Load the customer's billing information? This will remove any currently entered billing information.", 'kkart' ),
					'load_shipping'                 => __( "Load the customer's shipping information? This will remove any currently entered shipping information.", 'kkart' ),
					'featured_label'                => __( 'Featured', 'kkart' ),
					'prices_include_tax'            => esc_attr( get_option( 'kkart_prices_include_tax' ) ),
					'tax_based_on'                  => esc_attr( get_option( 'kkart_tax_based_on' ) ),
					'round_at_subtotal'             => esc_attr( get_option( 'kkart_tax_round_at_subtotal' ) ),
					'no_customer_selected'          => __( 'No customer selected', 'kkart' ),
					'plugin_url'                    => KKART()->plugin_url(),
					'ajax_url'                      => admin_url( 'admin-ajax.php' ),
					'order_item_nonce'              => wp_create_nonce( 'order-item' ),
					'add_attribute_nonce'           => wp_create_nonce( 'add-attribute' ),
					'save_attributes_nonce'         => wp_create_nonce( 'save-attributes' ),
					'calc_totals_nonce'             => wp_create_nonce( 'calc-totals' ),
					'get_customer_details_nonce'    => wp_create_nonce( 'get-customer-details' ),
					'search_products_nonce'         => wp_create_nonce( 'search-products' ),
					'grant_access_nonce'            => wp_create_nonce( 'grant-access' ),
					'revoke_access_nonce'           => wp_create_nonce( 'revoke-access' ),
					'add_order_note_nonce'          => wp_create_nonce( 'add-order-note' ),
					'delete_order_note_nonce'       => wp_create_nonce( 'delete-order-note' ),
					'calendar_image'                => KKART()->plugin_url() . '/assets/images/calendar.png',
					'post_id'                       => isset( $post->ID ) ? $post->ID : '',
					'base_country'                  => KKART()->countries->get_base_country(),
					'currency_format_num_decimals'  => kkart_get_price_decimals(),
					'currency_format_symbol'        => get_kkart_currency_symbol( $currency ),
					'currency_format_decimal_sep'   => esc_attr( kkart_get_price_decimal_separator() ),
					'currency_format_thousand_sep'  => esc_attr( kkart_get_price_thousand_separator() ),
					'currency_format'               => esc_attr( str_replace( array( '%1$s', '%2$s' ), array( '%s', '%v' ), get_kkart_price_format() ) ), // For accounting JS.
					'rounding_precision'            => kkart_get_rounding_precision(),
					'tax_rounding_mode'             => kkart_get_tax_rounding_mode(),
					'product_types'                 => array_unique( array_merge( array( 'simple', 'grouped', 'variable', 'external' ), array_keys( kkart_get_product_types() ) ) ),
					'i18n_download_permission_fail' => __( 'Could not grant access - the user may already have permission for this file or billing email is not set. Ensure the billing email is set, and the order has been saved.', 'kkart' ),
					'i18n_permission_revoke'        => __( 'Are you sure you want to revoke access to this download?', 'kkart' ),
					'i18n_tax_rate_already_exists'  => __( 'You cannot add the same tax rate twice!', 'kkart' ),
					'i18n_delete_note'              => __( 'Are you sure you wish to delete this note? This action cannot be undone.', 'kkart' ),
					'i18n_apply_coupon'             => __( 'Enter a coupon code to apply. Discounts are applied to line totals, before taxes.', 'kkart' ),
					'i18n_add_fee'                  => __( 'Enter a fixed amount or percentage to apply as a fee.', 'kkart' ),
				);

				wp_localize_script( 'kkart-admin-meta-boxes', 'kkart_admin_meta_boxes', $params );
			}

			// Term ordering - only when sorting by term_order.
			if ( ( strstr( $screen_id, 'edit-pa_' ) || ( ! empty( $_GET['taxonomy'] ) && in_array( wp_unslash( $_GET['taxonomy'] ), apply_filters( 'kkart_sortable_taxonomies', array( 'product_cat' ) ) ) ) ) && ! isset( $_GET['orderby'] ) ) {

				wp_register_script( 'kkart_term_ordering', KKART()->plugin_url() . '/assets/js/admin/term-ordering' . $suffix . '.js', array( 'jquery-ui-sortable' ), $version );
				wp_enqueue_script( 'kkart_term_ordering' );

				$taxonomy = isset( $_GET['taxonomy'] ) ? kkart_clean( wp_unslash( $_GET['taxonomy'] ) ) : '';

				$kkart_term_order_params = array(
					'taxonomy' => $taxonomy,
				);

				wp_localize_script( 'kkart_term_ordering', 'kkart_term_ordering_params', $kkart_term_order_params );
			}

			// Product sorting - only when sorting by menu order on the products page.
			if ( current_user_can( 'edit_others_pages' ) && 'edit-product' === $screen_id && isset( $wp_query->query['orderby'] ) && 'menu_order title' === $wp_query->query['orderby'] ) {
				wp_register_script( 'kkart_product_ordering', KKART()->plugin_url() . '/assets/js/admin/product-ordering' . $suffix . '.js', array( 'jquery-ui-sortable' ), $version, true );
				wp_enqueue_script( 'kkart_product_ordering' );
			}

			// Reports Pages.
			if ( in_array( $screen_id, apply_filters( 'kkart_reports_screen_ids', array( $kkart_screen_id . '_page_kkart-reports', 'toplevel_page_kkart-reports', 'dashboard' ) ) ) ) {
				wp_register_script( 'kkart-reports', KKART()->plugin_url() . '/assets/js/admin/reports' . $suffix . '.js', array( 'jquery', 'jquery-ui-datepicker' ), $version );

				wp_enqueue_script( 'kkart-reports' );
				wp_enqueue_script( 'flot' );
				wp_enqueue_script( 'flot-resize' );
				wp_enqueue_script( 'flot-time' );
				wp_enqueue_script( 'flot-pie' );
				wp_enqueue_script( 'flot-stack' );
			}

			// API settings.
			if ( $kkart_screen_id . '_page_kkart-settings' === $screen_id && isset( $_GET['section'] ) && 'keys' == $_GET['section'] ) {
				wp_register_script( 'kkart-api-keys', KKART()->plugin_url() . '/assets/js/admin/api-keys' . $suffix . '.js', array( 'jquery', 'kkart_admin', 'underscore', 'backbone', 'wp-util', 'qrcode', 'kkart-clipboard' ), $version, true );
				wp_enqueue_script( 'kkart-api-keys' );
				wp_localize_script(
					'kkart-api-keys',
					'kkart_admin_api_keys',
					array(
						'ajax_url'         => admin_url( 'admin-ajax.php' ),
						'update_api_nonce' => wp_create_nonce( 'update-api-key' ),
						'clipboard_failed' => esc_html__( 'Copying to clipboard failed. Please press Ctrl/Cmd+C to copy.', 'kkart' ),
					)
				);
			}

			// System status.
			if ( $kkart_screen_id . '_page_kkart-status' === $screen_id ) {
				wp_register_script( 'kkart-admin-system-status', KKART()->plugin_url() . '/assets/js/admin/system-status' . $suffix . '.js', array( 'kkart-clipboard' ), $version );
				wp_enqueue_script( 'kkart-admin-system-status' );
				wp_localize_script(
					'kkart-admin-system-status',
					'kkart_admin_system_status',
					array(
						'delete_log_confirmation' => esc_js( __( 'Are you sure you want to delete this log?', 'kkart' ) ),
						'run_tool_confirmation'   => esc_js( __( 'Are you sure you want to run this tool?', 'kkart' ) ),
					)
				);
			}

			if ( in_array( $screen_id, array( 'user-edit', 'profile' ) ) ) {
				wp_register_script( 'kkart-users', KKART()->plugin_url() . '/assets/js/admin/users' . $suffix . '.js', array( 'jquery', 'kkart-enhanced-select', 'selectWoo' ), $version, true );
				wp_enqueue_script( 'kkart-users' );
				wp_localize_script(
					'kkart-users',
					'kkart_users_params',
					array(
						'countries'              => wp_json_encode( array_merge( KKART()->countries->get_allowed_country_states(), KKART()->countries->get_shipping_country_states() ) ),
						'i18n_select_state_text' => esc_attr__( 'Select an option&hellip;', 'kkart' ),
					)
				);
			}

			if ( KKART_Marketplace_Suggestions::show_suggestions_for_screen( $screen_id ) ) {
				$active_plugin_slugs = array_map( 'dirname', get_option( 'active_plugins' ) );
				wp_register_script(
					'marketplace-suggestions',
					KKART()->plugin_url() . '/assets/js/admin/marketplace-suggestions' . $suffix . '.js',
					array( 'jquery', 'underscore', 'js-cookie' ),
					$version,
					true
				);
				wp_localize_script(
					'marketplace-suggestions',
					'marketplace_suggestions',
					array(
						'dismiss_suggestion_nonce' => wp_create_nonce( 'add_dismissed_marketplace_suggestion' ),
						'active_plugins'           => $active_plugin_slugs,
						'dismissed_suggestions'    => KKART_Marketplace_Suggestions::get_dismissed_suggestions(),
						'suggestions_data'         => KKART_Marketplace_Suggestions::get_suggestions_api_data(),
						'manage_suggestions_url'   => admin_url( 'admin.php?page=kkart-settings&tab=advanced&section=kkart_com' ),
						'in_app_purchase_params'   => KKART_Admin_Addons::get_in_app_purchase_url_params(),
						'i18n_marketplace_suggestions_default_cta'
							=> esc_html__( 'Learn More', 'kkart' ),
						'i18n_marketplace_suggestions_dismiss_tooltip'
							=> esc_attr__( 'Dismiss this suggestion', 'kkart' ),
						'i18n_marketplace_suggestions_manage_suggestions'
							=> esc_html__( 'Manage suggestions', 'kkart' ),
					)
				);
				wp_enqueue_script( 'marketplace-suggestions' );
			}

		}

	}

endif;

return new KKART_Admin_Assets();
