import pwd
import os

class ClPwd:
    class NoSuchUserException(Exception):
        def __init__(self, user):
            Exception.__init__(self, "No such user (%s)" % (user,))

    def __init__(self, min_uid = None):
        self._user_key_map = {}
        self._uid_key_map = {}
        self._user_full_map = {}
        self._uid_full_map = {}
        if min_uid is None:
            self._min_uid = self.get_sys_min_uid(500)
        else:
            self._min_uid = min_uid

    def get_user_dict(self):
        self._load_passwd_database()
        return self._user_key_map

    def get_uid_dict(self):
        self._load_passwd_database()
        return self._uid_key_map

    def get_user_full_dict(self):
        self._load_passwd_database()
        return self._user_full_map

    def get_uid_full_dict(self):
        self._load_passwd_database()
        return self._uid_full_map

    def get_pw_by_name(self, user):
        """
        Return pw_entry for user
        """
        try:
            return self.get_user_full_dict()[user]
        except KeyError:
            raise ClPwd.NoSuchUserException(user)

    def get_pw_by_uid(self, uid):
        """
        Return list of passwd entries for uid
        """
        try:
            return self.get_uid_full_dict()[uid]
        except KeyError:
            raise ClPwd.NoSuchUserException(uid)

    def get_uid(self, user):
        """
        Returns uid for user
        """
        try:
            return self.get_user_full_dict()[user].pw_uid
        except KeyError:
            raise ClPwd.NoSuchUserException(user)

    def get_homedir(self, user):
        """
        Returns homedir for a user
        @param user: string
        @return: string
        """
        try:
            return self.get_user_full_dict()[user].pw_dir
        except KeyError:
            raise ClPwd.NoSuchUserException(user)

    def _load_passwd_database(self):
        """
        Loads the passwd database and fills user_to_uid and user_to_homedir maps
        """
        if not self._uid_full_map:
            for entry in pwd.getpwall():
                self._user_full_map[entry.pw_name] = entry
                if entry.pw_uid not in self._uid_full_map:
                    self._uid_full_map[entry.pw_uid] = []
                self._uid_full_map[entry.pw_uid].append(entry)
                if entry.pw_uid >= self._min_uid:
                    self._user_key_map[entry.pw_name] = entry
                    if entry.pw_uid not in self._uid_key_map:
                        self._uid_key_map[entry.pw_uid] = []
                    self._uid_key_map[entry.pw_uid].append(entry)

    def get_names(self, uid):
        """
        Return names of users with uid specified
        @param uid: int
        @return: list of strings
        """
        try:
            entries = self.get_uid_full_dict()[uid]
        except KeyError:
            raise ClPwd.NoSuchUserException(uid)

        return [entry.pw_name for entry in entries]


    def get_sys_min_uid(self, def_min_uid = 500):
        """
        Return system defined MIN_UID from /etc/login.def or def_min_uid
        @param def_min_uid: int
        @return: MIN_UID: int
        """

        LOGIN_DEF_FILE = '/etc/login.defs'
        if (os.path.exists(LOGIN_DEF_FILE)):
            with open(LOGIN_DEF_FILE, 'r') as lines:
                for line in lines:
                    if line.startswith('UID_MIN'):
                        try:
                            return int(line.split('UID_MIN')[1].strip())
                        except ValueError:
                            pass

        return def_min_uid
