<?php
/*This file is part of Silex - see http://projects.silexlabs.org/?/silex

Silex is  2010-2011 Silex Labs and is released under the GPL License:

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License (GPL) as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version. 

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

To read the license please visit http://www.gnu.org/copyleft/gpl.html

File: version_editor.php
This file contains the classes of the version API of Silex server.

Author: Thomas Ftiveau (abojad)  <http://tofee.fr> or <thomas.fetiveau.tech@gmail.com>

TODO list:
      - TODO add logger and logging instructions adapted to production in functions that do not have already
      - TODO FolderModel->checkRights : return false ou message d'erreur ?
      - TODO FileModel->checkRights : return false ou message d'erreur ?
      - TODO check if not better to use is_writable / is_readable or @fopen ?!
 */

require_once("logger.php");
require_once("consts.php");

/*
   Class: SilexTree
   This class is the main entry point to manage a silex server version. Its logic embeds the generation of version data, the manipulation of these data and the persisting logic of these data to XML files.
*/
class SilexTree
{
	private $logger = null;
	
	/*
		Variable: $version
		The label of the version
	*/
	public $version;
   
	/*
		Variable: $rootFolder
		A FolderModel object corresponding to the root folder of the silex server
	*/
	public $rootFolder;
   
	/*
		Constructor: SilexTree
		Initializes the SilexTree
		
		Parameters:
		  $version - optional, the version tag of this Silex version
	 */
	public function __construct($version=null)
	{
		$this->logger = new logger("SilexTree");
		
		$this->version = $version;
		
		$this->rootFolder = new FolderModel();
		$this->rootFolder->name = '';
		$this->rootFolder->path = DIRECTORY_SEPARATOR;
	}
	
	/*
		Function: toXML
		Converts the php SilexTree object to a DOMDocument object
		
		Returns:
		  a DOMDocument
	*/
	public function toXML()
	{
		$versionDoc = new DOMDocument('1.0', 'utf-8');
		
		$silexTreeElement = $versionDoc->createElement('SilexTree');
		$silexTreeNode = $versionDoc->appendChild($silexTreeElement);
		$silexTreeNode->setAttribute('version', $this->version);
		
		if(!empty($this->rootFolder))
			$this->rootFolder->toXML($silexTreeElement, $versionDoc);
		
		return $versionDoc;
	}

	/*
		Function: fromXML
		Converts a DOMDocument to a SilexTree object
		
		Parameters:
		  $versionDOMDocument - The DOMDocument containing the data
		  $fetchVersionOnly - optional, if set, it will return an empty SilexTree with its version tag set only. default is null.

		Returns:
		a SilexTree corresponding to the DOMDocument
	*/
	public static function fromXML($versionDOMDocument, $fetchVersionOnly = false)
	{
		$rootElement = $versionDOMDocument->documentElement;
	
		$silexTree = new SilexTree($rootElement->getAttribute('version'));
		
		if( !$fetchVersionOnly )
			foreach ($rootElement->childNodes as $item)
			{
				if($item->nodeName == "folder")
					$silexTree->rootFolder->folders[] = FolderModel::fromXML($item, $silexTree->rootFolder->path);
				else if($item->nodeName == "file")
					$silexTree->rootFolder->files[] = FileModel::fromXML($item, $silexTree->rootFolder->path);
				// else: it's a #text
			}
		
		return $silexTree;
	}
   
	/*
		Function: loadVersion
		This method loads into a SilexTree object the information contained in a version.xml file
	
		Returns:
		a SilexTree corresponding to the data from xml file or null if it failed to load the file
		null if $versionFilePath not pointing a real file or if XML data loading failed
	*/
	public static function loadVersion($versionFilePath, $fetchVersionOnly = null)
	{
		$xmlDoc = new DOMDocument();
		if(is_file($versionFilePath))
			if($xmlDoc->load($versionFilePath))
				return self::fromXML($xmlDoc, $fetchVersionOnly);
		return null;
	}
   
	/*
		Function: saveVersion
		This method persists the information of a SilexTree object into a version.xml file
	
		Returns:
		the error code
	*/
	public function saveVersion($versionFilePath)
	{
		$versionDOMDocument = $this->toXML();
		//TODO add eventual cheks & error codes
		$fp = fopen($versionFilePath, 'wb');
		fwrite($fp, $versionDOMDocument->saveXML() );
		fclose($fp);
	}
	
	/*
		Function: createFromFileTree
		This method creates a SilexTree object by bowsing the actual Silex server file tree
		ATTENTION, The SilexTree generated by this method will have its version empty as it doesn't correspond to a published version yet
		
		Returns:
		a SilexTree corresponding to the Silex server file tree
		false on failure (may happen if READ permissions on file tree aren't set for the PHP process)
	*/
	public static function createFromFileTree($excludePaths = null, $requirePaths = null)
	{
		$silexTree = new SilexTree();
		
		if($silexTree->rootFolder = FolderModel::fromFileTree(DIRECTORY_SEPARATOR, '', $excludePaths, $requirePaths))
			return $silexTree;
		
		return false;
	}
	
	
	/*
		Function: checkRights
		This methods checks the desired access rights on a entire silex server file tree
		
		Parameters:
		  $rights - an array of rights (defined in consts.php) corresponding to the permissions to check the files and folders against.
		
		Returns
		a report containing the FileModel and FolderModel objects corresponding to the files and folders not having the desired access permissions
	*/
	public function checkRights($rights)
	{
		$report = array();
		$this->rootFolder->checkRights($rights, true, $report);
		
		return $report;
	}
	
	
	/*
		Function: diffComp
		Compare to another SilexTree (works as a difference: = this - anotherSilexTree)
		
		Parameters:
		  $otherSilexTree - the other SilexTree object to compare with
		  $matchSignatures - true if we want to compare the file signatures. If set to false, it will just report the files and folders missing in the other SilexTree object
		
		Returns:
		a report in a form of an array containing FileModel and FolderModel objects corresponding to all the files and folders of this SilexTree that are different or not existing in the other one.
	*/
	public function diffComp($otherSilexTree, $matchSignatures=true)
	{
		$report = array();
		$this->rootFolder->diffComp($otherSilexTree->rootFolder, $report, $matchSignatures);
		return $report;
	}
}


/*
   Class: SilexElement
   This class is the main entry point to manage a Silex element version (Silex plugins, fonts, layouts...). Its logic embeds the generation of version data, the manipulation of these data and the persisting logic of these data to XML files.
*/
class SilexElement
{
	private $logger = null;

	/*
		Variable: $id
		The id of the element in the Silex exchange platform
	*/
	public $id;
	
	/*
		Variable: $version
		The label of the element's version
	*/
	public $version;
	
	/*
		Variable: $compatibleVersions
		The list of Silex server versions compatible with this element
	*/
	public $compatibleVersions;
	
	/*
	
	/*
		Variable: $dependencies
		an array of wordpress 'postID'
	*/
	public $dependencies;
	
	/*
		Variable: $rootFolder
		A FolderModel object corresponding to the root folder of the silex server (each Silex element is always packaged relatively to the Silex server root folder.)
	*/
	public $rootFolder;
	
	  
	/*
		Constructor: SilexElement
		Initializes the SilexElement
		
		Parameters:
		  $id - optional, the Silex exchange platform id of the Silex element
		  $version - optional, the version tag of this Silex element
	 */
	public function __construct($id=null, $version=null)
	{
		$this->logger = new logger("SilexElement");
		
		$this->id = $id;
		
		$this->version = $version;
		
		$this->rootFolder = new FolderModel();
		$this->rootFolder->name = '';
		$this->rootFolder->path = DIRECTORY_SEPARATOR;
		
		$this->dependencies = array();
	}
	
	/*
		Function: toXML
		Converts the php SilexElement object to a DOMDocument object
		
		Returns:
		  a DOMDocument
	*/
	public function toXML()
	{
		$versionDoc = new DOMDocument('1.0', 'utf-8');
		
		$silexElementDOMElement = $versionDoc->createElement('SilexElement');
		$silexElementNode = $versionDoc->appendChild($silexElementDOMElement);
		$silexElementNode->setAttribute('version', $this->version);
		$silexElementNode->setAttribute('id', $this->id);
		
		if(!empty($this->compatibleVersions))
		{
			$compatibleVersionsElement = $versionDoc->createElement('compatibleVersions');
			$compatibleVersionsNode = $silexElementDOMElement->appendChild($compatibleVersionsElement);
			foreach($this->compatibleVersions as $compatibleVersion)
			{
				$compatibleVersionElement = $versionDoc->createElement('compatibleVersion');
				$compatibleVersionElement->appendChild($versionDoc->createCDATASection($compatibleVersion));
				$compatibleVersionsElement->appendChild($compatibleVersionElement);
			}
		}
		
		// dependencies
		$depElement = $versionDoc->createElement('dependencies');
		$depRootNode = $silexElementDOMElement->appendChild($depElement);
		for ($idx=0;$idx < count($this->dependencies); $idx++)
		{
			$depElement = $versionDoc->createElement('dependency');
			$depNode = $depRootNode->appendChild($depElement);
			$depNode->setAttribute('postID', $this->dependencies[$idx]);
		}
		
		if(!empty($this->rootFolder))
			$this->rootFolder->toXML($silexElementDOMElement, $versionDoc);
		
		return $versionDoc;
	}

	/*
		Function: fromXML
		Converts a DOMDocument to a SilexElement object
		
		Parameters:
		  $versionDOMDocument - The DOMDocument containing the data
		  $fetchVersionOnly - optional, if set, it will return an empty SilexElement with only its version tag set. default is false.

		Returns:
		a SilexElement corresponding to the DOMDocument
	*/
	public static function fromXML($versionDOMDocument, $fetchVersionAndIdOnly = false)
	{
		$rootElement = $versionDOMDocument->documentElement;
	
		$silexElement = new SilexElement($rootElement->getAttribute('id'), $rootElement->getAttribute('version'));
		
		if( !$fetchVersionAndIdOnly )
			foreach ($rootElement->childNodes as $item)
			{
				if($item->nodeName == "compatibleVersions")
				{
					$compatibleVersionItems = array();
					foreach ($item->childNodes as $compatibleVersionItem)
						$compatibleVersionItems[] = $compatibleVersionItem->textContent;
					$silexElement->compatibleVersions = $compatibleVersionItems;
				}				
				else if($item->nodeName == "folder")
				{
					$silexElement->rootFolder->folders[] = FolderModel::fromXML($item, $silexElement->rootFolder->path);
				}
				else if($item->nodeName == "file")
				{
					$silexElement->rootFolder->files[] = FileModel::fromXML($item, $silexElement->rootFolder->path);
				}
				else if($item->nodeName == "dependencies")
				{
					foreach ($item->childNodes as $depItem)
					{
						$silexElement->dependencies[] = $depItem->getAttribute('postID');
					}
				}
				// else: it's a #text
			}
		
		return $silexElement;
	}
   
	/*
		Function: loadVersion
		This method loads into a SilexElement object the information contained in a version xml file
	
		Returns:
		a SilexElement corresponding to the data from xml file or null if it failed to load the file
		null if $versionFilePath not pointing a real file or if XML data loading failed
	*/
	public static function loadVersion($versionFilePath, $fetchVersionAndIdOnly = null)
	{
		$xmlDoc = new DOMDocument();
		if(is_file($versionFilePath))
			if($xmlDoc->load($versionFilePath))
				return self::fromXML($xmlDoc, $fetchVersionAndIdOnly);
		return null;
	}
   
	/*
		Function: saveVersion
		This method persists the information of a SilexElement object into a version xml file
	
		Returns:
		the error code
	*/
	public function saveVersion($versionFilePath)
	{
		$versionDOMDocument = $this->toXML();
		//TODO add eventual cheks & error codes
		if($fp = fopen($versionFilePath, 'wb'))
		{
			fwrite($fp, $versionDOMDocument->saveXML() );
			fclose($fp);
		}
		// else TODO manage this case
	}
	
	/*
		Function: createFromFileTree
		This method creates a SilexElement object by bowsing a file tree corresponding to an element ready to publish
		ATTENTION, The SilexElement generated by this method will have its version and id empty as it doesn't correspond to an already published element yet 
		
		Parameters:
			$pathToTheElementFileTree - the path to the element file tree ready to publish (the element must be in a "silex server like" structure, ie: if the element is a plugin, the element file tree must be a single folder containing a "/plugins" folder, containing a "plugin name" folder, containing the plugin's files
			$excludePaths - optional, the paths to the files and folders not to include in the version file
		
		Returns:
		a SilexElement corresponding to the Silex element file tree
		false on failure (may happen if READ permissions on file tree aren't set for the PHP process)
	*/
	public static function createFromFileTree($pathToTheElementFileTree, $excludePaths = null)
	{
		$silexElement = new SilexElement();
		
		if($silexElement->rootFolder = FolderModel::fromFileTree(DIRECTORY_SEPARATOR, '', $excludePaths, null, $pathToTheElementFileTree))
			return $silexElement;
		
		return false;
	}
	
	
	/*
		Function: checkRights
		This methods checks the desired access rights on a entire Element file tree
		
		Parameters:
		  $rights - an array of rights (defined in consts.php) corresponding to the permissions to check the files and folders against.
		
		Returns
		a report containing the FileModel and FolderModel objects corresponding to the files and folders not having the desired access permissions
	*/
	public function checkRights($rights)
	{
		$report = array();
		$this->rootFolder->checkRights($rights, true, $report);
		
		return $report;
	}
	
	
	/*
		Function: diffComp
		Compare to another SilexElement or SilexTree (works as a difference: = this - anotherSilexTree)
		
		Parameters:
		  $otherSilexTree - the other SilexTree or SilexElement object to compare with
		  $matchSignatures - true if we want to compare the file signatures. If set to false, it will just report the files and folders missing in the other SilexTree object
		
		Returns:
		a report in a form of an array containing FileModel and FolderModel objects corresponding to all the files and folders of this SilexElement that are different or not existing in the other one.
	*/
	public function diffComp($otherSilexTree, $matchSignatures=true)
	{
		$report = array();
		$this->rootFolder->diffComp($otherSilexTree->rootFolder, $report, $matchSignatures);
		return $report;
	}
	
	
	/*
		Function: compToLocalSignatures
		Compare the files signatures of the SilexElement object to the local file ones if the same files (path+name) exist.
		
		Parameters:
		   rootpath - the path to the root dir of the local element to compare to (if you want to compare to an installed element, specify the Silex server root dir)
		
		Returns:
		a report in a form of an array containing FileModel objects of this SilexElement that exist in the local path specified in parameter and have a different signature.
	*/
	public function compToLocalSignatures($rootpath)
	{
		$report = array();
		$this->rootFolder->compToLocalSignatures($rootpath, $report);
		return $report;
	}
}


/*
   Class: FolderModel
   This class describes the version API's data model and operations of a folder.
*/
class FolderModel
{
	private $logger = null;
	
	/*
		Variable: $name
		the name of this folder
	*/
	public $name;
   
	/*
		Variable: $files
		an array of FileModel objects = The list of files contained in this folder
	*/
	public $files;
   
	/*
		Variable: $folders
		an array of FolderModel objects = The list of other folders contained in this folder
	*/
	public $folders;
   
	/*
		Variable: $path
		The path of this folder, relative to Silex server or element root path
	*/
	public $path;

   
	/*
		Constructor: FolderModel
		
		Default initialization of a FolderModel object
	 */
	public function __construct()
	{
		$this->logger = new logger("FolderModel");
		
		$this->folders = array();
		$this->files = array();
		
        $a = func_get_args();
        $i = func_num_args();
        if ( $i > 0 && method_exists($this,$f='__construct'.$i) )
            call_user_func_array(array($this,$f),$a);
	}
	/*
		Constructor: FolderModel
		
		Initialization of a FolderModel object
		
		Parameters:
		
		  $uri - the uri (relative to the Silex server root path) of the folder that will be converted into the path and name attributes
	 */
	function __construct1($uri)
	{
		$this->__construct();
		$this->name = basename($uri);
		(dirname($uri) != ".") ? $this->path = dirname($uri) : $this->path = "";
	}
	/*
		Constructor: FolderModel
		
		Initialization of a FolderModel object
		
		Parameters:
		
		  $path - the folder's path
		  $name - the folder's name
	 */
	function __construct2($path, $name)
	{
		$this->__construct();
		$this->name = $name;
		$this->path = formatPath($path);
	}
	
	/*
		Function: toString
		Returns a string representing this FolderModel object
	*/
	public function __toString()
    {
		$toString = "";
		
		if(isset($this->path))
			$toString .= $this->path;
		
		if(isset($this->name))
			$toString .= $this->name;
		
        return $toString;
    }
	
	/*
		Function: equals
		tells if a folder equals another.
		ATTENTION, this method in its current implementation just compares the path and name of the folder and doesn't take into consideration the folder's contents
		
		Returns:
		true if equals, false if not
	*/
	public function equals($folderModelToCompare)
	{
		if( formatPath($this->path.$this->name) == formatPath($folderModelToCompare->path.$folderModelToCompare->name) )
			return true;
		return false;
	}
	
	/*
		Function: diffComp
		Compare to another FolderModel (works as a difference: = this - anotherFolderModel)
		
		Parameters:
		  $folderModelToCompare - the other FolderModel object to compare with
		  &$report - a reference to a report to fill with the eventual files or folder that are different or missing in the other FolderModel
		  $matchSignatures - true if we want to compare also the file signatures. If set to false, it will just report the files and folders missing in the other FolderModel object.
	*/
	public function diffComp($folderModelToCompare, &$report, $matchSignatures=true)
	{
		foreach($this->folders as $folder)
		{
			$i=0;
			$otherFoldersSize = count($folderModelToCompare->folders);
			
			while($i < $otherFoldersSize && !$folder->equals($folderModelToCompare->folders[$i]))
				$i++;
				
			if($i < $otherFoldersSize) {
				$folder->diffComp($folderModelToCompare->folders[$i], $report, $matchSignatures);}
			else
				$report[] = $folder;
		}
		
		foreach($this->files as $file)
		{
			$i=0;
			$otherFilesSize = count($folderModelToCompare->files);
			while( $i < $otherFilesSize &&
				( !$file->equals($folderModelToCompare->files[$i]) && $matchSignatures || 
					!$matchSignatures && !$file->pathEquals($folderModelToCompare->files[$i])) )
				$i++;
			if($i == $otherFilesSize)
				$report[] = $file;
		}
	}
	
	/*
		Function: compToLocalSignatures
		
		
		Parameters:
		  
	*/
	public function compToLocalSignatures($rootpath, &$report)
	{
		if( is_dir($rootpath . DIRECTORY_SEPARATOR . $this->path . DIRECTORY_SEPARATOR  . $this->name) )
		{
			foreach($this->folders as $folder)
				$folder->compToLocalSignatures($rootpath, $report);
			
			foreach($this->files as $file)
				$file->compToLocalSignatures($rootpath, $report);
		}
	}
	
	/*
		Function: checkRights
		Checks a FolderModel object's permissions against desired permissions
		
		Parameters:
		  $rights - an array of rights (see consts.php) to check the FolderModel object against.
		  $recusive - tells if we want to check the inner folders recursively, default is false.
		  &$report - optional, a reference to a report to fill in with the FildeModel or FolderModel objects corresponding to the file and folders not having the required permissions
		  $rootPath - optional, the rootpath defining where to check the permissions. By default, check in the silex server root directory (in a Silex server context, where the ROOTPATH constant points to it).
		
		Returns:
		a boolean that tells if the folder's access rights are sufficient
	*/
	public function checkRights($rights, $recusive = false, &$report = null, $rootPath=ROOTPATH)
	{
		$rightsSufficient = false;
		$fullPath = $rootPath . $this->path . $this->name;
		
		if( is_array($rights) )
		{			
			foreach($rights as $right)
			{
				if($right == READ_RIGHTS)
					$rightsSufficient = is_dir($fullPath) && is_readable($fullPath);
				
				if($right == WRITE_RIGHTS) {
					$rightsSufficient = is_dir($fullPath) && is_writable($fullPath); // take care, is_writable tells false whether the file perms aren't OK or if the file doesn't exist
					while(!$rightsSufficient && !file_exists($fullPath)) { // we want to know if the parent directory  perms would allow to write the in it
						$fullPath = dirname($fullPath);
						$rightsSufficient = is_dir($fullPath) && is_writable($fullPath);
					}
					$fullPath = $rootPath . $this->path . $this->name;
				}
				
				if($right == EXECUTE_RIGHTS)
					$rightsSufficient = is_dir($fullPath) && is_executable($fullPath);
			}
			
			if($recusive)
			{			
				if ($dh = @opendir($rootPath . $this->path . $this->name))
				{
					while ( ( $innerItem = readdir($dh) ) !== false )
					{
						if ($innerItem === '.' || $innerItem === '..' || $innerItem === '.svn' || $innerItem === '.DS_Store' || $innerItem === 'Thumbs.db' )
							continue;
						
						if (is_file($rootPath . $this->path . $this->name . DIRECTORY_SEPARATOR . $innerItem)) {
							$innerFile = new FileModel($this->path . $this->name . DIRECTORY_SEPARATOR, $innerItem);
							$innerFile->checkRights($rights, $report);
						}
							
						else {
							$innerFolder = new FolderModel($this->path . $this->name . DIRECTORY_SEPARATOR, $innerItem);
							$innerFolder->checkRights($rights, true, $report);
						}
					}
					closedir($dh);
				} // else failed to open dir
			}
			
			if(is_array($report) && !$rightsSufficient)
				$report[] = $this;
			
			return $rightsSufficient;
		}
		else
			if ($this->logger) $this->logger->info(' checkRights : the $rights parameter must be an array ');
		
		return false; // TODO ou message d'erreur ?!
	}
	
	/*
		Function: toXML
		Converts the php FolderModel object to a DOMElement object and append it to its DOMElement parent
		
		Parameters:
		  &$parentFolderElement - a reference to the DOMElement object corresponding to the parent folder
		  &$domDoc - a reference to the DOMDocument object of the XML document
	*/
	public function toXML(&$parentFolderElement, &$domDoc)
	{
		if($this->name != '') // we do not want the root dir to appear in the xml
		{
			$folderElement = $domDoc->createElement('folder');
			$folderNode = $parentFolderElement->appendChild($folderElement);
			$folderNode->setAttribute('name', $this->name);
		}
		else
		{
			$folderElement = &$parentFolderElement;
		}
		
		
		if(is_array($this->folders))
			foreach($this->folders as $childFolder)
			{
				$childFolder->toXML($folderElement, $domDoc);
			}
		
		if(is_array($this->files))
			foreach($this->files as $childFile)
			{
				$childFile->toXML($folderElement, $domDoc);
			}
	}
	
	/*
		Function: fromXML
		convert a XML "folder" DOMNode  to a FolderModel object
		
		Parameters:
		  $folderDOMNode - The DOMNode object corresponding to this folder
		  $path - The path of this folder from its parent element
	
		Returns:
		the FolderModel object corresponding to the DOMNode  passed in parameter
	*/
	public static function fromXML($folderDOMNode, $path)
	{
		$silexFolder = new FolderModel($path, $folderDOMNode->getAttribute('name'));
		
		foreach ($folderDOMNode->childNodes as $item)
		{
			if($item->nodeName == "folder")
				$silexFolder->folders[] = FolderModel::fromXML($item, $silexFolder->path.$silexFolder->name.DIRECTORY_SEPARATOR);
			else if($item->nodeName == "file")
				$silexFolder->files[] = FileModel::fromXML($item, $silexFolder->path.$silexFolder->name.DIRECTORY_SEPARATOR);
			// else: it's a #text
		}
		
		return $silexFolder;
	}
	
	/*
		Function: fromFileTree
		Generate a FolderModel object from the local file system
		
		Parameters:
		  $path - the path of the folder on the file system
		  $name - the name of the folder on the file system
		  $excludePaths - optional, an array of paths to exclude (if $path.$name in this array, the folder/file in the file system will be ignored)
		  $requirePaths - optional, array of paths of files to set their updateRequired tag to true
		  $rootPath - optional, tells from which root directory do we extract the informations. By default, we extract them from the Silex server root dir (if we execute the version API in a Silex server context, thus given that the ROOTPATH constant points to the currently running Silex server root dir).
		
		Returns:
		the FolderModel object 
		false on failure
	*/
	public static function fromFileTree($path, $name, $excludePaths = null, $requirePaths = null, $rootPath=ROOTPATH)
	{		
		$currentFolder = new FolderModel($path, $name);
		
		if ($dh = @opendir($rootPath . $currentFolder->path.$currentFolder->name))
		{
			while ( ( $file = readdir($dh) ) !== false )
			{
				if ( $file === '.' || $file === '..' || $file === '.svn' )
					continue;
					
				if(!empty($excludePaths) && 
					in_array(formatPath($currentFolder->path . $currentFolder->name . DIRECTORY_SEPARATOR . $file), $excludePaths)) 
				{
					$currentFolder->logger->debug(formatPath($currentFolder->path . $currentFolder->name . DIRECTORY_SEPARATOR . $file).'  EXCLUDED');
					continue;
				}
				
				if ( is_file($rootPath . $currentFolder->path . $currentFolder->name . DIRECTORY_SEPARATOR . $file) )
				{
					if( $newFileModel = FileModel::fromFileTree($currentFolder->path . $currentFolder->name . DIRECTORY_SEPARATOR, $file, $requirePaths, $rootPath) )
						$currentFolder->files[] = $newFileModel;
				}
				else
				{
					$currentFolder->folders[] = FolderModel::fromFileTree($currentFolder->path . $currentFolder->name . DIRECTORY_SEPARATOR, 
						$file, $excludePaths, $requirePaths, $rootPath);
				}
			}
			closedir($dh);
		}
		else
		{
			if ($currentFolder->logger) $currentFolder->logger->err(" ERROR when opendir ". $rootPath . $currentFolder->path.$currentFolder->name);
			return false;
		}
		
		return $currentFolder;
	}
}



/*
   Class: FileModel
   This class describes the version API's data model and operations of a file.
*/
class FileModel
{
	private $logger = null;
	
	/*
		Variable: $name
		the name of this file
	*/
	public $name;
   
	/*
		Variable: $signature
		The signature of this file
	*/
	public $signature;
   
	/*
		Variable: $updateRequired
		The updateRequired tag of this file
	*/
	public $updateRequired;
   
	/*
		Variable: $path
		The path of this file, relative to the Element or Silex root dir
	*/
	public $path;
   
   
	/*
		Constructor: FileModel
		Default initialization of a FolderModel object
	 */
	public function __construct()
	{
		$this->logger = new logger("FileModel");
		
        $a = func_get_args();
        $i = func_num_args();
        if ( $i > 0 && method_exists($this,$f='__construct'.$i) )
            call_user_func_array(array($this,$f),$a);
	}
	/*
		Constructor: FileModel
		Initialization of a FileModel object
		
		Parameters:
		  $uri - The uri (relative to the silex server root directory) of the file
	 */
	function __construct1($uri)
	{
		$this->__construct();
		$fullPathFormatted = formatPath($uri);
		$this->name = basename($fullPathFormatted);
		(dirname($fullPathFormatted) != ".") ? $this->path = dirname($fullPathFormatted).DIRECTORY_SEPARATOR : $this->path = "";
	}
	/*
		Constructor: FileModel
		Initialization of a FileModel object
		
		Parameters:
		  $path - the path of the file
		  $name - the name of the file
	 */
	function __construct2($path, $name)
	{
		$this->__construct();
		$this->name = $name;
		$this->path = formatPath($path);
	}
	
	/*
		Function: toString
		Returns a string representing this FileModel object
	*/
	public function __toString()
    {
		$toString = "";
		
		if(isset($this->path))
			$toString .= $this->path;
		
		if(isset($this->name))
			$toString .= $this->name;
		
        return $toString;
    }
	
	/*
		Function: 
		Tells wether this FileModel object equals another one or not
		
		Parameters:
		  $fileModelToCompare - the other FileModel object to compare with.
	*/
	public function equals($fileModelToCompare)
	{
		if( $this->pathEquals($fileModelToCompare) )
			if( $this->signature == $fileModelToCompare->signature )
				return true;
		return false;
	}
	
	/*
		Function: pathEquals
		Tells wether this FileModel object points to the same location as another FileModel object
		
		Parameters:
		  $fileModelToCompare - the other FileModel object to compare with.
	*/
	public function pathEquals($fileModelToCompare)
	{
		if( formatPath($this->path.$this->name) ==  formatPath($fileModelToCompare->path.$fileModelToCompare->name) )
			return true;
		return false;
	}
	
	/*
		Function: checkRights
		This method checks if this FileModel objects has desired access permissions
		
		Parameters:
		  $rights - an array containing the rights to check on the file (see consts.php)
		  &$report - optional, a reference to a report to fill with files not having the required permissions
		  $rootPath - optional, tells in which root directory we check the file's permissions (by default, the Silex server directory, if we run the version API in a Silex server context)
		
		Returns:
		a boolean that tells if the file's access rights are sufficient
	*/
	public function checkRights($rights, &$report = null, $rootPath=ROOTPATH)
	{
		$rightsSufficient = false;
		$fullPath = $rootPath . $this->path . $this->name;
		
		if( is_array($rights) )
		{
			foreach($rights as $right)
			{
				if($right == READ_RIGHTS)
					$rightsSufficient = is_file($fullPath) && is_readable($fullPath);
					
				if($right == WRITE_RIGHTS)
				{
					$rightsSufficient = is_file($fullPath) && is_writable($fullPath); // take care, this function tells false whether the file perms aren't OK or if the file doesn't exist
					while(!$rightsSufficient && !file_exists($fullPath)) { // we want to know if the parent directory  perms would allow to write the in it
						$fullPath=dirname($fullPath);
						$rightsSufficient = is_dir($fullPath) && is_writable($fullPath);
					}
					$fullPath = $rootPath . $this->path . $this->name;
				}
				
				if($right == EXECUTE_RIGHTS)
					$rightsSufficient = is_file($fullPath) && is_executable($fullPath);
			}
			
			if(is_array($report) && !$rightsSufficient)
				$report[] = $this;
		}
		
		return $rightsSufficient; // TODO ou message d'erreur ?!
	}
	
	/*
		Function: toXML
		Converts the php FileModel object to a DOMElement object and append it to its DOMElement parent
		
		Parameters:
		  &$parentFolderElement - a reference to the DOMElement object corresponding to the parent folder
		  &$domDoc - a reference to the DOMDocument object
	*/
	public function toXML(&$parentFolderElement, &$domDoc)
	{
		$fileElement = $domDoc->createElement('file');
		$folderNode = $parentFolderElement->appendChild($fileElement);
		$folderNode->setAttribute('name', $this->name);
		$folderNode->setAttribute('signature', $this->signature);
		if($this->updateRequired!=null)
			$folderNode->setAttribute('updateRequired', $this->updateRequired);
	}
   
	/*
		Function: fromXML
		Converts a XML "file" DOMNode to a FileModel object
		
		Parameters:
		  $fileDOMNode - The DOMNode object of the corresponding FileModel object
		  $path - The path of the parent element
		
		Returns:
		the FileModel object corresponding to the DOMNode passed in parameter
	*/
	public static function fromXML($fileDOMNode, $path)
	{
		$silexFile = new FileModel($path, $fileDOMNode->getAttribute('name'));
		
		$silexFile->signature = $fileDOMNode->getAttribute('signature');
		
		if($fileDOMNode->hasAttribute('updateRequired'))
			$silexFile->updateRequired = $fileDOMNode->getAttribute('updateRequired');
	
		return $silexFile;
	}
	
	/*
		Function: fromFileTree
		Creates a FileModel object and Initializes it with data from a file system location.
		
		Parameters:
		  $path - the path of the file
		  $name - the name of the file
		  $requirePaths - optional, an array of paths of files to set their "update required" tag to true
		  $rootPath - optional, tells from which root directory we extract the data. By default, the Silex server root directory, if we run in a Silex server context. Otherwise, you may define this parameter to the directory you want.
		
		Returns:
		a FileModel object corresponding to the pointed file system location
	*/
	public static function fromFileTree($path, $name, $requirePaths = null, $rootPath=ROOTPATH)
	{
		if( $name != ".DS_Store" && $name != "Thumbs.db" )
		{
			$currentFile = new FileModel($path, $name);
			
			if(!empty($requirePaths))
				if( in_array(formatPath($path . $name), $requirePaths) )
					$currentFile->updateRequired = "true";
				else $currentFile->updateRequired = "false";
			
			$currentFile->signature = self::getFileSignature($rootPath . $currentFile->path . $currentFile->name);
			
			return $currentFile;
		}
		
		return null;
	}
	
	/*
		Function: compToLocalSignatures
		
		
		Parameters:
		  
	*/
	public function compToLocalSignatures($rootpath, &$report)
	{
		if( is_file($rootpath . DIRECTORY_SEPARATOR . $this->path . DIRECTORY_SEPARATOR  . $this->name) )
			if( self::getFileSignature($rootpath . DIRECTORY_SEPARATOR . $this->path . DIRECTORY_SEPARATOR  . $this->name) != $this->signature )
				$report[] = $this;
	}
	
	/*
		Function: getFileSignature
		Generate the signature of the desired file
		
		Parameters:
		  $fileUri - the uri of the file the get the signature

		Returns:
		the file signature of a file
	*/
	public static function getFileSignature($fileUri)
	{
		if(is_file($fileUri))
			return sha1_file($fileUri);
	}
}

/*
	Function: formatPath
	Clean a given path by deleting double directory separators
	
	Parameters:
	  $path - the path to clean
*/
function formatPath($path)
{
	if(!empty($path))
		return str_replace ( DIRECTORY_SEPARATOR.DIRECTORY_SEPARATOR , DIRECTORY_SEPARATOR  , $path );
	return $path;
}

?>
