<?php
namespace Redaxscript\View;

use Redaxscript\Admin;
use Redaxscript\Config;
use Redaxscript\Content;
use Redaxscript\Html;
use Redaxscript\Language;
use Redaxscript\Model;
use Redaxscript\Module;
use Redaxscript\Registry;
use Redaxscript\Request;
use Redaxscript\Validator;
use function array_replace_recursive;

/**
 * children class to create the extra
 *
 * @since 4.0.0
 *
 * @package Redaxscript
 * @category View
 * @author Henry Ruhs
 */

class Extra extends ViewAbstract
{
	/**
	 * options of the extra
	 *
	 * @var array
	 */

	protected array $_optionArray =
	[
		'tag' =>
		[
			'title' => 'h3',
			'box' => 'div'
		],
		'className' =>
		[
			'title' => 'rs-title-extra',
			'box' => 'rs-box-extra'
		],
		'orderColumn' => 'rank'
	];

	/**
	 * constructor of the class
	 *
	 * @since 4.0.0
	 *
	 * @param Registry $_registry instance of the registry class
	 * @param Request $_request instance of the request class
	 * @param Language $_language instance of the language class
	 * @param Config $_config instance of the config class
	 */

	public function __construct(protected Registry $_registry, protected Request $_request, protected Language $_language, protected Config $_config)
	{
		parent::__construct($this->_registry, $this->_language);
	}

	/**
	 * init the class
	 *
	 * @since 4.0.0
	 *
	 * @param array $optionArray options of the extra
	 *
	 * @return self
	 */

	public function init(array $optionArray = []) : self
	{
		$this->_optionArray = array_replace_recursive($this->_optionArray, $optionArray);
		return $this;
	}

	/**
	 * render the view
	 *
	 * @since 4.0.0
	 *
	 * @param int $extraId identifier of the extra
	 *
	 * @return string
	 */

	public function render(int $extraId = null) : string
	{
		if ($this->_registry->get('extraReplace'))
		{
			return Module\Hook::trigger('extraReplace');
		}
		$output = Module\Hook::trigger('extraStart');
		$accessValidator = new Validator\Access();
		$extras = null;
		$contentParser = new Content\Parser($this->_registry, $this->_request, $this->_language, $this->_config);
		$adminDock = new Admin\View\Helper\Dock($this->_registry, $this->_language);
		$adminDock->init();
		$loggedIn = $this->_registry->get('loggedIn');
		$token = $this->_registry->get('token');
		$categoryId = $this->_registry->get('categoryId');
		$articleId = $this->_registry->get('articleId');
		$firstParameter = $this->_registry->get('firstParameter');
		$myGroups = $this->_registry->get('myGroups');

		/* html element */

		$element = new Html\Element();
		$titleElement = $element
			->copy()
			->init($this->_optionArray['tag']['title'],
			[
				'class' => $this->_optionArray['className']['title']
			]);
		$boxElement = $element
			->copy()
			->init($this->_optionArray['tag']['box'],
			[
				'class' => $this->_optionArray['className']['box']
			]);
		$extras = $this->queryExtras($extraId);

		/* process extras */

		foreach ($extras as $value)
		{
			$validateContent = true;
			if ($value->category)
			{
				$validateContent = (string)$value->category === $categoryId;
			}
			if ($value->article)
			{
				$validateContent = (string)$value->article === $articleId;
			}
			if ($accessValidator->validate($value->access, $myGroups) && $validateContent)
			{
				$output .= Module\Hook::trigger('extraFragmentStart', (array)$value);
				if ($value->headline)
				{
					$output .= $titleElement
						->attr('id', 'extra-' . $value->alias)
						->text($value->title);
				}
				$contentParser->process($value->text);
				$output .= $boxElement->html($contentParser->getOutput()) . Module\Hook::trigger('extraFragmentEnd', (array)$value);

				/* admin dock */

				if ($loggedIn === $token && $firstParameter !== 'logout')
				{
					$output .= $adminDock->render('extras', $value->id);
				}
			}
		}
		$output .= Module\Hook::trigger('extraEnd');
		return $output;
	}

	/**
	 * query the extras
	 *
	 * @since 4.0.0
	 *
	 * @param int $extraId identifier of the extra
	 *
	 * @return object|null
	 */

	public function queryExtras(int $extraId = null) : ?object
	{
		$extraModel = new Model\Extra();
		$language = $this->_registry->get('language');

		/* query extras */

		if ($extraId)
		{
			return $extraModel->getSiblingByIdAndLanguageAndOrder($extraId, $language, $this->_optionArray['orderColumn']);
		}
		return $extraModel->getByLanguageAndOrder($language, $this->_optionArray['orderColumn']);
	}
}
