<?php
namespace Redaxscript;

use function array_key_exists;
use function file_get_contents;
use function is_array;
use function json_decode;

/**
 * children class to request globals
 *
 * @since 2.2.0
 *
 * @package Redaxscript
 * @category Request
 * @author Henry Ruhs
 */

class Request extends Singleton
{
	/**
	 * array of the request
	 *
	 * @var array
	 */

	protected static array $_requestArray = [];

	/**
	 * init the class
	 *
	 * @since 2.4.0
	 */

	public function init() : void
	{
		self::$_requestArray =
		[
			'server' => $_SERVER ?? [],
			'get' => $_GET ?? [],
			'post' => $_POST ?? [],
			'files' => $_FILES ?? [],
			'stream' => self::_loadStream(),
			'session' => $_SESSION ?? [],
			'cookie' => $_COOKIE ?? []
		];
	}

	/**
	 * get the value from globals
	 *
	 * @since 3.0.0
	 *
	 * @param string $key key of the item
	 *
	 * @return mixed
	 */

	public function get(string $key) : mixed
	{
		if (is_array(self::$_requestArray) && array_key_exists($key, self::$_requestArray))
		{
			return self::$_requestArray[$key];
		}
		return null;
	}

	/**
	 * get the array from globals
	 *
	 * @since 4.0.0
	 *
	 * @return array
	 */

	public function getArray() : array
	{
		return self::$_requestArray;
	}

	/**
	 * set the value to globals
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 * @param mixed $value value of the item
	 */

	public function set(string $key, mixed $value = null) : void
	{
		self::$_requestArray[$key] = $GLOBALS[$key] = $value;
	}

	/**
	 * get the value from server
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 *
	 * @return mixed
	 */

	public function getServer(string $key) : mixed
	{
		if (is_array(self::$_requestArray['server']) && array_key_exists($key, self::$_requestArray['server']))
		{
			return self::$_requestArray['server'][$key];
		}
		return null;
	}

	/**
	 * set the value to server
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 * @param mixed $value value of the item
	 */

	public function setServer(string $key, mixed $value = null) : void
	{
		self::$_requestArray['server'][$key] = $value;
	}

	/**
	 * get the value from query
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 *
	 * @return mixed
	 */

	public function getQuery(string $key) : mixed
	{
		if (is_array(self::$_requestArray['get']) && array_key_exists($key, self::$_requestArray['get']))
		{
			return self::$_requestArray['get'][$key];
		}
		return null;
	}

	/**
	 * set the value to query
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 * @param mixed $value value of the item
	 */

	public function setQuery(string $key, mixed $value = null) : void
	{
		self::$_requestArray['get'][$key] = $value;
	}

	/**
	 * get the value from post
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 *
	 * @return mixed
	 */

	public function getPost(string $key) : mixed
	{
		if (is_array(self::$_requestArray['post']) && array_key_exists($key, self::$_requestArray['post']))
		{
			return self::$_requestArray['post'][$key];
		}
		return null;
	}

	/**
	 * set the value to post
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 * @param mixed $value value of the item
	 */

	public function setPost(string $key, mixed $value = null) : void
	{
		self::$_requestArray['post'][$key] = $value;
	}

	/**
	 * get the value from files
	 *
	 * @since 3.0.0
	 *
	 * @param string $key key of the item
	 *
	 * @return mixed
	 */

	public function getFiles(string $key) : mixed
	{
		if (is_array(self::$_requestArray['files']) && array_key_exists($key, self::$_requestArray['files']))
		{
			return self::$_requestArray['files'][$key];
		}
		return null;
	}

	/**
	 * set the value to files
	 *
	 * @since 3.0.0
	 *
	 * @param string $key key of the item
	 * @param mixed $value value of the item
	 */

	public function setFiles(string $key, mixed $value = null) : void
	{
		self::$_requestArray['files'][$key] = $value;
	}

	/**
	 * get the value from stream
	 *
	 * @since 4.2.0
	 *
	 * @param string $key key of the item
	 *
	 * @return mixed
	 */

	public function getStream(string $key) : mixed
	{
		if (is_array(self::$_requestArray['stream']) && array_key_exists($key, self::$_requestArray['stream']))
		{
			return self::$_requestArray['stream'][$key];
		}
		return null;
	}

	/**
	 * set the value to stream
	 *
	 * @since 4.2.0
	 *
	 * @param string $key key of the item
	 * @param mixed $value value of the item
	 */

	public function setStream(string $key, mixed $value = null) : void
	{
		self::$_requestArray['stream'][$key] = $value;
	}

	/**
	 * get the value from session
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 *
	 * @return mixed
	 */

	public function getSession(string $key) : mixed
	{
		if (is_array(self::$_requestArray['session']) && array_key_exists($key, self::$_requestArray['session']))
		{
			return self::$_requestArray['session'][$key];
		}
		return null;
	}

	/**
	 * set the value to session
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 * @param mixed $value value of the item
	 */

	public function setSession(string $key, mixed $value = null) : void
	{
		self::$_requestArray['session'][$key] = $_SESSION[$key] = $value;
	}

	/**
	 * refresh the session
	 *
	 * @since 2.6.2
	 */

	public function refreshSession() : void
	{
		self::$_requestArray['session'] = $_SESSION ? : [];
	}

	/**
	 * get the value from cookie
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 *
	 * @return mixed
	 */

	public function getCookie(string $key) : mixed
	{
		if (is_array(self::$_requestArray['cookie']) && array_key_exists($key, self::$_requestArray['cookie']))
		{
			return self::$_requestArray['cookie'][$key];
		}
		return null;
	}

	/**
	 * set the value to cookie
	 *
	 * @since 2.2.0
	 *
	 * @param string $key key of the item
	 * @param mixed $value value of the item
	 */

	public function setCookie(string $key, mixed $value = null) : void
	{
		self::$_requestArray['cookie'][$key] = $_COOKIE[$key] = $value;
	}

	/**
	 * refresh the cookie
	 *
	 * @since 2.6.2
	 */

	public function refreshCookie() : void
	{
		self::$_requestArray['cookie'] = $_COOKIE ? : [];
	}

	/**
	 * load the stream
	 *
	 * @since 4.2.0
	 *
	 * @return array
	 */

	protected function _loadStream() : array
	{
		return (array)json_decode(file_get_contents('php://input'));
	}
}
