<?php
namespace Redaxscript\Module;

use Redaxscript\Config;
use Redaxscript\Installer;
use Redaxscript\Language;
use Redaxscript\Model;
use Redaxscript\Registry;
use Redaxscript\Request;
use function array_key_exists;
use function array_merge;
use function is_array;
use function is_dir;

/**
 * parent class to create a module
 *
 * @since 2.2.0
 *
 * @package Redaxscript
 * @category Module
 * @author Henry Ruhs
 */

class Module
{
	/**
	 * array of the module
	 *
	 * @var array
	 */

	protected static array $_moduleArray = [];

	/**
	 * constructor of the class
	 *
	 * @since 3.0.0
	 *
	 * @param Registry $_registry instance of the registry class
	 * @param Request $_request instance of the request class
	 * @param Language $_language instance of the language class
	 * @param Config $_config instance of the config class
	 */

	public function __construct(protected Registry $_registry, protected Request $_request, protected Language $_language, protected Config $_config)
	{
	}

	/**
	 * init the class
	 *
	 * @since 2.4.0
	 *
	 * @param array $moduleArray custom module setup
	 */

	public function init(array $moduleArray = []) : void
	{
		static::$_moduleArray = array_merge(static::$_moduleArray, $moduleArray);

		/* load the language */

		if (is_array(static::$_moduleArray) && array_key_exists('alias', static::$_moduleArray))
		{
			$this->_language->load(
			[
				'modules' . DIRECTORY_SEPARATOR . static::$_moduleArray['alias'] . DIRECTORY_SEPARATOR . 'languages' . DIRECTORY_SEPARATOR . 'en.json',
				'modules' . DIRECTORY_SEPARATOR . static::$_moduleArray['alias'] . DIRECTORY_SEPARATOR . 'languages' . DIRECTORY_SEPARATOR . $this->_registry->get('language') . '.json'
			]);
		}
	}

	/**
	 * install the module
	 *
	 * @since 2.6.0
	 *
	 * @return bool
	 */

	public function install() : bool
	{
		if (is_array(static::$_moduleArray) && array_key_exists('alias', static::$_moduleArray))
		{
			$moduleModel = new Model\Module();
			$moduleModel->createByArray(static::$_moduleArray);

			/* create from sql */

			$directory = 'modules' . DIRECTORY_SEPARATOR . static::$_moduleArray['alias'] . DIRECTORY_SEPARATOR . 'database';
			if (is_dir($directory))
			{
				$installer = new Installer($this->_registry, $this->_request, $this->_language, $this->_config);
				$installer->init($directory);
				$installer->rawCreate();
			}
			$moduleModel->clearCache();
			return $moduleModel->query()->where('alias', static::$_moduleArray['alias'])->count() === 1;
		}
		return false;
	}

	/**
	 * uninstall the module
	 *
	 * @since 2.6.0
	 *
	 * @return bool
	 */

	public function uninstall() : bool
	{
		if (is_array(static::$_moduleArray) && array_key_exists('alias', static::$_moduleArray))
		{
			$moduleModel = new Model\Module();
			$moduleModel->deleteByAlias(static::$_moduleArray['alias']);

			/* drop from sql */

			$directory = 'modules' . DIRECTORY_SEPARATOR . static::$_moduleArray['alias'] . DIRECTORY_SEPARATOR . 'database';
			if (is_dir($directory))
			{
				$installer = new Installer($this->_registry, $this->_request, $this->_language, $this->_config);
				$installer->init($directory);
				$installer->rawDrop();
			}
			$moduleModel->clearCache();
			return $moduleModel->query()->where('alias', static::$_moduleArray['alias'])->count() === 0;
		}
		return false;
	}

	/**
	 * reinstall the module
	 *
	 * @since 4.3.0
	 *
	 * @return bool
	 */

	public function reinstall() : bool
	{
		return $this->uninstall() && $this->install();
	}
}
