<?php

declare(strict_types = 1);

namespace Redaxscript;

use function is_numeric;
use function random_int;

/**
 * parent class to provide a mathematical task
 *
 * @since 2.0.0
 *
 * @package Redaxscript
 * @category Captcha
 * @author Henry Ruhs
 */

class Captcha
{
	/**
	 * captcha operator mode
	 */

	protected int $_mode;

	/**
	 * task to be solved
	 */

	protected string $_task;

	/**
	 * solution to the task
	 *
	 */

	protected int $_solution;

	/**
	 * allowed range for the task
	 *
	 * @var array<string>
	 */

	protected array $_rangeArray =
	[
		'min' => 1,
		'max' => 10
	];

	/**
	 * array of mathematical operators used for the task
	 *
	 * @var array<string>
	 */

	protected array $_operatorArray =
	[
		1 => 'plus',
		-1 => 'minus'
	];

	/**
	 * constructor of the class
	 *
	 * @since 2.4.0
	 *
	 * @param Language $_language instance of the language class
	 */

	public function __construct(protected Language $_language)
	{
	}

	/**
	 * init the class
	 *
	 * @since 2.4.0
	 *
	 * @param ?int $mode captcha operator mode
	 */

	public function init(int $mode = null) : void
	{
		if (is_numeric($mode))
		{
			$this->_mode = $mode;
		}
		else
		{
			$settingModel = new Model\Setting();
			$this->_mode = (int)$settingModel->get('captcha');
		}
		$this->_create();
	}

	/**
	 * get the task
	 *
	 * @since 2.0.0
	 */

	public function getTask() : string
	{
		return $this->_task;
	}

	/**
	 * get the solution
	 *
	 * @since 2.6.0
	 */

	public function getSolution() : int
	{
		return $this->_solution;
	}

	/**
	 * get the minimum range
	 *
	 * @since 2.6.0
	 */

	public function getMin() : int
	{
		return $this->_rangeArray['min'];
	}

	/**
	 * get the maximum range
	 *
	 * @since 2.6.0
	 */

	public function getMax() : int
	{
		return $this->_rangeArray['max'];
	}

	/**
	 * get the mathematical operator used for the task
	 *
	 * @since 2.0.0
	 */

	protected function _getOperator() : int
	{
		if ($this->_mode === 2)
		{
			return 1;
		}
		if ($this->_mode === 3)
		{
			return -1;
		}
		return random_int(0, 1) * 2 - 1;
	}

	/**
	 * create a task of two numbers between allowable range
	 *
	 * @since 2.0.0
	 */

	protected function _create() : void
	{
		/* range */

		$min = $this->getMin();
		$max = $this->getMax();

		/* random numbers */

		$a = random_int($min + 1, $max);
		$b = random_int($min, $a - 1);

		/* operator */

		$c = $this->_getOperator();
		$operator = $this->_operatorArray[$c];

		/* solution and task */

		$this->_solution = $a + $b * $c;
		$this->_task = $this->_language->get('_number')[$a] . ' ' . $this->_language->get($operator) . ' ' . $this->_language->get('_number')[$b];
	}
}
