<?php namespace Anomaly\Streams\Platform\Support;

use Anomaly\Streams\Platform\Routing\UrlGenerator;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Http\Request;
use StringTemplate\Engine;

/**
 * Class Parser
 *
 * @link          http://anomaly.is/streams-platform
 * @author        AnomalyLabs, Inc. <hello@anomaly.is>
 * @author        Ryan Thompson <ryan@anomaly.is>
 * @package       Anomaly\Streams\Platform\Support
 */
class Parser
{

    /**
     * The URL generator.
     *
     * @var UrlGenerator
     */
    protected $url;

    /**
     * The string parser.
     *
     * @var Engine
     */
    protected $parser;

    /**
     * The request object.
     *
     * @var Request
     */
    protected $request;

    /**
     * Create a new Parser instance.
     *
     * @param UrlGenerator $url
     * @param Engine       $parser
     * @param Request      $request
     */
    public function __construct(UrlGenerator $url, Engine $parser, Request $request)
    {
        $this->url     = $url;
        $this->parser  = $parser;
        $this->request = $request;
    }

    /**
     * Parse data into the target recursively.
     *
     * @param       $target
     * @param array $data
     * @return mixed
     */
    public function parse($target, array $data = [])
    {
        $data = $this->prepareData($data);

        /**
         * If the target is an array
         * then parse it recursively.
         */
        if (is_array($target)) {
            foreach ($target as $key => &$value) {
                $value = $this->parse($value, $data);
            }
        }

        /**
         * if the target is a string and is in a parsable
         * format then parse the target with the payload.
         */
        if (is_string($target) && str_contains($target, ['{', '}'])) {
            $target = $this->parser->render($target, $data);
        }

        return $target;
    }

    /**
     * Prepare the data.
     *
     * @param array $data
     * @return array
     */
    protected function prepareData(array $data)
    {
        return $this->toArray($this->mergeDefaultData($data));
    }

    /**
     * Merge default data.
     *
     * @param array $data
     * @return array
     */
    protected function mergeDefaultData(array $data)
    {
        $url     = $this->urlData();
        $request = $this->requestData();

        return array_merge(compact('url', 'request'), $data);
    }

    /**
     * Prep data for parsing.
     *
     * @param array $data
     * @return array
     */
    protected function toArray(array $data)
    {
        foreach ($data as $key => &$value) {
            if (is_object($value) && $value instanceof Arrayable) {
                $value = $value->toArray();
            }
        }

        return $data;
    }

    /**
     * Return the URL data.
     *
     * @return array
     */
    protected function urlData()
    {
        return [
            'previous' => $this->url->previous()
        ];
    }

    /**
     * Return the request data.
     *
     * @return array
     */
    protected function requestData()
    {
        $request = [
            'path' => $this->request->path(),
            'uri'  => $this->request->getRequestUri(),
        ];

        if ($route = $this->request->route()) {

            $request['route'] = [
                'uri'                      => $route->getUri(),
                'parameters'               => $route->parameters(),
                'parameters.to_urlencoded' => array_map(
                    function ($parameter) {
                        return urlencode($parameter);
                    },
                    $route->parameters()
                ),
                'parameter_names'          => $route->parameterNames(),
                'compiled'                 => [
                    'static_prefix'     => $route->getCompiled()->getStaticPrefix(),
                    'parameters_suffix' => str_replace(
                        $route->getCompiled()->getStaticPrefix(),
                        '',
                        $this->request->getRequestUri()
                    )
                ]
            ];
        }

        return $request;
    }
}
