/*
 * Copyright 2007-2017 Charles du Jeu - Abstrium SAS <team (at) pyd.io>
 * This file is part of Pydio.
 *
 * Pydio is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Pydio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Pydio.  If not, see <http://www.gnu.org/licenses/>.
 *
 * The latest code can be found at <https://pydio.com>.
 */

'use strict';

exports.__esModule = true;

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

var _mixinsFormMixin = require('../mixins/FormMixin');

var _mixinsFormMixin2 = _interopRequireDefault(_mixinsFormMixin);

var _FileDropzone = require('./FileDropzone');

var _FileDropzone2 = _interopRequireDefault(_FileDropzone);

var React = require('react');
var PydioApi = require('pydio/http/api');

var _require$requireLib = require('pydio').requireLib('hoc');

var NativeFileDropProvider = _require$requireLib.NativeFileDropProvider;

// Just enable the drop mechanism, but do nothing, it is managed by the FileDropzone
var BinaryDropZone = NativeFileDropProvider(_FileDropzone2['default'], function (items, files, props) {});

/**
 * UI for displaying and uploading an image,
 * using the binaryContext string.
 */
exports['default'] = React.createClass({
    displayName: 'InputImage',

    mixins: [_mixinsFormMixin2['default']],

    propTypes: {
        attributes: React.PropTypes.object,
        binary_context: React.PropTypes.string
    },

    componentWillReceiveProps: function componentWillReceiveProps(newProps) {
        var imgSrc = undefined;
        if (newProps.value && !this.state.reset) {
            if (!this.state.value || this.state.value != newProps.value) {
                imgSrc = this.getBinaryUrl(newProps.value, this.state.temporaryBinary && this.state.temporaryBinary == newProps.value);
            }
        } else if (newProps.attributes['defaultImage']) {
            if (this.state.value) {
                //this.setState({ value:'ajxp-remove-original' });
            }
            imgSrc = newProps.attributes['defaultImage'];
        }
        if (imgSrc) {
            this.setState({ imageSrc: imgSrc, reset: false });
        }
    },

    getInitialState: function getInitialState() {
        var imgSrc = undefined,
            originalBinary = undefined;
        if (this.props.value) {
            imgSrc = this.getBinaryUrl(this.props.value);
            originalBinary = this.props.value;
        } else if (this.props.attributes['defaultImage']) {
            imgSrc = this.props.attributes['defaultImage'];
        }
        return { imageSrc: imgSrc, originalBinary: originalBinary };
    },

    getBinaryUrl: function getBinaryUrl(binaryId) {
        var isTemporary = arguments.length <= 1 || arguments[1] === undefined ? false : arguments[1];

        var url = global.pydio.Parameters.get('ajxpServerAccess') + "&get_action=" + this.props.attributes['loadAction'];
        if (!isTemporary) {
            url += "&binary_id=" + binaryId;
        } else {
            url += "&tmp_file=" + binaryId;
        }
        if (this.props.binary_context) {
            url += "&" + this.props.binary_context;
        }
        return url;
    },

    getUploadUrl: function getUploadUrl(paramsOnly) {
        var params = "get_action=" + this.props.attributes['uploadAction'];
        if (this.props.binary_context) {
            params += "&" + this.props.binary_context;
        }
        if (paramsOnly) {
            return params;
        } else {
            return global.pydio.Parameters.get('ajxpServerAccess') + "&" + params;
        }
    },

    uploadComplete: function uploadComplete(newBinaryName) {
        var prevValue = this.state.value;
        this.setState({
            temporaryBinary: newBinaryName,
            value: null
        });
        if (this.props.onChange) {
            var additionalFormData = { type: 'binary' };
            if (this.state.originalBinary) {
                additionalFormData['original_binary'] = this.state.originalBinary;
            }
            this.props.onChange(newBinaryName, prevValue, additionalFormData);
        }
    },

    htmlUpload: function htmlUpload() {
        global.formManagerHiddenIFrameSubmission = (function (result) {
            result = result.trim();
            this.uploadComplete(result);
            global.formManagerHiddenIFrameSubmission = null;
        }).bind(this);
        this.refs.uploadForm.submit();
    },

    onDrop: function onDrop(files, event, dropzone) {
        if (PydioApi.supportsUpload()) {
            this.setState({ loading: true });
            PydioApi.getClient().uploadFile(files[0], "userfile", this.getUploadUrl(true), (function (transport) {
                // complete
                var result = transport.responseText.trim().replace(/<\w+(\s+("[^"]*"|'[^']*'|[^>])+)?>|<\/\w+>/gi, '');
                result = result.replace('parent.formManagerHiddenIFrameSubmission("', '').replace('");', '');
                this.uploadComplete(result);
                this.setState({ loading: false });
            }).bind(this), (function (transport) {
                // error
                this.setState({ loading: false });
            }).bind(this), function (computableEvent) {
                // progress
                // console.log(computableEvent);
            });
        } else {
                this.htmlUpload();
            }
    },

    clearImage: function clearImage() {
        var _this = this;

        if (global.confirm('Do you want to remove the current image?')) {
            (function () {
                var prevValue = _this.state.value;
                _this.setState({
                    value: 'ajxp-remove-original',
                    reset: true
                }, (function () {
                    this.props.onChange('ajxp-remove-original', prevValue, { type: 'binary' });
                }).bind(_this));
            })();
        }
    },

    render: function render() {
        var coverImageStyle = {
            backgroundImage: "url(" + this.state.imageSrc + ")",
            backgroundPosition: "50% 50%",
            backgroundSize: "cover"
        };
        var icons = [];
        if (this.state && this.state.loading) {
            icons.push(React.createElement('span', { key: 'spinner', className: 'icon-spinner rotating', style: { opacity: '0' } }));
        } else {
            icons.push(React.createElement('span', { key: 'camera', className: 'icon-camera', style: { opacity: '0' } }));
        }

        return React.createElement(
            'div',
            null,
            React.createElement(
                'div',
                { className: 'image-label' },
                this.props.attributes.label
            ),
            React.createElement(
                'form',
                { ref: 'uploadForm', encType: 'multipart/form-data', target: 'uploader_hidden_iframe', method: 'post', action: this.getUploadUrl() },
                React.createElement(
                    BinaryDropZone,
                    { onDrop: this.onDrop, accept: 'image/*', style: coverImageStyle },
                    icons
                )
            ),
            React.createElement(
                'div',
                { className: 'binary-remove-button', onClick: this.clearImage },
                React.createElement('span', { key: 'remove', className: 'mdi mdi-close' }),
                ' RESET'
            ),
            React.createElement('iframe', { style: { display: "none" }, id: 'uploader_hidden_iframe', name: 'uploader_hidden_iframe' })
        );
    }

});
module.exports = exports['default'];
