<?php
require_once ('Zend/Db/Table/Row/Abstract.php');
require_once ('user/models/vo/Comment.php');
require_once ('user/models/UserUtil.php');
require_once ('poll/models/Polls.php');

if(class_exists('Content', false))
	return;
class Content extends Zend_Db_Table_Row_Abstract implements Precurio_ObjectInterface,Precurio_ExtranetInterface, Precurio_SearchableInterface {

	CONST APPID = 'cms';
	const IMAGE_ICON = 0;
	const IMAGE_THUMBNAIL = 1;
	const IMAGE_LARGE = 2;
	const IMAGE_PROFILE = 3;
	const IMAGE_251 = 4;
	const IMAGE_271 = 5;
	const IMAGE_517 = 6;

	const PATH_PHOTOS = '/uploads/photo/';
	const PATH_PROFILE_PICS = '/uploads/profilepic/';
	const PATH_RESOURCES = '/uploads/resource/';
	const PATH_THUMBNAILS = '/uploads/thumb/';
	const PATH_ICONS = '/uploads/icon/';
	const PATH_TMP = '/uploads/tmp/';
	const PATH_CONTENT_IMAGE_251 = '/uploads/content/images/251/';
	const PATH_CONTENT_IMAGE_271 = '/uploads/content/images/271/';
	const PATH_CONTENT_IMAGE_517 = '/uploads/content/images/517/';

	const WIDTH_ICON = 50;
	const WIDTH_THUMBNAIL = 130;
	const WIDTH_PROFILE_PIC = 180;
	const WIDTH_LARGE = 465;
	const WIDTH_LOGO = 100;
	const WIDTH_IMAGE_251 = 251;
	const WIDTH_IMAGE_271 = 271;
	const WIDTH_IMAGE_517 = 517;

	const ACTIVITY_NEW = "new";
	const ACTIVITY_EDIT = "edit";
	const ACTIVITY_VIEW = "view";
	const ACTIVITY_READ = "read";
	const ACTIVITY_SHARE = "share";
	const ACTIVITY_EXTRANET_SHARE = "extranet_share";

	const VIEW_URL = "/cms/view/{1}";

	/**
	 * get the content id
	 * @return int
	 */
	public function getId()
	{
		return $this->id;
	}

	/**
	 * get the title property
	 * This is typically used as the headline for news, and title for articles.
	 * @return string
	 */
	public function getTitle()
	{
		return $this->title;
	}
	/**
	 * get the summary
	 * @return string
	 */
	public function getSummary($limit=300, $includeReadmore=false)
	{
		$content = $this->getContent();
		$baseUrl = Zend_Controller_Front::getInstance()->getBaseUrl();
		$tr = Zend_Registry::get('Zend_Translate');
		if(Precurio_Utils::isNull($content)) return "";
		
		$limit = strlen($content) < $limit ? strlen($content) : $limit;
	    
	    $summary = addslashes(substr(strip_tags($this->getContent()),0,$limit));
        
	    if($limit < strlen($content))$summary = $summary.'...';
	    
	    if($includeReadmore){
	        $summary = $summary.'<a href="'.$baseUrl.$this->getUrl().'">'.$tr->translate('read more').'</a>';
	    }
		
		$this->summary = $summary;
		
		return strip_tags($this->summary,'<a><b><i><u>');
	}
	/**
	 * Gets content owner
	 * @return User
	 */
	public function getUser()
	{
		return UserUtil::getUser($this->user_id);
	}
	/**
	 * Gets content owner (proxy to getUser())
	 * @return User
	 */
	public function getOwner()
	{
		return $this->getUser();
	}
	/**
	 * gets the content of the article
	 * @return string
	 */
	public function getContent()
	{
		if(Precurio_Utils::isNull($this->body))
		{
			if(!Precurio_Utils::isNull($this->summary))
			{
				return $this->summary;
			}
			else
			{
			//TODO if body is null, fetch content from url.

			}
		}
		return $this->body;
	}
/**
	 * gets the content of the article
	 * proxies to getContent()
	 * @return string
	 */
	public function getBody()
	{
		return $this->getContent();
	}
/**
	 * get the user_id of person who created the content
	 * @return int
	 */
	public function getUserId()
	{
		return $this->user_id;
	}
/**
	 * gets the url
	 * @return string
	 */
	public function getUrl()
	{
		$this->url = trim($this->url);
		if(empty($this->url) || $this->url == "#")
			return getLocalizedString(self::VIEW_URL,$this->getContentId());
		return $this->url;
	}
	public function getExtranetUrl()
	{
		return getLocalizedString(self::VIEW_URL,$this->getContentId());
	}
/**
	 * gets the keword
	 * @return string
	 */
	public function getKeywords()
	{
		$str = " ";
		$translate = Zend_Registry::get('Zend_Translate');
		if($this->isAdvert())$str .= $translate->translate("advert").' '.$translate->translate("advertisement").' ';
		if($this->isNews())$str .= $translate->translate("news").' ';
		if($this->isArticle())$str .= $translate->translate("featured").' '.$translate->translate("article").' ';
		if($this->is_photo)$str .= $translate->translate("picture").' '.$translate->translate("photo").' ';

		return trim($this->keyword.$str);
	}
	/**
	 * Tries to fetch an image from a content.
	 * Usually used if a content image was not uploaded and a thumbnail is required
	 * @return string
	 */
	public function getImageFromContent()
	{
		if(empty($this->body) || strlen($this->body) < 20)return "";
		$document = new DOMDocument();
		libxml_use_internal_errors(true);
		$document->loadHTML($this->body);
		libxml_clear_errors();
		$images = $document->getElementsByTagName('img');
		foreach($images as $img)
		{
			$src = $img->getAttribute('src');
			if($src)return $src;
		}
		return "";
	}
/**
	 * gets the image path of the article
	 * @return string
	 */
	public function getImagePath($type=100)
	{
		$filename = basename($this->image_path);
		$filepath = "";
		switch($type)
		{
			case self::IMAGE_LARGE:
				$filepath = self::PATH_PHOTOS;
				break;
			case self::IMAGE_THUMBNAIL:
				$filepath = self::PATH_THUMBNAILS;
				break;
			case self::IMAGE_ICON:
				$filepath = self::PATH_ICONS;
				break;
			case self::WIDTH_IMAGE_251:
				$filepath = self::PATH_CONTENT_IMAGE_251;
				break;
			case self::WIDTH_IMAGE_271:
				$filepath = self::PATH_CONTENT_IMAGE_271;
				break;
			case self::WIDTH_IMAGE_517:
				$filepath = self::PATH_CONTENT_IMAGE_517;
				break;
			default:
				$filename = $this->image_path;
				break;

		}

		$filename = $filepath.$filename;
		if(!is_file(getcwd().$filename))
			$filename = $this->image_path;
		return $filename;

	}
	/**
	 * Returns the category id for a content. Since a content can belong to mulitple
	 * categorys, we pick the first one.
	 * @return int
	 */
	public function getCategoryId()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$rows = $table->fetchAll($table->select()->where('content_id = ?',$this->getContentId())->where('active = 1'));
		//check all relationships, only return one with a valid relationship (a real category)
		foreach ($rows as $row)
		{
			if($row->category_id > 0)
				return $row->category_id;
		}

		//no valid relationship.
		if(!empty($row))
			return $row->category_id;

		if($this->isPublic())
			return RootFolder::getSpecialIdFromAccessType(Category::ACCESS_PUBLIC);
		if($this->isShared())
			return RootFolder::getSpecialIdFromAccessType(Category::ACCESS_SHARED);
		if($this->isPrivate())
			return RootFolder::getSpecialIdFromAccessType(Category::ACCESS_PRIVATE);
	}
/**
	 * Whether or not the content is active
	 * Deleted contents are inactive
	 * @return boolean
	 */
	public function isActive()
	{
		return $this->active;
	}
/**
	 * Whether or not the content is a news
	 * Deleted contents are inactive
	 * @return boolean
	 */
	public function isNews()
	{
		return false;
	}
	/**
	 * Whether or not the content is an article
	 * Deleted contents are inactive
	 * @return boolean
	 */
	public function isArticle()
	{
		return false;
	}
	/**
	 * Whether or not the content is an advert
	 * @return boolean
	 */
	public function isAdvert()
	{
		return false;
	}
	/**
	 * Whether or not the content is from an RSS source
	 * @return boolean
	 */
	public function isRSS()
	{
		return false;
	}
/**
	 * get the rss source id
	 * @return int
	 */
	public function getRssSourceId()
	{
		return $this->rss_source_id;
	}
/**
	 * get the date_created of the content
	 * @return int
	 */
	public function getDateCreated()
	{
		return trim($this->date_created);
	}
	public function getDateAdded()
	{
		$date =  new Precurio_Date($this->getDateCreated());
		return $date;
	}
	public function getDateUpdated()
	{
		$date =  new Precurio_Date($this->getDateLastUpdated(),null,'en_GB');//we are forcing to en_GB locale because of mysql
		//also get the date created
		//then return the most recent between date updated and date created.
		//All this was done because of an issue with demo data.
		$date2 = $this->getDateAdded();
		return $date > $date2 ? $date : $date2;
	}
/**
	 * get the last updated of the content
	 * @return int
	 */
	public function getDateLastUpdated()
	{
		$time = $this->last_updated;
		if(is_object($this->last_updated) && get_class($this->last_updated) == "DateTime")
			$time =  $this->last_updated->getTimestamp();//for mssql server
		return $time;
	}
	public function getSharedUsers()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USERS,rowClass=>'User'));
		$select = $table->select();
		$select->setIntegrityCheck(false);
		$select = $select->from(array('a'=>PrecurioTableConstants::USERS))
		->join(array('b'=>PrecurioTableConstants::SHARED_CONTENTS),'a.user_id = b.user_id',array('content_id'))
		->where('content_id = ?',$this->getContentId() );
		$users = $table->fetchAll($select);
		return $users;

	}
	/**
	 * Gives the specified users access to this content. Does not clear previously set users.
	 * @param int|array $user_ids
	 * @param int $sharer_id OPTIONAL - The user who shared the content. Defaults to active user.
	 */
	public function addSharedUsers($user_ids,$sharer_id=0)
	{
		if(is_int($user_ids))
		{
			$user_id = $user_ids;
			$user_ids = array();
			$user_ids[] = $user_id;
		}

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::SHARED_CONTENTS));
		$data = array();
		$data['content_id'] = $this->getContentId();
		$data['sharer_id'] = ($sharer_id ? $sharer_id :Precurio_Session::getCurrentUserId());
		$data['date_created'] = Precurio_Date::now()->getTimestamp();
		foreach($user_ids as $user_id)
		{
			$data['user_id'] = $user_id;
			$row = $table->createRow($data);
			$row->save();

			$this->recordActivity(self::ACTIVITY_SHARE, $data['sharer_id'],null,$data['user_id']);
		}
		//all shared contents must have a category id of -3.Same thing also done in groupShare
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$row = $table->fetchRow($table->select()->where('category_id = ?', RootFolder::SPECIAL_ID_SHARED)->where('content_id = ?',$this->getContentId())->where('active = 1'));


		if(empty($row))
		{
			$table->insert(array(
				'category_id'=>RootFolder::SPECIAL_ID_SHARED,
				'content_id'=>$this->getContentId(),
				'user_id'=>$data['sharer_id'],
				'date_created'=>time(),
				'active'=>1
			));
		}
		return;
	}
	/**
	 * Removes specified users from having access to this content.
	 * This function was created because of the helpdesk module
	 * @param int|array $user_ids
	 */
	public function removeSharedUsers($user_ids)
	{
		if(is_int($user_ids))
		{
			$user_id = $user_ids;
			$user_ids = array();
			$user_ids[] = $user_id;
		}

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::SHARED_CONTENTS));
		$content_id = $this->getContentId();
		foreach($user_ids as $user_id)
		{
			$table->delete("user_id = $user_id AND content_id = $content_id");
		}
	}
	/**
	 * Shares a content with users, will remove any existing user sharing
	 * @param array|int $user_ids
	 * @return null
	 */
	public function setSharedUsers($user_ids)
	{
		if(empty($user_ids))$user_ids = array();
		//get users already shared with.
		$sharedUsers = $this->getSharedUsers();
		$sharedUserIds  = array();
		foreach($sharedUsers as $user)$sharedUserIds[] = $user->user_id;

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::SHARED_CONTENTS));
		//delete all previous share.
		$where = $table->getAdapter()->quoteInto('content_id = ?',$this->getContentId() );
		$table->delete($where);
		$data = array();
		$data['content_id'] = $this->getContentId();
		$data['sharer_id'] = Precurio_Session::getCurrentUserId();
		$data['date_created'] = Precurio_Date::now()->getTimestamp();
		foreach($user_ids as $user_id)
		{
			$data['user_id'] = $user_id;
			$row = $table->createRow($data);
			$row->save();
			if(!in_array($user_id, $sharedUserIds))//not shared with this user before
			{
				$this->recordActivity(self::ACTIVITY_SHARE, $data['sharer_id'], $data['date_created'],$data['user_id']);
			}
		}
		if(!count($user_ids)) return;

		//all shared contents must have a category id of -3.Same thing also done in groupShare
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$row = $table->fetchRow($table->select()->where('category_id = ?', RootFolder::SPECIAL_ID_SHARED)->where('content_id = ?',$this->getContentId())->where('active = 1'));


		if(empty($row))
		{
			$table->insert(array(
				'category_id'=>RootFolder::SPECIAL_ID_SHARED,
				'content_id'=>$this->getContentId(),
				'user_id'=>Precurio_Session::getCurrentUserId(),
				'date_created'=>time(),
				'active'=>1
			));
		}
		return;
	}
	public function getComments()
	{
		return Comment::getAll($this->getItemId(), $this->getAppId());
	}
	public function getContentId()
	{
		return isset($this->content_id) ? $this->content_id : $this->id;
	}
	public function getShortHeadline($n=46)
	{
		if(strlen($this->getTitle())>$n)
			return substr($this->getTitle(),0,46).'...';
		else
			return $this->getTitle();
	}
	public function getSinceWhen()
	{
		$date = new Precurio_Date($this->getDateCreated());
		return $date->getHowLongAgo();
	}
	public function getFullName()
	{
		$user = UserUtil::getUser($this->getUserId());
		if(empty($user))return "";
		return $user->getFullName();
	}
	public function getOwnersName()
	{
		return $this->getFullName();
	}
	public function getAccessType()
	{
		if($this->isShared())return Category::ACCESS_SHARED;
		if($this->isPublic())return Category::ACCESS_PUBLIC;
		if($this->isPrivate())return Category::ACCESS_PRIVATE;
	}
	/**
	 * Gets Categorys that contain this content
	 * @return Zend_Db_Table_Rowset of Category objects
	 */
	public function getCategorys()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Category'));
		$select = new Zend_Db_Table_Select($table);
		$select->setIntegrityCheck(false);
		$select = $select->from(array('a' => PrecurioTableConstants::CATEGORYS))
						->join(array('b' => PrecurioTableConstants::CONTENT_CATEGORYS),'a.id = b.category_id',array())
						->where('b.content_id = ?',$this->getContentId())
						->where('a.active=1')
						->where('b.active=1')
						->order('a.title asc');
		return $table->fetchAll($select);

	}
	/**
	 * Determines if the content also belongs to a particular group
	 * @param $group_id int
	 * @return Boolean
	 */
	public function belongsToGroup($group_id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CONTENTS));
		$row = $table->fetchRow($table->select()->where('content_id = ?',$this->getContentId())->where('group_id = ?',$group_id));
		return $row == null ? false : true;
	}
	/**
	 * Returns an array of group objects that have been given access to the content
	 * @return array
	 */
	public function getGroups()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CONTENTS));
		$items = $table->fetchAll($table->select()->where('content_id = ?',$this->getContentId())->where('active = 1'));
		$groups = array();
		foreach ($items as $item)
		{
			$group = UserUtil::getGroup($item->group_id);
			if(!empty($group))$groups[] = $group;
		}
		return $groups;
	}
	/**
	 * Shares a content with groups, will remove any existing group sharing
	 * @param array|int $group_ids
	 * @return null
	 */
	public function groupShare($group_ids)
	{
		if(empty($group_ids))$group_ids = array();
		if(!is_array($group_ids))$group_ids = array($group_ids);

		//get users already shared with.
		$sharedUsers = $this->getSharedUsers();
		$sharedUserIds  = array();
		foreach($sharedUsers as $user)$sharedUserIds[] = $user->user_id;

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CONTENTS));
		$table->update(array('active'=>0), "content_id = ".$this->getContentId());

		$user_id = Precurio_Session::getCurrentUserId();
		$time = Precurio_Date::now()->getTimestamp();
		foreach($group_ids as $id)
		{
			$table->insert(array(
				'content_id'=>$this->getContentId(),
				'group_id'=>$id,
				'task_id'=>0,
				'date_created'=>$time,
				'active'=>1
			));
			//notify group members
			$members = UserUtil::getGroup($id)->getUsers();;
			foreach($members as $user)
			{
				if(!in_array($user->getId(), $sharedUserIds))//not shared with this user before
				{
					$this->recordActivity(self::ACTIVITY_SHARE, $user_id,null,$user->getId());
				}
			}
		}
		if(!count($group_ids)) return;

		//all shared contents must have a category id of -3.Same thing also done in setSharedUsers
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$row = $table->fetchRow($table->select()->where('category_id = ?', RootFolder::SPECIAL_ID_SHARED)->where('content_id = ?',$this->getContentId())->where('active = 1'));


		if(empty($row))
		{
			$table->insert(array(
				'category_id'=>RootFolder::SPECIAL_ID_SHARED,
				'content_id'=>$this->getContentId(),
				'user_id'=>Precurio_Session::getCurrentUserId(),
				'date_created'=>time(),
				'active'=>1
			));
		}
		return;
	}
	public function userHasRated($user_id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_RATINGS));
		$row = $table->fetchRow($table->select()->where('content_id = ?',$this->getContentId())->where('user_id = ?',$user_id));
		return $row == null ? false : true;
	}
	/**
	 * Delte content
	 * @param int $parent_id - If $deleteCopies is false, the function only removes the content from the category specified by parent_id (OPTIONAL)
	 * @param int $deleteCopies - If set to true, content is deleted. Defaults to true (OPTIONAL).
	 */
	public function do_delete($parent_id=0,$deleteCopies = true)
	{
		if($deleteCopies)
		{
			$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT));
			$row = $table->find($this->getContentId())->current();
			if($row)
			{
				$row->active = 0;
				$row->save();
				$this->index();
			}
		}
		else
		{
			if($parent_id < 0)
			{
				//this usually should not happen
				//but if this happens, then you are trying to delete an app document which has copies.)
				return;
			}
			$category = Category::getCategory($parent_id);
			$category->removeContents(array($this->getContentId()));
		}

	}

	public function isPublic()
	{
		return $this->is_public;
	}
	public function isShared()
	{
		//shared with users?
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::SHARED_CONTENTS));
		$row = $table->fetchRow($table->select()->where('content_id = ?',$this->getContentId())->where('active = 1'));
		if(!empty($row))
			return true;

		//shared with groups?
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CONTENTS));
		$row = $table->fetchRow($table->select()->where('content_id = ?',$this->getContentId())->where('active = 1'));
		if(!empty($row))
			return true;

		return false;
	}
	public function isPrivate()
	{
		return (!$this->isShared() && !$this->isPublic());//not shared and not public
	}
	public function isDocument()
	{
		return $this->is_document;
	}
	public function isPage()
	{
		return $this->is_page;
	}
	/**
	 * Determines if a user has access to a particular content.
	 * Note that this has no effect on any previously set access rules. I.e If a user
	 * has not been given "view" privilege to the content module, he will not be able
	 * to view the content even if this function returns TRUE.
	 * @param $user_id
	 * @return Boolean
	 */
	public function canAccess($user_id)
	{
		if(!$this->isActive())return false;//if the content has been deleted, then no access;
		if($this->user_id == $user_id)return true;//content creator can access

		$user = UserUtil::getUser($user_id);
		if(empty($user))return false;//probably user no longer exists
		if($this->isPublic() && !$user->isExternal())return true;//extranet users do not have access to public documents.
		//ok, content is not public
		//check if content has been shared with a group the user belongs to
		$groups = $user->getGroups();
		foreach ($groups as $group)
		{
			if($this->belongsToGroup($group->id))
				return true;
		}
		//ok, content was not shared with any group the user belongs to
		//check if content was shared specifically with user
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::SHARED_CONTENTS));
		$row = $table->fetchRow($table->select()->where('user_id = ?',$user_id)->where('content_id = ?',$this->getContentId()));
		if(!empty($row))
		{
			return true;
		}

		//ok content was not shared, now check if folder was shared
		if($user->isExternal())return false;//extranet users should not have access to any document nor directly shared
		$config = Zend_Registry::get('config');
		if(!empty($config->inherit_folder_access))
		{

			$categorys = $this->getCategorys();
			foreach($categorys as $category)
			{
				if($category->canAccess($user_id))
					return true;
			}
		}

		return false;
	}
	/**
	 * Determines if a user can modify the content.
	 * A user given modiy access has full rights to the content
	 * @param int $user_id
	 * @return boolean
	 */
	public function canModify($user_id)
	{
		return $this->user_id == $user_id || UserUtil::isAllowed("admin_index");
	}
	/**
	 * Determines if a user can share the content.
	 * @param int $user_id
	 * @return boolean
	 */
	public function canShare($user_id)
	{
		return $this->user_id == $user_id;
	}
	/**
	 * Determines a content has been locked for editting.
	 * Locked documents cannot be updated.
	 * @return int - The user id of the person that locked it, or 0 if not locked.
	 */
	public function isLocked()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_LOCK_HISTORY));
		$row = $table->fetchRow($table->select()->where('content_id = ?',$this->getContentId())->where('active = 1')->order('id desc'));
		if(empty($row))
			return 0;
		if($row->lock == 0)
			return 0;

		return $row->user_id;
	}
	/**
	 * Lock a content for editing
	 */
	public function lock()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_LOCK_HISTORY));
		$table->insert(array(
			'content_id'=>$this->getContentId(),
			'user_id'=>Precurio_Session::getCurrentUserId(),
			'lock'=>1,
			'active'=>1,
			'date_created'=>time()
		));
	}
	/**
	 * unlock a locked document
	 */
	public function unlock()
	{
		$content_id = $this->getContentId();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_LOCK_HISTORY));
		$table->update(array("active"=>0), "content_id = $content_id");
		$table->insert(array(
			'content_id'=>$this->getContentId(),
			'user_id'=>Precurio_Session::getCurrentUserId(),
			'lock'=>0,
			'active'=>1,
			'date_created'=>time()
		));
		return;
	}
	public function addDocument($fileCtrl='document')
	{
		//return;
		$root = Zend_Registry::get('root');
		$upload = new Precurio_Upload;
		$upload->uploadFile(self::PATH_TMP);
		$filename = $upload->_files[$fileCtrl];
		$filePath = $root.'/public/'.self::PATH_TMP.$filename;
		if($fp = fopen($filePath,'rb'))
		{
			$file_content = fread($fp,filesize($filePath));
			fclose($fp);
			$data = array(
				'content_id'=>$this->getContentId(),
				'file_name'=>$upload->getFileTitle($fileCtrl),
				'file_type'=>$upload->getFileExt($fileCtrl),
				'file_size'=>filesize($filePath),
				'file_author'=>$upload->getFileOwner($filePath),
				'file_date_created'=>filemtime($filePath),
				'file_content'=>$file_content,
				'active'=>1,
				'date_created'=>Precurio_Date::now()->getTimestamp()
			);
			$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENTS));
			$row = $table->createRow($data);
			$document_id = $row->save();

			unlink($filePath);//delete the file
		}

	}

	/**
	 * Get template to use for content. (templates are found in default/layouts/[THEME]/templates
	 * @return string
	 */
	public function getTemplate()
	{
		if(empty($this->template))return $this->getDefaultTemplate();
		return $this->template;
	}
	/**
	 * Get default template.
	 * This method will be overwritten by subclasses of content (e.g. Page, Video, etc)
	 * @return string
	 */
	public function getDefaultTemplate()
	{
		return 'content';
	}
	/**
	 * Determine if comments has been disabled for this content
	 * @return boolean
	 */
	public function commentsDisabled()
	{
		return isset($this->disable_comments) ? $this->disable_comments : ($this->isPage() ? true: false);
	}
	/**
	 * Determine if you want to hide key content info like author, date etc. Defaults to false.
	 * This method will be overwitten by subclasses that need to hide info e.g Page
	 * @return boolean
	 */
	public function hideInfo()
	{
		return false;
	}
	/**
	 * @param $fileCtrl String - Name of the input file control used for the file upload, default='file'
	 * @param $isProfilePhoto Boolean - Set whether this is a profile picture or not, default=false
	 * @param $isLogo Boolean - Set whether this is a logo image, default = false
	 * @return String - Path to thumbnail version of the photo.
	 */
	public static function addPhoto($fileCtrl='file',$isProfilePhoto=false,$isLogo=false)
	{
		$upload = new Precurio_Upload;
		$upload->uploadFile(self::PATH_TMP);
		$filename = $upload->_files[$fileCtrl];
		if($isProfilePhoto)
		{
			$upload->setMaxWidth(self::WIDTH_PROFILE_PIC);
			$upload->imgResize($upload->_files[$fileCtrl],self::PATH_TMP, self::WIDTH_PROFILE_PIC,true,self::PATH_PROFILE_PICS,true);
		}

		if(!$isLogo)
		{
			$upload->setMaxWidth(self::WIDTH_LARGE);
			$upload->imgResize($filename,self::PATH_TMP, self::WIDTH_LARGE,true,self::PATH_PHOTOS);
		}


		$upload->setMaxWidth(self::WIDTH_THUMBNAIL);
		$upload->imgResize($filename,self::PATH_TMP, self::WIDTH_THUMBNAIL,true,self::PATH_THUMBNAILS,true);

		$upload->setMaxWidth(self::WIDTH_ICON);
		$upload->imgResize($filename,self::PATH_TMP, self::WIDTH_ICON,true,self::PATH_ICONS,true);

		$filePath = $isProfilePhoto ? self::PATH_PROFILE_PICS.$filename : self::PATH_THUMBNAILS.$filename;

		return $filePath;

	}

	/**
	 * @param $fileCtrl String - Name of the input file control used for the file upload, default='file'
	 * @return String - Path to 271 width version of the photo.
	 */
	public static function addContentImage($fileCtrl='file')
	{
		$upload = new Precurio_Upload;
		$upload->uploadFile(self::PATH_TMP);

		$filename = $upload->_files[$fileCtrl];

		$upload->setMaxWidth(self::WIDTH_IMAGE_251);
		$upload->imgResize($filename,self::PATH_TMP, self::WIDTH_IMAGE_251,true,self::PATH_CONTENT_IMAGE_251);

		$upload->setMaxWidth(self::WIDTH_IMAGE_271);
		$upload->imgResize($filename,self::PATH_TMP, self::WIDTH_IMAGE_271,true,self::PATH_CONTENT_IMAGE_271);

		$upload->setMaxWidth(self::WIDTH_IMAGE_517);
		$upload->imgResize($filename,self::PATH_TMP, self::WIDTH_IMAGE_517,true,self::PATH_CONTENT_IMAGE_517);


		$filePath = self::PATH_CONTENT_IMAGE_271.$filename;

		return $filePath;

	}

	/**
	 * Create a new content.
	 * @param array $data
	 * @param boolean $logActivity - Determine is activity should be logged. This is used by Content sub classes to prevent double notification. Defauts to True
	 * @return id
	 */
	public static function createNew($data,$logActivity=true)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT));
		$row = $table->createRow();$row->setFromArray($data);
		$content_id = $row->save();
		if($logActivity)
			Precurio_Activity::create(self::APPID,self::ACTIVITY_NEW,$content_id,$data['user_id']);
		try
		{
			if($logActivity)
			{
				$content = MyContents::getContent($content_id,false);
				$content->index();
			}
		}
		catch(Exception $e)
		{

		}
		return $content_id;
	}
	/**
	 * Called when content needs to be updated
	 * @param array $params
	 * @param int $user_id - the user performing the update
	 * @return int  - content id
	 * @version 4
	 */
	public function do_update($params,$user_id)
	{
		$this->setFromArray($params);
		$this->save();
		$this->recordActivity(self::ACTIVITY_EDIT, $user_id);
		try
		{
			$this->index();
		}
		catch(Exception $e)
		{

		}
		return $this->getContentId();
	}

	/**
	 * Method is called by the controller anytime content is viewed by a user.
	 * @param int $user_id
	 */
	public function viewed($user_id)
	{
		if(empty($user_id))return;

		//first log a record of this user view in the database.
		$this->recordActivity(self::ACTIVITY_VIEW, $user_id);

		//now lets see if we should increase hits
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT,'rowClass'=>'Content'));
		$content = $table->find($this->getContentId())->current();
		//do not increase hits for content author
		if($content->user_id == $user_id)
		{
			unset($content);
			return;
		}

		//if content has been viewed by user in current session, do not increase hits
		try
		{
			$ns = new Zend_Session_Namespace('user');
			if(!$ns->viewed_content)$ns->viewed_content = array();
			if(in_array($this->getContentId(), $ns->viewed_content))
			{
				unset($content);
				return;
			}
			$ns->viewed_content[] = $this->getContentId();
		}
		catch (Exception $e){}

		$content->num_of_hits++;
		$content->save();
	}
	/**
	 * Removes all content to category mapping for the content.
	 * The Content simply becomes ophaned (i.e. having no category)
	 */
	public function removeCategorys()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$table->update(array('active'=>0),'content_id = '.$this->getContentId());
	}
	/**
	 *  Returns view activity logs
	 * @param $unique boolean - Set to true if you do not want multiple view log from a user. Default is true.
	 * @return array
	 */
	public function getViewLog($unique=true)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::ACTIVITY_LOG,'rowClass'=>'ActionLog'));
		$rows = $table->fetchAll($table->select()->where('active = 1')
				->where('item_id = ?',$this->getId())
				->where('appid = ?',$this->getAppId())
				->where('type = ?',self::ACTIVITY_VIEW));
		if(!$unique)return $rows;
		$temp = array();
		foreach($rows as $row)
		{
			$temp[$row->user_id] = $row;
		}
		return $temp;
	}
	/**
	 * Shares a content with external users.
	 * @param array $emails
	 */
	public function extranetShare($emails)
	{
		if(empty($emails))return;
		if(!is_array($emails))$emails = array($emails);

		$user_id = Precurio_Session::getCurrentUserId();
		$time = Precurio_Date::now()->getTimestamp();

		$activity_id = $this->recordActivity(self::ACTIVITY_EXTRANET_SHARE, $user_id,$time,-1);
		foreach($emails as $email)
		{
			Precurio_Extranet::invite($user_id, $email,$activity_id);
		}
		return;
	}
	/* Get all users with access to this content.
	 * Returns null if it is a public content
	 * @see Precurio_ObjectInterface::getUsers()
	 */
	public function getUsers()
	{
		if($this->isPublic())return null;

		$result = array();

		//get groups and add members
		$groups = $this->getGroups();
		foreach($groups as $group)
		{
			$members = $group->getUserMembers();
			foreach($members as $user)
				$result[$user->user_id] = $user;
		}

		//get users
		$users = $this->getSharedUsers();
		foreach($users as $user)
			$result[$user->user_id] = $user;

		//also include content owner
		$user = $this->getOwner();
		$result[$user->user_id] = $user;

		return $result;

	}
	public function getItemId()
	{
		return $this->getContentId();
	}
	public static function getAppId()
	{
		return Content::APPID;
	}
	public function __toString()
	{
		return $this->getTitle();
	}
	/**
	 * Determines if a user has read a content.
	 * If content has a "confirm read" option enabled, the function uses that, otherwise we assume a view is a read.
	 * @param unknown $user_id
	 * @return boolean
	 */
	public function isRead($user_id)
	{
		if($this->isPage())return true;//pages are read. This prevents the "Mark as Read" button from showing on pages.
		if(isset($this->confirm_read) && $this->confirm_read)
			return Precurio_Activity::exists(self::getAppId(), self::ACTIVITY_READ, $this->getContentId(), $user_id);
		else
			return Precurio_Activity::exists(self::getAppId(), self::ACTIVITY_VIEW, $this->getContentId(), $user_id);
	}
	/**
	 * Generates an excel report for the content and returns a path to the file.
	 * @return string
	 */
	public function downloadExcel()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::ACTIVITY_LOG));

		$rows = $table->fetchAll($table->select()->where('type = ?',self::ACTIVITY_READ)->where('appid = ?',self::getAppId())->where('item_id = ?',$this->getItemId()));


		$data = "<table>";
		$data .= "<tr><th>User</th><th>Read Date</th></tr>";

		foreach($rows as $row)
		{
			$user = UserUtil::getUser($row->user_id);
			$date = new Precurio_Date(trim($row->date_created));
			$data .= "<tr><td>".$user->getFullName()."</td><td>".$date->toString()."</td></tr>";
		}

		$data .= "</table>";

		$root = Zend_Registry::get('root');
		$fileFn = new Precurio_FileFn();
		$filePath = $root.'/public/'.self::PATH_TMP.$fileFn->cleanFilename($this->getTitle()).".xls";
		file_put_contents($filePath,$data);

		return $filePath;
	}
	public function index()
	{
		//contents without title will be ignored.
		if(Precurio_Utils::isNull($this->title))
			return false;

		$index = Precurio_Search::getIndex();

		//first remove content from the index if it exists
		$hits = $index->find('id:' . $this->getId().' AND module:'.self::APPID);
		foreach ($hits as $hit)
			$index->delete($hit->id);
		$index->commit();

		if(!$this->isActive())return;

		//now create a new index document
		$doc = new Zend_Search_Lucene_Document();

		$content = $this;

		$doc->addField(Zend_Search_Lucene_Field::keyword('id',$content->getId()));
		$doc->addField(Zend_Search_Lucene_Field::keyword('user_id',$content->getUserId()));
		$doc->addField(Zend_Search_Lucene_Field::text('title',$content->getTitle()));
		$doc->addField(Zend_Search_Lucene_Field::text('fullname',$content->getFullName()));
		if($content->getUrl() != '#')
			$doc->addField(Zend_Search_Lucene_Field::text('url',$content->getUrl()));
		$doc->addField(Zend_Search_Lucene_Field::text('tags',$content->getKeywords()));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('image_path',$content->getImagePath()));
		$doc->addField(Zend_Search_Lucene_Field::text('summary',$content->getSummary()));
		$doc->addField(Zend_Search_Lucene_Field::unStored('body',$content->getBody()));
		$doc->addField(Zend_Search_Lucene_Field::unStored('comments',$this->getCommentsContent()));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('is_photo',$content->is_photo));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('is_document',$content->is_document));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('date',$content->getDateAdded()));
		$doc->addField(Zend_Search_Lucene_Field::keyword('module',$content->getAppId()));

		$index->addDocument($doc);

		return $index;
	}
	/**
	 * Gets all the comment as a string for search purpose
	 * @return string
	 */
	private function getCommentsContent()
	{
		$comments = Comment::getAll($this->getId(), self::APPID);
		$body = "";
		foreach($comments as $comment)
			$body = $body.' '.$comment->message;
		return $body;
	}
	protected function recordActivity($type,$user_id,$date_created=null,$onlyNotifyId=0)
	{
		return Precurio_Activity::create($this->getAppId(),$type,$this->getItemId(),$user_id,$date_created,$onlyNotifyId);
	}
	/* (non-PHPdoc)
	 * @see Precurio_ExtranetInterface::addExtranetUser()
	 */
	public function addExtranetUser($user_id,$sharer_id)
	{
		$this->addSharedUsers($user_id,$sharer_id);
	}
	
	public function hasImage()
	{
	    return !empty($this->image_path);
	}
	
	public function hasPoll()
	{
	    try{
	        return $this->poll_id > 0;
	    }
	    catch(Zend_Db_Exception $e){
	        return false;
	    } 
	        
	}
	/**
	 * Gets a poll attached to a content
	 * @return Poll
	 */
	public function getPoll()
	{
	    try{
	        return Polls::getPoll($this->poll_id);
	    }
	    catch (Zend_Db_Exception $e){
	        return new stdClass();
	    }
	}
}
?>