<?php

require_once ('Zend/Db/Table/Row/Abstract.php');
require_once('user/models/UserUtil.php');
require_once 'workflow/models/vo/Process.php';

class UserProcess extends Zend_Db_Table_Row_Abstract implements Precurio_ObjectInterface {
	CONST APPID = 'workflow';
	const VIEW_URL = "/workflow/view/{1}";

	const ACTIVITY_NEW = "new";
	const ACTIVITY_WAITING_APPROVAL = "pending";
	const ACTIVITY_APPROVED = "approved";
	const ACTIVITY_REJECTED = "rejected";
	const ACTIVTY_TRANSFERED = "transfered";
	const ACTIVITY_COMPLETED = "completed";
	const ACTIVITY_EMAIL_FORM = "form_email";
	/**
	 * get the id
	 * @return int
	 */
	public function getId()
	{
		return $this->id;
	}
	public function getItemId()
	{
		return $this->getId();
	}
	public function getAppId()
	{
		return self::APPID;
	}
	public function getTitle()
	{
		return $this->getDisplayName();
	}
	public function getUrl()
	{
		return getLocalizedString(self::VIEW_URL,$this->getId());
	}
	protected function recordActivity($type,$user_id,$date_created=null,$onlyNotifyId=0)
	{
		Precurio_Activity::create($this->getAppId(),$type,$this->getItemId(),$user_id,$date_created,$onlyNotifyId);
		//all activities are also logged on the original user process. This is just so that we can get an activity log on for the workflow
		$id = $this->getStartProcess()->getId();
		Precurio_Activity::create($this->getAppId(),$type,$id,$user_id,$date_created,-2);
		return;
	}
	public function getProcessId()
	{
		return $this->process_id;
	}
	public function getStateId()
	{
		return $this->state_id;
	}
	public function getFormId()
	{
		return $this->form_id;
	}
	public function getUserId()
	{
		return $this->user_id;
	}
	public function getTaskId()
	{
		return $this->task_id;

	}
	public function getDate()
	{
		$date = new Precurio_Date($this->date_created);
		return $date->get(Precurio_Date::DATE_SHORT);
	}
	public function getStatus()
	{
		//if you are going to change this status to reflect 'approved' or 'not approved'
		//for every user, then consider the fact that in the case of a rejection, i.e non approval
		// only the rejectee and the process owner will be able to know the particlar form
		//was 'NOT APPROVED', every other user inbetween the process, sees it as 'APPROVED'
		//this behaviour can be changed by making sure ProcessManager::rejectProcess() updates
		//the 'rejected' flag of all process within a form.
		$tr = Zend_Registry::get('Zend_Translate');
		if($this->task_id == 0)
			return $this->completed == 1 ? ($this->rejected == 1 ? $tr->translate('NOT APPROVED') :$tr->translate('APPROVED')) : $tr->translate('PENDING');
		else
			return $this->completed == 1 ? $tr->translate('DONE') : $tr->translate('PENDING');
	}
	public function getStatusCode()
	{
		if($this->isRejected())return Task::STATUS_REJECT;
		if($this->isCompleted())return Task::STATUS_COMPLETE;
		return Task::STATUS_OPEN;
	}
	public function getComments()
	{
		return Comment::getAll($this->getItemId(), self::APPID);
	}
	public function getFormCode()
	{
		$process = $this->getProcess();
		return $process->getCode().'-'.$this->getFormId();
	}
	public function getDisplayName($withFormCode = false)
	{
		$process = $this->getProcess();
		$displayName = $process->getDisplayName();

		if($withFormCode)
		{
			$displayName = $displayName.' - '.$this->getFormCode();
		}
		return $displayName;
	}
	public function getTableName()
	{
		$process = $this->getProcess();
		return $process->getTableName();
	}
	public function isApproved()
	{
	    return  $this->isCompleted() && !$this->rejected;
	}
	public function isRejected()
	{
		return $this->rejected;
	}
	public function isCompleted()
	{
		return ($this->completed == Task::STATUS_COMPLETE);
	}
	/**
	 * @return Process
	 */
	public function getProcess()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::WORKFLOW,'rowClass'=>'Process'));
		$process = $table->fetchRow($table->select()->where('id = ?',$this->process_id));
		return $process;
	}
	public function getColourCode()
	{
		$process = $this->getProcess();
		return $process->getColourCode();
	}
	/**
	 * Returns the actual owner of the entire process,i.e the person who initiated the process
	 * @return User
	 */
	public function getOwner()
	{
		$userProcess = $this->getStartProcess();
		if(empty($userProcess))
		{
			$this->do_delete();//delete this user process;
			return "Original process has been deleted, please refresh browser";
		}
		return UserUtil::getUser($userProcess->getUserId());
	}
	/**
	 * Gets the initialising user process. I.e. The start state user process
	 * @return UserProcess
	 */
	public function getStartProcess()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS,'rowClass'=>'UserProcess'));
		return  $table->fetchRow($table->select()->where('task_id = 0')->where('process_id = ?',$this->getProcessId())->where('form_id = ?', $this->getFormId()));

	}
	/**
	 * Determines if a user can view a form, by checking if the user is in the process flow.
	 * Method is a proxy to canAccess()
	 * @param $user_id int user id
	 * @return Boolean
	 */
	public function userCanView($user_id)
	{
		return $this->canAccess($user_id);
	}
	/**
	 * Returns the User Object currently handling the next approval level
	 * @return User
	 */
	public function getCurrentApprover()
	{
		//the next approver is the person who has a task that is not completed.
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS));
		$user_process = $table->fetchRow($table->select()->where('task_id <> 0')
															->where('completed = 0')
															->where('form_id = ?', $this->getFormId())
															->where('process_id = ?',$this->getProcessId())
															->where('active = 1'));
		if($user_process == null)return "N/A";

		return UserUtil::getUser($user_process->user_id);
	}

	public function getRejectionComment()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS_REJECT));
		$row = $table->fetchRow($table->select()->where('user_process_id = ?',$this->getId()));
		if($row == null)return '[NO REASON GIVEN]';
		return Precurio_Utils::isNull($row->comment) ? '[NO REASON GIVEN]' : $row->comment;
	}
	/**
	 * Returns activity log for workflow
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getChanges()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::ACTIVITY_LOG, 'rowClass'=>'UserActivity'));
		$select = $table->select()
		->where('active=1')
		->where('item_id = ?',$this->getStartProcess()->getId())
		->where('appid = ?',$this->getAppId())
		->order('date_created ASC');

		return $table->fetchAll($select);
	}
	/**
	 * Proxy to function getChanges()
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getHistory()
	{
		return $this->getChanges();
	}
	/**
	 * Return all users that can access this workflow. They must have been involved at one stage of the workflow.
	 * @return collection
	 */
	public function getUsers()
	{
		//users who have been part of the approval process also have access.
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USERS,'rowClass'=>'User'));
		$select = $table->select(false);
		$select->setIntegrityCheck(false);
		$select->setTable($table);

		$select = $select->distinct()
		->from(array('a' => PrecurioTableConstants::USERS))
		->join(array('b' => PrecurioTableConstants::USER_PROCESS),'a.user_id = b.user_id',array('form_id','process_id'))
		->where('a.active = 1 AND b.active= 1')
		->where('form_id = ?',$this->getFormId())
		->where('process_id = ?',$this->getProcessId());

		$rows = $table->fetchAll($select);
		return $rows;
	}
	/**
	 * Determines if a user has view access to view this process
	 * @param int $user_id
	 * @return boolean
	 */
	public function canAccess($user_id)
	{
		if($user_id == $this->user_id)
			return true;
		if($this->userIsObserver($user_id))
			return true;
		//users who have been part of the approval process also have access.
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS));
		$row = $table->fetchRow($table->select()->where('user_id = ?',$user_id)
												->where('form_id = ?',$this->getFormId())
												->where('process_id = ?',$this->getProcessId()));
		return !empty($row);
	}

	/**
	 * Determines if a user has view access to modify this process. Only modify action is delete
	 * @param int $user_id
	 * @return boolean
	 */
	public function canModify($user_id)
	{
		return ($this->getStartProcess()->user_id == $user_id);
	}
	public function getPrimaryField()
	{
		$process = $this->getProcess();

		if(empty($process->title_field))
			return $this->getFormCode();

		$titleField = $process->title_field;
		$table = new Zend_Db_Table(array('name'=>$process->getTableName()));
		$row = $table->fetchRow($table->select()->where('id = ?',$this->getFormId()));
		return '('.$this->getFormCode().')  '.substr($row->{$titleField},0,50);

	}
	/**
	 * Gets the form Data
	 * @return Zend_Db_Table_Row_Abstract
	 */
	public function getFormData()
	{
		$process = $this->getProcess();
		
		$table = new Zend_Db_Table(array('name'=>$process->getTableName()));
		$row = $table->fetchRow($table->select()->where('id = ?',$this->getFormId()));
		
		return $row;
	}
	public function do_delete()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS));
		$rows = $table->fetchAll($table->select()->where('process_id = ?',$this->getProcessId())
										->where('form_id = ?',$this->getFormId()));
		foreach ($rows as $row)
		{
			$row->active = 0;
			$row->save();
		}
	}

	/**
	 * transfer process to another user
	 * @param int $user_id
	 * @param int $from_user_id
	 */
	public function transfer($user_id,$from_user_id)
	{
		if(empty($this->task_id) || ($this->user_id != $from_user_id))return;

		//delete process
		$this->active = 0;
		$this->save();

		//transfer task to new user. transfer function also deletes  task to old user relationship
		$task = Task::get($this->task_id);
		$task->transfer($from_user_id, $user_id, "");

		//create new approval process for the same state
		$state = ProcessState::get($this->state_id);
		$processManager = new ProcessManager($this->form_id);
		$processManager->createApprovalProcess($state, $this->task_id);
	}
	/**
	 * This function simply marks a process task has completed
	 * @return void
	 */
	public function taskCompleted()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK));
		$task = $table->find($this->getTaskId())->current();
		$task->status = Task::STATUS_COMPLETE;
		$task->save();
	}
	/**
	 * Called when a process is approved.
	 */
	public function approve()
	{
		//first mark task as complete
		$this->taskCompleted();

		//then mark user process as complete
		$this->completed = Task::STATUS_COMPLETE;
		$this->save();

		$this->recordActivity(self::ACTIVITY_APPROVED, $this->user_id,null,$this->getOwner()->getId());
	}
	/**
	 * Called when a process is not approved
	 */
	public function reject()
	{
		//first mark task as complete
		$this->taskCompleted();

		//then mark user process as complete and rejected
		$this->completed = Task::STATUS_COMPLETE;
		$this->rejected = 1;
		$this->save();
		$this->recordActivity(self::ACTIVITY_REJECTED, $this->user_id);//only owner of process can possibly reject

		$this->complete();
	}
	/**
	 * The overall process is complete.
	 * This function only gets called, if there is no more state left.
	 * @param int $user_id - User that completed the workflow
	 */
	public function complete($user_id = 0)
	{
		if(empty($user_id))$user_id = Precurio_Session::getCurrentUserId();
		$startProcess = $this->getStartProcess();
		$startProcess->completed = Task::STATUS_COMPLETE;
		$startProcess->rejected = $this->rejected;//if this process is rejected, also set the start process
		$startProcess->save();

		$this->recordActivity(self::ACTIVITY_COMPLETED, $user_id,null,$startProcess->user_id);//only notify process owner
	}
	public function isActive()
	{
		return $this->active;
	}
	public function __toString()
	{
		return $this->getTitle();
	}
	/**
	 * Determines if the user is an observer
	 * @param int $user_id
	 * @return boolean
	 */
	public function userIsObserver($user_id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::WORKFLOW_OBSERVERS));
		$row =  $table->fetchRow($table->select()->where("process_id = ?",$this->getProcessId())->where("user_id = ?",$user_id)->where("active = 1"));
		return empty($row) ? false : true;
	}
}

?>