<?php

require_once 'task/models/vo/Task.php';
require_once 'workflow/models/vo/ProcessState.php';
require_once 'workflow/models/vo/UserProcess.php';
require_once 'user/models/vo/User.php';
require_once 'user/models/UserUtil.php';
require_once 'task/models/Tasks.php';
class ProcessManager
{
	/**
	 * @var ProcessState
	 */
	private $currentState;
	private $form_id;

	private $approver_id; //will be set in the 'user can change approver' scenerio

	/**
	 * @param $form_id int - will be 0 for a new form
	 * @param $approver_id int - The user that will approve the next process. Parameter is only set when 'user can change approver'.
	 * @return unknown_type
	 */
	public function __construct($form_id,$approver_id = 0)
	{
		$this->form_id = $form_id;
		$this->approver_id = $approver_id;
	}
	public function newProcess($startState)
	{
		$this->currentState = $startState;
		$userProcess = $this->createApprovalProcess($startState,0);
		$this->createNextStep($userProcess);
	}
	/**
	 * @param $userProcess UserProcess
	 * @return void
	 */
	public function rejectProcess($userProcess)
	{
		//then mark user process as complete
		$userProcess->reject();
	}
	/**
	 * @param $userProcess UserProcess
	 * @param $currentState ProcessState
	 * @return void
	 */
	public function approveProcess($userProcess,$currentState)
	{
		$this->currentState = $currentState;

		$userProcess->approve();

		//then create next step. but before then, make sure no other approval is pending.
		//NOTE: Since our workflows are designed to operate serially/synchronously, another pending approval
		//existing , means the current process has a twin parallel process which must be approved, before going to the next state.
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS,'rowClass'=>'UserProcess'));
		$row = $table->fetchRow($table->select()->where('process_id = ?',$userProcess->getProcessId())->where('form_id = ?',$userProcess->getFormId())->where('task_id <> 0')->where('completed <> ?',Task::STATUS_COMPLETE)->where('active = 1'));

		if($row == null)$this->createNextStep($userProcess);

	}
	/**
	 * This will handle every thing involved with the next step, after the current state.
	 * @param $userProcess UserProcess
	 * @return void
	 */
	private function createNextStep($userProcess)
	{
		//get next state(s)
		$states = $this->getNextStates();

		if($states->count() == 0)//there is no next state. process is complete
		{
			$userProcess->complete();
		}

		foreach($states as $state)
		{
			$task_id = $this->createApprovalTask($state,$userProcess);
			$this->createApprovalProcess($state,$task_id);
		}

	}
	/**
	 * Fetch the state(s) after the current state. Based on position property of the state.
	 * @return Zend_Db_Table_Rowset
	 */
	private function getNextStates()
	{
		$currentState = $this->currentState;
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::WORKFLOW_STATES,'rowClass'=>'ProcessState'));
		$states = $table->fetchAll($table->select()->where('process_id = ?',$currentState->getProcessId())
															->where('position = ?',($currentState->getPosition()+1)));

		return $states;
	}
	/**
	 * Create a task for the user approving the next state.
	 * @param $nextState ProcessState
	 * @param $userProcess UserProcess
	 * @return void
	 */
	private function createApprovalTask($nextState,$userProcess)
	{
		$currentState = $this->currentState;

		$start = Precurio_Date::now()->getTimestamp();
		$end = $start + ($nextState->getDuration() * 60 * 60 );

		$tr = Zend_Registry::get('Zend_Translate');
		$data = array(
				'type'=>Task::TYPE_WORKFLOW,
				'creator_group_id'=>0,
				'creator_user_id'=>Precurio_Session::getCurrentUserId(),
				'user_id'=>$this->getGoodApproverId($nextState,$userProcess->getOwner()),
				'title'=>$tr->translate(PrecurioStrings::WORKFLOWAPPROVAL).' - '.$userProcess->getFormCode().' : '.$userProcess->getDisplayName().' ('.$userProcess->getOwner()->getFullName().')',
				'duration'=>$nextState->getDuration(),
				'start_time'=>$start,
				'end_time'=>$end,
				'description'=>'',
				'status'=>Task::STATUS_OPEN,
				'is_transferable'=>0,
				'date_created'=>$start,
				'active'=>1
				);
		$tasks = new Tasks();
		$task_id = $tasks->addTask($data,true);
		return $task_id;
	}
	/**
	 * Creates a user approval process for the next state
	 * @param $nextState ProcessState
	 * @param $task_id int
	 * @return UserProcess
	 */
	public function createApprovalProcess($nextState,$task_id)
	{
		//first get user that was assigned the task. this is the same as the user approving
		//the process for $nextState.
		//if there is no task, then it is a new request, not an approval process
		if($task_id == 0)
		{
			$user_id = Precurio_Session::getCurrentUserId();
		}
		else
		{
			// note that, it is one approval process to a task. if their should be 2 parallel approval
			//process, then it is 2 different task.
			$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK_USERS));
			$row = $table->fetchRow($table->select()->where('task_id = ?',$task_id)->where('active = 1')->where('is_active = 1'));
			if($row == null)
			{
				$tr = Zend_Registry::get('Zend_Translate');
				throw new Precurio_Exception($tr->translate(PrecurioStrings::APPLICATION_LOGIC_ERROR),Precurio_Exception::EXCEPTION_APPLICATION_ERROR,1003);
			}
			$user_id = $row->user_id;
		}


		$data = array(
			'user_id'=>$user_id,
			'process_id'=>$nextState->getProcessId(),
			'state_id'=>$nextState->getId(),
			'form_id'=>$this->form_id,
			'task_id'=>$task_id,
			'date_created'=>Precurio_Date::now()->getTimestamp(),
			'completed'=>($task_id == 0 ? Task::STATUS_ONHOLD : Task::STATUS_OPEN),//yea, only a request process gets it completed status "on hold"
			'active'=>1
		);

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS));
		$row = $table->createRow($data);
		$user_process_id = $row->save();


		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS,'rowClass'=>'UserProcess'));
		$userProcess = $table->find($user_process_id)->current();
		//now we need to update the task description, since we now have a user_process_id
		if($task_id != 0)
		{
			$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK));
			$task = $table->find($task_id)->current();

			$baseUrl  =  Zend_Controller_Front::getInstance()->getRequest()->getBaseUrl();
			$description = "<a href='$baseUrl/workflow/view/{$user_process_id}'> click here to view form </a>";
			$task->description = $description;
			$task->save();

			//workflow approval requested activity
			Precurio_Activity::create($userProcess->getAppId(),UserProcess::ACTIVITY_WAITING_APPROVAL,$userProcess->getItemId(),$user_id);
			Precurio_Activity::create($userProcess->getAppId(),UserProcess::ACTIVITY_WAITING_APPROVAL,$userProcess->getStartProcess()->getItemId(),$user_id,null,-2);
			
			//now send form as an email if the option has been enabled.
			if(isset($nextState->include_form_in_notification) && $nextState->include_form_in_notification)
			{
				//create an activity but do not notify any one. We want to create a custom message
				$activity_id = Precurio_Activity::create($userProcess->getAppId(),UserProcess::ACTIVITY_EMAIL_FORM,$userProcess->getStartProcess()->getItemId(),$user_id,null,-2);
				//first get the email format to be used. This format can be changed by the admin
				$activity = UserActivity::get($activity_id);
				$emailformat = Precurio_Activity::getMessageFormat($activity,"mail");
				
				//now lets get the form data  and convert it to an html table
				$formData = $userProcess->getFormData();
				$formData = $formData->toArray();
				
				
				$style = "text-align: left;min-width: 200px";
				$table = "<table>\n";
				$table .= "\t<tr>";
				$table .= "<th style='$style'>Field</th><th style='$style'>Value</th>";
				$table .= "</tr>\n";
				foreach($formData as $field=>$value)
				{
					if(substr($field, 0,9)=="signature")continue;
					if($field == "date_created")continue;
					if($field == "last_updated")continue;
					if($field == "user_id")continue;
					
					$table .= "\t<tr>";
					$table .= '<td>' . $field. '</td>';
					$table .= '<td>' . htmlspecialchars($value). '</td>';
					$table .= "</tr>\n";
				}
				$table .= '</table>';
				
				//now lets send email.
				
				$config = Zend_Registry::get("config");
				$mail = new Precurio_Mail();
				$mail->sendData(array(
					'subject' => $activity->getMailSubject(),
					'date_created' => Precurio_Date::now()->getTimestamp(),
					'activity_id' => $activity->getId(),
					'to' => UserUtil::getUser($user_id)->getEmail(),
					'body' => getLocalizedString($emailformat,$table,"",$config->base_url.$userProcess->getUrl()),
					'from' => $mail->getFrom(),
				));
				
				
			}
		}
		else
		{
			//new workflow activity
			Precurio_Activity::create($userProcess->getAppId(),UserProcess::ACTIVITY_NEW,$userProcess->getItemId(),$user_id);
		}

		return $userProcess;

	}
	/**
	 * Return the user_id of the User approving for state $state.
	 * Serves as a "good" validator for getApproverId()
	 * This function should have been private, but we need it in ProcessForm
	 * for the 'user can change approver' scenerio
	 * @param $state ProcessState
	 * @param $owner User - Owner of the process. If null, default to current user.
	 * @return int
	 */
	public function getGoodApproverId($state,$owner = null)
	{
		if(empty($owner))$owner = UserUtil::getUser(Precurio_Session::getCurrentUserId());
		if(!Precurio_Utils::isNull($this->approver_id))//an approver has been selected by the user
		{
			if($this->getNextStates()->count() == 1)//the 'user can change approver' scenerio is only valid for a serial workflow
			{
				return $this->approver_id;
			}

		}
		$log = Zend_Registry::get('log');
		try
		{
			$user_id = $this->getApproverId($state,$owner);
			$user = UserUtil::getUser($user_id);
			$log->info('Found a user '.$user->getFullName()." ($user_id) to approve state ".$state->getId());
			$log->info('Found a GOOD user '.$user->getFullName()." ($user_id) to approve state ".$state->getId());
		}
		catch(Exception $e)//error is thrown by getApproverId, if not user could be found.
		{
			if($this->currentState->isStartState())
			{
				$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS,'rowClass'=>'UserProcess'));
				$newProcess = $table->fetchRow($table->select()->where('user_id = ?',Precurio_Session::getCurrentUserId())
															->where('completed = ?',Task::STATUS_ONHOLD)
															->where('form_id = ?',$this->form_id)
															->where('process_id = ?',$this->currentState->getProcessId()));
				if($newProcess)$newProcess->delete();
			}
			if($this->form_id == 0)//this is means, the user is just initiating the process, and the next approval user cannot be chosen. simply let the current user approver himself instead of throwing error
			{
			    $log->warn('We are allowing you approve your form. No good approver found for form '.$this->form_id.' belonging to process '.$this->currentState->getProcessId());
				return Precurio_Session::getCurrentUserId();
			}
			$log->err('No good approver found for form '.$this->form_id.' belonging to process '.$this->currentState->getProcessId());
			throw $e;
		}



		$outObj = $user->getOutOfOffice();
		if($outObj != null)
		{
			$log->info('User '.$user->getFullName()." ($user_id) is not in office to approve state ".$state->getId());
			$user = $outObj->getProxy();
			$user_id = $user->getId();
			$log->info('Proxy User '.$user->getFullName()." ($user_id) to approve state ".$state->getId());
		}
		$log->info('Returned a user '.$user->getFullName()." ($user_id) to approve state ".$state->getId());
		return $user->getId();
	}
	/**
	 * Return the user_id of the User approving for state $state, this function should never
	 * be called directly, except throug getGoodApproverId();
	 * @param $state ProcessState
	 * @param $owner User - Owner of the process.
	 * @return int
	 */
	private function getApproverId($state,$owner)
	{
		$tr = Zend_Registry::get('Zend_Translate');

		//first see if the state uses supervisor approval
		if($state->useSupervisor())
		{
			//the current user logged in user is the person approving the current state.
			//we need to get his boss.
			$boss = UserUtil::getUser(Precurio_Session::getCurrentUserId())->getBoss();

			$log = Zend_Registry::get('log');
			if($boss)//there is a boss
			{
				if($boss->isActive())//we have to check if boss is active so we do not get stuck in a never ending do..while loop
					return $boss->getId();
				else
					$log->warn('Use Supervisor was enabled but supervisor is no longer active. Form '.$this->form_id.' belonging to process '.$this->currentState->getProcessId());
			}
			else
			{
				$log->warn('Use Supervisor was enabled but there was no supervisor for form '.$this->form_id.' belonging to process '.$this->currentState->getProcessId());
			}
		}

		$userList = $this->getPossibleUsersFromList($state, $owner);

		return $userList[0]->getId();//return user with the least open tasks
	}
	/**
	 * This is where we perform the algorithm.
	 * We look at the state, and return valid users that can approve for that state, based on user configuration.
	 * @param ProcessState $state
	 * @param User $owner
	 * @throws Precurio_Exception
	 * @return array
	 */
	private function getPossibleUsersFromList($state,$owner)
	{
		$tr = Zend_Registry::get('Zend_Translate');
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::WORKFLOW_APPROVAL_ACCESS));
		$accessList = $table->fetchAll($table->select()->where('process_id = ?',$state->getProcessId())
				->where('state_id = ?',($state->getId())));


		if($accessList->count() == 0)
		{
			//instead of throwing error. Just pick all users
			$userList = UserUtil::getUsers();
		}
		else
			$userList = $this->getUsersFromAccessList($accessList);


		$user = $owner;

		if($state->isDepartmental())
		{
			$userList = $this->filterUsersByDepartment($userList,$user->getDepartmentId());
		}

		if($state->isLocational())
		{
			$userList = $this->filterUsersByLocation($userList,$user->getLocationId());
		}
		if(count($userList) == 0)
		{
			throw new Precurio_Exception($tr->translate(PrecurioStrings::INVALIDPROCESSCONFIG),Precurio_Exception::EXCEPTION_INVALID_PROCESS_CONFIG,2003);
		}

		if(!function_exists("taskSort")){
    		function taskSort($userA,$userB)
    		{
    			$a = count($userA->getOpenTasks());
    			$b = count($userB->getOpenTasks());
    			if ( $a == $b )
    				return 0;
    			else if ( $a < $b )
    				return -1;
    			else
    				return 1;
    		};
		}
		//if more than one user, sort user list by number of open tasks.
		if(is_array($userList) && count($userList) > 1) usort($userList, 'taskSort');
		
		return $userList;
	}
	/**
	 * Gets an array of users that can approve for a state
	 * @param ProcessState $state
	 * @param User $owner
	 * @return array
	 */
	public function getStateApprovers($state,$owner)
	{
		return $this->getPossibleUsersFromList($state, $owner);
	}
	/**
	 * @param $users Array of User objects to filter
	 * @param $dept_id int the department Id, not the group_id of the department
	 * @param $useCurrentUserDepartment - Also filter by department of the currently logged in user
	 * @return Array of user objects that passes the filter
	 */
	private function filterUsersByDepartment($users,$dept_id,$useCurrentUserDepartment=true)
	{
		$filter = array();
		$currentUser = UserUtil::getUser();
		$current_dept_id = $currentUser->getDepartmentId();
		foreach($users as $user)
		{
			if($user->getDepartmentId() == $dept_id)
				$filter[$user->user_id] = $user;
			//see if you should also include users that belong to current user's department
			if($useCurrentUserDepartment && $user->getDepartmentId() == $current_dept_id)
				$filter[$user->user_id] = $user;
		}
		return $filter;
	}
	/**
	 * @param $users Array of User objects to filter
	 * @param $location_id int the location Id, not the group_id of the location
	 * @param $useCurrentUserLocation  - Also filter by location of the currently logged in user
	 * @return Array of user objects that passes the filter
	 */
	private function filterUsersByLocation($users,$location_id,$useCurrentUserLocation = true)
	{
		$filter = array();
		$currentUser = UserUtil::getUser();
		$current_loc_id = $currentUser->getLocationId();
		foreach($users as $user)
		{
			if($user->getLocationId() == $location_id)
				$filter[$user->user_id] = $user;
			//see if you should also include users that belong to current user's location
			if($useCurrentUserLocation && $user->getLocationId() == $current_loc_id)
				$filter[$user->user_id] = $user;
		}
		return $filter;
	}
	/**
	 * @param $accessList Zend_Db_Table_Rowset , containing row items from workflow_approval_access
	 * @return Array of User Objects
	 */
	private function getUsersFromAccessList($accessList)
	{
		$userList = array();
		foreach($accessList as $access)
		{
			if($access->user_id == 0)
			{
				$users =  UserUtil::getGroup($access->group_id)->getUsers();
				foreach($users as $user)
				{
					$userList[] = $user;
				}
			}
			else
			{
				$userList[] = UserUtil::getUser($access->user_id);
			}
		}
		return $userList;
	}

	/**
	 * A setter function for current state.
	 * @param $currentState ProcessState
	 * @return void
	 */
	public function setCurrentState($currentState)
	{
		$this->currentState = $currentState;
	}
	/**
	 * Returns all pending requests and approval associated with the current user
	 * To get only request,run task_id == 0 test on each object, and otherwise for approvals
	 * Note that this function returns a rowset of UserProcess Objects.
	 * @return arry
	 */
	public static function getMyProcesses()
	{
		$result = array();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS,'rowClass'=>'UserProcess'));
		$processes = $table->fetchAll($table->select()->where('active = 1')->where('user_id = ?',Precurio_Session::getCurrentUserId())->order('id desc'));
		foreach($processes as $userp)
			$result[$userp->id] = $userp;
		
		//get processes for which current user is observer.
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::WORKFLOW_OBSERVERS));
		$items = $table->fetchAll($table->select()->where('active = 1')->where('user_id = ?',Precurio_Session::getCurrentUserId()));

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS,'rowClass'=>'UserProcess'));
		foreach($items as $item)
		{
			$processes = $table->fetchAll($table->select()->where('active = 1')->where('process_id = ?',$item->process_id)->order('id desc'));
			foreach($processes as $userp)
				$result[$userp->id] = $userp;
		}
		
		return $result;

	}
	/**
	 * Returns all the workflows the current user can initiate, sorted by department
	 * @return Array of Process objects
	 */
	public static function getAllWorkflows()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::WORKFLOW,'rowClass'=>'Process'));
		$rowSet = $table->fetchAll($table->select()->where('active=1')->order('department_id'));
		//now determine the ones i have access to
		$currentUser = UserUtil::getUser(Precurio_Session::getCurrentUserId());
		$temp = array();
		foreach($rowSet as $row)
		{
			if(UserUtil::isAllowed('admin_index'))
			{
				$temp[] = $row;
			}
			else if($row->userCanRequest($currentUser))
			{
				$temp[] = $row;
			}
		}
		return $temp;
	}
	/**
	 * @param $id int
	 * @return UserProcess
	 */
	public static function getUserProcess($id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USER_PROCESS,'rowClass'=>'UserProcess'));
		$row = $table->fetchRow($table->select()->where('id = ?',$id));
		return $row;
	}
}

?>