<?php

require_once ('Zend/Db/Table/Row/Abstract.php');
require_once ('task/models/vo/TaskFolder.php');
class Task extends Zend_Db_Table_Row_Abstract implements Precurio_ObjectInterface,Precurio_SearchableInterface,Precurio_ExtranetInterface{
	CONST APPID = 'task';
	const STATUS_OPEN = 0;
	const STATUS_COMPLETE = 1;
	const STATUS_ONHOLD = -1;
	const STATUS_REJECT = 3;
	const TYPE_ADVERT = 'ADVERT'; //this must correspond to feild names of content approval settings
	const TYPE_NEWS = 'NEWS';//this must correspond to feild names of content approval settings
	const TYPE_FEATURED = 'FEATURED';//this must correspond to feild names of content approval settings
	const TYPE_GROUP_CONTENT = "GROUP_CONTENT";
	const TYPE_WORKFLOW = "WORKFLOW";
	const TYPE_CONTENT = "content";

	const VIEW_URL = "/task/index/view/id/{1}";

	const ACTIVITY_NEW = "new";
	const ACTIVITY_COMPLETE = "complete";
	const ACTIVITY_FAILED = "due";
	const ACTIVITY_TRANSFER = "transfer";
	const ACTIVITY_ADD_USER = "add_user";
	const ACTIVITY_ADD_FILE = "add_file";

	public static $statusStr = array(
			Task::STATUS_OPEN=>"Open",
			Task::STATUS_COMPLETE=>"Complete",
			Task::STATUS_ONHOLD=>"Pending",
			Task::STATUS_REJECT=>"Rejected",
	);

	public function getId()
	{
		return $this->id;
	}
	public function getDescription()
	{
		return $this->description;
	}
	public function getTaskId()
	{
		return $this->id;
	}
	public function isComplete()
	{
		return ($this->status == self::STATUS_COMPLETE || $this->status == self::STATUS_REJECT);
	}
	public function getStatusStr()
	{
		return self::$statusStr[$this->status];
	}
	public function getStatus()
	{
		return $this->status;
	}
	public function isDue()
	{
		$now = Precurio_Date::now()->getTimestamp();
		return  ($now >= $this->end_time && !empty($this->end_time));
	}
	public function isMine()
	{
		$my_id = Precurio_Session::getCurrentUserId();
		return $this->creator_user_id == $my_id;
	}
	public function isAssignedToMe()
	{
		return $this->getUserId() == Precurio_Session::getCurrentUserId();
	}
	/**
	 * Determines if the task is a group/team task
	 * @return boolean
	 */
	public function isTeam()
	{
		return !empty($this->creator_group_id);
	}
	/**
	 * Get the group associated with the task;if any.
	 * @return NULL| Team
	 */
	public function getGroup()
	{
		if(!$this->isTeam())return null;
		return Team::get($this->creator_group_id);
	}
	/**
	 * Gets the user that created the task
	 * @return Ambigous <User, NULL, Zend_Db_Table_Row_Abstract, unknown>
	 */
	public function getOwner()
	{
		return UserUtil::getUser($this->creator_user_id);
	}

	/**
	 * Gets the name of the user performing the task
	 * @return string
	 */
	public function getTaskUserName()
	{
		return $this->getUser()->getFullName();
	}
	private $user;
	/**
	 * Gets the user performing the task
	 * @return User
	 */
	public function getUser()
	{
		if(!Precurio_Utils::isNull($this->user))
			return $this->user;
		$user_id = $this->getUserId();
		if(empty($user_id))return null;
		$this->user = UserUtil::getUser($user_id);
		return $this->user;
	}
	/**
	 * Get ID of user performing the task.
	 */
	public function getUserId()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK_USERS));
		$select = $table->select()->where('task_id = ?',$this->getId())
				->where('is_active = 1')
				->where('active = 1')->order('id asc');
		$row = $table->fetchRow($select);
		if(empty($row))return 0;
		return $row->user_id;
	}
	public function getDueDate($short=false)
	{
		$date = new Precurio_Date($this->end_time);
		return ($short ? $date->get(Precurio_Date::TIME_SHORT).' '.$date->get(Precurio_Date::DATE_SHORT) : $date->get(Precurio_Date::DATETIME));
	}
	/**
	 * Returns the due date as a date object.
	 * @return Precurio_Date
	 */
	public function getEndDate()
	{
		return new Precurio_Date($this->end_time);
	}
	/**
	 * Determines whether a task has a due date
	 * @version 4
	 * @return boolean
	 */
	public function hasDate()
	{
		return !empty($this->end_time);
	}
	public function getDateCreated()
	{
		$date = new Precurio_Date($this->date_created);
		return $date->get(Precurio_Date::DATE_LONG);
	}
	public function getLastModified()
	{
		$date = new Precurio_Date($this->last_updated);
		return $date->get(Precurio_Date::DATE_LONG);
	}
	/**
	 * Gets the date the task was completed
	 * @return Ambigous string
	 */
	public function getDateCompleted()
	{
		if(empty($this->complete_time))return $this->getDueDate();
		$date = new Precurio_Date($this->complete_time);
		return $date->get(Zend_Date::DATETIME);
	}
	/**
	 * Determines if a user has access to the task;
	 * @param int $user_id
	 * @version 4
	 * @return boolean
	 */
	public function canAccess($user_id)
	{
		if($user_id == $this->creator_user_id)return true;
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK_USERS));
		$select = $table->select()->where('task_id = ?',$this->getId())
				->where('user_id = ?',$user_id)
				->where('active = 1');
		$row = $table->fetchRow($select);
		if(!empty($row))return true;
		return false;
	}
	/**
	 * Determines if a user can modify the task;
	 * @param int $user_id
	 * @version 4
	 * @return boolean
	 */
	public function canModify($user_id)
	{
		return $this->creator_user_id == $user_id;
	}
	public function getComments()
	{
		return Comment::getAll($this->getTaskId(), self::APPID);
	}
	public function isApproval()
	{
		return !Precurio_Utils::isNull($this->type);
	}
	public function getTitle()
	{
		return $this->title;
	}
	/**
	 * Returns Task Object with ID = $id
	 * @param $id int
	 * @return Task
	 */
	public static function get($id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK, 'rowClass'=>'Task'));

		$task = $table->fetchRow($table->select()->where('id = ?' ,$id));

		if($task->creator_group_id)
			$task = TeamTask::get($id, $task->creator_group_id);

		return $task;
	}
	/**
	 * Transfer a task from one user to another
	 * @param int $user_id
	 * @param int $to_user_id
	 * @param user $reason
	 */
	public function transfer($user_id,$to_user_id,$reason)
	{
		$from_user_id = $this->getUserId();
		if($from_user_id == $to_user_id)return;//there is no change.

		//deactive active user.
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK_USERS));
		$table->update(array('is_active'=>0), "task_id = ".$this->getId());

		//add user.
		$this->addUser($to_user_id,true,false,true);

		//log transfer
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK_TRANSFERS));
		$newTransfer = $table->createRow(array(
										'task_id'=>$this->getId(),
										'to_user_id'=>$to_user_id,
										'user_id'=>$user_id,
										'from_user_id'=>$from_user_id,
										'reason'=>$reason,
										'date_created'=>time()));
		$newTransfer->save();

		//create activity
		$this->recordActivity(Task::ACTIVITY_TRANSFER, $user_id);
	}
	/**
	 * Add a user to a task
	 * @param int $user_id
	 * @param boolean $isTransfer
	 * @param boolean $isProxy
	 * @version 4
	 * @return int
	 *
	 */
	public function addUser($user_id,$isTransfer=0,$isProxy=0,$isActive=0)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK_USERS));
		$data = array();

		if($isActive)$table->update(array('is_active'=>0), 'task_id = '.$this->getTaskId());
		if($this->canAccess($user_id))//user has access either via task_users OR as a member of team
		{
			if($isActive)
			{
				$num = $table->update(array('is_active'=>1), 'task_id = '.$this->getTaskId().' AND user_id = '.$user_id);
				if($num > 0)return $this;//user already has access via task_users.
			}
		}
		$data['user_id'] = $user_id;
		$data['task_id'] = $this->getId();
		$data['is_transfer'] = $isTransfer;
		$data['is_proxy'] = $isProxy;
		$data['is_active'] = $isActive;
		$data['date_created'] = Precurio_Date::now()->getTimestamp();
		$id = $table->insert($data);

		$this->recordActivity(Task::ACTIVITY_ADD_USER, $user_id);
		return $id;
	}
	/**
	 * Removes a user from a task
	 * @param int $user_id
	 */
	public function removeUser($user_id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK_USERS));
		$table->update(array('active'=>0), "user_id = $user_id AND task_id = ".$this->getId());
		return;
	}
	/**
	 * Updates the due date of a task
	 * @param timestamp $timestamp
	 * @return Task
	 */
	public function updateDate($timestamp)
	{
		$this->end_time = $timestamp;
		//write to database
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK));
		$task = $table->fetchRow($table->select()->where('id = ?',$this->getId()));
		$task->end_time = $timestamp;
		$task->save();
		return $this;
	}
	/**
	 * Updates the status of a task
	 * @param int $status
	 * @param int $user_id -  the user performing the update
	 * @return Task
	 */
	public function updateStatus($status,$user_id)
	{
		$status = (int)$status;
		if(!isset(self::$statusStr[$status]))return;
		$this->status = $status;
		//write to database
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK));
		$task = $table->fetchRow($table->select()->where('id = ?',$this->getId()));
		$task->status = $status;
		if($status == Task::STATUS_COMPLETE)$task->complete_time = Precurio_Date::now()->getTimestamp();
		$task->save();

		if($status == Task::STATUS_COMPLETE)
			$this->recordActivity(Task::ACTIVITY_COMPLETE, $user_id);
		return $this;
	}
	public function do_delete()
	{
		$this->active = 0;
		//write to database
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::TASK));
		$task = $table->fetchRow($table->select()->where('id = ?',$this->getId()));
		$task->active = 0;
		$task->save();

		$this->index();
	}
	/**
	 * Get all users in this task
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getUsers()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USERS,'rowClass'=>'User'));
		$select = $table->select(false);
		$select->setIntegrityCheck(false);
		$select->setTable($table);

		$select = $select->distinct()
				->from(array('a' => PrecurioTableConstants::USERS))
				->join(array('b' => PrecurioTableConstants::TASK_USERS),'a.user_id = b.user_id',array('is_proxy','is_transfer','is_active','date_assigned'=>'date_created'))
				->where('a.active = 1 AND b.active= 1 AND  b.task_id = ?' ,$this->getTaskId())
				->order('a.first_name asc');

		$rows = $table->fetchAll($select);
		//now we need to add the creator to this list
		$result = array();
		foreach($rows as $row)
		{
			$result[$row->id] = $row;
		}
		$owner = $this->getOwner();
		$result[$owner->id] = $owner;
		return $result;
	}
	protected function recordActivity($type,$user_id,$date_created=null,$onlyNotifyId=0)
	{
		return Precurio_Activity::create(Task::APPID,$type,$this->getId(),$user_id,$date_created,$onlyNotifyId);
	}
	/**
	 * Get url to view task
	 * @return string
	 */
	public function getUrl()
	{
		return getLocalizedString(Task::VIEW_URL,$this->getId());
	}
	public function getFolder()
	{
		return new TaskFolder($this->getId());
	}
	public function getItemId()
	{
		return $this->getId();
	}
	public function isActive()
	{
		return $this->active;
	}
	public function __toString()
	{
		return $this->title;
	}

	/**
	 * Exports the event as an iCal
	 * @return string
	 */
	public function export($user_id=0)
	{
		require_once( "others/iCalcreator.class.php" );
		$tz = date_default_timezone_get();
		if(empty($user_id))$user_id = Precurio_Session::getCurrentUserId();
		$sessionUser = UserUtil::getUser($user_id);
		$u_id = $sessionUser->getUsername().'-task'.$this->getId();

		$user = $this->getUser();
		if($user)
			$organizer = $user->getFullName()." <".$user->getEmail().">";

		$config = Zend_Registry::get('config');
		$url = $config->base_url.$this->getUrl();

		$date = new Zend_Date($this->date_created);
		$startArr = getdate($date->getTimestamp());
		$endArr = getdate($this->getEndDate()->getTimestamp());

		$config = array( "unique_id" =>$u_id, "TZID"      => $tz );
		$v = new vcalendar( $config );                             // create a new calendar object instance

		//$v->setProperty( "method", "PUBLISH" );                    // required of some calendar software
		//$v->setProperty( "x-wr-calname", "Precurio" );      // required of some calendar software
		//$v->setProperty( "X-WR-CALDESC", "Intranet Tasks and Events" ); // required of some calendar software
		$v->setProperty( "X-WR-TIMEZONE", $tz );                   // required of some calendar software

		$xprops = array( "X-LIC-LOCATION" => $tz );                // required of some calendar software
		iCalUtilityFunctions::createTimezone( $v, $tz, $xprops );  // create timezone component(-s) opt. 1
		// based on present date

		$vtodo = & $v->newComponent( "vevent" );                  // create an event calendar component
		$vtodo->setProperty( "dtstart", array( "year"  => $startArr['year']
				, "month" =>    $startArr['mon']
				, "day"   =>    $startArr['mday']
				, "hour"  =>    $startArr['hours']
				, "min"   =>    $startArr['minutes']
				, "sec"   =>    $startArr['seconds'] ));
		if($this->hasDate())
		{

			$vtodo->setProperty( "due",   array( "year"  => $endArr['year']
					, "month" =>    $endArr['mon']
					, "day"   =>    $endArr['mday']
					, "hour"  =>    $endArr['hours']
					, "min"   =>    $endArr['minutes']
					, "sec"   =>    $endArr['seconds'] ));
		}
		else
			$vtodo->setProperty("duration" , array( "day" => 1 ));
		$vtodo->setProperty( "summary", $this->getTitle() );
		$vtodo->setProperty( "description", strip_tags($this->getDescription()) );
		$vtodo->setProperty( "url",$url );
		$vtodo->setProperty( "status",($this->isComplete() ? "COMPLETED" : "NEEDS-ACTION"));
		if($user)
			$vtodo->setProperty( "organizer", $organizer );


		$valarm = & $vtodo->newComponent( "valarm" );             // create an event alarm
		$valarm->setProperty("action", "DISPLAY" );
		$valarm->setProperty("description", $vtodo->getProperty( "description" ));

		$d = sprintf( '%04d%02d%02d %02d%02d%02d',  $startArr['year'], $startArr['mon'], ($startArr['mday']-1), $startArr['hours'], $startArr['minutes'], $startArr['seconds']);
		iCalUtilityFunctions::transformDateTime( $d, $tz, "UTC", "Ymd\THis\Z");
		$valarm->setProperty( "trigger", $d );                     // create alarm trigger (in UTC datetime)

		//echo $v->createCalendar();die('hekk');
		return $v->createCalendar();
	}
	public function index()
	{
		$index = Precurio_Search::getIndex();

		//first remove user from the index if it exists
		$hits = $index->find('id:' . $this->getId().' AND module:'.self::APPID);
		foreach ($hits as $hit)
			$index->delete($hit->id);
		$index->commit();

		if(!$this->isActive())return;

		$task = $this;

		$doc = new Zend_Search_Lucene_Document();
		$doc->addField(Zend_Search_Lucene_Field::keyword('id',$task->id));
		$doc->addField(Zend_Search_Lucene_Field::keyword('user_id',$task->creator_user_id));
		$doc->addField(Zend_Search_Lucene_Field::keyword('type',$task->type));
		$doc->addField(Zend_Search_Lucene_Field::text('title',$task->title));
		$doc->addField(Zend_Search_Lucene_Field::text('description',$task->description));
		$doc->addField(Zend_Search_Lucene_Field::keyword('status',$task->getStatus()));
		$doc->addField(Zend_Search_Lucene_Field::unStored('comments',$this->getCommentsContent()));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('due_date',$task->getDueDate()));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('date',$task->getDateCreated()));
		$doc->addField(Zend_Search_Lucene_Field::keyword('url',$task->getUrl()));
		$doc->addField(Zend_Search_Lucene_Field::keyword('module',self::APPID));

		$index->addDocument($doc);
		return $index;
	}

	/**
	 * Gets all the comment as a string for search purpose
	 * @return string
	 */
	private function getCommentsContent()
	{
		$comments = Comment::getAll($this->getId(), self::APPID);
		$body = "";
		foreach($comments as $comment)
			$body = $body.' '.$comment->message;
		return $body;
	}
	public function getExtranetUrl()
	{
		return getLocalizedString("/extranet/task/view/id/{1}",$this->getId());
	}
	public function addExtranetUser($user_id,$sharer_id)
	{
		$this->addUser($user_id);
	}
	public function init()
	{
		$this->_data['end_time'] = trim($this->_data['end_time']);
		$this->_data['start_time'] = trim($this->_data['start_time']);
		$this->_data['complete_time'] = trim($this->_data['complete_time']);
		$this->_data['type'] = trim($this->_data['type']);
		parent::init();
	}
}

?>