<?php
require_once ('Zend/Db/Table/Row/Abstract.php');
require_once ('others/class.httpdownload.php');
require_once ('document/models/vo/IDocumentObject.php');
require_once ('document/models/Documents.php');
require_once ('cms/models/vo/Content.php');
if(!class_exists('Content'))
	require('cms/models/vo/Content.php');
require_once ('user/models/UserUtil.php');
/**
 * Ideally, a one to one relationship exists between contents and documents,
 * but in future, this may become a one to many relationship as a content may contain
 * more than one document.
 * The current database implementation allows for this futuristic behavior and the current
 * class implementation also allows for this too, contrary to what an Inheritance design
 * may have lead you to believe.
 *
 * But note that, we will continue to assume the ideal scenerio till specified otherwise.
 *
 * Also note this statements, which are all true
 * "1) All documents are contents while not all contents are documents"
 * "2) Contents which are documents can also be articles, adverts,or news"
 * The second statement means, Content has a one to one relationship with many objects
 * which gives the wrong assumption that Content has a one to many relationship.
 * Infact, the only place Content has a one to many relationship is with Categorys
 * and Groups. I.e, a content can belong to more than one Category or Group
 * @author brain
 *
 */
class Document extends Content implements IDocumentObject
{

	const VIEW_URL = '/document/index/view/id/{1}';
	const APPID  = "document";

	const ACTIVITY_NEW = "new";
	const ACTIVITY_SHARE = "share";
	const ACTIVITY_VIEW = "view";
	const ACTIVITY_DOWNLOAD = "download";
	const ACTIVITY_NEW_VERSION = "new_version";
	const ACTIVITY_CHECKOUT = "checkout";

	/**
	 * This are the required content properties.
	 * They must be included into the row data via a join statement with Content table
	 * @var array
	 */
	public static $requiredContentProp = array(
		'title',
		'summary',
		'body',
		'image_path',
		'user_id',
		'keyword',
		'last_updated',
		'num_of_hits',
		'rating',
		'is_public',
		'active');
	/**
	 * This are the required document properties.
	 * This was done so we don't load "file_content" into memory. (performance!)
	 * They must be included into the row data via a join statement with Document table
	 * @version 4
	 * @var array
	 */
	public static $requiredDocumentProp = array('id','content_id','file_name','file_type','file_author','file_size','file_date_created','date_created','last_modified');

	public function getItemId()
	{
		return $this->id;
	}
	public static function getAppId()
	{
		return Document::APPID;
	}
	public function getTitle($includeVersion=false)
	{
		if(empty($this->title))
		{
			$this->title = $this->file_name;
		}
		$title = $this->title;
		if($includeVersion)
		{
			$versionNumber = isset($this->version) ? $this->version : $this->getMostRecentVersion();
			if(!empty($versionNumber) && $versionNumber != "1.0")
			{
				$filename = pathinfo($title,PATHINFO_FILENAME);
				$filename = $filename." [v".$versionNumber."]";

				$title = $filename.".".$this->file_type;
			}
		}
		return $title;
	}

	/**
	 * Returns the size of the document
	 * @param $formatted OPTIONAL - Set to true if you want the size formatted eg 10MB. Default is true
	 * @return string|long
	 */
	public function getSize($formatted = true)
	{
		return ($formatted ? Document::formatSize($this->file_size) : $this->file_size);
	}
	public static function formatSize($size)
	{
		$units = array(' B', ' KB', ' MB', ' GB', ' TB');
	    for ($i = 0; $size >= 1024 && $i < 4; $i++) $size /= 1024;
	    return round($size, 1).' '.$units[$i];
	}
	/**
	 * Returns the date modified as a Date object
	 * @return Precurio_Date
	 */
	public function getDateModified()
	{
		$time = $this->last_modified;
		if(is_object($this->last_modified) && get_class($this->last_modified) == "DateTime")
			$time =  $this->last_modified->getTimestamp();//for mssql server
		return new Precurio_Date($time);
	}
	public function getDateCreated()
	{
		return new Precurio_Date(trim($this->date_created));
	}
	/**
	 * Returns the user who last modified the contents of this folder
	 * @return User
	 */
	public function getLastModifiedBy()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_HISTORY));
		$row = $table->fetchRow($table->select()->where('content_id = ?',$this->getContentId())->order('date_created desc'));
		if(empty($row))
		{
			return $this->getOwner();
		}
		return UserUtil::getUser($row->user_id);
	}
	public function getImagePath($type=100)
	{
		$imagePath = parent::getImagePath($type);
		if(empty($imagePath))return "/library/themes/default/css/images/".$this->getIcon();
		return $imagePath;
	}
	/**
	 * Returns the icon to use for display
	 * @return string
	 */
	public function getIcon()
	{
		switch (substr(strtolower($this->file_type),0,3))
		{
			case 'pdf':
				return 'File-PDF-Acrobat.png';
			case 'zip':
			case 'rar':
			case 'tar':
				return 'File-ZIP-Archive.png';
			case 'doc':
			case 'rtf':
				return 'File-Doc-Text-Word.png';
			case 'xls':
				return 'File-XLS-Excel.png';
			case 'css':
				return 'File-CSS-Stylesheet.png';
			case 'psd':
				return 'File-PSD-Photoshop.png';
			case 'mp3':
				return 'File-MP3-Music.png';
			case 'htm':
				return 'File-HTML.png';
			case 'php':
				return 'File-PHP.png';
			case 'png':
				return 'File-PNG-Image.png';
			case 'jpg':
				return 'File-JPG-Image.png';
		}
		return "File.png";
	}
	/**
	 * Returns the url to view the document(without prefixing the base url)
	 * @return string
	 */
	public function getUrl()
	{
		return getLocalizedString(self::VIEW_URL,$this->getId());
	}
	public function getExtranetUrl()
	{
		return getLocalizedString("/document/view/id/{1}",$this->getId());
	}
	public function getType()
	{
		return strtoupper($this->file_type);
	}

	/**
	 * Returns the category id of the document.
	 * Since a document can belong to several categories, the only way to accurately
	 * get the applicable category, is to check the document browsing history
	 * @return int
	 */
	public function getParentId()
	{
		$history = FolderHistory::get();
		$items = $history->getAll();
		if(empty($items))
		{
			return RootFolder::getSpecialIdFromAccessType($this->getAccessType());
		}
		$parent_id = $items[count($items) - 1];
		return $parent_id;
	}
	/**
	 * Gets folders that contain this document
	 * @return Zend_Db_Table_Rowset of Folder objects
	 */
	public function getFolders()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Folder'));
		$select = new Zend_Db_Table_Select($table);
		$select->setIntegrityCheck(false);
		$select = $select->from(array('a' => PrecurioTableConstants::CATEGORYS))
						->join(array('b' => PrecurioTableConstants::CONTENT_CATEGORYS),'a.id = b.category_id',array())
						->where('b.content_id = ?',$this->getContentId())
						->where('a.active=1')
						->where('b.active=1')
						->order('a.title asc');
		return $table->fetchAll($select);

	}
	public function rename($title)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT));
		$row = $table->find($this->getContentId())->current();
		if(!$row)return;
		$row->title  = $title;
		$row->save();
	}
	public function download($user_id = 0)
	{
		if(empty($user_id))$user_id = Precurio_Session::getCurrentUserId();//this is bad, we try to avoid access user session from models
		$root = Zend_Registry::get('root');
		$filename = $root.'/public/'.Content::PATH_TMP.$this->getId().'.'.strtolower($this->file_type);

		file_put_contents($filename,$this->getFileContent());

		$frontController = Zend_Controller_Front::getInstance();
        $baseUrl = $frontController->getBaseUrl();

        if(substr(strtolower($baseUrl),-4) == '.php')
			$baseUrl = dirname($baseUrl);

		$url = $baseUrl.Content::PATH_TMP.$this->getFullFilename() ;
//echo $filename.'<br/>'.$url.'<br/>'.mb_detect_encoding($filename);die();
		$object = new httpdownload;
	    $object->set_byfile($filename);
		$object->use_resume = false;
		//$object->filename = $this->getFullFilename();//the problem with this is that, it doesn't use the renamed file name
		$object->filename = $this->getTitle(true);
		$object->download();

		unlink($filename);//delete the file
		unset($object);

		$this->recordActivity(self::ACTIVITY_DOWNLOAD, $user_id);
		return;
	}
	public function getFileContent()
	{
		return $this->file_content;
	}
	public function getFullFilename()
	{
		return $this->file_name.'.'.strtolower($this->file_type);
	}
	public function getFileTitle()
	{
		return $this->getFullFilename();
	}
	public function isWord()
	{
		return (substr(strtolower($this->file_type),0,3) == 'doc');
	}

	public function isPdf()
	{
		return (substr(strtolower($this->file_type),0,3) == 'pdf'); ;
	}
	/**
	 * Determines if the document belongs primarily to another module such as helpdesk
	 * @return boolean;
	 */
	public function isFromOtherModule()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$row = $table->fetchRow($table->select()->where('content_id = ?',$this->getContentId())
												->where('category_id <= ?', RootFolder::SPECIAL_ID_HELPDESK)
												->where('active = 1'));
		return !empty($row);

	}
	/**
	 * Return records of version history
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getVersions()
	{

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_VERSIONS));

		$select = new Zend_Db_Table_Select($table);
		$select->setIntegrityCheck(false);
		$select = $select->from(array('a' => PrecurioTableConstants::DOCUMENT_VERSIONS))
		//->joinLeft(array('b' => PrecurioTableConstants::DOCUMENT_HISTORY),'a.document_history_id = b.id')
		->where('a.document_id = ?',$this->getId())
		->where("a.active = 1")
		->order("a.id desc");

		$rows = $table->fetchAll($select);
		return $rows;
	}
	/**
	 * Sets a version as the current document version. Returns true if it was successfully set.
	 * @param string $version
	 * @return boolean
	 */
	public function setCurrentVersion($version)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_VERSIONS));
		$row = $table->fetchRow($table->select()->where("document_id = ?",$this->getId())
				->where("version = ?",$version)
				->where("active = 1"));
		if(empty($row))//if current version not set, use the most recent version
		{
			return false;
		}
		$table->update(array("current"=>0),"document_id = ".$this->getId());
		$row->current = 1;
		$row->save();
		return true;
	}
	/**
	 * Gets the current/active version of a document
	 * @param boolean $returnNull - If set to true, will return null (instead of "1") if document has no version history
	 * @return string
	 * @version 4
	 */
	public function getCurrentVersion($returnNull = false)
	{
		$db = Zend_Registry::get('db');
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_VERSIONS));
		$row = $table->fetchRow($table->select()->where("document_id = ?",$this->getId())
										->where($db->quoteIdentifier('current')." = 1")
										->where("active = 1"));
		if(empty($row))//if current version not set, use the most recent version
		{
			$row = $table->fetchRow($table->select()->where("document_id = ?",$this->getId())
										->where("active = 1")
										->order("id desc"));
			if(empty($row))
			{
				return $returnNull ? null : "1.0";
			}
			return $row->version;
		}
		return $row->version;

	}
	/**
	 * Gets the most recent version of a document.
	 * @return Ambigous <int, string>
	 */
	public function getMostRecentVersion()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_VERSIONS));
		$row = $table->fetchRow($table->select()->where("document_id = ?",$this->getId())
				->where("active = 1")
				->order("id desc"));
		if(empty($row))//if current version not set, use the most recent version
		{
			return "1.0";
		}
		return $row->version;
	}
	/**
	 * Determines if a document has been checked out
	 * @return int - The user id of the person that locked it, or 0 if not locked.
	 */
	public function isCheckedOut()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_CHECKOUT));
		$row = $table->fetchRow($table->select()->where('document_id = ?',$this->getId())->where('active = 1')->order('id desc'));
		if(empty($row))
			return 0;
		if($row->check_out == 0)
			return 0;

		return $row->user_id;
	}
	/**
	 * Checks out a document
	 */
	public function checkOut()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_CHECKOUT));
		$table->insert(array(
			'document_id'=>$this->getId(),
			'user_id'=>Precurio_Session::getCurrentUserId(),
			'check_out'=>1,
			'check_in'=>0,
			'version'=>$this->getCurrentVersion(),
			'active'=>1,
			'cancelled'=>0,
			'date_created'=>time()
		));

		$this->recordActivity(self::ACTIVITY_CHECKOUT, Precurio_Session::getCurrentUserId());
	}
	/**
	 * Cancels a checkout
	 */
	public function cancelCheckOut()
	{
		$id = $this->getId();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_CHECKOUT));
		$table->update(array("active"=>0,"cancelled"=>1), "document_id = $id and check_out = 1 and active = 1");
		return;
	}
	public function checkIn()
	{
		$id = $this->getId();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_CHECKOUT));
		$table->update(array("active"=>0), "document_id = $id");
		$table->insert(array(
			'document_id'=>$id,
			'user_id'=>Precurio_Session::getCurrentUserId(),
			'check_out'=>0,
			'check_in'=>1,
			'version'=>$this->getCurrentVersion(),
			'active'=>1,
			'cancelled'=>0,
			'date_created'=>time()
		));
	}
	/**
	 * Sets the password to a document. Only the owner of the document can password it
	 */
	public function setPassword($password)
	{
		if(Precurio_Session::getCurrentUserId() != $this->getUserId())
			return;
		$this->removePassword();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_PASSWORD));
		$table->insert(array(
			'document_id'=>$this->getId(),
			'user_id'=>Precurio_Session::getCurrentUserId(),
			'password'=>md5($password),
			'active'=>1,
			'date_created'=>time()
		));
		return;
	}
	/**
	 * Checks if a document has been passworded by its owner
	 * @return boolean|string - Returns the password if set, or false if not set.
	 */
	public function isPassworded()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_PASSWORD));
		$row = $table->fetchRow($table->select()->where('document_id = ?',$this->getId())
												->where('active=1'));
		if(empty($row))return false;
		return $row->password;
	}
	/**
	 * Removes all set password on the document
	 */
	public function removePassword()
	{
		if(Precurio_Session::getCurrentUserId() != $this->getUserId())
			return;
		$id = $this->getId();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_PASSWORD));
		$table->update(array('active'=>0), "document_id = $id");
		return;
	}
	/**
	 * Returns download activity logs
	 * @param $unique boolean - Set to true if you do not want multiple download log from a user. Default is true.
	 * @return array
	 */
	public function getDownloadLog($unique=true)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::ACTIVITY_LOG,'rowClass'=>'ActionLog'));
		$rows = $table->fetchAll($table->select()->where('active = 1')
				->where('item_id = ?',$this->getId())
				->where('appid = ?',$this->getAppId())
				->where('type = ?',self::ACTIVITY_DOWNLOAD));

		if(!$unique)return $rows;
		$temp = array();
		foreach($rows as $row)
		{
			$temp[$row->user_id] = $row;
		}
		return $temp;
	}
	/**
	 * Gets a document by its Id
	 * @param int $id
	 * @throws Precurio_Exception
	 * @return Document
	 */
	public static function get($id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT,'rowClass'=>'Document'));

		$select = new Zend_Db_Table_Select($table);
		$select->setIntegrityCheck(false);
		$select = $select->from(array('a' => PrecurioTableConstants::CONTENT),Document::$requiredContentProp)
						->join(array('b' => PrecurioTableConstants::DOCUMENTS),'a.id = b.content_id')
						->where('b.id = ?',$id);
		$row = $table->fetchRow($select);

		$tr = Zend_Registry::get('Zend_Translate');
		if(empty($row))
			throw new Precurio_Exception($tr->translate(PrecurioStrings::INVALIDCONTENT),Precurio_Exception::EXCEPTION_INVALID_CONTENT);
		return $row;
	}
	/**
	 * Gets a particular document version.
	 * This function is typically called on document download
	 * @param int $id
	 * @param string $versionNumber
	 * @throws Precurio_Exception
	 * @return Document
	 */
	public static function getVersion($id,$versionNumber)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_VERSIONS));
		$row = $table->fetchRow($table->select()->where('version = ?',$versionNumber)
										->where('document_id = ?',$id)
										->where('active = 1'));
		if(empty($row))//version does not exists
		{
			$doc = self::get($id);//just get the document
			if(empty($versionNumber) || $versionNumber == "0")//see if a version number was passed at all.
			{
				$currentVersion = $doc->getCurrentVersion(true);//no version number was actually passed, get the current version
				if(empty($currentVersion))//there is no current version too. Just return the document then.
					return $doc;
				else
					return self::getVersion($id, $currentVersion);//there is a current version, return the current version instead
			}
			return $doc;//just in case a wrong version number was passed.
		}
		//if history id is 0 return current document
		if($row->document_history_id == 0)
		{
			return self::get($id);
		}
		//code below is the same as Document::get(), except that am joining with table
		//document_history instead of document and we are using the document_history_id
		//The only reason this is possible is because 'document' table and 'document_history'
		//table is exactly the same.
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT,'rowClass'=>'Document'));

		$select = new Zend_Db_Table_Select($table);
		$select->setIntegrityCheck(false);
		$select = $select->from(array('a' => PrecurioTableConstants::CONTENT),Document::$requiredContentProp)
						->join(array('b' => PrecurioTableConstants::DOCUMENT_HISTORY),'a.id = b.content_id')
						->join(array('c' => PrecurioTableConstants::DOCUMENT_VERSIONS),'b.id = c.document_history_id',array('version','current'))
						->where('b.id = ?',$row->document_history_id);
		$row = $table->fetchRow($select);

		$tr = Zend_Registry::get('Zend_Translate');
		if(empty($row))
			throw new Precurio_Exception($tr->translate(PrecurioStrings::INVALIDCONTENT),Precurio_Exception::EXCEPTION_INVALID_CONTENT);

		//now replace the history_id with the document id itself.
		//If we don't do this. Passworded documents with history will misbehave.
		$row->id = $id;
		return $row;
	}
	/**
	 * Update the contents of the file. Usually called when a new version of document is uploaded
	 * @param string $filename - name of newly uploaded file. (File have been uploaded to Content::PATH_TMP folder)
	 * @param boolean $majorUpdate - Set to false, if document update is not a major update. Default is true. (OPTIONAL, default = true)
	 * @param string $notes - Notes regarding this update (OPTIONAL)
	 */
	public function updateContent($filename,$majorUpdate=true,$notes="")
	{
		$document_id  = $this->getId();
		//if document is locked or checkout, and current user cannot modify document
		if(($this->isLocked() || $this->isCheckedOut()) && (!$this->canModify(Precurio_Session::getCurrentUserId())))
		{
			return;//dont throw error, just ignore.
			//throw new Precurio_Exception(PrecurioStrings::CANNOTACCESSRESOURCE, Precurio_Exception::EXCEPTION_NO_ACCESS);
		}

		$root = Zend_Registry::get('root');
		$filePath = $root.'/public/'.Content::PATH_TMP.$filename;
		$fileFn = new Precurio_FileFn();

		if($fp = fopen($filePath,'rb'))
		{
			$file_content = fread($fp,filesize($filePath));
			$file_size = filesize($filePath);
			fclose($fp);
			//push current version to document history
			$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_HISTORY));
			$history_id = $table->insert(array(
				'user_id'=>$this->user_id,
				'content_id'=>$this->content_id,
				'file_name'=>$this->file_name,
				'file_type'=>$this->file_type,
				'file_author'=>$this->file_author,
				'file_size'=>$this->file_size,
				'file_date_created'=>$this->file_date_created,
				'file_content'=>$this->file_content,
				'date_created'=>trim($this->date_created)
			));
			//if document has never been versioned, create a new version.
			//this inserts version "1" into the document version history
			if($this->getCurrentVersion(true) == null)
				$this->addVersion($history_id,true);
			else
			{
				//set the history id for the last uploaded version.
				//Note, the last uploaded version will always have document history id set to 0
				$table= new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_VERSIONS));
				$table->update(array('document_history_id'=>$history_id), "document_id = $document_id and document_history_id = 0 ");
			}

			//update content

			$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENTS));
			$doc = $table->find($this->getId())->current();
			$doc->file_content = $file_content;
			$doc->file_size = $file_size;
			$doc->date_created = Precurio_Date::now()->getTimestamp();
			$doc->save();

			unlink($filePath);//delete the file
			@unlink($root.'/public/'.Content::PATH_TMP.$document_id.'.'.$this->file_type);//remove previous downloaded copy if it exists.

			$this->addVersion(0,$majorUpdate,$notes);//create new version

			$user_id = Precurio_Session::getCurrentUserId();

			$this->recordActivity(self::ACTIVITY_NEW_VERSION, $user_id);

			if($this->isCheckedOut())
				$this->checkIn();


			$this->index();
		}

	}
	/**
	 * Adds a new version to the version history
	 * @param int $document_history_id a value of zero means this version has not been moved to the history and so is the most recent version. Note that the most recent version is not necessarily the current version since a user can set an older version as the current version.
	 * @param boolean $major - Set to true if you want to add a major version. This affects only the version numbering
	 * @param string $notes - Notes regarding this update (OPTIONAL)
	 * @return $string - The added version.
	 */
	private function addVersion($document_history_id,$major=true,$notes="")
	{
		$currentVersion = $this->getCurrentVersion(true);
		if(empty($currentVersion))
		{
			$newVersion = "1.0";
		}
		else
		{
			$arr = explode(".", $currentVersion);
			if($major)
			{
				$newVersion = $arr[0] + 1;
			}
			else
			{
				//change 1.4 to 1.5 or  1.1.2 to 1.1.3
				if(count($arr) > 1)
				{
					$arr[count($arr)-1] = $arr[count($arr)-1] + 1;
				}
				else
				{
					$arr[1] = "1";
				}
				$newVersion = implode(".",$arr);
			}
		}

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DOCUMENT_VERSIONS));
		$table->insert(array(
			'document_history_id'=>$document_history_id,
			'document_id'=>$this->getId(),
			'version'=>$newVersion,
			'notes'=>$notes,
			'user_id'=>Precurio_Session::getCurrentUserId(),
			'major'=>$major,
			'active'=>1,
			'date_created'=>time()
		));

		if(empty($document_history_id))//most recent is automatically set to current version
			$this->setCurrentVersion($newVersion);
		return $newVersion;
	}

	public function updateDetails($details,$keywords=null,$isPublic=null)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT));
		$row = $table->find($this->getContentId())->current();
		$row->body = $details;
		if($keywords !== null)
			$row->keyword = $keywords;
		if($isPublic !== null)
			$row->is_public = $isPublic;
		$row->save();

		$this->index();
	}

	/**
	 * Emails a document as an email attachment
	 * @param string $recipients - Semicolon separated list of recipient emails
	 * @param string $subject - Subject for the email
	 * @param string $message - Body of the email
	 */
	public function email($recipients,$subject,$message)
	{
		try
		{
			$mail = new Precurio_Mail();
			$mail->createAttachment($this->getFileContent(),
							Zend_Mime::TYPE_OCTETSTREAM,
							Zend_Mime::DISPOSITION_ATTACHMENT,
							Zend_Mime::ENCODING_BASE64,
							$this->getFileTitle());
			$recipients = explode(";",$recipients);
			foreach($recipients as $recipient)
			{
				$validator = new Zend_Validate_EmailAddress();
				if($validator->isValid($recipient))
					$mail->addTo($recipient);
			}
			$user = UserUtil::getUser();
			$mail->setFrom($user->getEmail(),$user->getFullName());
			$mail->setReplyTo($user->getEmail(),$user->getFullName());
			$mail->setBodyHtml($message);
			$mail->setBodyText(strip_tags($message));
			$mail->setSubject($subject);
			$mail->send();
		}
		catch(Exception $e)
		{
			$log = Zend_Registry::get('log');
       		$log->err($e);
		}
	}
	/*
	 * this function was removed for performance issues
	public function isPublic()
	{
		return $this->getAccessType() == Category::ACCESS_PUBLIC;
	}
	*/

	protected function recordActivity($type,$user_id,$date_created=null,$onlyNotifyId=0)
	{
		$team_id = 0;
		if(isset($this->group_id)){
			$team_id = $this->group_id;}
		elseif(isset($this->team_id)){//for the future....just in case ;)
			$team_id = $this->team_id;}
		else{
			$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::ACTIVITY_LOG));
			$row = $table->fetchRow($table->select()->where('appid = ?',$this->getAppId())
										->where('item_id = ?',$this->getItemId())
										->where('user_id = ?',$user_id)
										->where('type = ?',self::ACTIVITY_VIEW)
										->order('id desc'));
			if($row && !empty($row->group_id))
				$team_id = $row->group_id;
		}


		return Precurio_Activity::create($this->getAppId(),$type,$this->getItemId(),$user_id,$date_created,$onlyNotifyId,$team_id);
	}

	/**
	 * Returns the mime type of the document.
	 * Function is currently not used anywhere, added 20/09/2011
	 * @return string
	 */
	public function getMimeType()
	{
		$filename = $this->getFullFilename();
        preg_match("|\.([a-z0-9]{2,4})$|i", $filename, $fileSuffix);

        switch(strtolower($fileSuffix[1]))
        {
            case "js" :
                return "application/x-javascript";

            case "json" :
                return "application/json";

            case "jpg" :
            case "jpeg" :
            case "jpe" :
                return "image/jpg";

            case "png" :
            case "gif" :
            case "bmp" :
            case "tiff" :
                return "image/".strtolower($fileSuffix[1]);

            case "css" :
                return "text/css";

            case "xml" :
                return "application/xml";

            case "doc" :
            case "docx" :
                return "application/msword";

            case "xls" :
            case "xlt" :
            case "xlm" :
            case "xld" :
            case "xla" :
            case "xlc" :
            case "xlw" :
            case "xll" :
                return "application/vnd.ms-excel";

            case "ppt" :
            case "pps" :
                return "application/vnd.ms-powerpoint";

            case "rtf" :
                return "application/rtf";

            case "pdf" :
                return "application/pdf";

            case "html" :
            case "htm" :
            case "php" :
                return "text/html";

            case "txt" :
                return "text/plain";

            case "mpeg" :
            case "mpg" :
            case "mpe" :
                return "video/mpeg";

            case "mp3" :
                return "audio/mpeg3";

            case "wav" :
                return "audio/wav";

            case "aiff" :
            case "aif" :
                return "audio/aiff";

            case "avi" :
                return "video/msvideo";

            case "wmv" :
                return "video/x-ms-wmv";

            case "mov" :
                return "video/quicktime";

            case "zip" :
                return "application/zip";

            case "tar" :
                return "application/x-tar";

            case "swf" :
                return "application/x-shockwave-flash";

            default :
            $root = Zend_Registry::get('root');
			$filename = $root.'/public/'.Content::PATH_TMP.$filename;
			if(file_exists($filename))
			{
				$finfo = finfo_open(FILEINFO_MIME_TYPE); // return mime type ala mimetype extension
				$mime =  finfo_file($finfo, $filename);
				finfo_close($finfo);
				return $mime;
			}
            return "unknown/" . trim($fileSuffix[0], ".");
        }
	}
	public function index()
	{
		$index = Precurio_Search::getIndex();

		//first remove content from the index if it exists
		$hits = $index->find('id:' . $this->getId().' AND module:'.self::APPID);
		foreach ($hits as $hit)
			$index->delete($hit->id);
		$index->commit();


		$file = $this;
		$content = MyContents::getContent($this->getContentId(),false);
		if(!$content->isActive())return;//if content was deleted, do not bother indexing.

		$config = Zend_Registry::get('config');
		
		if(isset($config->search_file_content) && $config->search_file_content)
		{
    		try
    		{
    		    //lets form the physical file from database
    		    $root = Zend_Registry::get('root');
    		    $filename = $root.'/public/'.Content::PATH_TMP.$file->getFullFilename();
    		    file_put_contents($filename,$file->getContent());
    		    
    			if(strtolower($file->getType()) == 'docx')
    			{
    				$doc = Zend_Search_Lucene_Document_Docx::loadDocxFile($filename);
    			}
    			if(strtolower($file->getType()) == 'xlxx')
    			{
    				$doc = Zend_Search_Lucene_Document_Xlsx::loadXlsxFile($filename);
    			}
    			if(strtolower($file->getType()) == 'pptx')
    			{
    				$doc = Zend_Search_Lucene_Document_Pptx::loadPptxFile($filename);
    			}
    			if($file->isPdf())
    			{
    
    			}
    			@unlink($filename);//delete the file
    		}
    		catch(Exception $e)
    		{
    			@unlink($filename);//delete the file
    			unset($doc);
    
    			$log = Zend_Registry::get('log');
    			$log->err($e);
    		}
		}
		
		if(!isset($doc))$doc = new Zend_Search_Lucene_Document();
		
		$doc->addField(Zend_Search_Lucene_Field::keyword('id',$this->getId()));
		$doc->addField(Zend_Search_Lucene_Field::keyword('user_id',$content->getUserId()));
		$doc->addField(Zend_Search_Lucene_Field::text('title',$this->getTitle()));
		$doc->addField(Zend_Search_Lucene_Field::text('fullname',$content->getFullName()));
		if($content->getUrl() != '#')
			$doc->addField(Zend_Search_Lucene_Field::text('url',$this->getUrl()));
		$doc->addField(Zend_Search_Lucene_Field::text('tags',$content->getKeywords()));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('image_path',$content->getImagePath()));
		$doc->addField(Zend_Search_Lucene_Field::unStored('body',$content->getBody()));
		$doc->addField(Zend_Search_Lucene_Field::unStored('comments',$this->getCommentsContent()));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('date',$content->getDateAdded()));
		$doc->addField(Zend_Search_Lucene_Field::keyword('module',$this->getAppId()));

		$index->addDocument($doc);

		return $index;
	}
	/**
	 * Gets all the comment as a string for search purpose
	 * @return string
	 */
	private function getCommentsContent()
	{
		$comments = Comment::getAll($this->getId(), self::APPID);
		$body = "";
		foreach($comments as $comment)
			$body = $body.' '.$comment->message;
		return $body;
	}

	public function init()
	{
		$this->_data['file_type'] = trim($this->_data['file_type']);
		$this->_data['file_date_created'] = trim($this->_data['file_date_created']);
		parent::init();
	}
}

?>