<?php
require_once ('Zend/Db/Table/Row/Abstract.php');
require_once ('user/models/vo/Comment.php');
require_once ('discussion/models/vo/DiscussionCategory.php');
require_once ('discussion/models/vo/DiscussionFolder.php');
class Discussion extends Zend_Db_Table_Row_Abstract implements Precurio_ObjectInterface, Precurio_SearchableInterface, Precurio_ExtranetInterface  {
	CONST APPID = 'discussion';
	const VIEW_URL = "/discussion/index/view/id/{1}";

	const ACTIVITY_NEW = "new";
	const ACTIVITY_COMMENT = "comment";
	const ACTIVITY_ADD_USER = "add_user";
	const ACTIVITY_ADD_FILE = "add_file";
	const ACTIVITY_VIEW = "view";
	public function getId()
	{
		return $this->id;
	}
	public function getTitle()
	{
		return $this->title;
	}

	/**
	 * Gets the date the discussion was created
	 * @return Precurio_Date
	 */
	public function getDate()
	{
		return new Precurio_Date($this->date_created);
	}
	/**
	 * Determines if a user has access to the discussion
	 * @param int $user_id
	 * @return boolean
	 */
	public function canAccess($user_id)
	{
		$restricted = false;//determines if the discussion is restricted.
		//check if there is any access restriction on the discussion.
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSION_USERS));
		$rows = $table->fetchAll($table->select()->where('discussion_id = ?',$this->getId())->where('active = 1'));
		if(count($rows) > 1)$restricted = true;
		
		if($this->is_team) //all team discussions are restricted
		    $restricted = true;
		
		//check if the user was given direct access to the discussion
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSION_USERS));
		$row = $table->fetchRow($table->select()->where('discussion_id = ?',$this->getId())->where('active = 1')->where('user_id = ?',$user_id));
		if(!empty($row))return true;
		
		if($this->is_team) //if group discussion, see if user is member of any group with access.
		{
		    $table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSION_GROUPS));
		    $rows = $table->fetchAll($table->select()->where('discussion_id = ?',$this->getId())->where('active = 1'));
		    foreach ($rows as $row)
		    {
		        $group = UserUtil::getGroup($row->group_id);
		        if($group->containsMember($user_id))
		        {
		            return true;
		        }
		    }
		}

		//user was not given direct access. see what access is on category.
		if($restricted)return false;//irrespective of category access, if user was not specifically given access, then deny.
		
		//get category
		$category = $this->getCategory();
		return empty($category) ? false: $category->canAccess($user_id);
	}
	/**
	 * Gets the last post/comment in the discussion
	 * @return Comment|string
	 */
	public function getLastPost()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::COMMENTS));
		$row = $table->fetchRow($table->select()->where('appid = ?',self::APPID)
					->where('item_id = ?',$this->getId())->where('active = 1')->order('date_created desc'));
		if(!empty($row))
		{
			return Comment::getComment($row->id);
		}
		return "";
	}
	public function getPostCount()
	{
		return $this->getPosts()->count();
	}
	/**
	 * Posts a message or comment in the discussion
	 * @param int $user_id
	 * @param string $message
	 */
	public function postMessage($user_id,$message)
	{
		$message = trim($message);
		if(!empty($message))
			Comment::create($user_id, $message, $this->getId(), self::APPID);

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSIONS));
		$table->update(array('last_updated'=>Precurio_Date::now()->getTimestamp()), "id = ".$this->getId());
	}
	/**
	 * Gets all the posts in the discussion
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getPosts()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::COMMENTS));
		return $table->fetchAll($table->select()->where('appid = ?',self::APPID)
				->where('item_id = ?',$this->getId())->where('active = 1')->order('date_created asc'));
	}
	/**
	 * Returns a collection of User objects that have access to the discussion.
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getUsers()
	{
		$results = array();//contains all users with access to the discussion and discussion category

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USERS,'rowClass'=>'User'));
		$select = $table->select(false);
		$select->setIntegrityCheck(false);
		$select->setTable($table);

		$select = $select
		->from(array('a' => PrecurioTableConstants::USERS))
		->join(array('b' =>PrecurioTableConstants::DISCUSSION_USERS),'a.user_id = b.user_id',array('discussion_id'))
		->where('a.active = 1 AND b.active= 1 AND  b.discussion_id = ?' ,$this->getId())
		->order('b.id desc');

		$rows = $table->fetchAll($select);

		foreach($rows as $row)
			$results[$row->user_id] = $row;

		$category = $this->getCategory();
		if($category)$groups = $category->getGroups();
		if(isset($groups) && count($groups))
		{
			foreach($groups as $group)
			{
				$members = $group->getUserMembers();
				foreach($members as $user)
					$results[$user->user_id] = $user;
			}
		}
		else //probably a public forum discussion. get participants instead
		{
			$users = $this->getParticipants();
			foreach($users as $user)
				$results[$user->user_id] = $user;
		}

		return $results;
	}
	/**
	 * Returns a collection of users actively participating in the discussion
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getParticipants()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USERS,'rowClass'=>'User'));
		$select = $table->select(false);
		$select->setIntegrityCheck(false);
		$select->setTable($table);

		$select = $select->distinct()
		->from(array('a' => PrecurioTableConstants::USERS))
		->join(array('b' =>PrecurioTableConstants::COMMENTS),'a.user_id = b.user_id',array())
		->where('a.active = 1 AND b.active = 1')
		->where("b.item_id = ?",$this->getId())
		->where("b.appid = ?",self::APPID);

		$rows = $table->fetchAll($select);
		return $rows;
	}
	/**
	 * Determines if a discussion contains unread posts by user.
	 * It does this by simply checking the last time a user viewed a discussions vs the number of new messages since then
	 * @param int $user_id
	 * @return boolean
	 */
	public function hasUpdates($user_id)
	{
		//@todo: review performance impact of this logic (test for 20 discussion)
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::COMMENTS));
		$select = $table->select()->where('item_id= ? ',$this->getId())
		->where('appid= ? ',Discussion::APPID)
		->where('active=1')->where('user_id <> ?',$user_id)->order('date_created desc');

		$row = $table->fetchRow($select);

		if(empty($row))return false;


		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::ACTIVITY_LOG, 'rowClass'=>'UserActivity'));
		$item = $table->fetchRow($table->select()->where('appid = ?',self::APPID)
				->where('type = ?',self::ACTIVITY_VIEW)
				->where('item_id = ?',$this->getId())->order('date_created desc'));
		if(empty($item))return true;

		if($row->date_created > $item->date_created )return true;

		return false;
	}
	public function addUser($user_id)
	{
		if(empty($user_id))return;
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSION_USERS));
		$time = Precurio_Date::now()->getTimestamp();
		$table->insert(array('user_id'=>$user_id,'discussion_id'=>$this->getId(),'date_created'=>$time,'active'=>1));

		if($user_id != $this->creator_user_id)//do not trigger activity for creator
			Precurio_Activity::create(self::APPID,self::ACTIVITY_NEW,$this->getId(),$this->creator_user_id,null,$user_id);
	}
	public function removeUser($user_id)
	{
		if(empty($user_id))return;
		if($this->creator_user_id == $user_id)return;//you can't remove the person who started the discussion
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSION_USERS));
		$table->update(array('active'=>0),'discussion_id = '.$this->getId().' and user_id = '.$user_id);
	}
	/**
	 * Gets the user that created the discussion
	 * @return Ambigous <User, NULL, Zend_Db_Table_Row_Abstract, unknown>
	 */
	public function getUser()
	{
		return UserUtil::getUser($this->creator_user_id);
	}
	/**
	 * Get id of User that started the discussion
	 * @return int
	 */
	public function getUserId()
	{
		return $this->creator_user_id;
	}
	/**
	 * Change the subject/title of the discussion
	 * @param string $subject
	 */
	public function changeSubject($subject)
	{
		if(empty($subject))return;
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSIONS));
		$table->update(array('title'=>$subject),'id = '.$this->getId());
	}
	/**
	 * Creates a discussion object
	 * @param $data
	 * @return Discussion
	 */
	public static function createNew($data)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSIONS));
		$row = $table->createRow();
		$row->setFromArray($data);
		$id = $row->save();
		$discussion = self::get($id);
		//initialise
		Precurio_Activity::create(self::APPID,self::ACTIVITY_NEW,$id,$data['creator_user_id']);

		$discussion->addUser($data['creator_user_id']);

		if(!empty($data['message']))
		{
			Comment::create($data['creator_user_id'],$data['message'], $id, self::APPID);
		}

		return $discussion;
	}
	/**
	 * Gets a discussion object.
	 * Returns null if object does not exist
	 * @param int $id -
	 * @return Discussion
	 */
	public static function get($id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSIONS,'rowClass'=>'Discussion'));
		$row = $table->fetchRow($table->select()->where('id = ?',$id));
		return $row;
	}
	public static function getAll($user_id = 0)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSIONS,'rowClass'=>'Discussion'));
		$select = $table->select(false);
		$select->setIntegrityCheck(false);
		$select->setTable($table);

		$select = $select->distinct()
		->from(array('a' => PrecurioTableConstants::DISCUSSIONS))
		->join(array('b' => PrecurioTableConstants::DISCUSSION_USERS),'a.id = b.discussion_id',array('user_id'))
		->where('a.is_team = 0 AND a.active = 1 AND b.active= 1 AND (a.creator_user_id = ? OR b.user_id = ?)' ,$user_id)
		->order('last_updated desc');
		$items = $table->fetchAll($select);
		//remove duplicates caused by joining tables
		$result = array();
		foreach($items as $item)
		{
			$result[$item->id] = $item;
		}
		return $result;
	}
	public function getUrl()
	{
		return getLocalizedString(self::VIEW_URL,$this->getId());
	}
	public function getItemId()
	{
		return $this->getId();
	}
	public function isActive()
	{
		return $this->active;
	}
	/**
	 * Deletes the discussion
	 */
	public function do_delete()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::DISCUSSIONS));
		$table->update(array('active'=>0),'id = '.$this->getId());
	}
	/**
	 * Returns the category the discussion belongs to
	 * @return DiscussionCategory
	 */
	public function getCategory()
	{
		return DiscussionCategory::get($this->category_id);
	}
	public function __toString()
	{
		return $this->getTitle();
	}
	public function index()
	{
		$index = Precurio_Search::getIndex();

		//first remove user from the index if it exists
		$hits = $index->find('id:' . $this->getId().' AND module:'.self::APPID);
		foreach ($hits as $hit)
			$index->delete($hit->id);
		$index->commit();

		if(!$this->isActive())return;


		$doc = new Zend_Search_Lucene_Document();
		$doc->addField(Zend_Search_Lucene_Field::keyword('id',$this->id));
		$doc->addField(Zend_Search_Lucene_Field::keyword('user_id',$this->creator_user_id));
		$doc->addField(Zend_Search_Lucene_Field::text('title',$this->getTitle()));
		$doc->addField(Zend_Search_Lucene_Field::unStored('comments',$this->getCommentsContent()));
		$doc->addField(Zend_Search_Lucene_Field::unIndexed('date',$this->getDate()));
		$doc->addField(Zend_Search_Lucene_Field::keyword('url',$this->getUrl()));
		$doc->addField(Zend_Search_Lucene_Field::keyword('module',self::APPID));

		$index->addDocument($doc);
		return $index;
	}
	/**
	 * Gets all the comment as a string for search purpose
	 * @return string
	 */
	private function getCommentsContent()
	{
		$comments = Comment::getAll($this->getId(), self::APPID);
		$body = "";
		foreach($comments as $comment)
			$body = $body.' '.$comment->message;
		return $body;
	}

	public function getExtranetUrl()
	{
		return getLocalizedString("/extranet/discussion/view/id/{1}",$this->id);
	}
	public function addExtranetUser($user_id, $sharer_id)
	{
		$this->addUser($user_id);
	}
	public function getFolder()
	{
		return new DiscussionFolder($this->getId());
	}
	public function init()
	{
		$this->_data['last_updated'] = trim($this->_data['last_updated']);
		parent::init();
	}
}

?>