<?php
require_once ('user/models/UserUtil.php');
require_once ('cms/models/vo/Content.php');
require_once ('cms/models/vo/GroupToCategory.php');
require_once ('document/models/Folders.php');
require_once ('Zend/Db/Table/Row/Abstract.php');
class Category extends Zend_Db_Table_Row_Abstract{

	const ACCESS_PUBLIC = 0;
	const ACCESS_PRIVATE = -1;
	const ACCESS_SHARED = 1;

	/**
	 * This is the class used to represent a category record.
	 * This variable was actually implemented so that 'Category' and
	 * 'Folder' can use same class implementation e.g getCategoryChildren().
	 * @var unknown_type
	 */
	protected $categoryClass = 'Category';
	public function getId()
	{
		return $this->id;
	}
	public function getTitle()
	{
		return $this->title;
	}
	/**
	 * Gets category owner
	 * @return User
	 */
	public function getUser()
	{
		return UserUtil::getUser($this->user_id);
	}
	/* Get all users with access to this category.
	 * Returns null if it is a public category
	*/
	public function getUsers()
	{
		if($this->isPublic())return null;

		$result = array();

		//get groups and add members
		$groups = $this->getGroups();
		foreach($groups as $group)
		{
			$members = $group->getUserMembers();
			foreach($members as $user)
				$result[$user->user_id] = $user;
		}

		//get users
		$users = $this->getSharedUsers();
		foreach($users as $user)
			$result[$user->user_id] = $user;

		//also include content owner
		$user = $this->getOwner();
		$result[$user->user_id] = $user;

		return $result;

	}
	/**
	 * Gets category owner (proxy to getUser())
	 * @return User
	 */
	public function getOwner()
	{
	    $user = $this->getUser();
	    
		return $user ? $user : new stdClass();
	}
	/**
	 * Returns the full name of the user that owns the category
	 * @return string
	 */
	public function getOwnersName()
	{
	    $user = $this->getUser();
		return $user ? $user->getFullName() : "";
	}
	/**
	 * Returns the date created as a Date object
	 * @return Precurio_Date
	 */
	public function getDateCreated()
	{
		return new Precurio_Date(trim($this->date_created));
	}
	public function getDescription()
	{
		return $this->description;
	}
	/**
	 * Returns the access type
	 * @return int
	 */
	public function getAccessType()
	{
		return $this->access_type;
	}
	/**
	 * Returns the access type as a string, e.g. "Public"
	 * @return string
	 */
	public function getAccessTypeStr()
	{
		return Category::getAccessStr($this->access_type);
	}
	public function getParentId()
	{
		return $this->parent_id;
	}
	/**
	 * Returns the parent category
	 * @return Category
	 */
	public function getParent()
	{
		$tr = Zend_Registry::get('Zend_Translate');
		if($this->parent_id == 0 )
			return '['.$tr->translate('None').']';
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>$this->categoryClass));
		$item = $table->fetchRow($table->select()->where('id = ?',$this->parent_id)->where('active=1'));
		if($item==null)
			return $tr->translate('Orphaned Category');
		return $item;
	}
	/**
	 * Get all direct children of the category (children include sub categories and contents)
	 * @return array
	 */
	public function getChildren()
	{
		$categories = $this->getCategoryChildren(false);
		$contents = $this->getContentChildren(false);
		return array_merge($categories,$contents);
	}
	/**
	 * Returns the sub-categorys on a category
	 * @param boolean $recursive |default true| - flag to determine if you want to recursively get sub categories of sub categories.
	 * @param int $category_id |default 0| - id of category you want to get it's children. (Mostly used internaly when recurive=true)
	 * @return array
	 */
	public function getCategoryChildren($recursive = true,$category_id=0)
	{
		if($category_id == 0)$category_id = $this->id;
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>$this->categoryClass));
		$items = $table->fetchAll($table->select()->where('parent_id = ?',$category_id)->where('is_folder = ?',$this->is_folder)->where('active=1')->order('title asc'));
		if(!isset($categorys))
			$categorys = array();
		$user_id = Precurio_Session::getCurrentUserId();
		foreach($items as $item)
		{
			if(!$item->canAccess($user_id))continue;
			$categorys[] = $item;
			if($recursive)
			{
				$categorys = array_merge($categorys,$this->getCategoryChildren($recursive,$item->id));

			}
		}
		return $categorys;
	}
	/**
	 * Enter description here ...
	 * @param unknown_type $recursive
	 * @return array
	 */
	public function getContentChildren($recursive = true)
	{
		$categorys = array();
		$contents = array();
		//if you want to get all children (both direct and indirect), you have to first get all sub categorys
		if($recursive)
		{
			$categorys = $this->getCategoryChildren(true);
		}
		array_unshift($categorys,$this);//add this category
		
		$user_id = Precurio_Session::getCurrentUserId();
		
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT,'rowClass'=>'Content'));
		foreach ($categorys as $category)
		{
			$select = new Zend_Db_Table_Select($table);
			$select->setIntegrityCheck(false);
			$select = $select->from(array('a' => PrecurioTableConstants::CONTENT))
							->join(array('b' => PrecurioTableConstants::CONTENT_CATEGORYS),'a.id = b.content_id',array())
							->where('b.category_id = ?',$category->id)
							->where('a.active=1')
							->where('b.active=1');
			$temp = $table->fetchAll($select);
			foreach($temp as $t)//get contents Zend_Db_Table_RowSet into array , by looping.
			{
			    if(!$t->canAccess($user_id))continue;
				$contents[$t->id] = $t;//using $contents[$t->id] instead of  solves the issue of duplicate contents, since  already existing contents will replace themselves
			}
		}

		return $contents;
	}

	public function deActivateCategory()
	{
		$this->active = 0;
		$this->deleted = 1;
		$this->save();

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$items = $table->fetchAll("category_id = $this->id");

		foreach($items as $obj)
		{
			$obj->active = 0;
			$obj->save();
		}
	}
	/**
	 * Add contents to a category
	 * @param array|int $content_ids
	 * @return null
	 */
	public function addContents($content_ids)
	{
		$user_id = Precurio_Session::getCurrentUserId();
		$date_created  = time();
		if($this->isLocked() && $this->user_id != $user_id)
		{
			throw new Precurio_Exception(PrecurioStrings::FOLDERISLOCKED, Precurio_Exception::EXCEPTION_INVALID_USER_ACTION);
			return;
		}
		if(!is_array($content_ids))$content_ids = array($content_ids);
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		foreach($content_ids as $id)
		{
			$table->insert(array(
				'category_id'=>$this->id,
				'content_id'=>$id,
				'user_id'=>$user_id,
				'date_created'=>$date_created,
				'active'=>1
			));
		}

		//if category has been shared with groups, also share the contents with groups. This is to support team rooms feature
		$groups = $this->getGroups();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CONTENTS));
		foreach ($groups as $group)
		{
			$table->insert(array(
				'group_id'=>$group->id,
				'content_id'=>$id,
				'task_id'=>0,
				'date_created'=>$date_created,
				'active'=>1
			));
		}

		return;
	}
	/**
	 * Remove content from category, this does not delete the contents themeselves.
	 * It simply removes the relationship
	 * @param array|int $content_ids
	 * @return null
	 */
	public function removeContents($content_ids)
	{
		if(!is_array($content_ids))$content_ids = array($content_ids);
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$id = $this->id;
		foreach($content_ids as $content_id)
		{
			$table->delete("content_id = $content_id and category_id = $id");
		}
		return;
	}
	/**
	 * @param array|int $group_ids
	 * @return null
	 */
	public function removeGroups($group_ids)
	{
		if(!is_array($group_ids))$group_ids = array($group_ids);
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CATEGORYS));
		$id = $this->id;
		foreach($group_ids as $group_id)
		{
			$table->delete("group_id = $group_id and category_id = $id");
		}
		return;
	}
	/**
	 * Indicates whether category already contains a particular content
	 * @param int $content_id
	 * @return boolean
	 */
	public function hasContent($content_id)
	{
		if(empty($content_id))return false;
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$item = $table->fetchRow($table->select()->where('content_id = ?',$content_id)->where('category_id = ?',$this->id)->where('active = 1'));
		return (empty($item) ? false : true);
	}

	/**
	 * Returns an array of group objects that have been given access to shared category
	 * @return array
	 */
	public function getGroups()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CATEGORYS));
		$items = $table->fetchAll($table->select()->where('category_id = ?',$this->id)->where('active = 1'));
		$groups = array();
		foreach ($items as $item)
		{
			$group = UserUtil::getGroup($item->group_id);
			if(!empty($group))$groups[] = $group;
		}
		return $groups;
	}
	/**
	 * Add groups to a shared category
	 * @param array|int $group_ids
	 * @return null
	 */
	public function addGroups($group_ids)
	{
		if(!is_array($group_ids))$group_ids = array($group_ids);
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CATEGORYS));
		$user_id = Precurio_Session::getCurrentUserId();
		foreach($group_ids as $id)
		{
			$table->insert(array(
				'category_id'=>$this->id,
				'group_id'=>$id,
				'user_id'=>$user_id,
				'date_created'=>time(),
				'active'=>1
			));
		}
		return;
	}
/**
	 * Works like addGroups(), but removes any existing group sharing
	 * @param array|int $group_ids
	 * @return null
	 */
	public function groupShare($group_ids)
	{
		if(empty($group_ids))$group_ids = array();//convert null,0 or "" to an empty array, so they are not added to an array by statement below.
		if(!is_array($group_ids))$group_ids = array($group_ids);

		//get users already shared with.
		$sharedUsers = $this->getSharedUsers();
		$sharedUserIds  = array();
		foreach($sharedUsers as $user)$sharedUserIds[] = $user->user_id;

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CATEGORYS));
		$table->update(array('active'=>0), "category_id = ".$this->getId());

		$user_id = Precurio_Session::getCurrentUserId();
		foreach($group_ids as $id)
		{
			$table->insert(array(
				'category_id'=>$this->id,
				'group_id'=>$id,
				'user_id'=>$user_id,
				'date_created'=>time(),
				'active'=>1
			));
			//notify group members
			$members = UserUtil::getGroup($id)->getUsers();
			foreach($members as $user)
			{
				if(!in_array($user->getId(), $sharedUserIds))//not shared with this user before
				{
					//@todo folder share notification. Means we have to make category an object
				}
			}
		}
		return;
	}
	/**
	 * Gets a rowset of GroupToCategory objects
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getGroupsToCategory()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::GROUP_CATEGORYS,'rowClass'=>'GroupToCategory'));
		return $table->fetchAll($table->select()->where('category_id = ?',$this->id)->where('active = 1'));
	}
	public function __toString()
	{
		return $this->title;
	}

	/**
	 * Determines is a category is publicly accessible
	 * @return boolean
	 */
	public function isPublic()
	{
		return $this->access_type == self::ACCESS_PUBLIC ? 1 : 0;
	}
	/**
	 * Determines is a category is only accessible by the creator
	 * @return boolean
	 */
	public function isPrivate()
	{
		return $this->access_type == self::ACCESS_PRIVATE;
	}
	/**
	 * Determines is a category is accessible by selected groups
	 * @return boolean
	 */
	public function isShared()
	{
		return $this->access_type == self::ACCESS_SHARED;
	}
	/**
	 * Checks if it has been locked by author. Used by folders
	 * @return boolean
	 */
	public function isLocked()
	{
		return $this->is_locked;
	}
	public function update($data)
	{
		unset($data['id']);
		$this->setFromArray($data);
		$this->save();
	}
	/**
	 * Determines if a user has access to a this category.
	 * Note that this has no effect on any previously set access rules. I.e If a user
	 * has not been given "view" privilege to the document module, he will not be able
	 * to view the categorys/folder even if this function returns TRUE.
	 * @param $user_id
	 * @return Boolean
	 */
	public function canAccess($user_id)
	{
		if($this->user_id == $user_id)return true;//content creator can access
		if($this->isPublic())return true;
		//ok, category is not public
		//check if category has been shared with a group the user belongs to
		$groups = $this->getGroups();
		foreach($groups as $group)
		{
			if($group->containsMember($user_id))
				return true;
		}
		//ok, category was not shared with any group the user belongs to
		//check if category was shared specifically with user
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::SHARED_CATEGORYS));
		$row = $table->fetchRow($table->select()->where('user_id = ?',$user_id)->where('category_id = ?',$this->getId()));
		if(!empty($row))
		{
			return true;
		}
		//we just want to know if the category was shared with any user
		$shared = $table->fetchAll($table->select()->where('category_id = ?',$this->getId()));

		//If the category was not restricted (i.e. not shared with user or group)
		//then inherit the parent access.
		if(count($shared) == 0 && count($groups) == 0)
		{
    		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Category'));
    		$parent = $table->fetchRow($table->select()->where('id = ?',$this->parent_id)->where('active=1'));

			if(!empty($parent))
			{
				return $parent->canAccess($user_id);
			}
		}
		return false;

	}
	/**
	 * Determines if a user can modify the category.
	 * Modification include, rename, delete, move, share
	 * @param int $user_id
	 * @return boolean
	 */
	public function canModify($user_id)
	{
		return $this->user_id == $user_id || UserUtil::isAllowed("admin_index");
	}
	/**
	 * Returns the number of direct children the category contains.
	 * Children are contents + categorys.
	 * @return number
	 */
	public function getNumberOfChildren()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		$numContents = $table->fetchAll($table->select()->where('active=1')->where('category_id = ?',$this->getId()))->count();

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS));
		$numCategorys = $table->fetchAll($table->select()->where('active=1')->where('parent_id = ?',$this->getId()))->count();

		return $numCategorys + $numContents;
	}
	public function do_delete($deleteChildren = false)
	{
		//first delete category children
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS));
		if(!$deleteChildren)
		{
			$table->update(array('parent_id'=>$this->parent_id), "parent_id = ".$this->id);
		}
		else
		{
			$table->update(array('active'=>0), "parent_id = ".$this->id);
		}
		//then delete content children
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CONTENT_CATEGORYS));
		if(!$deleteChildren)
		{
			if($this->parent_id > 0)//do not move files to root folder
				$table->update(array('category_id'=>$this->parent_id), "category_id = ".$this->id);
		}
		else
		{
			$table->update(array('active'=>0), "category_id = ".$this->id);
		}

		//finally delete this category

		$this->active = 0;
			//if user chooses to move children to parent folder (i.e. not delete children),
			// then category is deleted permanently.
		$this->deleted = $deleteChildren ? 0 : 1;
		$this->save();

	}

	/**
	 * Returna  rowset of User objects
	 * @return Zend_Db_Table_RowSet
	 */
	public function getSharedUsers()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::USERS,'rowClass'=>'User'));
		$select = $table->select();
		$select->setIntegrityCheck(false);
		$select = $select->from(array('a'=>PrecurioTableConstants::USERS))
							->join(array('b'=>PrecurioTableConstants::SHARED_CATEGORYS),'a.user_id = b.user_id',array('category_id'))
							->where('category_id = ?',$this->getId() );
		$users = $table->fetchAll($select);
		return $users;
	}
	public function setSharedUsers($user_ids)
	{
		if(empty($user_ids))$user_ids = array();
		//get users already shared with.
		$sharedUsers = $this->getSharedUsers();
		$sharedUserIds  = array();
		foreach($sharedUsers as $user)$sharedUserIds[] = $user->user_id;

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::SHARED_CATEGORYS));
		//delete all previous share.
		$where = $table->getAdapter()->quoteInto('category_id = ?',$this->getId() );
		$table->delete($where);
		$data = array();
		$data['category_id'] = $this->getId();
		$data['sharer_id'] = Precurio_Session::getCurrentUserId();
		$data['date_created'] = Precurio_Date::now()->getTimestamp();
		foreach($user_ids as $user_id)
		{
			$data['user_id'] = $user_id;
			$row = $table->createRow($data);
			$row->save();
			if(!in_array($user_id, $sharedUserIds))//not shared with this user before
			{
				//@todo folder share notification. Means we have to make category an object
			}
		}
	}
	public function isParent($category)
	{
		if(is_int($category))$category = Category::getCategory($category);


	}
	/**
	 * Returns categorys that are public
	 * @param $is_folder boolean  - Fetch Folders instead. Default = false do not fetch folders.
	 * @version 4 compatible
	 * @return Zend_Db_Table_Rowset
	 */
	public static function getPublicCategorys($is_folder=0)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Category'));
		$items = $table->fetchAll($table->select()->where('active=1')->where('is_folder = ?',$is_folder)->where('access_type=?',Category::ACCESS_PUBLIC)->where("appid='' OR appid='cms' OR appid='0'"));
		return $items;
	}
	/**
	 * Returns folders/categorys that are shared and accessible by the current user.
	 *
	 * For a shared category to be accessible by a user, the user must be a member of one
	 * of the groups given access to the category.
	 * @param $user_id int
	 * @param $is_folder boolean  - Fetch Folders instead. Default = 0 do not fetch folders.
	 * @return array
	 */
	public static function getSharedCategorys($user_id=0,$is_folder=0)
	{
		if(empty($user_id))$user_id = Precurio_Session::getCurrentUserId();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Category'));
		$items = $table->fetchAll($table->select()->where('active=1')->where('is_folder = ?',$is_folder)->where('access_type=?',Category::ACCESS_SHARED)->where("appid='' OR appid='cms'"));
		$result = array();
		foreach($items as $category)
		{
			$groups = $category->getGroups();
			foreach($groups as $group)
			{
				if($group->containsMember($user_id))
				{
					$result[] = $category;
					break;//check next category.
				}
			}
		}
		return $result;
	}
	/**
	 * Returns folders/categorys that are only accessible by the user
	 * @param $user_id int
	 * @param $is_folder boolean  - Fetch Folders instead. Default = 0 do not fetch folders.
	 * @return Zend_Db_Table_Rowset
	 */
	public static function getPrivateCategorys($user_id=0,$is_folder=0)
	{
		if(empty($user_id))$user_id = Precurio_Session::getCurrentUserId();
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Category'));
		$items = $table->fetchAll($table->select()->where('active=1')
												->where('is_folder = ?',$is_folder)
												->where('access_type=?',Category::ACCESS_PRIVATE)
												->where('user_id = ?',$user_id)
												->where("appid='' OR appid='cms'"));
		return $items;
	}
	/**
	 * Get the string representation of an access type
	 * @param $accessType int
	 * @return string
	 */
	public static function getAccessStr($accessType)
	{
		$tr = Zend_Registry::get('Zend_Translate');
		if($accessType == Category::ACCESS_SHARED)return 'Shared';
		if($accessType == Category::ACCESS_PUBLIC)return 'Public';
		if($accessType == Category::ACCESS_PRIVATE) return 'Private';
	}

	/**
	 * @param int $id id of category
	 * @return Category
	 */
	public static function getCategory($id)
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Category'));
		$item = $table->fetchRow($table->select()->where('id = ?',$id));
		return $item;
	}

	/**
	 * Returns all content categorys
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public static function getAll()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Category'));
		$items = $table->fetchAll($table->select()->where('active=1')->where('is_folder = 0')->where('appid = ? OR appid = "cms"',"")->order('title'));
		return $items;
	}

	/**
	 * Get all root content categorys that are public
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public static function getRootCategorys()
	{
		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS,'rowClass'=>'Category'));
		$items = $table->fetchAll($table->select()->where('active=1')->where('is_folder=0')
				->where('parent_id=0')->where('appid = ? OR appid = "cms"',"")->order('title'));
		return $items;
	}
	/**
	 * Create a new category and returns the id (an alternative to Folders::createNew())
	 * @param array $data
	 * @return int
	 * @version 4
	 */
	public static function create($data)
	{
		if(!empty($data['id']))return $data["id"];//do not create if id already exists

		$table = new Zend_Db_Table(array('name'=>PrecurioTableConstants::CATEGORYS));
		$row = $table->createRow();$row->setFromArray($data);
		$id = $row->save();
		return $id;
	}
}

?>