#!/usr/bin/perl
# encoding: utf-8
#
# author: Kyle Yetter
#

package IMH::CPanel::User;

use strict;
use warnings;
use Cpanel::Config::LoadCpUserFile ();
use Cpanel::Config::userdata ();
use File::Slurp qw( slurp read_dir write_file );
use User::pwent;
use IMH::CPanel::FindUser;
use overload '""' => \&to_s;

*load_cp_user_file = *Cpanel::Config::LoadCpUserFile::loadcpuserfile;
*load_user_data    = *Cpanel::Config::userdata::load_userdata;

sub new {
  my ( $class, $user ) = @_;
  my $obj = bless( {}, $class || __PACKAGE__ );
  $obj->initialize( $user );
  return $obj;
}

sub initialize {
  my ( $self, $user )              = @_;
  my $name                         = identify_user( $user );

  $name or die "`$user' is not a valid cPanel user";

  $self->{ name }                  = $user;
  $self->{ pwent }                 = getpwnam( $user );
  $self->{ profile }               = load_cp_user_file( $user );
  $self->{ domain_groups }         = load_user_data( $user );
  $self->{ domain_data }           = {};
}

sub name {
  my ( $self ) = @_;
  return $self->{ name };
}

sub home {
  my ( $self ) = @_;
  return $self->{ pwent }->dir;
}

sub uid {
  my ( $self ) = @_;
  return $self->{ pwent }->uid;
}

sub gid {
  my ( $self ) = @_;
  return $self->{ pwent }->gid;
}

sub domain_data {
  my ( $self, $domain ) = @_;
  my $user = $self->{ name };
  $self->{ domain_data }->{ $domain } ||= load_user_data( $user, $domain );
  return $self->{ domain_data }->{ $domain };
}

sub domains {
  my ( $self ) = @_;
  my $profile = $self->{ profile };
  my @domains = ( $profile->{ 'DOMAIN' }, @{ $profile->{ 'DOMAINS' } } );
  return( wantarray ? @domains : [ @domains ] );
}

sub primary_domain {
  my ( $self ) = @_;
  my $doms     = $self->{ domain_groups };
  return $doms->{ main_domain };
}

sub parked_domains {
  my ( $self ) = @_;
  my $doms     = $self->{ domain_groups };
  my $parked   = $doms->{ parked_domains } || [];
  return( wantarray ? @$parked : $parked );
}

sub addon_domains {
  my ( $self ) = @_;
  my $doms     = $self->{ domain_groups };
  my @addons   = keys %{ $doms->{ addon_domains } || {} };
  return( wantarray ? @addons : [ @addons ] );
}

sub subdomains {
  my ( $self, $all ) = @_;
  my $doms           = $self->{ domain_groups };
  my %subdomains     = map { $_ => 1 } @{ $doms->{ sub_domains } || [] };
  unless ( $all ) {
    my @addons  = values %{ $doms->{ addon_domains } || {} };
    for ( @addons ) { delete $subdomains{ $_ }; }
  }
  my @subdomains = keys %subdomains;
  return( wantarray ? @subdomains : [ @subdomains ] );
}


sub document_root {
  my $self   = shift;
  my $domain = shift || $self->primary_domain;
  return $self->domain_data( $domain )->{ documentroot };
}

sub ip {
  my ( $self ) = @_;
  my $profile  = $self->{ profile };
  return $profile->{ IP };
}

sub start_date {
  my ( $self ) = @_;
  my $profile  = $self->{ profile };
  return( 0 + $profile->{ STARTDATE } );
}

sub plan {
  my ( $self ) = @_;
  my $profile  = $self->{ profile };
  return $profile->{ PLAN };
}

sub is_suspended {
  my ( $self ) = @_;
  my $name = $self->{ name };
  return( -f "/var/cpanel/suspended/$name" );
}

sub suspension_note {
  my ( $self ) = @_;
  my $name = $self->{ name };
  if ( -f "/var/cpanel/suspended/$name" ) {
    return slurp( "/var/cpanel/suspended/$name" );
  } else {
    return undef;
  }
}

sub domlog_paths {
  my $self   = shift;
  my @doms   = @_ ? @_ : $self->domains;
  my %logs;

  for my $dom ( @doms ) {
    my $data  = $self->domain_data( $dom );
    for my $log_info ( @{ $data->{ customlog } } ) {
      $logs{ $log_info->{ target } } = 1;
    }
  }

  my @logs = sort keys %logs;
  return( wantarray ? @logs : [ @logs ] );
}


sub to_s {
  my ( $self ) = @_;
  my $name = $self->{ name };
  return "$name";
}
