<?php

/**
 * Openbiz Cubi Application Platform
 *
 * LICENSE http://code.google.com/p/openbiz-cubi/wiki/CubiLicense
 *
 * @package   cubi.service
 * @copyright Copyright (c) 2005-2011, Openbiz Technology LLC
 * @license   http://code.google.com/p/openbiz-cubi/wiki/CubiLicense
 * @link      http://code.google.com/p/openbiz-cubi/
 * @version   $Id: pdfService.php 3506 2012-06-25 06:32:24Z agus.suhartono@gmail.com $
 */
/**
 * @package PluginService
 */
require_once(MODULE_PATH . "/pdf/lib/mpdf50/mpdf.php");
define('_MPDF_PATH', MODULE_PATH . "/pdf/lib/mpdf50/");

/**
 * pdfService - 
 * class pdfService is the plug-in service of printing openbiz form to pdf 
 * 
 * @package PluginService
 * @author rocky swen 
 * @copyright Copyright (c) 2003
 * @version $Id: pdfService.php 3506 2012-06-25 06:32:24Z agus.suhartono@gmail.com $
 * @access public 
 */
class pdfService
{

    protected $pdfObj;

    public function __construct()
    {
        $this->pdfObj = new mPDF('c'); // use default. check http://mpdf1.com/manual/index.php?tid=184
        //$this->pdfObj = new mPDF('zh-CN','A4');   // chinese example
    }

    /**
     * Get PDF object
     * 
     * @return type 
     */
    public function getPdfObject()
    {
        return $this->pdfObj;
    }

    /**
     * Set config
     * 
     * @param type $config 
     */
    public function setConfig($config = null)
    {
        if ($config == null)
        {
            $config = $this->GetDefaultConfig();
        }

        if ($config['url'])
        {
            $this->pdfObj->setBasePath($config['url']);
        } else
        {
            $this->pdfObj->setBasePath(SITE_URL);
        }

        $this->_setPageHeader($config);
        $this->_setPageFooter($config);
        
        //set protection
        if ($config['password'] || $config['readonly_password'])
        {
            $protect_array = array();
            if ($config['protect_copy'])
            {
                array_push($protect_array, "copy");
            }
            if ($config['protect_print'])
            {
                array_push($protect_array, "print");
            }
            if ($config['protect_modify'])
            {
                array_push($protect_array, "modify");
            }
            if ($config['protect_annot'])
            {
                array_push($protect_array, "annot-forms");
            }
            $this->pdfObj->SetProtection($protect_array, $config['readonly_password'], $config['password']);
        }

        // set watermark
        $this->_setWatermark($config);

        //meta setting
        if ($config['meta_title'])
        {
            switch ($config['meta_title'])
            {
                case "DEFAULT":
                    break;
                case "NONE":
                    $this->pdfObj->SetTitle("");
                    break;
                default:
                    $this->pdfObj->SetTitle($config['meta_title']);
                    break;
            }
        }

        if ($config['meta_author'])
        {
            switch ($config['meta_author'])
            {
                case "DEFAULT_USERNAME":
                    $this->pdfObj->setAuthor(BizSystem::getUserProfile("username"));
                    break;
                case "DEFAULT_DISPLAY_NAME":
                    $this->pdfObj->setAuthor(BizSystem::getUserProfile("profile_display_name"));
                    break;
                case "DEFAULT":
                    $this->pdfObj->setAuthor("Openbiz Cubi");
                    break;
                case "NONE":
                    $this->pdfObj->setAuthor("");
                    break;
                default:
                    $this->pdfObj->setAuthor($config['meta_author']);
                    break;
            }
        }

        if ($config['meta_creator'])
        {
            switch ($config['meta_creator'])
            {
                case "DEFAULT":
                    $this->pdfObj->SetAuthor("Openbiz PDF Printer");
                    break;
                case "NONE":
                    $this->pdfObj->SetCreator("");
                    break;
                default:
                    $this->pdfObj->SetCreator($config['meta_creator']);
                    break;
            }
        }

        if ($config['meta_subject'])
        {
            $this->pdfObj->SetSubject($config['meta_subject']);
        }

        if ($config['meta_keywords'])
        {
            $this->pdfObj->SetKeywords($config['meta_keywords']);
        }
    }

    /**
     * Write HTML
     * @param string $html 
     */
    public function writeHTML($html)
    {
        $this->pdfObj->WriteHTML($html);
    }

    /**
     * Return PDF output
     * 
     * @param string $filename
     * @return string 
     */
    public function Output($filename = null)
    {
        if ($filename)
        {
            $file = $filename;
            touch($file);
        } else
        {
            $tmpfile = APP_FILE_PATH . "/tmpfiles";
            if (!is_dir($tmpfile))
            {
                mkdir($tmpfile);
            }

            $this->CleanFiles($tmpfile, 100);
            $file_tmp = tempnam($tmpfile, 'tmp');
            $file = $file_tmp . '.pdf';
            $file = str_replace("\\", "/", $file);
            unlink($file_tmp);
        }
        $this->pdfObj->Output($file);
        $path_parts = pathinfo($file);
        $file_download = APP_FILE_URL . "/tmpfiles/" . $path_parts['basename'];

        return $file_download;
    }

    /**
     * Clean temporary file
     * 
     * @param string $directory
     * @param number $seconds 
     */
    function cleanFiles($directory, $seconds)
    {
        //Delete temporary files
        $t = time();
        $directoryHandler = opendir($directory);
        while ($file = readdir($directoryHandler))
        {
            if (substr($file, 0, 3) == 'tmp' && substr($file, -4) == '.pdf')
            {
                $path = $directory . '/' . $file;
                if ($t - filemtime($path) > $seconds)
                    unlink($path);
            }
        }
        closedir($directoryHandler);
    }

    /**
     * Get default config
     * 
     * @return array 
     */
    public function getDefaultConfig()
    {
        $system_config = array();
        $config = array();

        $systemConfigArr = BizSystem::getObject("pdf.do.PdfDO", 1)->directfetch();
        foreach ($systemConfigArr as $item)
        {
            $system_config[$item['name']] = $item['value'];
        }
        return $config;
    }

    /**
     * Set Pdf watermark
     * @param array $config 
     */
    private function _setWatermark($config)
    {
        if ($config['watermark_type'])
        {
            if ($config['watermark_alpha'] > 1)
            {
                $alpha = (float) ("0." . $config['watermark_alpha']);
            } else
            {
                $alpha = $config['watermark_alpha'];
            }
            switch ($config['watermark_type'])
            {
                case "Picture":
                    $this->pdfObj->SetWatermarkImage("file://" . APP_HOME . $config['watermark_picture'], $alpha, $config['watermark_size'], $config['watermark_position']);
                    $this->pdfObj->showWatermarkImage = true;
                    break;
                case "Text":
                    $this->pdfObj->SetWatermarkText($config['watermark_text'], $alpha);
                    $this->pdfObj->showWatermarkText = true;
                    break;
            }
        }
    }

    /**
     * Set Pdf header
     * @param array $config 
     */
    private function _setPageHeader($config)
    {
        if ($config['page_header_type'])
        {
            switch ($config['page_header_type'])
            {
                case "Html":
                    switch ($config['page_header_html_even_type'])
                    {
                        case "CUSTOM":
                            $this->pdfObj->SetHTMLHeader($config['page_header_html_odd'], "O");
                            $this->pdfObj->SetHTMLHeader($config['page_header_html_even'], "E");
                            break;
                        case "SAME":
                            $this->pdfObj->SetHTMLHeader($config['page_header_html_odd'], "O");
                            $this->pdfObj->SetHTMLHeader($config['page_header_html_odd'], "E");
                            break;
                    }
                    break;
                case "Text":
                    $header_odd = array(
                        'L' => array(
                            'content' => $config['page_header_text_left'],
                            'font-size' => $config['page_header_text_left_font'],
                            'font-style' => $config['page_header_text_left_style'],
                            'color' => "#" . $config['page_header_text_left_color']
                        ),
                        'C' => array(
                            'content' => $config['page_header_text_center'],
                            'font-size' => $config['page_header_text_center_font'],
                            'font-style' => $config['page_header_text_center_style'],
                            'color' => "#" . $config['page_header_text_center_color']
                        ),
                        'R' => array(
                            'content' => $config['page_header_text_right'],
                            'font-size' => $config['page_header_text_right_font'],
                            'font-style' => $config['page_header_text_right_style'],
                            'color' => "#" . $config['page_header_text_right_color']
                        ),
                        'line' => $config['page_header_text_line'],
                    );
                    $header_even = array();
                    $header_even["L"] = $header_odd["R"];
                    $header_even["C"] = $header_odd["C"];
                    $header_even["R"] = $header_odd["L"];
                    $header_even["line"] = $header_odd["line"];

                    switch ($config['page_header_text_even_type'])
                    {
                        case "MIRROR":
                            $this->pdfObj->SetHeader($header_odd, "O");
                            $this->pdfObj->SetHeader($header_even, "E");
                            break;
                        case "SAME":
                            $this->pdfObj->SetHeader($header_odd, "O");
                            $this->pdfObj->SetHeader($header_odd, "E");
                            break;
                    }
                    break;
            }
        }
    }

    /**
     * Set page footer
     * @param array $config 
     */
    private function _setPageFooter($config)
    {
        if ($config['page_footer_type'])
        {
            switch ($config['page_footer_type'])
            {
                case "Html":
                    switch ($config['page_footer_html_even_type'])
                    {
                        case "CUSTOM":
                            $this->pdfObj->SetHTMLFooter($config['page_footer_html_odd'], "O");
                            $this->pdfObj->SetHTMLFooter($config['page_footer_html_even'], "E");
                            break;
                        case "SAME":
                            $this->pdfObj->SetHTMLFooter($config['page_footer_html_odd'], "O");
                            $this->pdfObj->SetHTMLFooter($config['page_footer_html_odd'], "E");
                            break;
                    }
                    break;
                case "Text":
                    $footer_odd = array(
                        'L' => array(
                            'content' => $config['page_footer_text_left'],
                            'font-size' => $config['page_footer_text_left_font'],
                            'font-style' => $config['page_footer_text_left_style'],
                            'color' => "#" . $config['page_footer_text_left_color']
                        ),
                        'C' => array(
                            'content' => $config['page_footer_text_center'],
                            'font-size' => $config['page_footer_text_center_font'],
                            'font-style' => $config['page_footer_text_center_style'],
                            'color' => "#" . $config['page_footer_text_center_color']
                        ),
                        'R' => array(
                            'content' => $config['page_footer_text_right'],
                            'font-size' => $config['page_footer_text_right_font'],
                            'font-style' => $config['page_footer_text_right_style'],
                            'color' => "#" . $config['page_footer_text_right_color']
                        ),
                        'line' => $config['page_footer_text_line'],
                    );
                    $footer_even = array();
                    $footer_even["L"] = $footer_odd["R"];
                    $footer_even["C"] = $footer_odd["C"];
                    $footer_even["R"] = $footer_odd["L"];
                    $footer_even["line"] = $footer_odd["line"];

                    switch ($config['page_footer_text_even_type'])
                    {
                        case "MIRROR":
                            $this->pdfObj->SetFooter($footer_odd, "O");
                            $this->pdfObj->SetFooter($footer_even, "E");
                            break;
                        case "SAME":
                            $this->pdfObj->SetFooter($footer_odd, "O");
                            $this->pdfObj->SetFooter($footer_odd, "E");
                            break;
                    }
                    break;
            }
        }
    }

}
