<?php

/**
 *
 * liveSite - Enterprise Website Platform
 * 
 * @author      Camelback Web Architects
 * @link        https://livesite.com
 * @copyright   2001-2019 Camelback Consulting, Inc.
 * @license     https://opensource.org/licenses/mit-license.html MIT License
 *
 */

function array_map_recursive($callback, $array)
{
    $r = array();
    if (is_array($array)) {
        foreach($array as $key => $value) {
            if (is_scalar($value)) {
                $r[$key] = $callback($value);
            } else {
                $r[$key] = array_map_recursive($callback, $value);
            }
        }
    }
    
    return $r;
}

function array_stripslashes($array)
{
    return array_map_recursive('stripslashes', $array);
}

function db_connect() {

    // Create a class to hold the MySQL connection object so that we can access it anywhere.
    // We check if class already exists, because something might run this function multiple times.
    if (!class_exists('db')) {
        class db {
            public static $con;
        }
    }

    db::$con = @mysqli_connect(DB_HOST, DB_USERNAME, DB_PASSWORD, DB_DATABASE);
    
    if (db::$con) {
        init_mysql_charset();
        define('DB_CONNECTED', true);

    } else {
        define('DB_CONNECTED', false);
        output_error('Sorry, this website could not connect to the database.  The server administrator should check the status of the database.  If there is not a problem with the database, then the server administrator should verify that the database information in the config.php file in the software directory is correct.');
    }

    // Disable MySQL strict mode, because later versions of MySQL enable strict mode by default,
    // and liveSite is not compatible with strict mode.  This will also remove all other sql modes,
    // however that should be fine.
    mysqli_query(db::$con, "SET SESSION sql_mode = ''");

    // If this site has enabled the legacy MySQL connection in the config.php and it is using a version
    // of PHP that supports that (i.e. before PHP 7), then create a legacy DB connection.  Sites might
    // enable this feature because they have old custom code in hook code, styles, custom layouts, or etc.
    // that uses the old mysql extension and relies on liveSite to start that connection.
    // If a site enables this feature, then a second db connection will be opened for every request,
    // which might have a negative performance effect.

    if (defined('DB_LEGACY') and DB_LEGACY === true and function_exists('mysql_connect')) {

        $link=@mysql_connect(DB_HOST, DB_USERNAME, DB_PASSWORD);

        if ($link and @mysql_select_db(DB_DATABASE)) {

            $mysql_version = preg_replace('#[^0-9\.]#', '', mysql_get_server_info());

            if (version_compare($mysql_version, '5.5.3', '>=') == true) {
                $mysql_character_set = 'utf8mb4';
            } else {
                $mysql_character_set = 'utf8';
            }

            if (function_exists('mysql_set_charset')) {
                @mysql_set_charset($mysql_character_set);
            }

            @mysql_query("SET NAMES '" . $mysql_character_set . "' COLLATE '" . $mysql_character_set . "_unicode_ci'");
        }
    }
}

function init_mysql_charset() {

    $mysql_version = preg_replace('#[^0-9\.]#', '', mysqli_get_server_info(db::$con));

    if (version_compare($mysql_version, '5.5.3', '>=') == true) {
        $mysql_character_set = 'utf8mb4';
    } else {
        $mysql_character_set = 'utf8';
    }

    // If the mysqli_set_charset function is available (PHP 5.2.3+)
    // then use it to set the character set.  This is necessary
    // so that mysqli_real_escape_string will be secure.
    if (function_exists('mysqli_set_charset')) {
        mysqli_set_charset(db::$con, $mysql_character_set);
    }

    // Even though we most likely ran mysqli_set_charset above,
    // we have to run the command below also so that all charset
    // and collation settings get set properly.  For example,
    // mysqli_set_charset does not set collation_connection
    // to the collation that we want.
    mysqli_query(db::$con, "SET NAMES '" . $mysql_character_set . "' COLLATE '" . $mysql_character_set . "_unicode_ci'");
}

// Use this to run any type of general db query (e.g. select, insert, update, etc.).  It will
// attempt to return a value or array that you might expect, however there might be some cases
// where you need to use one of the specialized db functions (e.g. db_items) if you need the data
// returned in a certain way.  This function tries to predict how you want result data returned.

function db($query) {

    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // If this was a query like an insert or update and it was successful (true), or if the query
    // failed (false), then return the result.
    if (($result === true) or ($result === false)) {
        return $result;
    }

    // Otherwise a resource should have been returned, so deal with that.

    $number_of_rows = mysqli_num_rows($result);

    // If there are no result rows, then the query might have been an insert or update type
    // or maybe there is just no result from a select, so return empty string.
    if (!$number_of_rows) {
        return '';
    }

    // If there is just one row, then return data in a certain way.
    if ($number_of_rows == 1) {

        // If there is only one field/column, then just return that one value.
        if (mysqli_num_fields($result) == 1) {
            $row = mysqli_fetch_row($result);
            return $row[0];

        // Otherwise there is more than one field/column, so return the whole array.
        } else {
            return mysqli_fetch_assoc($result);
        }

    // Otherwise there is more than one row, so return data in a different way.
    } else {

        // If there is only one field/column, then return an array of values.
        if (mysqli_num_fields($result) == 1) {

            $values = array();
            
            while ($row = mysqli_fetch_row($result)) {
                $values[] = $row[0];
            }
            
            return $values;

        // Otherwise there is more than one field/column, so return an array of items.
        } else {

            $items = array();

            while ($item = mysqli_fetch_assoc($result)) {
                $items[] = $item;
            }
            
            return $items;

        }

    }

}

// Create function that can be used to query for a single value (e.g. SELECT COUNT(*) FROM page).
function db_value($query) {
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_row($result);
    return $row[0];
}

// Create function that can be used to get an array of values (e.g. SELECT page_id FROM page)
function db_values($query) {

    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    $values = array();
    
    while ($row = mysqli_fetch_row($result)) {
        $values[] = $row[0];
    }
    
    return $values;
}

// Create function that can be used to get an array of values for a single item (e.g. the properties for a single page)
function db_item($query) {
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    return mysqli_fetch_assoc($result);
}

// Create function that can be used to get an array of multiple items (e.g. the properties of many pages)
function db_items($query, $key_column = '') {
    
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    $items = array();

    // If a key column was supplied, then prepare array with
    // that column value as the key for the array item.
    if ($key_column != '') {
        while ($item = mysqli_fetch_assoc($result)) {
            $items[$item[$key_column]] = $item;
        }

    // Otherwise a key column was not supplied, so prepare array
    // without specific key values (i.e. just use default: 0, 1, 2, etc.).
    } else {
        while ($item = mysqli_fetch_assoc($result)) {
            $items[] = $item;
        }
    }
    
    return $items;
}

// Create a function that can be used to escape content that will be placed in HTML.
// The benefit of this function is that it provides a shorthand version of htmlspecialchars.
// It also sets all of the correct settings for htmlspecialchars that we want
// so that we don't rely on the defaults, because the defaults vary depending on the PHP version.
function h($content)
{
    return htmlspecialchars($content, ENT_COMPAT | ENT_HTML401, 'UTF-8');
}

// escape data to be safe for MySQL queries
function escape($string) {
    return mysqli_real_escape_string(db::$con, $string);
}

// Duplicate of function above with just a shorter name.
function e($string) {
    return mysqli_real_escape_string(db::$con, $string);
}

// escape special characters in SQL LIKE comparison
function escape_like($string)
{
    $string = str_replace('%', '\%', $string);
    $string = str_replace('_', '\_', $string);
    
    return $string;
}

// escape special characters in a javascript string
function escape_javascript($string)
{
    // escape backslashes
    $string = str_replace('\\', '\\\\', $string);
    
    // escape single quotes
    $string = str_replace("'", "\\'", $string);
    
    // escape double quotes
    $string = str_replace('"', '\\"', $string);
    
    // escape new line
    $string = str_replace("\r\n", '\n', $string);
    $string = str_replace("\n", '\n', $string);

    $string = str_ireplace('</script>', '<\/script>', $string);
    
    return $string;
}

// create function to escape quotation marks for CSV data

function escape_csv($string) {
    return str_replace('"', '""', $string);
}

// Create a function that is used for outputting URLs
// in anchor hrefs where the URLs might be untrusted because they came from the
// query string or etc.  This prevents XSS by preventing a URL
// like "javascript:example()".  Currently, this function does not support
// rare URLs like "mailto:" or relative URLs like "example", because we don't use those.

function escape_url($url) {

    // If the URL starts with a valid prefix (e.g. http),
    // and not something like "javascript:example()", then return the URL.
    if (
        (mb_substr($url, 0, 1) == '/')
        || (mb_substr($url, 0, 3) == '../')
        || (mb_substr($url, 0, 2) == '//')
        || (mb_substr($url, 0, 7) == 'http://')
        || (mb_substr($url, 0, 8) == 'https://')
    ) {
        return $url;

    // Otherwise the URL is not valid, so return false.
    } else {
        return false;
    }
}

function unhtmlspecialchars($string) {

    // We added ENT_QUOTES and the corresponding &#39; and &#039; lines below,
    // in v9.0.0 because CKEditor converts single quotes to entities
    // which caused the {{name: 'example'}} feature in form list view
    // advanced search feature to break.

    // if this version of PHP already has a function for this, then use the function
    if (function_exists('htmlspecialchars_decode')) {
        return htmlspecialchars_decode($string, ENT_QUOTES);
        
    // else this version of PHP does not have a function for this, so use our own code
    } else {
        $string = str_replace('&amp;', '&', $string);
        $string = str_replace('&quot;', '"', $string);
        $string = str_replace('&#39;', '\'', $string);
        $string = str_replace('&#039;', '\'', $string);
        $string = str_replace('&lt;', '<', $string);
        $string = str_replace('&gt;', '>', $string);
        
        return $string;
    }
}

function mysqli_fetch_items($result) {

    $items = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $items[] = $row;
    }
    
    return $items;
}

// Used for critical errors which aborts any page rendering
// that we might have done so far and shows an error.

function output_error($error_message, $response_code = 0) {

    set_response_code($response_code);

    $mysql_error = '';

    if (db::$con) {
        $mysql_error = mysqli_error(db::$con);
    }

    // If debug is enabled or if the error was generated by the install/update script
    // and there is a MySQL error, then output the error.
    if (
        (
            (defined('DEBUG') and DEBUG)
            || (defined('INSTALL_OR_UPDATE') and INSTALL_OR_UPDATE)
        )
        && ($mysql_error)
    ) {
        $debug_message = h($mysql_error) . '<br />';
        
    } elseif ($mysql_error) {
        $debug_message = 'The administrator can turn on Verbose Database Errors in the settings in order to see more information about this error the next time it occurs.';
        
    } else {
        $debug_message = '';
    }

    $content = '<div style="color: red">Error: ' . $error_message . '<br />' . $debug_message . '</div>';
    
    // If the db connection failed or there was an error, then just output error message content
    // without error page, in order to avoid infinite errors.
    if (!db::$con or $mysql_error != '') {
        echo $content;
        exit();
        
    // else there is not a MySQL error, so output error message content on error page
    } else {
        // If we don't know if the user is logged in (maybe because the error happened
        // early), then set that the user is not logged,
        // We need to set this constant, because other logic might run
        // during the output of the error screen, that assumes the constant is set.
        // For example, we had issue where setTimezone would try to use USER_TIMEZONE
        // and a PHP error would occur, because the constant was not set.
        if (!defined('USER_LOGGED_IN')) {
            define('USER_LOGGED_IN', false);
        }

        echo get_error_screen($content);
        exit();
    }
}

// The following function is used to return our standard error styling around
// an error message.  This is used when we want to show an error on the page
// that generated the error, so an editor can continue to interact with the page
// via the toolbar and etc., instead of using output_error() which is generally
// used for critical errors which aborts the current page rendering
// and shows the error page, which does not have a toolbar and etc.

function error($content, $response_code = 0) {

    set_response_code($response_code);

    return
        '<div class="software_error">
            <img src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/icon_error.png" alt="Error" title="" class="icon" style="float: none; display: inline-block; height: 1em; width: 1em; min-height: 16px; min-width: 16px">
            <div style="display: inline; vertical-align: top">
                <span class="description">An error occurred:</span> <span class="error">' . $content . '</span>
            </div>
        </div>';

}

// The username property can either be a username or email address.
function validate_login($username, $password) {

    // check to see if the login information is valid by trying to find a user
    $query =
        "SELECT
            user_id as id,
            user_role as role
        FROM user
        WHERE
            (
                (user_username = '" . escape($username) . "')
                OR (user_email = '" . escape($username) . "')
            )
            AND (user_password = '" . escape($password) . "')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if a user was found, then the login information is valid
    if (mysqli_num_rows($result) > 0) {
        return true;
    // else a user was not found, so the login information is not valid, so return that
    } else {
        return false;
    }
}

// The username can either be a username or email address for a user.
function validate_username($username)
{
    $query = "SELECT user_id FROM user WHERE (user_username = '" . escape($username) . "') OR (user_email = '" . escape($username) . "')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');
    if (mysqli_num_rows($result) > 0) {
        return true;
    } else {
        return false;
    }
}

// checks that user is logged in and returns array with user information
function validate_user()
{
    // if the username or password is not in the session, then the user has not logged or their session has expired, so prepare error and forward user to login screen
    if ((!isset($_SESSION['sessionusername'])) || (!isset($_SESSION['sessionpassword']))) {
        include_once('liveform.class.php');
        $liveform = new liveform('login');
        $liveform->mark_error('', 'You have not logged in, or your session has expired. Please login.');
        header('Location: ' . URL_SCHEME . HOSTNAME . PATH . SOFTWARE_DIRECTORY . '/index.php?send_to=' . urlencode(get_request_uri()));
        exit();
        
    // else if the login is invalid, then clear session and cookies and output error
    } else if (validate_login($_SESSION['sessionusername'], $_SESSION['sessionpassword']) == FALSE) {
        $logged_in_as_different_user = $_SESSION['software']['logged_in_as_different_user'];

        session_unset();
        session_destroy();

        // If the user was not logged in as a different user,
        // then also clear remember me cookies.  We don't want to clear
        // remember me cookies if user was logged in as a different user,
        // because we want the user to be able to automatically log back in
        // under their own user account now.
        if ($logged_in_as_different_user == false) {
            setcookie('software[username]', '', time() - 1000, '/');
            setcookie('software[password]', '', time() - 1000, '/');
        }

        output_error('<a href="javascript:history.go(-1)">Invalid login</a>');
    }
    
    $query = "SELECT * "
            ."FROM user "
            ."WHERE user_username = '" . escape($_SESSION['sessionusername']) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');
    $row = mysqli_fetch_array($result);
    $num_rows = mysqli_num_rows($result);

    // initialize variables
    $user_id =              $row['user_id'];
    $user_username =        $row['user_username'];
    $user_email =           $row['user_email'];
    $user_password =        $row['user_password'];
    $user_role =            $row['user_role'];
    $user_manage_contacts = $row['user_manage_contacts'];
    $user_manage_visitors = $row['user_manage_visitors'];
    $user_manage_ecommerce = $row['user_manage_ecommerce'];
    $user_view_card_data = $row['user_view_card_data'];
    $manage_ecommerce_reports = $row['manage_ecommerce_reports'];
    $user_set_offline_payment = $row['user_set_offline_payment'];
    $user_create_pages = $row['user_create_pages'];
    $user_delete_pages = $row['user_delete_pages'];    
    $user_manage_forms = $row['user_manage_forms'];
    $user_manage_calendars = $row['user_manage_calendars'];
    $user_publish_calendar_events = $row['user_publish_calendar_events'];
    $user_manage_emails =   $row['user_manage_emails'];
    $user_home =            $row['user_home'];
    $user_set_page_type_email_a_friend = $row['user_set_page_type_email_a_friend'];
    $user_set_page_type_folder_view = $row['user_set_page_type_folder_view'];
    $user_set_page_type_photo_gallery = $row['user_set_page_type_photo_gallery'];
    $user_set_page_type_custom_form = $row['user_set_page_type_custom_form'];
    $user_set_page_type_custom_form_confirmation = $row['user_set_page_type_custom_form_confirmation'];
    $user_set_page_type_form_list_view = $row['user_set_page_type_form_list_view'];
    $user_set_page_type_form_item_view = $row['user_set_page_type_form_item_view'];
    $user_set_page_type_form_view_directory = $row['user_set_page_type_form_view_directory'];
    $user_set_page_type_calendar_view = $row['user_set_page_type_calendar_view'];
    $user_set_page_type_calendar_event_view = $row['user_set_page_type_calendar_event_view'];
    $user_set_page_type_catalog = $row['user_set_page_type_catalog'];
    $user_set_page_type_catalog_detail = $row['user_set_page_type_catalog_detail'];
    $user_set_page_type_express_order = $row['user_set_page_type_express_order'];
    $user_set_page_type_order_form = $row['user_set_page_type_order_form'];
    $user_set_page_type_shopping_cart = $row['user_set_page_type_shopping_cart'];
    $user_set_page_type_shipping_address_and_arrival = $row['user_set_page_type_shipping_address_and_arrival'];
    $user_set_page_type_shipping_method = $row['user_set_page_type_shipping_method'];
    $user_set_page_type_billing_information = $row['user_set_page_type_billing_information'];
    $user_set_page_type_order_preview = $row['user_set_page_type_order_preview'];
    $user_set_page_type_order_receipt = $row['user_set_page_type_order_receipt'];
    $user_reward_points = $row['user_reward_points'];
    
    if ($user_manage_contacts == 'yes') {
        $user_manage_contacts = TRUE;
    } else {
        $user_manage_contacts = FALSE;
    }
    
    if ($user_manage_visitors == 'yes') {
        $user_manage_visitors = TRUE;
    } else {
        $user_manage_visitors = FALSE;
    }
    
    if ($user_manage_ecommerce == 'yes') {
        $user_manage_ecommerce = TRUE;
    } else {
        $user_manage_ecommerce = FALSE;
    }
    
    if ($user_manage_forms == 'yes') {
        $user_manage_forms = true;
    } else {
        $user_manage_forms = false;
    }
    
    if ($user_manage_calendars == 'yes') {
        $user_manage_calendars = true;
    } else {
        $user_manage_calendars = false;
    }
    
    if ($user_publish_calendar_events == 'yes') {
        $user_publish_calendar_events = true;
    } else {
        $user_publish_calendar_events = false;
    }

    if ($user_manage_emails == 'yes') {
        $user_manage_emails = TRUE;
    } else {
        $user_manage_emails = FALSE;
    }
    
    // convert integer database values to boolean values for newer fields
    settype($user_view_card_data, 'boolean');
    settype($manage_ecommerce_reports, 'boolean');
    settype($user_set_offline_payment, 'boolean');
    settype($user_create_pages, 'boolean');
    settype($user_delete_pages, 'boolean');
    settype($user_set_page_type_email_a_friend, 'boolean');
    settype($user_set_page_type_folder_view, 'boolean');
    settype($user_set_page_type_photo_gallery, 'boolean');
    settype($user_set_page_type_custom_form, 'boolean');
    settype($user_set_page_type_custom_form_confirmation, 'boolean');
    settype($user_set_page_type_form_list_view, 'boolean');
    settype($user_set_page_type_form_item_view, 'boolean');
    settype($user_set_page_type_form_view_directory, 'boolean');
    settype($user_set_page_type_calendar_view, 'boolean');
    settype($user_set_page_type_calendar_event_view, 'boolean');
    settype($user_set_page_type_catalog, 'boolean');
    settype($user_set_page_type_catalog_detail, 'boolean');
    settype($user_set_page_type_express_order, 'boolean');
    settype($user_set_page_type_order_form, 'boolean');
    settype($user_set_page_type_shopping_cart, 'boolean');
    settype($user_set_page_type_shipping_address_and_arrival, 'boolean');
    settype($user_set_page_type_shipping_method, 'boolean');
    settype($user_set_page_type_billing_information, 'boolean');
    settype($user_set_page_type_order_preview, 'boolean');
    settype($user_set_page_type_order_receipt, 'boolean');
    
    $user = 
        array (
            "id" => $user_id, 
            "username" => $user_username, 
            "email_address" => $user_email, 
            "role" => $user_role,
            "manage_contacts" => $user_manage_contacts,
            "manage_visitors" => $user_manage_visitors,
            "manage_ecommerce" => $user_manage_ecommerce,
            "view_card_data" => $user_view_card_data,
            "manage_ecommerce_reports" => $manage_ecommerce_reports,
            "set_offline_payment" => $user_set_offline_payment,
            "create_pages" => $user_create_pages,
            "delete_pages" => $user_delete_pages,
            "manage_forms" => $user_manage_forms,
            "manage_calendars" => $user_manage_calendars,
            "publish_calendar_events" => $user_publish_calendar_events,
            "manage_emails" => $user_manage_emails,
            "home" => $user_home, 
            'set_page_type_email_a_friend' => $user_set_page_type_email_a_friend,
            'set_page_type_folder_view' => $user_set_page_type_folder_view,
            'set_page_type_photo_gallery' => $user_set_page_type_photo_gallery,
            'set_page_type_custom_form' => $user_set_page_type_custom_form,
            'set_page_type_custom_form_confirmation' => $user_set_page_type_custom_form_confirmation,
            'set_page_type_form_list_view' => $user_set_page_type_form_list_view,
            'set_page_type_form_item_view' => $user_set_page_type_form_item_view,
            'set_page_type_form_view_directory' => $user_set_page_type_form_view_directory,
            'set_page_type_calendar_view' => $user_set_page_type_calendar_view,
            'set_page_type_calendar_event_view' => $user_set_page_type_calendar_event_view,
            'set_page_type_catalog' => $user_set_page_type_catalog,
            'set_page_type_catalog_detail' => $user_set_page_type_catalog_detail,
            'set_page_type_express_order' => $user_set_page_type_express_order,
            'set_page_type_order_form' => $user_set_page_type_order_form,
            'set_page_type_shopping_cart' => $user_set_page_type_shopping_cart,
            'set_page_type_shipping_address_and_arrival' => $user_set_page_type_shipping_address_and_arrival,
            'set_page_type_shipping_method' => $user_set_page_type_shipping_method,
            'set_page_type_billing_information' => $user_set_page_type_billing_information,
            'set_page_type_order_preview' => $user_set_page_type_order_preview,
            'set_page_type_order_receipt' => $user_set_page_type_order_receipt,
            "reward_points" => $user_reward_points
        );
    return ($user);
}

// check that user has access to area of software
function validate_area_access($user, $minimum_role)
{
    // convert role description into number
    switch($minimum_role)
    {
        case 'administrator':
            $minimum_role_num = 0;
            break;
        case 'designer':
            $minimum_role_num = 1;
            break;
        case 'manager':
            $minimum_role_num = 2;
            break;
        case 'user':
            $minimum_role_num = 3;
            break;
    }
    // if user has a user role and does not have edit rights -> output error and return false
    if (($user['role'] == 3) && (no_acl_check($user['id']) == FALSE)) {
        log_activity("access denied because user had no edit rights", $_SESSION['sessionusername']);
        output_error('Access denied. <a href="javascript:history.go(-1)">Go back</a>.');
        return FALSE;
    }
    // if user's role is high enough, then user has access -> return true
    if ($user['role'] <= $minimum_role_num) {
        return TRUE;
    // else user's role is not high enough -> output error and return false
    } else {
        log_activity("access denied because user attempted to access area for higher role", $_SESSION['sessionusername']);
        output_error('Access denied. <a href="javascript:history.go(-1)">Go back</a>.');
        return FALSE;
    }
}

function validate_contact_access($user, $contact_id)
{
    // if user has a role greater than user role, user has access, so return true
    if ($user['role'] < 3) {
        return true;
        
    // else user does not have a role greater than user role, so do further checks to see if user has access
    } else {
        // get all contact groups that contact is in
        $query =
            "SELECT contact_group_id as id
            FROM contacts_contact_groups_xref
            WHERE contact_id = '" . escape($contact_id) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $contact_groups = array();
        
        // loop through all contact groups in order to add them to array
        while ($row = mysqli_fetch_assoc($result)) {
            $contact_groups[] = $row;
        }
        
        // loop through all contact groups in order to determine if user has access to contact group
        foreach ($contact_groups as $contact_group) {
            // if user has access to contact group, then user has access to contact, so return true
            if (validate_contact_group_access($user, $contact_group['id'])) {
                return true;
            }
        }
        
        // we have not returned true by now, so user does not have access to contact, so return false
        return false;
    }
}

function validate_contacts_access($user, $only_return = false)
{
    // if user has a role greater than user role, return true
    if ($user['role'] < 3) {
        return true;
        
    // else if user has access to manage contacts
    } elseif ($user['manage_contacts'] == true) {
        // check if user has access to at least one contact group
        $query =
            "SELECT count(*)
            FROM users_contact_groups_xref
            WHERE user_id = '" . escape($user['id']) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_row($result);
        
        // if user has access to at least one contact group, return true
        if ($row[0] > 0) {
            return true;
            
        // else user does not have access to at least one contact group
        } else {
            if ($only_return == false) {
                log_activity("access denied to contacts because user did not have access to at least one contact group", $_SESSION['sessionusername']);
                output_error('Access denied. You do not have access to contacts because you do not have access to at least one contact group. <a href="javascript:history.go(-1)">Go back</a>.');
            }
            return false;
        }
        
    // else user does not have access, so return false
    } else {
        if ($only_return == false) {
            log_activity("access denied to contacts", $_SESSION['sessionusername']);
            output_error('Access denied. <a href="javascript:history.go(-1)">Go back</a>.');
        }
        return false;
    }
}

function validate_contact_group_access($user, $contact_group_id)
{
    // if user has a role greater than user role, user has access, so return true
    if ($user['role'] < 3) {
        return true;
        
    // else user does not have a role greater than user role, so do further checks to see if user has access
    } else {
        // check to see if user has access to contact group
        $query =
            "SELECT user_id
            FROM users_contact_groups_xref
            WHERE
                (user_id = '" . escape($user['id']) . "')
                AND (contact_group_id = '" . escape($contact_group_id) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // if user has access to contact group, return true
        if (mysqli_num_rows($result) > 0) {
            return true;
        
        // else user does not have access to contact group, so return false
        } else {
            return false;
        }
    }
}

function validate_ecommerce_access($user)
{
    // if user has a role greater than user role OR user has access to manage e-commerce, return true
    if ($user['role'] < 3 || $user['manage_ecommerce'] == true) {
        return true;
    // else user does not have access, so return false
    } else {
        log_activity("access denied to commerce", $_SESSION['sessionusername']);
        output_error('Access denied. <a href="javascript:history.go(-1)">Go back</a>.');
        return false;
    }
}

// Used by the order report & shipping report screens to verify that the user
// has access.
function validate_ecommerce_report_access() {

    if (
        !USER_LOGGED_IN
        or ((USER_ROLE == 3) and !USER_MANAGE_ECOMMERCE_REPORTS)
    ) {
        log_activity('access denied to commerce reports');
        output_error('Access denied.');
    }

}

function validate_forms_access($user) {

    // if user has a role greater than user role OR user has access to manage forms, return true
    if ($user['role'] < 3 || $user['manage_forms'] == true) {
        return true;
    // else user does not have access, so return false
    } else {
        log_activity("access denied to forms", $_SESSION['sessionusername']);
        output_error('Access denied. <a href="javascript:history.go(-1)">Go back</a>.');
        return false;
    }
}

function validate_visitors_access($user)
{
    // if user has a role greater than user role OR user has access to manage visitors, return true
    if ($user['role'] < 3 || $user['manage_visitors'] == true) {
        return true;
    // else user does not have access, so return false
    } else {
        output_error('Access denied. <a href="javascript:history.go(-1)">Go back</a>.');
        return false;
    }
}

function validate_calendars_access($user, $only_return = false)
{
    // if user has a role greater than user role, return true
    if ($user['role'] < 3) {
        return true;
        
    // else if user has access to manage calendars
    } elseif ($user['manage_calendars'] == true) {
        // check if user has access to at least one calendar
        $query =
            "SELECT count(*)
            FROM users_calendars_xref
            WHERE user_id = '" . escape($user['id']) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_row($result);
        
        // if user has access to at least one calendar, return true
        if ($row[0] > 0) {
            return true;
            
        // else user does not have access to at least one calendar
        } else {
            if ($only_return == false) {
                log_activity("access denied to calendars because user did not have access to at least one calendar", $_SESSION['sessionusername']);
                output_error('Access denied. You do not have access to calendars because you do not have access to at least one calendar. <a href="javascript:history.go(-1)">Go back</a>.');
            }
            return false;
        }
        
    // else user does not have access, so return false
    } else {
        if ($only_return == false) {
            log_activity("access denied to calendars", $_SESSION['sessionusername']);
            output_error('Access denied. <a href="javascript:history.go(-1)">Go back</a>.');
        }
        return false;
    }
}

function validate_calendar_access($calendar_id)
{
    // if user has a role greater than user role, user has access, so return true
    if (USER_ROLE < 3) {
        return true;
        
    // else user does not have a role greater than user role, so do further checks to see if user has access
    } else {
        // check to see if user has access to calendar
        $query =
            "SELECT user_id
            FROM users_calendars_xref
            WHERE
                (user_id = '" . USER_ID . "')
                AND (calendar_id = '" . escape($calendar_id) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // if user has access to calendar, return true
        if (mysqli_num_rows($result) > 0) {
            return true;
        
        // else user does not have access to calendar, so return false
        } else {
            return false;
        }
    }
}

function validate_calendar_event_access($calendar_event_id)
{
    // if user has a role greater than user role, user has access, so return true
    if (USER_ROLE < 3) {
        return true;
        
    // else user does not have a role greater than user role, so do further checks to see if user has access
    } else {
        // get all calendars that this event is in
        $query =
            "SELECT calendar_id
            FROM calendar_events_calendars_xref
            WHERE calendar_event_id = '" . escape($calendar_event_id) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $selected_calendars = array();
        
        // loop through all selected calendars in order to build array
        while ($row = mysqli_fetch_assoc($result)) {
            $selected_calendars[] = $row['calendar_id'];
        }
        
        // loop through all selected calendars in order to determine if user has access to any calendars that this calendar event is in
        foreach ($selected_calendars as $calendar_id) {
            // if user has access to calendar, then user has access to calendar event, so return true
            if (validate_calendar_access($calendar_id) == true) {
                return true;
            }
        }
        
        // if we have gotten here then the user does not have access to any calendars that calendar event is in, so return false
        return false;
    }
}

function validate_email_access($user)
{
    // if user has a role greater than user role OR user has access to manage emails, return true
    if ($user['role'] < 3 || $user['manage_emails'] == TRUE) {
        return TRUE;
    // else user does not have access, so return false
    } else {
        log_activity("access denied to e-mail page", $_SESSION['sessionusername']);
        output_error('Access denied. <a href="javascript:history.go(-1)">Go back</a>.');
        return FALSE;
    }
}

function check_folder_access_in_array($folder_id, $folders_that_user_has_access_to)
{
    global $user;
    
    // if user has a role that is greater than a basic user or user has access to folder, then return true
    if (((isset($user['role']) == true) && ($user['role'] < 3)) || (in_array($folder_id, $folders_that_user_has_access_to) == true)) {
        return true;
        
    // else user does not have access
    } else {
        return false;
    }
}

function get_folders_that_user_has_access_to($user_id)
{
    // get folders with access set that user has access to
    $folders_with_access_set = array();
    
    $query =
        "SELECT aclfolder_folder as folder_id
        FROM aclfolder
        WHERE
            (aclfolder_user = '" . escape($user_id) . "')
            AND (aclfolder_rights = '2')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    while ($row = mysqli_fetch_assoc($result)) {
        $folders_with_access_set[] = $row['folder_id'];
    }
    
    // get all folders
    $folders = array();
    
    $query =
        "SELECT
            folder_id as id,
            folder_parent as parent_folder_id
        FROM folder";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    while ($row = mysqli_fetch_assoc($result)) {
        $folders[] = $row;
    }
    
    $folders_that_user_has_access_to = array();
    
    // loop through all folders with access set
    foreach ($folders_with_access_set as $folder_id) {
        // add folder to folders that user has access to
        $folders_that_user_has_access_to[] = $folder_id;
        
        // add child folders to folders that user has access to
        $folders_that_user_has_access_to = array_merge($folders_that_user_has_access_to, get_child_folders($folder_id, $folders));
    }
    
    // remove duplicate folders from array
    $folders_that_user_has_access_to = array_unique($folders_that_user_has_access_to);
    
    return $folders_that_user_has_access_to;
}

function get_child_folders($folder_id, $folders)
{
    $child_folders = array();
    
    // loop through all folders
    foreach ($folders as $folder) {
        // if folder is a child, then add folder to array
        if ($folder['parent_folder_id'] == $folder_id) {
            $child_folders[] = $folder['id'];
            
            $child_folders = array_merge($child_folders, get_child_folders($folder['id'], $folders));
        }
    }
    
    return $child_folders;
}

// We look at the server hostname to determine if we host this site or not.  We use this for various
// purposes include showing custom messages for hosted customers.

function we_host() {

    $server_hostname = php_uname('n');

    if  (
        stripos($server_hostname, '.livesite.com') !== false
        or stripos($server_hostname, '.getlivesite.com') !== false
        or stripos($server_hostname, '.livesitehost.com') !== false
    ) {
        return true;
    } else {
        return false;
    }
}

function output_header($toolbar = false)
{
    
    //Output menu
    global $user;
    
    $output_body_class = '';
    
    // get the name of the PHP script that was just accessed in order to determine which tab should be highlighted
    $url_path_parts = explode('/', $_SERVER['SCRIPT_NAME']);
    $file_name = $url_path_parts[count($url_path_parts) - 1];
    
    // determine the body class based on the current script that is running (used for highlighting current tab)
    switch($file_name) {
        case 'view_folders.php':
        case 'view_folders.php':
        case 'add_folder.php':
        case 'edit_folder.php':
        case 'duplicate_folder.php':
            $output_body_class = ' class="folders"';
            break;

        case 'view_pages.php':
        case 'view_short_links.php':
        case 'add_short_link.php':
        case 'edit_short_link.php':
        case 'view_auto_dialogs.php':
        case 'add_auto_dialog.php':
        case 'edit_auto_dialog.php':
        case 'add_page.php':
        case 'edit_page.php':
        case 'edit_form_list_view.php':
        case 'edit_form_item_view.php':
        case 'edit_comment.php':
            $output_body_class = ' class="pages"';
            break;
            
        case 'optimize_content.php':
            switch ($_GET['item_type']) {
                case 'page':
                    $output_body_class = ' class="pages"';
                    break;
                
                case 'product':
                case 'product_group':
                    $output_body_class = ' class="ecommerce"';
                    break;
            }
            
            break;
        
        case 'toolbar.php':
            if (isset($_GET['page_id']) == true) {
                $output_body_class = ' class="toolbar pages"';
            } else if (isset($_GET['campaign_id']) == true) {
                $output_body_class = ' class="toolbar campaigns"';
            }
            break;
        
        case 'view_files.php':
        case 'add_file.php':
        case 'edit_file.php':
        case 'image_editor_edit.php':
        case 'editor_select_image.php':
            $output_body_class = ' class="files"';
            break;
            
        case 'view_contacts.php':
        case 'add_contact.php':
        case 'edit_contact.php':
        case 'import_contacts.php':
        case 'view_contact_groups.php':
        case 'add_contact_group.php':
        case 'edit_contact_group.php':
        case 'organize_contacts.php':
            $output_body_class = ' class="contacts"';
            break;
            
        case 'view_forms.php':
        case 'view_submitted_forms.php':
        case 'edit_submitted_form.php':
        case 'import_submitted_forms.php':
            $output_body_class = ' class="forms"';
            break;

        case 'view_fields.php':
        case 'add_field.php':
        case 'edit_field.php':
            if (isset($_GET['product_id']) == true) {
                $output_body_class = ' class="ecommerce"';
            } else {
                $output_body_class = ' class="pages"';
            }
            break;
        
        case 'calendars.php':
        case 'view_calendars.php':
        case 'add_calendar.php':
        case 'edit_calendar.php':
        case 'add_calendar_event.php':
        case 'edit_calendar_event.php':
        case 'view_calendar_event_locations.php':
        case 'add_calendar_event_location.php':
        case 'edit_calendar_event_location.php':
            $output_body_class = ' class="calendars"';
            break;
            
        case 'view_email_campaigns.php':
        case 'view_email_campaign_history.php':
        case 'add_email_campaign.php':
        case 'edit_email_campaign.php':
        case 'view_email_campaign_profiles.php':
        case 'add_email_campaign_profile.php':
        case 'edit_email_campaign_profile.php':
        case 'import_email_campaign_profiles.php':
            $output_body_class = ' class="campaigns"';
            break;
            
        case 'view_ecommerce.php':
        case 'view_orders.php':
        case 'view_order_reports.php':
        case 'view_order_report.php':
        case 'view_order.php':
        case 'view_products.php':
        case 'add_product.php':
        case 'edit_product.php':
        case 'import_products.php':
        case 'view_product_groups.php':
        case 'add_product_group.php':
        case 'edit_product_group.php':
        case 'duplicate_product_group.php':
        case 'edit_featured_and_new_items.php':
        case 'view_product_attributes.php':
        case 'add_product_attribute.php':
        case 'edit_product_attribute.php':
        case 'view_gift_cards.php':
        case 'add_gift_card.php':
        case 'edit_gift_card.php':
        case 'view_tax_zones.php':
        case 'add_tax_zone.php':
        case 'edit_tax_zone.php':
        case 'view_shipping_methods.php':
        case 'add_shipping_method.php':
        case 'edit_shipping_method.php':
        case 'view_zones.php':
        case 'add_zone.php':
        case 'edit_zone.php':
        case 'view_arrival_dates.php':
        case 'add_arrival_date.php':
        case 'edit_arrival_date.php':
        case 'view_countries.php':
        case 'add_country.php':
        case 'edit_country.php':
        case 'view_states.php':
        case 'add_state.php':
        case 'edit_state.php':
        case 'view_verified_shipping_addresses.php':
        case 'add_verified_shipping_address.php':
        case 'edit_verified_shipping_address.php':
        case 'view_containers.php':
        case 'add_container.php':
        case 'edit_container.php':
        case 'view_shipping_report.php':
        case 'view_referral_sources.php':
        case 'add_referral_source.php':
        case 'edit_referral_source.php':
        case 'view_key_codes.php':
        case 'add_key_code.php':
        case 'edit_key_code.php':
        case 'import_key_codes.php':
        case 'delete_key_codes.php':
        case 'view_offers.php':
        case 'add_offer.php':
        case 'edit_offer.php':
        case 'view_offer_rules.php':
        case 'add_offer_rule.php':
        case 'edit_offer_rule.php':
        case 'view_offer_actions.php':
        case 'add_offer_action.php':
        case 'edit_offer_action.php':
        case 'view_commissions.php':
        case 'edit_commission.php':
        case 'view_recurring_commission_profiles.php':
        case 'edit_recurring_commission_profile.php':
        case 'view_currencies.php':
        case 'add_currency.php':
        case 'edit_currency.php':
        case 'view_orders_for_contact.php':
        case 'preview_form.php':
        case 'view_ship_date_adjustments.php':
        case 'add_ship_date_adjustment.php':
        case 'edit_ship_date_adjustment.php':
            $output_body_class = ' class="ecommerce"';
            break;

        case 'view_users.php':
        case 'add_user.php':
        case 'edit_user.php':
        case 'import_users.php':
            $output_body_class = ' class="users"';
            break;
        
        case 'view_visitor_reports.php':
        case 'view_visitor_report.php':
        case 'view_visitor.php':
            $output_body_class = ' class="statistics"';
            break;
            
        case 'view_ads.php':
        case 'add_ad.php':
        case 'edit_ad.php':
            $output_body_class = ' class="ads"';
            break;
        
        case 'view_design.php':
        case 'view_styles.php':
        case 'add_style.php':
        case 'add_system_style.php':
        case 'edit_system_style.php':
        case 'view_system_style_source.php':
        case 'add_custom_style.php':
        case 'edit_custom_style.php':
        case 'add_menu.php':
        case 'edit_menu.php':
        case 'view_menus.php':
        case 'view_regions.php':
        case 'view_themes.php':
        case 'add_theme_file.php':
        case 'edit_theme_file.php':
        case 'edit_javascript.php':
        case 'edit_theme_css.php':
        case 'theme_designer.php':
        case 'add_common_region.php':
        case 'add_dynamic_region.php':
        case 'add_login_region.php':
        case 'edit_common_region.php':
        case 'edit_designer_region.php':
        case 'add_designer_region.php':
        case 'edit_dynamic_region.php':
        case 'edit_login_region.php':
        case 'view_menu_items.php':
        case 'add_menu_item.php':
        case 'edit_menu_item.php':
        case 'view_design_files.php':
        case 'add_design_file.php':
        case 'edit_design_file.php':
        case 'add_ad_region.php':
        case 'edit_ad_region.php':
        case 'import_design.php':
        case 'import_zip.php':
        case 'migration.php':
        case 'find_and_replace.php':
        case 'generate_layout.php':
            $output_body_class = ' class="design"';
            break;
            
        case 'welcome.php':
        case 'view_log.php':
            $output_body_class = ' class="welcome"';
            break;
    }
    
    $output_parent_target = '';
    
    // if the header is being outputted for the toolbar, then prepare to output parent target
    if ($toolbar == true) {
        $output_parent_target = ' target="_parent"';
    }
    
    $output_software_update_available = '';
    
    // if software update check is enabled and there is a software update available, then output notice
    if (
        (
            (defined('SOFTWARE_UPDATE_CHECK') == FALSE)
            || (SOFTWARE_UPDATE_CHECK == TRUE)
        )
        && (SOFTWARE_UPDATE_AVAILABLE == TRUE)
    ) {
        $output_software_update_available_start = '';
        $output_software_update_available_end = '';
        
        // if the site is not private labeled, then prepare to add link
        if (PRIVATE_LABEL == FALSE) {
            $output_software_update_available_start = '<a href="https://livesite.com/software-update-available" target="_blank" class="notice">';
            $output_software_update_available_end = '</a>';
            
        // else the site is private labeled, so don't add link, just use span tags
        } else {
            $output_software_update_available_start = '<span class="notice">';
            $output_software_update_available_end = '</span>';
        }
        
        $output_software_update_available = $output_software_update_available_start . 'Software Update Available' . $output_software_update_available_end;
    }
    
    $output_site_settings = '';
    
    // if the user is at least a manager then output site settings
    if ($user['role'] < 3) {
        $output_site_settings = '<a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/settings.php"' . $output_parent_target . '>Site Settings</a>&nbsp;&nbsp;&nbsp;&nbsp;';
    }
    
    $output_start_page_link = '';
    
    // if the user has a start page, then prepare to output start page link
    if ($user['home'] != 0) {
        // get start page name
        $start_page_name = get_page_name($user['home']);
        
        // if a start page name was found, then the page still exists, so prepare to output start page link
        if ($start_page_name != '') {
            $output_start_page_link = '<a href="' . OUTPUT_PATH . h($start_page_name) .'"' . $output_parent_target . '>My Start Page</a>&nbsp;&nbsp;&nbsp;';
        }
    }
    
    $output_home_page_link_url = '';
    
    // get home page(s)
    $query = "SELECT page_name FROM page WHERE page_home = 'yes'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if there is one home page, then prepare URL with page name
    if (mysqli_num_rows($result) == 1) {
        $row = mysqli_fetch_assoc($result);
        $home_page_name = $row['page_name'];
        
        $output_home_page_link_url = OUTPUT_PATH . h(encode_url_path($home_page_name));
        
    // else there is not a home page or there are multiple home pages, so prepare general home URL
    // so that they receive an error for no home page or receive a random home page if there are multiple
    } else {
        $output_home_page_link_url = OUTPUT_PATH;
    }
    
    $output_menu = '';

    // if user has access to edit content
    if (($user['role'] < 3) || (no_acl_check($user['id']) == TRUE)) {
        $output_menu .=
            '<li id="menu_folders"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_folders.php"' . $output_parent_target . '>Folders</a></li>
            <li id="menu_pages"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_pages.php"' . $output_parent_target . '>Pages</a></li>
            <li id="menu_files"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_files.php"' . $output_parent_target . '>Files</a></li>';
    }
    
    // if calendars module is enabled and user has access to manage calendars, show calendars button
    if ((CALENDARS === true) && (($user['role'] < 3) || ($user['manage_calendars'] == true))) {
        $output_menu .= '<li id="menu_calendars"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_calendars.php"' . $output_parent_target . '>Calendars</a></li>';
    }
    
    // if forms module is enabled and user has access to manage forms, show forms button
    if ((FORMS === true) && (($user['role'] < 3) || ($user['manage_forms'] == true))) {
        $output_menu .= '<li id="menu_forms"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_submitted_forms.php"' . $output_parent_target . '>Forms</a></li>';
    }
    
    // if user is manager or above or has access, show visitors
    if (($user['role'] < 3) || ($user['manage_visitors'] == true)) {
        $output_menu .=
            '<li id="menu_statistics"><a href="view_visitor_reports.php"' . $output_parent_target . '>Visitors</a></li>';
    }
    
    // if user has access to contacts, show contacts menu button
    if (($user['role'] < 3) || ($user['manage_contacts'] == true)) {
        $output_menu .= '<li id="menu_contacts"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_contacts.php"' . $output_parent_target . '>Contacts</a></li>';
    }
    
    // if user is manager or above, show users
    if ($user['role'] < 3) {
        $output_menu .=
            '<li id="menu_users"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_users.php"' . $output_parent_target . '>Users</a></li>';
    }
    
    // if the user's role is Manager or above OR if the user has rights to manage e-mails, show e-mail menu button
    if (($user['role'] < 3) || ($user['manage_emails'] == true)) {
        $output_menu .= '<li id="menu_campaigns"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_email_campaigns.php"' . $output_parent_target . '>Campaigns</a></li>';
    }
    
    // If ecommerce is enabled and user has access to manage ecommerce or
    // ecommerce reports, then show commerce button.
    if (
        (ECOMMERCE === true)
        and
        (
            ($user['role'] < 3)
            or USER_MANAGE_ECOMMERCE
            or USER_MANAGE_ECOMMERCE_REPORTS
        )
    ) {

        // If this user has access to manage all commerce, then link to orders screen.
        if (($user['role'] < 3) or USER_MANAGE_ECOMMERCE) {
            $commerce_url = 'view_orders.php';

        // Otherwise this user only has access to commerce reports, so link to
        // order reports.
        } else {
            $commerce_url = 'view_order_reports.php';
        }


        $output_menu .=
            '<li id="menu_ecommerce">
                <a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/' . $commerce_url . '" style="white-space: nowrap"' . $output_parent_target . '>Commerce</a>
            </li>';

    }
    
    // If ads are enabled and the user is at least a manager
    // or if they have access to edit any ad regions, then show ads.
    if (
        (ADS === true)
        &&
        (
            ($user['role'] < 3)
            || (count(get_items_user_can_edit('ad_regions', $user['id'])) > 0)
        )
    ) {
        $output_menu .= '<li id="menu_ads"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_ads.php"' . $output_parent_target . '>Ads</a></li>';
    }
    
    // if user has access to manage design elements, show design menu
    if ($user['role'] <= 1) {
        $output_menu .= '<li id="menu_design"><a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_styles.php"' . $output_parent_target . '>Design</a></li>';
    }
    
    if (PRIVATE_LABEL == FALSE) {
        $software_title = 'liveSite - ';
    }

    $output_javascript = '';

    // If the message check is enabled and this site is not private labeled
    // and the user is currently viewing a back-end screen
    // (i.e. this is not the header being loaded for the toolbar),
    // then determine if we should show a message to the user.
    if (
        (
            (defined('CHECK') == false)
            || (CHECK == true)
        )
        && (PRIVATE_LABEL == false)
        && ($toolbar == false)
    ) {
        // Get a message that is appropriate to show the user now.
        // Either a message that has not been shown to the user before,
        // or a recurring message where the frequency value has passed
        // since the last time the message was shown to this user.
        $message = db_item(
            "SELECT
                messages.id,
                messages.title,
                messages.url,
                messages.width,
                messages.height,
                users_messages_xref.timestamp
            FROM messages
            LEFT JOIN users_messages_xref ON
                (messages.id = users_messages_xref.message_id)
                AND (users_messages_xref.user_id = '" . USER_ID . "')
            WHERE
                (users_messages_xref.message_id IS NULL)
                OR
                (
                    (messages.frequency > 0)
                    AND (((UNIX_TIMESTAMP() - users_messages_xref.timestamp) / 60 / 60) >= messages.frequency)
                )
            ORDER BY messages.id
            LIMIT 1");

        // If a message was found, then show message to user.
        if ($message) {
            $output_dialog_properties = encode_json(array(
                'title' => $message['title'],
                'url' => $message['url'],
                'width' => intval($message['width']),
                'height' => intval($message['height'])
            ));

            $output_javascript .=
                '
                $(document).ready(function() {
                    open_dialog(' . $output_dialog_properties . ');
                });';
            
            // If this message has been shown to this user before,
            // then update the timestamp for that existing record.
            if ($message['timestamp']) {
                db(
                    "UPDATE users_messages_xref
                    SET timestamp = UNIX_TIMESTAMP()
                    WHERE
                        (user_id = '" . USER_ID . "')
                        AND (message_id = '" . e($message['id']) . "')");

            // Otherwise this is the first time that this message
            // has been shown to this user, so create new record with timestamp.
            } else {
                db(
                    "INSERT INTO users_messages_xref (
                        user_id,
                        message_id,
                        timestamp)
                    VALUES (
                        '" . USER_ID . "',
                        '" . e($message['id']) . "',
                        UNIX_TIMESTAMP())");
            }
        }
    }
    
    // if the path is more than just "/", then add the path to the home title
    if (PATH != '/') {
        $output_home_title = h(HOSTNAME) . OUTPUT_PATH;
        
    // else the path is just "/", so set the home title to just the hostname
    } else {
        $output_home_title = h(HOSTNAME);
    }
    
    return
        '<!DOCTYPE html>
        <html lang="en">
            <head>
                <meta charset="utf-8">
                <title>' . $software_title . h($_SERVER['HTTP_HOST']) . '</title>
                ' . get_generator_meta_tag() . '
                ' . output_control_panel_header_includes() . '
                <script type="text/javascript">
                    ' . $output_javascript . '
                </script>
            </head>
            <body' . $output_body_class . '>
                <div id="header">
                    <table width="100%" cellspacing="0" cellpadding="0" border="0">
                        <tr>
                            <td rowspan="2" id="logo">
                                <a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/welcome.php"' . $output_parent_target . '><img src="' . LOGO_URL . '" width="150" border="0" alt="logo" title="" /></a><br />
                            </td>
                            <td>
                                <div id="top_line">
                                    <table style="width: 100%; text-align: right" cellspacing="0" cellpadding="0" border="0">
                                        <tr>
                                            <td><div id="notification">' . $output_software_update_available . '</div>' . $output_site_settings . 'Hi, <a href="' . h(get_page_type_url('my account')) . '"' . $output_parent_target . '>' . h($_SESSION['sessionusername']) . '</a>&nbsp;|&nbsp;<a href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/logout.php"' . $output_parent_target . '>Logout</a>&nbsp;&nbsp;&nbsp;&nbsp;' . $output_start_page_link . '<a href="' . $output_home_page_link_url . '"' . $output_parent_target . '><img id="home_icon" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/icon_home_white_small.png" width="16" height="14" alt="Home" title="Home (' . $output_home_title . ')" border="0" /></a></td>
                                        </tr>
                                    </table>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <td style="vertical-align: bottom; width: 100%">
                                <div id="menu"><ul>' . $output_menu . '</ul></div>
                            </td>
                        </tr>
                    </table>
                </div>
                ';
}

// this function outputs the includes for the files needed for the control panel
function output_control_panel_header_includes() {

    $output_view_order_print_stylesheet = '';
    
    // if this is the view order screen, then also output stylesheet for printing
    if ($_SERVER['PHP_SELF'] == PATH . SOFTWARE_DIRECTORY . '/view_order.php') {
        $output_view_order_print_stylesheet = '<link rel="stylesheet" type="text/css" href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_order_print.css" media="print" />';
    }

    $help_url = '';

    // If the site is not private labeled, then get help URL.  We hide the help button if site
    // is private labeled, so no reason to get help url.
    if (!PRIVATE_LABEL) {
        require(dirname(__FILE__) . '/get_help_url.php');
        $help_url = get_help_url();
    }

    // if CDN is enabled, then use Google CDN for jQuery for performance reasons
    if (
        (defined('CDN') == FALSE)
        || (CDN == TRUE)
    ) {
        $output_jquery =
            '<script src="//ajax.googleapis.com/ajax/libs/jquery/1.7.2/jquery.min.js"></script>
            <script src="//ajax.googleapis.com/ajax/libs/jqueryui/1.8.21/jquery-ui.min.js"></script>';

    // else CDN is disabled, so use local jQuery
    } else {
        $output_jquery =
            '<script src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/jquery/jquery-1.7.2.min.js"></script>
            <script src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/jquery/jquery-ui-1.8.21.min.js"></script>';
    }
    
    return
        '<link rel="stylesheet" type="text/css" href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/jquery/theme/standard.' . ENVIRONMENT_SUFFIX . '.css" />
        <link rel="stylesheet" type="text/css" href="' . CONTROL_PANEL_STYLESHEET_URL . '" />
        ' . $output_view_order_print_stylesheet . '
        <script type="text/javascript">
            var path = "' . escape_javascript(PATH) . '";
            var software_directory = "' . escape_javascript(SOFTWARE_DIRECTORY) . '";
            var software_token = "' . $_SESSION['software']['token'] . '";
            var help_url = "' . $help_url . '";
        </script>
        ' . $output_jquery . '
        <script type="text/javascript" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/backend.' . ENVIRONMENT_SUFFIX . '.js?v=' . @filemtime(dirname(__FILE__) . '/backend.' . ENVIRONMENT_SUFFIX . '.js') . '"></script>';
}

function output_footer()
{
    // if private label is disabled, then show Camelback footer
    if (PRIVATE_LABEL == FALSE) {
        $output_footer_table =
            '<table>
                <tr>
                    <td class="logo"><a href="https://livesite.com" target="_blank"><img src="images/logo_footer.png" width="150" height="50" alt="Camelback Web Architects" title="" /></a></td>
                    <td class="slogan">&nbsp;&nbsp;&nbsp;simply innovative.</td>
                    <td class="links">&nbsp;</td>
                    <td class="version_and_copyright">
                        v' . VERSION . '&nbsp;' . EDITION . '&nbsp;&nbsp;&copy;&nbsp;' . date('Y', time()) . '
                    </td>
                </tr>
            </table>';
        
    // else if private label is enabled, then show private labeled footer
    } else {
        $output_footer_logo = '';
        
        // if the footer logo is set, then prepare to output footer logo
        if ((defined('FOOTER_LOGO_URL') == true) && (FOOTER_LOGO_URL != '')) {
            $output_link_start = '';
            $output_link_end = '';
            
            // if there is a footer logo link url, then prepare to output link
            if ((defined('FOOTER_LOGO_LINK_URL') == true) && (FOOTER_LOGO_LINK_URL != '')) {
                $output_link_start = '<a href="' . h(FOOTER_LOGO_LINK_URL) . '" target="_blank">';
                $output_link_end = '</a>';
            }
            
            $output_footer_logo = $output_link_start . '<img src="' . h(FOOTER_LOGO_URL) . '" width="150" alt="logo" title="" />' . $output_link_end;
        }
        
        $output_footer_link_1 = '';
        
        // if the footer link 1 label is set, then prepare to output it
        if ((defined('FOOTER_LINK_1_LABEL') == true) && (FOOTER_LINK_1_LABEL != '')) {
            $output_link_start = '';
            $output_link_end = '';
            
            // if there is a footer link 1 url, then prepare to output it
            if ((defined('FOOTER_LINK_1_URL') == true) && (FOOTER_LINK_1_URL != '')) {
                $output_link_start = '<a href="' . h(FOOTER_LINK_1_URL) . '" target="_blank">';
                $output_link_end = '</a>';
            }
            
            $output_footer_link_1 = $output_link_start . h(FOOTER_LINK_1_LABEL) . $output_link_end;
        }
        
        $output_footer_link_2 = '';
        
        // if the footer link 2 label is set, then prepare to output it
        if ((defined('FOOTER_LINK_2_LABEL') == true) && (FOOTER_LINK_2_LABEL != '')) {
            $output_link_start = '';
            $output_link_end = '';
            
            // if there is a footer link 2 url, then prepare to output it
            if ((defined('FOOTER_LINK_2_URL') == true) && (FOOTER_LINK_2_URL != '')) {
                $output_link_start = '<a href="' . h(FOOTER_LINK_2_URL) . '" target="_blank">';
                $output_link_end = '</a>';
            }
            
            $output_footer_link_2 = $output_link_start . h(FOOTER_LINK_2_LABEL) . $output_link_end;
        }
        
        $output_footer_link_3 = '';
        
        // if the footer link 3 label is set, then prepare to output it
        if ((defined('FOOTER_LINK_3_LABEL') == true) && (FOOTER_LINK_3_LABEL != '')) {
            $output_link_start = '';
            $output_link_end = '';
            
            // if there is a footer link 3 url, then prepare to output it
            if ((defined('FOOTER_LINK_3_URL') == true) && (FOOTER_LINK_3_URL != '')) {
                $output_link_start = '<a href="' . h(FOOTER_LINK_3_URL) . '" target="_blank">';
                $output_link_end = '</a>';
            }
            
            $output_footer_link_3 = $output_link_start . h(FOOTER_LINK_3_LABEL) . $output_link_end;
        }
        
        $output_footer_link_4 = '';
        
        // if the footer link 4 label is set, then prepare to output it
        if ((defined('FOOTER_LINK_4_LABEL') == true) && (FOOTER_LINK_4_LABEL != '')) {
            $output_link_start = '';
            $output_link_end = '';
            
            // if there is a footer link 4 url, then prepare to output it
            if ((defined('FOOTER_LINK_4_URL') == true) && (FOOTER_LINK_4_URL != '')) {
                $output_link_start = '<a href="' . h(FOOTER_LINK_4_URL) . '" target="_blank">';
                $output_link_end = '</a>';
            }
            
            $output_footer_link_4 = $output_link_start . h(FOOTER_LINK_4_LABEL) . $output_link_end;
        }
        
        $output_footer_link_5 = '';
        
        // if the footer link 5 label is set, then prepare to output it
        if ((defined('FOOTER_LINK_5_LABEL') == true) && (FOOTER_LINK_5_LABEL != '')) {
            $output_link_start = '';
            $output_link_end = '';
            
            // if there is a footer link 5 url, then prepare to output it
            if ((defined('FOOTER_LINK_5_URL') == true) && (FOOTER_LINK_5_URL != '')) {
                $output_link_start = '<a href="' . h(FOOTER_LINK_5_URL) . '" target="_blank">';
                $output_link_end = '</a>';
            }
            
            $output_footer_link_5 = $output_link_start . h(FOOTER_LINK_5_LABEL) . $output_link_end;
        }
        
        $output_footer_table =
            '<table>
                <tr>
                    <td class="logo" style="height: 51px;">' . $output_footer_logo . '</td>
                    <td class="links">
                        ' . $output_footer_link_1 . '
                        ' . $output_footer_link_2 . '
                        ' . $output_footer_link_3 . '
                        ' . $output_footer_link_4 . '
                        ' . $output_footer_link_5 . '
                    </td>
                    <td class="version_and_copyright">
                        v' . VERSION . '&nbsp;' . EDITION . '&nbsp;&nbsp;&copy;&nbsp;' . date('Y', time()) . '
                    </td>
                </tr>
            </table>';
    }
    
    return
                '<div style="padding: 1em 0 1em 0" id="footer">
                    ' . $output_footer_table . '
                </div>
            </body>
        </html>';
}

function output_header_secure()
{
    // if CDN is enabled, then use Google CDN for jQuery for performance reasons
    if (
        (defined('CDN') == FALSE)
        || (CDN == TRUE)
    ) {
        $output_jquery =
            '<script src="//ajax.googleapis.com/ajax/libs/jquery/1.7.2/jquery.min.js"></script>
            <script src="//ajax.googleapis.com/ajax/libs/jqueryui/1.8.21/jquery-ui.min.js"></script>';

    // else CDN is disabled, so use local jQuery
    } else {
        $output_jquery =
            '<script src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/jquery/jquery-1.7.2.min.js"></script>
            <script src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/jquery/jquery-ui-1.8.21.min.js"></script>';
    }
    
    return
        '<!DOCTYPE html>
        <html lang="en">
            <head>
                <meta charset="utf-8">
                <title>' . h($_SERVER['HTTP_HOST']) . '</title>
                ' . get_generator_meta_tag() . '
                <link rel="stylesheet" type="text/css" href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/jquery/theme/standard.' . ENVIRONMENT_SUFFIX . '.css" />
                <link rel="stylesheet" type="text/css" href="' . CONTROL_PANEL_STYLESHEET_URL . '" />
                <script type="text/javascript">
                    var path = "' . escape_javascript(PATH) . '";
                    var software_directory = "' . escape_javascript(SOFTWARE_DIRECTORY) . '";
                    var software_token = "' . $_SESSION['software']['token'] . '";
                </script>
                ' . $output_jquery . '
                <script type="text/javascript" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/backend.' . ENVIRONMENT_SUFFIX . '.js?v=' . @filemtime(dirname(__FILE__) . '/backend.' . ENVIRONMENT_SUFFIX . '.js') . '"></script>
            </head>
            <body>
                <div id="content">';
}

function output_footer_secure()
{
    return
                '</div>
            </body>
        </html>';
}

// determine what access control type a folder has
function get_access_control_type($folder_id)
{
    $query = "SELECT folder_parent, folder_access_control_type FROM folder WHERE folder_id = '" . escape($folder_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');
    $row = mysqli_fetch_assoc($result);

    if ($row['folder_access_control_type']) {
        return $row['folder_access_control_type'];
    }

    // if this folder is the root folder, return public
    if ($row['folder_parent'] == 0) {
        return 'public';
    // else this folder is not the root folder, so use recursion to find access control of folder parent
    } else {
        return get_access_control_type($row['folder_parent']);
    }
}

function get_access_control_type_name($access_control_type)
{
    switch ($access_control_type) {
        case 'public':
            return 'Public';
            break;
            
        case 'guest':
            return 'Guest';
            break;
        
        case 'private':
            return 'Private';
            break;
        
        case 'registration':
            return 'Registration';
            break;
        
        case 'membership':
            return 'Membership';
            break;
    }
}

// You can pass the access control type in order to limit the folders to those with that type of access control.
function select_folder($folder_id = 0, $parent_folder_id = 0, $excluded_folder_id = 0, $level = 0, $folders = array(), $folders_that_user_has_access_to = array(), $access_control_type = '')
{
    global $user;
    
    $output = '';
    
    // if this is the first time this function has run, then get folders and folders that user has has access to
    if ($parent_folder_id == 0) {
        // get all folders
        $query =
            "SELECT
                folder_id as id,
                folder_name as name,
                folder_parent as parent_folder_id,
                folder_archived as archived
            FROM folder
            ORDER BY folder_level, folder_order, folder_name";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

        while ($row = mysqli_fetch_assoc($result)) {
            $folders[] = $row;
        }
        
        // if user is a basic user, then get folders that user has access to
        if ($user['role'] == 3) {
            $folders_that_user_has_access_to = get_folders_that_user_has_access_to($user['id']);
        }
    }
    
    $child_folders = array();
    
    // loop through folders array in order to get all folders that are in parent folder
    foreach ($folders as $folder) {
        // if the parent folder id for this folder is equal to the parent folder id, then this is a child folder, so add to array
        if ($folder['parent_folder_id'] == $parent_folder_id) {
            $child_folders[] = $folder;
        }
    }
    
    // loop through child folders
    foreach ($child_folders as $folder) {
        // if folder id is not equal to excluded folder id, then continue to prepare option and get child folders
        if ($folder['id'] != $excluded_folder_id) {
            // If user has access to folder, or if the folder is the selected folder,
            // and the access control type is valid, then output option for folder.
            if (
                (
                    (check_folder_access_in_array($folder['id'], $folders_that_user_has_access_to) == true)
                    || ($folder['id'] == $folder_id)
                )
                &&
                (
                    ($access_control_type == '')
                    || ($access_control_type == get_access_control_type($folder['id']))
                )
            ) {
                // prepare indentation
                $indentation = '';
                
                for ($i = 1; $i <= $level; $i++)
                {
                    $indentation .= '&nbsp;&nbsp;&nbsp;&nbsp;';
                }
                
                $next_level = $level + 1;

                // if this folder is the selected folder, then this option should be selected
                if ($folder['id'] == $folder_id) {
                    $selected = ' selected';
                } else {
                    $selected = '';
                }
                
                $archived_label = '';
                
                // if this folder is archived, then output archived beside the folder name
                if ($folder['archived'] == '1') {
                    $archived_label = ' [ARCHIVED]';
                }
                
                // output option row
                $output .= '<option value="' . $folder['id'] . '"' . $selected . '>' . $indentation . h($folder['name'] . $archived_label) . '</option>';
            }
            
            // get options for child folders
            $output .= select_folder($folder_id, $folder['id'], $excluded_folder_id, $next_level, $folders, $folders_that_user_has_access_to, $access_control_type);
        }
    }
    
    return $output;
}

// output selection field for selecting access control for a folder
function select_access_control_type($access_control_type = '', $default = true)
{
    $output = '';

    if ($default == true) {
        $output .= '<option value="">Default (inherit)</option>';
    }

    // let's list access control types
    $access_control_types[] = 'Public';
    $access_control_types[] = 'Guest';
    $access_control_types[] = 'Registration';
    $access_control_types[] = 'Membership';
    $access_control_types[] = 'Private';

    foreach ($access_control_types as $value) {
        // if this access control type is the selected access control type, select this option
        if ((mb_strtolower($value) == mb_strtolower($access_control_type))) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output .= '<option value="' . mb_strtolower($value) . '"' . $selected . ' class="' . mb_strtolower($value) . '">' . $value . '</option>';
    }

    return $output;
}

// outputs the form's <select> style list for selecting a style
// this function will only get non-mobile styles
function select_style($style = '', $default_label = 'Default (inherit)')
{
    $output .= '<option value="0">' . h($default_label) . '</option>';

    $result = mysqli_query(db::$con, "SELECT style_id, style_name FROM style WHERE style_layout != 'one_column_mobile' ORDER BY style_name") or output_error('Query failed');
    while ($row = mysqli_fetch_assoc($result)) {
        // find if style is selected
        if ($row['style_id'] == $style) {
            $selected = ' selected';
        } else {
            $selected = '';
        }
        // output option row
        $output .= '<option value="' . $row['style_id']. '"' .$selected. '>' . $row['style_name'] . '</option>';
    }
    return $output;
}

// create function that will get options for the mobile style pick list
function get_mobile_style_options($mobile_style_id = '', $default_label = 'Default (inherit)')
{
    $output = '<option value="0">' . h($default_label) . '</option>';

    $sql_mobile_style_filter = "";

    // If mobile is enabled in the site settings,
    // then only get styles that can be mobile styles.
    if (MOBILE == true) {
        $sql_mobile_style_filter = "WHERE (style_type = 'custom') OR (style_layout = 'one_column_mobile')";
    }

    // get all styles that might be mobile styles
    $query =
        "SELECT
            style_id AS id,
            style_name AS name
        FROM style
        $sql_mobile_style_filter
        ORDER BY style_name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $mobile_styles = mysqli_fetch_items($result);

    // loop through the mobile styles in order to prepare options
    foreach ($mobile_styles as $mobile_style) {
        // if this mobile style is the selected mobile style, then select it
        if ($mobile_style['id'] == $mobile_style_id) {
            $selected = ' selected="selected"';

        // else this mobile style is not the selected mobile style, so do not select it
        } else {
            $selected = '';
        }

        $output .= '<option value="' . $mobile_style['id'] . '"' . $selected . '>' . h($mobile_style['name']) . '</option>';
    }
    
    return $output;
}

// returns content for column header depending on asc or desc
function asc_or_desc($column, $type, $extras = '')
{
    if($_GET['sort'] == $column) {
        if($_GET['order'] == 'desc') {
            return("<a href=\"$type.php?sort=$column&order=asc$extras\" class=\"link-top-row\">$column</a> <img src=\"images/icon_desc.gif\" width=\"8\" height=\"7\">");
        } else {
            return("<a href=\"$type.php?sort=$column&order=desc$extras\" class=\"link-top-row\">$column</a> <img src=\"images/icon_asc.gif\" width=\"8\" height=\"7\">");
        }
    } else {
        return("<a href=\"$type.php?sort=$column$extras\" class=\"link-top-row\">$column</a>");
    }
}

// return content for column heading
function get_column_heading($column, $sort, $order, $extras = '')
{
    if ($column == $sort) {
        if ($order == 'asc') {
            return('<a href="' . h($_SERVER["PHP_SELF"]) . '?sort=' . h(urlencode($column)) . '&order=desc' . $extras . '" class="link-top-row">' . h($column) . '</a> <img src="images/icon_asc.gif" width="8" height="7" />');
        } else {
            return('<a href="' . h($_SERVER["PHP_SELF"]) . '?sort=' . h(urlencode($column)) . '&order=asc' . $extras . '" class="link-top-row">' . h($column) . '</a> <img src="images/icon_desc.gif" width="8" height="7" />');
        }
    } else {
        return('<a href="' . h($_SERVER["PHP_SELF"]) . '?sort=' . h(urlencode($column)) . '&order=asc' . $extras . '" class="link-top-row">' . h($column) . '</a>');
    }
}

function get_acl_folder_tree($type, $parent_folder_id = 0, $level = 0, $folders = array(), $folders_that_user_has_access_to = array(), $user_id = '')
{
    $output = '';
    
    // if this is the first time this function has run, then get folders
    if ($parent_folder_id == 0) {
        // get all folders
        $query =
            "SELECT
                folder_id as id,
                folder_name as name,
                folder_parent as parent_folder_id
            FROM folder
            ORDER BY folder_level, folder_order, folder_name";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

        while ($row = mysqli_fetch_assoc($result)) {
            $folders[] = $row;
        }
    }
    
    $child_folders = array();
    
    // loop through folders array in order to get all folders that are in parent folder
    foreach ($folders as $folder) {
        // if the parent folder id for this folder is equal to the parent folder id, then this is a child folder, so add to array
        if ($folder['parent_folder_id'] == $parent_folder_id) {
            $child_folders[] = $folder;
        }
    }
    
    // loop through child folders
    foreach ($child_folders as $folder) {
        // prepare indentation
        $indentation = '';
        
        for ($i = 1; $i <= $level; $i++) {
            $indentation .= '&nbsp;&nbsp;&nbsp;&nbsp;';
        }
        
        $next_level = $level + 1;
        
        // if type is edit or type is view and folder is private, then output line for this folder
        if (($type == 'edit') || (($type == 'view') && (get_access_control_type($folder['id']) == 'private'))) {
            $checked = '';
            
            // if the user has access to this folder, then prepare to check check box
            if (in_array($folder['id'], $folders_that_user_has_access_to) == true) {
                $checked = ' checked="checked"';
            }

            $output_onclick = '';
            $output_expiration_date_container = '';

            // If the type if view, then output expiration date field.
            if ($type == 'view') {
                $output_onclick = ' onclick="show_or_hide_view_expiration_date(' . $folder['id'] . ')"';

                // Assume that the expiration date field should not be shown until we find out otherwise.
                $output_expiration_date_container_style = ' style="display: none"';
                $output_expiration_date = '';
                $output_expiration_date_style = '';
                $output_expiration_date_picker = '';

                // If the check box is checked, then show expiration date field and get expiration date if one exists.
                if ($checked != '') {
                    $output_expiration_date_container_style = '';

                    // Get expiration date if one exists.
                    $expiration_date = db_value(
                        "SELECT expiration_date
                        FROM aclfolder
                        WHERE
                            (aclfolder_user = '" . escape($user_id) . "')
                            AND (aclfolder_folder = '" . $folder['id'] . "')
                            AND (aclfolder_rights = '1')");

                    // If an expiration date was found, then prepare to prefill field with value.
                    if ($expiration_date != '0000-00-00') {
                        $output_expiration_date = prepare_form_data_for_output($expiration_date, 'date');

                        // If the expiration date has expired, then make expiration date red.
                        if ($expiration_date < date('Y-m-d')) {
                            $output_expiration_date_style = '; color: red';
                        }
                    }

                    $output_expiration_date_picker =
                        '<script>
                            $("#view_' . $folder['id'] . '_expiration_date").datepicker({
                                dateFormat: date_picker_format
                            });
                        </script>';
                }

                $output_expiration_date_container = '<span id="view_' . $folder['id'] . '_expiration_date_container"' . $output_expiration_date_container_style . '>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Expiration Date: <input type="text" id="view_' . $folder['id'] . '_expiration_date" name="view_' . $folder['id'] . '_expiration_date" value="' . $output_expiration_date . '" size="10" maxlength="10" style="margin: .1em 0' . $output_expiration_date_style . '" onfocus="this.style.color = \'\'" />' . $output_expiration_date_picker . '</span>';
            }
            
            $output .= '<div style="white-space: nowrap"><input type="checkbox" name="' . $type . '_' . $folder['id'] . '" id="' . $type . '_' . $folder['id'] . '" value="1" class="checkbox"' . $checked . $output_onclick . ' /><label for="' . $type . '_' . $folder['id'] . '"> ' . $indentation . h($folder['name']) . '</label>' . $output_expiration_date_container . '</div>';
        }
        
        // get lines for child folders
        $output .= get_acl_folder_tree($type, $folder['id'], $next_level, $folders, $folders_that_user_has_access_to, $user_id);
    }
    
    return $output;
}

// find if there is an entry for the user in the access control list to edit any folder
function no_acl_check($user_id)
{
    $result=mysqli_query(db::$con, "SELECT aclfolder_user FROM aclfolder WHERE aclfolder_user = '" . escape($user_id) . "' AND aclfolder_rights = '2'") or output_error('Query failed');
    if(mysqli_num_rows($result)) {
        return TRUE;
    } else {
        return FALSE;
    }
}

// Gets the activated style for a particular folder and device type.
// This is used in order to figure out the style for a page, when a page does not
// have a style set for it.  This function uses recursion to continue to look up
// into the tree into parent folders if a child folder does not have a style set.

function get_style($folder_id, $device_type = 'desktop') {

    switch ($device_type) {
        case 'desktop':
            $query =
                "SELECT
                    folder_parent,
                    folder_style,
                    folder_level
                FROM folder
                WHERE folder_id = '" . escape($folder_id) . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed');
            $row = mysqli_fetch_assoc($result);

            $folder_parent = $row['folder_parent'];
            $folder_style = $row['folder_style'];
            $folder_level = $row['folder_level'];

            // if there is a style for this folder, then return it
            if ($folder_style) {
                return $folder_style;

            // else if this is the root folder, then there is no where else to look, so return 0
            } else if ($folder_level == 0) {
                return 0;

            // else use recursion to check for a style in the parent folder
            } else {
                return get_style($folder_parent, $device_type);
            }

            break;
        
        case 'mobile':
            $query =
                "SELECT
                    folder_parent,
                    mobile_style_id,
                    folder_level
                FROM folder
                WHERE folder_id = '" . escape($folder_id) . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed');
            $row = mysqli_fetch_assoc($result);

            $folder_parent = $row['folder_parent'];
            $mobile_style_id = $row['mobile_style_id'];
            $folder_level = $row['folder_level'];

            // if there is a mobile style for this folder, then return it
            if ($mobile_style_id != 0) {
                return $mobile_style_id;

            // else if this is the root folder, then there is no where else to look, so return 0
            } else if ($folder_level == 0) {
                return 0;

            // else use recursion to check for a style in the parent folder
            } else {
                return get_style($folder_parent, $device_type);
            }

            break;
    }

}

// Get the style that should be shown for a particular page.  If the visitor is
// a designer and is previewing styles, then this function will return the style
// that is currently being previewed.  Otherwise, if the visitor is a common
// website visitor, then it will return the activated style for the page.

function get_preview_style($properties) {

    $page_id = $properties['page_id'];
    $folder_id = $properties['folder_id'];
    $page_style_id = $properties['page_style_id'];
    $page_mobile_style_id = $properties['page_mobile_style_id'];
    $device_type = $properties['device_type'];

    $style_id = '';

    // If theme/style preview is enabled, and this page is not being loaded in the
    // theme designer, or if the theme that is being edited in the theme designer
    // is the same as the theme that is being previewed, then check if we should use
    // a preview style instead of the activated style.
    if (
        (isset($_SESSION['software']['preview_theme_id']) == true)
        &&
        (
            ($_GET['edit_theme'] != 'true')
            || ($_GET['theme_id'] == $_SESSION['software']['preview_theme_id'])
        )
    ) {
        // If the selected theme is the activated theme, then get style in a certain way.
        if (
            ($_SESSION['software']['preview_theme_id'])
            && ($_SESSION['software']['preview_theme_id'] == db_value("SELECT id FROM files WHERE activated_" . $device_type . "_theme = '1'"))
        ) {
            // If a style has been selected, then use that style.
            if (isset($_SESSION['software']['preview_style']['theme_' . $_SESSION['software']['preview_theme_id'] . '_page_' . $page_id . '_' . $device_type]) == true) {
                // If the selected style is blank, then that means,
                // the default option has been selected, so get style from folders.
                if (!$_SESSION['software']['preview_style']['theme_' . $_SESSION['software']['preview_theme_id'] . '_page_' . $page_id . '_' . $device_type]) {
                    $style_id = get_style($folder_id, $device_type);

                    // If the device type is set to mobile and a default mobile style id
                    // could not be found, then get desktop style id because we fallback
                    // to a desktop style when a mobile style cannot be found for a page
                    if (
                        ($device_type == 'mobile')
                        && ($style_id == 0)
                    ) {
                        $style_id = get_style($folder_id, 'desktop');
                    }

                // Otherwise the selected style is not blank, so use that style.
                } else {
                    $style_id = $_SESSION['software']['preview_style']['theme_' . $_SESSION['software']['preview_theme_id'] . '_page_' . $page_id . '_' . $device_type];
                }
            }

        // Otherwise the selected theme is not the activated theme,
        // so get style in a different way.
        } else {
            // If a style has not been selected yet, then check for preview style in database.
            if (isset($_SESSION['software']['preview_style']['theme_' . $_SESSION['software']['preview_theme_id'] . '_page_' . $page_id . '_' . $device_type]) == false) {
                $style_id = db_value(
                    "SELECT style_id
                    FROM preview_styles
                    WHERE
                        (page_id = '" . e($page_id) . "')
                        AND (theme_id = '" . e($_SESSION['software']['preview_theme_id']) . "')
                        AND (device_type = '" . e($device_type) . "')");

            // Otherwise, use style that has been selected.
            } else {
                $style_id = $_SESSION['software']['preview_style']['theme_' . $_SESSION['software']['preview_theme_id'] . '_page_' . $page_id . '_' . $device_type];
            }
        }
    }

    // If no style is being previewed then get activated style.
    if (!$style_id) {

        $activated_style = get_activated_style(array(
            'page_id' => $page_id,
            'folder_id' => $folder_id,
            'page_style_id' => $page_style_id,
            'page_mobile_style_id' => $page_mobile_style_id,
            'device_type' => $device_type));

        $style_id = $activated_style['id'];

        // If there was no mobile style, then the device type might have been
        // switched to desktop, so update the device type.
        $device_type = $activated_style['device_type'];

    }

    return array(
        'id' => $style_id,
        'device_type' => $device_type);

}

// Gets the activated style for a particular page and device type.
// If the page does not have style properties set on it,
// then it will check its parent folders for the style.
// This function ignores any style that might be being previewed.
// It just gets the current production style for a page.

function get_activated_style($properties) {

    $page_id = $properties['page_id'];
    $folder_id = $properties['folder_id'];
    $page_style_id = $properties['page_style_id'];
    $page_mobile_style_id = $properties['page_mobile_style_id'];
    $device_type = $properties['device_type'];

    $style_id = '';

    // get the style id differently based on the device type
    switch ($device_type) {
        case 'desktop':
        default:
            $style_id = $page_style_id;

            // if the page does not have a desktop style, then get desktop style from parent folders
            if ($style_id == 0) {
                $style_id = get_style($folder_id, 'desktop');
            }

            break;
        
        case 'mobile':
            $style_id = $page_mobile_style_id;

            // if the page does not have a mobile style, then get mobile style from parent folders
            if ($style_id == 0) {
                $style_id = get_style($folder_id, 'mobile');

                // If a mobile style could not be found, then get desktop style and switch device type to desktop,
                // so that we don't use mobile theme.
                if ($style_id == 0) {
                    $style_id = $page_style_id;

                    // if the page does not have a desktop style, then get desktop style from parent folders
                    if ($style_id == 0) {
                        $style_id = get_style($folder_id, 'desktop');
                    }

                    $device_type = 'desktop';
                }
            }

            break;
    }

    return array(
        'id' => $style_id,
        'device_type' => $device_type);
}

function get_random_string($properties = array()) {

    $type = $properties['type'];
    $length = $properties['length'];
    $characters = $properties['characters'];

    if (!$length) {
        $length = 10;
    }

    if (!$characters) {

        switch ($type) {

            case 'letters_and_numbers':
            default:
                $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
                break;

            case 'lowercase_letters_and_numbers':
                $characters = 'abcdefghijklmnopqrstuvwxyz0123456789';
                break;

            case 'uppercase_letters_and_numbers':
                $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
                break;

            case 'lowercase_letters':
                $characters = 'abcdefghijklmnopqrstuvwxyz';
                break;

            case 'uppercase_letters':
                $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
                break;

            case 'numbers':
                $characters = '0123456789';
                break;
        }
    }    

    $string = '';

    $max = mb_strlen($characters, '8bit') - 1;

    for ($i = 0; $i < $length; $i++) {

        // If random_int() is available (PHP 7+), then use it because it is more random than mt_rand().
        if (function_exists('random_int')) {
            $random_number = random_int(0, $max);
            
        } else {
            $random_number = mt_rand(0, $max);
        }

        $string .= $characters[$random_number];
    }

    return $string;
}

// outputs the form's <select> page list for selecting a page (value is page id)
function select_page($page_id = '', $page_type = '')
{
    global $user;
    
    $folders_that_user_has_access_to = array();
    
    // if user is a basic user, then get folders that user has access to
    if ($user['role'] == 3) {
        $folders_that_user_has_access_to = get_folders_that_user_has_access_to($user['id']);
    }
    
    // if a page type was given, prepare to only return pages with that page type
    if ($page_type) {

        // If multiple page types were passed in an array, then prepare filter for all of them.
        if (is_array($page_type)) {

            $where .= "WHERE (";

            foreach ($page_type as $key => $type) {

                if ($key != 0) {
                    $where .= "OR ";
                }

                $where .= "(page.page_type = '" . e($type) . "') ";
            }

            $where .= ")";

        // Otherwise only one page type was passed, so prepare single filter.
        } else {
            $where = "WHERE page.page_type = '" . e($page_type) . "'";
        }

    } else {
        $where = '';
    }
    
    // if there is not already a where statement, then output the starting where part
    if ($where == '') {
        $where .= 'WHERE ';
    
    // else there is already a where statement, so add an and
    } else {
        $where .= ' AND ';
    }
    
    $where .= '(folder.folder_archived = "0")';
    
    // get pages
    $query =
        "SELECT
            page.page_id as id,
            page.page_name as name,
            page.page_folder as folder_id
        FROM page
        LEFT JOIN folder ON page.page_folder = folder.folder_id
        $where
        ORDER BY page.page_name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $pages = array();
    
    // loop through all pages so they can be added to array
    while ($row = mysqli_fetch_assoc($result)) {
        $pages[] = $row;
    }
    
    // loop through all pages so their options can be outputted
    foreach ($pages as $page) {
        // if user has access to folder that page is in, then output option
        if (check_folder_access_in_array($page['folder_id'], $folders_that_user_has_access_to) == true) {
            // if this page should be selected, then select it
            if ($page['id'] == $page_id) {
                $selected = ' selected="selected"';
            } else {
                $selected = '';
            }
            
            // output option
            $output .= '<option value="' . $page['id'] . '"' . $selected . '>' . h($page['name']) . '</option>';
        }
    }
    
    return $output;
}

// outputs the form's <select> page list for selecting a custom form
function select_custom_form($page_id = '', $user = '')
{
    // get pages
    $query = "SELECT
                page.page_id,
                page.page_name,
                page.page_folder,
                custom_form_pages.form_name
             FROM page
             LEFT JOIN custom_form_pages ON page.page_id = custom_form_pages.page_id
             WHERE page.page_type = 'custom form'
             ORDER BY custom_form_pages.form_name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while ($row = mysqli_fetch_assoc($result)) {
        if ($row['page_id'] == $page_id) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        // if access does not need to be validated
        // or custom form is the current selected custom form
        // or user has access to this custom form,
        // prepare to output custom form option
        if (!$user || $selected || (check_edit_access($row['page_folder']) == true)) {
            if ($row['form_name']) {
                $name = $row['form_name'];
            } else {
                $name = $row['page_name'];
            }
            
            $output .= '<option value="' . $row['page_id'] . '"' . $selected . '>' . h($name) . '</option>';
        }
    }
    
    return $output;
}

// outputs the form's <select> list for selecting a page type
function select_page_type($page_type = '', $user)
{
    $output = '';
    
    switch ($page_type) {
        case 'change password':
            $change_password_status = ' selected="selected"';
            break;
                        
        case 'set password':
            $set_password_status = ' selected="selected"';
            break;
                        
        case 'email a friend':
            $email_a_friend_status = ' selected="selected"';
            break;
            
        case 'error':
            $error_status = ' selected="selected"';
            break;

        case 'folder view':
            $folder_view_status = ' selected="selected"';
            break;
            
        case 'forgot password':
            $forgot_password_status = ' selected="selected"';
            break;
            
        case 'login':
            $login_status = ' selected="selected"';
            break;
            
        case 'logout':
            $logout_status = ' selected="selected"';
            break;
            
        case 'my account':
            $my_account_status = ' selected="selected"';
            break;
            
        case 'my account profile':
            $my_account_profile_status = ' selected="selected"';
            break;
            
        case 'email preferences':
            $email_preferences_status = ' selected="selected"';
            break;
            
        case 'view order':
            $view_order_status = ' selected="selected"';
            break;
            
        case 'update address book':
            $update_address_book_status = ' selected="selected"';
            break;
            
        case 'search results':
            $search_results_status = ' selected="selected"';
            break;
            
        case 'registration entrance':
            $registration_entrance_status = ' selected="selected"';
            break;
            
        case 'registration confirmation':
            $registration_confirmation_status = ' selected="selected"';
            break;
            
        case 'membership entrance':
            $membership_entrance_status = ' selected="selected"';
            break;
            
        case 'membership confirmation':
            $membership_confirmation_status = ' selected="selected"';
            break;
            
        case 'custom form':
            $custom_form_status = ' selected="selected"';
            break;
            
        case 'custom form confirmation':
            $custom_form_confirmation_status = ' selected="selected"';
            break;
            
        case 'form list view':
            $form_list_view_status = ' selected="selected"';
            break;
            
        case 'form item view':
            $form_item_view_status = ' selected="selected"';
            break;
            
        case 'form view directory':
            $form_view_directory_status = ' selected="selected"';
            break;
            
        case 'calendar view':
            $calendar_view_status = ' selected="selected"';
            break;
            
        case 'calendar event view':
            $calendar_event_view_status = ' selected="selected"';
            break;
            
        case 'catalog':
            $catalog_status = ' selected="selected"';
            break;
            
        case 'catalog detail':
            $catalog_detail_status = ' selected="selected"';
            break;

        case 'express order':
            $express_order_status = ' selected="selected"';
            break;
            
        case 'order form':
            $order_form_status = ' selected="selected"';
            break;
            
        case 'shopping cart':
            $shopping_cart_status = ' selected="selected"';
            break;

        case 'shipping address and arrival':
            $shipping_address_and_arrival_status = ' selected="selected"';
            break;

        case 'shipping method':
            $shipping_method_status = ' selected="selected"';
            break;

        case 'billing information':
            $billing_information_status = ' selected="selected"';
            break;

        case 'order preview':
            $order_preview_status = ' selected="selected"';
            break;

        case 'order receipt':
            $order_receipt_status = ' selected="selected"';
            break;
            
        case 'affiliate sign up form':
            $affiliate_sign_up_form_status = ' selected="selected"';
            break;

        case 'affiliate sign up confirmation':
            $affiliate_sign_up_confirmation_status = ' selected="selected"';
            break;
            
        case 'affiliate welcome':
            $affiliate_welcome_status = ' selected="selected"';
            break;
            
        case 'photo gallery':
            $photo_gallery_status = ' selected="selected"';
            break;
    }
    
    // output the standard page type option
    $output .= '<option value="standard">Standard</option>';
    
    $output_miscellaneous_options = '';
    
    // if user is above a basic user role, then prepare system page types
    if ($user['role'] < 3) {
        $output_miscellaneous_options .=
            '<option value="change password"' . $change_password_status . '>Change Password</option>' .
            '<option value="set password"' . $set_password_status . '>Set Password</option>' ;
    }
    
    // if user is at least a manager or if they are able to create this type of page, then output the page type option
    if (($user['role'] < 3) || ($user['set_page_type_email_a_friend'] == TRUE)) {
        $output_miscellaneous_options .= '<option value="email a friend"' . $email_a_friend_status . '>E-mail a Friend</option>';
    }
    
    // if user is above a basic user role, then output error page type option.
    if ($user['role'] < 3) {
        $output_miscellaneous_options .= '<option value="error"' . $error_status . '>Error</option>';
    }

    // If user is at least a manager or if they are able to create this type of page, then output the page type option.
    if (($user['role'] < 3) || ($user['set_page_type_folder_view'] == TRUE)) {
        $output_miscellaneous_options .= '<option value="folder view"' . $folder_view_status . '>Folder View</option>';
    }

    // If user is above a basic user role, then output more page type options.
    if ($user['role'] < 3) {
        $output_miscellaneous_options .=
            '<option value="forgot password"' . $forgot_password_status . '>Forgot Password</option>
            <option value="login"' . $login_status . '>Login</option>
            <option value="logout"' . $logout_status . '>Logout</option>';
    }
    
    // if user is at least a manager or if they are able to create this type of page, then output the page type option
    if (($user['role'] < 3) || ($user['set_page_type_photo_gallery'] == TRUE)) {
        $output_miscellaneous_options .= '<option value="photo gallery"' . $photo_gallery_status . '>Photo Gallery</option>';
    }
    
    // if user is above a basic user role, then prepare system page types
    if ($user['role'] < 3) {
        $output_miscellaneous_options .= '<option value="search results"' . $search_results_status . '>Search Results</option>';
    }
    
    // if there are miscellaneous page types to output then output them
    if ($output_miscellaneous_options != '') {
        $output .= 
            '<optgroup label="Miscellaneous">
                ' . $output_miscellaneous_options . '
            </optgroup>';
    }
    
    // if user is above a basic user role, then prepare system page types
    if ($user['role'] < 3) {
        $output .=
            '<optgroup label="Registration">
                <option value="registration entrance"' . $registration_entrance_status . '>Registration Entrance</option>
                <option value="registration confirmation"' . $registration_confirmation_status . '>Registration Confirmation</option>
            </optgroup>
            <optgroup label="Membership">
                <option value="membership entrance"' . $membership_entrance_status . '>Membership Entrance</option>
                <option value="membership confirmation"' . $membership_confirmation_status . '>Membership Confirmation</option>
            </optgroup>
            <optgroup label="My Account">
                <option value="my account"' . $my_account_status . '>My Account</option>
                <option value="my account profile"' . $my_account_profile_status . '>My Account Profile</option>
                <option value="email preferences"' . $email_preferences_status . '>E-mail Preferences</option>';
        
        // if e-commerce module is active, then prepare e-commerce page types
        if (ECOMMERCE == true) {
            $output .=
                '<option value="view order"' . $view_order_status . '>View Order</option>
                <option value="update address book"' . $update_address_book_status . '>Update Address Book</option>';
        }
        
        $output .= '</optgroup>';
    }
    
    // if forms module is active, then use forms page types
    if (FORMS == true) {
        $output_form_options = '';
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_custom_form'] == TRUE)) {
            $output_form_options .='<option value="custom form"' . $custom_form_status . '>Custom Form</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_custom_form_confirmation'] == TRUE)) {
            $output_form_options .='<option value="custom form confirmation"' . $custom_form_confirmation_status . '>Custom Form Confirmation</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_form_list_view'] == TRUE)) {
            $output_form_options .='<option value="form list view"' . $form_list_view_status . '>Form List View</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_form_item_view'] == TRUE)) {
            $output_form_options .='<option value="form item view"' . $form_item_view_status . '>Form Item View</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_form_view_directory'] == TRUE)) {
            $output_form_options .='<option value="form view directory"' . $form_view_directory_status . '>Form View Directory</option>';
        }
        
        // if there are form options, then output the group
        if ($output_form_options != '') {
            $output .=
                '<optgroup label="Forms">
                    ' . $output_form_options . '
                </optgroup>';
        }
    }
    
    // if calendars module is active and user has access to calendars, then use calendars page types
    if ((CALENDARS == true) && (($user['role'] < 3) || ($user['manage_calendars'] == true))) {
        $output_calendar_options = '';
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_calendar_view'] == TRUE)) {
            $output_calendar_options .= '<option value="calendar view"' . $calendar_view_status . '>Calendar View</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_calendar_event_view'] == TRUE)) {
            $output_calendar_options .= '<option value="calendar event view"' . $calendar_event_view_status . '>Calendar Event View</option>';
        }
        
        // if there are calendar options to output, then output them
        if ($output_calendar_options != '') {
            $output .=
                '<optgroup label="Calendars">
                    ' . $output_calendar_options . '
                </optgroup>';
        }
    }

    // if e-commerce module is active, then use e-commerce page types
    if (ECOMMERCE == true) {
        $output_commerce_options = '';
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_catalog'] == TRUE)) {
            $output_commerce_options .='<option value="catalog"' . $catalog_status . '>Catalog</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_catalog_detail'] == TRUE)) {
            $output_commerce_options .='<option value="catalog detail"' . $catalog_detail_status . '>Catalog Detail</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_express_order'] == TRUE)) {
            $output_commerce_options .='<option value="express order"' . $express_order_status . '>Express Order</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_order_form'] == TRUE)) {
            $output_commerce_options .='<option value="order form"' . $order_form_status . '>Order Form</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_shopping_cart'] == TRUE)) {
            $output_commerce_options .='<option value="shopping cart"' . $shopping_cart_status . '>Shopping Cart</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_shipping_address_and_arrival'] == TRUE)) {
            $output_commerce_options .='<option value="shipping address and arrival"' . $shipping_address_and_arrival_status . '>Shipping Address &amp; Arrival</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_shipping_method'] == TRUE)) {
            $output_commerce_options .='<option value="shipping method"' . $shipping_method_status . '>Shipping Method</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_billing_information'] == TRUE)) {
            $output_commerce_options .='<option value="billing information"' . $billing_information_status . '>Billing Information</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_order_preview'] == TRUE)) {
            $output_commerce_options .='<option value="order preview"' . $order_preview_status . '>Order Preview</option>';
        }
        
        // if user is at least a manager or if they are able to create this type of page, then output the page type option
        if (($user['role'] < 3) || ($user['set_page_type_order_receipt'] == TRUE)) {
            $output_commerce_options .='<option value="order receipt"' . $order_receipt_status . '>Order Receipt</option>';
        }
        
        // if there are ecommerce options to output, then output them
        if ($output_commerce_options != '') {
            $output .=
                '<optgroup label="E-Commerce">
                    ' . $output_commerce_options . '
                </optgroup>';
        }
    }
    
    // if affiliate program is active and user is above a basic user role, then use affiliate page types
    if ((AFFILIATE_PROGRAM == true) && ($user['role'] < 3)) {
        $output .=
            '<optgroup label="Affiliate Program">
                <option value="affiliate sign up form"' . $affiliate_sign_up_form_status . '>Affiliate Sign Up Form</option>
                <option value="affiliate sign up confirmation"' . $affiliate_sign_up_confirmation_status . '>Affiliate Sign Up Confirmation</option>
                <option value="affiliate welcome"' . $affiliate_welcome_status . '>Affiliate Welcome</option>
            </optgroup>';
    }

    return $output;
}

function get_product_group_options($product_group_id = 0, $parent_product_group_id = 0, $excluded_product_group_id = 0, $level = 0, $product_groups = array(), $include_select_product_groups = TRUE, $format = 'text', $include_blank_option = FALSE, $include_disabled = true)
{
    global $user;
    
    // If the format is text, then prepare output variable for that.
    if ($format == 'text') {
        $output = '';

    // Otherwise the format is array, so prepare output variable for that.
    } else {
        $output = array();
    }
    
    // If this is the first time this function has run, then get all product groups and put them in an array,
    // and determine if a blank option should be added.
    if (count($product_groups) == 0) {

        $sql_where = "";

        // If disabled product groups should not be included, then add SQL where
        // filter.
        if (!$include_disabled) {
            $sql_where = "WHERE enabled = '1'";
        }

        // get all product groups
        $query =
            "SELECT
                id,
                name,
                enabled,
                parent_id,
                display_type
            FROM product_groups
            $sql_where
            ORDER BY sort_order, name";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // add each product group to an array
        while ($row = mysqli_fetch_assoc($result)) {
            $product_groups[] = $row;
        }

        // If a blank option should be included, then include it.
        if ($include_blank_option == TRUE) {
            // If the format is text, then get prepare blank option in a certain way.
            if ($format == 'text') {
                $output .= '<option value=""></option>';

            // Otherwise the format is array, so prepare blank option a different way.
            } else {
                $output[] = array(
                    'label' => '',
                    'value' => ''
                );
            }
        }
    }
    
    $child_product_groups = array();
    
    // loop through product groups array in order to get all product groups that are in parent product group
    foreach ($product_groups as $product_group) {
        // if the parent product group id for this product group is equal to the parent product group id,
        // and this product group is a browse product group or select product groups should be included
        // then this is a child product group and it should be added, so add to array
        if (
            ($product_group['parent_id'] == $parent_product_group_id)
            && (($product_group['display_type'] == 'browse') || ($include_select_product_groups == TRUE))
        ) {
            $child_product_groups[] = $product_group;
        }
    }
    
    // loop through child product groups
    foreach ($child_product_groups as $child_product_group) {
        // if product group id is not equal to excluded product group id, then continue to prepare option and get child product groups
        if ($child_product_group['id'] != $excluded_product_group_id) {            
            // prepare indentation
            $indentation = '';
            
            for ($i = 1; $i <= $level; $i++)
            {
                $indentation .= '&nbsp;&nbsp;&nbsp;&nbsp;';
            }
            
            $next_level = $level + 1;

            $disabled = '';
            
            // If this product group is disabled, then output "[DISABLED]" on the
            // end of the label.
            if (!$child_product_group['enabled']) {
                $disabled = ' [DISABLED]';
            }

            // If the format is text, then prepare output for that format.
            if ($format == 'text') {
                // if this product group is the selected product group, then this option should be selected
                if ($child_product_group['id'] == $product_group_id) {
                    $selected = ' selected';
                } else {
                    $selected = '';
                }
                
                $output .= '<option value="' . $child_product_group['id'] . '"' . $selected . '>' . $indentation . h($child_product_group['name']) . $disabled . '</option>';

            // Otherwise the format is array, so prepare output for that format.
            } else {
                $output[] = array(
                    'label' => $indentation . h($child_product_group['name']) . $disabled,
                    'value' => $child_product_group['id']
                );
            }

            // if this product group is a browse product group, then get child product groups
            if ($child_product_group['display_type'] == 'browse') {
                // If the format is text, then get child product groups in a certain way.
                if ($format == 'text') {
                    // get options for child product groups
                    $output .= get_product_group_options($product_group_id, $child_product_group['id'], $excluded_product_group_id, $next_level, $product_groups, $include_select_product_groups, $format, $include_blank_option, $include_disabled);

                // Otherwise the format is array, so get child product groups in a different way.
                } else {
                    $output = array_merge($output, get_product_group_options($product_group_id, $child_product_group['id'], $excluded_product_group_id, $next_level, $product_groups, $include_select_product_groups, $format, $include_blank_option, $include_disabled));
                }
            }
        }
    }
    
    return $output;
}

// get an array of ad regions for pick lists
function get_ad_region_options()
{
    global $user;
    
    // Setup first option
    $ad_region_options[''] = '';
    
    $sql_join = '';
    $where = '';

    // if user is a user role, then prepare sql join and where
    if ($user['role'] == 3) {
        $sql_join = 'LEFT JOIN users_ad_regions_xref ON ad_regions.id = users_ad_regions_xref.ad_region_id';
        $where = "WHERE users_ad_regions_xref.user_id = '" . escape($user['id']) . "'";
    }
    
    // get ad regions
    $query =
        "SELECT
            id,
            name
        FROM ad_regions
        $sql_join
        $where
        ORDER BY name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // loop through each ad region and add it to array
    while ($row = mysqli_fetch_assoc($result)) {
        $ad_region_options[h($row['name'])] = $row['id'];
    }
    
    return $ad_region_options;
}

// get an array of states for pick lists
function get_state_options()
{
    // check if foreign states exist
    // foreign states are states that belong to a country that is not the default country
    $query =
        "SELECT COUNT(*)
        FROM states
        LEFT JOIN countries ON states.country_id = countries.id
        WHERE countries.default_selected = '0'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_row($result);

    $foreign_states = FALSE;

    // if foreign states exist, then remember that
    if ($row[0] > 0) {
        $foreign_states = TRUE;
    }
    
    // get states
    $query = 
        "SELECT
            states.id,
            states.name,
            countries.name as country_name
        FROM states
        LEFT JOIN countries ON states.country_id = countries.id
        ORDER BY
            countries.name ASC,
            states.name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $states = array();
    
    // loop through each state and add it to array
    while ($row = mysqli_fetch_assoc($result)) {
        $states[] = $row;
    }
    
    $state_options = array();
    $state_options[''] = '';
    
    // loop through each state in order to prepare options
    foreach ($states as $state) {
        // if foreign states exist, then output state label with country name prefix in order to prevent confusion
        if ($foreign_states == TRUE) {
            $output_state_label = h($state['country_name']) . ': ' . h($state['name']);
            
        // else foreign states do not exist, so just use state name for label
        } else {
            $output_state_label = h($state['name']);
        }
        
        $state_options[$output_state_label] = $state['id'];
    }
    
    return $state_options;
}

// outputs the form's <select> list for selecting an image file name (value is the image file name)
function select_image_options($image_name = '', $design = FALSE)
{
    $folders_that_user_has_access_to = array();
    global $user;
    
    // if user is a basic user, then get folders that user has access to
    if ($user['role'] == 3) {
        $folders_that_user_has_access_to = get_folders_that_user_has_access_to($user['id']);
    }
    
    // if design files should not be included, then include filter
    if ($design == FALSE) {
        $sql_design = "AND (design = 0)";
    }
    
    // get image list
    $query =
        "SELECT
            name,
            folder
        FROM files
        WHERE
            (
                (type = 'gif')
                || (type = 'jpg')
                || (type = 'jpeg')
                || (type = 'png')
                || (type = 'bmp')
                || (type = 'tif')
                || (type = 'tiff')
            )
            $sql_design
            AND (attachment = 0)
        ORDER BY name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // loop through each image and create an <option> tag
    while ($row = mysqli_fetch_assoc($result)) {
        // if the user has access to this folder
        if (($row['name'] == $image_name) || (check_folder_access_in_array($row['folder'], $folders_that_user_has_access_to) == true)) {
            if ($row['name'] == $image_name) {
                $selected = ' selected="selected"';
            } else {
                $selected = '';
            }
            $output .= '<option value="' . h($row['name']) . '"' . $selected . '>' . h($row['name']) . '</option>';
        }
    }
    return $output;
}

// outputs the form's <select> list for selecting a recurring payment period
function select_payment_period($payment_period = '')
{
    $payment_periods[] = '';
    $payment_periods[] = 'Monthly';
    $payment_periods[] = 'Weekly';
    $payment_periods[] = 'Every Two Weeks';
    $payment_periods[] = 'Twice every Month';
    $payment_periods[] = 'Every Four Weeks';
    $payment_periods[] = 'Quarterly';
    $payment_periods[] = 'Twice every Year';
    $payment_periods[] = 'Yearly';

    foreach ($payment_periods as $value) {
        if ($value == $payment_period) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output .= '<option value="' . $value . '"' . $selected . '>' . $value . '</option>';
    }

    return $output;
}

function get_payment_period_options()
{
    $payment_period_options = array();
    
    $payment_period_options[''] = '';
    $payment_period_options['Monthly'] = 'Monthly';
    $payment_period_options['Weekly'] = 'Weekly';
    $payment_period_options['Every Two Weeks'] = 'Every Two Weeks';
    $payment_period_options['Twice every Month'] = 'Twice every Month';
    $payment_period_options['Every Four Weeks'] = 'Every Four Weeks';
    $payment_period_options['Quarterly'] = 'Quarterly';
    $payment_period_options['Twice every Year'] = 'Twice every Year';
    $payment_period_options['Yearly'] = 'Yearly';
    
    return $payment_period_options;
}

// outputs the form's <select> list for selecting a selection type for a product
function select_selection_type($selected_selection_type = '')
{
    $selection_types = array();

    $selection_types[] = array('value' => 'checkbox', 'name' => 'Checkbox');
    $selection_types[] = array('value' => 'quantity', 'name' => 'Quantity');
    $selection_types[] = array('value' => 'donation', 'name' => 'Donation');
    $selection_types[] = array('value' => 'autoselect', 'name' => 'Auto-Select');
    
    $output = '';
    
    foreach ($selection_types as $selection_type) {
        // if this selection type is the selected selection type then select this option
        if ($selection_type['value'] == $selected_selection_type) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output .= '<option value="' . $selection_type['value'] . '"' . $selected . '>' . $selection_type['name'] . '</option>';
    }

    return $output;
}

function get_registration_entrance_screen()
{
    // find if there is a registration entrance page
    $query = "SELECT page_id FROM page WHERE page_type = 'registration entrance'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is a registration entrance page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not a registration entrance page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_registration_entrance.php');

        $output = output_header_secure() . get_registration_entrance() . output_footer_secure();
    }

    return $output;
}

function get_membership_entrance_screen()
{
    // find if there is a membership entrance page
    $query = "SELECT page_id FROM page WHERE page_type = 'membership entrance'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is a membership entrance page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not a membership entrance page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_membership_entrance.php');

        $output = output_header_secure() . get_membership_entrance() . output_footer_secure();
    }

    return $output;
}

function get_registration_confirmation_screen()
{
    // find if there is a registration confirmation page
    $query = "SELECT page_id FROM page WHERE page_type = 'registration confirmation'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is a registration confirmation page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not a registration confirmation page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_registration_confirmation_screen_content.php');

        $output = output_header_secure() . get_registration_confirmation_screen_content() . output_footer_secure();
    }

    return $output;
}

function get_membership_confirmation_screen()
{
    // find if there is a membership confirmation page
    $query = "SELECT page_id FROM page WHERE page_type = 'membership confirmation'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is a membership confirmation page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not a membership confirmation page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_membership_confirmation_screen_content.php');

        $output = output_header_secure() . get_membership_confirmation_screen_content() . output_footer_secure();
    }

    return $output;
}

function get_my_account_profile_screen()
{
    // find if there is a my account profile page
    $query = "SELECT page_id FROM page WHERE page_type = 'my account profile'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is a my account profile page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not a my account profile page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_my_account_profile.php');

        $output = output_header_secure() . get_my_account_profile() . output_footer_secure();
    }

    return $output;
}

function get_email_preferences_screen()
{
    // find if there is an e-mail preferences page
    $query = "SELECT page_id FROM page WHERE page_type = 'email preferences'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is an e-mail preferences page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not an e-mail preferences page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_email_preferences.php');

        $output =
            output_header_secure() .
            get_email_preferences() .
            output_footer_secure();
    }

    return $output;
}

function get_view_order_screen()
{
    // find if there is a view order page
    $query = "SELECT page_id FROM page WHERE page_type = 'view order'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // if there is a view order page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');
        
        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);
    
    // else there is not a view order page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_view_order_screen_content.php');

        $output = output_header_secure() . get_view_order_screen_content(array()) . output_footer_secure();
    }

    return $output;
}

function get_update_address_book_screen()
{
    // find if there is an update address book page
    $query = "SELECT page_id FROM page WHERE page_type = 'update address book'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // if there is an update address book page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not an update address book page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_update_address_book.php');

        $output = output_header_secure() . get_update_address_book() . output_footer_secure();
    }

    return $output;
}

function get_error_screen($content)
{
    // if the connection to the database was successful
    if (defined('DB_CONNECTED') and DB_CONNECTED == true) {
        // find if there is an error page
        $query = "SELECT page_id FROM page WHERE page_type = 'error'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed');

        // if there is an error page
        if (mysqli_num_rows($result) > 0) {
            $row = mysqli_fetch_assoc($result);

            require_once(dirname(__FILE__) . '/get_page_content.php');

            // get page content
            $output = get_page_content($row['page_id'], $content, $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);
            return $output;
        }
    }

    $output = output_header_secure() . $content . output_footer_secure();
    return $output;
}

function get_forgot_password_screen() {
    // find if there is a forgot password page
    $query = "SELECT page_id FROM page WHERE page_type = 'forgot password'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is a forgot password page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $content, $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not a forgot password page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_forgot_password.php');

        $output =
            output_header_secure() .
            get_forgot_password() .
            output_footer_secure();
    }

    return $output;
}

function get_login_screen()
{
    // find if there is a login page
    $query = "SELECT page_id FROM page WHERE page_type = 'login'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is a login page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not a login page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_login.php');

        $output = output_header_secure() . get_login() . output_footer_secure();
    }

    return $output;
}

function get_logout_screen()
{
    // find if there is a logout page
    $query = "SELECT page_id FROM page WHERE page_type = 'logout'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is a logout page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not a login page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_logout_screen_content.php');

        $output = output_header_secure() . get_logout_screen_content() . output_footer_secure();
    }

    return $output;
}



function get_affiliate_sign_up_form_screen()
{
    // find if there is an affiliate sign up form page
    $query = "SELECT page_id FROM page WHERE page_type = 'affiliate sign up form'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is an affiliate sign up form page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not an affiliate sign up form page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_affiliate_sign_up_form_screen_content.php');

        $output = output_header_secure() . get_affiliate_sign_up_form_screen_content($properties) . output_footer_secure();
    }

    return $output;
}

function get_affiliate_sign_up_confirmation_screen()
{
    // check if there is a affiliate sign up confirmation page
    $query = "SELECT page_id FROM page WHERE page_type = 'affiliate sign up confirmation'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed');

    // if there is an affiliate sign up confirmation page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = false, $dynamic_properties = array(), $toolbar = FALSE, $_SESSION['software']['device_type']);

    // else there is not an affiliate sign up confirmation page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_affiliate_sign_up_confirmation_screen_content.php');

        $output = output_header_secure() . get_affiliate_sign_up_confirmation_screen_content() . output_footer_secure();
    }

    return $output;
}

function get_affiliate_welcome_screen()
{
    // find if there is an affiliate welcome page
    $query = "SELECT page_id FROM page WHERE page_type = 'affiliate welcome'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // if there is an affiliate welcome page
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);

        require_once(dirname(__FILE__) . '/get_page_content.php');

        // get page content
        $output = get_page_content($row['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = true);

    // else there is not an affiliate welcome page, so use default screen
    } else {
        require_once(dirname(__FILE__) . '/get_affiliate_welcome_screen_content.php');

        $output = output_header_secure() . get_affiliate_welcome_screen_content() . output_footer_secure();
    }

    return $output;
}

function log_activity($description, $user = '')
{
    // If a username was not passed, and a user is logged in,
    // then use that user's username.
    if (
        ($user == '')
        && defined('USER_USERNAME')
    ) {
        $user = USER_USERNAME;
    }

    $query = "INSERT INTO log (log_id, log_description, log_ip, log_user, log_timestamp) "
            ."VALUES ('', '" . escape($description) . "', '" . escape($_SERVER['REMOTE_ADDR']) . "', '" . escape($user) . "', UNIX_TIMESTAMP())";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // get a random number between 1 and 100 in order to determine if we should delete old log entries
    // there is a 1 in 100 chance that we will delete old log entries each time a log entry is added
    $random_number = rand(1, 100);
    
    // if the random number is 1, then delete old log entries
    // all log entries before 6 months ago are deleted
    if ($random_number == 1) {
        $query = "DELETE FROM log WHERE log_timestamp < (UNIX_TIMESTAMP() - 15552000)";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    }
}

function get_number_of_contacts($contact_group, $require_email = false)
{
    // if contact group is [None]
    if ($contact_group == '[None]') {
        $where = "WHERE (contacts_contact_groups_xref.contact_group_id IS NULL)";
    
    // else contact group is not [None]
    } else {
        $where = "WHERE (contacts_contact_groups_xref.contact_group_id = '" . escape($contact_group) . "')";
    }
    
    if ($require_email == true) {
        $where .=
            " AND (opt_in != 0)
            AND (email_address != '')";
    }

    // get number of contacts in group
    $query =
        "SELECT count(contacts.id)
        FROM contacts
        LEFT JOIN contacts_contact_groups_xref ON contacts.id = contacts_contact_groups_xref.contact_id
        $where";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_row($result);
    return $row[0];
}

function validate_date($date)
{
    // if format of date is valid
    if (preg_match('/(\d{1,2})[-,\/](\d{1,2})[-,\/](\d{4})/', $date, $date_parts) == 1) {
        $year = $date_parts[3];

        if (DATE_FORMAT == 'month_day') {
            $month = $date_parts[1];
            $day = $date_parts[2];
        } else {
            $month = $date_parts[2];
            $day = $date_parts[1];
        }
        
        // if date is valid (i.e. does the day exist for the month)
        if (checkdate($month, $day, $year) == true) {
            return true;
        } else {
            return false;
        }
        
    } else {
        return false;
    }
}

function validate_date_and_time($date_and_time)
{
    $date_and_time_parts = explode(' ', $date_and_time, 2);
    $date = $date_and_time_parts[0];
    $time = $date_and_time_parts[1];
    
    if ((validate_date($date) == true) && (validate_time($time) == true)) {
        return true;
    } else {
        return false;
    }
}

function validate_email_address($email_address)
{
    if (preg_match("/^([-!#\$%&'*+.\/0-9=?A-Z^_`a-z{|}~])+@([-!#\$%&'*+\/0-9=?A-Z^_`a-z{|}~]+\\.)+[a-zA-Z]{2,6}\$/i", $email_address) == 1) {
        return true;
    } else {
        return false;
    }
}

function validate_time($time)
{
    // if format of time is not valid
    if (preg_match('/(\d{1,2}:\d{1,2}) (AM|PM)/i', $time, $time_parts) == 0) {
        return false;
    }
    
    $hour = $time_parts[1];
    $minute = $time_parts[2];
    $second = $time_parts[4];
    
    // if format of time is not valid
    if (($hour == 0) || ($hour > 12) || ($minute > 59) || ($second > 59)) {
        return false;
    } else {
        return true;
    }
}

function prepare_page_for_email($html) {
    
    // find if there is a base tag in the HTML
    $base_in_html = preg_match('/<\s*base\s+[^>]*href\s*=\s*["\'](?:http:\/\/|https:\/\/|ftp:\/\/).*?["\']/is', $html);

    // if there is not a base tag in the HTML, add base tag and convert relative links to absolute links
    if (!$base_in_html) {
        $base = '<head>' . "\n" . '<base href="' . URL_SCHEME . HOSTNAME_SETTING . '/" />';
        $html = preg_replace('/<head>/i', $base, $html);

        // change relative URLs to absolute URLs for links
        $html = preg_replace('/(<\s*a\s+[^>]*href\s*=\s*["\'])(?!ftp:\/\/|https:\/\/|mailto:|http:\/\/)(?:\/|\.\.\/|\.\/|)(.*?["\'].*?>)/is', "$1" . URL_SCHEME . HOSTNAME_SETTING . "/$2", $html);

        // change relative URLs to absolute URLs for images
        $html = preg_replace('/(<\s*img\s+[^>]*src\s*=\s*["\'])(?!http:\/\/|https:\/\/)(?:\/|\.\.\/|\.\/|)(.*?["\'].*?>)/is', "$1" . URL_SCHEME . HOSTNAME_SETTING . "/$2", $html);

        // change relative URLs to absolute URLs for CSS background images
        $html = preg_replace('/(background-image\s*:\s*url\s*\(\s*(?:"|\'|))(?!http:\/\/|https:\/\/)(?:\/|\.\.\/|\.\/|)(.*?(?:"|\'|).*?\))/is', "$1" . URL_SCHEME . HOSTNAME_SETTING . "/$2", $html);

        // change relative URLs to absolute URLs for HTML background images
        $html = preg_replace('/(background\s*=\s*["\'])(?!http:\/\/|https:\/\/)(?:\/|\.\.\/|\.\/|)(.*?["\'])/is', "$1" . URL_SCHEME . HOSTNAME_SETTING . "/$2", $html);
    }

    // wrap long lines (RFC 821)
    $html = wordwrap($html, 900, "\n", 1);

    return $html;
}

// outputs the form's <select> product list for selecting a product

function select_product($product_id = '') {

    $query =
        "SELECT
            id,
            name,
            enabled,
            short_description,
            price
        FROM products
        ORDER BY name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    while ($row = mysqli_fetch_assoc($result)) {

        if ($row['id'] == $product_id) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output_disabled_label = '';
        
        // If this product is disabled, then output "disabled" next to the name and short description.
        if ($row['enabled'] == 0) {
            $output_disabled_label = ' [DISABLED]';
        }

        $output .= '<option value="' . $row['id'] . '"' . $selected . '>' . h($row['name']) . ' - ' . h($row['short_description']) . ' (' . prepare_amount($row['price'] / 100) . ')' . $output_disabled_label . '</option>';
    }

    return $output;
}

// get an array of products for pick lists
function get_product_options($include_blank_option = true) {
    
    $product_options = array();

    if ($include_blank_option) {
        $product_options[''] = '';
    }
    
    // get all products
    $query =
        "SELECT
            id,
            name,
            enabled,
            short_description,
            price
        FROM products
        ORDER BY name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // loop through each product in order to add it to array
    while ($row = mysqli_fetch_assoc($result)) {
        $output_disabled_label = '';
        
        // If this product is disabled, then output "disabled" next to the name and short description.
        if ($row['enabled'] == 0) {
            $output_disabled_label = ' [DISABLED]';
        }

        $label = h($row['name']) . ' - ' . h($row['short_description']) . ' (' . prepare_amount($row['price'] / 100) . ')' . $output_disabled_label;
        
        $product_options[$label] = $row['id'];
    }
    
    return $product_options;
}

function get_recipient_options()
{
    $recipient_options = array();
    
    $recipient_options[''] = '';
    $recipient_options['myself'] = 'myself';
    
    // if it has not been done already, add recipients to session
    initialize_recipients();
    
    // if there is at least one recipient stored in session
    if (isset($_SESSION['ecommerce']['recipients']) == true) {
        // loop through all recipients to build recipient options
        foreach ($_SESSION['ecommerce']['recipients'] as $recipient) {
            $value = h($recipient);
            $recipient_options[$value] = $value;
        }
    }
    
    $recipient_options['- add name below -'] = '- add name below -';
    
    return $recipient_options;
}

function select_country($country_id = '')
{
    $query = "SELECT id, name, default_selected ".
             "FROM countries ".
             "ORDER BY name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while ($row = mysqli_fetch_assoc($result)) {
        // if (this country is the selected country OR (this is the default selected country AND there is no selected country)), select this country by default
        if (($row['id'] == $country_id) || (($row['default_selected'] == 1) && ($country_id == ''))) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }
        $output .= '<option value="' . $row['id'] . '"' . $selected . '>' . h($row['name']) . '</option>';
    }
    return $output;
}

// outputs the options for a drop-down selection field for selecting a referral source
function select_referral_source($referral_source_code = '')
{
    $query = "SELECT name, code FROM referral_sources ORDER BY sort_order";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while ($row = mysqli_fetch_assoc($result)) {
        if (($referral_source_code) && ($row['code'] == $referral_source_code)) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }
        $output .= '<option value="' . $row['code'] . '"' . $selected . '>' . h($row['name']) . '</option>';
    }
    return $output;
}

// output options for drop-down selection for selecting an offer rule
function select_offer_rule($offer_rule_id = '')
{
    // get offer rule
    $query = "SELECT id, name FROM offer_rules ORDER BY name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while ($row = mysqli_fetch_assoc($result)) {
        if ($row['id'] == $offer_rule_id) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }
        $output .= '<option value="' . $row['id'] . '"' . $selected . '>' . $row['name'] . '</option>';
    }
    return $output;
}

// output options for drop-down selection for selecting an offer action type
function select_offer_action_type($offer_action_type = '')
{
    switch ($offer_action_type) {
        case 'discount order':
            $order_discount_status = ' selected="selected"';
            break;

        case 'discount product':
            $product_discount_status = ' selected="selected"';
            break;

        case 'add product':
            $add_product_status = ' selected="selected"';
            break;
            
        case 'discount shipping':
            $shipping_discount_status = ' selected="selected"';
            break;
    }
    
    $output_discount_shipping_option = '';
    
    // if shipping is on, then prepare to output discount shipping option
    if (ECOMMERCE_SHIPPING == true) {
        $output_discount_shipping_option = '<option value="discount shipping"' . $shipping_discount_status . '>Discount Shipping</option>';
    }

    return
        '<option value="discount order"' . $order_discount_status . '>Discount Order</option>
        <option value="discount product"' . $product_discount_status . '>Discount Product</option>
        <option value="add product"' . $add_product_status . '>Add Product</option>
        ' . $output_discount_shipping_option;
}

// output options for drop-down selection for selecting a field type
function select_field_type($field_type = '', $form_type = 'custom')
{
    switch ($field_type) {
        case 'text box':
            $text_box_status = ' selected="selected"';
            break;

        case 'text area':
            $text_area_status = ' selected="selected"';
            break;
        
        case 'pick list':
            $pick_list_status = ' selected="selected"';
            break;
        
        case 'radio button':
            $radio_button_status = ' selected="selected"';
            break;
        
        case 'check box':
            $check_box_status = ' selected="selected"';
            break;
        
        case 'file upload':
            $file_upload_status = ' selected="selected"';
            break;
            
        case 'date':
            $date_status = ' selected="selected"';
            break;
            
        case 'date and time':
            $date_and_time_status = ' selected="selected"';
            break;
            
        case 'email address':
            $email_address_status = ' selected="selected"';
            break;
        
        case 'information':
            $information_status = ' selected="selected"';
            break;
            
        case 'time':
            $time_status = ' selected="selected"';
            break;
    }
    
    $output_file_upload_field_option = '';
    
    // if this is a custom form, then output file upload field type option
    if ($form_type == 'custom') {
        $output_file_upload_field_option = '<option value="file upload"' . $file_upload_status . '>File Upload</option>'; 
    }
    
    $output =
        '<optgroup label="Standard">
            <option value="text box"' . $text_box_status . '>Text Box</option>
            <option value="text area"' . $text_area_status . '>Text Area</option>
            <option value="pick list"' . $pick_list_status . '>Pick List</option>
            <option value="radio button"' . $radio_button_status . '>Radio Button</option>
            <option value="check box"' . $check_box_status . '>Check Box</option>
            ' . $output_file_upload_field_option . '
        </optgroup>
        <optgroup label="Special">
            <option value="date"' . $date_status . '>Date</option>
            <option value="date and time"' . $date_and_time_status . '>Date &amp; Time</option>
            <option value="email address"' . $email_address_status . '>E-mail Address</option>
            <option value="information"' . $information_status . '>Information</option>
            <option value="time"' . $time_status . '>Time</option>
        </optgroup>';
    
    return $output;
}

// output options for drop-down selection for selecting a field's position
function select_field_position($position = '', $field_id = '', $page_or_product_id, $page_type, $form_type) {

    if ($form_type == 'product') {
        $form_fields_identifier_column = 'product_id';
    } else {
        $form_fields_identifier_column = 'page_id';
    }

    // Prepare sql filter in order to get correct fields

    $form_type_filter =
        "form_fields." . $form_fields_identifier_column . " = '" . e($page_or_product_id) . "'";

    // If the page type is express order then we need to add an extra filter for the form type
    if ($page_type == 'express order') {
        $form_type_filter .= " AND form_fields.form_type = '" . e($form_type) . "'";
    }
    
    if ($position == 'top') {
        $top_selected = ' selected="selected"';
    } else {
        $top_selected = '';
    }
    
    $output = '<option value="top"' . $top_selected . '>Top</option>';
    
    // get all fields
    $query = "SELECT
                id,
                name
             FROM form_fields
             WHERE $form_type_filter
             ORDER BY sort_order";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    while ($row = mysqli_fetch_assoc($result)) {
        if ($row['id'] != $field_id) {
            if ($row['id'] == $position) {
                $selected = ' selected="selected"';
            } else {
                $selected = '';
            }
            
            $output .= '<option value="' . $row['id'] . '"' . $selected . '>Below ' . h($row['name']) . '</option>';
        }
    }

    return $output;
}

// output options for drop-down selection for selecting a contact field
function select_contact_field($selected_contact_field = '')
{
    $contact_fields = array();

    $contact_fields[] = array('value' => 'salutation', 'name' => 'Salutation');
    $contact_fields[] = array('value' => 'first_name', 'name' => 'First Name');
    $contact_fields[] = array('value' => 'last_name', 'name' => 'Last Name');
    $contact_fields[] = array('value' => 'suffix', 'name' => 'Suffix');
    $contact_fields[] = array('value' => 'nickname', 'name' => 'Nickname');
    $contact_fields[] = array('value' => 'company', 'name' => 'Company');
    $contact_fields[] = array('value' => 'title', 'name' => 'Title');
    $contact_fields[] = array('value' => 'department', 'name' => 'Department');
    $contact_fields[] = array('value' => 'office_location', 'name' => 'Office Location');
    $contact_fields[] = array('value' => 'business_address_1', 'name' => 'Business Address 1');
    $contact_fields[] = array('value' => 'business_address_2', 'name' => 'Business Address 2');
    $contact_fields[] = array('value' => 'business_city', 'name' => 'Business City');
    $contact_fields[] = array('value' => 'business_state', 'name' => 'Business State');
    $contact_fields[] = array('value' => 'business_zip_code', 'name' => 'Business Zip Code');
    $contact_fields[] = array('value' => 'business_country', 'name' => 'Business Country');
    $contact_fields[] = array('value' => 'business_phone', 'name' => 'Business Phone');
    $contact_fields[] = array('value' => 'business_fax', 'name' => 'Business Fax');
    $contact_fields[] = array('value' => 'home_address_1', 'name' => 'Home Address 1');
    $contact_fields[] = array('value' => 'home_address_2', 'name' => 'Home Address 2');
    $contact_fields[] = array('value' => 'home_city', 'name' => 'Home City');
    $contact_fields[] = array('value' => 'home_state', 'name' => 'Home State');
    $contact_fields[] = array('value' => 'home_zip_code', 'name' => 'Home Zip Code');
    $contact_fields[] = array('value' => 'home_country', 'name' => 'Home Country');
    $contact_fields[] = array('value' => 'home_phone', 'name' => 'Home Phone');
    $contact_fields[] = array('value' => 'home_fax', 'name' => 'Home Fax');
    $contact_fields[] = array('value' => 'mobile_phone', 'name' => 'Mobile Phone');
    $contact_fields[] = array('value' => 'email_address', 'name' => 'E-mail Address');
    $contact_fields[] = array('value' => 'website', 'name' => 'Website');
    $contact_fields[] = array('value' => 'lead_source', 'name' => 'Lead Source');
    $contact_fields[] = array('value' => 'opt_in', 'name' => 'Opt-In');
    $contact_fields[] = array('value' => 'description', 'name' => 'Description');
    $contact_fields[] = array('value' => 'affiliate_name', 'name' => 'Affiliate Name');

    foreach ($contact_fields as $contact_field) {
        // if this contact field is the selected contact field, select this option
        if ($contact_field['value'] == $selected_contact_field) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output .= '<option value="' . $contact_field['value'] . '"' . $selected . '>' . $contact_field['name'] . '</option>';
    }

    return $output;
}

// Gets the number of users who have access to the control panel to edit something.
// The $exception_user_id is used by the edit user screen to ignore the current
// user that is being edited when figuring out the number of edit users.

function get_number_of_edit_users($exception_user_id = 0) {

    // Get all editors that we can figure out by just looking at the user table.

    $sql_calendars = "";

    if (CALENDARS) {
        $sql_calendars = "OR (user_manage_calendars = 'yes')";
    }

    $sql_ecommerce = "";

    if (ECOMMERCE) {

        $sql_offline_payment = "";

        if (ECOMMERCE_OFFLINE_PAYMENT) {
            $sql_offline_payment = "OR (user_set_offline_payment = '1')";
        }

        $sql_ecommerce =
            "OR (user_manage_ecommerce = 'yes')
            OR (manage_ecommerce_reports = '1')
            $sql_offline_payment";

    }

    $general_editors = db_values(
        "SELECT user_id FROM user
        WHERE
            (user_role < 3)
            $sql_calendars
            OR (user_manage_visitors = 'yes')
            OR (user_manage_contacts = 'yes')
            OR (user_manage_emails = 'yes')
            $sql_ecommerce");

    // Get other types of editors that we need to check other tables for.

    $folder_editors = db_values(
        "SELECT DISTINCT(aclfolder_user) FROM aclfolder
        LEFT JOIN user ON aclfolder.aclfolder_user = user.user_id
        WHERE (aclfolder.aclfolder_rights = '2') AND (user.user_role = '3')");

    $common_region_editors = db_values(
        "SELECT DISTINCT(users_common_regions_xref.user_id)
        FROM users_common_regions_xref
        LEFT JOIN user ON users_common_regions_xref.user_id = user.user_id
        WHERE user.user_role = '3'");

    $menu_editors = db_values(
        "SELECT DISTINCT(users_menus_xref.user_id)
        FROM users_menus_xref
        LEFT JOIN user ON users_menus_xref.user_id = user.user_id
        WHERE user.user_role = '3'");

    $ad_region_editors = array();

    if (ADS) {
        $ad_region_editors = db_values(
            "SELECT DISTINCT(users_ad_regions_xref.user_id)
            FROM users_ad_regions_xref
            LEFT JOIN user ON users_ad_regions_xref.user_id = user.user_id
            WHERE user.user_role = '3'");
    }

    // Combine all the different types of editors into one array so we can remove duplicates and count.
    $editors = array_merge(
        $general_editors, $folder_editors, $common_region_editors, $menu_editors, $ad_region_editors);

    // Remove duplicate users from the array, so we can get an accurate unique count.
    $editors = array_unique($editors);

    // If there is an exception user (i.e. user being edited), then remove it from the array,
    // because we don't want to count it.
    if ($exception_user_id) {
        if (($key = array_search($exception_user_id, $editors)) !== false) {
            unset($editors[$key]);
        }
    }

    // Return the total number of editors.
    return count($editors);

}

function logout()
{
    $username = $_SESSION['sessionusername'];
    $logged_in_as_different_user = $_SESSION['software']['logged_in_as_different_user'];
    
    session_unset();
    session_destroy();

    // If the user was not logged in as a different user,
    // then also clear remember me cookies.  We don't want to clear
    // remember me cookies if user was logged in as a different user,
    // because we want the user to be able to automatically log back in
    // under their own user account now.
    if ($logged_in_as_different_user == false) {
        setcookie('software[username]', '', time() - 1000, '/');
        setcookie('software[password]', '', time() - 1000, '/');
    }

    // remove the device type cookie, so if this visitor is a control panel user that was just previewing
    // mobile then the mobile site will not appear by default in their desktop web browser the next time
    // the visitor visits the site
    setcookie('software[device_type]', '', time() - 1000, '/');
    unset($_COOKIE['software']['device_type']);

    // if user was logged in
    if ($username) {
        log_activity("user logged out", $username);

    // else there was no user to logout
    } else {
        output_error('You are not logged in.');
    }
}

function add_order_item($product_id, $quantity, $donation_amount, $ship_to, $add_name, $calendar_event_id = 0, $recurrence_number = 0)
{
    // Check if product exists and is enabled and get product info.
    $query =
        "SELECT
            id,
            name,
            price,
            shippable,
            selection_type,
            submit_form,
            form
        FROM products
        WHERE
            (id = '" . escape($product_id) . "')
            AND (enabled = '1')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $product = mysqli_fetch_assoc($result);

    // If an enabled product was not found, then return false.
    if ($product['id'] == '') {
        return false;
    }

    // if shipping is on and product is shippable, create ship to record or get existing ship to id
    if ((ECOMMERCE_SHIPPING == true) && ($product['shippable'] == 1)) {
        $ship_to_id = create_or_get_ship_to($ship_to, $add_name);

    // else shipping is not on and/or product is not shippable
    } else {
        $ship_to_id = 0;
    }

    // Don't allow a negative donation amount.
    if ($donation_amount < 0) {
        $donation_amount = 0;
    }

    // find out if product has already been added to order
    // ignore order items that have been added by offers because they might be discounted and we don't want to mess with that
    $query =
        "SELECT id
        FROM order_items
        WHERE
            (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
            AND (product_id = '" . escape($product_id) . "')
            AND (ship_to_id = '" . escape($ship_to_id) . "')
            AND (calendar_event_id = '" . escape($calendar_event_id) . "')
            AND (recurrence_number = '" . escape($recurrence_number) . "')
            AND (added_by_offer = '0')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if product has already been added to order, update order item
    if (mysqli_num_rows($result) > 0) {
        // get order item id
        $row = mysqli_fetch_assoc($result);
        $order_item_id = $row['id'];
        
        // if this product is a donation product, update price
        if ($product['selection_type'] == 'donation') {
            $query =
                "UPDATE order_items
                SET
                    price = (price + '" . e($donation_amount) . "')
                WHERE
                    (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                    AND (product_id = '" . escape($product_id) . "')
                    AND (ship_to_id = '" . escape($ship_to_id) . "')
                    AND (calendar_event_id = '" . escape($calendar_event_id) . "')
                    AND (recurrence_number = '" . escape($recurrence_number) . "')";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
        // else this product is not a donation product, so update quantity
        } else {
            $query =
                "UPDATE order_items
                SET
                    quantity = (quantity + '" . e($quantity) . "')
                WHERE
                    (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                    AND (product_id = '" . escape($product_id) . "')
                    AND (ship_to_id = '" . escape($ship_to_id) . "')
                    AND (calendar_event_id = '" . escape($calendar_event_id) . "')
                    AND (recurrence_number = '" . escape($recurrence_number) . "')";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            // if there is a ship to for this order item
            if ($ship_to_id) {
                // get ship to information
                $query =
                    "SELECT complete
                    FROM ship_tos
                    WHERE id = '" . escape($ship_to_id) . "'";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                $row = mysqli_fetch_assoc($result);
                
                $complete = $row['complete'];
                
                // if ship to is complete, then update shipping cost for ship to
                if ($complete == 1) {
                    require_once(dirname(__FILE__) . '/shipping.php');
                    update_shipping_cost_for_ship_to($ship_to_id);
                }
            }
        }

    // else product has not been added to order, so create order item record
    } else {
        // if this product is a donation product, use donation amount for price of order item
        if ($product['selection_type'] == 'donation') {
            $price = $donation_amount;
            
        // else this product is not a donation product, so use price of product for price of order item
        } else {
            $price = $product['price'];
        }

        $add_watcher = '';

        // If this product submits a form and add watcher info is in the session,
        // then add watcher info for this order item, so that a watcher is added
        // to the form and page when the order is submitted.
        if (($product['submit_form'] == 1) && ($_SESSION['software']['product_submit_form']['add_watcher'] != '')) {
            $add_watcher = $_SESSION['software']['product_submit_form']['add_watcher'];
        }
        
        $query = "INSERT INTO order_items (
                    order_id,
                    ship_to_id,
                    product_id,
                    product_name,
                    quantity,
                    price,
                    calendar_event_id,
                    recurrence_number,
                    add_watcher)
                 VALUES (
                    '" . $_SESSION['ecommerce']['order_id'] . "',
                    '" . escape($ship_to_id) . "',
                    '" . escape($product_id) . "',
                    '" . escape($product['name']) . "',
                    '" . escape($quantity) . "',
                    '" . escape($price) . "',
                    '" . escape($calendar_event_id) . "',
                    '" . escape($recurrence_number) . "',
                    '" . escape($add_watcher) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $order_item_id = mysqli_insert_id(db::$con);
        
        // if there is a ship to for this order item
        if ($ship_to_id) {
            // get ship to information
            $query =
                "SELECT
                    complete,
                    country,
                    state
                FROM ship_tos
                WHERE id = '" . escape($ship_to_id) . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            $row = mysqli_fetch_assoc($result);
            
            $complete = $row['complete'];
            $country_code = $row['country'];
            $state_code = $row['state'];
            
            // if ship to is complete, then do some checks
            if ($complete == 1) {
                // if order item is valid for destination and arrival date, then update shipping for ship to
                if ((validate_product_for_destination($product_id, $country_code, $state_code) == true) && (validate_order_item_for_arrival_date($order_item_id) == true)) {
                    require_once(dirname(__FILE__) . '/shipping.php');
                    update_shipping_cost_for_ship_to($ship_to_id);
                
                // else there are problems, so mark ship to as incomplete
                } else {
                    $query = "UPDATE ship_tos SET complete = 0 WHERE id = '" . escape($ship_to_id) . "'";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                }
            }
        }

        // If this product has a product form, and there is prefill info in the session,
        // then determine if we need to prefill a product form field.
        // Prefill data is added to the session via the do.php script
        // which allows a product form field to be prefilled with a value,
        // based on where a visitor came from.
        // It is necessary to prefill the product form fields now,
        // when the order item is added, because if we prefilled on the shopping cart
        // the values would not actually be saved until the visitor updated the cart,
        // which the visitor might not do before saving or sharing the cart link.
        if (
            ($product['form'] == 1)
            && (isset($_SESSION['software']['prefill_product_form']))
            && (is_array($_SESSION['software']['prefill_product_form']))
        ) {
            // Loop through the fields to be prefilled.
            foreach ($_SESSION['software']['prefill_product_form'] as $field) {
                $value = $field['value'];

                // Check if there is a field in the product form with the field name in the session.
                $field = db_item(
                    "SELECT
                        id,
                        name,
                        type
                    FROM form_fields
                    WHERE
                        (product_id = '" . $product['id'] . "')
                        AND (name = '" . e($field['name']) . "')
                    LIMIT 1");

                // If a field was found, then continue to prefill field.
                if ($field['id']) {
                    // assume that the form data type is standard until we find out otherwise
                    $form_data_type = 'standard';
                    
                    // if the form field's type is date, date and time, or time, then set form data type to the form field type
                    if (
                        ($field['type'] == 'date')
                        || ($field['type'] == 'date and time')
                        || ($field['type'] == 'time')
                    ) {
                        $form_data_type = $field['type'];
                        
                    // else if the form field is a wysiwyg text area, then set type to html
                    } elseif (($field['type'] == 'text area') && ($field['wysiwyg'] == 1)) {
                        $form_data_type = 'html';
                    }

                    // Prefill value for field in database.
                    db(
                        "INSERT INTO form_data (
                            order_id,
                            order_item_id,
                            quantity_number,
                            form_field_id,
                            data,
                            name,
                            type)
                        VALUES (
                            '" . $_SESSION['ecommerce']['order_id'] . "',
                            '" . $order_item_id . "',
                            '1',
                            '" . $field['id'] . "',
                            '" . escape(prepare_form_data_for_input($value, $field['type'])) . "',
                            '" . escape($field['name']) . "',
                            '$form_data_type')");
                }
            }
        }
    }

    return $order_item_id;
}

function remove_order_item($order_item_id)
{
    // before we remove order item from order, get ship to id, so we can remove ship to if necessary
    $query = "SELECT ship_to_id FROM order_items WHERE id = '" . escape($order_item_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $ship_to_id = $row['ship_to_id'];

    // delete order item
    $query = "DELETE FROM order_items WHERE id = '" . escape($order_item_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    db("DELETE FROM order_item_gift_cards WHERE order_item_id = '" . escape($order_item_id) . "'");
    
    // delete product form data for order item
    $query = "DELETE FROM form_data WHERE (order_item_id = '" . escape($order_item_id) . "') AND (order_item_id != '0')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // get all order items for ship to id, so we can figure out if we may delete ship to record
    $query = "SELECT id FROM order_items WHERE ship_to_id = '" . escape($ship_to_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // if there are no order items left for ship to, then delete ship to
    if (mysqli_num_rows($result) == 0) {
        $query = "DELETE FROM ship_tos WHERE id = '$ship_to_id'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // delete custom shipping form data for ship to
        $query = "DELETE FROM form_data WHERE (ship_to_id = '$ship_to_id') AND (ship_to_id != '0')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
    // else there are order items left for ship to, so update shipping cost for ship to, if necessary
    } else {
        // if there is a ship to for this order item
        if ($ship_to_id) {
            // get ship to information
            $query =
                "SELECT complete
                FROM ship_tos
                WHERE id = '" . escape($ship_to_id) . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            $row = mysqli_fetch_assoc($result);
            
            $complete = $row['complete'];
            
            // if ship to is complete, then update shipping cost for ship to
            if ($complete == 1) {
                require_once(dirname(__FILE__) . '/shipping.php');
                update_shipping_cost_for_ship_to($ship_to_id);
            }
        }
    }
}

function create_or_get_ship_to($ship_to, $add_name)
{
    // get ship to name
    // if add name text field was used, use that name for ship to name
    if ($add_name) {
        $ship_to_name = $add_name;

    // else if a name was selected from the ship_to selection drop-down, then use that value
    } elseif (($ship_to != '') && ($ship_to != '- add name below -')) {
        $ship_to_name = $ship_to;

    // else use 'myself'
    } else {
        $ship_to_name = 'myself';
    }

    // figure out if ship_to record has already been created for ship to name
    $query = "SELECT id FROM ship_tos WHERE ship_to_name = '" . escape($ship_to_name) . "' AND order_id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // if ship_to record has already been created, get ship to id
    if (mysqli_num_rows($result)) {
        $row = mysqli_fetch_assoc($result);
        $ship_to_id = $row['id'];

    // else ship_to record has not been created, so create record
    } else {
        $query = "INSERT INTO ship_tos (order_id, ship_to_name) VALUES ('" . $_SESSION['ecommerce']['order_id'] . "', '" . escape($ship_to_name) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $ship_to_id = mysqli_insert_id(db::$con);
    }
    
    // add recipient to session so that ship to name appears in ship to pick lists
    add_recipient($ship_to_name);
    
    return $ship_to_id;
}

function get_page_name($page_id) {

    return db("SELECT page_name FROM page WHERE page_id = '" . e($page_id) . "'");
}

function encrypt_credit_card_number($credit_card_number, $key)
{
    // if the key is too large, then someone has messed with it, so output error, in order to avoid PHP error
    if (mb_strlen($key) > 32) {
        output_error('Encryption key is too large.');
    }
    
    $iv_size = mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_CBC);
    
    // if this server is on Windows, then seed random number generator and use MCRYPT_RAND
    if (mb_strtoupper(mb_substr(PHP_OS, 0, 3)) == 'WIN') {
        srand((double) microtime() * 1000000);
        $source = MCRYPT_RAND;
        
    // else this server is not on Windows, so use MCRYPT_DEV_URANDOM
    // we previously used MCRYPT_DEV_RANDOM, but we had to change to MCRYPT_DEV_URANDOM,
    // because MCRYPT_DEV_RANDOM caused random delays of 5 seconds to several minutes
    } else {
        $source = MCRYPT_DEV_URANDOM;
    }

    $iv = mcrypt_create_iv($iv_size, $source);
    
    // prepend iv and encrypt credit card number
    $encrypted_credit_card_number = $iv . mcrypt_encrypt(MCRYPT_RIJNDAEL_256, $key, $credit_card_number, MCRYPT_MODE_CBC, $iv);
    
    // base64 encode encrypted credit card number so that it does not contain binary characters and can be stored easily
    $encrypted_credit_card_number = base64_encode($encrypted_credit_card_number);
    
    return $encrypted_credit_card_number;
}

function decrypt_credit_card_number($credit_card_number, $key)
{
    // if the key is too large, then someone has messed with it, so output error, in order to avoid PHP error
    if (mb_strlen($key) > 32) {
        output_error('Encryption key is too large.');
    }
    
    // base64 decode in order to get binary encrypted credit card number
    $credit_card_number = base64_decode($credit_card_number);
    
    $iv_size = mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_CBC);
    
    // get iv from the front of the encrypted credit card number
    $iv = substr($credit_card_number, 0, $iv_size);
    
    // remove iv from the encrypted credit card number
    $credit_card_number = substr($credit_card_number, $iv_size);
    
    // decrypt the credit card number
    $decrypted_credit_card_number = mcrypt_decrypt(MCRYPT_RIJNDAEL_256, $key, $credit_card_number, MCRYPT_MODE_CBC, $iv);
    
    // remove null values from the end
    $decrypted_credit_card_number = rtrim($decrypted_credit_card_number, "\0");
    
    return $decrypted_credit_card_number;
}

function protect_credit_card_number($card_number)
{
    $protected_length = mb_strlen($card_number) - 4;
    for ($i = 1; $i <= $protected_length; $i++) {
        $credit_card_number .= '*';
    }
    $credit_card_number .= mb_substr($card_number, -4);

    return $credit_card_number;
}

function protect_card_verification_number($card_verification_number)
{
    $protected_length = mb_strlen($card_verification_number);
    
    $protected_card_verification_number = '';
    
    for ($i = 1; $i <= $protected_length; $i++) {
        $protected_card_verification_number .= '*';
    }

    return $protected_card_verification_number;
}

function get_valid_zones($ship_to_id, $country_code, $state_code)
{
    // get all products in cart for this ship to
    $query = "SELECT product_id
             FROM order_items
             WHERE order_id = '" . $_SESSION['ecommerce']['order_id'] . "' AND ship_to_id = '" . escape($ship_to_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    $order_items = array();
    
    // add all items in cart to array
    while ($row = mysqli_fetch_assoc($result)) {
        $order_items[] = $row['product_id'];
    }

    $zones_for_order_items = array();

    // loop through all order items
    foreach ($order_items as $key => $product_id) {
        $zones_for_order_item = array();
        
        // get all allowed zones for this order item
        $query = "SELECT zone_id
                 FROM products_zones_xref
                 WHERE product_id = '$product_id'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        while ($row = mysqli_fetch_assoc($result)) {
            $zones_for_order_item[] = $row['zone_id'];
        }
        
        // if this is the first order item, load $zones_for_order_items array with values
        if ($key == 0) {
            $zones_for_order_items = $zones_for_order_item;
        }
        
        // set zones for order items to only allowed zones from past order items and allowed zones for this order item
        $zones_for_order_items = array_intersect($zones_for_order_items, $zones_for_order_item);
    }
    
    $zones_for_destination = get_valid_zones_for_destination($country_code, $state_code);
    
    $zones = array();
    
    // loop through all valid zones for order items
    foreach ($zones_for_order_items as $zone_id) {
        // if zone is also valid for destination, then zone is valid
        if (in_array($zone_id, $zones_for_destination) == true) {
            $zones[] = $zone_id;
        }
    }

    return $zones;
}

function get_valid_zones_for_destination($country_code, $state_code) {
    // declare zones for country array that we will use to store all valid zones for recipient's country
    $zones_for_country = array();

    // get all zones that contain country
    $query = "SELECT zone_id
             FROM zones_countries_xref
             LEFT JOIN countries ON countries.id = zones_countries_xref.country_id
             WHERE countries.code = '" . escape($country_code) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while ($row = mysqli_fetch_assoc($result)) {
        $zones_for_country[] = $row['zone_id'];
    }

    // declare zones for state array that we will use to store all valid zones for recipient's state
    $zones_for_state = array();

    // declare zones array that we will use to store all valid zones for recipient's address
    $zones_for_destination = array();
    
    // get country id
    $query = "SELECT id FROM countries WHERE code = '" . escape($country_code) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $country_id = $row['id'];

    // find out if recipient's state is found in database and if it belongs to receipient's country
    $query =
        "SELECT id
        FROM states
        WHERE
            (code = '" . escape($state_code) . "')
            AND (country_id = '$country_id')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // if recipient's state was found in database and it belongs to recipient's country, find what zones contain recipient's state
    if (mysqli_num_rows($result) > 0) {
        $query = "SELECT zone_id
                 FROM zones_states_xref
                 LEFT JOIN states ON states.id = zones_states_xref.state_id
                 WHERE states.code = '" . escape($state_code) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        while ($row = mysqli_fetch_assoc($result)) {
            $zones_for_state[] = $row['zone_id'];
        }

        // loop through all zones for country
        foreach ($zones_for_country as $zone_id) {
            // if zone id in zone for country is also a valid zone for state, then it is a valid zone, so add it to zones array
            if (in_array($zone_id, $zones_for_state) == true) {
                $zones_for_destination[] = $zone_id;
            }
        }

    // else recipient's state was not found in database, so valid zones are all valid zones for country
    } else {
        $zones_for_destination = $zones_for_country;
    }
    
    return $zones_for_destination;
}

function validate_product_for_destination($product_id, $country_code, $state_code)
{
    // get country id
    $query = "SELECT id FROM countries WHERE code = '" . escape($country_code) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $country_id = $row['id'];
    
    // get all allowed zones for this product
    $query = "SELECT zone_id
             FROM products_zones_xref
             WHERE product_id = '" . escape($product_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $zones = array();
    
    // loop through all allowed zones for this product
    while ($row = mysqli_fetch_assoc($result)) {
        $zones[] = $row['zone_id'];
    }
    
    foreach ($zones as $key => $zone_id) {
        // determine if recipient's country is in this zone
        $query = "SELECT zone_id FROM zones_countries_xref
                 LEFT JOIN countries ON zones_countries_xref.country_id = countries.id
                 WHERE (zone_id = '$zone_id') AND (countries.code = '" . escape($country_code) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

        // if result was found, then product can be shipped to recipient's country
        if (mysqli_num_rows($result) > 0) {
            $valid_product = true;
        } else {
            $valid_product = false;
        }

        // if product is valid for recipient's country, check on state
        if ($valid_product == true) {
            // find out if recipient's state is found in database and if it belongs to recipient's country
            $query =
                "SELECT id
                FROM states
                WHERE
                    (code = '" . escape($state_code) . "')
                    AND (country_id = '$country_id')";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

            // if recipient's state was found in database and it belongs to recipient's country, check to see if this product can be shipped to recipient's state
            if (mysqli_num_rows($result) > 0) {
                // determine if recipient's state is in this zone
                $query = "SELECT zone_id FROM zones_states_xref
                         LEFT JOIN states ON zones_states_xref.state_id = states.id
                         WHERE (zone_id = '$zone_id') AND (states.code = '" . escape($state_code) . "')";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

                // if result was found, then product can be shipped to recipient's state
                if (mysqli_num_rows($result) > 0) {
                    $valid_product = true;

                // else result was not found, so product cannot be shipped to recipient's state
                } else {
                    $valid_product = false;
                }
            }
        }
        
        // if this product is valid for the destination, return true
        if ($valid_product == true) {
            return true;
        }
    }

    return false;
}

function get_order_subtotal() {

    $query = "SELECT price, quantity FROM order_items WHERE order_id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $order_subtotal = 0;
    
    while($row = mysqli_fetch_assoc($result)) {
        $order_subtotal += $row['price'] * $row['quantity'];
    }

    return $order_subtotal;
}

// Create function that is responsible for removing order items from an order
// if product no longer exists or is disabled and also updating order item prices.

function update_order_item_prices() {

    // get all order items for this order
    $query = "SELECT
                order_items.id as id,
                order_items.offer_id,
                products.id as product_id,
                products.enabled AS product_enabled,
                products.price as product_price,
                products.selection_type as product_selection_type
             FROM order_items
             LEFT JOIN products ON order_items.product_id = products.id
             WHERE order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    $order_items = array();

    while ($row = mysqli_fetch_assoc($result)) {
        $order_items[] = $row;
    }

    // loop through all order items
    foreach ($order_items as $key => $value) {
        // If a product was found for the order item and the product is enabled,
        // then continue to determine if price needs to be updated.
        if (
            ($order_items[$key]['product_id'])
            && ($order_items[$key]['product_enabled'] == 1)
        ) {
            // if order item was not modified by a special offer and order item is not a donation, update order item price
            if (($order_items[$key]['offer_id'] == 0) && ($order_items[$key]['product_selection_type'] != 'donation')) {
                $query = "UPDATE order_items
                         SET price = '" . $order_items[$key]['product_price'] . "'
                         WHERE id = '" . $order_items[$key]['id'] . "'";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            }
        
        // Otherwise a product was not found for order item (an admin recently deleted product),
        // or product is now disabled, so remove order item from cart.
        } else {
            remove_order_item($order_items[$key]['id']);
        }
    }
}

function update_order_item_taxes()
{
    // get information from order
    $query = "SELECT
                billing_state,
                billing_country,
                tax_exempt
             FROM orders
             WHERE id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $billing_state = $row['billing_state'];
    $billing_country = $row['billing_country'];
    $tax_exempt = $row['tax_exempt'];
    
    // if order is tax-exempt, clear tax from all order items
    if ($tax_exempt) {
        $query = "UPDATE order_items
                 SET tax = 0
                 WHERE order_id = '" . $_SESSION['ecommerce']['order_id'] . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
    // else order is not tax-exempt, so we need to apply taxes to order items
    } else {
        // get tax rate for billing address
        $billing_address_tax_rate = get_tax_rate_for_address($billing_country, $billing_state);
        
        // get all ship tos
        $query = "SELECT
                    DISTINCT order_items.ship_to_id,
                    ship_tos.state,
                    ship_tos.country
                 FROM order_items
                 LEFT JOIN ship_tos ON order_items.ship_to_id = ship_tos.id
                 WHERE order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "'
                 ORDER BY order_items.ship_to_id";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

        $ship_tos = array();

        // foreach ship to, add ship to to array
        while ($row = mysqli_fetch_assoc($result)) {
            $ship_tos[] = $row;
        }

        // loop through all ship tos
        foreach ($ship_tos as $ship_to) {
            // if this ship to is a real recipient, get tax rate for shipping address
            if ($ship_to['ship_to_id'] > 0) {
                $shipping_address_tax_rate = get_tax_rate_for_address($ship_to['country'], $ship_to['state']);
            }
            
            // get all order items for this ship to
            $query = "SELECT
                        order_items.id,
                        order_items.price,
                        products.taxable
                     FROM order_items
                     LEFT JOIN products ON order_items.product_id = products.id
                     WHERE order_id = '" . $_SESSION['ecommerce']['order_id'] . "' AND ship_to_id = '" . $ship_to['ship_to_id'] . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

            $order_items = array();

            // foreach order item for this ship to, add order item to array
            while ($row = mysqli_fetch_assoc($result)) {
                $order_items[] = $row;
            }

            // foreach order item
            foreach ($order_items as $order_item) {
                // if order item is taxable
                if ($order_item['taxable']) {
                    // if this order item has a recipient
                    if ($ship_to['ship_to_id'] > 0) {
                        // if shipping address is in a tax zone, apply tax
                        if ($shipping_address_tax_rate) {
                            $apply_tax = true;
                            $tax_rate = $shipping_address_tax_rate;
                            
                        // else shipping address is not in a tax zone, so don't apply tax
                        } else {
                            $apply_tax = false;
                        }
                    
                    // else this order item does not have a recipient
                    } else {
                        // if billing address is in a tax zone, apply tax
                        if ($billing_address_tax_rate) {
                            $apply_tax = true;
                            $tax_rate = $billing_address_tax_rate;
                            
                        // else billing address is not in a tax zone, so don't apply tax
                        } else {
                            $apply_tax = false;
                        }
                    }
                    
                // else order item is not taxable
                } else {
                    $apply_tax = false;
                }
                
                // if taxes should be applied to this order item, apply them
                if ($apply_tax == true) {
                    $tax_amount = round($tax_rate / 100 * $order_item['price']);
                    
                    $query = "UPDATE order_items
                             SET tax = '$tax_amount'
                             WHERE id = '" . $order_item['id'] . "'";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                    
                // else taxes should not be applied to order item, so clear taxes
                } else {
                    $query = "UPDATE order_items
                             SET tax = '0'
                             WHERE id = '" . $order_item['id'] . "'";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                }
            }
        }
    }
}

function validate_offer($offer_id, $ship_to_id = 0, $subtotal = '') {
    
    // get offer info
    $query =
        "SELECT
            offers.scope,
            offer_rules.id AS offer_rule_id,
            offer_rules.required_subtotal,
            offer_rules.required_quantity
        FROM offers
        LEFT JOIN offer_rules ON offer_rules.id = offers.offer_rule_id
        WHERE
            (offers.id = '" . e($offer_id) . "')
            AND (offers.status = 'enabled')
            AND (offers.start_date <= CURRENT_DATE())
            AND (CURRENT_DATE() <= offers.end_date)";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if an offer was not found, then the offer is not valid, so return false
    if (mysqli_num_rows($result) == 0) {
        return false;
    }
    
    $row = mysqli_fetch_assoc($result);
    
    $scope = $row['scope'];
    $offer_rule_id = $row['offer_rule_id'];
    $required_subtotal = $row['required_subtotal'];
    $required_quantity = $row['required_quantity'];

    // If this offer does not have an offer rule, then offer is valid.
    if (!$offer_rule_id) {
        return true;
    }

    $required_products = db_items("
        SELECT product_id AS id FROM offer_rules_products_xref
        WHERE offer_rule_id = '" . e($offer_rule_id) . "'");

    // If there is no required subtotal and no required products, then the offer is valid.
    if (!$required_subtotal and !$required_products) {
        return true;
    }
    
    // if this offer needs to be validated at a recipient level
    if (
        (ECOMMERCE_SHIPPING == true)
        && (ECOMMERCE_RECIPIENT_MODE == 'multi-recipient')
        && ($scope == 'recipient')
        && ($ship_to_id)
    ) {

        // Assume that required subtotal is not valid until we find otherwise.
        $required_subtotal_valid = false;

        // If there is a required subtotal for the recipient,
        // then determine if this recipient has that subtotal.
        if ($required_subtotal) {

            // Since the scope is recipient, we consider the subtotal requirement
            // to be the recipient's subtotal and not the entire order subtotal.

            $recipient_subtotal = db_value(
                "SELECT SUM(price * CAST(quantity AS signed))
                FROM order_items
                WHERE ship_to_id = '" . e($ship_to_id) . "'");

            if ($recipient_subtotal >= $required_subtotal) {
                $required_subtotal_valid = true;
            }

        // Otherwise there is no subtotal rule, so subtotal is valid.
        } else {
            $required_subtotal_valid = true;
        }

        // Assume that required product is not valid until we find otherwise.
        $required_product_valid = false;

        // If the required subtotal is valid then continue to check if required product is valid.
        if ($required_subtotal_valid) {

            // If there is a required product for the recipient,
            // then determine if recipient has that product quantity.
            if ($required_products) {

                require_once(dirname(__FILE__) . '/check_for_products_in_cart.php');

                if (
                    check_for_products_in_cart(array(
                        'products' => $required_products,
                        'quantity' => $required_quantity,
                        'recipient' => array('id' => $ship_to_id)))
                ) {
                    $required_product_valid = true;
                }

            // Otherwise there is no required product, so it is valid.
            } else {
                $required_product_valid = true;
            }

        }

        // If both the required subtotal and required product requirements
        // have been met, then the offer is valid.
        if ($required_subtotal_valid and $required_product_valid) {
            return true;
        } else {
            return false;
        }
        
    // else this offer needs to be validated at an order level
    } else {

        // if a subtotal was not passed then get dynamic subtotal
        if ($subtotal === '') {
            $subtotal = get_order_subtotal();
        }

        require_once(dirname(__FILE__) . '/check_for_products_in_cart.php');
        
        // If required subtotal is valid and there are no required products or required products
        // and quantity are in cart, then offer is valid.
        if (
            $subtotal >= $required_subtotal
            and (
                !$required_products
                or check_for_products_in_cart(array(
                    'products' => $required_products,
                    'quantity' => $required_quantity))
            )
        ) {
            return true;
        
        // else offer is not valid
        } else {
            return false;
        }
    }
}

function get_offer_code_for_special_offer_code($special_offer_code) {
    
    $offer_code = '';

    $query = "SELECT code FROM offers WHERE code = '" . escape($special_offer_code) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // If an offer was found, use that code.
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $offer_code = $row['code'];
        
    // Otherwise an offer was not found, so check if there is an active key code.
    } else {
        $offer_code = db_value(
            "SELECT offers.code
            FROM key_codes
            LEFT JOIN offers ON key_codes.offer_code = offers.code
            WHERE
                (key_codes.code = '" . e($special_offer_code) . "')
                AND (key_codes.enabled = '1')
                AND
                (
                    (key_codes.expiration_date = '0000-00-00')
                    OR (key_codes.expiration_date >= CURRENT_DATE())
                )");
    }
    
    return $offer_code;
}

// Create function that is responsible for updating and applying offers to an order.

function apply_offers_to_cart() {

    // get all order items that are discounted by an offer in order to remove discount so we can add discounts later
    // this does not include order items that were added and discounted by an offer, because we will deal with those later
    $query =
        "SELECT
            order_items.id,
            order_items.added_by_offer,
            products.id AS product_id,
            products.price AS product_price
        FROM order_items
        LEFT JOIN products ON order_items.product_id = products.id
        WHERE
            (order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
            AND (order_items.discounted_by_offer = '1')
            AND (order_items.added_by_offer = '0')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $order_items = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $order_items[] = $row;
    }
    
    // loop through the order items that are discounted by offers in order to remove discount
    foreach ($order_items as $order_item) {
        // if a product was found for this order item, then remove discount
        if ($order_item['product_id'] != '') {
            $query =
                "UPDATE order_items
                SET
                    price = '" . $order_item['product_price'] . "',
                    offer_id = '0',
                    discounted_by_offer = '0'
                WHERE id = '" . $order_item['id'] . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // else a product was not found for this order item (e.g. an admin recently deleted product), so remove order item from order
        } else {
            remove_order_item($order_item['id']);
        }
    }
    
    // if shipping is on, then remove shipping discounts so we can cleanly add shipping discounts later
    if (ECOMMERCE_SHIPPING == TRUE) {
        $query =
            "UPDATE ship_tos
            SET
                shipping_cost = original_shipping_cost,
                original_shipping_cost = '0',
                offer_id = '0'
            WHERE
                (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                AND (offer_id != '0')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    }

    // get special offer code for this order
    $query = "SELECT special_offer_code FROM orders WHERE id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $special_offer_code = $row['special_offer_code'];
    
    $offer_code = '';
    
    // if there is a special offer code for this order, then get offer code because the special offer code might just be a key code
    if ($special_offer_code != '') {
        $offer_code = get_offer_code_for_special_offer_code($special_offer_code);
    }
    
    // Get all active offers so we can apply offers to order.
    // We order by offer code in order to help us process competing offers later.
    $query =
        "SELECT
            offers.id,
            offers.code,
            offers.scope,
            offers.multiple_recipients,
            offers.description,
            offers.upsell,
            offers.upsell_message,
            offers.upsell_trigger_subtotal,
            offers.upsell_trigger_quantity,
            offers.upsell_action_button_label,
            offers.upsell_action_page_id,
            offers.only_apply_best_offer,
            offer_rules.id AS offer_rule_id,
            offer_rules.required_subtotal AS offer_rule_required_subtotal,
            offer_rules.required_quantity AS offer_rule_required_quantity
        FROM offers
        LEFT JOIN offer_rules ON offers.offer_rule_id = offer_rules.id
        WHERE
            (offers.status = 'enabled')
            AND (offers.start_date <= CURRENT_DATE())
            AND (CURRENT_DATE() <= offers.end_date)
            AND ((offers.require_code = 0) OR (offers.code = '" . escape($offer_code) . "'))
        ORDER BY offers.code ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $offers = array();

    while ($row = mysqli_fetch_assoc($result)) {
        $offers[] = $row;
    }
    
    // prepare to get best offers (including non-competing offers and competing offers that are the best offer)
    // competing offers are offers that share the same code but where only the best offer will be applied
    $best_offers = array();
    
    // create array to store the best offer id's for offer codes, so we don't have to look up the best offer multiple times
    $best_offer_ids = array();
    
    // loop through the offers in order to get best offers
    foreach ($offers as $key => $offer) {
        $previous_offer_code = '';
    
        // if this is not the first offer, then get previous offer code
        if ($key != 0) {
            $previous_offer_code = $offers[$key - 1]['code'];
        }
        
        $next_offer_code = '';
    
        // if this is not the last offer, then get next offer code
        if ($key != (count($offers) - 1)) {
            $next_offer_code = $offers[$key + 1]['code'];
        }
    
        // if this is not the first offer and the previous offer had the same code,
        // or if this is not the last offer and the next offer had the same code,
        // and if this offer is set so only the best offer is applied,
        // then this is a competing offer, so determine if offer is the best offer
        if (
            (
                ($key != 0) && ($offer['code'] == $previous_offer_code)
                || ($key != (count($offers) - 1)) && ($offer['code'] == $next_offer_code)
            )
            && ($offer['only_apply_best_offer'] == 1)
        ) {
            $best_offer_id = '';
            
            // if the best offer id has already been found for this offer code, then get it
            if (isset($best_offer_ids[$offer['code']]) == TRUE) {
                $best_offer_id = $best_offer_ids[$offer['code']];
                
            // else the best offer id has not already been found for this offer code, so get it
            // and then remember it by adding to array
            } else {
                $best_offer_id = get_best_offer_id($offer['code']);
                $best_offer_ids[$offer['code']] = $best_offer_id;
            }
            
            // if this offer is the best offer, then add it to array
            if ($offer['id'] == $best_offer_id) {
                $best_offers[] = $offer;
            }
            
        // else this is not a competing offer, so it is a best offer, so add it to array
        } else {
            $best_offers[] = $offer;
        }
    }
    
    // set active offers to all the best offers
    $offers = $best_offers;
    
    // get order items so we can prepare array for ship tos and order items so we can loop through them later when preparing to apply offers
    $query =
        "SELECT
            order_items.id,
            order_items.product_id,
            order_items.added_by_offer,
            order_items.ship_to_id,
            order_items.quantity,
            products.price AS product_price,
            ship_tos.complete,
            ship_tos.shipping_method_id,
            ship_tos.shipping_cost
        FROM order_items
        LEFT JOIN ship_tos ON order_items.ship_to_id = ship_tos.id
        LEFT JOIN products ON order_items.product_id = products.id
        WHERE order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $ship_tos = array();

    while ($row = mysqli_fetch_assoc($result)) {
        // if this ship to has not been added yet, then add it
        if (isset($ship_tos[$row['ship_to_id']]) == FALSE) {
            $ship_tos[$row['ship_to_id']] = array();
            $ship_tos[$row['ship_to_id']]['id'] = $row['ship_to_id'];
            $ship_tos[$row['ship_to_id']]['complete'] = $row['complete'];
            $ship_tos[$row['ship_to_id']]['shipping_method_id'] = $row['shipping_method_id'];
            $ship_tos[$row['ship_to_id']]['shipping_cost'] = $row['shipping_cost'];
            $ship_tos[$row['ship_to_id']]['order_items'] = array();
        }
        
        $ship_tos[$row['ship_to_id']]['order_items'][] = $row;
    }
    
    // loop through all active offers in order to get offer actions
    foreach ($offers as $key => $offer) {
        $query =
            "SELECT
                offer_actions.id,
                offer_actions.name,
                offer_actions.type,
                offer_actions.discount_order_amount,
                offer_actions.discount_order_percentage,
                offer_actions.discount_product_product_id,
                offer_actions.discount_product_amount,
                offer_actions.discount_product_percentage,
                offer_actions.add_product_product_id,
                offer_actions.add_product_quantity,
                offer_actions.add_product_discount_amount,
                offer_actions.add_product_discount_percentage,
                offer_actions.discount_shipping_percentage,
                discount_products.id AS discount_product_id,
                discount_products.shippable AS discount_product_shippable,
                add_products.id AS add_product_id,
                add_products.shippable AS add_product_shippable
            FROM offers_offer_actions_xref
            LEFT JOIN offer_actions ON offers_offer_actions_xref.offer_action_id = offer_actions.id
            LEFT JOIN products AS discount_products ON offer_actions.discount_product_product_id = discount_products.id
            LEFT JOIN products AS add_products ON offer_actions.add_product_product_id = add_products.id
            WHERE offers_offer_actions_xref.offer_id = '" . $offer['id'] . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $offers[$key]['offer_actions'] = mysqli_fetch_items($result);
    }
    
    $discounted_order_items = array();
    
    // get subtotal
    // we need the subtotal because we need to keep track of how the subtotal is changed as order items are discounted
    // in order for us to determine if offers are valid
    $subtotal = get_order_subtotal();
    
    // prepare array to store upsell offers
    // upsell offers are offers that have not had their requirements met but have hit their offer upsell triggers
    $upsell_offers = array();
    
    // loop through the offers in order to prepare to discount order items
    // we have to discount order items before we apply other offers,
    // because discounted order items affect the subtotal which might affect the rules for an offer
    // we are not actually discounting order items in the database in this loop for database efficiency,
    // since multiple offers might discount the same order item
    foreach ($offers as $offer) {
        // loop through the offers actions for this offer, in order to discount order items
        foreach ($offer['offer_actions'] as $offer_action) {
            // if this offer action discounts order items, then continue
            if (
                ($offer_action['type'] == 'discount product')
                && ($offer_action['discount_product_id'] != '')
                &&
                (
                    ($offer_action['discount_product_amount'] != 0)
                    || ($offer_action['discount_product_percentage'] != 0)
                )
            ) {
                $scope = '';
                
                // if this offer should be applied at the order level then remember that
                if (
                    (ECOMMERCE_SHIPPING == FALSE)
                    || (ECOMMERCE_RECIPIENT_MODE == 'single recipient')
                    || ($offer['scope'] == 'order')
                    || ($offer_action['discount_product_shippable'] == 0)
                ) {
                    $scope = 'order';
                
                // else this offer should be applied at the recipient level, so remember that
                } else {
                    $scope = 'recipient';
                }
                
                // if the scope is order and the offer is valid or the scope is recipient,
                // then prepare to discount order items
                if (
                    (
                        ($scope == 'order')
                        && (validate_offer($offer['id'], 0, $subtotal) == TRUE)
                    )
                    || ($scope == 'recipient')
                ) {
                    // loop through ship tos in order to loop through order items
                    foreach ($ship_tos as $ship_to) {
                        // if the scope is recipient and the offer is valid for this recipient or the scope is order,
                        // then prepare to discount order items
                        if (
                            (
                                ($scope == 'recipient')
                                && (validate_offer($offer['id'], $ship_to['id'], $subtotal) == TRUE)
                            )
                            || ($scope == 'order')
                        ) {
                            // loop through order items for this ship to in order to prepare to discount them
                            foreach ($ship_to['order_items'] as $order_item) {
                                // if this offer discounts this product
                                // and this order item was not added by an offer,
                                // then prepare to discount it
                                if (
                                    ($offer_action['discount_product_product_id'] == $order_item['product_id'])
                                    && ($order_item['added_by_offer'] == 0)
                                ) {
                                    $discounted_price = 0;
                                    
                                    // if discount is by amount, then get discounted price
                                    if ($offer_action['discount_product_amount']) {
                                        $discounted_price = $order_item['product_price'] - $offer_action['discount_product_amount'];
                                    
                                    // else discount is by percentage, so get discounted price
                                    } else {
                                        $discounted_price = $order_item['product_price'] - ($order_item['product_price'] * ($offer_action['discount_product_percentage'] / 100));
                                    }
                                    
                                    // if the discounted price is less than 0, then set discounted price to 0
                                    if ($discounted_price < 0) {
                                        $discounted_price = 0;
                                    }
                                    
                                    // if this offer is going to discount the product more than any previous offer so far, use this offer
                                    if (
                                        (isset($discounted_order_items[$order_item['id']]) == FALSE)
                                        || ($discounted_price < $discounted_order_items[$order_item['id']]['price'])
                                    ) {
                                        // if this is the first offer that has discounted this order item, then prepare array
                                        if (isset($discounted_order_items[$order_item['id']]) == FALSE) {
                                            $discounted_order_items[$order_item['id']] = array();
                                            $discounted_order_items[$order_item['id']]['id'] = $order_item['id'];
                                        }
                                        
                                        $discounted_order_items[$order_item['id']]['price'] = $discounted_price;
                                        $discounted_order_items[$order_item['id']]['offer_id'] = $offer['id'];
                                        $discounted_order_items[$order_item['id']]['offer_action_id'] = $offer_action['id'];
                                        
                                        // add previous discount for this order item (if one exists) back to subtotal
                                        $subtotal = $subtotal + $discounted_order_items[$order_item['id']]['discount'];
                                        
                                        // set new discount for this order item
                                        $discounted_order_items[$order_item['id']]['discount'] = ($order_item['product_price'] - $discounted_price) * $order_item['quantity'];
                                        
                                        // reduce subtotal by new discount for this order item
                                        // we do this so when we validate other offers, we will be validating with the most recent subtotal that we know about
                                        $subtotal = $subtotal - $discounted_order_items[$order_item['id']]['discount'];
                                    }
                                }
                            }
                            
                        // else this offer is not valid, so if an upsell has not already been added for it,
                        // and there is an upsell, then add upsell
                        } else if (
                            (isset($upsell_offers[$offer['id']]) == FALSE)
                            && (check_upsell($offer, $ship_to['id'], $subtotal) == TRUE)
                        ) {
                            $upsell_offers[$offer['id']] = $offer;
                        }
                    }
                    
                // else this offer is not valid, so if an upsell has not already been added for it,
                // and there is an upsell, then add upsell
                } else if (
                    (isset($upsell_offers[$offer['id']]) == FALSE)
                    && (check_upsell($offer, 0, $subtotal) == TRUE)
                ) {
                    $upsell_offers[$offer['id']] = $offer;
                }
            }
        }
    }
    
    // loop through discounted order items in order to discount them
    foreach ($discounted_order_items as $discounted_order_item) {
        $query =
            "UPDATE order_items
            SET
                price = '" . $discounted_order_item['price'] . "',
                offer_id = '" . $discounted_order_item['offer_id'] . "',
                offer_action_id = '" . $discounted_order_item['offer_action_id'] . "',
                discounted_by_offer = 1
            WHERE id = '" . $discounted_order_item['id'] . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    }
    
    // Get all order items that were added by an offer in order to determine if they are still valid.
    // We don't just want to remove them all and then add them down below because it would be too
    // much thrashing on the database and we would lose info about recipient it was added to and etc.
    // We join the offers_offer_actions_xref in order to join the offer actions table in order to
    // verify that offer action is still allowed for offer.

    $query =
        "SELECT
            order_items.id,
            order_items.ship_to_id,
            order_items.offer_id,
            order_items.product_id,
            offers.require_code AS offer_require_code,
            offers.code AS offer_code,
            offers.scope AS offer_scope,
            offers.multiple_recipients AS offer_multiple_recipients,
            offers.only_apply_best_offer AS offer_only_apply_best_offer,
            products.shippable AS product_shippable,
            products.price AS product_price,
            offer_actions.id AS offer_action_id,
            offer_actions.type AS offer_action_type,
            offer_actions.add_product_product_id AS offer_action_add_product_product_id,
            offer_actions.add_product_quantity AS offer_action_add_product_quantity,
            offer_actions.add_product_discount_amount AS offer_action_add_product_discount_amount,
            offer_actions.add_product_discount_percentage AS offer_action_add_product_discount_percentage
        FROM order_items
        LEFT JOIN offers ON order_items.offer_id = offers.id
        LEFT JOIN offers_offer_actions_xref ON ((order_items.offer_id = offers_offer_actions_xref.offer_id) AND (order_items.offer_action_id = offers_offer_actions_xref.offer_action_id))
        LEFT JOIN offer_actions ON offers_offer_actions_xref.offer_action_id = offer_actions.id
        LEFT JOIN products ON order_items.product_id = products.id
        WHERE
            (order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
            AND (order_items.added_by_offer = '1')
        ORDER BY order_items.id ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $order_items = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $order_items[] = $row;
    }
    
    // create array that will keep track of offers and actions that have added products
    $offers_and_offer_actions = array();
    
    // loop through the order items that were added by offers in order to determine if they are still valid
    foreach ($order_items as $order_item) {

        // if this offer does not require a code or it has the same code that the customer entered,
        // and this offer is valid,
        // and not only the best offer should be applied for this code or it is the best offer for the code,
        // and the action is still set to add product,
        // and the action still adds the same product as the product for this order item,
        // and
            // if this product has not already been added for this offer and action,
            // or the product is allowed to be added multiple times,
        // then the order item is valid, so update quantity and discount for product
        if (
            (($order_item['offer_require_code'] == 0) || (mb_strtolower($order_item['offer_code']) == mb_strtolower($offer_code)))
            && (validate_offer($order_item['offer_id'], $order_item['ship_to_id']) == TRUE)
            &&
            (
                ($order_item['offer_only_apply_best_offer'] == 0)
                || ($order_item['offer_id'] == get_best_offer_id($order_item['offer_code']))
            )
            && ($order_item['offer_action_type'] == 'add product')
            && ($order_item['offer_action_add_product_product_id'] == $order_item['product_id'])
            &&
            (
                (in_array($order_item['offer_id'] . '_' . $order_item['offer_action_id'], $offers_and_offer_actions) == FALSE)
                || (ECOMMERCE_SHIPPING == FALSE)
                || (ECOMMERCE_RECIPIENT_MODE == 'single recipient')
                || ($order_item['offer_scope'] == 'order')
                || ($order_item['product_shippable'] == 0)
                || ($order_item['offer_multiple_recipients'] == 1)
            )
        ) {
            $discounted_price = 0;
            
            // if discount is by amount, get discounted price
            if ($order_item['offer_action_add_product_discount_amount'] != 0) {
                $discounted_price = $order_item['product_price'] - $order_item['offer_action_add_product_discount_amount'];
                
            // else discount is by percentage, so get discounted price
            } else {
                $discounted_price = $order_item['product_price'] - ($order_item['product_price'] * ($order_item['offer_action_add_product_discount_percentage'] / 100));
            }
            
            $discounted_by_offer = 0;
            
            // if the product that was added is also being discounted, prepare to mark order item accordingly
            if (($order_item['offer_action_add_product_discount_amount'] != 0) || ($order_item['offer_action_add_product_discount_percentage'] != 0)) {                                
                $discounted_by_offer = 1;
            }
            
            // update order item quantity and price and mark order item as having been affected by an offer
            $query =
                "UPDATE order_items
                SET
                    quantity = '" . $order_item['offer_action_add_product_quantity'] . "',
                    price = '" . $discounted_price . "',
                    discounted_by_offer = '" . $discounted_by_offer . "'
                WHERE id = '" . $order_item['id'] . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            // if it has not already been remembered that this offer and action have added a product, then remember that
            if (in_array($order_item['offer_id'] . '_' . $order_item['offer_action_id'], $offers_and_offer_actions) == FALSE) {
                $offers_and_offer_actions[] = $order_item['offer_id'] . '_' . $order_item['offer_action_id'];
            }
        
        // else the order item is no longer valid, so remove order item
        } else {
            remove_order_item($order_item['id']);
        }
    }
    
    $best_order_discount = 0;
    $best_order_discount_offer_id = 0;
    
    // pending offers are offers that add product(s) that are ready for the customer to select to add them to the order
    $pending_offers = array();
    
    $discounted_ship_tos = array();
    
    // loop through the offers in order prepare to apply offers that discount order, add product, and/or discount shipping
    foreach ($offers as $offer) {
        // loop through the offers actions for this offer, in order to prepare to apply offers
        foreach ($offer['offer_actions'] as $offer_action) {
            // apply offer differently depending on the type of action
            switch ($offer_action['type']) {
                case 'discount order':
                    // if this action actually discounts the order,
                    // then prepare to apply discount
                    if (
                        ($offer_action['discount_order_amount'] != 0)
                        || ($offer_action['discount_order_percentage'] != 0)
                    ) {
                        // if this offer is valid,
                        // then prepare to apply discount
                        if (validate_offer($offer['id']) == TRUE) {
                            $order_discount = 0;
                            
                            // if discount is by amount, get order discount
                            if ($offer_action['discount_order_amount'] != 0) {
                                $subtotal = get_order_subtotal();

                                // If the subtotal is greater than or equal to the discount,
                                // then use the whole discount.
                                if ($subtotal >= $offer_action['discount_order_amount']) {
                                    $order_discount = $offer_action['discount_order_amount'];

                                // Otherwise the subtotal is less than the possible discount,
                                // so just set the discount to the amount of the subtotal.
                                // This prevents us from showing a larger discount than
                                // was actually applied on the cart, preview, and etc. screens.
                                // It also prevents the total from ever showing a negative number,
                                // which would be confusing to the customer.
                                } else {
                                    $order_discount = $subtotal;
                                }
                                
                            // else discount is by percentage, so get order discount
                            } else {
                                $order_discount = round(get_order_subtotal() * ($offer_action['discount_order_percentage'] / 100));
                            }
                            
                            // if this offer provides the best order discount so far then remember that
                            if ($order_discount > $best_order_discount) {
                                $best_order_discount = $order_discount;
                                $best_order_discount_offer_id = $offer['id'];
                            }
                        
                        // else this offer is not valid, so if an upsell has not already been added for it,
                        // and there is an upsell, then add upsell
                        } else if (
                            (isset($upsell_offers[$offer['id']]) == FALSE)
                            && (check_upsell($offer) == TRUE)
                        ) {
                            $upsell_offers[$offer['id']] = $offer;
                        }
                    }
                    
                    break;
                    
                case 'add product':
                    // if this action actually adds a product, then continue to add pending offer
                    if (
                        ($offer_action['add_product_id'] != '')
                        && ($offer_action['add_product_quantity'] != 0)
                    ) {
                        // if this offer should be applied at the order level then continue to add pending offer
                        if (
                            (ECOMMERCE_SHIPPING == FALSE)
                            || (ECOMMERCE_RECIPIENT_MODE == 'single recipient')
                            || ($offer['scope'] == 'order')
                            || ($offer_action['add_product_shippable'] == 0)
                        ) {
                            // if this offer is valid at the order level, then continue to add pending offer
                            if (validate_offer($offer['id']) == TRUE) {
                                // check if this offer and action have already been applied to this order
                                $query =
                                    "SELECT id
                                    FROM order_items
                                    WHERE
                                        (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                                        AND (offer_id = '" . $offer['id'] . "')
                                        AND (offer_action_id = '" . $offer_action['id'] . "')";
                                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                
                                // if this offer and action have not already been applied to this order, add pending offer
                                if (mysqli_num_rows($result) == 0) {
                                    // if this offer has not been added as a pending offer yet, then prepare array
                                    if (isset($pending_offers[$offer['id']]) == FALSE) {
                                        $pending_offers[$offer['id']] = $offer;
                                        $pending_offers[$offer['id']]['offer_actions'] = array();
                                    }
                                    
                                    $pending_offers[$offer['id']]['offer_actions'][] = $offer_action;
                                }
                                
                            // else this offer is not valid, so if an upsell has not already been added for it,
                            // and there is an upsell, then add upsell
                            } else if (
                                (isset($upsell_offers[$offer['id']]) == FALSE)
                                && (check_upsell($offer) == TRUE)
                            ) {
                                $upsell_offers[$offer['id']] = $offer;
                            }
                            
                        // else this offer should be applied at the recipient level, so continue to add pending offer
                        } else {
                            // check if this offer and action have already been applied to this order
                            $query =
                                "SELECT id
                                FROM order_items
                                WHERE
                                    (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                                    AND (offer_id = '" . $offer['id'] . "')
                                    AND (offer_action_id = '" . $offer_action['id'] . "')";
                            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                            
                            // if this offer and action have not already been applied to order or offer is allowed to be applied to multiple recipients, then continue to add pending offer
                            if ((mysqli_num_rows($result) == 0) || ($offer['multiple_recipients'] == 1)) {
                                // loop through recipients
                                foreach ($ship_tos as $ship_to) {
                                    // if offer is valid for this recipient
                                    if (validate_offer($offer['id'], $ship_to['id']) == TRUE) {
                                        // check if this offer and action have already been applied to this recipient
                                        $query =
                                            "SELECT id
                                            FROM order_items
                                            WHERE
                                                (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                                                AND (ship_to_id = '" . $ship_to['id'] . "')
                                                AND (offer_id = '" . $offer['id'] . "')
                                                AND (offer_action_id = '" . $offer_action['id'] . "')";
                                        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                        
                                        // if this offer and action have not already been applied to this recipient, then continue to add pending offer
                                        if (mysqli_num_rows($result) == 0) {
                                            // if this offer has not been added as a pending offer yet, then prepare array
                                            if (isset($pending_offers[$offer['id']]) == FALSE) {
                                                $pending_offers[$offer['id']] = $offer;
                                                $pending_offers[$offer['id']]['offer_actions'] = array();
                                            }
                                            
                                            // if this offer action has not been added as a pending offer action yet, then add it and prepare allowed recipients
                                            if (isset($pending_offers[$offer['id']]['offer_actions'][$offer_action['id']]) == FALSE) {
                                                $pending_offers[$offer['id']]['offer_actions'][$offer_action['id']] = $offer_action;
                                                $pending_offers[$offer['id']]['offer_actions'][$offer_action['id']]['allowed_recipients'] = array();
                                            }
                                            
                                            $pending_offers[$offer['id']]['offer_actions'][$offer_action['id']]['allowed_recipients'][] = $ship_to['id'];
                                        }
                                        
                                    // else this offer is not valid, so if an upsell has not already been added for it,
                                    // and there is an upsell, then add upsell
                                    } else if (
                                        (isset($upsell_offers[$offer['id']]) == FALSE)
                                        && (check_upsell($offer, $ship_to['id']) == TRUE)
                                    ) {
                                        $upsell_offers[$offer['id']] = $offer;
                                    }
                                }
                            }
                        }
                    }
                    
                    break;
                    
                case 'discount shipping':
                    // if shipping is on and there is an actual discount shipping percentage, then continue to discount shipping
                    if (
                        (ECOMMERCE_SHIPPING == true)
                        && ($offer_action['discount_shipping_percentage'] != 0)
                    ) {
                        $scope = '';
                        
                        // if this offer should be applied at the order level then remember that
                        if (
                            (ECOMMERCE_RECIPIENT_MODE == 'single recipient')
                            || ($offer['scope'] == 'order')
                        ) {
                            $scope = 'order';
                        
                        // else this offer should be applied at the recipient level, so remember that
                        } else {
                            $scope = 'recipient';
                        }
                        
                        // if the scope is order and the offer is valid or the scope is recipient,
                        // then prepare to discount shipping
                        if (
                            (
                                ($scope == 'order')
                                && (validate_offer($offer['id']) == TRUE)
                            )
                            || ($scope == 'recipient')
                        ) {
                            // loop through the ship tos in order to discount shipping
                            foreach ($ship_tos as $ship_to) {
                                // if this is a real ship to,
                                // and the ship to is complete,
                                // and if the scope is recipient and the offer is valid for this recipient,
                                // or the scope is order,
                                // then prepare to discount shipping
                                if (
                                    ($ship_to['id'] != 0)
                                    && ($ship_to['complete'] == 1)
                                    &&
                                    (
                                        (
                                            ($scope == 'recipient')
                                            && (validate_offer($offer['id'], $ship_to['id']) == TRUE)
                                        )
                                        || ($scope == 'order')
                                    )
                                ) {
                                    // check if this offer and action is valid for the shipping method
                                    $query =
                                        "SELECT offer_action_id
                                        FROM offer_actions_shipping_methods_xref
                                        WHERE
                                            (offer_action_id = '" . $offer_action['id'] . "')
                                            AND (shipping_method_id = '" . $ship_to['shipping_method_id'] . "')";
                                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                    
                                    // if this offer and action is valid for the shipping method then continue to prepare to discount shipping
                                    if (mysqli_num_rows($result) != 0) {
                                        $original_shipping_cost = $ship_to['shipping_cost'];
                                        
                                        $discounted_shipping_cost = $original_shipping_cost - ($original_shipping_cost * ($offer_action['discount_shipping_percentage'] / 100));
                                        
                                        // if this offer is going to discount the ship to more than any previous offer so far, use this offer
                                        if (
                                            (isset($discounted_ship_tos[$ship_to['id']]) == FALSE)
                                            || ($discounted_shipping_cost < $discounted_ship_tos[$ship_to['id']]['shipping_cost'])
                                        ) {
                                            // if this is the first offer that has discounted this ship to, then prepare array
                                            if (isset($discounted_ship_tos[$ship_to['id']]) == FALSE) {
                                                $discounted_ship_tos[$ship_to['id']] = array();
                                                $discounted_ship_tos[$ship_to['id']]['id'] = $ship_to['id'];
                                            }
                                            
                                            $discounted_ship_tos[$ship_to['id']]['shipping_cost'] = $discounted_shipping_cost;
                                            $discounted_ship_tos[$ship_to['id']]['original_shipping_cost'] = $original_shipping_cost;
                                            $discounted_ship_tos[$ship_to['id']]['offer_id'] = $offer['id'];
                                        }
                                    }
                                    
                                // else this offer is not valid, so if an upsell has not already been added for it,
                                // and there is an upsell, then add upsell
                                } else if (
                                    (isset($upsell_offers[$offer['id']]) == FALSE)
                                    && (check_upsell($offer, $ship_to['id']) == TRUE)
                                ) {
                                    $upsell_offers[$offer['id']] = $offer;
                                }
                            }
                            
                        // else this offer is not valid, so if an upsell has not already been added for it,
                        // and there is an upsell, then add upsell
                        } else if (
                            (isset($upsell_offers[$offer['id']]) == FALSE)
                            && (check_upsell($offer) == TRUE)
                        ) {
                            $upsell_offers[$offer['id']] = $offer;
                        }
                    }
                    
                    break;
            }
        }
    }
    
    // if there is an order discount, then add order discount to order
    if ($best_order_discount_offer_id != 0) {
        $query = "UPDATE orders SET discount_offer_id = '" . $best_order_discount_offer_id . "' WHERE id = '" . $_SESSION['ecommerce']['order_id'] . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $_SESSION['ecommerce']['order_discount'] = $best_order_discount;
        
    // else there is not an order discount, so if there was before, then remove it
    } else if (isset($_SESSION['ecommerce']['order_discount']) == TRUE) {
        // remove order discount until we find out below if there is an order discount
        $query = "UPDATE orders SET discount_offer_id = '0' WHERE id = '" . $_SESSION['ecommerce']['order_id'] . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        unset($_SESSION['ecommerce']['order_discount']);
    }
    
    // if shipping is on, then loop through discounted ship tos in order to discount them
    if (ECOMMERCE_SHIPPING == TRUE) {
        foreach ($discounted_ship_tos as $discounted_ship_to) {
            $query =
                "UPDATE ship_tos
                SET
                    shipping_cost = '" . $discounted_ship_to['shipping_cost'] . "',
                    original_shipping_cost = '" . $discounted_ship_to['original_shipping_cost'] . "',
                    offer_id = '" . $discounted_ship_to['offer_id'] . "'
                WHERE id = '" . $discounted_ship_to['id'] . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        }
    }
    
    return array(
        'pending_offers' => $pending_offers,
        'upsell_offers' => $upsell_offers
    );
}

function check_upsell($offer, $ship_to_id = 0, $subtotal = '') {
    
    // if upsell is disabled for offer, then return false
    if (!$offer['upsell']) {
        return false;
    }
    
    // if a subtotal was not passed then get dynamic subtotal
    if ($subtotal === '') {
        $subtotal = get_order_subtotal();
    }

    $required_products = db_items("
        SELECT product_id AS id FROM offer_rules_products_xref
        WHERE offer_rule_id = '" . e($offer['offer_rule_id']) . "'");

    require_once(dirname(__FILE__) . '/check_for_products_in_cart.php');
    
    // if required subtotal is valid and required product and quantity is valid with triggers taken into account,
    // then offer should be displayed as an upsell offer
    if (
        (
            ($offer['upsell_trigger_subtotal'] == 0)
            or ($offer['offer_rule_required_subtotal'] == 0)
            or ($subtotal >= $offer['offer_rule_required_subtotal'] - $offer['upsell_trigger_subtotal'])
        )
        and
        (
            !$offer['upsell_trigger_quantity']
            or !$offer['offer_rule_required_quantity']
            or !$required_products
            or check_for_products_in_cart(array(
                'products' => $required_products,
                'quantity' => $offer['offer_rule_required_quantity'] - $offer['upsell_trigger_quantity'],
                'recipient' => array('id' => $ship_to_id)))
        )
    ) {
        return true;
    }
    
    return false;
}

// get the best offer for offers that share the same offer code
// we determine the best offer by finding the offer which gives the largest discount
function get_best_offer_id($offer_code, $scope = '') {

    $sql_scope = "";
    
    // if scope is defined, then prepare to only get offers for that scope
    if ($scope != '') {
        $sql_scope = "AND (offers.scope = '" . $scope . "')";
    }
    
    // Get all active offers for the offer code so we can determine which is the best.
    $query =
        "SELECT
            offers.id,
            offers.scope,
            offers.multiple_recipients
        FROM offers
        WHERE
            (offers.status = 'enabled')
            AND (offers.start_date <= CURRENT_DATE())
            AND (CURRENT_DATE() <= offers.end_date)
            AND (offers.code = '" . escape($offer_code). "')
            " . $sql_scope;
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $offers = array();

    while ($row = mysqli_fetch_assoc($result)) {
        $offers[] = $row;
    }
    
    // if there is one offer, it is the best offer, so return id
    if (count($offers) == 1) {
        return $offers[0]['id'];
    }
    
    $best_order_discount = 0;
    $best_order_discount_offer_id = 0;
    
    // get order items so we can prepare array for ship tos and order items so we can loop through them later to determine discounts
    $query =
        "SELECT
            order_items.id,
            order_items.product_id,
            order_items.added_by_offer,
            order_items.ship_to_id,
            order_items.quantity,
            products.price AS product_price,
            ship_tos.complete,
            ship_tos.shipping_method_id,
            ship_tos.shipping_cost,
            ship_tos.original_shipping_cost
        FROM order_items
        LEFT JOIN ship_tos ON order_items.ship_to_id = ship_tos.id
        LEFT JOIN products ON order_items.product_id = products.id
        WHERE order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $ship_tos = array();

    while ($row = mysqli_fetch_assoc($result)) {
        // if this ship to has not been added yet, then add it
        if (isset($ship_tos[$row['ship_to_id']]) == FALSE) {
            $ship_tos[$row['ship_to_id']] = array();
            $ship_tos[$row['ship_to_id']]['id'] = $row['ship_to_id'];
            $ship_tos[$row['ship_to_id']]['complete'] = $row['complete'];
            $ship_tos[$row['ship_to_id']]['shipping_method_id'] = $row['shipping_method_id'];
            $ship_tos[$row['ship_to_id']]['shipping_cost'] = $row['shipping_cost'];
            $ship_tos[$row['ship_to_id']]['original_shipping_cost'] = $row['original_shipping_cost'];
            $ship_tos[$row['ship_to_id']]['order_items'] = array();
        }
        
        $ship_tos[$row['ship_to_id']]['order_items'][] = $row;
    }
    
    $largest_discount_amount = 0;
    $largest_number_of_added_products = 0;
    $best_offer_id = 0;
    
    // loop through active offers so we can determine which offer gives the largest discount
    foreach ($offers as $offer) {

        // get offer actions for this offer
        $query =
            "SELECT
                offer_actions.id,
                offer_actions.type,
                offer_actions.discount_order_amount,
                offer_actions.discount_order_percentage,
                offer_actions.discount_product_product_id,
                offer_actions.discount_product_amount,
                offer_actions.discount_product_percentage,
                offer_actions.add_product_product_id,
                offer_actions.add_product_quantity,
                offer_actions.add_product_discount_amount,
                offer_actions.add_product_discount_percentage,
                offer_actions.discount_shipping_percentage,
                discount_products.id AS discount_product_id,
                discount_products.shippable AS discount_product_shippable,
                add_products.id AS add_product_id,
                add_products.shippable AS add_product_shippable,
                add_products.price AS add_product_price
            FROM offers_offer_actions_xref
            LEFT JOIN offer_actions ON offers_offer_actions_xref.offer_action_id = offer_actions.id
            LEFT JOIN products AS discount_products ON offer_actions.discount_product_product_id = discount_products.id
            LEFT JOIN products AS add_products ON offer_actions.add_product_product_id = add_products.id
            WHERE offers_offer_actions_xref.offer_id = '" . $offer['id'] . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $offer_actions = array();

        while ($row = mysqli_fetch_assoc($result)) {
            $offer_actions[] = $row;
        }
        
        $discounted_order_items = array();
        
        // get subtotal
        // we need the subtotal because we need to keep track of how the subtotal is changed as order items are discounted
        // in order for us to determine if offers are valid
        $subtotal = get_order_subtotal();
        
        // loop through the offer actions in order to determine order item discounts
        // we check for order item discounts first, because they will affect the subtotal which could affect the validity of other types of offers
        foreach ($offer_actions as $offer_action) {
            // if this offer action discounts order items, then continue
            if (
                ($offer_action['type'] == 'discount product')
                && ($offer_action['discount_product_id'] != '')
                &&
                (
                    ($offer_action['discount_product_amount'] != 0)
                    || ($offer_action['discount_product_percentage'] != 0)
                )
            ) {
                $scope = '';
                
                // if this offer should be applied at the order level then remember that
                if (
                    (ECOMMERCE_SHIPPING == FALSE)
                    || (ECOMMERCE_RECIPIENT_MODE == 'single recipient')
                    || ($offer['scope'] == 'order')
                    || ($offer_action['discount_product_shippable'] == 0)
                ) {
                    $scope = 'order';
                
                // else this offer should be applied at the recipient level, so remember that
                } else {
                    $scope = 'recipient';
                }
                
                // if the scope is order and the offer is valid or the scope is recipient,
                // then determine discount amount
                if (
                    (
                        ($scope == 'order')
                        && (validate_offer($offer['id'], 0, $subtotal) == TRUE)
                    )
                    || ($scope == 'recipient')
                ) {
                    // loop through ship tos in order to loop through order items
                    foreach ($ship_tos as $ship_to) {
                        // if the scope is recipient and the offer is valid for this recipient or the scope is order,
                        // then determine discount amount
                        if (
                            (
                                ($scope == 'recipient')
                                && (validate_offer($offer['id'], $ship_to['id'], $subtotal) == TRUE)
                            )
                            || ($scope == 'order')
                        ) {
                            // loop through order items for this ship to in order to determine discount amount
                            foreach ($ship_to['order_items'] as $order_item) {
                                // if this offer discounts this product
                                // and this order item was not added by an offer,
                                // then determine discount
                                if (
                                    ($offer_action['discount_product_product_id'] == $order_item['product_id'])
                                    && ($order_item['added_by_offer'] == 0)
                                ) {
                                    $discounted_price = 0;
                                    
                                    // if discount is by amount, then get discounted price
                                    if ($offer_action['discount_product_amount']) {
                                        $discounted_price = $order_item['product_price'] - $offer_action['discount_product_amount'];
                                    
                                    // else discount is by percentage, so get discounted price
                                    } else {
                                        $discounted_price = $order_item['product_price'] - ($order_item['product_price'] * ($offer_action['discount_product_percentage'] / 100));
                                    }
                                    
                                    // if the discounted price is less than 0, then set discounted price to 0
                                    if ($discounted_price < 0) {
                                        $discounted_price = 0;
                                    }
                                    
                                    // if this offer is going to discount the product more than any previous offer so far, use this offer
                                    if (
                                        (isset($discounted_order_items[$order_item['id']]) == FALSE)
                                        || ($discounted_price < $discounted_order_items[$order_item['id']]['price'])
                                    ) {
                                        // if this is the first offer that has discounted this order item, then prepare array
                                        if (isset($discounted_order_items[$order_item['id']]) == FALSE) {
                                            $discounted_order_items[$order_item['id']] = array();
                                        }
                                        
                                        $discounted_order_items[$order_item['id']]['price'] = $discounted_price;
                                        
                                        // add previous discount for this order item (if one exists) back to subtotal
                                        $subtotal = $subtotal + $discounted_order_items[$order_item['id']]['discount'];
                                        
                                        // set new discount for this order item
                                        $discounted_order_items[$order_item['id']]['discount'] = ($order_item['product_price'] - $discounted_price) * $order_item['quantity'];
                                        
                                        // reduce subtotal by new discount for this order item
                                        // we do this so when we validate other offers, we will be validating with the most recent subtotal that we know about
                                        $subtotal = $subtotal - $discounted_order_items[$order_item['id']]['discount'];
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        
        $best_order_discount = 0;
        $add_product_discount = 0;
        $discounted_ship_tos = array();
        $number_of_added_products = 0;
        
        // loop through the offer actions in order to determine the discount amount for discount order, add product, and/or discount shipping actions
        foreach ($offer_actions as $offer_action) {
            // determine discount amount differently based on the type of action
            switch ($offer_action['type']) {
                case 'discount order':
                    // if this action actually discounts the order,
                    // then continue to determine discount amount
                    if (
                        ($offer_action['discount_order_amount'] != 0)
                        || ($offer_action['discount_order_percentage'] != 0)
                    ) {
                        // if this offer is valid,
                        // then determine discount amount
                        if (validate_offer($offer['id'], 0, $subtotal) == TRUE) {
                            $order_discount = 0;
                            
                            // if discount is by amount, get order discount
                            if ($offer_action['discount_order_amount'] != 0) {
                                $order_discount = $offer_action['discount_order_amount'];
                                
                            // else discount is by percentage, so get order discount
                            } else {
                                $order_discount = round($subtotal * ($offer_action['discount_order_percentage'] / 100));
                            }
                            
                            // if the order discount is greater than the subtotal, then set the order discount to the subtotal
                            if ($order_discount > $subtotal) {
                                $order_discount = $subtotal;
                            }
                            
                            // if this is the best order discount, then remember that
                            if ($order_discount > $best_order_discount) {
                                $best_order_discount = $order_discount;
                            }
                        }
                    }
                    
                    break;
                    
                case 'add product':
                    // If this action actually adds a product, then continue to determine discount
                    // amount and number of products added.
                    if (
                        ($offer_action['add_product_id'] != '')
                        && ($offer_action['add_product_quantity'] != 0)
                    ) {
                        // if this offer should be applied at the order level then continue to determine discount amount
                        if (
                            (ECOMMERCE_SHIPPING == FALSE)
                            || (ECOMMERCE_RECIPIENT_MODE == 'single recipient')
                            || ($offer['scope'] == 'order')
                            || ($offer_action['add_product_shippable'] == 0)
                        ) {
                            // if this offer is valid at the order level, then continue to to determine discount amount
                            if (validate_offer($offer['id'], 0, $subtotal) == TRUE) {

                                // If the product is discounted then get discount.
                                if (
                                    $offer_action['add_product_discount_amount']
                                    or $offer_action['add_product_discount_percentage']
                                ) {

                                    $discounted_price = 0;
                                    
                                    // if discount is by amount, then get discounted price
                                    if ($offer_action['add_product_discount_amount']) {
                                        $discounted_price = $offer_action['add_product_price'] - $offer_action['add_product_discount_amount'];
                                    
                                    // else discount is by percentage, so get discounted price
                                    } else {
                                        $discounted_price = $offer_action['add_product_price'] - ($offer_action['add_product_price'] * ($offer_action['add_product_discount_percentage'] / 100));
                                    }
                                    
                                    // if the discounted price is less than 0, then set discounted price to 0
                                    if ($discounted_price < 0) {
                                        $discounted_price = 0;
                                    }
                                    
                                    // add discount for this added product to add product discount (multiply discount by quantity)
                                    $add_product_discount = $add_product_discount + (($offer_action['add_product_price'] - $discounted_price) * $offer_action['add_product_quantity']);
                                }

                                $number_of_added_products += $offer_action['add_product_quantity'];
                            }
                            
                        // else this offer should be applied at the recipient level, so continue to determine discount amount for added product
                        } else {
                            // loop through recipients in order to determine the discount amount for the added product
                            foreach ($ship_tos as $ship_to) {
                                // if offer is valid for this recipient then continue to determine the discount amount for the added product
                                if (validate_offer($offer['id'], $ship_to['id'], $subtotal) == TRUE) {

                                    // If the product is discounted then get discount.
                                    if (
                                        $offer_action['add_product_discount_amount']
                                        or $offer_action['add_product_discount_percentage']
                                    ) {

                                        $discounted_price = 0;
                                        
                                        // if discount is by amount, then get discounted price
                                        if ($offer_action['add_product_discount_amount']) {
                                            $discounted_price = $offer_action['add_product_price'] - $offer_action['add_product_discount_amount'];
                                        
                                        // else discount is by percentage, so get discounted price
                                        } else {
                                            $discounted_price = $offer_action['add_product_price'] - ($offer_action['add_product_price'] * ($offer_action['add_product_discount_percentage'] / 100));
                                        }
                                        
                                        // if the discounted price is less than 0, then set discounted price to 0
                                        if ($discounted_price < 0) {
                                            $discounted_price = 0;
                                        }
                                        
                                        // add discount for this added product to discount amount (multiply discount by quantity)
                                        $add_product_discount = $add_product_discount + (($offer_action['add_product_price'] - $discounted_price) * $offer_action['add_product_quantity']);
                                    }

                                    $number_of_added_products += $offer_action['add_product_quantity'];
                                    
                                    // if this offer is not allowed to be applied to multiple recipients,
                                    // then we don't need to loop through any more recipients so break out of the loop
                                    if ($offer['multiple_recipients'] == 0) {
                                        break;
                                    }
                                }
                            }
                        }
                    }
                    
                    break;
                    
                case 'discount shipping':
                    // if shipping is on and there is an actual discount shipping percentage, then continue to determine discount amount
                    if (
                        (ECOMMERCE_SHIPPING == true)
                        && ($offer_action['discount_shipping_percentage'] != 0)
                    ) {
                        $scope = '';
                        
                        // if this offer should be applied at the order level then remember that
                        if (
                            (ECOMMERCE_RECIPIENT_MODE == 'single recipient')
                            || ($offer['scope'] == 'order')
                        ) {
                            $scope = 'order';
                        
                        // else this offer should be applied at the recipient level, so remember that
                        } else {
                            $scope = 'recipient';
                        }
                        
                        // if the scope is order and the offer is valid or the scope is recipient,
                        // then continue to determine discount amount
                        if (
                            (
                                ($scope == 'order')
                                && (validate_offer($offer['id'], 0, $subtotal) == TRUE)
                            )
                            || ($scope == 'recipient')
                        ) {
                            // loop through the ship tos in order to determine discount amount
                            foreach ($ship_tos as $ship_to) {
                                // if this is a real ship to,
                                // and the ship to is complete,
                                // and if the scope is recipient and the offer is valid for this recipient,
                                // or the scope is order,
                                // then determine discount amount
                                if (
                                    ($ship_to['id'] != 0)
                                    && ($ship_to['complete'] == 1)
                                    &&
                                    (
                                        (
                                            ($scope == 'recipient')
                                            && (validate_offer($offer['id'], $ship_to['id'], $subtotal) == TRUE)
                                        )
                                        || ($scope == 'order')
                                    )
                                ) {
                                    // check if this offer and action is valid for the shipping method
                                    $query =
                                        "SELECT offer_action_id
                                        FROM offer_actions_shipping_methods_xref
                                        WHERE
                                            (offer_action_id = '" . $offer_action['id'] . "')
                                            AND (shipping_method_id = '" . $ship_to['shipping_method_id'] . "')";
                                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                    
                                    // if this offer and action is valid for the shipping method then determine discount amount
                                    if (mysqli_num_rows($result) != 0) {
                                        // if there is an original shipping cost, then use that for the original shipping cost
                                        if ($ship_to['original_shipping_cost'] != 0) {
                                            $original_shipping_cost = $ship_to['original_shipping_cost'];
                                            
                                        // else there is not an original shipping cost, so use shipping cost for original shipping cost
                                        } else {
                                            $original_shipping_cost = $ship_to['shipping_cost'];
                                        }
                                        
                                        $discounted_shipping_cost = $original_shipping_cost - ($original_shipping_cost * ($offer_action['discount_shipping_percentage'] / 100));
                                        
                                        // if this offer is going to discount the ship to more than any previous offer so far, use this offer
                                        if (
                                            (isset($discounted_ship_tos[$ship_to['id']]) == FALSE)
                                            || ($discounted_shipping_cost < $discounted_ship_tos[$ship_to['id']]['shipping_cost'])
                                        ) {
                                            // if this is the first offer that has discounted this ship to, then prepare array
                                            if (isset($discounted_ship_tos[$ship_to['id']]) == FALSE) {
                                                $discounted_ship_tos[$ship_to['id']] = array();
                                            }
                                            
                                            $discounted_ship_tos[$ship_to['id']]['shipping_cost'] = $discounted_shipping_cost;
                                            
                                            // set new discount for this order item
                                            $discounted_ship_tos[$ship_to['id']]['discount'] = $original_shipping_cost - $discounted_shipping_cost;
                                        }
                                    }
                                }
                            }
                        }
                    }
                    
                    break;
            }
        }
        
        // start discount amount off with best order discount for this offer
        $discount_amount = $best_order_discount;
        
        // loop through the discounted order items for this offer in order to add to discount amount
        foreach ($discounted_order_items as $discounted_order_item) {
            $discount_amount += $discounted_order_item['discount'];
        }
        
        // add the add product discount to the discount amount
        $discount_amount += $add_product_discount;
        
        // loop through the discounted ship tos for this offer in order to add to discount amount
        foreach ($discounted_ship_tos as $discounted_ship_to) {
            $discount_amount += $discounted_ship_to['discount'];
        }
        
        // If this offer is the first offer in this group or if the discount amount for this offer
        // is the largest discount amount, or if this offer has the same discount however it adds
        // more products, then remember that this is the best offer so far.  We take into account
        // added products, because sometimes an offer will add a free product, where the price of
        // the product is already zero, and is not discounted by the offer.  Therefore, we want
        // to give some credit to the offer as possibly being the best offer if it adds products.
        // A discount has the highest importance, and the number of added products has a
        // secondary importance.
        if (
            ($best_offer_id == 0)
            or ($discount_amount > $largest_discount_amount)
            or (
                $discount_amount == $largest_discount_amount
                and $number_of_added_products > $largest_number_of_added_products
            )
        ) {
            $largest_discount_amount = $discount_amount;
            $largest_number_of_added_products = $number_of_added_products;
            $best_offer_id = $offer['id'];
        }
    }
    
    return $best_offer_id;
}

function get_best_shipping_discount_offer($ship_to_id, $shipping_method_id) {

    // get special offer code that shopper might have entered
    $query = "SELECT special_offer_code FROM orders WHERE id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $special_offer_code = $row['special_offer_code'];
    
    $offer_code = '';
    
    // if there is a special offer code for this order, then get offer code because the special offer code might just be a key code
    if ($special_offer_code != '') {
        $offer_code = get_offer_code_for_special_offer_code($special_offer_code);
    }
    
    // get all active offers so we can find the best shipping discount
    $query =
        "SELECT
            offers.id,
            offers.code,
            offers.only_apply_best_offer
        FROM offers
        WHERE
            (offers.status = 'enabled')
            AND (offers.start_date <= CURRENT_DATE())
            AND (CURRENT_DATE() <= offers.end_date)
            AND ((offers.require_code = 0) OR (offers.code = '" . escape($offer_code) . "'))
        ORDER BY offers.code ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $offers = array();

    while ($row = mysqli_fetch_assoc($result)) {
        $offers[] = $row;
    }
    
    // prepare to get best offers (including non-competing offers and competing offers that are the best offer)
    // competing offers are offers that share the same code but where only the best offer will be applied
    $best_offers = array();
    
    // create array to store the best offer id's for offer codes, so we don't have to look up the best offer multiple times
    $best_offer_ids = array();
    
    // loop through the offers in order to get best offers
    foreach ($offers as $key => $offer) {
        $previous_offer_code = '';
    
        // if this is not the first offer, then get previous offer code
        if ($key != 0) {
            $previous_offer_code = $offers[$key - 1]['code'];
        }
        
        $next_offer_code = '';
    
        // if this is not the last offer, then get next offer code
        if ($key != (count($offers) - 1)) {
            $next_offer_code = $offers[$key + 1]['code'];
        }
    
        // if this is not the first offer and the previous offer had the same code,
        // or if this is not the last offer and the next offer had the same code,
        // and if this offer is set so only the best offer is applied,
        // then this is a competing offer, so determine if offer is the best offer
        if (
            (
                ($key != 0) && ($offer['code'] == $previous_offer_code)
                || ($key != (count($offers) - 1)) && ($offer['code'] == $next_offer_code)
            )
            && ($offer['only_apply_best_offer'] == 1)
        ) {
            $best_offer_id = '';
            
            // if the best offer id has already been found for this offer code, then get it
            if (isset($best_offer_ids[$offer['code']]) == TRUE) {
                $best_offer_id = $best_offer_ids[$offer['code']];
                
            // else the best offer id has not already been found for this offer code, so get it
            // and then remember it by adding to array
            } else {
                $best_offer_id = get_best_offer_id($offer['code']);
                $best_offer_ids[$offer['code']] = $best_offer_id;
            }
            
            // if this offer is the best offer, then add it to array
            if ($offer['id'] == $best_offer_id) {
                $best_offers[] = $offer;
            }
            
        // else this is not a competing offer, so it is a best offer, so add it to array
        } else {
            $best_offers[] = $offer;
        }
    }
    
    // set active offers to all the best offers
    $offers = $best_offers;
    
    $best_offer_id = 0;
    $best_discount_shipping_percentage = 0;
    
    // loop through all active offers so we can find the best shipping discount
    foreach ($offers as $offer) {
        // if the offer is valid for this recipient, then continue to check if this offer has the best shipping discount
        if (validate_offer($offer['id'], $ship_to_id) == TRUE) {
            // get offer actions for this offer that discount shipping and that are valid for this shipping method
            $query =
                "SELECT offer_actions.discount_shipping_percentage
                FROM offers_offer_actions_xref
                LEFT JOIN offer_actions ON offers_offer_actions_xref.offer_action_id = offer_actions.id
                LEFT JOIN offer_actions_shipping_methods_xref ON offer_actions.id = offer_actions_shipping_methods_xref.offer_action_id
                WHERE
                    (offers_offer_actions_xref.offer_id = '" . $offer['id'] . "')
                    AND (offer_actions.type = 'discount shipping')
                    AND (offer_actions.discount_shipping_percentage != '0')
                    AND (offer_actions_shipping_methods_xref.shipping_method_id = '" . escape($shipping_method_id) . "')";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            $offer_actions = array();

            while ($row = mysqli_fetch_assoc($result)) {
                $offer_actions[] = $row;
            }
            
            // loop through the offer actions in order to find the best shipping discount
            foreach ($offer_actions as $offer_action) {
                // if this offer provides the best shipping discount so far then remember that
                if ($offer_action['discount_shipping_percentage'] > $best_discount_shipping_percentage) {
                    $best_offer_id = $offer['id'];
                    $best_discount_shipping_percentage = $offer_action['discount_shipping_percentage'];
                }
            }
        }
    }
    
    // if an offer was found, then return it
    if ($best_discount_shipping_percentage > 0) {
        return array(
            'id' => $best_offer_id,
            'discount_shipping_percentage' => $best_discount_shipping_percentage
        );
        
    // else an offer was not found, so return false
    } else {
        return FALSE;
    }
}

// create function that will check if there are any active shipping discounts
// we do this in order to improve performance by not running unnecessary queries
function check_if_active_shipping_discount_offer_exists()
{
    // get special offer code that shopper might have entered
    $query = "SELECT special_offer_code FROM orders WHERE id = '" . $_SESSION['ecommerce']['order_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $special_offer_code = $row['special_offer_code'];
    
    $offer_code = '';
    
    // if there is a special offer code for this order, then get offer code because the special offer code might just be a key code
    if ($special_offer_code != '') {
        $offer_code = get_offer_code_for_special_offer_code($special_offer_code);
    }
    
    $query =
        "SELECT COUNT(offers_offer_actions_xref.offer_action_id)
        FROM offers_offer_actions_xref
        LEFT JOIN offers ON offers_offer_actions_xref.offer_id = offers.id
        LEFT JOIN offer_actions ON offers_offer_actions_xref.offer_action_id = offer_actions.id
        WHERE
            (offers.status = 'enabled')
            AND (offers.start_date <= CURRENT_DATE())
            AND (CURRENT_DATE() <= offers.end_date)
            AND ((offers.require_code = 0) OR (offers.code = '" . escape($offer_code) . "'))
            AND (offer_actions.type = 'discount shipping')
            AND (offer_actions.discount_shipping_percentage != '0')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_row($result);
    
    // if there is at least one active shipping discount offer, then return that
    if ($row[0] > 0) {
        return TRUE;
        
    // else there is not an active shipping discount offer, so return that
    } else {
        return FALSE;
    }
}

function add_pending_offers($liveform) {

    // if pending offers form was submitted
    if ($_POST['pending_offers']) {
        // loop through all submitted fields in order to determine if a pending offer was requested to be added
        foreach ($_POST as $key => $value) {
            // if the name of the field starts with "add_pending_offer_",
            // then a pending offer was requested to be added, so continue to add pending offers
            if (mb_substr($key, 0, 18) == 'add_pending_offer_') {
                // get the offer id and offer action id from the field name
                $offer_id_and_offer_action_id = mb_substr($key, 18);
                $offer_id_and_offer_action_id_parts = explode('_', $offer_id_and_offer_action_id);
                $offer_id = $offer_id_and_offer_action_id_parts[0];
                $offer_action_id = $offer_id_and_offer_action_id_parts[1];
                
                // get special offer code for this order
                $query = "SELECT special_offer_code FROM orders WHERE id = '" . $_SESSION['ecommerce']['order_id'] . "'";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                $row = mysqli_fetch_assoc($result);
                $special_offer_code = $row['special_offer_code'];
                
                $offer_code = '';
                
                // if there is a special offer code for this order, then get offer code because the special offer code might just be a key code
                if ($special_offer_code != '') {
                    $offer_code = get_offer_code_for_special_offer_code($special_offer_code);
                }
                
                // get offer and action info for the pending offer that the customer requested to add
                $query =
                    "SELECT
                        offer_actions.id,
                        offer_actions.add_product_product_id,
                        offer_actions.add_product_quantity,
                        offer_actions.add_product_discount_amount,
                        offer_actions.add_product_discount_percentage,
                        offers.id AS offer_id,
                        offers.scope AS offer_scope,
                        offers.multiple_recipients AS offer_multiple_recipients,
                        add_products.shippable AS add_product_shippable,
                        add_products.price AS add_product_price,
                        add_products.name AS add_product_name
                    FROM offers_offer_actions_xref
                    LEFT JOIN offer_actions ON offers_offer_actions_xref.offer_action_id = offer_actions.id
                    LEFT JOIN offers ON offers_offer_actions_xref.offer_id = offers.id
                    LEFT JOIN products AS add_products ON offer_actions.add_product_product_id = add_products.id
                    WHERE
                        (offer_actions.id = '" . escape($offer_action_id) . "')
                        AND (offers.id = '" . escape($offer_id) . "')
                        AND (offers.status = 'enabled')
                        AND (offers.start_date <= CURRENT_DATE())
                        AND (CURRENT_DATE() <= offers.end_date)
                        AND ((offers.require_code = 0) OR (offers.code = '" . escape($offer_code) . "'))
                        AND (offer_actions.type = 'add product')
                        AND (add_products.id IS NOT NULL)
                        AND (offer_actions.add_product_quantity != '0')";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                
                // if an offer was not found
                if (mysqli_num_rows($result) == 0) {
                    $liveform->mark_error('pending_offer', 'The offer that you tried to add no longer exists or is no longer active.');
                }
                
                // if there are no errors so far, continue
                if ($liveform->check_form_errors() == FALSE) {
                    $offer_action = mysqli_fetch_assoc($result);
                    
                    // if shipping is on, recipient mode is multi-recipient, and the offer is adding a product that is shippable
                    // make sure that a recipient was selected or entered
                    if (
                        (ECOMMERCE_SHIPPING == TRUE)
                        && (ECOMMERCE_RECIPIENT_MODE == 'multi-recipient')
                        && ($offer_action['add_product_shippable'] == 1)
                    ) {
                        $ship_to = $_POST['pending_offer_' . $offer_action['offer_id'] . '_' . $offer_action['id'] . '_ship_to'];
                        $add_name = $_POST['pending_offer_' . $offer_action['offer_id'] . '_' . $offer_action['id'] . '_add_name'];
                        
                        if ($add_name == 'or add name') {
                            $add_name = '';
                        }
                        
                        // if ship to was not selected and a name was not entered, then prepare error
                        if (!$ship_to && !$add_name) {
                            $liveform->mark_error('pending_offer_' . $offer_action['offer_id'] . '_' . $offer_action['id'] . '_ship_to', 'The offer that you attempted to add requires a recipient.');
                            $liveform->mark_error('pending_offer_' . $offer_action['offer_id'] . '_' . $offer_action['id'] . '_add_name', '');
                            $liveform->assign_field_value('pending_offer_' . $offer_action['offer_id'] . '_' . $offer_action['id'] . '_add_name', 'or add name');
                            
                        // else get ship to id
                        } else {
                            if ($add_name) {
                                $ship_to_name = $add_name;
                            } else {
                                $ship_to_name = $ship_to;
                            }
                            
                            // get ship to id for ship to name
                            $query = "SELECT id FROM ship_tos WHERE (order_id = '" . escape($_SESSION['ecommerce']['order_id']) . "') AND (ship_to_name = '" . escape($ship_to_name) . "')";
                            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                            $row = mysqli_fetch_assoc($result);
                            
                            $ship_to_id = $row['id'];
                        }
                    }
                    
                    // if there are no errors so far, continue
                    if ($liveform->check_form_errors() == FALSE) {
                        // if this offer should be applied at the order level then continue to add pending offer
                        if (
                            (ECOMMERCE_SHIPPING == FALSE)
                            || (ECOMMERCE_RECIPIENT_MODE == 'single recipient')
                            || ($offer_action['offer_scope'] == 'order')
                            || ($offer_action['add_product_shippable'] == 0)
                        ) {
                            // if offer is valid for order
                            if (validate_offer($offer_action['offer_id']) == TRUE) {
                                // check if this offer and action have already been applied to order
                                $query =
                                    "SELECT id
                                    FROM order_items
                                    WHERE
                                        (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                                        AND (offer_id = '" . $offer_action['offer_id'] . "')
                                        AND (offer_action_id = '" . $offer_action['id'] . "')";
                                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                
                                // if this offer and action have not already been applied to this order, add pending offer
                                if (mysqli_num_rows($result) == 0) {
                                    $discounted_price = 0;
                                    
                                    // if discount is by amount, get discounted price
                                    if ($offer_action['add_product_discount_amount'] != 0) {
                                        $discounted_price = $offer_action['add_product_price'] - $offer_action['add_product_discount_amount'];
                                    
                                    // else discount is by percentage, get discounted price
                                    } else {
                                        $discounted_price = $offer_action['add_product_price'] - ($offer_action['add_product_price'] * ($offer_action['add_product_discount_percentage'] / 100));
                                    }
                                    
                                    // if product that is being added is also being discounted, prepare to mark order item accordingly
                                    if (($offer_action['add_product_discount_amount'] > 0) || ($offer_action['add_product_discount_percentage'] > 0)) {                                
                                        $discounted_by_offer = 1;
                                    } else {
                                        $discounted_by_offer = 0;
                                    }
                                    
                                    if ((ECOMMERCE_SHIPPING == true) && $offer_action['add_product_shippable']) {
                                        $ship_to_id = create_or_get_ship_to($ship_to, $add_name);
                                    }
                                    
                                    $query =
                                        "INSERT INTO order_items (
                                            order_id,
                                            ship_to_id,
                                            product_id,
                                            product_name,
                                            quantity,
                                            price,
                                            offer_id,
                                            offer_action_id,
                                            added_by_offer,
                                            discounted_by_offer
                                        ) VALUES (
                                            '" . $_SESSION['ecommerce']['order_id'] . "',
                                            '$ship_to_id',
                                            '" . $offer_action['add_product_product_id'] . "',
                                            '" . escape($offer_action['add_product_name']) . "',
                                            '" . $offer_action['add_product_quantity'] . "',
                                            '$discounted_price',
                                            '" . $offer_action['offer_id'] . "',
                                            '" . $offer_action['id'] . "',
                                            1,
                                            $discounted_by_offer)";
                                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                    $order_item_id = mysqli_insert_id(db::$con);
                                    
                                    // if there is a ship to for this order item
                                    if ($ship_to_id) {
                                        // get ship to information
                                        $query =
                                            "SELECT
                                                complete,
                                                country,
                                                state
                                            FROM ship_tos
                                            WHERE id = '" . escape($ship_to_id) . "'";
                                        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                        $row = mysqli_fetch_assoc($result);
                                        
                                        $complete = $row['complete'];
                                        $country_code = $row['country'];
                                        $state_code = $row['state'];
                                        
                                        // if ship to is complete, then do some checks
                                        if ($complete == 1) {
                                            // if order item is valid for destination and arrival date, then update shipping for ship to
                                            if ((validate_product_for_destination($offer_action['add_product_product_id'], $country_code, $state_code) == TRUE) && (validate_order_item_for_arrival_date($order_item_id) == TRUE)) {
                                                require_once(dirname(__FILE__) . '/shipping.php');
                                                update_shipping_cost_for_ship_to($ship_to_id);
                                            
                                            // else there are problems, so mark ship to as incomplete
                                            } else {
                                                $query = "UPDATE ship_tos SET complete = 0 WHERE id = '" . escape($ship_to_id) . "'";
                                                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                            }
                                        }
                                    }
                                }
                            }
                            
                        // else this offer should be applied at the recipient level, so continue to add pending offer
                        } else {
                            // check if this offer and action have already been applied to order
                            $query =
                                "SELECT id
                                FROM order_items
                                WHERE
                                    (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                                    AND (offer_id = '" . $offer_action['offer_id'] . "')
                                    AND (offer_action_id = '" . $offer_action['id'] . "')";
                            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                            
                            // if offer has not already been applied to order or offer is allowed to be applied to multiple recipients, then continue
                            if ((mysqli_num_rows($result) == 0) || ($offer_action['offer_multiple_recipients'] == 1)) {
                                // if offer is valid for this recipient
                                if (validate_offer($offer_action['offer_id'], $ship_to_id) == TRUE) {
                                    // check if offer and action have already been applied to this recipient
                                    $query =
                                        "SELECT id
                                        FROM order_items
                                        WHERE
                                            (order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                                            AND (ship_to_id = '$ship_to_id')
                                            AND (offer_id = '" . $offer_action['offer_id'] . "')
                                            AND (offer_action_id = '" . $offer_action['id'] . "')";
                                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                    
                                    // if offer has not already been applied to this recipient, then add pending offer
                                    if (mysqli_num_rows($result) == 0) {
                                        $discounted_price = 0;
                                        
                                        // if discount is by amount, get discounted price
                                        if ($offer_action['add_product_discount_amount'] != 0) {
                                            $discounted_price = $offer_action['add_product_price'] - $offer_action['add_product_discount_amount'];
                                        
                                        // else discount is by percentage, get discounted price
                                        } else {
                                            $discounted_price = $offer_action['add_product_price'] - ($offer_action['add_product_price'] * ($offer_action['add_product_discount_percentage'] / 100));
                                        }
                                        
                                        // if product that is being added is also being discounted, prepare to mark order item accordingly
                                        if (($offer_action['add_product_discount_amount'] > 0) || ($offer_action['add_product_discount_percentage'] > 0)) {                                
                                            $discounted_by_offer = 1;
                                        } else {
                                            $discounted_by_offer = 0;
                                        }
                                        
                                        $query =
                                            "INSERT INTO order_items (
                                                order_id,
                                                ship_to_id,
                                                product_id,
                                                product_name,
                                                quantity,
                                                price,
                                                offer_id,
                                                offer_action_id,
                                                added_by_offer,
                                                discounted_by_offer
                                            ) VALUES (
                                                '" . $_SESSION['ecommerce']['order_id'] . "',
                                                '$ship_to_id',
                                                '" . $offer_action['add_product_product_id'] . "',
                                                '" . escape($offer_action['add_product_name']) . "',
                                                '" . $offer_action['add_product_quantity'] . "',
                                                '$discounted_price',
                                                '" . $offer_action['offer_id'] . "',
                                                '" . $offer_action['id'] . "',
                                                1,
                                                $discounted_by_offer)";
                                        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                        $order_item_id = mysqli_insert_id(db::$con);
                                        
                                        // if there is a ship to for this order item
                                        if ($ship_to_id) {
                                            // get ship to information
                                            $query =
                                                "SELECT
                                                    complete,
                                                    country,
                                                    state
                                                FROM ship_tos
                                                WHERE id = '" . escape($ship_to_id) . "'";
                                            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                            $row = mysqli_fetch_assoc($result);
                                            
                                            $complete = $row['complete'];
                                            $country_code = $row['country'];
                                            $state_code = $row['state'];
                                            
                                            // if ship to is complete, then do some checks
                                            if ($complete == 1) {
                                                // if order item is valid for destination and arrival date, then update shipping for ship to
                                                if ((validate_product_for_destination($offer_action['add_product_product_id'], $country_code, $state_code) == TRUE) && (validate_order_item_for_arrival_date($order_item_id) == TRUE)) {
                                                    require_once(dirname(__FILE__) . '/shipping.php');
                                                    update_shipping_cost_for_ship_to($ship_to_id);
                                                
                                                // else there are problems, so mark ship to as incomplete
                                                } else {
                                                    $query = "UPDATE ship_tos SET complete = 0 WHERE id = '" . escape($ship_to_id) . "'";
                                                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                
                header('Location: ' . URL_SCHEME . HOSTNAME . PATH . get_page_name($_POST['page_id']));
                exit();
            }
        }
    }
}

function select_month($month) {
    $months = array();
    
    $months[] = array('name' => 'January', 'value' => '01');
    $months[] = array('name' => 'February', 'value' => '02');
    $months[] = array('name' => 'March', 'value' => '03');
    $months[] = array('name' => 'April', 'value' => '04');
    $months[] = array('name' => 'May', 'value' => '05');
    $months[] = array('name' => 'June', 'value' => '06');
    $months[] = array('name' => 'July', 'value' => '07');
    $months[] = array('name' => 'August', 'value' => '08');
    $months[] = array('name' => 'September', 'value' => '09');
    $months[] = array('name' => 'October', 'value' => '10');
    $months[] = array('name' => 'November', 'value' => '11');
    $months[] = array('name' => 'December', 'value' => '12');

    foreach ($months as $key => $value) {
        // if this month is the selected month, select this option
        if ($months[$key]['value'] == $month) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output .= '<option value="' . $months[$key]['value'] . '"' . $selected . '>' . $months[$key]['name'] . '</option>';
    }

    return $output;
}

function select_day($day) {
    $days = array();
    
    $days[] = array('name' => '1', 'value' => '01');
    $days[] = array('name' => '2', 'value' => '02');
    $days[] = array('name' => '3', 'value' => '03');
    $days[] = array('name' => '4', 'value' => '04');
    $days[] = array('name' => '5', 'value' => '05');
    $days[] = array('name' => '6', 'value' => '06');
    $days[] = array('name' => '7', 'value' => '07');
    $days[] = array('name' => '8', 'value' => '08');
    $days[] = array('name' => '9', 'value' => '09');
    $days[] = array('name' => '10', 'value' => '10');
    $days[] = array('name' => '11', 'value' => '11');
    $days[] = array('name' => '12', 'value' => '12');
    $days[] = array('name' => '13', 'value' => '13');
    $days[] = array('name' => '14', 'value' => '14');
    $days[] = array('name' => '15', 'value' => '15');
    $days[] = array('name' => '16', 'value' => '16');
    $days[] = array('name' => '17', 'value' => '17');
    $days[] = array('name' => '18', 'value' => '18');
    $days[] = array('name' => '19', 'value' => '19');
    $days[] = array('name' => '20', 'value' => '20');
    $days[] = array('name' => '21', 'value' => '21');
    $days[] = array('name' => '22', 'value' => '22');
    $days[] = array('name' => '23', 'value' => '23');
    $days[] = array('name' => '24', 'value' => '24');
    $days[] = array('name' => '25', 'value' => '25');
    $days[] = array('name' => '26', 'value' => '26');
    $days[] = array('name' => '27', 'value' => '27');
    $days[] = array('name' => '28', 'value' => '28');
    $days[] = array('name' => '29', 'value' => '29');
    $days[] = array('name' => '30', 'value' => '30');
    $days[] = array('name' => '31', 'value' => '31');

    foreach ($days as $key => $value) {
        // if this day is the selected day, select this option
        if ($days[$key]['value'] == $day) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output .= '<option value="' . $days[$key]['value'] . '"' . $selected . '>' . $days[$key]['name'] . '</option>';
    }

    return $output;
}

function select_year($years, $year) {
    foreach ($years as $value) {
        // if this year is the selected year, select this option
        if ($value == $year) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output .= '<option value="' . $value . '"' . $selected . '>' . $value . '</option>';
    }

    return $output;
}

// output options for drop-down selection for selecting a user's role
function select_user_role($selected_user_role, $parent_user_role)
{
    $user_roles = array();
    
    // if parent user is an administrator then prepare certain roles for picklist
    if ($parent_user_role == 0) {
        $user_roles[] = array('value' => '0', 'name' => 'Administrator');
        $user_roles[] = array('value' => '1', 'name' => 'Designer');
    }

    $user_roles[] = array('value' => '2', 'name' => 'Manager');
    $user_roles[] = array('value' => '3', 'name' => 'User');

    $output = '';

    foreach ($user_roles as $user_role) {
        // if this user role is the selected user role, select this option
        if ($user_role['value'] == $selected_user_role) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        $output .= '<option value="' . $user_role['value'] . '"' . $selected . '>' . $user_role['name'] . '</option>';
    }

    return $output;
}

// output options for drop-down selection for selecting a contact group
function select_contact_group($contact_group_id = '', $user = '')
{
    $output = '';
    
    // get contact groups
    $query =
        "SELECT
           id,
           name
        FROM contact_groups
        ORDER BY name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    while ($row = mysqli_fetch_assoc($result)) {
        $id = $row['id'];
        $name = $row['name'];
        
        if ($id == $contact_group_id) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }

        // if access does not need to be validated
        // or contact group is the current selected contact group
        // or user has access to this contact group,
        // prepare to output contact group option
        if (!$user || $selected || (validate_contact_group_access($user, $id) == true)) {
            $output .= '<option value="' . $id . '"' . $selected . '>' . h($name) . '</option>';
        }
    }
    
    return $output;
}

// Prepare options for a pick list of offers.
function select_offer($offer_id = '') {
    $output = '';

    // Get all offers.
    $offers = db_items(
        "SELECT
            id,
            code,
            description
        FROM offers
        ORDER BY code ASC");
    
    $output = '';
    
    // Loop through offers in order to prepare options.
    foreach ($offers as $offer) {
        // If the user has access to commerce or this is the selected offer, then output it.
        if (USER_MANAGE_ECOMMERCE || ($offer['id'] == $offer_id)) {
            $selected = '';

            // If this offer is the selected offer, then select it by default.
            if ($offer['id'] == $offer_id) {
                $selected = ' selected="selected"';
            }
            
            // Start the label with the offer code.
            $output_label = h($offer['code']);
            
            // If there is a description, then add it to the label.
            if ($offer['description'] != '') {
                $output_label .= ' - ';
                
                // If the description is greater than 50 characters, then truncate it to 50 characters.
                if (mb_strlen($offer['description']) > 50) {
                    $offer['description'] = mb_substr($offer['description'], 0, 50) . '...';
                }
                
                $output_label .= $offer['description'];
            }
            
            $output .= '<option value="' . $offer['id'] . '"' . $selected . '>' . $output_label . '</option>';
        }
    }

    return $output;
}

function initialize_order() {

    // If the visitor already has an order in the session, then use it. We have already verified
    // the status of the order in init.php, to make sure it is still incomplete, so we don't need
    // to do that here.
    if ($_SESSION['ecommerce']['order_id']) {

        // If the user is logged in and not ghosting, then check if we should update user/contact
        // for order, because this function is run when an update is made to the order, and we want
        // to set the user/contact for an order when a user updates the order.
        if (USER_LOGGED_IN and !$_SESSION['software']['ghost']) {

            $order = db_item(
                "SELECT user_id, contact_id FROM orders
                WHERE id = '" . e($_SESSION['ecommerce']['order_id']) . "'");

            // If the order's user/contact info is different from this user, then update
            // user/contact info for order.
            if ($order['user_id'] != USER_ID or $order['contact_id'] != USER_CONTACT_ID) {
                db(
                    "UPDATE orders
                    SET
                        user_id = '" . e(USER_ID) . "',
                        contact_id = '" . e(USER_CONTACT_ID) . "'
                    WHERE id = '" . e($_SESSION['ecommerce']['order_id']) . "'");
            }
        }

    // Otherwise the visitor does not have an active order in session, so create order.
    } else {

        $reference_code = generate_order_reference_code();
        
        $offline_payment_allowed = '0';
        
        // if offline payment is on, and if only on specific orders is off, then set the allow offline payment to 1
        if ((ECOMMERCE_OFFLINE_PAYMENT == TRUE) && (ECOMMERCE_OFFLINE_PAYMENT_ONLY_SPECIFIC_ORDERS == FALSE)) {
            $offline_payment_allowed = '1';
        }
        
        $query = "INSERT INTO orders (
                    order_date,
                    last_modified_timestamp,
                    user_id,
                    contact_id,
                    reference_code,
                    tracking_code,
                    utm_source,
                    utm_medium,
                    utm_campaign,
                    utm_term,
                    utm_content,
                    currency_code,
                    affiliate_code,
                    http_referer,
                    ip_address,
                    offline_payment_allowed)
                 VALUES (
                    UNIX_TIMESTAMP(),
                    UNIX_TIMESTAMP(),
                    '" . e(USER_ID) . "',
                    '" . e(USER_CONTACT_ID) . "',
                    '$reference_code',
                    '" . escape(get_tracking_code()) . "',
                    '" . e($_SESSION['software']['utm_source']) . "',
                    '" . e($_SESSION['software']['utm_medium']) . "',
                    '" . e($_SESSION['software']['utm_campaign']) . "',
                    '" . e($_SESSION['software']['utm_term']) . "',
                    '" . e($_SESSION['software']['utm_content']) . "',
                    '" . escape(VISITOR_CURRENCY_CODE) . "',
                    '" . escape(get_affiliate_code()) . "',
                    '" . escape($_SESSION['software']['http_referer']) . "',
                    IFNULL(INET_ATON('" . escape($_SERVER['REMOTE_ADDR']) . "'), 0),
                    '" . $offline_payment_allowed . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

        // store order id in session
        $_SESSION['ecommerce']['order_id'] = mysqli_insert_id(db::$con);
        
        // if visitor tracking is on, update visitor record with order information, if visitor has not already created a previous order
        if (VISITOR_TRACKING == true) {
            $query = "UPDATE visitors
                     SET
                        order_id = '" . $_SESSION['ecommerce']['order_id'] . "',
                        order_created = '1',
                        stop_timestamp = UNIX_TIMESTAMP()
                     WHERE (id = '" . $_SESSION['software']['visitor_id'] . "') AND (order_created = '0')";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        }
    }
}

function update_visitor_page_data($page_name)
{
    // check to see if landing page name has already been set
    $query = "SELECT landing_page_name
             FROM visitors
             WHERE id = '" . $_SESSION['software']['visitor_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $existing_landing_page_name = $row['landing_page_name'];
    
    // if there is not an existing landing page name, prepare sql for landing page name update
    if (!$existing_landing_page_name) {
        $sql_landing_page_name = "landing_page_name = '" . escape($page_name) ."',";
    } else {
        $sql_landing_page_name = '';
    }
    
    // update visitor record
    $query = "UPDATE visitors
             SET
                $sql_landing_page_name
                page_views = page_views + 1,
                stop_timestamp = UNIX_TIMESTAMP()
             WHERE id = '" . $_SESSION['software']['visitor_id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
}

function get_page_type_properties($page_id, $page_type, $collection = '') {

    $page_type_table_name = str_replace(' ', '_', $page_type) . '_pages';

    $sql_collection = '';

    // If this is a form list or item view, then deal with collection.
    if ($page_type == 'form list view' or $page_type == 'form item view') {

        // If the collection was not passed, then assume A collection.
        if (!$collection) {
            $collection = 'a';
        }

        $sql_collection = "AND (collection = '$collection')";

    }

    return db_item(
        "SELECT *
        FROM $page_type_table_name
        WHERE
            (page_id = '" . e($page_id) . "')
            $sql_collection");

}

function check_for_page_type_properties($page_type)
{
    if (
        ($page_type == 'email a friend')
        || ($page_type == 'folder view')
        || ($page_type == 'photo gallery')
        || ($page_type == 'search results')
        || ($page_type == 'custom form')
        || ($page_type == 'custom form confirmation')
        || ($page_type == 'form list view')
        || ($page_type == 'form item view')
        || ($page_type == 'form view directory')
        || ($page_type == 'calendar view')
        || ($page_type == 'calendar event view')
        || ($page_type == 'catalog')
        || ($page_type == 'catalog detail')
        || ($page_type == 'express order')
        || ($page_type == 'order form')
        || ($page_type == 'shopping cart')
        || ($page_type == 'shipping address and arrival')
        || ($page_type == 'shipping method')
        || ($page_type == 'billing information')
        || ($page_type == 'order preview')
        || ($page_type == 'order receipt')
        || ($page_type == 'affiliate sign up form')
        || ($page_type == 'update address book')
    ) {
        return true;
    } else {
        return false;
    }
}

function create_or_update_page_type_record($page_type, $properties)
{
    $page_type_table_name = str_replace(' ', '_', $page_type) . '_pages';

    $sql_collection = "";

    // If this is a form list or item view, then deal with the collection.
    if ($page_type == 'form list view' or $page_type == 'form item view') {

        // If the collection is not set, then set to default collection A.
        if (!$properties['collection']) {
            $properties['collection'] = 'a';
        }

        $sql_collection = "AND (collection = '" . e($properties['collection']) . "')";

    }
    
    // check to see if there is already a page type record
    $query =
        "SELECT COUNT(*)
        FROM $page_type_table_name
        WHERE
            (page_id = '" . e($properties['page_id']) . "')
            $sql_collection";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_row($result);
    $number_of_records = $row[0];
    
    // if a record does not exist, create record
    if ($number_of_records == 0) {
        $count = 1;
        $columns = '';
        $values = '';
        
        foreach ($properties as $field => $value) {
            $columns .= $field;
            $values .= "'" . e($value) . "'";
            
            if ($count < count($properties)) {
                $columns .= ', ';
                $values .= ', ';
            }
                
            $count++;
        }
        
        $query = "INSERT INTO $page_type_table_name ($columns) VALUES ($values)";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    // else a record exists, so update record
    } else {
        $count = 1;
        $sql_update = '';
        
        foreach ($properties as $field => $value) {
            $sql_update .= "$field = '" . e($value) . "'";
            
            if ($count < count($properties)) {
                $sql_update .= ', ';
            }
                
            $count++;
        }
        
        $query =
            "UPDATE $page_type_table_name
            SET $sql_update
            WHERE
                (page_id = '" . e($properties['page_id']) . "')
                $sql_collection";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    }
}

function get_page_type_name($page_type)
{
    switch ($page_type) {
        case 'standard':
            return 'Standard';
            break;
            
        case 'change password':
            return 'Change Password';
            break;
        
        case 'set password':
            return 'Set Password';
            break;
        
        case 'email a friend':
            return 'E-mail a Friend';
            break;
        
        case 'error':
            return 'Error';
            break;

        case 'folder view':
            return 'Folder View';
            break;
        
        case 'forgot password':
            return 'Forgot Password';
            break;
        
        case 'login':
            return 'Login';
            break;
        
        case 'logout':
            return 'Logout';
            break;
        
        case 'photo gallery':
            return 'Photo Gallery';
            break;
        
        case 'membership confirmation':
            return 'Membership Confirmation';
            break;
        
        case 'membership entrance':
            return 'Membership Entrance';
            break;
        
        case 'my account':
            return 'My Account';
            break;
        
        case 'my account profile':
            return 'My Account Profile';
            break;
        
        case 'email preferences':
            return 'E-mail Preferences';
            break;
        
        case 'view order':
            return 'View Order';
            break;
        
        case 'update address book':
            return 'Update Address Book';
            break;

        case 'custom form':
            return 'Custom Form';
            break;
        
        case 'custom form confirmation':
            return 'Custom Form Confirmation';
            break;

        case 'form list view':
            return 'Form List View';
            break;
            
        case 'form item view':
            return 'Form Item View';
            break;
            
        case 'form view directory':
            return 'Form View Directory';
            break;
            
        case 'calendar view':
            return 'Calendar View';
            break;
            
        case 'calendar event view':
            return 'Calendar Event View';
            break;
            
        case 'catalog':
            return 'Catalog';
            break;
            
        case 'catalog detail':
            return 'Catalog Detail';
            break;
        
        case 'express order':
            return 'Express Order';
            break;
        
        case 'order form':
            return 'Order Form';
            break;
        
        case 'shopping cart':
            return 'Shopping Cart';
            break;
        
        case 'shipping address and arrival':
            return 'Shipping Address & Arrival';
            break;
        
        case 'shipping method':
            return 'Shipping Method';
            break;

        case 'billing information':
            return 'Billing Information';
            break;
        
        case 'order preview':
            return 'Order Preview';
            break;
        
        case 'order receipt':
            return 'Order Receipt';
            break;
        
        case 'registration confirmation':
            return 'Registration Confirmation';
            break;
        
        case 'registration entrance':
            return 'Registration Entrance';
            break;
        
        case 'search results':
            return 'Search Results';
            break;
            
        case 'affiliate sign up form':
            return 'Affiliate Sign Up Form';
            break;
            
        case 'affiliate sign up confirmation':
            return 'Affiliate Sign Up Confirmation';
            break;
            
        case 'affiliate welcome':
            return 'Affiliate Welcome';
            break;
    }
}

function get_currency_name_from_code($currency_code)
{
    // if the currency code is not blank, then get name from code
    if ($currency_code != '') {
        $query = "SELECT name FROM currencies WHERE code = '" . escape($currency_code) . "'";
        $result = mysqli_query(db::$con, $query) or output_error("Query failed.");
        $row = mysqli_fetch_assoc($result);
        return $row['name'];
        
    // else the currency code is blank, so return blank
    } else {
        return '';
    }
}

function get_login_region_content($login_region_id)
{
    // if the user is not logged in then prepare header, possibly body with form, and footer
    if (isset($_SESSION['sessionusername']) == false) {
        // get information from the database
        $query = 
            "SELECT 
                not_logged_in_header,
                login_form,
                not_logged_in_footer
            FROM login_regions
            WHERE id = '$login_region_id'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_assoc($result);
        
        $login_region_header = $row['not_logged_in_header'];
        $login_form = $row['login_form'];
        $login_region_footer = $row['not_logged_in_footer'];
        
        // assume that the login region body should be blank, until we find out otherwise
        $login_region_body = '';
        
        // if the login form is enabled, then output it
        if ($login_form == 1) {

            $liveform = new liveform('login');
            
            // if software is in secure mode, then make sure that form is submitted to a secure URL
            if (URL_SCHEME == 'https://') {
                $action_url = 'https://' . HOSTNAME . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/index.php';
                
            // else software is not in secure mode, so just use relative URL
            } else {
                $action_url = OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/index.php';
            }

            $output_remember_me = '';

            // if the remember me feature is enabled, then deal with it
            if (REMEMBER_ME == TRUE) {
                // if the form has not been submitted yet and the visitor checked remember me during the last login,
                // then check the remember me check box by default
                if (
                    ($liveform->field_in_session('email') == FALSE)
                    && ($_COOKIE['software']['remember_me'] == 'true')
                ) {
                    $liveform->assign_field_value('remember_me', '1');
                }

                $output_remember_me = '<div>' . $liveform->output_field(array('type'=>'checkbox', 'name'=>'remember_me', 'id'=>'remember_me', 'value'=>'1', 'class'=>'software_input_checkbox')) . '<label for="remember_me">Remember Me</label></div>';
            }
            
            $output_forgot_password_link = '';
            
            // if the forgot password link is enabled in the settings, output link
            if (FORGOT_PASSWORD_LINK == true) {
                $output_forgot_password_link = '<div><a class="forgot_button" href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/forgot_password.php?send_to=' . h(urlencode(get_request_uri())) . '">Forgot password?</a></div>' . "\n";
            }
            
            $login_region_body =
                $liveform->output_errors() . '
                <form name="login" action="' . $action_url . '" method="post" style="margin: 0px;">
                    ' . get_token_field() . '
                    <input type="hidden" name="login_region" id="login_region" value="true" />
                    <input type="hidden" name="send_to" id="send_to" value="' . h(get_request_uri()) . '" />
                    <input type="hidden" name="require_cookies" value="true" />
                    <div class="username_label"><label for="email">Email:</label></div>
                    <div>' .
                        $liveform->field(array(
                            'type' => 'email',
                            'id' => 'email',
                            'name' => 'email',
                            'class' => 'software_input_text',
                            'required' => 'true',
                            'autocomplete' => 'email',
                            'spellcheck' => 'false')) . '
                    </div>
                    <div class="password_label"><label for="password">Password:</label></div>
                    <div>' .
                        $liveform->field(array(
                            'type' => 'password',
                            'id' => 'password',
                            'name' => 'password',
                            'class' => 'software_input_password',
                            'required' => 'true',
                            'autocomplete' => 'current-password',
                            'spellcheck' => 'false')) . '
                    </div>
                    ' . $output_remember_me . '
                    <div style="margin-top: .5em;"><input type="submit" name="submit_login" value="Login" class="software_input_submit_primary login_button" /></div>
                    ' . $output_forgot_password_link . '
                </form>';
            
            $liveform->remove();
        }
        
    // else the user is logged in, so prepare header, body with username, and footer
    } else {
        // get header and footer information from the datbase
        $query = 
            "SELECT 
                logged_in_header,
                logged_in_footer
            FROM login_regions
            WHERE id = '$login_region_id'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_assoc($result);
        
        $login_region_header = $row['logged_in_header'];
        $login_region_footer = $row['logged_in_footer'];
        
        $output_badge = '';
        
        // Get badge info for user.
        $query = "SELECT user_badge AS badge, user_badge_label AS badge_label FROM user WHERE user_username = '" . escape($_SESSION['sessionusername']) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_assoc($result);
        $badge = $row['badge'];
        $badge_label = $row['badge_label'];
        
        // If badge is enabled for user and there is a badge label, then output badge.
        if (
            ($badge == 1)
            &&
            (
                ($badge_label != '')
                || (BADGE_LABEL != '')
            )
        ) {
            // If the user's badge label is blank, then use default label.
            if ($badge_label == '') {
                $badge_label = BADGE_LABEL;
            }

            $output_badge = ' <span class="software_badge ' . h(get_class_name($badge_label)) . '">' . h($badge_label) . '</span>';
        }
        
        $login_region_body = h($_SESSION['sessionusername']) . $output_badge;

    }
    
    $login_region_content = 
        '<div class="software_login_region">
            ' . $login_region_header . '
            ' . $login_region_body . '
            ' . $login_region_footer . '
        </div>';
    
    return $login_region_content;

}

function get_salutation_options()
{
    return array(
        '' => '',
        'Dr.' => 'Dr.',
        'Miss' => 'Miss',
        'Mr.' => 'Mr.',
        'Mrs.' => 'Mrs.',
        'Ms.' => 'Ms.',
        'Prof.' => 'Prof.',
        'Rev.' => 'Rev.');
}

function get_suffix_options()
{
    return array(
        '' => '',
        'I' => 'I',
        'II' => 'II',
        'III' => 'III',
        'IV' => 'IV',
        'Jr.' => 'Jr.',
        'Sr.' => 'Sr.');
}

function get_tax_rate_for_address($country_code, $state_code) {
    // declare tax zones for country array that we will use to store all valid tax zones for country
    $tax_zones_for_country = array();

    // get all tax zones that contain country
    $query = "SELECT tax_zone_id
             FROM tax_zones_countries_xref
             LEFT JOIN countries ON countries.id = tax_zones_countries_xref.country_id
             WHERE countries.code = '" . escape($country_code) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while ($row = mysqli_fetch_assoc($result)) {
        $tax_zones_for_country[] = $row['tax_zone_id'];
    }

    // declare tax zones for state array that we will use to store all valid tax zones for state
    $tax_zones_for_state = array();
    
    // get country id
    $query = "SELECT id FROM countries WHERE code = '" . escape($country_code) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $country_id = $row['id'];

    // find out if state is in database and if it belongs to country
    $query =
        "SELECT id
        FROM states
        WHERE
            (code = '" . escape($state_code) . "')
            AND (country_id = '$country_id')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

    // if state was found in database and it belongs to country, find what tax zones contain state
    if (mysqli_num_rows($result) > 0) {
        $query = "SELECT tax_zone_id
                 FROM tax_zones_states_xref
                 LEFT JOIN states ON states.id = tax_zones_states_xref.state_id
                 WHERE states.code = '" . escape($state_code) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        while ($row = mysqli_fetch_assoc($result)) {
            $tax_zones_for_state[] = $row['tax_zone_id'];
        }

        // loop through all tax zones for country
        foreach ($tax_zones_for_country as $tax_zone_id) {
            // if tax zone id in tax zone for country is also a valid tax zone for state, then it is a valid tax zone
            if (in_array($tax_zone_id, $tax_zones_for_state) == true) {
                $valid_tax_zone_id = $tax_zone_id;
                break;
            }
        }

    // else state was not found in database, so valid tax zone is valid tax zone for country
    } else {
        $valid_tax_zone_id = $tax_zones_for_country[0];
    }
    
    // if a valid tax zone was found, get tax rate
    if ($valid_tax_zone_id) {
        $query = "SELECT tax_rate FROM tax_zones WHERE id = '$valid_tax_zone_id'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_assoc($result);
        $tax_rate = $row['tax_rate'];
    }
    
    // if a tax rate was found return tax rate
    if ($tax_rate) {
        return $tax_rate;
        
    // else a tax rate was not found, so return false
    } else {
        return false;
    }
}

function generate_order_reference_code() {
    
    $characters = array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z');
    
    for($i = 1; $i <= 10; $i++) {
        $index = mt_rand(0, 35);
        $reference_code .= $characters[$index];
    }
    
    // check to see if reference code is already in use
    $query = "SELECT id FROM orders WHERE reference_code = '$reference_code'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if reference code is already in use, use recursion to generate a new reference code
    if (mysqli_num_rows($result) > 0) {
        return generate_order_reference_code();
        
    // else reference code is not already in use, so return reference code
    } else {
        return $reference_code;
    }
}

function generate_affiliate_code()
{
    $characters = array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z');
    
    for($i = 1; $i <= 5; $i++) {
        $index = mt_rand(0, 35);
        $affiliate_code .= $characters[$index];
    }
    
    // check to see if affiliate code is already in use
    $query = "SELECT id FROM contacts WHERE affiliate_code = '$affiliate_code'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if affiliate code is already in use, use recursion to generate a new affiliate code
    if (mysqli_num_rows($result) > 0) {
        return generate_affiliate_code();
        
    // else affiliate code is not already in use, so return affiliate code
    } else {
        return $affiliate_code;
    }
}

function get_request_uri()
{
    // the order of the conditionals below is important, because newer versions of IIS supply a REQUEST_URI value,
    // but the value does not contain the original pretty URL and might also have other problems
    
    // if HTTP_X_REWRITE_URL is set (i.e. ISAPI_Rewrite is being used on IIS)
    if ($_SERVER['HTTP_X_REWRITE_URL']) {
        return $_SERVER['HTTP_X_REWRITE_URL'];
    
    // else if REQUEST_URI is set (i.e. non IIS web server)
    } elseif ($_SERVER['REQUEST_URI']) {
        return $_SERVER['REQUEST_URI'];
        
    // else no REQUEST_URI or HTTP_X_REWRITE_URL can be found (i.e. IIS web server without ISAPI_Rewrite)
    } else {
        // if QUERY_STRING is set then return PHP_SELF and QUERY_STRING
        if ($_SERVER['QUERY_STRING']) {
            return $_SERVER['PHP_SELF'] . '?' . $_SERVER['QUERY_STRING'];
            
        // else QUERY_STRING is not set, so return just PHP_SELF
        } else {
            return $_SERVER['PHP_SELF'];
        }
    }
}

function prepare_form_data_for_input($data, $type)
{
    if ($data) {
        switch ($type) {
            case 'date':
                $date_parts = preg_split('/[-,\/]/', $data);
                
                $year = $date_parts[2];

                if (DATE_FORMAT == 'month_day') {
                    $month = $date_parts[0];
                    $day = $date_parts[1];
                } else {
                    $month = $date_parts[1];
                    $day = $date_parts[0];
                }
                
                $year = str_pad($year, 4, '0', STR_PAD_LEFT);
                $month = str_pad($month, 2, '0', STR_PAD_LEFT);
                $day = str_pad($day, 2, '0', STR_PAD_LEFT);
                
                return $year . '-' . $month . '-' . $day;
                break;
                
            case 'date and time':
                $date_and_time_parts = explode(' ', $data, 2);
                $date = $date_and_time_parts[0];
                $time = $date_and_time_parts[1];
                
                $date_parts = preg_split('/[-,\/]/', $date);
                
                $year = $date_parts[2];

                if (DATE_FORMAT == 'month_day') {
                    $month = $date_parts[0];
                    $day = $date_parts[1];
                } else {
                    $month = $date_parts[1];
                    $day = $date_parts[0];
                }
                
                $year = str_pad($year, 4, '0', STR_PAD_LEFT);
                $month = str_pad($month, 2, '0', STR_PAD_LEFT);
                $day = str_pad($day, 2, '0', STR_PAD_LEFT);

                preg_match('/(\d+):(\d+) (AM|PM)/i', $time, $time_parts);
                
                $hour = $time_parts[1];
                $minute = $time_parts[2];
                $am_pm = mb_strtoupper($time_parts[3]);
                
                // convert hour to military format
                if (($am_pm == 'PM') && ($hour != 12)) {
                    $hour = $hour + 12;
                }
                
                // convert 12 AM to 0 AM
                if (($am_pm == 'AM') && ($hour == 12)) {
                    $hour = 0;
                }
                
                $hour = str_pad($hour, 2, '0', STR_PAD_LEFT);
                $minute = str_pad($minute, 2, '0', STR_PAD_LEFT);
                
                return $year . '-' . $month . '-' . $day . ' ' . $hour . ':' . $minute;
                break;
                
            case 'time':
                preg_match('/(\d+):(\d+) (AM|PM)/i', $data, $time_parts);
                
                $hour = $time_parts[1];
                $minute = $time_parts[2];
                $am_pm = mb_strtoupper($time_parts[3]);
                
                // convert hour to military format
                if (($am_pm == 'PM') && ($hour != 12)) {
                    $hour = $hour + 12;
                }
                
                // convert 12 AM to 0 AM
                if (($am_pm == 'AM') && ($hour == 12)) {
                    $hour = 0;
                }
                
                $hour = str_pad($hour, 2, '0', STR_PAD_LEFT);
                $minute = str_pad($minute, 2, '0', STR_PAD_LEFT);
                
                return $hour . ':' . $minute;
                break;
                
            default:
                return $data;
                break;
        }
    } else {
        return $data;
    }
}

function prepare_form_data_for_output($data, $type, $prepare_for_html = true, $date_format = '')
{
    if ($data) {
        switch ($type) {
            case 'date':
                // if a date format was passed, then output date with that format
                if ($date_format != '') {
                    // if the format is relative, then get relative time
                    if ($date_format == 'relative') {
                        // If this is being prepared for HTML, then add tooltip with absolute date and time,
                        // so a user can hover over the relative time and see the absolute date and time
                        if ($prepare_for_html == TRUE) {
                            $output = get_relative_time(array('timestamp' => strtotime($data), 'format' => 'html', 'type' => 'date'));

                            // because we have HTML in this output now,
                            // set the prepare for HTML value to false, so down below we don't escape HTML
                            $prepare_for_html = FALSE;

                        // Otherwise this is being prepared for plain text, so do not add any HTML.
                        } else {
                            $output = get_relative_time(array('timestamp' => strtotime($data), 'format' => 'plain_text', 'type' => 'date'));
                        }

                    // else the format is not relative, so use format
                    } else {
                        $output = date($date_format, strtotime($data));
                    }

                // else a date format was not passed, so output date with standard format
                } else {
                    $date_parts = explode('-', $data);

                    $month = $date_parts[1];

                    // If the first character of the month is 0, then remove the 0.
                    if (mb_substr($month, 0, 1) == '0') {
                        $month = mb_substr($month, 1);
                    }

                    $day = $date_parts[2];

                    // If the first character of the day is 0, then remove the 0.
                    if (mb_substr($day, 0, 1) == '0') {
                        $day = mb_substr($day, 1);
                    }

                    $year = $date_parts[0];

                    if (DATE_FORMAT == 'month_day') {
                        $output = $month . '/' . $day . '/' . $year;
                    } else {
                        $output = $day . '/' . $month . '/' . $year;
                    }
                }

                break;
                
            case 'date and time':
                // if a date format was passed, then output date with that format
                if ($date_format != '') {
                    // if the format is relative, then get relative time
                    if ($date_format == 'relative') {
                        // If this is being prepared for HTML, then add tooltip with absolute date and time,
                        // so a user can hover over the relative time and see the absolute date and time
                        if ($prepare_for_html == TRUE) {
                            $output = get_relative_time(array('timestamp' => strtotime($data), 'format' => 'html'));

                            // because we have HTML in this output now,
                            // set the prepare for HTML value to false, so down below we don't escape HTML
                            $prepare_for_html = FALSE;

                        // Otherwise this is being prepared for plain text, so do not add any HTML.
                        } else {
                            $output = get_relative_time(array('timestamp' => strtotime($data), 'format' => 'plain_text'));
                        }

                    // else the format is not relative, so use format
                    } else {
                        // If the visitor is logged in,
                        // and the user has a specific timezone set,
                        // and the date format contains a timezone,
                        // then get date & time in user's timezone.
                        if (
                            (USER_LOGGED_IN)
                            && (USER_TIMEZONE != '')
                            &&
                            (
                                (preg_match("/[^\\\]T/", $date_format) == 1)
                                || (preg_match("/[^\\\]e/", $date_format) == 1)
                            )
                        ) {
                            $date = new DateTime('@' . strtotime($data));
                            $date->setTimezone(new DateTimeZone(USER_TIMEZONE));
                            $output = $date->format($date_format);

                        // Otherwise get date & time in site's timezone.
                        } else {
                            $output = date($date_format, strtotime($data));
                        }
                    }

                // else a date format was not passed, so output date with standard format
                } else {
                    $date_and_time_parts = explode(' ', $data, 2);
                    $date = $date_and_time_parts[0];
                    $time = $date_and_time_parts[1];
                    
                    $date_parts = explode('-', $date);

                    $month = $date_parts[1];

                    // If the first character of the month is 0, then remove the 0.
                    if (mb_substr($month, 0, 1) == '0') {
                        $month = mb_substr($month, 1);
                    }

                    $day = $date_parts[2];

                    // If the first character of the day is 0, then remove the 0.
                    if (mb_substr($day, 0, 1) == '0') {
                        $day = mb_substr($day, 1);
                    }

                    $year = $date_parts[0];
                    
                    $time_parts = explode(':', $time);
                    
                    $hour = $time_parts[0];
                    $minute = $time_parts[1];
                    
                    // if it is AM
                    if ($hour <= 11) {
                        $am_pm = 'AM';
                        
                    // else it is PM
                    } else {
                        $am_pm = 'PM';
                    }
                    
                    // if hour is equal to 0 then convert to 12
                    if ($hour == 0) {
                        $hour = 12;
                        
                    // else if hour is greater or equal to 13, subtract 12
                    } elseif ($hour >= 13) {
                        $hour = $hour - 12;
                    }

                    // If the first character of the hour is 0, then remove the 0.
                    if (mb_substr($hour, 0, 1) == '0') {
                        $hour = mb_substr($hour, 1);
                    }
                    
                    $minute = str_pad($minute, 2, '0', STR_PAD_LEFT);

                    if (DATE_FORMAT == 'month_day') {
                        $output = $month . '/' . $day . '/' . $year . ' ' . $hour . ':' . $minute . ' ' . $am_pm;
                    } else {
                        $output = $day . '/' . $month . '/' . $year . ' ' . $hour . ':' . $minute . ' ' . $am_pm;
                    }
                }

                break;
            
            case 'time':
                $time_parts = explode(':', $data);
                
                $hour = $time_parts[0];
                $minute = $time_parts[1];
                
                // if it is AM
                if ($hour <= 11) {
                    $am_pm = 'AM';
                    
                // else it is PM
                } else {
                    $am_pm = 'PM';
                }
                
                // if hour is equal to 0 then convert to 12
                if ($hour == 0) {
                    $hour = 12;
                    
                // else if hour is greater or equal to 13, subtract 12
                } elseif ($hour >= 13) {
                    $hour = $hour - 12;
                }

                // If the first character of the hour is 0, then remove the 0.
                if (mb_substr($hour, 0, 1) == '0') {
                    $hour = mb_substr($hour, 1);
                }
                
                $minute = str_pad($minute, 2, '0', STR_PAD_LEFT);
                
                $output = $hour . ':' . $minute . ' ' . $am_pm;
                break;
                
            default:
                $output = $data;
                break;
        }
        
        if ($prepare_for_html == true) {
            $output = h($output);
            $output = nl2br($output);
        }
        
    } else {
        $output = $data;
    }
    
    return $output;
}

function get_standard_fields_for_view()
{
    $standard_fields =
        array(

            array(
                'name' => 'Reference Code',
                'value' => 'reference_code',
                'sql_name' => 'forms.reference_code',
                'type' => ''
            ),

            array(
                'name' => 'Complete',
                'value' => 'complete',
                'sql_name' => 'CASE WHEN (forms.complete) THEN "Complete" ELSE "" END',
                'type' => ''
            ),

            array(
                'name' => 'Address Name',
                'value' => 'address_name',
                'sql_name' => 'forms.address_name',
                'type' => ''
            ),
            
            array(
                'name' => 'Tracking Code',
                'value' => 'tracking_code',
                'sql_name' => 'forms.tracking_code',
                'type' => ''
            ),
            
            array(
                'name' => 'Affiliate Code',
                'value' => 'affiliate_code',
                'sql_name' => 'forms.affiliate_code',
                'type' => ''
            ),
            
            array(
                'name' => 'Referring URL',
                'value' => 'referring_url',
                'sql_name' => 'forms.http_referer',
                'type' => ''
            ),
            
            array(
                'name' => 'Submitter',
                'value' => 'submitter',
                'sql_name' => 'submitter.user_username',
                'type' => 'username'
            ),
            
            array(
                'name' => 'Submitted Date & Time',
                'value' => 'submitted_date_and_time',
                'sql_name' => 'FROM_UNIXTIME(forms.submitted_timestamp)',
                'type' => 'date and time'
            ),
            
            array(
                'name' => 'Last Modifier',
                'value' => 'last_modifier',
                'sql_name' => 'last_modifier.user_username',
                'type' => 'username'
            ),
            
            array(
                'name' => 'Last Modified Date & Time',
                'value' => 'last_modified_date_and_time',
                'sql_name' => 'FROM_UNIXTIME(forms.last_modified_timestamp)',
                'type' => 'date and time'
            ),

            array(
                'name' => 'Number of Views',
                'value' => 'number_of_views',
                'sql_name' => 'submitted_form_info.number_of_views',
                'type' => ''
            ),
                
            array(
                'name' => 'Number of Comments',
                'value' => 'number_of_comments',
                'sql_name' => 'submitted_form_info.number_of_comments',
                'type' => ''
            ),

            array(
                'name' => 'Newest Comment Name',
                'value' => 'newest_comment_name',
                'sql_name' => 'newest_comment.name',
                'type' => ''
            ),
                
            array(
                'name' => 'Newest Comment',
                'value' => 'newest_comment',
                'sql_name' => 'newest_comment.message',
                'type' => ''
            ),
                
            array(
                'name' => 'Newest Comment Date & Time',
                'value' => 'newest_comment_date_and_time',
                'sql_name' => 'FROM_UNIXTIME(newest_comment.created_timestamp)',
                'type' => 'date and time'
            ),
                
            array(
                'name' => 'Newest Comment ID',
                'value' => 'newest_comment_id',
                'sql_name' => 'newest_comment.id',
                'type' => ''
            ),
                
            array(
                'name' => 'Newest Activity Date & Time',
                'value' => 'newest_activity_date_and_time',
                'sql_name' => 'FROM_UNIXTIME(GREATEST(forms.submitted_timestamp, IFNULL(newest_comment.created_timestamp, \'\')))',
                'type' => 'date and time'
            ),

            array(
                'name' => 'Comment Attachments',
                'value' => 'comment_attachments',
                'sql_name' =>
                    "(SELECT GROUP_CONCAT(files.name ORDER BY comments.id ASC SEPARATOR '||')
                    FROM comments
                    LEFT JOIN files ON files.id = comments.file_id
                    WHERE
                        (comments.page_id = submitted_form_info.page_id)
                        AND (comments.item_id = forms.id)
                        AND (comments.item_type = 'submitted_form')
                        AND (comments.published = 1)
                        AND (comments.file_id != 0))",
                'type' => ''
            )

        );
    
    return $standard_fields;
}

// This function is used by form list view to create sql for filters.

function prepare_sql_operation($operator, $operand_1, $operand_2) {

    $quote = '';

    // If the operator is a size operator and the second operand is not numeric,
    // then prepare to add single quotes around the second operand.  We have to eliminate
    // quotes for numeric values so that MySQL will compare the operands correctly.
    if (
        (
            ($operator == 'is less than')
            || ($operator == 'is less than or equal to')
            || ($operator == 'is greater than')
            || ($operator == 'is greater than or equal to')
        )
        && (is_numeric($operand_2) == FALSE)
    ) {
        $quote = "'";
    }

    switch ($operator) {
        case 'contains':
        default:
            return "IFNULL(" . $operand_1 . ", '') LIKE '%" . e(escape_like($operand_2)) . "%'";
            break;
            
        case 'does not contain':
            return "IFNULL(" . $operand_1 . ", '') NOT LIKE '%" . e(escape_like($operand_2)) . "%'";
            break;
            
        case 'is equal to':
            return "IFNULL(" . $operand_1 . ", '') = '" . e($operand_2) . "'";
            break;
            
        case 'is not equal to':
            return "IFNULL(" . $operand_1 . ", '') != '" . e($operand_2) . "'";
            break;
            
        case 'is less than':
            return "IFNULL(" . $operand_1 . ", '') < " . $quote . e($operand_2) . $quote;
            break;
            
        case 'is less than or equal to':
            return "IFNULL(" . $operand_1 . ", '') <= " . $quote . e($operand_2) . $quote;
            break;
            
        case 'is greater than':
            return "IFNULL(" . $operand_1 . ", '') > " . $quote . e($operand_2) . $quote;
            break;
            
        case 'is greater than or equal to':
            return "IFNULL(" . $operand_1 . ", '') >= " . $quote . e($operand_2) . $quote;
            break;
    }
}

// this function is used by form list view to convert a dynamic value to a static value
function get_dynamic_value($dynamic_value, $dynamic_value_attribute = '')
{
    switch ($dynamic_value) {
        case 'current date':
            return date('Y-m-d');
            break;
            
        case 'current date and time':
            return date('Y-m-d H:i:s');
            break;
            
        case 'current time':
            return date('H:i:s');
            break;
            
        case 'days ago':
            return date('Y-m-d H:i:s', time() - (86400 * $dynamic_value_attribute));
            break;
            
        case 'viewer':
            return $_SESSION['sessionusername'];
            break;
            
        case 'viewers email address':
            $viewers_email_address = '';
            
            // if there is a user logged in then get the user's e-mail address
            if (isset($_SESSION['sessionusername']) == TRUE) {
                $query = "SELECT user_email FROM user WHERE user_username = '" . escape($_SESSION['sessionusername']) . "'";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                $row = mysqli_fetch_assoc($result);
                $viewers_email_address = $row['user_email'];
            }
            
            return $viewers_email_address;
            break;
    }
}

function generate_email_recipient_reference_code()
{
    $characters = array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z');
    
    for($i = 1; $i <= 10; $i++) {
        $index = mt_rand(0, 35);
        $reference_code .= $characters[$index];
    }
    
    // check to see if reference code is already in use
    $query = "SELECT id FROM email_recipients WHERE reference_code = '$reference_code'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if reference code is already in use, use recursion to generate a new reference code
    if (mysqli_num_rows($result) > 0) {
        return generate_form_reference_code();
        
    // else reference code is not already in use, so return reference code
    } else {
        return $reference_code;
    }
}

function generate_form_reference_code()
{
    $characters = array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z');
    
    for($i = 1; $i <= 10; $i++) {
        $index = mt_rand(0, 35);
        $reference_code .= $characters[$index];
    }
    
    // check to see if reference code is already in use
    $query = "SELECT id FROM forms WHERE reference_code = '$reference_code'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if reference code is already in use, use recursion to generate a new reference code
    if (mysqli_num_rows($result) > 0) {
        return generate_form_reference_code();
        
    // else reference code is not already in use, so return reference code
    } else {
        return $reference_code;
    }
}

function generate_commission_reference_code()
{
    $characters = array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z');
    
    for($i = 1; $i <= 10; $i++) {
        $index = mt_rand(0, 35);
        $reference_code .= $characters[$index];
    }
    
    // check to see if reference code is already in use
    $query = "SELECT id FROM commissions WHERE reference_code = '$reference_code'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if reference code is already in use, use recursion to generate a new reference code
    if (mysqli_num_rows($result) > 0) {
        return generate_commission_reference_code();
        
    // else reference code is not already in use, so return reference code
    } else {
        return $reference_code;
    }
}

function generate_encryption_key()
{
    $characters = array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z');
    
    $key = '';
    
    for ($i = 1; $i <= 32; $i++) {
        $index = mt_rand(0, 35);
        $key .= $characters[$index];
    }
    
    return $key;
}

function get_tracking_code()
{
    // if there is a tracking code in the session, then return it
    if ($_SESSION['software']['tracking_code']) {
        return $_SESSION['software']['tracking_code'];
        
    // else if there is a tracking code in a cookie, then return it
    } elseif ($_COOKIE['software']['tracking_code']) {
        return $_COOKIE['software']['tracking_code'];
        
    // else no tracking code can be found, so return empty string
    } else {
        return '';
    }
}

function get_affiliate_code()
{
    // if there is a tracking code in the session, then return it
    if ($_SESSION['software']['affiliate_code']) {
        return $_SESSION['software']['affiliate_code'];
        
    // else if there is a tracking code in a cookie, then return it
    } elseif ($_COOKIE['software']['affiliate_code']) {
        return $_COOKIE['software']['affiliate_code'];
        
    // else no tracking code can be found, so return empty string
    } else {
        return '';
    }
}

function validate_order_item_for_arrival_date($order_item_id)
{
    // get information for order item
    $query =
        "SELECT
            ship_tos.id AS ship_to_id,
            ship_tos.arrival_date,
            ship_tos.zip_code,
            ship_tos.country,
            ship_tos.shipping_method_id
        FROM order_items
        LEFT JOIN ship_tos ON order_items.ship_to_id = ship_tos.id
        WHERE order_items.id = '" . e($order_item_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);

    $ship_to_id = $row['ship_to_id'];
    $arrival_date = $row['arrival_date'];
    $zip_code = $row['zip_code'];
    $country = $row['country'];
    $shipping_method_id = $row['shipping_method_id'];
    
    // if the requested arrival date is not at once
    if ($arrival_date != '0000-00-00') {
        // determine if there is a shipping cut-off for the arrival date and shipping method
        $query =
            "SELECT
                shipping_cutoffs.date_and_time
            FROM shipping_cutoffs
            LEFT JOIN arrival_dates ON shipping_cutoffs.arrival_date_id = arrival_dates.id
            WHERE
                (arrival_dates.arrival_date = '" . escape($arrival_date) . "')
                AND (shipping_cutoffs.shipping_method_id = '$shipping_method_id')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // if there is a shipping cut-off, then the order item is valid for the arrival date, because we ignore the product preparation time when there is a shipping cut-off
        if (mysqli_num_rows($result) > 0) {
            return TRUE;
            
        // else there is not a shipping cut-off, so determine if the order item is valid for the arrival date by looking at the transit time
        } else {

            require_once(dirname(__FILE__) . '/shipping.php');

            $response = get_delivery_date(array(
                'ship_to_id' => $ship_to_id,
                'shipping_method' => array('id' => $shipping_method_id),
                'zip_code' => $zip_code,
                'country' => $country
            ));
        
            $delivery_date = $response['delivery_date'];

            if (!$delivery_date or $delivery_date > $arrival_date) {
                return false;

            } else {
                return true;
            }
        }
        
    // else the requested arrival date is at once, so the product is valid for the requested arrival date
    } else {
        return true;
    }
}

// get the javascript code for initializing tiny_mce wysiwyg editors
// You can pass a $style_theme_name in order to override the activated theme that is loaded for the rich-text editor.
// If a theme is currently being previewed, then that will be used instead of the style theme.
function get_wysiwyg_editor_code($editor_ids, $activate_editors = true, $folder_id = 0, $edit_region_dialog = FALSE, $style_theme_name = '')
{
    // get values for page editor buttons from config table so that we will know which buttons to output
    $query =
        "SELECT
            page_editor_version,
            page_editor_font,
            page_editor_font_size,
            page_editor_font_style,
            page_editor_font_color,
            page_editor_background_color
        FROM config";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);

    $page_editor_version = $row['page_editor_version'];
    $page_editor_font_style = $row['page_editor_font_style'];
    $page_editor_font = $row['page_editor_font'];
    $page_editor_font_size = $row['page_editor_font_size'];
    $page_editor_font_color = $row['page_editor_font_color'];
    $page_editor_background_color = $row['page_editor_background_color'];

    if ($page_editor_version == 'latest') {

        $theme_name = '';

        // If the visitor is previewing a theme, then use that theme.
        if (isset($_SESSION['software']['preview_theme_id']) == TRUE) {
            // If the visitor has selected a theme to preview, then use that theme.
            // This check is necessary because a user might have selected the none theme in the previous pick list.
            if ($_SESSION['software']['preview_theme_id'] != '') {
                $theme_name = db_value("SELECT name FROM files WHERE id = '" . escape($_SESSION['software']['preview_theme_id']) . "'");
            }

        // Otherwise if there is a style theme name, then use that.
        } else if ($style_theme_name != '') {
            $theme_name = $style_theme_name;
            
        // Otherwise use activated theme.
        } else {
            // get theme name differently based on the device type (i.e. desktop or mobile)
            switch ($_SESSION['software']['device_type']) {
                // if the device type is desktop then get the activated desktop theme name
                case 'desktop':
                default:
                    $theme_name = db_value("SELECT name FROM files WHERE activated_desktop_theme = '1'");
                    break;
                
                // if the device type is mobile, then get the activated mobile theme name
                // and fall back to the activated desktop theme if necessary
                case 'mobile':
                    $theme_name = db_value("SELECT name FROM files WHERE activated_mobile_theme = '1'");

                    // if an activated mobile theme could not be found, then get activated desktop theme name
                    if ($theme_name == '') {
                        $theme_name = db_value("SELECT name FROM files WHERE activated_desktop_theme = '1'");
                    }

                    break;
            }
        }

        $output_custom_format_toolbar_property = '';
        $output_editor_properties = '';
        $custom_formats_found = false;
        
        // If a theme was found and the theme file exists, then use stylesheet for editor.
        if (($theme_name != '') && (file_exists(FILE_DIRECTORY_PATH . '/' . $theme_name) == TRUE)) {
            $output_editor_properties .= 'contentsCss: ["' . PATH . h($theme_name) . '", "' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/ckeditor_4_4_0/contents.css"]';

            $output_custom_formats = '';

            $custom_formats = get_custom_formats($theme_name);

            // if there is at least one custom format, then loop through all custom formats in order to prepare lists for editor
            if (count($custom_formats) > 0) {
                $output_custom_format_toolbar_property = '"Styles", ';

                // loop through the custom formats in order to prepare list of data
                foreach ($custom_formats as $custom_format) {
                    $output_class_name = escape_javascript($custom_format['name']);
                    
                    // if this is not the first custom format, then add separation
                    if ($output_custom_formats != '') {
                        $output_custom_formats .= ',';
                    }
                    
                    // prepare style format for this custom format differently based on its class name
                    // we do special things for the default custom formats
                    switch ($custom_format['name']) {
                        case 'heading-primary':
                        case 'heading-secondary':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: ["h1", "h2", "h3", "h4", "h5", "h6"], attributes: {"class": "' . $output_class_name . '"}}';
                            break;   

                        case 'image-primary':
                        case 'image-secondary':
                        case 'image-left-primary':
                        case 'image-left-secondary':
                        case 'image-right-primary':
                        case 'image-right-secondary':
                        case 'image-mobile-hide':
                        case 'image-desktop-hide':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "img", attributes: {"class": "' . $output_class_name . '"}}';
                            break;
                            
                        case 'link-button-primary-large':
                        case 'link-button-primary-small':
                        case 'link-button-secondary-large':
                        case 'link-button-secondary-small':
                        case 'link-content-more':
                        case 'link-menu-item':
                        case 'link-mobile-hide':
                        case 'link-desktop-hide':
                        case 'list-accordion-expanded':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "a", attributes: {"class": "' . $output_class_name . '"}}';
                            break;

                        case 'list-accordion':
                        case 'list-tabs':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: ["ul", "ol"], attributes: {"class": "' . $output_class_name . '"}}';
                            break;                                

                        case 'paragraph-indent':
                        case 'paragraph-box-example':
                        case 'paragraph-box-notice':
                        case 'paragraph-box-primary':
                        case 'paragraph-box-secondary':
                        case 'paragraph-box-warning':
                        case 'paragraph-no-margin':
                        case 'paragraph-no-margin-top':
                        case 'paragraph-no-margin-bottom':
                        case 'paragraph-mobile-hide':
                        case 'paragraph-desktop-hide':
                        case 'video-primary':
                        case 'video-secondary':
                        case 'video-left-primary':
                        case 'video-left-secondary':
                        case 'video-right-primary':
                        case 'video-right-secondary':
                        case 'video-mobile-hide':
                        case 'video-desktop-hide':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "p", attributes: {"class": "' . $output_class_name . '"}}';
                            break;
                            
                        case 'table-primary':
                        case 'table-secondary':
                        case 'table-left':
                        case 'table-right':
                        case 'table-center':
                        case 'table-mobile-hide':
                        case 'table-desktop-hide':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "table", attributes: {"class": "' . $output_class_name . '"}}';
                            break;
                            
                        case 'table-row-header':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "thead", attributes: {"class": "' . $output_class_name . '"}}';
                            break;
                            
                        case 'table-row-body':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "tbody", attributes: {"class": "' . $output_class_name . '"}}';
                            break;
                            
                        case 'table-row-footer':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "tfoot", attributes: {"class": "' . $output_class_name . '"}}';
                            break;
                            
                        case 'table-cell-header':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "th", attributes: {"class": "' . $output_class_name . '"}}';
                            break;
                            
                        case 'table-cell-data':
                        case 'table-cell-mobile-fill':
                        case 'table-cell-mobile-wrap':
                        case 'table-cell-mobile-hide':
                        case 'table-cell-desktop-hide':
                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "td", attributes: {"class": "' . $output_class_name . '"}}';
                            break;
                            
                        default:
                            // If an element was supplied with the custom format, then use that.
                            if ($custom_format['element']) {
                                $element = escape_javascript($custom_format['element']);

                            // Otherwise use span element.
                            } else {
                                $element = 'span';
                            }

                            $output_custom_formats .= '{name: "' . $output_class_name . '", element: "' . $element . '", attributes: {"class": "' . $output_class_name . '"}}';

                            break;
                    }
                }

                if ($output_editor_properties != '') {
                    $output_editor_properties .= ',';
                }

                $output_editor_properties .= 'stylesSet: [' . $output_custom_formats . ']';

                $custom_formats_found = true;
            }

        } else {
            $output_editor_properties .= 'contentsCss: "' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/ckeditor_4_4_0/contents.css"';
        }

        if ($custom_formats_found == false) {
            if ($output_editor_properties != '') {
                $output_editor_properties .= ',';
            }

            $output_editor_properties .= 'stylesSet: []';
        }

        $output_folder_id = '';

        if ($folder_id) {
            $output_folder_id = '&folder_id=' . $folder_id;
        }

        $output_filebrowser_properties = '';

        // If the user is logged in and the user is a manager or above,
        // or if the user has edit access to at least one folder,
        // then output settings and plugin so that user can browse images/link items,
        // and upload files.
        if (
            (USER_LOGGED_IN == true)
            &&
            (
                (USER_ROLE < 3)
                || (no_acl_check(USER_ID) == true)
            )
        ) {
            $output_filebrowser_properties =
                'filebrowserBrowseUrl: "' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/editor_select_page_or_file.php",
                filebrowserImageBrowseUrl: "' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/editor_select_image.php",
                filebrowserWindowWidth: "850",
                filebrowserWindowHeight: "740",';
        }

        $output_remove_sourcedialog_plugin = '';
        $output_editor_initialization = '';

        if ($edit_region_dialog == true) {
            // do nothing

        } else {
            $output_remove_sourcedialog_plugin = ',sourcedialog';

            foreach ($editor_ids as $editor_id) {
                $output_editor_initialization .= 'CKEDITOR.replace("' . $editor_id . '", software_ckeditor_config);';
            }

            $output_editor_initialization =
                'if (typeof(software_$) != "undefined") {
                    software_$(document).ready(function() {
                        ' . $output_editor_initialization . '
                    });
                } else {
                    $(document).ready(function() {
                        ' . $output_editor_initialization . '
                    });
                }';
        }

        $output_font_style_toolbar_property = '';

        if ($page_editor_font_style == 1) {
            $output_font_style_toolbar_property = '"Format", ';
        }

        $output_font_toolbar_property = '';

        if ($page_editor_font == 1) {
            $output_font_toolbar_property = '"Font", ';
        }

        $output_font_size_toolbar_property = '';

        if ($page_editor_font_size == 1) {
            $output_font_size_toolbar_property = '"FontSize", ';
        }

        $output_font_color_toolbar_property = '';

        if ($page_editor_font_color == 1) {
            $output_font_color_toolbar_property = '"TextColor", ';
        }

        $output_background_color_toolbar_property = '';

        if ($page_editor_background_color == 1) {
            $output_background_color_toolbar_property = '"BGColor", ';
        }

        // We have to remove the various plugins listed by removePlugins,
        // because we were using those plugins at one point during testing
        // and we told ckbuilder to add them to the bundled ckeditor.js file.
        // We are no longer using those plugins in production.
        // We don't want to bother having to run ckbuilder again for now, so we are just
        // dynamically telling ckeditor to remove them.  We have to tell ckeditor
        // to remove them, because we have since deleted their directories
        // so a JavaScript error would appear otherwise.  When we update ckeditor
        // in the future, then tell ckbuilder next time, that we don't want those plugins,
        // and then we can remove them from the list below.

        // Setting customConfig to an empty string is necessary so that CKEditor
        // does not try to load the non-existent config.js file in the ckeditor directory.
        // We load a dynamic config object and do not use the external config.js feature.
        // This avoids slow load-speed and browser generating 404 errors.

        return
            '<script src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/ckeditor_4_4_0/ckeditor.js"></script>
            <script>
                var software_editor_version = "' . $page_editor_version . '";
                CKEDITOR.disableAutoInline = true;
                var software_ckeditor_config = {
                    allowedContent: true,
                    autoGrow_onStartup: true,
                    customConfig: "",
                    dialogFieldsDefaultValues: {
                        table: {
                            info: {
                                txtWidth: "100%",
                                txtBorder: "0",
                                txtCellSpace: "",
                                txtCellPad: ""
                            }
                        }
                    },
                    disableNativeSpellChecker: false,
                    ' . $output_filebrowser_properties . '
                    filebrowserUploadUrl: "' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/editor_add_file.php?token=' . escape_javascript($_SESSION['software']['token']). $output_folder_id . '",
                    image_previewText: " ",
                    removeButtons: "addFile,addImage",
                    removeDialogTabs: "image:Link",
                    removePlugins: "codemirror,div,pagebreak,smiley,youtube' . $output_remove_sourcedialog_plugin . '",
                    resize_dir: "both",
                    title: false,
                    toolbar: [
                        [' . $output_font_style_toolbar_property . $output_custom_format_toolbar_property . $output_font_toolbar_property . $output_font_size_toolbar_property . $output_font_color_toolbar_property . $output_background_color_toolbar_property . '"Bold", "Italic", "Underline", "Strike", "Subscript", "Superscript", "JustifyLeft", "JustifyCenter", "JustifyRight", "JustifyBlock", "Indent", "Outdent", "-", "RemoveFormat"], "/",
                        ["Link", "Unlink", "Image", "oembed", "Table", "BulletedList", "NumberedList", "HorizontalRule", "Blockquote", "Anchor", "SpecialChar"], ["Cut", "Copy", "Paste", "PasteText", "PasteFromWord", "SelectAll", "NewPage", "Find"], ["Undo", "Redo"], ["Maximize"], ["ShowBlocks", "Source", "Sourcedialog"]
                    ],
                    ' . $output_editor_properties . '};
                ' . $output_editor_initialization . '
            </script>';

    // else set the version path to the older version
    } else {
        $page_editor_font = '';
        if ($row['page_editor_font'] == 1) {
            $page_editor_font = "fontselect,";
        }

        $page_editor_font_size = '';
        if ($row['page_editor_font_size'] == 1) {
            $page_editor_font_size = "fontsizeselect,";
        }

        $page_editor_font_style = '';
        if ($row['page_editor_font_style'] == 1) {
            $page_editor_font_style = "formatselect,";
        }

        $page_editor_font_color = '';
        if ($row['page_editor_font_color'] == 1) {
            $page_editor_font_color = "forecolor,";
        }

        $page_editor_background_color = '';
        if ($row['page_editor_background_color'] == 1) {
            $page_editor_background_color = "backcolor,|,";
        }
        
        $page_editor_style_select = '';
        $page_editor_style_sheet = '';
        $output_theme_advanced_styles = '';
        $output_style_formats = '';

        $theme_name = '';

        // If the visitor is previewing a theme, then use that theme.
        if (isset($_SESSION['software']['preview_theme_id']) == TRUE) {
            // If the visitor has selected a theme to preview, then use that theme.
            // This check is necessary because a user might have selected the none theme in the previous pick list.
            if ($_SESSION['software']['preview_theme_id'] != '') {
                $theme_name = db_value("SELECT name FROM files WHERE id = '" . escape($_SESSION['software']['preview_theme_id']) . "'");
            }

        // Otherwise if there is a style theme name, then use that.
        } else if ($style_theme_name != '') {
            $theme_name = $style_theme_name;
            
        // Otherwise use activated theme.
        } else {
            // get theme name differently based on the device type (i.e. desktop or mobile)
            switch ($_SESSION['software']['device_type']) {
                // if the device type is desktop then get the activated desktop theme name
                case 'desktop':
                default:
                    $theme_name = db_value("SELECT name FROM files WHERE activated_desktop_theme = '1'");
                    break;
                
                // if the device type is mobile, then get the activated mobile theme name
                // and fall back to the activated desktop theme if necessary
                case 'mobile':
                    $theme_name = db_value("SELECT name FROM files WHERE activated_mobile_theme = '1'");

                    // if an activated mobile theme could not be found, then get activated desktop theme name
                    if ($theme_name == '') {
                        $theme_name = db_value("SELECT name FROM files WHERE activated_desktop_theme = '1'");
                    }

                    break;
            }
        }
        
        // if a theme was found and the theme file exists, then use stylesheet for editor and get custom formats
        if (($theme_name != '') && (file_exists(FILE_DIRECTORY_PATH . '/' . $theme_name) == TRUE)) {
            $page_editor_style_sheet = 'content_css : "' . PATH . h($theme_name) . '",';

            $custom_formats = get_custom_formats($theme_name);

            // if there is at least one custom format, then loop through all custom formats in order to prepare lists for TinyMCE
            // we still use theme_advanced_styles so that pick lists in dialog windows contain the correct list of classes
            if (count($custom_formats) > 0) {
                $page_editor_style_select = 'styleselect,';
                
                // loop through the custom formats in order to prepare list of data
                foreach ($custom_formats as $custom_format) {
                    $output_class_name = escape_javascript($custom_format['name']);
                    
                    // if this is not the first custom format, then add separation
                    if ($output_theme_advanced_styles != '') {
                        $output_theme_advanced_styles .= ';';
                    }
                    
                    $output_theme_advanced_styles .= $output_class_name . '=' . $output_class_name;
                    
                    // if this is not the first custom format, then add separation
                    if ($output_style_formats != '') {
                        $output_style_formats .= ',';
                    }
                    
                    // prepare style format for this custom format differently based on its class name
                    // we do special things for the default custom formats
                    switch ($custom_format['name']) {
                        case 'heading-primary':
                        case 'heading-secondary':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'h1,h2,h3,h4,h5,h6\', classes: \'' . $output_class_name . '\'}';
                            break;   

                        case 'image-primary':
                        case 'image-secondary':
                        case 'image-left-primary':
                        case 'image-left-secondary':
                        case 'image-right-primary':
                        case 'image-right-secondary':
                        case 'image-mobile-hide':
                        case 'image-desktop-hide':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'img\', classes: \'' . $output_class_name . '\'}';
                            break;
                            
                        case 'link-button-primary-large':
                        case 'link-button-primary-small':
                        case 'link-button-secondary-large':
                        case 'link-button-secondary-small':
                        case 'link-content-more':
                        case 'link-menu-item':
                        case 'link-mobile-hide':
                        case 'link-desktop-hide':
                        case 'list-accordion-expanded':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'a\', classes: \'' . $output_class_name . '\'}';
                            break;

                        case 'list-accordion':
                        case 'list-tabs':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'ul, ol\', classes: \'' . $output_class_name . '\'}';
                            break;                                

                        case 'paragraph-indent':
                        case 'paragraph-box-example':
                        case 'paragraph-box-notice':
                        case 'paragraph-box-primary':
                        case 'paragraph-box-secondary':
                        case 'paragraph-box-warning':
                        case 'paragraph-no-margin':
                        case 'paragraph-no-margin-top':
                        case 'paragraph-no-margin-bottom':
                        case 'paragraph-mobile-hide':
                        case 'paragraph-desktop-hide':
                        case 'video-primary':
                        case 'video-secondary':
                        case 'video-left-primary':
                        case 'video-left-secondary':
                        case 'video-right-primary':
                        case 'video-right-secondary':
                        case 'video-mobile-hide':
                        case 'video-desktop-hide':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', block: \'p\', classes: \'' . $output_class_name . '\'}';
                            break;
                            
                        case 'table-primary':
                        case 'table-secondary':
                        case 'table-left':
                        case 'table-right':
                        case 'table-center':
                        case 'table-mobile-hide':
                        case 'table-desktop-hide':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'table\', classes: \'' . $output_class_name . '\'}';
                            break;
                            
                        case 'table-row-header':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'thead\', classes: \'' . $output_class_name . '\'}';
                            break;
                            
                        case 'table-row-body':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'tbody\', classes: \'' . $output_class_name . '\'}';
                            break;
                            
                        case 'table-row-footer':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'tfoot\', classes: \'' . $output_class_name . '\'}';
                            break;
                            
                        case 'table-cell-header':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'th\', classes: \'' . $output_class_name . '\'}';
                            break;
                            
                        case 'table-cell-data':
                        case 'table-cell-mobile-fill':
                        case 'table-cell-mobile-wrap':
                        case 'table-cell-mobile-hide':
                        case 'table-cell-desktop-hide':
                            $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'td\', classes: \'' . $output_class_name . '\'}';
                            break;
                            
                        default:
                            // If an element was supplied with the custom format, then use that.
                            if ($custom_format['element']) {
                                $output_style_formats .= '{title: \'' . $output_class_name . '\', selector: \'' . escape_javascript($custom_format['element']) . '\', classes: \'' . $output_class_name . '\'}';

                            // Otherwise use span element.
                            } else {
                                $output_style_formats .= '{title: \'' . $output_class_name . '\', inline: \'span\', classes: \'' . $output_class_name . '\'}';
                            }

                            break;
                    }
                }
                
                $output_theme_advanced_styles = 'theme_advanced_styles: "' . $output_theme_advanced_styles . '",';
                $output_style_formats = 'style_formats: [' . $output_style_formats . '],';
            }
        }
        
        $output_editor_initialization = 'elements : "';
        
        foreach ($editor_ids as $editor_id) {
            $output_editor_initialization .= $editor_id . ',';
        }
        
        $output_editor_initialization = mb_substr($output_editor_initialization, 0, -1);
        $output_editor_initialization .= '",';
        
        $output_editor_activation = '';
        if ($activate_editors == true) {
            $output_editor_activation = 'mode : "exact",';
        } else {
            $output_editor_activation = 'mode : "none",';
        }
        
        $version_path = '3_5_10';
        $output_plugins = 'lists,';
        $output_spellchecker_button = '';
        $output_browser_spellcheck = 'browser_spellcheck: true,';
        
        $output_edit_region_dialog_properties = '';
        
        // if this is for the edit region dialog, then output properties for it
        if ($edit_region_dialog == TRUE) {
            $output_theme_advanced_resizing = 'false';

            // init_instance_callback is used in order to call a function to resize the editor after it is fully loaded
            // auto_focus is used in order to place the cursor in the editor.
            // the following function causes the cursor to sometimes be hidden in Firefox, so don't use it instead.
            // tinyMCE.execCommand('mceFocus', false, 'software_edit_region_textarea')
            
            $output_edit_region_dialog_properties =
                'init_instance_callback: function() {software_activate_edit_region_dialog();},
                auto_focus: "software_edit_region_textarea",';
        } else {
            $output_theme_advanced_resizing = 'true';
        }

        // Assume that CDN is disabled until we find out otherwise.
        // We pass the CDN value so we know which jQuery location to use for plugin pages in the rich-text editor.
        $cdn = 'false';

        // if CDN is enabled, then use Google CDN for jQuery for performance reasons
        if (
            (defined('CDN') == FALSE)
            || (CDN == TRUE)
        ) {
            $cdn = 'true';
        }
        
        // we removed shape from the list of allowed attributes for an anchor in extended_valid_elements below
        // in order to workaround a bug with IE 9 where updates would not be saved if a link existed in the content
        // "media_strict: false" was added in order to prevent TinyMCE from removing embed tag (using extended_valid_elements did not work),
        // which resulted in Flash movies not working in IE 9 (64 bit).  Not sure if this affected IE 9 (32 bit).
        
        return
            '<script language="javascript" type="text/javascript" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/tiny_mce_' . $version_path . '/tiny_mce.js"></script>
            <script language="javascript" type="text/javascript">
                var software_editor_version = "' . $page_editor_version . '";
                tinyMCE.init({
                    ' . $output_editor_activation . '
                    ' . $output_editor_initialization . '
                    ' . $output_edit_region_dialog_properties . '
                    theme: "advanced",
                    skin: "cirkuit",
                    plugins: "advimage,advlink,' . $output_plugins . 'contextmenu,fullscreen,inlinepopups,media,paste,searchreplace,tabfocus,table",
                    theme_advanced_buttons1: "' . $page_editor_font_style . $page_editor_style_select . $page_editor_font  . $page_editor_font_size . $page_editor_font_color . $page_editor_background_color . 'bold,italic,underline,strikethrough,blockquote' . $output_spellchecker_button . '",
                    theme_advanced_buttons2: "' . 'cut,copy,paste,pastetext,pasteword,selectall,newdocument,|,sub,sup,|,justifyleft,justifycenter,justifyright,justifyfull,|,bullist,numlist,outdent,indent,|,hr,link,unlink,anchor,image,media",
                    theme_advanced_buttons3: "tablecontrols,|,visualaid,|,charmap,|,removeformat,cleanup,|,search,replace,|,undo,redo,|,fullscreen,code",
                    theme_advanced_toolbar_location: "top",
                    theme_advanced_toolbar_align: "left",
                    theme_advanced_path_location: "bottom",
                    media_strict: false,
                    ' . $page_editor_style_sheet . '
                    ' . $output_theme_advanced_styles . '
                    ' . $output_style_formats . '
                    extended_valid_elements: "#p[id|class|align|style],-div[id|class|align|style],span[id|class|align|style],a[accesskey|align|charset|class|coords|dir|href|hreflang|id|lang|name|onblur|onclick|ondblclick|onfocus|onkeydown|onkeypress|onkeyup|onmousedown|onmousemove|onmouseout|onmouseover|onmouseup|rel|rev|style|tabindex|title|target|type],-ul[type|class|compact|align],-ol[type|class|compact|align],iframe[align|class|frameborder|height|id|longdesc|marginheight|marginwidth|name|scrolling|src|style|title|width]",
                    external_link_list_url: "' . PATH . SOFTWARE_DIRECTORY . '/tiny_mce_' . $version_path . '/get_page_list.php",
                    external_files_list_url: "' . PATH . SOFTWARE_DIRECTORY . '/tiny_mce_' . $version_path . '/get_file_list.php",
                    external_folder_list_url: "' . PATH . SOFTWARE_DIRECTORY . '/tiny_mce_' . $version_path . '/get_folder_list.php",
                    external_image_list_url: "' . PATH . SOFTWARE_DIRECTORY . '/tiny_mce_' . $version_path . '/get_image_list.php",
                    media_external_list_url: "' . PATH . SOFTWARE_DIRECTORY . '/tiny_mce_' . $version_path . '/get_media_list.php",
                    width: \'100%\',
                    theme_advanced_resizing: ' . $output_theme_advanced_resizing . ',
                    theme_advanced_resizing_use_cookie: false,
                    convert_urls: false,
                    ' . $output_browser_spellcheck . '
                    remove_script_host: false,
                    folder_id: "' . $folder_id . '",
                    environment_suffix: "' . ENVIRONMENT_SUFFIX . '",
                    cdn: ' . $cdn . '
                });
            </script>';
    }
}

function get_calendar($calendar_id, $calendars, $view, $status, $user, $date, $link, $number_of_upcoming_events = '', $return = 'html') {
    
    // if at least one calendar was supplied, continue
    if ($calendars) {
        // assume that calendar id has not been verified until we find out otherwise
        $calendar_id_verified = false;
        
        // loop through calendars in order to verify if supplied calendar is one of the supplied calendars
        foreach ($calendars as $calendar) {
            if ($calendar['id'] == $calendar_id) {
                $calendar_id_verified = true;
            }
        }
        
        // clear calendar id if it was not verified
        if ($calendar_id_verified == false) {
            $calendar_id = '';
        }
        
        switch ($view) {
            case '':
            case 'monthly':
                // if date is supplied, then use date
                if ($date) {
                    // Split date into parts.
                    $date_parts = explode('-', $date);
                    $start_month = $date_parts[0];
                    $start_day = '01';
                    $start_year = $date_parts[2];
                    $start_timestamp = mktime(0, 0, 0, $start_month, $start_day, $start_year);
                    
                    $stop_month = $start_month;
                    $stop_day = date('t', $start_timestamp);
                    $stop_year = $start_year;
                    $stop_timestamp = mktime(0, 0, 0, $stop_month, $stop_day, $stop_year);

                // else date is not supplied, so use current month
                } else {
                    $timestamp = time();
                    
                    $start_month = date('m', $timestamp);
                    $start_day = '01';
                    $start_year = date('Y', $timestamp);
                    $start_timestamp = mktime(0, 0, 0, $start_month, $start_day, $start_year);

                    $stop_month = date('m', $timestamp);
                    $stop_day = date('t', $timestamp);
                    $stop_year = date('Y', $timestamp);
                    $stop_timestamp = mktime(0, 0, 0, $stop_month, $stop_day, $stop_year);
                }
                
                $decrease_timestamp = mktime(0, 0, 0, $start_month - 1, 1, $start_year);
                $decrease_date = date('m', $decrease_timestamp) . '-' . date('d', $decrease_timestamp) . '-' . date('Y', $decrease_timestamp);

                $increase_timestamp = mktime(0, 0, 0, $start_month + 1, 1, $start_year);
                $increase_date = date('m', $increase_timestamp) . '-' . date('d', $increase_timestamp) . '-' . date('Y', $increase_timestamp);
                
                $output_date_selection = '<a href="?date=' . h($decrease_date) . '&amp;calendar_id=' . h($calendar_id) . '&amp;view=' . h($view) . '&amp;status=' . h($status) . '" class="button_3d_secondary software_button_small_secondary" class="previous">&lt;</a> 
                                          <a href="?date" class="button_3d_secondary software_button_small_secondary this_month">This Month</a>
                                          <a href="?date=' . h($increase_date) . '&amp;calendar_id=' . h($calendar_id) . '&amp;view=' . h($view) . '&amp;status=' . h($status) . '" class="button_3d_secondary software_button_small_secondary" class="next">&gt;</a> &nbsp; 
                                          <span style="font-weight: bold; vertical-align: middle; font-size: 150%">' . h(date('F Y', $start_timestamp)) . '</span>';
                
                break;
                
            case 'weekly':
                // if date is supplied, then use date
                if ($date) {
                    // Split date into parts.
                    $date_parts = explode('-', $date);
                    $date_month = $date_parts[0];
                    $date_day = $date_parts[1];
                    $date_year = $date_parts[2];
                    $date_timestamp = mktime(0, 0, 0, $date_month, $date_day, $date_year);

                // else date is not supplied, so use current week
                } else {
                    $date_timestamp = time();
                }
                
                // if start date is Sunday
                if (date('l', $date_timestamp) == 'Sunday') {
                    $week_start_time = strtotime('Sunday', $date_timestamp);
                } else {
                    $week_start_time = strtotime('last Sunday', $date_timestamp);
                }
                
                $start_month = date('m', $week_start_time);
                $start_day = date('d', $week_start_time);
                $start_year = date('Y', $week_start_time);
                $start_timestamp = mktime(0, 0, 0, $start_month, $start_day, $start_year);
                
                $week_stop_time = strtotime('Saturday', $week_start_time);
                
                $stop_month = date('m', $week_stop_time);
                $stop_day = date('d', $week_stop_time);
                $stop_year = date('Y', $week_stop_time);
                $stop_timestamp = mktime(0, 0, 0, $stop_month, $stop_day, $stop_year);
                
                $decrease_timestamp = strtotime('last sunday 12:00:00', $start_timestamp);
                $decrease_date = date('m', $decrease_timestamp) . '-' . date('d', $decrease_timestamp) . '-' . date('Y', $decrease_timestamp);
                
                $increase_timestamp = strtotime('2 Sunday', $start_timestamp);
                $increase_date = date('m', $increase_timestamp) . '-' . date('d', $increase_timestamp) . '-' . date('Y', $increase_timestamp);
                
                $output_date_selection = '<a href="?date=' . h($decrease_date) . '&amp;calendar_id=' . h($calendar_id) . '&amp;view=' . h($view) . '&amp;status=' . h($status) . '" class="software_button_small_secondary">&lt;</a> <span style="font-weight: bold; vertical-align: middle;">' . get_absolute_time(array('timestamp' => $week_start_time, 'type' => 'date', 'size' => 'long')) . ' - ' . get_absolute_time(array('timestamp' => $week_stop_time, 'type' => 'date', 'size' => 'long')) . '</span> <a href="?date=' . h($increase_date) . '&amp;calendar_id=' . h($calendar_id) . '&amp;view=' . h($view) . '&amp;status=' . h($status) . '" class="software_button_small_secondary">&gt;</a> <a href="?date" class="software_button_small_secondary this_week">This Week</a>';
                
                break;

            case 'upcoming':
                // set current timestamp
                $date_timestamp = time();
                
                // set start date
                $start_month = date('m', $date_timestamp);
                $start_day = date('d', $date_timestamp);
                $start_year = date('Y', $date_timestamp);

                // set end date
                $stop_month = '99';
                $stop_day = '99';
                $stop_year = '9999';
                
                break;
        }
        
        $start_date = $start_month . '-' . $start_day . '-' . $start_year;
        $start_date_for_comparison = $start_year . '-' . $start_month . '-' . $start_day;
        $stop_date = $stop_month . '-' . $stop_day . '-' . $stop_year;
        $stop_date_for_comparison = $stop_year . '-' . $stop_month . '-' . $stop_day;
        
        $output_calendar_pick_list = '';
        
        // if the view is monthly, weekly, or if we are in the backend, then prepare calendar update form
        if (($view == 'monthly') || ($view == 'weekly') || ($user)) {
            // if there is more than one calendar, then prepare to output calendar pick list
            if (count($calendars) > 1) {
                $output_calendar_options = '<option value="">-All Calendars-</option>';
                
                // loop through all calendars in order to prepare calendar pick list
                foreach ($calendars as $calendar) {
                    // if calendar is the current selected calendar, prepare to select calendar
                    if ($calendar['id'] == $calendar_id) {
                        $selected = ' selected="selected"';
                    } else {
                        $selected = '';
                    }
                    
                    // prepare to output calendar option
                    $output_calendar_options .= '<option value="' . h($calendar['id']) . '"' . $selected . '>' . h($calendar['name']) . '</option>';
                }

                $output_calendar_pick_list_class = '';
                
                // if this is for a front-end view, then set class
                if (!$user) {
                    $output_calendar_pick_list_class = ' class="software_select"';
                }
                
                $output_calendar_pick_list = '<select name="calendar_id"' . $output_calendar_pick_list_class . '>' . $output_calendar_options . '</select>';
            }
            
            $output_view_pick_list_class = '';
            $output_view_monthly_selected = '';
            $output_view_weekly_selected = '';
            
            switch ($view) {
                case 'monthly':
                    $output_view_monthly_selected = ' selected="selected"';
                    break;
                    
                case 'weekly':
                    $output_view_weekly_selected = ' selected="selected"';
                    break;
            }
            
            // if this is for a front-end view, then set class
            if (!$user) {
                $output_view_pick_list_class = ' class="software_select"';
            }
            
            $output_status_pick_list = '';
            
            // if a user was supplied, then this is for the back-end, so prepare to output status pick list
            if ($user) {
                $all_selected = '';
                $published_selected = '';
                $not_published_selected = '';
                
                switch ($status) {
                    case '':
                        $all_selected = ' selected="selected"';
                        break;
                        
                    case 'published':
                        $published_selected = ' selected="selected"';
                        break;
                        
                    case 'not_published':
                        $not_published_selected = ' selected="selected"';
                        break;
                }
                
                // output status pick list
                $output_status_pick_list = '<td class="mobile_left mobile_width" style="padding-right: 1em; padding-bottom: .5em;">Status: <select name="status"><option value=""' . $all_selected . '>-All-</option><option value="published"' . $published_selected . '>Published</option><option value="not_published"' . $not_published_selected . '>*Not Published</option></select></td>';
            }
            
            // if this is for a back-end view then set the update button class
            if ($user) {
                $output_update_button_class = ' class="submit_small_secondary"';
                
            // else this is for a front-end view
            } else {
                $output_update_button_class = ' class="software_input_submit_small_secondary"';
            }
            
            $output_update_calendar_view_form = 
                '<div class="calendar_view">
                    <div style="float: left; padding-right: 1em; padding-bottom: .5em; margin-top: .25em;">' . $output_date_selection . '</div>
                    <div class="desktop_right" style="float: right">
                        <form method="get">
                            <input type="hidden" name="date" value="' . h($date) . '">
                            <table>
                                <tr>
                                    <td class="mobile_left mobile_width" style="padding-right: 1em; padding-bottom: .5em;">
                                        ' . $output_calendar_pick_list . '
                                    </td>
                                    <td class="mobile_left mobile_width" style="padding-right: 1em; padding-bottom: .5em;">
                                        <select name="view"' . $output_view_pick_list_class . '>
                                            <option value="monthly"' . $output_view_monthly_selected . '>Monthly</option>
                                            <option value="weekly"' . $output_view_weekly_selected . '>Weekly</option>
                                        </select>
                                    </td>
                                        ' . $output_status_pick_list . '
                                    <td class="mobile_left mobile_width" style="padding-right: 0; padding-bottom: .5em;">
                                        <input type="submit" name="submit_update" value="Update"' . $output_update_button_class . '/>
                                    </td>
                                </tr>
                            </table>
                        </form>
                    </div>
                    <div style="clear: both"></div>
                </div>';
        }
        
        // prepare sql for status
        
        $sql_status = "";
        
        switch ($status) {
            case 'published':
                $sql_status = "(calendar_events.published = '1')";
                break;
                
            case 'not_published':
                $sql_status = "(calendar_events.published = '0')";
                break;
        }
        
        $sql_calendar = "";
        
        // if a calendar was supplied, prepare SQL to only get calendar events for that calendar
        if ($calendar_id) {
            // if there is sql for the status, then prepare AND
            if ($sql_status != '') {
                $sql_calendar .= "AND ";
            }
            $sql_join = "LEFT JOIN calendar_events_calendars_xref ON calendar_events.id = calendar_events_calendars_xref.calendar_event_id";
            $sql_calendar .= "(calendar_events_calendars_xref.calendar_id = '" . escape($calendar_id) . "')";
            
        // else a calendar was not supplied, so prepare SQL to only get calendars that were supplied
        } else {
            // If this is not the back end
            if (!$user) {
                // if there is sql for the status, then prepare AND
                if ($sql_status != '') {
                    $sql_calendar .= "AND ";
                }
                
                $sql_calendars = "";
                
                // For each calendar, create an or sql statement
                foreach ($calendars as $calendar) {
                    if ($sql_calendars) {
                        $sql_calendars .= " OR ";
                    }
                    
                    $sql_calendars .= "(calendar_events_calendars_xref.calendar_id = '" . escape($calendar['id']) . "')";
                }
                
                $sql_calendar .= "($sql_calendars)";
                $sql_join = "LEFT JOIN calendar_events_calendars_xref ON calendar_events.id = calendar_events_calendars_xref.calendar_event_id";
            }
        }
        
        $sql_where = "";
        
        if (($sql_status != '') || ($sql_calendar != '')) {
            $sql_where =
                "WHERE
                    $sql_status
                    $sql_calendar";
        }

        // get calendar event exceptions
        $query =
            "SELECT
                calendar_event_id,
                recurrence_number
            FROM calendar_event_exceptions";
        $result = mysqli_query(db::$con, $query);

        $calendar_event_exceptions = array();
        // Place all of the exceptions into an array
        while ($row = mysqli_fetch_assoc($result)) {
            $calendar_event_exceptions[] = $row;
        }

        // get calendar events
        $query =
            "SELECT
                calendar_events.id,
                calendar_events.name,
                calendar_events.published,
                calendar_events.short_description,
                calendar_events.all_day,
                calendar_events.start_time,
                calendar_events.recurrence_number,
                calendar_events.recurrence_type,
                calendar_events.recurrence_day_sun,
                calendar_events.recurrence_day_mon,
                calendar_events.recurrence_day_tue,
                calendar_events.recurrence_day_wed,
                calendar_events.recurrence_day_thu,
                calendar_events.recurrence_day_fri,
                calendar_events.recurrence_day_sat,
                calendar_events.recurrence_month_type,
                calendar_events.reservations
            FROM calendar_events
            $sql_join
            $sql_where";
        $result = mysqli_query(db::$con, $query);
        $calendar_events = array();

        // Add each event information to an array
        while ($row = mysqli_fetch_assoc($result)) {
            $calendar_events[] = $row;
        }

        $events = array();
        $processed_events = array();
        
        // loop through all calendar events
        foreach ($calendar_events as $calendar_event) {
            $id = $calendar_event['id'];
            $name = $calendar_event['name'];
            $published = $calendar_event['published'];
            $short_description = $calendar_event['short_description'];
            $all_day = $calendar_event['all_day'];
            $event_start_date_and_time = $calendar_event['start_time'];
            $recurrence_number = $calendar_event['recurrence_number'];
            $recurrence_type = $calendar_event['recurrence_type'];
            $recurrence_day_sun = $calendar_event['recurrence_day_sun'];
            $recurrence_day_mon = $calendar_event['recurrence_day_mon'];
            $recurrence_day_tue = $calendar_event['recurrence_day_tue'];
            $recurrence_day_wed = $calendar_event['recurrence_day_wed'];
            $recurrence_day_thu = $calendar_event['recurrence_day_thu'];
            $recurrence_day_fri = $calendar_event['recurrence_day_fri'];
            $recurrence_day_sat = $calendar_event['recurrence_day_sat'];
            $recurrence_month_type = $calendar_event['recurrence_month_type'];
            $reservations = $calendar_event['reservations'];
            
            // If the event id has not already been processed, process it and log that it has been processed.
            if (!in_array($id, $processed_events)) {
                // Keep track that we have already processed this event.
                $processed_events[] = $id;
                // if a user was not supplied or user has access to calendar event, then continue
                if (!$user || (validate_calendar_event_access($id) == true)) {
                    // split event start date and time into parts
                    $event_start_date_and_time_parts = explode(' ', $event_start_date_and_time);
                    $event_start_date = $event_start_date_and_time_parts[0];
                    $event_start_time = $event_start_date_and_time_parts[1];
                    
                    $calendar_exceptions = array();

                    // if recurrence number is greater than zero, then split event start date into parts, that we will use later
                    if ($recurrence_number > 0) {
                        $event_start_date_parts = explode('-', $event_start_date);
                        $event_start_year = $event_start_date_parts[0];
                        $event_start_month = $event_start_date_parts[1];
                        $event_start_day = $event_start_date_parts[2];
                        
                        // Loop through all calendar exceptions and separate the ones we need for the event we are working with into $calendar_exceptions
                        foreach ($calendar_event_exceptions as $calendar_event_exception) {
                            if ($calendar_event_exception[calendar_event_id] == $id) {
                                $calendar_exceptions[] = $calendar_event_exception[recurrence_number];
                            }
                        }

                        // If this is a monthly event and the month type is "day of the week",
                        // then determine which week in the month the event is on.
                        // If the week is 1-4 then we will use that, however if the week is 5,
                        // then we interpret that as the last week.
                        if (
                            ($recurrence_type == 'month')
                            && ($recurrence_month_type == 'day_of_the_week')
                        ) {
                            $day_of_the_week = date('l', strtotime($event_start_date));
                            $first_day_of_the_month_timestamp = strtotime($event_start_year . '-' . $event_start_month . '-01');

                            $week = '';

                            // Create a loop in order to determine which week event falls on.
                            // We only loop through 4 weeks, because we are going to set "last" below for 5th week.
                            for ($week_index = 0; $week_index <= 3; $week_index++) {
                                // If the event is in this week, then remember the week number and break out of this loop.
                                if ($event_start_date == date('Y-m-d', strtotime('+' . $week_index . ' week ' . $day_of_the_week, $first_day_of_the_month_timestamp))) {
                                    $week = $week_index + 1;
                                    break;
                                }
                            }

                            // If a week was not found, then that means it falls on the 5th week,
                            // so set it to be the last week.
                            if ($week == '') {
                                $week = 'last';
                            }
                        }
                    }
                    
                    // loop in order to create a new event for each recurrence
                    for ($i = 0; $i <= $recurrence_number; $i++) {
                        // We will use this variable to stop the calculations if the recurrence number is hidden by an exception
                        $halt_on_exception = '0';
                        $hidden_exception = '0';
                        // If calendar_exceptions is not empty.
                        if (count($calendar_exceptions) > 0) {
                            // Check if this recurrence is an exception.
                            if (in_array($i, $calendar_exceptions)) {
                                // If we are not in software's backend, do not worry about this date anymore.
                                if (!$user) {
                                    $halt_on_exception = '1';
                                // else, we are in softwares backend, so print the correct link (remove, unremove) to all the user to make an exception
                                } else {
                                    $halt_on_exception = '0';
                                    $hidden_exception = '1';
                                }
                            // The recurrence was not an exception, so do not hide it.
                            } else {
                                $hidden_exception = '0';
                            }
                        }
                        
                        // if recurrence number is greater than 0, then adjust event start date
                        if ($i > 0) {
                            // adjust event start date depending on recurrence type
                            switch ($recurrence_type) {
                                // Daily
                                case 'day':
                                    $count = 0;

                                    // Loop through days in the future until we find a date that is valid
                                    // based on the valid days of the week that were selected.
                                    while (true) {
                                        $new_time = strtotime('+1 day', strtotime($event_start_date));
                                        $event_start_date = date('Y-m-d', $new_time);
                                        $day_of_the_week = strtolower(date('D', $new_time));

                                        // If this day of the week is valid for this calendar event,
                                        // then we have found a valid date, so break out of the loop.
                                        if (${'recurrence_day_' . $day_of_the_week} == 1) {
                                            break;
                                        }

                                        $count++;

                                        // If we have already looped 7 times, then something is wrong,
                                        // so break out of this loop and the recurrence loop above.
                                        // This should never happen but is added just in case in order to
                                        // prevent an endless loop.
                                        if ($count == 7) {
                                            break 3;
                                        }
                                    }

                                    break;
                                    
                                // Weekly
                                case 'week':
                                    $new_time = mktime(0, 0, 0, $event_start_month, $event_start_day + (7 * $i), $event_start_year);
                                    $event_start_date = date('Y', $new_time) . '-' . date('m', $new_time) . '-' . date('d', $new_time);
                                    break;
                                
                                // Monthly
                                case 'month':
                                    switch ($recurrence_month_type) {
                                        case 'day_of_the_month':
                                            $new_time = mktime(0, 0, 0, $event_start_month + $i, 1, $event_start_year);
                                            $new_event_start_year = date('Y', $new_time);
                                            $new_event_start_month = date('m', $new_time);
                                            $new_event_start_day = $event_start_day;

                                            // if date is not valid, then get last date for month
                                            if (checkdate($new_event_start_month, $new_event_start_day, $new_event_start_year) == false) {
                                                $new_event_start_day = date('t', mktime(0, 0, 0, $new_event_start_month, 1, $new_event_start_year));
                                            }

                                            $event_start_date = $new_event_start_year . '-' . $new_event_start_month . '-' . $new_event_start_day;

                                            break;

                                        case 'day_of_the_week':
                                            $first_day_of_the_month_timestamp = mktime(0, 0, 0, $event_start_month + $i, 1, $event_start_year);

                                            // If the week is 1-4 then find the date in a certain way.
                                            if ($week != 'last') {
                                                $week_index = $week - 1;

                                                $new_time = strtotime('+' . $week_index . ' week ' . $day_of_the_week, $first_day_of_the_month_timestamp);

                                            // Otherwise the week is last, so find the date in a different way.
                                            } else {
                                                $last_day_of_the_month_timestamp = strtotime(date('Y-m-t', $first_day_of_the_month_timestamp));

                                                // If the last day of the month happens to be the right day of the week,
                                                // then thats that day that we want.
                                                if (date('l', $last_day_of_the_month_timestamp) == $day_of_the_week) {
                                                    $new_time = $last_day_of_the_month_timestamp;

                                                // Otherwise find the day of the week that we want in the last week of the month.
                                                } else {
                                                    $new_time = strtotime('last ' . $day_of_the_week, $last_day_of_the_month_timestamp);
                                                }
                                            }

                                            $event_start_date = date('Y-m-d', $new_time);

                                            break;
                                    }

                                    break;
                                
                                // Yearly
                                case 'year':
                                    $new_event_start_year = $event_start_year + $i;
                                    $new_event_start_month = $event_start_month;
                                    $new_event_start_day = $event_start_day;
                                    
                                    // if date is not valid, then get last date for month
                                    if (checkdate($new_event_start_month, $new_event_start_day, $new_event_start_year) == false) {
                                        $new_event_start_day = date('t', mktime(0, 0, 0, $new_event_start_month, 1, $new_event_start_year));
                                    }
                                    
                                    $event_start_date = $new_event_start_year . '-' . $new_event_start_month . '-' . $new_event_start_day;
                                    break;
                            }
                        }

                        if ($halt_on_exception == '0') {
                            // if the event start date is in the selected date range, add event to array
                            if (($start_date_for_comparison <= $event_start_date) && ($event_start_date <= $stop_date_for_comparison)) {
                                $events[] =
                                    array (
                                        'id' => $id,
                                        'name' => $name,
                                        'published' => $published,
                                        'short_description' => $short_description,
                                        'event_start_date' => $event_start_date,
                                        'event_start_time' => $event_start_time,
                                        'event_start_date_and_time' => $event_start_date . ' ' . $event_start_time,
                                        'recurring_event' => $recurrence_number,
                                        'recurrence_number' => $i,
                                        'all_day' => $all_day,
                                        'hidden_exception' => $hidden_exception,
                                        'reservations' => $reservations,
                                    );
                            }
                        }

                        // if the event start date is greater than or equal to stop date, then break out of loop,
                        // because it is not possible for further recurrences to be in date range
                        if ($event_start_date >= $stop_date_for_comparison) {
                            break;
                        }
                    }
                }
            }
        }
        
        // create array for storing event start dates and times, so that we can sort events later
        $event_start_dates_and_times = array();

        // If this is for the front-end, then create array to store the calendars that calendar events are assigned to.
        // We store this data in an array, so that we don't have to do duplicate
        // database queries for multiple recurrences of the same event.
        if (!$user) {
            $calendar_events_calendars = array();
        }
        
        // Loop through events in order to populate start date and times array
        // and to get calendars that each event is assigned to.
        foreach ($events as $key => $event) {
            $event_start_dates_and_times[$key] = $event['event_start_date_and_time'];

            // If this is for the front-end, then prepare to get calendars for this event
            // so we can output a class for each calendar (e.g. in order to add a different
            // background color behind each event based on the calendar that the event is in).
            if (!$user) {
               // If this event has not already been processed, then get calendars that this event is assigned to.
               if (isset($calendar_events_calendars[$event['id']]) == FALSE) {
                    $query = "SELECT calendar_id FROM calendar_events_calendars_xref WHERE calendar_event_id = '" . $event['id'] . "'";
                    $result = mysqli_query(db::$con, $query);

                    $calendar_events_calendars[$event['id']] = array();

                    while ($row = mysqli_fetch_assoc($result)) {
                        $calendar_events_calendars[$event['id']][] = $row['calendar_id'];
                    }
               }

               // Store list of calendars for this event.
               $events[$key]['calendars'] = $calendar_events_calendars[$event['id']];
           }
        }
        
        // sort events by date and time
        array_multisort($event_start_dates_and_times, SORT_ASC, $events);
        
        // Output Calendar based on View
        
        if ($view == 'upcoming') {
            
            // If the request asked for the data as an array instead of HTML, then just return that.
            if ($return == 'array') {
                if ($number_of_upcoming_events) {
                    return array_slice($events, 0, $number_of_upcoming_events);
                } else {
                    return $events;
                }
            }
            
            // Output Upcoming View

                // if there is at least one event, prepare to list events
                if ($events) {
                    // If the date format is month and then day, then use that format.
                    if (DATE_FORMAT == 'month_day') {
                        $month_and_day_format = 'F j';

                    // Otherwise the date format is day and then month, so use that format.
                    } else {
                        $month_and_day_format = 'j F';
                    }

                    $output_calendar_data = '';
                    $event_counter = 0;
                    
                    // loop through all events in order to build list
                    foreach ($events as $key => $event) {
                        // increment the event counter
                        $event_counter++;
                        
                        // if event start date is not equal to previous event start date, then open new container for this day
                        if ($event['event_start_date'] != $events[$key - 1]['event_start_date']) {
                            $output_calendar_data .= '<p class="row_' . ($event_counter % 2) . '"><span style="font-weight:bold;">' . h(date('l, ' . $month_and_day_format, strtotime($event['event_start_date']))) . '</span><br>';
                        }

                        $output_calendar_classes = '';

                        // Loop through the calendars for this event in order to add a class for calendar that the event is assigned to.
                        // (e.g. in order to add a different background color behind each event based on the calendar that the event is in).
                        foreach ($event['calendars'] as $calendar) {
                            // If a class has already been added, then add a space for separation.
                            if ($output_calendar_classes != '') {
                                $output_calendar_classes .= ' ';
                            }
                            
                            $output_calendar_classes .= 'calendar_' . $calendar;
                        }
                        
                        $recurring_query_string = '';
                        
                        // If the event is recurring.
                        if ($event['recurrence_number']) {
                            $recurring_query_string = '&recurrence_number=' . $event['recurrence_number'];
                        }
                        
                        // if there should be a link, then prepare link
                        if ($link) {
                            // get additional calendar event data
                            $calendar_event = get_calendar_event($event['id'], $event['recurrence_number']);
                            
                            $output_availability_icon = '';
                            
                            // if this event is published
                            // and it does not have an exception
                            // and reservations are enabled
                            // and the reservation product still exists
                            // and the reservation product is enabled
                            // then determine which availability icon we should show
                            if (
                                ($event['published'] == 1)
                                && ($event['hidden_exception'] == 0)
                                && ($event['reservations'] == 1)
                                && ($calendar_event['product_id'] != '')
                                && ($calendar_event['product_enabled'] == 1)
                            ) {
                                // if the event is not in the past
                                // and reservations are not limited
                                // or the number of remaining spots is greater than 0
                                // and inventory is disabled for product
                                // or the inventory quantity is greater than 0
                                // or the product is allowed to be backordered
                                // then this calendar event is available, so output available icon
                                if (
                                    (strtotime($calendar_event['end_date_and_time']) >= time())
                                    &&
                                    (
                                        ($calendar_event['limit_reservations'] == 0)
                                        || ($calendar_event['number_of_remaining_spots'] > 0)
                                    )
                                    &&
                                    (
                                        ($calendar_event['inventory'] == 0)
                                        || ($calendar_event['inventory_quantity'] > 0)
                                        || ($calendar_event['backorder'] == 1)
                                    )
                                ) {
                                    $output_availability_icon = '<span style="color: green; font-weight: bold; font-size: 130%; margin-right: .25em">&#9679;</span>';
                                    
                                // else this calendar event is not available, so output unavailable icon
                                } else {
                                    $output_availability_icon = '<span style="color: lightgrey; font-weight: bold; font-size: 130%; margin-right: .25em">&#9679;</span>';
                                }
                            }
                            
                            $output_link = $output_availability_icon . '<a href="' . h(escape_url($link)) . '?id=' . h($event['id']) . h($recurring_query_string) . '">' . h($event['name']) . '</a>';
                            
                        // else there should not be a link, so prepare plain text
                        } else {
                            $output_link = h($event['name']);
                        }
                        
                        // output event on it's own line
                        $output_calendar_data .= '<span class="' . $output_calendar_classes . '">' . $output_link . '</span><br />';
                        
                        // if next event start date is not equal to current event start date, or if this is the last event to be outputted, then close container for this day
                        if (($events[$key + 1]['event_start_date'] != $event['event_start_date']) || ($event_counter == $number_of_upcoming_events)) {
                            $output_calendar_data .= '</p>';
                        }
                        
                        // if the number of upcoming events is greater than zero, and if the event counter is equal to the number of upcoming events
                        // then break loop.
                        if (($number_of_upcoming_events > 0) && ($event_counter == $number_of_upcoming_events)) {
                            break;
                        }
                    }
                    
                // else there are no events, so prepare notice
                } else {
                    $output_calendar_data = '<div>There are no upcoming calendar events.</div>';
                }
                
                $output_calendar = '<div class="upcoming">' . $output_calendar_data . '</div>';
                
        } else {

            // Output Monthly View

            if ($view != 'weekly') {
                $days = array();

                // loop through all events in order to add events to days array
                foreach ($events as $key => $event) {
                    // split event start date into parts in order to get day of the month
                    $event_start_date_parts = explode('-', $event['event_start_date']);
                    $event_start_day = $event_start_date_parts[2];
                    
                    // remove 0 before day number
                    settype($event_start_day, "integer");
                    
                    // add event to days array
                    $days[$event_start_day][] = $event;
                }

                $output_calendar_data = '';

                $start_day_of_week = date('w', $start_timestamp);
                $stop_day_of_week = date('w', $stop_timestamp);

                $output_calendar_data .= '<tr class="data">';

                // loop through all days from month before that appear greyed-out on calendar
                for ($i = 1; $i <= $start_day_of_week; $i++) {
                    $output_calendar_data .= '<td class="inactive">&nbsp;</td>';
                }

                // loop through all days in the month
                for ($i = 1; $i <= $stop_day; $i++) {
                    $day_timestamp = mktime(0, 0, 0, $start_month, $i, $start_year);
                    $current_day_timestamp = mktime(0, 0, 0, date('m'), date('d'), date('Y'));
                    $day_of_week = date('l', $day_timestamp);
                    
                    // if this day is a Sunday and this is not the first day, then start new row
                    if (($day_of_week == 'Sunday') && ($i != 1)) {
                        $output_calendar_data .= '<tr class="data">';
                    }
                    
                    if ($day_timestamp == $current_day_timestamp) {
                        $output_day_number = '<span class="today">' . $i . '</span>';
                    } else {
                        $output_day_number = $i;
                    }
                    
                    // start cell
                    $output_calendar_data .=
                        '<td>
                            <div style="text-align: right; margin-bottom: 10px">' . $output_day_number . '</div>';
                    
                    // if there are events for this day
                    if ($days[$i]) {
                        // loop through all events for this day
                        foreach ($days[$i] as $event) {
                            // Initialize variables
                            $start_strikethrough_tag = '';
                            $end_strikethrough_tag = '';
                            $exception_link = '';
                            $recurring_query_string = '';
                            $asterisk = '';
                            $link_access = TRUE;
                            $output_calendar_classes = '';

                            // If this is for the front-end, then output a class for each calendar that the event is in
                            // (e.g. in order to add a different background color behind each event based on the calendar that the event is in).
                            if (!$user) {
                                // Loop through the calendars for this event in order to prepare classes.
                                foreach ($event['calendars'] as $calendar) {
                                    // If a class has already been added, then add a space for separation.
                                    if ($output_calendar_classes != '') {
                                        $output_calendar_classes .= ' ';
                                    }

                                    $output_calendar_classes .= 'calendar_' . $calendar;
                                }
                            }
                            
                            // get additional calendar event data
                            $calendar_event = get_calendar_event($event['id'], $event['recurrence_number']);
                            
                            // If the event is recurring.
                            if ($event['recurrence_number']) {
                                $recurring_query_string = '&recurrence_number=' . $event['recurrence_number'];
                            }
                            
                            // If we are in softwares backend, then prepare items that only appear in the backend
                            if ($user) {
                                // If this event is an exception, then prepare to output strikethrough tag and remove exception link
                                if ($event['hidden_exception'] == '1') {
                                    $start_strikethrough_tag = '<span class="calendar_event_exception">';
                                    $end_strikethrough_tag = '</span>';
                                    
                                    // If the calendar event is not published, or the user is greater than a basic user, or the user has publish rights, then prepare to output exception link
                                    if (($event['published'] == 0) || ($user['role'] < 3) || ($user['publish_calendar_events'] == TRUE)) {
                                        $exception_link = '<a href="update_calendar_event_exception.php?action=delete&amp;calendar_event_id=' . h($event['id']) . '&amp;recurrence_number=' . h($event['recurrence_number']) . get_token_query_string_field() . '">Unremove</a>';
                                    }
                                    
                                // Else this event is not an exception, so if it is a recurring event, then prepare to output add exception link
                                } else if ($event['recurring_event'] > 0) {
                                    // If the calendar event is not published, or the user is greater than a basic user, or the user has publish rights, then prepare to output exception link
                                    if (($event['published'] == 0) || ($user['role'] < 3) || ($user['publish_calendar_events'] == TRUE)) {
                                        $exception_link = '<a href="update_calendar_event_exception.php?action=create&amp;calendar_event_id=' . h($event['id']) . '&amp;recurrence_number=' . h($event['recurrence_number']) . get_token_query_string_field() . '">Remove</a>';
                                    }
                                }
                                
                                // if event is not published, then prepare asterisk
                                if ($event['published'] == 0) {
                                    $asterisk = '*';
                                    
                                // Else the event is published, so check to make sure that user has access to edit calendar event
                                } else {
                                    // if user is a basic user, and does not have rights to publish calendar events then the user should not have access to the link
                                    if (($user['role'] == 3) && ($user['publish_calendar_events'] == FALSE)) {
                                        $link_access = FALSE;
                                    }
                                }
                            }
                            
                            $output_availability_icon = '';
                            
                            // if this event is published
                            // and it does not have an exception
                            // and reservations are enabled
                            // and the reservation product still exists
                            // and the reservation product is enabled
                            // then determine which availability icon we should show
                            if (
                                ($event['published'] == 1)
                                && ($event['hidden_exception'] == 0)
                                && ($event['reservations'] == 1)
                                && ($calendar_event['product_id'] != '')
                                && ($calendar_event['product_enabled'] == 1)
                            ) {
                                // if the event is not in the past
                                // and reservations are not limited
                                // or the number of remaining spots is greater than 0
                                // and inventory is disabled for product
                                // or the inventory quantity is greater than 0
                                // or the product is allowed to be backordered
                                // then this calendar event is available, so output available icon
                                if (
                                    (strtotime($calendar_event['end_date_and_time']) >= time())
                                    &&
                                    (
                                        ($calendar_event['limit_reservations'] == 0)
                                        || ($calendar_event['number_of_remaining_spots'] > 0)
                                    )
                                    &&
                                    (
                                        ($calendar_event['inventory'] == 0)
                                        || ($calendar_event['inventory_quantity'] > 0)
                                        || ($calendar_event['backorder'] == 1)
                                    )
                                ) {
                                    $output_availability_icon = '<span style="color: green; font-weight: bold; font-size: 130%; margin-right: .25em">&#9679;</span>';
                                    
                                // else this calendar event is not available, so output unavailable icon
                                } else {
                                    $output_availability_icon = '<span style="color: lightgrey; font-weight: bold; font-size: 130%; margin-right: .25em">&#9679;</span>';
                                }
                            }
                            
                            // If there is a page to link and the user has access to the link, then prepare to output the event name with a link
                            if ($link && ($link_access == TRUE)) {
                                $output_link = '<a href="' . h(escape_url($link)) . '?id=' . h($event['id']) . h($recurring_query_string) . '">' . $output_availability_icon . '<strong>' . h($event['name']) . '</strong></a><br>';
                                
                            // Else there should not be a link, so prepare to just output the event name
                            } else {
                                $output_link = $output_availability_icon . '<strong>' . h($event['name']) . '</strong><br>';
                            }
                            
                            $output_time = '';
                            
                            // If there is a time range, then output it.
                            if ($calendar_event['time_range'] != '') {
                                $output_time = $calendar_event['time_range'] . '<br>';
                            }
                            
                            $output_calendar_data .=
                                '<div class="' . $output_calendar_classes . '" style="margin-bottom: .8em">
                                    ' . $start_strikethrough_tag . '
                                    ' . $asterisk . $output_link . '
                                    ' . $output_time . '
                                    ' . $end_strikethrough_tag . '
                                    ' . $exception_link . '
                                </div>';
                        }
                    }
                    
                    // end cell
                    $output_calendar_data .= '</td>';
                    
                    // if this day is a Saturday and this is not the last day, then close row
                    if (($day_of_week == 'Saturday') && ($i != $stop_day)) {
                        $output_calendar_data .= '</tr>';
                    }
                }

                // loop through all days from month after that appear greyed-out on calendar
                for ($i = $stop_day_of_week + 1; $i <= 6; $i++) {
                    $output_calendar_data .= '<td class="inactive">&nbsp;</td>';
                }

                $output_calendar_data .= '</tr>';
                
                // if this is for a back-end view
                if ($user) {
                    $output_monthly_calendar_class = ' class="monthly_calendar"';
                    
                // else this is for a front-end view
                } else {
                    $output_monthly_calendar_class = ' class="software_monthly_calendar"';
                }
                
                $output_calendar = 
                    '<div class="monthly">
                        <table' . $output_monthly_calendar_class . '>
                        <tr class="heading">
                            <th>Sunday</th>
                            <th>Monday</th>
                            <th>Tuesday</th>
                            <th>Wednesday</th>
                            <th>Thursday</th>
                            <th>Friday</th>
                            <th>Saturday</th>
                        </tr>
                        ' . $output_calendar_data . '
                        </table>
                    </div>';
            }

            // Output Weekly View (including Monthly View for responsive designs)

                // if there is at least one event, prepare to list events
                if ($events) {
                    $output_fieldset_class = '';
                    
                    // if this is for a front-end view
                    if (!$user) {
                        $output_fieldset_class = ' class="software_fieldset"';
                    }
                    
                    $output_legend_class = '';
                    
                    // if this is for a front-end view
                    if (!$user) {
                        $output_legend_class = ' class="software_legend"';
                    }
                    
                    $output_calendar_data = '';
                    
                    // loop through all events in order to build list
                    foreach ($events as $key => $event) {
                        // if event start date is not equal to previous event start date, then open new container for this day
                        if ($event['event_start_date'] != $events[$key - 1]['event_start_date']) {
                            $output_calendar_data .=
                                '<fieldset style="margin-bottom: 15px"' . $output_fieldset_class . '>
                                    <legend' . $output_legend_class . '>' . get_absolute_time(array('timestamp' => strtotime($event['event_start_date']), 'type' => 'date', 'size' => 'long')) . '</legend>
                                    <div style="margin: 10px">';
                        }
                        
                        // Initialize variables
                        $start_strikethrough_tag = '';
                        $end_strikethrough_tag = '';
                        $exception_link = '';
                        $recurring_query_string = '';
                        $asterisk = '';
                        $link_access = TRUE;
                        $output_calendar_classes = '';

                        // If this is for the front-end, then output a class for each calendar that the event is in
                        // (e.g. in order to add a different background color behind each event based on the calendar that the event is in).
                        if (!$user) {
                            // Loop through the calendars for this event in order to prepare classes.
                            foreach ($event['calendars'] as $calendar) {
                                // If a class has already been added, then add a space for separation.
                                if ($output_calendar_classes != '') {
                                    $output_calendar_classes .= ' ';
                                }

                                $output_calendar_classes .= 'calendar_' . $calendar;
                            }
                        }
                        
                        // get additional calendar event data
                        $calendar_event = get_calendar_event($event['id'], $event['recurrence_number']);
                        
                        // If we are in softwares backend, then prepare items that only appear in the backend
                        if ($user) {
                            // If this event is an exception, then prepare to output strikethrough tag and remove exception link
                            if ($event['hidden_exception'] == '1') {
                                $start_strikethrough_tag = '<span class="calendar_event_exception">';
                                $end_strikethrough_tag = '</span>';
                                
                                // If the calendar event is not published, or the user is greater than a basic user, or the user has publish rights, then prepare to output exception link
                                if (($event['published'] == 0) || ($user['role'] < 3) || ($user['publish_calendar_events'] == TRUE)) {
                                    $exception_link = '<a href="update_calendar_event_exception.php?action=delete&amp;calendar_event_id=' . h($event['id']) . '&amp;recurrence_number=' . h($event['recurrence_number']) . get_token_query_string_field() . '">Unremove</a><br>';
                                }
                                
                            // Else this event is not an exception, so if it is a recurring event, then prepare to output add exception link
                            } else if ($event['recurring_event'] > 0) {
                                // If the calendar event is not published, or the user is greater than a basic user, or the user has publish rights, then prepare to output exception link
                                if (($event['published'] == 0) || ($user['role'] < 3) || ($user['publish_calendar_events'] == TRUE)) {
                                    $exception_link = '<a href="update_calendar_event_exception.php?action=create&amp;calendar_event_id=' . h($event['id']) . '&amp;recurrence_number=' . h($event['recurrence_number']) . get_token_query_string_field() . '">Remove</a><br>';
                                }
                            }
                            
                            // if event is not published, then prepare asterisk
                            if ($event['published'] == 0) {
                                $asterisk = '*';
                                
                            // Else the event is published, so check to make sure that user has access to edit calendar event
                            } else {
                                // if user is a basic user, and does not have rights to publish calendar events then the user should not have access to the link
                                if (($user['role'] == 3) && ($user['publish_calendar_events'] == FALSE)) {
                                    $link_access = FALSE;
                                }
                            }
                            
                        // Else we are working in the front-end (calendar view page type)
                        } else {
                            // If the event is recurring.
                            if ($event['recurrence_number']) {
                                $recurring_query_string = '&recurrence_number=' . $event['recurrence_number'];
                            }
                        }
                        
                        $output_availability_icon = '';
                        
                        // if this event is published
                        // and it does not have an exception
                        // and reservations are enabled
                        // and the reservation product still exists
                        // and the reservation product is enabled
                        // then determine which availability icon we should show
                        if (
                            ($event['published'] == 1)
                            && ($event['hidden_exception'] == 0)
                            && ($event['reservations'] == 1)
                            && ($calendar_event['product_id'] != '')
                            && ($calendar_event['product_enabled'] == 1)
                        ) {
                            // if the event is not in the past
                            // and reservations are not limited
                            // or the number of remaining spots is greater than 0
                            // and inventory is disabled for product
                            // or the inventory quantity is greater than 0
                            // or the product is allowed to be backordered
                            // then this calendar event is available, so output available icon
                            if (
                                (strtotime($calendar_event['end_date_and_time']) >= time())
                                &&
                                (
                                    ($calendar_event['limit_reservations'] == 0)
                                    || ($calendar_event['number_of_remaining_spots'] > 0)
                                )
                                &&
                                (
                                    ($calendar_event['inventory'] == 0)
                                    || ($calendar_event['inventory_quantity'] > 0)
                                    || ($calendar_event['backorder'] == 1)
                                )
                            ) {
                                $output_availability_icon = '<span style="color: green; font-weight: bold; font-size: 130%; margin-right: .25em">&#9679;</span>';
                                
                            // else this calendar event is not available, so output unavailable icon
                            } else {
                                $output_availability_icon = '<span style="color: lightgrey; font-weight: bold; font-size: 130%; margin-right: .25em">&#9679;</span>';
                            }
                        }
                        
                        // If there is a page to link and the user has access to the link, then prepare to output the event name with a link
                        if ($link && ($link_access == TRUE)) {
                            $output_link = '<a href="' . h(escape_url($link)) . '?id=' . h($event['id']) . h($recurring_query_string) . '">' . $output_availability_icon . '<strong>' . h($event['name']) . '</strong></a><br>';
                            
                        // Else there should not be a link, so prepare to just output the event name
                        } else {
                            $output_link = $output_availability_icon . '<strong>' . h($event['name']) . '</strong><br>';
                        }
                        
                        $output_time = '';
                        
                        // If there is a time range, then output it.
                        if ($calendar_event['time_range'] != '') {
                            $output_time = $calendar_event['time_range'] . '<br>';
                        }
                        
                        $output_calendar_data .=
                            '<p class="' . $output_calendar_classes . '">
                                ' . $start_strikethrough_tag . '
                                ' . $asterisk . $output_link . '
                                ' . $output_time . '
                                ' . $end_strikethrough_tag . '
                                ' . $exception_link . '
                                ' . h($event['short_description']) . '
                            </p>';
                        
                        // if next event start date is not equal to current event start date, then close container for this day
                        if ($events[$key + 1]['event_start_date'] != $event['event_start_date']) {
                            $output_calendar_data .=
                                '   </div>
                                </fieldset>';
                        }
                    }
                    
                // else there are no events, so prepare notice
                } else {
                    $output_calendar_data = '<div>There are no calendar events for your selection.</div>';
                }
                
                // if monthly or backend screen (null)
                if (($view == 'monthly') || ($view == '')) {
                // hide weekly output if monthly is present (and display using media queries for responsive designs)
                    $output_calendar .= '<div class="weekly" style="display: none;">' . $output_calendar_data . '</div>';
                }
                else {
                    $output_calendar .= '<div class="weekly">' . $output_calendar_data . '</div>';
                }
        }
        
        return
            '<div class="software_calendar">
                ' . $output_update_calendar_view_form . '
                ' . $output_calendar . '
            </div>';
    
    // else no calendars were supplied, so output error
    } else {
        // if this view is for the back-end
        if ($user) {
            return 'There are no calendars, so no calendar events could be displayed.';
            
        // else this view is for the front-end
        } else {
            return 'There are no calendars selected for this view, so no calendar events could be displayed.';
        }
    }
}

function convert_html_to_text($html) {
    
    // remove white-space from beginning and end
    $content = trim($html);
    
    // replace various HTML elements with plain text
    $content = preg_replace("/\r/", '', $content);
    $content = preg_replace("/[\n\t]+/", ' ', $content);
    $content = preg_replace('/<title[^>]*>.*?<\/title>/i', '', $content);
    $content = preg_replace('/<script[^>]*>.*?<\/script>/i', '', $content);
    $content = preg_replace('/<style[^>]*>.*?<\/style>/i', '', $content);

    if (!function_exists('convert_html_to_text_heading_big_callback')) {
        function convert_html_to_text_heading_big_callback($matches) {
            return "\n\n" . mb_strtoupper($matches[1]) . "\n\n";
        }
    }

    $content = preg_replace_callback(
        '/<h[123][^>]*>(.+?)<\/h[123]>/i',
        'convert_html_to_text_heading_big_callback',
        $content
    );

    if (!function_exists('convert_html_to_text_heading_small_callback')) {
        function convert_html_to_text_heading_small_callback($matches) {
            return "\n\n" . ucwords($matches[1]) . "\n\n";
        }
    }

    $content = preg_replace_callback(
        '/<h[456][^>]*>(.+?)<\/h[456]>/i',
        'convert_html_to_text_heading_small_callback',
        $content
    );

    $content = preg_replace('/<p[^>]*>/i', "\n\n", $content);
    $content = preg_replace('/<br[^>]*>/i', "\n", $content);

    if (!function_exists('convert_html_to_text_bold_callback')) {
        function convert_html_to_text_bold_callback($matches) {
            return mb_strtoupper($matches[1]);
        }
    }

    $content = preg_replace_callback(
        '/<b[^>]*>(.+?)<\/b>/i',
        'convert_html_to_text_bold_callback',
        $content
    );

    if (!function_exists('convert_html_to_text_strong_callback')) {
        function convert_html_to_text_strong_callback($matches) {
            return mb_strtoupper($matches[1]);
        }
    }

    $content = preg_replace_callback(
        '/<strong[^>]*>(.+?)<\/strong>/i',
        'convert_html_to_text_strong_callback',
        $content
    );

    $content = preg_replace('/<i[^>]*>(.+?)<\/i>/i', '_\\1_', $content);
    $content = preg_replace('/<em[^>]*>(.+?)<\/em>/i', '_\\1_', $content);
    $content = preg_replace('/(<ul[^>]*>|<\/ul>)/i', "\n\n", $content);
    $content = preg_replace('/(<ol[^>]*>|<\/ol>)/i', "\n\n", $content);
    $content = preg_replace('/<link[^>]*>/i', '', $content);
    $content = preg_replace('/<li[^>]*>/i', "\n*", $content);
    $content = preg_replace('/<a href="([^"]+)"[^>]*>(.+?)<\/a>/i', '\\2 (\\1)', $content);
    $content = preg_replace('/<hr[^>]*>/i', "\n-------------------------\n", $content);
    $content = preg_replace('/(<table[^>]*>|<\/table>)/i', "\n\n", $content);
    $content = preg_replace('/(<tr[^>]*>|<\/tr>)/i', "\n", $content);
    $content = preg_replace('/<td[^>]*>(.+?)<\/td>/i', "\t\t\\1\n", $content);

    if (!function_exists('convert_html_to_text_table_heading_callback')) {
        function convert_html_to_text_table_heading_callback($matches) {
            return "\t\t" . mb_strtoupper($matches[1]) . "\n";
        }
    }

    $content = preg_replace_callback(
        '/<th[^>]*>(.+?)<\/th>/i',
        'convert_html_to_text_table_heading_callback',
        $content
    );

    $content = preg_replace('/&nbsp;/i', ' ', $content);
    $content = preg_replace('/&quot;/i', '"', $content);
    $content = preg_replace('/&gt;/i', '>', $content);
    $content = preg_replace('/&lt;/i', '<', $content);
    $content = preg_replace('/&amp;/i', '&', $content);
    $content = preg_replace('/&copy;/i', '(c)', $content);
    $content = preg_replace('/&trade;/i', '(tm)', $content);
    $content = preg_replace('/&#8220;/', '"', $content);
    $content = preg_replace('/&#8221;/', '"', $content);
    $content = preg_replace('/&#8211;/', '-', $content);
    $content = preg_replace('/&#8217;/', "'", $content);
    $content = preg_replace('/&#38;/', '&', $content);
    $content = preg_replace('/&#169;/', '(c)', $content);
    $content = preg_replace('/&#8482;/', '(tm)', $content);
    $content = preg_replace('/&#151;/', '--', $content);
    $content = preg_replace('/&#147;/', '"', $content);
    $content = preg_replace('/&#148;/', '"', $content);
    $content = preg_replace('/&#149;/', '*', $content);
    $content = preg_replace('/&reg;/i', '(R)', $content);
    $content = preg_replace('/&bull;/i', '*', $content);
    $content = preg_replace('/&[&;]+;/i', '', $content);
    
    // remove any remaining tags
    $content = strip_tags($content);
    
    // do not allow more than 2 empty lines in a row
    $content = preg_replace("/\n\s+\n/", "\n", $content);
    $content = preg_replace("/[\n]{3,}/", "\n\n", $content);
    
    // replace multiple spaces with a single space
    $content = preg_replace("/ +/", ' ', $content);

    // wrap content
    $content = wordwrap($content);
    
    return $content;
}

function convert_text_to_html($content)
{
    // get all URL's so that we can later replace them with links
    preg_match_all('/(((http|https|ftp):\/\/)(\S*?\.\S*?))(\s|\;|\)|\]|\[|\{|\}|,|"|\'|:|\<|$|\.\s)/i', $content, $matches);
    $urls = $matches[1];
    
    // if there is at least one URL, then convert all URL's to placeholders
    if (count($urls) > 0) {
        $content = preg_replace('/(((http|https|ftp):\/\/)(\S*?\.\S*?))(\s|\;|\)|\]|\[|\{|\}|,|"|\'|:|\<|$|\.\s)/i', '^^^placeholder^^^' . "$5", $content);
    }
    
    // prepare the content for HTML
    $content = h($content);
    
    // loop through the URL's in order to convert them to links
    foreach ($urls as $url) {
        $output_url = h($url);
        
        // convert URL to link
        $content = preg_replace('/' . preg_quote('^^^placeholder^^^') . '/', '<a href="' . $output_url . '" target="_blank" rel="nofollow">' . $output_url . '</a>', $content, 1);
    }
    
    // convert all new lines to line breaks
    $content = nl2br($content);
    
    return $content;
}

function check_calendar_event_location_availability($location_id, $event_start, $event_end, $event_id = "0")
{
    // Convert the timestamps to seconds
    $event_start = strtotime($event_start);
    $event_end = strtotime($event_end);

    // If they are editing an existing event, this should be > 0
    if ($event_id != "0") {
        $event_id_where_clause = ' AND (id != "' . escape($event_id) . '")';
    } else {
        $event_id_where_clause = '';
    }
    
    // get calendar event exceptions
    $query =
        "SELECT
            calendar_event_id,
            recurrence_number
        FROM calendar_event_exceptions";
    $result = mysqli_query(db::$con, $query);

    $calendar_event_exceptions = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $calendar_event_exceptions[] = $row;
    }

    // get calendar events
    $query =
        "SELECT
            id,
            calendar_events_calendar_event_locations_xref.calendar_event_location_id as calendar_event_location_id, 
            start_time as normal_start_time, 
            end_time as normal_end_time,
            UNIX_TIMESTAMP(start_time) as start_time, 
            UNIX_TIMESTAMP(end_time) as end_time,
            recurrence,
            recurrence_number,
            recurrence_type,
            recurrence_day_sun,
            recurrence_day_mon,
            recurrence_day_tue,
            recurrence_day_wed,
            recurrence_day_thu,
            recurrence_day_fri,
            recurrence_day_sat,
            recurrence_month_type
        FROM calendar_events
        LEFT JOIN calendar_events_calendar_event_locations_xref ON calendar_events.id = calendar_events_calendar_event_locations_xref.calendar_event_id
        WHERE 
            (calendar_events_calendar_event_locations_xref.calendar_event_location_id = '" . escape($location_id) . "')"
            . $event_id_where_clause;
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // Check to see if there is an event using the same room at the same time
    while ($row = mysqli_fetch_assoc($result)) {
        // If this is not a recurring event, then do a simple check
        if ($row['recurrence'] == 0) {
            $event_start_date_and_time = $row['start_time'];
            $event_end_date_and_time = $row['end_time'];
            if (($event_start_date_and_time <= $event_start) && ($event_end_date_and_time > $event_start)) { return $row['id']; }
            if (($event_start_date_and_time > $event_start) && ($event_start_date_and_time < $event_end)) { return $row['id']; }
            if (($event_end_date_and_time > $event_start) && ($event_end_date_and_time <= $event_end)) { return $row['id']; }
        } else {
            // This is a recurring event, check each day to make sure its not conflicting
            $event_start_date_and_time = $row['normal_start_time'];
            $event_end_date_and_time = $row['normal_end_time'];
            
            $recurrence_number = $row['recurrence_number'];
            $recurrence_type = $row['recurrence_type'];
            $recurrence_day_sun = $row['recurrence_day_sun'];
            $recurrence_day_mon = $row['recurrence_day_mon'];
            $recurrence_day_tue = $row['recurrence_day_tue'];
            $recurrence_day_wed = $row['recurrence_day_wed'];
            $recurrence_day_thu = $row['recurrence_day_thu'];
            $recurrence_day_fri = $row['recurrence_day_fri'];
            $recurrence_day_sat = $row['recurrence_day_sat'];
            $recurrence_month_type = $row['recurrence_month_type'];
            
            // split event start date and time into parts
            $event_start_date_and_time_parts = explode(' ', $event_start_date_and_time);
            $event_start_date = $event_start_date_and_time_parts[0];
            $event_start_time = $event_start_date_and_time_parts[1];
            
            // split event end date and time into parts
            $event_end_date_and_time_parts = explode(' ', $event_end_date_and_time);
            $event_end_time = $event_end_date_and_time_parts[1];
            
            $calendar_exceptions = array();
            
            // Split event start date into parts, that we will use later.
            $event_start_date_parts = explode('-', $event_start_date);
            $event_start_year = $event_start_date_parts[0];
            $event_start_month = $event_start_date_parts[1];
            $event_start_day = $event_start_date_parts[2];

            foreach ($calendar_event_exceptions as $calendar_event_exception) {
                if ($calendar_event_exception[calendar_event_id] == $row['id']) {
                    $calendar_exceptions[] = $calendar_event_exception[recurrence_number];
                }
            }

            // If this is a monthly event and the month type is "day of the week",
            // then determine which week in the month the event is on.
            // If the week is 1-4 then we will use that, however if the week is 5,
            // then we interpret that as the last week.
            if (
                ($recurrence_type == 'month')
                && ($recurrence_month_type == 'day_of_the_week')
            ) {
                $day_of_the_week = date('l', strtotime($event_start_date));
                $first_day_of_the_month_timestamp = strtotime($event_start_year . '-' . $event_start_month . '-01');

                $week = '';

                // Create a loop in order to determine which week event falls on.
                // We only loop through 4 weeks, because we are going to set "last" below for 5th week.
                for ($week_index = 0; $week_index <= 3; $week_index++) {
                    // If the event is in this week, then remember the week number and break out of this loop.
                    if ($event_start_date == date('Y-m-d', strtotime('+' . $week_index . ' week ' . $day_of_the_week, $first_day_of_the_month_timestamp))) {
                        $week = $week_index + 1;
                        break;
                    }
                }

                // If a week was not found, then that means it falls on the 5th week,
                // so set it to be the last week.
                if ($week == '') {
                    $week = 'last';
                }
            }
            
            // loop in order to create a new event for each recurrence
            for ($i = 0; $i <= $recurrence_number; $i++) {
                // We will use this variable to stop the calculations if the recurrence number is hidden by an exception
                $halt_on_exception = '0';
                if (count($calendar_exceptions) > 0) {
                    if (in_array($i, $calendar_exceptions)) {
                        $halt_on_exception = '1';
                    } else {
                        $halt_on_exception = '0';
                    }
                }
                
                // if recurrence number is greater than 0, then adjust event start date
                if ($i > 0) {
                    // adjust event start date depending on recurrence type
                    switch ($recurrence_type) {
                        // Daily
                        case 'day':
                            $count = 0;

                            // Loop through days in the future until we find a date that is valid
                            // based on the valid days of the week that were selected.
                            while (true) {
                                $new_time = strtotime('+1 day', strtotime($event_start_date));
                                $event_start_date = date('Y-m-d', $new_time);
                                $day_of_the_week = strtolower(date('D', $new_time));

                                // If this day of the week is valid for this calendar event,
                                // then we have found a valid date, so break out of the loop.
                                if (${'recurrence_day_' . $day_of_the_week} == 1) {
                                    break;
                                }

                                $count++;

                                // If we have already looped 7 times, then something is wrong,
                                // so break out of this loop and the recurrence loop above.
                                // This should never happen but is added just in case in order to
                                // prevent an endless loop.
                                if ($count == 7) {
                                    break 3;
                                }
                            }

                            break;
                            
                        // Weekly
                        case 'week':
                            $new_time = mktime(0, 0, 0, $event_start_month, $event_start_day + (7 * $i), $event_start_year);
                            $event_start_date = date('Y', $new_time) . '-' . date('m', $new_time) . '-' . date('d', $new_time);
                            break;

                        // Monthly
                        case 'month':
                            switch ($recurrence_month_type) {
                                case 'day_of_the_month':
                                    $new_time = mktime(0, 0, 0, $event_start_month + $i, 1, $event_start_year);
                                    $new_event_start_year = date('Y', $new_time);
                                    $new_event_start_month = date('m', $new_time);
                                    $new_event_start_day = $event_start_day;

                                    // if date is not valid, then get last date for month
                                    if (checkdate($new_event_start_month, $new_event_start_day, $new_event_start_year) == false) {
                                        $new_event_start_day = date('t', mktime(0, 0, 0, $new_event_start_month, 1, $new_event_start_year));
                                    }

                                    $event_start_date = $new_event_start_year . '-' . $new_event_start_month . '-' . $new_event_start_day;

                                    break;

                                case 'day_of_the_week':
                                    $first_day_of_the_month_timestamp = mktime(0, 0, 0, $event_start_month + $i, 1, $event_start_year);

                                    // If the week is 1-4 then find the date in a certain way.
                                    if ($week != 'last') {
                                        $week_index = $week - 1;

                                        $new_time = strtotime('+' . $week_index . ' week ' . $day_of_the_week, $first_day_of_the_month_timestamp);

                                    // Otherwise the week is last, so find the date in a different way.
                                    } else {
                                        $last_day_of_the_month_timestamp = strtotime(date('Y-m-t', $first_day_of_the_month_timestamp));

                                        // If the last day of the month happens to be the right day of the week,
                                        // then thats that day that we want.
                                        if (date('l', $last_day_of_the_month_timestamp) == $day_of_the_week) {
                                            $new_time = $last_day_of_the_month_timestamp;

                                        // Otherwise find the day of the week that we want in the last week of the month.
                                        } else {
                                            $new_time = strtotime('last ' . $day_of_the_week, $last_day_of_the_month_timestamp);
                                        }
                                    }

                                    $event_start_date = date('Y-m-d', $new_time);

                                    break;
                            }

                            break;
                        
                        // Yearly
                        case 'year':
                            $new_event_start_year = $event_start_year + $i;
                            $new_event_start_month = $event_start_month;
                            $new_event_start_day = $event_start_day;
                            
                            // if date is not valid, then get last date for month
                            if (checkdate($new_event_start_month, $new_event_start_day, $new_event_start_year) == false) {
                                $new_event_start_day = date('t', mktime(0, 0, 0, $new_event_start_month, 1, $new_event_start_year));
                            }
                            
                            $event_start_date = $new_event_start_year . '-' . $new_event_start_month . '-' . $new_event_start_day;
                            break;
                    }
                }
                
                $event_start_date_and_time = strtotime($event_start_date . ' ' . $event_start_time);
                $event_end_date_and_time = strtotime($event_start_date . ' ' . $event_end_time);
                
                // if the event start date is greater than or equal to stop date, then break out of loop,
                // because it is not possible for further recurrences to be in date range
                if ($event_start_date_and_time >= $event_end) {
                    break;
                }

                if ($halt_on_exception == '0') {
                    if (($event_start_date_and_time <= $event_start) && ($event_end_date_and_time > $event_start)) { return $row['id']; }
                    if (($event_start_date_and_time > $event_start) && ($event_start_date_and_time < $event_end)) { return $row['id']; }
                    if (($event_end_date_and_time > $event_start) && ($event_end_date_and_time <= $event_end)) { return $row['id']; }
                }
            }
        }
    }
    // If there was not an overlapped booking, then the room is available.
    return 'available';
}

function get_update_currency_form()
{
    // if multi-currency is disabled in the settings, then do not return form and return empty string
    if (ECOMMERCE_MULTICURRENCY == false) {
        return '';
    }

    // Get all currencies where the exchange rate is not 0, with base currency first.
    $currencies = db_items(
        "SELECT
            id,
            name,
            code,
            exchange_rate
        FROM currencies
        WHERE exchange_rate != '0'
        ORDER BY
            base DESC,
            name ASC");

    // If there is less than 2 currencies, then return empty string.
    if (count($currencies) < 2) {
        return '';
    }
    
    $output_currency_options = '';
    
    // loop through all currencies
    foreach ($currencies as $currency) {
        $selected = '';
        
        // if this currency matches which currency the user chose, display it as selected in the picklist.
        if ($currency['id'] == VISITOR_CURRENCY_ID) {
            $selected = ' selected="selected"';
        }
        
        // prepare option for currency
        $output_currency_options .= '<option value="' . $currency['id'] . '"' . $selected . '>' . h($currency['name']) . ' (' . h($currency['code']) . ')</option>';
    }
    
    return
        '<form action="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/update_currency.php" method="post" class="currency" style="text-align: right; margin-top: 15px; margin-bottom: 15px">
            ' . get_token_field() . '
            <input type="hidden" name="send_to" value="' . h(get_request_uri()) . '">
            <select name="currency_id" class="software_select">' . $output_currency_options . '</select> <input type="submit" name="submit" value="Update" class="software_input_submit_small_secondary">
        </form>';
}

function get_currency_amount($amount, $exchange_rate)
{
    return $amount * $exchange_rate;
}

function get_currency_options($currency_code = '')
{
    // Get currency names and codes, with the base currency first.
    $query =
        "SELECT
            name,
            code
        FROM currencies
        ORDER BY
            base DESC,
            name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // loop through each currency and create an <option> tag
    while ($row = mysqli_fetch_assoc($result)) {
        if ($row['code'] == $currency_code) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }
        
        $output .= '<option value="' . h($row['code']) . '"' . $selected . '>' . h($row['name']) . ' (' . h($row['code']) . ')</option>';
    }
    
    return $output;
}

function get_spell_checker_engine_info()
{
    $spell_checker_engine_info = array();
    
    // if pspell extension is loaded in PHP, then return its information
    if (function_exists("pspell_new")) {
        $spell_checker_engine_info['name'] = 'Pspell Extension';
        return $spell_checker_engine_info;
    }
    
    // if the server is not running Windows, then try to find aspell
    // we removed Windows support for aspell in v7 because it did not work with the new TinyMCE version (we do not know why)
    if (mb_strtoupper(mb_substr(PHP_OS, 0, 3)) != 'WIN') {
        // try to find aspell at /usr/bin/aspell
        $path = '/usr/bin/aspell';
        
        $data = shell_exec($path . ' --version');
        
        // if aspell was found at /usr/bin/aspell, then return its information
        if ($data != '') {
            $spell_checker_engine_info['name'] = 'Aspell';
            $spell_checker_engine_info['path'] = $path;
            return $spell_checker_engine_info;
        }
        
        // try to find aspell at /usr/local/bin/aspell
    	$path = '/usr/local/bin/aspell';
        
        $data = shell_exec($path . ' --version');
        
        // if aspell was found at /usr/local/bin/aspell, then return its information
        if ($data != '') {
            $spell_checker_engine_info['name'] = 'Aspell';
            $spell_checker_engine_info['path'] = $path;
            return $spell_checker_engine_info;
        }
    }
    
    // if we got here, then no spell checker engine could be found on the server, so return Google's information
    $spell_checker_engine_info['name'] = 'Google';
    return $spell_checker_engine_info;
}

// store list of recipients from address book in session
function initialize_recipients() {

    // If recipients have not already been initialized, and user is logged in and not ghosting,
    // store recipients from address book in session.
    if (
        !$_SESSION['ecommerce']['initialized_recipients']
        and USER_LOGGED_IN
        and !$_SESSION['software']['ghost']
    ) {
        // get user id
        $query = "SELECT user_id FROM user WHERE user_username = '" . escape($_SESSION['sessionusername']) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_assoc($result);
        $user_id = $row['user_id'];

        // get all recipients in address book
        $query = "SELECT ship_to_name FROM address_book WHERE user = '$user_id' ORDER BY ship_to_name";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

        // loop through all recipients in address book to make sure recipient is in ship to selection
        while ($row = mysqli_fetch_assoc($result)) {
            $recipient_in_session = false;

            // if there are recipients in the session
            if ($_SESSION['ecommerce']['recipients']) {
                // loop through all recipients
                foreach ($_SESSION['ecommerce']['recipients'] as $recipient) {
                    // if recipient from address book is already in session, remember that
                    if (mb_strtolower($recipient) == mb_strtolower($row['ship_to_name'])) {
                        $recipient_in_session = true;
                        break;
                    }
                }
            }

            // if recipient was not found in the session and recipient is not "myself", add recipient to session
            if (($recipient_in_session == false) && (mb_strtolower($row['ship_to_name']) != 'myself')) {
                $_SESSION['ecommerce']['recipients'][] = $row['ship_to_name'];
            }
        }
        
        // remember that recipients have been initialized
        $_SESSION['ecommerce']['initialized_recipients'] = true;
    }
}

// add a recipient to session so that ship to name appears in ship to pick lists
function add_recipient($ship_to_name)
{
    // if shipping is on and recipient mode is multi-recipient and ship to name is not myself, continue
    if ((ECOMMERCE_SHIPPING == true) && (ECOMMERCE_RECIPIENT_MODE == 'multi-recipient') && ($ship_to_name != 'myself')) {
        // initialize variable that will be used to determine if recipient is already in session
        $recipient_in_session = false;

        // if recipients have been stored in session
        if ($_SESSION['ecommerce']['recipients']) {
            // loop through all recipients
            foreach ($_SESSION['ecommerce']['recipients'] as $recipient) {
                // if recipient from address book is already in session, remember that
                if (mb_strtolower($recipient) == mb_strtolower($ship_to_name)) {
                    $recipient_in_session = true;
                    break;
                }
            }
        }

        // if recipient is not already in the session, add recipient to session
        if ($recipient_in_session == false) {
            // if recipients array in session is not set, create it
            if (!isset($_SESSION['ecommerce']['recipients'])) {
                $_SESSION['ecommerce']['recipients'] = array();
            }
            
            $_SESSION['ecommerce']['recipients'][] = $ship_to_name;
        }
    }
}

// get three digit country code from 2 character country code
function get_country_number($country_code)
{
    switch ($country_code) {
        case 'US': return '840';
        case 'AF': return '004';
        case 'AX': return '248';
        case 'AL': return '008';
        case 'DZ': return '012';
        case 'AS': return '016';
        case 'AD': return '020';
        case 'AO': return '024';
        case 'AI': return '660';
        case 'AQ': return '010';
        case 'AG': return '028';
        case 'AR': return '032';
        case 'AM': return '051';
        case 'AW': return '533';
        case 'AU': return '036';
        case 'AT': return '040';
        case 'AZ': return '031';
        case 'BS': return '044';
        case 'BH': return '048';
        case 'BD': return '050';
        case 'BB': return '052';
        case 'BY': return '112';
        case 'BE': return '056';
        case 'BZ': return '084';
        case 'BJ': return '204';
        case 'BM': return '060';
        case 'BT': return '064';
        case 'BO': return '068';
        case 'BA': return '070';
        case 'BW': return '072';
        case 'BV': return '074';
        case 'BR': return '076';
        case 'IO': return '086';
        case 'BN': return '096';
        case 'BG': return '100';
        case 'BF': return '854';
        case 'BI': return '108';
        case 'KH': return '116';
        case 'CM': return '120';
        case 'CA': return '124';
        case 'CV': return '132';
        case 'KY': return '136';
        case 'CF': return '140';
        case 'TD': return '148';
        case 'XX': return '830';
        case 'CL': return '152';
        case 'CN': return '156';
        case 'CX': return '162';
        case 'CC': return '166';
        case 'CO': return '170';
        case 'KM': return '174';
        case 'CD': return '180';
        case 'CG': return '178';
        case 'CK': return '184';
        case 'CR': return '188';
        case 'CI': return '384';
        case 'HR': return '191';
        case 'CU': return '192';
        case 'CY': return '196';
        case 'CZ': return '203';
        case 'DK': return '208';
        case 'DJ': return '262';
        case 'DM': return '212';
        case 'DO': return '214';
        case 'EC': return '218';
        case 'EG': return '818';
        case 'SV': return '222';
        case 'GQ': return '226';
        case 'ER': return '232';
        case 'EE': return '233';
        case 'ET': return '231';
        case 'FK': return '238';
        case 'FO': return '234';
        case 'FJ': return '242';
        case 'FI': return '246';
        case 'FR': return '250';
        case 'FX': return '249';
        case 'GF': return '254';
        case 'PF': return '258';
        case 'TF': return '260';
        case 'GA': return '266';
        case 'GM': return '270';
        case 'GE': return '268';
        case 'DE': return '276';
        case 'GH': return '288';
        case 'GI': return '292';
        case 'GR': return '300';
        case 'GL': return '304';
        case 'GD': return '308';
        case 'GP': return '312';
        case 'GU': return '316';
        case 'GT': return '320';
        case 'GN': return '324';
        case 'GW': return '624';
        case 'GY': return '328';
        case 'HT': return '332';
        case 'HM': return '334';
        case 'VA': return '336';
        case 'HN': return '340';
        case 'HK': return '344';
        case 'HU': return '348';
        case 'IS': return '352';
        case 'IN': return '356';
        case 'ID': return '360';
        case 'IR': return '364';
        case 'IQ': return '368';
        case 'IE': return '372';
        case 'IM': return '833';
        case 'IL': return '376';
        case 'IT': return '380';
        case 'JM': return '388';
        case 'JP': return '392';
        case 'JO': return '400';
        case 'KZ': return '398';
        case 'KE': return '404';
        case 'KI': return '296';
        case 'KP': return '408';
        case 'KR': return '410';
        case 'KW': return '414';
        case 'KG': return '417';
        case 'LA': return '418';
        case 'LV': return '428';
        case 'LB': return '422';
        case 'LS': return '426';
        case 'LR': return '430';
        case 'LY': return '434';
        case 'LI': return '438';
        case 'LT': return '440';
        case 'LU': return '442';
        case 'MO': return '446';
        case 'MK': return '807';
        case 'MG': return '450';
        case 'MW': return '454';
        case 'MY': return '458';
        case 'MV': return '462';
        case 'ML': return '466';
        case 'MT': return '470';
        case 'MH': return '584';
        case 'MQ': return '474';
        case 'MR': return '478';
        case 'MU': return '480';
        case 'YT': return '175';
        case 'MX': return '484';
        case 'FM': return '583';
        case 'MD': return '498';
        case 'MC': return '492';
        case 'MN': return '496';
        case 'MS': return '500';
        case 'MA': return '504';
        case 'MZ': return '508';
        case 'MM': return '104';
        case 'NA': return '516';
        case 'NR': return '520';
        case 'NP': return '524';
        case 'NL': return '528';
        case 'AN': return '530';
        case 'NC': return '540';
        case 'NZ': return '554';
        case 'NI': return '558';
        case 'NE': return '562';
        case 'NG': return '566';
        case 'NU': return '570';
        case 'NF': return '574';
        case 'MP': return '580';
        case 'NO': return '578';
        case 'OM': return '512';
        case 'PK': return '586';
        case 'PW': return '585';
        case 'PS': return '275';
        case 'PA': return '591';
        case 'PG': return '598';
        case 'PY': return '600';
        case 'PE': return '604';
        case 'PH': return '608';
        case 'PN': return '612';
        case 'PL': return '616';
        case 'PT': return '620';
        case 'PR': return '630';
        case 'QA': return '634';
        case 'RE': return '638';
        case 'RO': return '642';
        case 'RU': return '643';
        case 'RW': return '646';
        case 'SH': return '654';
        case 'KN': return '659';
        case 'LC': return '662';
        case 'PM': return '666';
        case 'VC': return '670';
        case 'WS': return '882';
        case 'SM': return '674';
        case 'ST': return '678';
        case 'SA': return '682';
        case 'SN': return '686';
        case 'CS': return '891';
        case 'SC': return '690';
        case 'SL': return '694';
        case 'SG': return '702';
        case 'SK': return '703';
        case 'SI': return '705';
        case 'SB': return '090';
        case 'SO': return '706';
        case 'ZA': return '710';
        case 'GS': return '239';
        case 'ES': return '724';
        case 'LK': return '144';
        case 'SD': return '736';
        case 'SR': return '740';
        case 'SJ': return '744';
        case 'SZ': return '748';
        case 'SE': return '752';
        case 'CH': return '756';
        case 'SY': return '760';
        case 'TW': return '158';
        case 'TJ': return '762';
        case 'TZ': return '834';
        case 'TH': return '764';
        case 'TL': return '626';
        case 'TG': return '768';
        case 'TK': return '772';
        case 'TO': return '776';
        case 'TT': return '780';
        case 'TN': return '788';
        case 'TR': return '792';
        case 'TM': return '795';
        case 'TC': return '796';
        case 'TV': return '798';
        case 'UG': return '800';
        case 'UA': return '804';
        case 'AE': return '784';
        case 'GB': return '826';
        case 'UM': return '581';
        case 'UY': return '858';
        case 'UZ': return '860';
        case 'VU': return '548';
        case 'VE': return '862';
        case 'VN': return '704';
        case 'VG': return '092';
        case 'VI': return '850';
        case 'WF': return '876';
        case 'EH': return '732';
        case 'YE': return '887';
        case 'ZM': return '894';
        case 'ZW': return '716';
        default: return '';
    }
}

// address type is either "po box" or "street address"
function get_address_type($address)
{
    // remove spaces from beginning and end of address
    $address = trim($address);
    
    // convert address to lowercase characters
    $address = mb_strtolower($address);
    
    // get first three characters of address
    $first_three_characters = mb_substr($address, 0, 3);
    
    // get first two characters of address
    $first_two_characters = mb_substr($address, 0, 2);
    
    // if the address is a po box, then prepare type
    if (
        (mb_strpos($first_three_characters, 'po ') !== false)
        || (mb_strpos($first_three_characters, 'p.o') !== false)
        || (mb_strpos($first_three_characters, 'p,o') !== false)
        || (mb_strpos($first_three_characters, 'pob') !== false)
        || (mb_strpos($first_three_characters, 'p o') !== false)
        || (mb_strpos($first_three_characters, 'p. ') !== false)
        || (mb_strpos($first_three_characters, 'p b') !== false)
        || (mb_strpos($first_three_characters, 'rr ') !== false)
        || (mb_strpos($first_three_characters, 'hc ') !== false)
        || (mb_strpos($first_three_characters, 'rou') !== false)
        || (mb_strpos($first_two_characters, 'hc') !== false)
        || (mb_strpos($first_two_characters, 'rr') !== false)
        || (mb_strpos($first_two_characters, 'rt') !== false)
    ) {
        $address_type = 'po box';
        
    // else the address is a street address, so prepare type
    } else {
        $address_type = 'street address';
    }
    
    return $address_type;
}

// Outputs dynamic menu system using values from the database.
// We don't currently support multiple menu items being set as the current menu item
// because it causes problems for the accordion menu.  The logic for the accordion menu
// assumes there is only one current menu item.  If multiple are set,
// then menu items do not collapse properly.  We should eventually spend some time and resolve this
// in order to add support for multiple current menu items.
function get_menu_content($menu_id, $parent_id = 0, $current_menu_item_id = 0) {

    // get menu information
    $menu = db_item(
        "SELECT
            name,
            effect,
            class
         FROM menus
         WHERE id = '" . $menu_id . "'");
    
    $output = '';
    
    // if parent_id is 0 then this is the first level of the menu
    if ($parent_id == 0) {

        $class = '';

        // If there is a custom class for the menu, then add a space on the beginning for
        // separation from the software_menu class.
        if ($menu['class']) {
            $class = ' ' . $menu['class'];
        }

        $output .= '<ul id="software_menu_' . $menu['name'] . '" class="software_menu' . h($class) . '">';
        
    // else, this is the second level of the menu
    } else {
        $output .= '<ul>';
    }
    
    // get menu items
    $query = 
        "SELECT 
           menu_items.id,
           menu_items.name,
           page.page_name AS link_page_name,
           page.page_folder AS link_page_folder_id,
           menu_items.link_url,
           menu_items.link_target,
           menu_items.security
        FROM menu_items
        LEFT JOIN page ON page.page_id = menu_items.link_page_id
        WHERE 
           (menu_items.parent_id = '" . escape($parent_id) . "') 
           AND (menu_items.menu_id = '" . $menu_id . "') 
        ORDER BY menu_items.sort_order";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $menu_items = array();
    
    // loop through all menu items in order to build array
    while ($row = mysqli_fetch_assoc($result)) {
        $menu_items[] = $row;
    }
    
    $first_menu_item = TRUE;
    
    // loop through all menu items in order to get content
    foreach ($menu_items as $menu_item) {
        // If security is disabled,
        // or if this menu item is not connected to a page
        // or this visitor has access to view this menu item's page,
        // then show the menu item and its child items, if any exist.
        if (
            ($menu_item['security'] == 0)
            || ($menu_item['link_page_name'] == '')
            || (check_view_access($menu_item['link_page_folder_id']) == true)
        ) {
            // assume that sub menus do not exist, until we find out otherwise
            $sub_menu_exists = false;
            
            // find out if sub menu exists
            $query = "SELECT COUNT(*) FROM menu_items WHERE parent_id = '" . $menu_item['id'] . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            $row = mysqli_fetch_row($result);
            
            // if sub menu exists, remember that
            if ($row[0] > 0) {
                $sub_menu_exists = true;
            }
            
            $output_link_href = '';
            $output_link_target = '';
            
            // if this is not an accordion menu or a sub-menu does not exist and there is a link, then prepare link
            if ((($menu['effect'] != 'Accordion') || ($sub_menu_exists == false)) && (($menu_item['link_page_name'] != '') || ($menu_item['link_url'] != ''))) {
                // if there is a link page name, then prepare link to page
                if ($menu_item['link_page_name']) {
                    $output_link_href = OUTPUT_PATH . h($menu_item['link_page_name']);
                    
                // else there is not a link page name, so prepare link to URL
                } else {
                    $output_link_href = h($menu_item['link_url']);
                }
                
                // if link should open in a new window, then prepare target
                if ($menu_item['link_target'] == 'New Window') {
                    $output_link_target = ' target="_blank"';
                }
                
            // else there should not be a link, so disable link
            } else {
                $output_link_href = 'javascript:void(0)';
            }
            
            // Create variable that we will use to build all of the classes that should exist for this menu item.
            $output_menu_item_class = '';

            // If this is a top-level menu item, then add class for that.
            if ($parent_id == 0) {
                $output_menu_item_class .= 'top_level';
            }

            // If this menu item is for the page that the visitor is currently on, then output current class.
            if ($menu_item['id'] == $current_menu_item_id) {
                // If there are other classes for this menu item, then add space for separation.
                if ($output_menu_item_class != '') {
                    $output_menu_item_class .= ' ';
                }

                $output_menu_item_class .= 'current';
            }

            // If this is the first menu item at this level, then output first class.
            if ($first_menu_item == TRUE) {
                // If there are other classes for this menu item, then add space for separation.
                if ($output_menu_item_class != '') {
                    $output_menu_item_class .= ' ';
                }
                
                $output_menu_item_class .= 'first';

                // Update variable so that we will know that the next menu item is not first.
                $first_menu_item = FALSE;
            }

            // If this menu item has a menu under it, then output parent class.
            if ($sub_menu_exists == TRUE) {
                // If there are other classes for this menu item, then add space for separation.
                if ($output_menu_item_class != '') {
                    $output_menu_item_class .= ' ';
                }
                
                $output_menu_item_class .= 'parent';
            }

            // If there is at least one class for the menu item, then prepare attribute.
            if ($output_menu_item_class != '') {
                $output_menu_item_class = ' class="' . $output_menu_item_class . '"';
            }
            
            // add opening li tag and anchor to output
            $output .=  '<li id="software_menu_item_' . $menu_item['id'] . '"' . $output_menu_item_class . '>' . '<a href="' . $output_link_href . '"' . $output_link_target . $output_menu_item_class . '>' . h($menu_item['name']) . '</a>';
            
            // if sub menu exists, then add sub menu content to output
            if ($sub_menu_exists == true) {
                $output .= get_menu_content($menu_id, $menu_item['id'], $current_menu_item_id);
            }
            
            // add closing li tag to output
            $output .= '</li>';
        }
    }
    
    // add closing ul tag
    $output .= '</ul>';
    
    return $output;
}

// get options for parent menu item liveform pick list
function get_menu_item_options($menu_id, $exception_menu_item_id = 0, $parent_id = 0, $level = 1, $selected_id = 0)
{
    // if this is the first level, then add -None- option
    if ($level == 1) {
        $options .= '<option value="0">-None-</option>';
    }
    
    // get menu items
    $query =
        "SELECT
            id,
            name
        FROM menu_items
        WHERE
            (menu_id = '" . escape($menu_id) . "')
            AND (id != '" . escape($exception_menu_item_id) . "')
            AND (parent_id = '" . escape($parent_id) . "')
        ORDER BY sort_order";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $menu_items = array();
    
    // loop through menu items in order to build array
    while ($row = mysqli_fetch_assoc($result)) {
        $menu_items[] = $row;
    }
    
    // loop through menu items in order to get options
    foreach ($menu_items as $menu_item) {
        $indent = '';
        
        // create indent
        for ($i = 2; $i <= $level; $i++) {
            $indent .= '&nbsp;&nbsp;&nbsp;&nbsp;';
        }
        
        if ($selected_id == $menu_item['id']) {
            $selected = ' selected="selected"';
        } else {
            $selected = '';
        }
        
        $options .= '<option value="' . $menu_item['id'] . '"' . $selected . '>' . $indent . h($menu_item['name']) . '</option>';
        
        // determine if there is a sub menu to display
        $query =
            "SELECT COUNT(*)
            FROM menu_items
            WHERE
                (parent_id = '" . $menu_item['id'] . "')
                AND (id != '" . escape($exception_menu_item_id) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_row($result);

        // if there is a sub menu, then get options for sub menu
        if ($row[0] > 0) {
            // get options for sub menu
            $sub_options = get_menu_item_options($menu_id, $exception_menu_item_id, $menu_item['id'], $level + 1, $selected_id);
            
            // add sub menu options to options
            $options .= $sub_options;
        }
    }
    
    return $options;
}

// get page options for liveform pick list
// $access: "edit" or "view".  This defines whether only pages should be shown where
// user has edit access or if pages that user has view access should be shown.
function get_page_options($page_id = '', $page_type = '', $access = 'edit')
{
    global $user;
    
    // If access is set to "edit", then get folders that user has edit access to.
    // We will use this further below in loop.
    if ($access == 'edit') {
        $folders_that_user_has_access_to = array();
        
        // if user is a basic user, then get folders that user has access to
        if ($user['role'] == 3) {
            $folders_that_user_has_access_to = get_folders_that_user_has_access_to($user['id']);
        }
    }
    
    $page_options = array();
    
    // Setup first option
    $page_options[''] = '';

    // if a page type was given, prepare to only return pages with that page type
    if ($page_type) {
        $where = "WHERE page.page_type = '" . escape($page_type) . "'";
    } else {
        $where = '';
    }
    
    // if there is not already a where statement, then output the starting where part
    if ($where == '') {
        $where .= 'WHERE ';
    
    // else add and so that we can add the where condition
    } else {
        $where .= ' AND ';
    }
    
    $where .= '(folder.folder_archived = "0")';
    
    // get pages
    $query =
        "SELECT
            page.page_id,
            page.page_name,
            page.page_folder
        FROM page
        LEFT JOIN folder ON page.page_folder = folder.folder_id
        $where
        ORDER BY page.page_name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    while ($row = mysqli_fetch_assoc($result)) {
        // If access is set to "edit" and the user has edit access to this page,
        // or if access is set to "view" and the user has view access to this page,
        // then include it
        if (
            (
                ($access == 'edit')
                && (check_folder_access_in_array($row['page_folder'], $folders_that_user_has_access_to) == true)
            )
            ||
            (
                ($access == 'view')
                && (check_view_access($row['page_folder']) == true)
            )
        ) {
            $page_options[h($row['page_name'])] = $row['page_id'];
        }
    }
    
    return $page_options;
}

// get form info (i.e. field content, wysiwyg fields, file upload exists)
function get_form_info($page_id, $product_id, $order_item_id, $quantity_number, $label_column_width, $office_use_only, $liveform, $interface, $editable = false, $device_type = 'desktop', $folder_id_for_default_value = 0, $reference_code_field_id = 0, $express_order_form_type = '', $prefix = '')
{
    $form_info = array();
    
    // if page id is not equal to 0, then this is a page form
    if ($page_id != 0) {
        // get page type
        $query = "SELECT page_type FROM page WHERE page_id = '" . escape($page_id) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_assoc($result);
        $page_type = $row['page_type'];
        
        $form_type = '';
        
        // get the form type by looking at the page type
        switch ($page_type) {
            case 'custom form':
                $form_type = 'custom';
                break;

            case 'express order':

                if ($express_order_form_type == 'shipping') {
                    $form_type = 'shipping';
                } else {
                    $form_type = 'billing';
                }

                break;

            case 'billing information':
                $form_type = 'billing';
                break;
                
            case 'shipping address and arrival':
                $form_type = 'shipping';
                break;
        }
    
    // else page id is 0, so this is a product form
    } else {
        $form_type = 'product';
    }
    
    $connect_to_contact = '';
    
    // if there is a connect to contact passed in the URL string then prepare and save it
    if (isset($_GET['connect_to_contact']) == TRUE) {
        $connect_to_contact = trim(mb_strtolower($_GET['connect_to_contact']));
    }

    $contact = array();
    
    // If this form is a custom form, and if connect to contact is on,
    // and if user is logged in, then get contact info, because fields might need to be prefilled.
    if (
        ($form_type == 'custom')
        && ($connect_to_contact != 'false')
        && (USER_LOGGED_IN == true)
    ) {
        $query =
            "SELECT contacts.*
            FROM user
            LEFT JOIN contacts ON user.user_contact = contacts.id
            WHERE user.user_username = '" . escape($_SESSION['sessionusername']) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $contact = mysqli_fetch_assoc($result);
    }

    $office_use_only_target_options = array();

    // If office use only fields will not appear on form, then get triggers for them,
    // so we can limit the options that appear in target pick lists.
    if ($office_use_only == false) {

        // if this is a product form, then prepare SQL
        if ($form_type == 'product') {
            $sql_where = "(product_id = '" . e($product_id) . "')";
            
        // else this is a form for a page, so prepare SQL in a different way
        } else {
            $sql_where = "(page_id = '" . e($page_id) . "')";

            if ($page_type == 'express order') {
                $sql_where .= " AND (form_type = '" . e($form_type) . "')";
            }
        }

        $office_use_only_fields = db_items(
            "SELECT
                id,
                contact_field,
                default_value,
                use_folder_name_for_default_value
            FROM form_fields
            WHERE
                $sql_where
                AND (type = 'pick list')
                AND (office_use_only = '1')");

        // Loop through the office use only fields in order to check if we need to apply triggers.
        foreach ($office_use_only_fields as $office_use_only_field) {
            // Get default value for office use only field so we can find if there is a trigger for this value.
            $default_value = '';
                
            // If field is set to use folder name for default value, then use it.
            if ($office_use_only_field['use_folder_name_for_default_value'] == 1) {
                $default_value = db_value("SELECT folder_name FROM folder WHERE folder_id = '" . escape($folder_id_for_default_value) . "'");

            // Otherwise use default value from field.
            } else {
                $default_value = $office_use_only_field['default_value'];
            }

            // Check if there is an option for this default value that has a trigger.
            $option = db_item(
                "SELECT
                    id,
                    target_form_field_id
                FROM form_field_options
                WHERE
                    (form_field_id = '" . $office_use_only_field['id'] . "')
                    AND (value = '$default_value')
                    AND (target_form_field_id != '0')");

            // If an option with a trigger for the default value was found, then add target options to array.
            if ($option != '') {
                $target_options = db_items("SELECT value FROM target_options WHERE trigger_option_id = '" . $option['id'] . "'");

                $office_use_only_target_options[$option['target_form_field_id']] = array();

                // Loop through target options in order to add them to array.
                foreach ($target_options as $target_option) {
                    $office_use_only_target_options[$option['target_form_field_id']][] = mb_strtolower($target_option['value']);
                }
            }
        }
    }
    
    $sql_custom_form_fields = '';
    $sql_where = '';
    $sql_where_office_use_only = '';
    
    // if this is a product form, then prepare SQL
    if ($form_type == 'product') {
        $sql_where = "product_id = '" . e($product_id) . "'";
        
    // else this is a form for a page, so prepare SQL in a different way
    } else {
        $sql_where = "(page_id = '" . e($page_id) . "')";

        if ($page_type == 'express order') {
            $sql_where .= " AND (form_type = '" . e($form_type) . "')";
        }
        
        // if this is a custom form, then prepare other SQL
        if ($form_type == 'custom') {
            $sql_custom_form_fields =
                ",
                contact_field,
                office_use_only";
            
            // if office use only fields should not be displayed, prepare to only select non office use only fields
            if ($office_use_only == false) {
                $sql_where_office_use_only = " AND (office_use_only = 0)";
            }
        }
    }
    
    // get all fields for this form
    $query =
        "SELECT
            id,
            name,
            label,
            type,
            required,
            information,
            default_value,
            use_folder_name_for_default_value,
            size,
            maxlength,
            wysiwyg,
            rows,
            cols,
            multiple,
            spacing_above,
            spacing_below
            $sql_custom_form_fields
        FROM form_fields
        WHERE
            ($sql_where)
            $sql_where_office_use_only
        ORDER BY sort_order";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $fields = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $fields[] = $row;
    }
    
    $label_column_width_added = false;
    $form_info['wysiwyg_fields'] = array();
    
    foreach ($fields as $field) {
        $output_label_column_width = '';
        
        // if the label column width is not blank and it has not been added and this field type is not information, then add it to this field
        if (($label_column_width != '') && ($label_column_width_added == false) && ($field['type'] != 'information')) {
            $output_label_column_width = 'width: ' . $label_column_width . '%';
            $label_column_width_added = true;
        }

        // Assume that field will be shown until we find out otherwise.
        $show = true;

        // If a query string value was passed that says to not show field then remember that.
        if ($_GET['show_' . $field['id']] == 'false') {
            $show = false;
        }

        $output_hidden_style = '';

        // If the field should not be shown, then hide the row.
        // The field will still be outputted and the default value will
        // be outputted with the submitted form.
        if ($show == false) {
            $output_hidden_style = ' style="display: none"';
        }

        // create unique and valid css class for field
        $output_row_class = 'ff_' . get_class_name($field['name']);
        
        // if form is a custom form and field is for office use only, then prepare to apply office use only style to row; append field label as class
        if (($form_type == 'custom') && ($field['office_use_only'] == 1)) {
            $row_class = ' class="' . $output_row_class .' software_office_use_only"';
            
        } else {
            $row_class = ' class="' . $output_row_class . '"';
        }
        
        if ($field['size'] == 0) {
            $field['size'] = '';
        }

        if ($field['maxlength'] == 0) {
            $field['maxlength'] = '';
        }

        if ($field['rows'] == 0) {
            $field['rows'] = '';
        }

        if ($field['cols'] == 0) {
            $field['cols'] = '';
        }
        
        if ($field['label'] && $field['required']) {
            $field['label'] .= '*';
        }
        
        $default_value = '';

        $value_from_query_string = trim($_GET['value_' . $field['id']]);

        // If a default value was passed in the query string, then use that.
        if ($value_from_query_string != '') {
            $default_value = $value_from_query_string;

        // Otherwise if edit mode is off and the form is a custom form and a contact was found for user
        // and this field is connected to a contact field, then set the default value for the field
        // to the contact field's value
        } else if (
            ($editable == false)
            && ($form_type == 'custom')
            && ($contact)
            && ($field['contact_field'] != '')
        ) {
            $default_value = $contact[$field['contact_field']];
            
        // Otherwise, if field is set to use folder name for default value, then use it.
        } else if ($field['use_folder_name_for_default_value'] == 1) {
            $default_value = db_value("SELECT folder_name FROM folder WHERE folder_id = '" . escape($folder_id_for_default_value) . "'");

        // Otherwise use default value from field.
        } else {
            $default_value = $field['default_value'];
        }
        
        // if field has options, get options
        if (($field['type'] == 'pick list') || ($field['type'] == 'radio button') || ($field['type'] == 'check box')) {
            $query =
                "SELECT
                    id,
                    label,
                    value,
                    default_selected
                FROM form_field_options
                WHERE form_field_id = '" . $field['id'] . "'
                ORDER BY sort_order";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            $options = array();
            
            while ($row = mysqli_fetch_assoc($result)) {
                // If there are not target options for this field,
                // or this option is a target option, then include this option.
                // Target options in this case are the only options that a different office use only field require to appear in pick list.
                if (
                    (isset($office_use_only_target_options[$field['id']]) == false)
                    || (in_array(mb_strtolower($row['value']), $office_use_only_target_options[$field['id']]) == true)
                ) {
                    $options[] = $row;
                }
            }
        }

        $output_spacing_row_class = 'class="spacing_row' . ($field['office_use_only'] ? ' software_office_use_only" ': '" ');
        
        // if field should have spacing above, add spacing
        if ($field['spacing_above']) {
            $form_info['content'] .=
                '<tr ' .$output_spacing_row_class . $output_hidden_style . '>
                    <td colspan="2">&nbsp;</td>
                </tr>';
        }
        
        // if field needs it, then get html field name
        switch ($field['type']) {
            case 'text box':
            case 'email address':
            case 'text area':
            case 'pick list':
            case 'radio button':
            case 'check box':
            case 'file upload':
            case 'date':
            case 'date and time':
            case 'time':
                // get html field name differently based on the form type
                switch ($form_type) {
                    case 'custom':
                        $html_field_name = $field['id'];
                        break;
                        
                    case 'billing':
                    case 'shipping':
                        $html_field_name = $prefix . 'field_' . $field['id'];
                        break;
                        
                    case 'product':
                        $html_field_name = 'order_item_' . $order_item_id . '_quantity_number_' . $quantity_number . '_form_field_' . $field['id'];
                        break;
                }
                
                break;
        }
        
        // if field is a radio button or checkbox, then get html option id prefix
        switch ($field['type']) {
            case 'radio button':
            case 'check box':
                // get html option id prefix differently based on the form type
                switch ($form_type) {
                    case 'custom':
                    case 'billing':
                    case 'shipping':
                        $html_option_id_prefix = $prefix . 'software_option_';
                        break;
                        
                    case 'product':
                        $html_option_id_prefix = 'order_item_' . $order_item_id . '_quantity_number_' . $quantity_number . '_software_option_';
                        break;
                }
                
                break;
        }

        $required = '';

        // If this field is required, and it is not a check box, or it is a check box and there is
        // just one check box option, then add required attribute.  We don't add the required
        // attribute, when there are multiple check box options, because it would require that all
        // of them be checked.
        if (
            $field['required']
            and
            (
                ($field['type'] != 'check box')
                or (count($options) == 1)
            )
        ) {
            $required = 'true';
        }
        
        // prepare to output field
        switch ($field['type']) {
            case 'text box':
            case 'email address':

                if ($field['type'] == 'email address') {
                    $type = 'email';
                } else {
                    $type = 'text';
                }

                $output_title_row = '';

                // If this field is a product submit form reference code field,
                // and the field does not have an error and the 
                // and the field has a value in it, then check if we can find a
                // submitted form for the reference code, and then get title for that form.
                // This will help the user understand which submitted form the reference code is related to.
                if (
                    ($field['id'] == $reference_code_field_id)
                    && ($liveform->check_field_error($html_field_name) == false)
                    && ($liveform->get_field_value($html_field_name) != '')
                    && ($submitted_form = db_item("SELECT id, page_id FROM forms WHERE reference_code = '" . e($liveform->get_field_value($html_field_name)) . "'"))
                ) {
                    $title_label = db_value(
                        "SELECT label
                        FROM form_fields
                        WHERE
                            page_id = '" . $submitted_form['page_id'] . "'
                            AND (rss_field = 'title')
                        ORDER BY sort_order");

                    $title = get_submitted_form_title($submitted_form['id']);

                    if ($title != '') {
                        $output_title_row =
                            '<tr' . $row_class . $output_hidden_style . '>
                                <td style="' . $output_label_column_width . '">' . $title_label . '</td>
                                <td>' . h($title) . '</td>
                            </tr>';
                    }
                }
                
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td>' . $liveform->output_field(array('type' => $type, 'name'=>$html_field_name, 'value'=>$default_value, 'size'=>$field['size'], 'maxlength'=>$field['maxlength'], 'class'=>'software_input_text', 'required' => $required)) . '</td>
                    </tr>' .
                    $output_title_row;

                break;
                
            case 'text area':
                
                // if field is wysiwyg, then prepare special values and output textarea so it takes up both columns
                if ($field['wysiwyg'] == 1) {
                    // add field to wysiwyg fields array, so that we can prepare JavaScript later
                    $form_info['wysiwyg_fields'][] = $html_field_name;
                    
                    // if rows was not set, then set default rows so that WYSIWYG editor appears correctly
                    if (!$field['rows']) {
                        $field['rows'] = 15;
                    }
                    
                    $style = '';
                    
                    // if cols was not set, then set default width so that WYSIWYG editor appears correctly
                    if (!$field['cols']) {
                        $style = 'width: 95%';
                    }
                    
                    $form_info['content'] .=
                        '<tr' . $row_class . $output_hidden_style . '>
                            <td colspan="2">
                                <div style="margin-bottom: .5em">' . $field['label'] . '</div>
                                <div>' . $liveform->output_field(array('type'=>'textarea', 'name'=>$html_field_name, 'id'=>$html_field_name, 'value'=>$default_value, 'maxlength'=>$field['maxlength'], 'rows'=>$field['rows'], 'cols'=>$field['cols'], 'class'=>'software_textarea', 'style'=>$style, 'required' => $required)) . '</div>
                            </td>
                        </tr>';
                    
                // else the field is not wysiwyg, so output two columns like normal
                } else {
                    $form_info['content'] .=
                        '<tr' . $row_class . $output_hidden_style . '>
                            <td style="vertical-align: top; ' . $output_label_column_width . '">' . $field['label'] . '</td>
                            <td style="vertical-align: top">' . $liveform->output_field(array('type'=>'textarea', 'name'=>$html_field_name, 'id'=>$html_field_name, 'value'=>$default_value, 'maxlength'=>$field['maxlength'], 'rows'=>$field['rows'], 'cols'=>$field['cols'], 'class'=>'software_textarea', 'required' => $required)) . '</td>
                        </tr>';
                }
                
                break;
                
            case 'pick list':
                $multiple = '';
                
                // if the pick list supports multiple selection, then alter html field name and setup pick list to allow multiple selection
                if ($field['multiple'] == 1) {
                    $html_field_name .= '[]';
                    $multiple = 'multiple';
                }
                
                $pick_list_options = array();
                
                foreach ($options as $option) {
                    $pick_list_options[$option['label']] =
                        array(
                            'value' => $option['value'],
                            'default_selected' => $option['default_selected']
                        );
                }
                
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="vertical-align: top; ' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td style="vertical-align: top">' . $liveform->output_field(array('type'=>'select', 'name'=>$html_field_name, 'value'=>$default_value, 'options'=>$pick_list_options, 'size'=>$field['size'], 'multiple'=>$multiple, 'class'=>'software_select', 'required' => $required)) . '</td>
                    </tr>';
                
                break;
                
            case 'radio button':
                $output_options = '';
                
                foreach ($options as $option) {
                    // if this radio button should be selected by default, prepare to select by default
                    if ($option['value'] == $default_value) {
                        $checked = 'checked';
                    } else {
                        $checked = '';
                    }
                    
                    $output_options .= $liveform->output_field(array('type'=>'radio', 'name'=>$html_field_name, 'id'=>$html_option_id_prefix . $option['id'], 'value'=>$option['value'], 'checked'=>$checked, 'class'=>'software_input_radio', 'required' => $required)) . '<label for="' . $html_option_id_prefix . $option['id'] . '"> ' . h($option['label']) . '</label><br />';
                }
                
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="vertical-align: top; ' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td style="vertical-align: top">' . $output_options . '</td>
                    </tr>';
                
                break;
                
            case 'check box':
                $output_options = '';
                
                // if there is more than one option for this check box group, then prepare check boxes to support multiple check boxes
                if (count($options) > 1) {
                    $html_field_name .= '[]';
                }
                
                foreach ($options as $option) {
                    // if this checkbox should be selected by default, prepare to select by default
                    if (($option['default_selected'] == 1) || ($option['value'] == $default_value)) {
                        $checked = 'checked';
                    } else {
                        $checked = '';
                    }
                    
                    $output_options .= $liveform->output_field(array('type'=>'checkbox', 'name'=>$html_field_name, 'id'=>$html_option_id_prefix . $option['id'], 'value'=>$option['value'], 'checked'=>$checked, 'class'=>'software_input_checkbox', 'required' => $required)) . '<label for="' . $html_option_id_prefix . $option['id'] . '"> ' . h($option['label']) . '</label><br />';
                }
                
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="vertical-align: top; ' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td style="vertical-align: top">' . $output_options . '</td>
                    </tr>';
                
                break;
                
            case 'file upload':
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td style="vertical-align: top">' . $liveform->output_field(array('type'=>'file', 'name'=>$html_field_name, 'size'=>$field['size'], 'class'=>'software_input_file', 'required' => $required)) . '</td>
                    </tr>';
            
                $form_info['file_upload_exists'] = true;
                
                break;
                
            case 'date':
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td style="vertical-align: top">
                            ' . $liveform->output_field(array('type'=>'text', 'id' => $html_field_name, 'name'=>$html_field_name, 'value'=>$default_value, 'size'=>$field['size'], 'maxlength'=>'10', 'class'=>'software_input_text', 'required' => $required)) . '
                            ' . get_date_picker_format() . '
                            <script>
                                software_$("#' . $html_field_name . '").datepicker({
                                    dateFormat: date_picker_format
                                });
                            </script>
                        </td>
                    </tr>';
                break;
                
            case 'date and time':
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td style="vertical-align: top">
                            ' . $liveform->output_field(array('type'=>'text', 'id' => $html_field_name, 'name'=>$html_field_name, 'value'=>$default_value, 'size'=>$field['size'], 'maxlength'=>'22', 'class'=>'software_input_text', 'required' => $required)) . '
                            <script src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/jquery/jquery-ui-timepicker-addon-1.2.1.min.js"></script>
                            ' . get_date_picker_format() . '
                            <script>
                                software_$("#' . $html_field_name . '").datetimepicker({
                                    dateFormat: date_picker_format,
                                    timeFormat: "h:mm TT"
                                });
                            </script>
                        </td>
                    </tr>';
                break;
                
            case 'information':
                // if this is the back-end (e.g. preview form feature for product from) then prepare content for output
                if ($interface == 'backend') {
                    $field['information'] = prepare_rich_text_editor_content_for_output($field['information']);
                }
                
                if ($editable == TRUE) {
                    // add the edit button to images in content
                    $field['information'] = add_edit_button_for_images('form_field', $field['id'], $field['information']);
                }
                
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td colspan="2" style="vertical-align: top; ' . $output_label_column_width . '">' . $field['information'] . '</td>
                    </tr>';
                break;
                
            case 'time':
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td style="vertical-align: top">' . $liveform->output_field(array('type'=>'text', 'name'=>$html_field_name, 'value'=>$default_value, 'size'=>$field['size'], 'maxlength'=>'11', 'class'=>'software_input_text', 'required' => $required)) . ' (Format: h:mm AM/PM)</td>
                    </tr>';
                break;
                
            default:
                $form_info['content'] .=
                    '<tr' . $row_class . $output_hidden_style . '>
                        <td style="' . $output_label_column_width . '">' . $field['label'] . '</td>
                        <td style="vertical-align: top"></td>
                    </tr>';
                break;
        }

        // if field should have spacing below, add spacing
        if ($field['spacing_below']) {
            $form_info['content'] .=
                '<tr '. $output_spacing_row_class . $output_hidden_style . '>
                    <td colspan="2">&nbsp;</td>
                </tr>';
        }
    }
    
    return $form_info;
}

// get the table rows of content for a submitted product form and use data from form fields (e.g. label)
function get_submitted_product_form_content_with_form_fields($order_item_id, $quantity_number)
{
    // get product info
    $query =
        "SELECT
            products.id,
            products.form_label_column_width
        FROM order_items
        LEFT JOIN products ON order_items.product_id = products.id
        WHERE order_items.id = '$order_item_id'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    
    $product_id = $row['id'];
    $label_column_width = $row['form_label_column_width'];
    
    // get all fields for this form
    $query =
        "SELECT
            id,
            label,
            type,
            information,
            wysiwyg,
            spacing_above,
            spacing_below
        FROM form_fields
        WHERE product_id = '$product_id'
        ORDER BY sort_order";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $form_fields = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $form_fields[] = $row;
    }
    
    $content = '';
    $label_column_width_added = false;
    
    // loop through form fields
    foreach ($form_fields as $form_field) { 
        $output_label_column_width = '';
        
        // if the label column width is not blank and it has not been added and this field type is not information, then add it to this field
        if (($label_column_width != '') && ($label_column_width_added == false) && ($form_field['type'] != 'information')) {
            $output_label_column_width = '; width: ' . $label_column_width . '%';
            $label_column_width_added = true;
        }
        
        // if field should have spacing above, add spacing
        if ($form_field['spacing_above']) {
            $content .=
                '<tr>
                    <td colspan="2">&nbsp;</td>
                </tr>';
        }
        
        // if field has an information type, then don't get data for field
        if ($form_field['type'] == 'information') {
            $content .=
                '<tr>
                    <td colspan="2" style="vertical-align: top' . $output_label_column_width . '">' . $form_field['information'] . '</td>
                </tr>';
        
        // else field does not have an information type, so get data for field
        } else {
            $data = '';
            
            $query =
                "SELECT data
                FROM form_data
                WHERE
                    (order_item_id = '$order_item_id')
                    AND (quantity_number = '$quantity_number')
                    AND (form_field_id = '" . $form_field['id'] . "')
                ORDER BY id";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            // if there is one value, then set data to value
            if (mysqli_num_rows($result) == 1) {
                $row = mysqli_fetch_assoc($result);
                $data = $row['data'];
                
            // else if there is more than one value, then set data for multiple values
            } elseif (mysqli_num_rows($result) > 1) {
                while ($row = mysqli_fetch_assoc($result)) {
                    // if there is already data, then add a comma and a space
                    if ($data != '') {
                        $data .= ', ';
                    }
                    
                    $data .= $row['data'];
                }
            }
            
            // if this form field is wysiwyg, then do not prepare for html
            if ($form_field['wysiwyg'] == 1) {
                $data = prepare_form_data_for_output($data, $form_field['type'], $prepare_for_html = false);
                
            // else this form field is not wysiwyg, so prepare for html
            } else {
                $data = prepare_form_data_for_output($data, $form_field['type'], $prepare_for_html = true);
            }
            
            $content .=
                '<tr>
                    <td style="vertical-align: top' . $output_label_column_width . '">' . $form_field['label'] . '</td>
                    <td style="vertical-align: top">' . $data . '</td>
                </tr>';
        }
        
        // if field should have spacing below, add spacing
        if ($form_field['spacing_below']) {
            $content .=
                '<tr>
                    <td colspan="2">&nbsp;</td>
                </tr>';
        }
    }
    
    return $content;
}

// get the table rows of content for a submitted product form but do not use data from form fields (e.g. label) because they might not exist
function get_submitted_product_form_content_without_form_fields($order_item_id, $quantity_number, $interface)
{
    // get form data items for order item
    $query =
        "SELECT
            form_field_id,
            data,
            name,
            type,
            count(*) as number_of_values
        FROM form_data
        WHERE
            (order_item_id = '$order_item_id')
            AND (quantity_number = '$quantity_number')
        GROUP BY form_field_id
        ORDER BY id";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $form_data_items = array();
    
    // loop through form data items in order to add them to array
    while ($row = mysqli_fetch_assoc($result)) {
        $form_data_items[] = $row;
    }
    
    $output_form_data_rows = '';
    
    // loop through form data items in order to prepare rows of data
    foreach ($form_data_items as $form_data_item) {
        $data = '';
        
        // if there is more than one value, then get all values so data can be set to all values
        if ($form_data_item['number_of_values'] > 1) {
            $query =
                "SELECT data
                FROM form_data
                WHERE
                    (order_item_id = '$order_item_id')
                    AND (quantity_number = '$quantity_number')
                    AND (form_field_id = '" . $form_data_item['form_field_id'] . "')
                ORDER BY id";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            while ($row = mysqli_fetch_assoc($result)) {
                // if data is not empty, then add comma and space
                if ($data != '') {
                    $data .= ', ';
                }
                
                $data .= $row['data'];
            }
            
        // else there is just one value, so set data
        } else {
            $data = $form_data_item['data'];
        }
        
        // if form data item type is html, then don't prepare output for html, because data is already html
        if ($form_data_item['type'] == 'html') {
            $output_data = prepare_form_data_for_output($data, $form_data_item['type'], $prepare_for_html = false);
            
        // else form data item type is not html, so prepare output for html
        } else {
            $output_data = prepare_form_data_for_output($data, $form_data_item['type'], $prepare_for_html = true);
        }
            
        $output_form_data_rows .=
            '<tr>
                <td style="vertical-align: top">' . h($form_data_item['name']) . ':</td>
                <td style="vertical-align: top">' . $output_data . '</td>
            </tr>';
    }
    
    return $output_form_data_rows;
}

// Create function that is used to show data for a submitted custom form
// for various different screens (e.g. order preview, order receipt)
// for custom billing forms that appear on express order and billing information pages
// and for custom shipping forms that appear on shipping address & arrival pages.
// This function gets all info for the form, including labels.
function get_submitted_form_content_with_form_fields($properties)
{
    $type = $properties['type'];
    $order_id = $properties['order_id'];
    $ship_to_id = $properties['ship_to_id'];

    // Prepare where part of queries differently based on the custom form type.
    switch ($type) {
        case 'custom_billing_form':
            $sql_where =
                "(order_id = '" . escape($order_id) . "')
                AND (order_id != '0')
                AND (ship_to_id = '0')
                AND (order_item_id = '0')";

            break;

        case 'custom_shipping_form':
            $sql_where = "(ship_to_id = '" . escape($ship_to_id) . "')";

            break;
    }

    // Get a field id for any submitted form data,
    // in order determine if there is a submitted form
    // and which page it came from.
    $field_id = db_value(
        "SELECT form_field_id
        FROM form_data
        WHERE $sql_where
        LIMIT 1");

    // If the visitor has not submitted form data, then return empty string.
    if ($field_id == '') {
        return '';
    }

    // Get page info for the field of data.
    $page = db_item(
        "SELECT
            page.page_id AS id,
            page.page_type AS type
        FROM form_fields
        LEFT JOIN page ON form_fields.page_id = page.page_id
        WHERE form_fields.id = '$field_id'");

    // If a page was not found for some reason or the page type does not support a custom form,
    // then return empty string.
    if (
        ($page['id'] == '')
        ||
        (
            ($page['type'] != 'express order')
            && ($page['type'] != 'billing information')
            && ($page['type'] != 'shipping address and arrival')
        )
    ) {
        return '';
    }

    $page_type_table_name = str_replace(' ', '_', $page['type']) . '_pages';

    $page_type = $page['type'];

    // Get more page type info for the page and verify that form is enabled for the page.

    // The properties for a custom shipping form on express order are unique, because
    // shipping and billing forms can appear on express order, so get values in unique way
    if ($page_type == 'express order' and $type == 'custom_shipping_form') {

        $page = db_item(
            "SELECT page_id AS id
            FROM $page_type_table_name
            WHERE
                (page_id = '" . $page['id'] . "')
                AND (shipping_form = '1')");

    } else {

        $page = db_item(
            "SELECT
                page_id AS id,
                form_name,
                form_label_column_width
            FROM $page_type_table_name
            WHERE
                (page_id = '" . $page['id'] . "')
                AND (form = '1')");
    }

    // If page type properties could not be found, or the page does not have a form
    // enabled anymore, then return empty string.
    if ($page['id'] == '') {
        return '';
    }

    $form_name = $page['form_name'];
    $label_column_width = $page['form_label_column_width'];

    if ($page_type == 'express order') {

        if ($type == 'custom_shipping_form') {
            $sql_field_where =
                "(page_id = '" . e($page['id']) . "') AND (form_type = 'shipping')";
        } else {
            $sql_field_where =
                "(page_id = '" . e($page['id']) . "') AND (form_type = 'billing')";
        }

    } else {
        $sql_field_where = "page_id = '" . e($page['id']) . "'";
    }

    // Get all fields for this form.
    $fields = db_items(
        "SELECT
            id,
            label,
            type,
            information,
            wysiwyg,
            spacing_above,
            spacing_below
        FROM form_fields
        WHERE $sql_field_where
        ORDER BY sort_order ASC");
    
    $output_rows = '';
    $label_column_width_added = false;
    
    // Loop through form fields.
    foreach ($fields as $field) { 
        $output_label_column_width = '';
        
        // If the label column width is not blank and it has not been added and this field type is not information,
        // then add it to this field.
        if (($label_column_width != '') && ($label_column_width_added == false) && ($field['type'] != 'information')) {
            $output_label_column_width = '; width: ' . $label_column_width . '%';
            $label_column_width_added = true;
        }
        
        // If field should have spacing above, add spacing.
        if ($field['spacing_above']) {
            $output_rows .=
                '<tr>
                    <td colspan="2">&nbsp;</td>
                </tr>';
        }
        
        // If field has an information type, then don't get data for field.
        if ($field['type'] == 'information') {
            $output_rows .=
                '<tr>
                    <td colspan="2" style="vertical-align: top' . $output_label_column_width . '">' . $field['information'] . '</td>
                </tr>';
        
        // Otherwise field does not have an information type, so get data for field.
        } else {
            $data = '';
            
            $query =
                "SELECT data
                FROM form_data
                WHERE
                    $sql_where
                    AND (form_field_id = '" . $field['id'] . "')
                ORDER BY id ASC";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            // If there is one value, then set data to value.
            if (mysqli_num_rows($result) == 1) {
                $row = mysqli_fetch_assoc($result);
                $data = $row['data'];
                
            // Otherwise there is more than one value, so set data for multiple values.
            } elseif (mysqli_num_rows($result) > 1) {
                while ($row = mysqli_fetch_assoc($result)) {
                    // if there is already data, then add a comma and a space
                    if ($data != '') {
                        $data .= ', ';
                    }
                    
                    $data .= $row['data'];
                }
            }
            
            // If this form field is wysiwyg, then do not prepare for html.
            if ($field['wysiwyg'] == 1) {
                $data = prepare_form_data_for_output($data, $field['type'], $prepare_for_html = false);
                
            // Otherwise this form field is not wysiwyg, so prepare for html.
            } else {
                $data = prepare_form_data_for_output($data, $field['type'], $prepare_for_html = true);
            }
            
            $output_rows .=
                '<tr>
                    <td style="vertical-align: top' . $output_label_column_width . '">' . $field['label'] . '</td>
                    <td style="vertical-align: top">' . $data . '</td>
                </tr>';
        }
        
        // If field should have spacing below, add spacing.
        if ($field['spacing_below']) {
            $output_rows .=
                '<tr>
                    <td colspan="2">&nbsp;</td>
                </tr>';
        }
    }

    // If there is a form name, then output fieldset and legend with table of rows.
    if ($form_name != '') {
        return
            '<div style="margin-top: .5em; margin-bottom: .5em">
                <fieldset class="software_fieldset">
                    <legend class="software_legend">' . h($form_name) . '</legend>
                    <table>
                        ' . $output_rows . '
                    </table>
                </fieldset>
            </div>';

    // Otherwise there is not a form name, so just output table without a fieldset.
    } else {
        return
            '<div style="margin-top: .5em; margin-bottom: .5em">
                <table>
                    ' . $output_rows . '
                </table>
            </div>';
    }
}

// This is used to get custom shipping/billing/product form field data that customer
// has entered so it can be displayed in custom layouts for review on
// express order, order preview, and order receipt.

function get_form_review_info($properties) {

    $type = $properties['type'];
    $order_id = $properties['order_id'];
    $ship_to_id = $properties['ship_to_id'];
    $order_item_id = $properties['order_item_id'];
    $quantity_number = $properties['quantity_number'];
    $product_id = $properties['product_id'];

    // Prepare where part of queries differently based on the custom form type.
    switch ($type) {
        case 'custom_billing_form':
            $sql_where =
                "(order_id = '" . e($order_id) . "')
                AND (order_id != '0')
                AND (ship_to_id = '0')
                AND (order_item_id = '0')";

            break;

        case 'custom_shipping_form':
            $sql_where = "(ship_to_id = '" . e($ship_to_id) . "')";

            break;

        case 'product_form':
            $sql_where =
                "(order_item_id = '" . e($order_item_id) . "')
                AND (quantity_number = '" . e($quantity_number) . "')";

            break;
    }

    // Get a field id for any submitted form data,
    // in order determine if there is a submitted form
    // and which page it came from.
    $field_id = db_value(
        "SELECT form_field_id
        FROM form_data
        WHERE $sql_where
        LIMIT 1");

    // If the visitor has not submitted form data, then return false.
    if ($field_id == '') {
        return false;
    }

    $title = '';

    // Get info about the source of the form differently based on the type.
    switch ($type) {
        case 'custom_billing_form':
        case 'custom_shipping_form':

            // Get page info for the field of data.
            $page = db_item(
                "SELECT
                    page.page_id AS id,
                    page.page_type AS type
                FROM form_fields
                LEFT JOIN page ON form_fields.page_id = page.page_id
                WHERE form_fields.id = '$field_id'");

            // If a page was not found for some reason or the page type does not support a custom form,
            // then return false.
            if (
                ($page['id'] == '')
                ||
                (
                    ($page['type'] != 'express order')
                    && ($page['type'] != 'billing information')
                    && ($page['type'] != 'shipping address and arrival')
                )
            ) {
                return false;
            }

            $page_type_table_name = str_replace(' ', '_', $page['type']) . '_pages';

            $page_type = $page['type'];

            // Get more page type info for the page and verify that form is enabled for the page.

            // The properties for a custom shipping form on express order are unique, because
            // shipping and billing forms can appear on express order, so get values in unique way
            if ($page_type == 'express order' and $type == 'custom_shipping_form') {

                $page = db_item(
                    "SELECT page_id AS id
                    FROM $page_type_table_name
                    WHERE
                        (page_id = '" . $page['id'] . "')
                        AND (shipping_form = '1')");

            } else {

                $page = db_item(
                    "SELECT
                        page_id AS id,
                        form_name
                    FROM $page_type_table_name
                    WHERE
                        (page_id = '" . $page['id'] . "')
                        AND (form = '1')");
            }

            // If page type properties could not be found, or the page does not have a form
            // enabled anymore, then return false.
            if ($page['id'] == '') {
                return false;
            }

            $title = $page['form_name'];

            if ($page_type == 'express order') {

                if ($type == 'custom_shipping_form') {
                    $sql_field_where =
                        "(page_id = '" . e($page['id']) . "') AND (form_type = 'shipping')";
                } else {
                    $sql_field_where =
                        "(page_id = '" . e($page['id']) . "') AND (form_type = 'billing')";
                }

            } else {
                $sql_field_where = "page_id = '" . e($page['id']) . "'";
            }

            break;

        case 'product_form':

            $sql_field_where = "product_id = '" . e($product_id) . "'";

            break;
    }

    // Get all fields for this form.
    $fields = db_items(
        "SELECT
            id,
            label,
            type,
            information,
            wysiwyg,
            spacing_above,
            spacing_below
        FROM form_fields
        WHERE $sql_field_where
        ORDER BY sort_order ASC");

    // Create a variable that we will use to remember if there
    // is at least one field that has data, so a designer
    // can decide if he/she wants to output form if it has no data.
    $data = false;
    
    // Loop through form fields.
    foreach ($fields as $key => $field) { 
        
        // If this is not an information field, then get data that customer entered.
        if ($field['type'] != 'information') {

            $field['data'] = '';
            
            $query =
                "SELECT data
                FROM form_data
                WHERE
                    $sql_where
                    AND (form_field_id = '" . $field['id'] . "')
                ORDER BY id ASC";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            // If there is one value, then set data to value.
            if (mysqli_num_rows($result) == 1) {
                $row = mysqli_fetch_assoc($result);
                $field['data'] = $row['data'];
                
            // Otherwise there is more than one value, so set data for multiple values.
            } elseif (mysqli_num_rows($result) > 1) {
                while ($row = mysqli_fetch_assoc($result)) {
                    // if there is already data, then add a comma and a space
                    if ($field['data'] != '') {
                        $field['data'] .= ', ';
                    }
                    
                    $field['data'] .= $row['data'];
                }
            }

            if ($field['data'] != '') {
                $data = true;
            }
            
            // If this form field is wysiwyg, then do not prepare for html.
            if ($field['wysiwyg'] == 1) {
                $field['data_info'] = prepare_form_data_for_output($field['data'], $field['type'], $prepare_for_html = false);
                
            // Otherwise this form field is not wysiwyg, so prepare for html.
            } else {
                $field['data_info'] = prepare_form_data_for_output($field['data'], $field['type'], $prepare_for_html = true);
            }

        }

        $fields[$key] = $field;
    
    }

    $info = array();

    $info['title'] = $title;
    $info['fields'] = $fields;
    $info['data'] = $data;

    return $info;

}

// Create function that is used to show data for a submitted custom billing or shipping form
// for the front-end and back-end view order screens.  The original form might have changed
// since the order was submitted, so we don't show labels in this function.
// We just output name and value for each field.
function get_submitted_form_content_without_form_fields($properties) {

    $type = $properties['type'];
    $order_id = $properties['order_id'];
    $ship_to_id = $properties['ship_to_id'];
    $style = $properties['style'];

    // Prepare where part of queries differently based on the custom form type.
    switch ($type) {
        case 'custom_billing_form':
            $sql_where =
                "(order_id = '" . escape($order_id) . "')
                AND (order_id != '0')
                AND (ship_to_id = '0')
                AND (order_item_id = '0')";

            break;

        case 'custom_shipping_form':
            $sql_where = "(ship_to_id = '" . escape($ship_to_id) . "')";

            break;
    }

    // Get form data items for order item.
    $form_data_items = db_items(
        "SELECT
            form_field_id,
            data,
            name,
            type,
            count(*) as number_of_values
        FROM form_data
        WHERE $sql_where
        GROUP BY form_field_id
        ORDER BY id");

    // If there is no form data, then return empty string.
    if (count($form_data_items) == 0) {
        return '';
    }
    
    $output_rows = '';
    
    // Loop through form data items in order to prepare rows of data.
    foreach ($form_data_items as $form_data_item) {
        $data = '';
        
        // If there is more than one value, then get all values so data can be set to all values.
        if ($form_data_item['number_of_values'] > 1) {
            $query =
                "SELECT data
                FROM form_data
                WHERE
                    $sql_where
                    AND (form_field_id = '" . $form_data_item['form_field_id'] . "')
                ORDER BY id";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            while ($row = mysqli_fetch_assoc($result)) {
                // If data is not empty, then add comma and space.
                if ($data != '') {
                    $data .= ', ';
                }
                
                $data .= $row['data'];
            }
            
        // Otherwise there is just one value, so set data.
        } else {
            $data = $form_data_item['data'];
        }
        
        // If form data item type is html, then don't prepare output for html, because data is already html.
        if ($form_data_item['type'] == 'html') {
            $output_data = prepare_form_data_for_output($data, $form_data_item['type'], $prepare_for_html = false);
            
        // Otherwise form data item type is not html, so prepare output for html.
        } else {
            $output_data = prepare_form_data_for_output($data, $form_data_item['type'], $prepare_for_html = true);
        }
            
        $output_rows .=
            '<tr>
                <td style="vertical-align: top">' . h($form_data_item['name']) . ':</td>
                <td style="vertical-align: top">' . $output_data . '</td>
            </tr>';
    }

    if ($style) {
        $output_style = $style;
    } else {
        $output_style = 'margin-top: .5em; margin-bottom: .5em';
    }

    return
        '<div style="' . $output_style . '">
            <table>
                ' . $output_rows . '
            </table>
        </div>';
}

function send_user_to_login_home()
{
    // get user information
    $user = validate_user();
    
    // get home page name
    $home_page_name = get_page_name($user['home']);
            
    // if there is a send to, then forward user to the send to
    if ((isset($_REQUEST['send_to']) == true) && ($_REQUEST['send_to'] != '')) {
        header('Location: ' . URL_SCHEME . HOSTNAME . $_REQUEST['send_to']);
        exit();
        
    // else if user has a home page, then forward user to that page
    } elseif ($home_page_name != '') {
        header('Location: ' . URL_SCHEME . HOSTNAME . PATH . encode_url_path($home_page_name));
        exit();
        
    // else if user's role is administrator, designer, or manager
    // or user has edit rights
    // or user has access to control panel
    // then forward user to control panel welcome screen
    } elseif (
        ($user['role'] < 3)
        || (no_acl_check($user['id']) == true)
        || ($user['manage_calendars'] == true)
        || ($user['manage_forms'] == true)
        || ($user['manage_visitors'] == true)
        || ($user['manage_contacts'] == true)
        || ($user['manage_emails'] == true)
        || ($user['manage_ecommerce'] == true)
        || $user['manage_ecommerce_reports']
        || (count(get_items_user_can_edit('ad_regions', $user['id'])) > 0)
    ) {
        header('Location: ' . URL_SCHEME . HOSTNAME . PATH . SOFTWARE_DIRECTORY . '/welcome.php');
        exit();
        
    // else send user to my account page
    } else {
        go(get_page_type_url('my account'));
    }
}

function get_price_range($product_group_id, $discounted_product_prices)
{
    // in order to resolve an apparent bug in Zend Guard, we had to rename $largest_price to $large_price for some reason
    
    // assume that there are not any non-donation products, until we find out otherwise
    $non_donation_products_exist = false;
    
    // assume that there are 0 products in this product group, until we find out otherwise
    // we use this to be able to show when there is a discount when there is only one product in a product group
    $number_of_products = 0;
    
    // assume that there are no discounted products until we find out otherwise
    // we use this to be able to show when there is a discount when there is only one product in a product group
    $discounted_products_exist = FALSE;
    
    // initialize variable for storing the original price for a discounted product
    // we use this to be able to show when there is a discount when there is only one product in a product group
    $original_price = 0;
    
    $product_groups = array();
    
    // get all product groups in this product group
    $query =
        "SELECT id
        FROM product_groups
        WHERE
            (parent_id = '" . escape($product_group_id) . "')
            AND (enabled = '1')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // loop through all product groups in order to add them to array
    while ($row = mysqli_fetch_assoc($result)) {
        $product_groups[] = $row;
    }
    
    // loop through all product groups in order to get price range
    foreach ($product_groups as $product_group) {
        $price_range = get_price_range($product_group['id'], $discounted_product_prices);
        
        // if non donation products exist in this product, then store that and determine if the smallest and largest prices need to be updated
        if ($price_range['non_donation_products_exist'] == true) {
            $non_donation_products_exist = true;
            
            // if smallest price is not set or the smallest price of this product group is the smallest price so far, then store smallest price
            if ((isset($smallest_price) == false) || ($price_range['smallest_price'] < $smallest_price)) {
                $smallest_price = $price_range['smallest_price'];
            }
            
            // if largest price is not set or the largest price of this product group is the largest price so far, then store largest price
            if ((isset($large_price) == false) || ($price_range['largest_price'] > $large_price)) {
                $large_price = $price_range['largest_price'];
            }
        }
        
        $number_of_products = $number_of_products + $price_range['number_of_products'];
        
        // if a discounted product exists in this product group, then remember that and store original price
        if ($price_range['discounted_products_exist'] == TRUE) {
            $discounted_products_exist = TRUE;
            $original_price = $price_range['original_price'];
        }
    }
    
    $products = array();
    
    // get all non-donation products that are in this product group
    $query =
        "SELECT
            products.id,
            products.price
        FROM products_groups_xref
        LEFT JOIN products ON products.id = products_groups_xref.product
        WHERE
            (products_groups_xref.product_group = '" . escape($product_group_id) . "')
            AND (products.enabled = '1')
            AND (products.selection_type != 'donation')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // loop through all products in order to add them to array
    while ($row = mysqli_fetch_assoc($result)) {
        $products[] = $row;
    }
    
    // update the number of products in this product group
    $number_of_products = $number_of_products + count($products);
    
    // loop through all products in order to get price range
    foreach ($products as $product) {
        // remember that non-donation products exist
        $non_donation_products_exist = true;
        
        // if this product is discounted by an offer, then remember that discounted products exist, store original price, and set the product price to the discounted price
        if (isset($discounted_product_prices[$product['id']]) == TRUE) {
            $discounted_products_exist = TRUE;
            $original_price = $product['price'];
            $product['price'] = $discounted_product_prices[$product['id']];
        }
        
        // if smallest price is not set or the price of this product is the smallest price so far, then store smallest price
        if ((isset($smallest_price) == false) || ($product['price'] < $smallest_price)) {
            $smallest_price = $product['price'];
        }
        
        // if largest price is not set or the price of this product is the largest price so far, then store largest price
        if ((isset($large_price) == false) || ($product['price'] > $large_price)) {
            $large_price = $product['price'];
        }
    }
    
    // if a smallest price has not been found, then set to 0
    if (isset($smallest_price) == false) {
        $smallest_price = 0;
    }
    
    // if a largest price has not been found, then set to 0
    if (isset($large_price) == false) {
        $large_price = 0;
    }
    
    $price_range = array(
        'smallest_price' => $smallest_price,
        'largest_price' => $large_price,
        'non_donation_products_exist' => $non_donation_products_exist,
        'number_of_products' => $number_of_products,
        'discounted_products_exist' => $discounted_products_exist,
        'original_price' => $original_price
    );
    
    return $price_range;
}

// Gets the price or price range for a product or product group,
// that is ready to be outputted for a catalog page.
function get_price_info($properties) {

    $item = $properties['item'];
    $discounted_product_prices = $properties['discounted_product_prices'];

    // if this item is a product group, then get price range for all product groups and products in this product group
    if ($item['type'] == 'product group') {
        
        // if there are non-donation products in product group, then output price range
        if ($item['price_range']['non_donation_products_exist'] == true) {
            // if the smallest price and largest price are the same, then just output a price without a range
            if ($item['price_range']['smallest_price'] == $item['price_range']['largest_price']) {
                // if there is only one product and it is discounted, then prepare to show original price and discounted price
                if (($item['price_range']['number_of_products'] == 1) && ($item['price_range']['discounted_products_exist'] == TRUE)) {
                    return prepare_price_for_output($item['price_range']['original_price'], TRUE, $item['price_range']['smallest_price'], 'html');
                    
                // else there is more than one product or there are no discounted products, so prepare to just show original price
                } else {
                    // if this product group has a select display type and there are discounted products in it, then prepare to output single price with discounted styling
                    if (($item['display_type'] == 'select') && ($item['price_range']['discounted_products_exist'] == TRUE)) {
                        return '<span class="software_discounted_price">' . prepare_price_for_output($item['price_range']['smallest_price'], FALSE, $discounted_price = '', 'html') . '</span>';
                        
                    // else this product group does not have a select display type or there are no discounted products in it, so prepare to output single price without discounted styling
                    } else {
                        return prepare_price_for_output($item['price_range']['smallest_price'], FALSE, $discounted_price = '', 'html');
                    }
                }
                
            // else the smallest price and largest price are not the same, so output price range
            } else {
                // if this product group has a select display type and there are discounted products in it, then prepare to output price range with discounted styling
                if (($item['display_type'] == 'select') && ($item['price_range']['discounted_products_exist'] == TRUE)) {
                    return '<span class="software_discounted_price">' . prepare_price_for_output($item['price_range']['smallest_price'], FALSE, $discounted_price = '', 'html', $show_code = FALSE) . ' - ' . prepare_price_for_output($item['price_range']['largest_price'], FALSE, $discounted_price = '', 'html') . '</span>';
                    
                // else this product group does not have a select display type or there are no discounted products in it, so prepare to output price without discounted styling
                } else {
                    return prepare_price_for_output($item['price_range']['smallest_price'], FALSE, $discounted_price = '', 'html', $show_code = FALSE) . ' - ' . prepare_price_for_output($item['price_range']['largest_price'], FALSE, $discounted_price = '', 'html');
                }
            }
        }
        
    // else this item is a product, so if product is not a donation, then output product's price
    } elseif ($item['selection_type'] != 'donation') {
        // assume that the product is not discounted, until we find out otherwise
        $discounted = FALSE;
        $discounted_price = '';

        // if the product is discounted, then prepare to show that
        if (isset($discounted_product_prices[$item['id']]) == TRUE) {
            $discounted = TRUE;
            $discounted_price = $discounted_product_prices[$item['id']];
        }
        
        return prepare_price_for_output($item['price'], $discounted, $discounted_price, 'html');
    }

    return '';

}

function get_number_of_payments_message()
{
    // if credit/debit card is selected as a payment method and a payment gateway is selected, then get message
    if ((ECOMMERCE_CREDIT_DEBIT_CARD == true) && (ECOMMERCE_PAYMENT_GATEWAY != '')) {
        switch (ECOMMERCE_PAYMENT_GATEWAY) {
            case 'Authorize.Net':
                return ' (1-9999 or leave blank for no limit)';
                
            case 'ClearCommerce':
                return ' (2-999)';
                
            case 'First Data Global Gateway':
                return ' (1-99)';
            
            case 'PayPal Payflow Pro':
            case 'PayPal Payments Pro':
                return ' (leave blank for no limit)';
        }
    }
    
    return ' (leave blank for no limit)';
}

// get the charset from HTML content
function get_charset($content)
{
    // look for the charset in a meta tag
    preg_match("/<meta[^>]*http-equiv[^>]*charset=(.*)[\"|']/i", $content, $matches);
    
    // if the charset was found, then return the charset
    if (isset($matches[1]) == true) {
        return mb_strtolower(trim($matches[1]));
    } else {
        return '';
    }
}

function get_month_name_from_number($month_number)
{
    switch ($month_number) {
        case '01':
            return 'January';
            break;
        
        case '02':
            return 'February';
            break;
        
        case '03':
            return 'March';
            break;
        
        case '04':
            return 'April';
            break;
        
        case '05':
            return 'May';
            break;
        
        case '06':
            return 'June';
            break;
        
        case '07':
            return 'July';
            break;
        
        case '08':
            return 'August';
            break;
        
        case '09':
            return 'September';
            break;
        
        case '10':
            return 'October';
            break;
        
        case '11':
            return 'November';
            break;
        
        case '12':
            return 'December';
            break;
    }
}

function validate_password_strength($password)
{
    // if there are less than 10 characters then return false
    if (mb_strlen($password) < 10) {
        return false;
    }
    
    // if there is not a capital letter, then return false
    if (preg_match('/[A-Z]/', $password) == 0) {
        return false;
    }
    
    // if there are less than 2 numbers, then return false
    if (preg_match_all('/[0-9]/', $password, $matches) < 2) {
        return false;
    }
    
    // if there is not a non-alphanumeric character, then return false
    if (preg_match('/[^A-Za-z0-9]/', $password) == 0) {
        return false;
    }
    
    // if we have gotten here, then the password is strong so return true
    return true;
}

function get_strong_password_requirements() {
    return
        '<div>Your password must contain at least:</div>
        <ul>
            <li>10 characters</li>
            <li>1 capital letter</li>
            <li>2 numbers</li>
            <li>1 non-alphanumeric character (e.g. !, $)</li>
        </ul>';
}

function get_thumbnail_dimensions($image_width, $image_height, $max_dimension) {
    // Get the aspect ratio of the image.

    $width_percentage = 0;

    if ($image_width != 0) {
        $width_percentage = $max_dimension / $image_width;
    }

    $height_percentage = 0;

    if ($image_height != 0) {
        $height_percentage = $max_dimension / $image_height;
    }
    
    // If the image width is wider than the maximum width allowed then continue the code.
    if ($image_width >= $max_dimension) {
        
        // If the width percentage ratio is smaller than or equal to the height percentage ratio then continue the code.
        if ($width_percentage <= $height_percentage){
            
            // NOTE: the following will scale the image and keep the original aspect ratio.
            // Set the thumbnail width to the value of the width percentage multiplied by the current image width.
            $thumbnail_width = $width_percentage * $image_width;
            
            // Set the thumbnail height to the value of the width percentage multiplied by the current image height.
            $thumbnail_height = $width_percentage * $image_height;
        
        // If the height percentage ratio is greater than the width then continue the code.
        }else{
            
            // Set the thumbnail width to the value of the height percentage multiplied by the current image width.
            $thumbnail_width = $height_percentage * $image_width;
            
            // Set the thumbnail height to the value of the height percentage multiplied by the current image height.
            $thumbnail_height = $height_percentage * $image_height;
        }
        
    // If the image width is less than the maximum width allowed but the height is greater than the allowed amount then continue the code.
    } else if ($image_height >= $max_dimension) {
        
        // If the width percentage ratio is smaller than or equal to the height percentage ratio then continue the code.
        if($width_percentage <= $height_percentage){
            
            // NOTE: the following will scale the image and keep the original aspect ratio.
            // Set the thumbnail width to the value of the width percentage multiplied by the current image width.
            $thumbnail_width = $width_percentage * $image_width;
            
            // Set the thumbnail height to the value of the width percentage multiplied by the current image height.
            $thumbnail_height = $width_percentage * $image_height;
        }else{
            
            // Set the thumbnail width to the value of the height percentage multiplied by the current image width.
            $thumbnail_width= $height_percentage * $image_width;
            
            // Set the thumbnail height to the value of the height percentage multiplied by the current image height.
            $thumbnail_height= $height_percentage * $image_height;
        }
        
    // If the image is smaller than or equal to the allowed image size then keep original size.
    } else {
        $thumbnail_width = $image_width;
        $thumbnail_height = $image_height;
    }
    
    // Output thumbnail size.
    return array(
        'width' => $thumbnail_width,
        'height' => $thumbnail_height);
}

function convert_bytes_to_string($bytes, $round = 0) { 
    
    $sizes = array('B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'); 
    $total = count($sizes); 
    for ($i=0; $bytes > 1024 && $i < $total; $i++) $bytes /= 1024; 
    return number_format(round($bytes, $round), $round, '.', ',') . ' ' . $sizes[$i];
}

function get_design_subnav() {
    global $user;

    $output_ad_region_link = '';

    if (ADS === true) {
        $output_ad_region_link = '<li><a href="view_regions.php?filter=all_ad_regions">All Ad Regions</a></li>';
    }
    
    // if the user is an administrator and dynamic regions are enabled, then prepare to output dynamic regions link
    if (($user['role'] < 1) && ((defined('DYNAMIC_REGIONS') == true) && (DYNAMIC_REGIONS == true))) {
        $output_dynamic_region_link = '<li><a href="view_regions.php?filter=all_dynamic_regions">All Dynamic Regions</a></li>';
    }

    $output_migration = '';

    if ((defined('MIG') == true) && (MIG == true)) {
        $output_migration = '<li><a href="migration.php">Migration</a></li>';
    }
    
    return ' 
        <table>
            <tr>
                <td>
                    <ul>
                        <li><a href="view_styles.php">All Page Styles</a></li>
                    </ul>
                </td>
                <td>
                    <ul>
                        <li><a href="view_menus.php">All Menus</a></li>
                    </ul>
                </td>
                <td>
                    <ul>
                        <li><a href="view_regions.php?filter=all_common_regions">All Common Regions</a></li>
                        <li><a href="view_regions.php?filter=all_designer_regions">All Designer Regions</a></li>
                        ' . $output_ad_region_link . '
                        ' . $output_dynamic_region_link . '
                        <li><a href="view_regions.php?filter=all_login_regions">All Login Regions</a></li>
                    </ul>
                </td>
                <td>
                    <ul>
                        <li><a href="view_themes.php">All Themes</a></li>
                    </ul>
                </td>
                <td>
                    <ul>
                        <li><a href="view_design_files.php">All Design Files</a></li>
                    </ul>
                </td>
                <td>
                    <ul>
                        <li><a href="import_design.php">Import My Site</a></li>
                        <li><a href="import_zip.php">Import ZIP File</a></li>
                        ' . $output_migration . '
                    </ul>
                </td>
            </tr>
        </table>';
}

function get_email_campaign_status_name($status)
{
    switch ($status) {
        case 'ready':
            if (defined('EMAIL_CAMPAIGN_JOB') and EMAIL_CAMPAIGN_JOB) {
                return 'Scheduled';
            } else {
                return 'Ready to Send';
            }
            break;
            
        case 'paused':
            return 'Paused';
            break;
        
        case 'cancelled':
            return 'Cancelled';
            break;
        
        case 'complete':
            return 'Complete';
            break;
    }
}

function generate_url_id()
{
    // if the URL array in the session has not been initialized yet, then initialize it
    if (isset($_SESSION['software']['urls']) == false) {
        $_SESSION['software']['urls'] = array();
    }
    
    // get current request uri
    $request_uri = get_request_uri();
    
    // check to see if the current URL has already been stored
    $url_id = array_search($request_uri, $_SESSION['software']['urls']);
    
    // if a URL id was found, then return it
    if ($url_id !== false) {
        return $url_id;
        
    // else a URL id was not found, so store URL in session and return URL id
    } else {
        $_SESSION['software']['urls'][] = $request_uri;
        
        return count($_SESSION['software']['urls']) - 1;
    }
}

// get product groups and products that should appear in search results for the site search and for the catalog search
// the logic below might seem more complex than necessary, which is because we are trying to minimize the number of database queries,
// in order to get good performance for sites with many product groups and products
function get_catalog_search_results($search_query, $product_group_id) {

    // If the product group is disabled, then return an empty array of results.
    if (!db_value("SELECT enabled FROM product_groups WHERE id = '" . e($product_group_id) . "'")) {
        return array();
    }

    // get all product groups so we can determine which product groups can appear in search results
    $query =
        "SELECT
            id,
            parent_id,
            display_type
        FROM product_groups
        WHERE enabled = '1'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $product_groups = array();
    
    // add each product group to an array
    while ($row = mysqli_fetch_assoc($result)) {
        $product_groups[] = $row;
    }
    
    // get all relationships between products and product groups so we can determine which products can appear in search results
    $query =
        "SELECT
            product as product_id,
            product_group as product_group_id
        FROM products_groups_xref";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $products_product_groups_xrefs = array();
    
    // add each relationship to an array
    while ($row = mysqli_fetch_assoc($result)) {
        $products_product_groups_xrefs[] = $row;
    }
    
    $items_that_can_appear_in_search_results = get_catalog_items_that_can_appear_in_search_results($product_group_id, $product_groups, $products_product_groups_xrefs);
    
    // initialize array for storing product groups and products that have been added to the items array, so that we don't add duplicate items
    $added_items = array();
    $added_items['product_groups'] = array();
    $added_items['products'] = array();
    
    // initialize array for storing items
    $items = array();
    
    // get search results for keywords matches
    $search_results = get_catalog_search_results_for_field('keywords', 'contains', $search_query, $items_that_can_appear_in_search_results, $added_items);
    $items = array_merge($items, $search_results['items']);
    $added_items = array_merge($added_items, $search_results['added_items']);
    
    // get search results for exact name matches
    $search_results = get_catalog_search_results_for_field('name', 'is equal to', $search_query, $items_that_can_appear_in_search_results, $added_items);
    $items = array_merge($items, $search_results['items']);
    $added_items = array_merge($added_items, $search_results['added_items']);
    
    // get search results for partial name matches
    $search_results = get_catalog_search_results_for_field('name', 'contains', $search_query, $items_that_can_appear_in_search_results, $added_items);
    $items = array_merge($items, $search_results['items']);
    $added_items = array_merge($added_items, $search_results['added_items']);
    
    // get search results for short description matches
    $search_results = get_catalog_search_results_for_field('short_description', 'contains', $search_query, $items_that_can_appear_in_search_results, $added_items);
    $items = array_merge($items, $search_results['items']);
    $added_items = array_merge($added_items, $search_results['added_items']);
    
    // get search results for full description matches
    $search_results = get_catalog_search_results_for_field('full_description', 'contains', $search_query, $items_that_can_appear_in_search_results, $added_items);
    $items = array_merge($items, $search_results['items']);
    $added_items = array_merge($added_items, $search_results['added_items']);
    
    // get search results for details matches
    $search_results = get_catalog_search_results_for_field('details', 'contains', $search_query, $items_that_can_appear_in_search_results, $added_items);
    $items = array_merge($items, $search_results['items']);
    $added_items = array_merge($added_items, $search_results['added_items']);
    
    // get search results for meta keywords matches
    $search_results = get_catalog_search_results_for_field('meta_keywords', 'contains', $search_query, $items_that_can_appear_in_search_results, $added_items);
    $items = array_merge($items, $search_results['items']);
    $added_items = array_merge($added_items, $search_results['added_items']);
    
    return $items;
}

// we will use the function below to get all product groups and products that can appear in search results,
// because they have a certain type and are in the product group tree scope
function get_catalog_items_that_can_appear_in_search_results($parent_product_group_id, $product_groups, $products_product_groups_xrefs)
{
    $items_that_can_appear_in_search_results = array();
    $items_that_can_appear_in_search_results['product_groups'] = array();
    $items_that_can_appear_in_search_results['products'] = array();
    
    $child_product_groups = array();
    
    // loop through product groups array in order to get all product groups that are in parent product group
    foreach ($product_groups as $product_group) {
        // if the parent product group id for this product group is equal to the parent product group id, then this is a child product group, so add to array
        if ($product_group['parent_id'] == $parent_product_group_id) {
            $child_product_groups[] = $product_group;
        }
    }
    
    // loop through child product groups
    foreach ($child_product_groups as $child_product_group) {
        // if this child product group is a select product group, then add it to array
        if ($child_product_group['display_type'] == 'select') {
            $items_that_can_appear_in_search_results['product_groups'][] = $child_product_group['id'];
            
        // else this child product group is a browse product group, so get items under this product group, via recursion
        } else {
            // get child items
            $child_items_that_can_appear_in_search_results = get_catalog_items_that_can_appear_in_search_results($child_product_group['id'], $product_groups, $products_product_groups_xrefs);
            
            // add child product groups to array
            $items_that_can_appear_in_search_results['product_groups'] = array_merge($items_that_can_appear_in_search_results['product_groups'], $child_items_that_can_appear_in_search_results['product_groups']);
            
            // add child products to array
            $items_that_can_appear_in_search_results['products'] = array_merge($items_that_can_appear_in_search_results['products'], $child_items_that_can_appear_in_search_results['products']);
        }
    }
    
    // loop through products_product_groups_xrefs array in order to get all products that are in the parent product group
    foreach ($products_product_groups_xrefs as $products_product_groups_xref) {
        // if this product is in the parent product group, then add it to the array
        if ($products_product_groups_xref['product_group_id'] == $parent_product_group_id) {
            $items_that_can_appear_in_search_results['products'][] = $products_product_groups_xref['product_id'];
        }
    }
    
    // remove duplicate entries
    $items_that_can_appear_in_search_results['product_groups'] = array_unique($items_that_can_appear_in_search_results['product_groups']);
    $items_that_can_appear_in_search_results['products'] = array_unique($items_that_can_appear_in_search_results['products']);
    
    return $items_that_can_appear_in_search_results;
}

// initialize a function for getting all items for search results for a certain field, operator, and search query
// we have created a function for this so that we don't have to duplicate this large amount of code for every field that needs to be searched
function get_catalog_search_results_for_field($field, $operator, $search_query, $items_that_can_appear_in_search_results, $added_items)
{
    // initialize array that will store items
    $items = array();
    
    // initialize array that will store data for sorting
    $item_names = array();
    
    // prepare sql comparison
    $sql_comparison = "";
    
    switch ($operator) {
        case 'contains':
            $sql_comparison = "LIKE '%" . escape(escape_like($search_query)) . "%'";
            break;
            
        case 'is equal to':
            $sql_comparison = "= '" . escape($search_query) . "'";
            break;
    }
    
    // get all select product groups where the field matches the comparison
    $query =
        "SELECT
            id,
            name,
            short_description,
            image_name
        FROM product_groups
        WHERE
            (enabled = '1')
            AND (display_type = 'select')
            AND ($field $sql_comparison)";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // loop through product groups in order to add them to array
    while ($row = mysqli_fetch_assoc($result)) {
        // if this product group can appear in search results and it has not been added already, then add it
        if ((in_array($row['id'], $items_that_can_appear_in_search_results['product_groups']) == true) && (in_array($row['id'], $added_items['product_groups']) == false)) {
            $items[] = array(
                'id' => $row['id'],
                'name' => $row['name'],
                'short_description' => $row['short_description'],
                'image_name' => $row['image_name'],
                'display_type' => 'select',
                'type' => 'product group'
            );
            
            $item_names[] = mb_strtolower($row['name']);
            
            // remember that this product group has been added
            $added_items['product_groups'][] = $row['id'];
        }
    }
    
    // get all products that are in a product group where the product's field matches the comparison
    $query =
        "SELECT
            products.id as product_id,
            products.name as product_name,
            products.short_description as product_short_description,
            products.image_name as product_image_name,
            products.price as product_price,
            products.selection_type as product_selection_type,
            product_groups.id as product_group_id,
            product_groups.name as product_group_name,
            product_groups.short_description as product_group_short_description,
            product_groups.image_name as product_group_image_name,
            product_groups.display_type as product_group_display_type
        FROM products_groups_xref
        LEFT JOIN products ON products_groups_xref.product = products.id
        LEFT JOIN product_groups ON products_groups_xref.product_group = product_groups.id
        WHERE
            (products.enabled = '1')
            AND (product_groups.enabled = '1')
            AND (products.$field $sql_comparison)";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // loop through products in order to add the product or its product group to the array
    while ($row = mysqli_fetch_assoc($result)) {
        // if this product's product group is a browse product group, then add continue to determine if product should be added to array
        if ($row['product_group_display_type'] == 'browse') {
            // if this product can appear in search results and it has not been added already, then add it
            if ((in_array($row['product_id'], $items_that_can_appear_in_search_results['products']) == true) && (in_array($row['product_id'], $added_items['products']) == false)) {
                $items[] = array(
                    'id' => $row['product_id'],
                    'name' => $row['product_name'],
                    'short_description' => $row['product_short_description'],
                    'image_name' => $row['product_image_name'],
                    'price' => $row['product_price'],
                    'selection_type' => $row['product_selection_type'],
                    'type' => 'product'
                );
                
                $item_names[] = mb_strtolower($row['product_name']);
                
                // remember that this product has been added
                $added_items['products'][] = $row['product_id'];
            }
            
        // else this product's product group is a select product group, so continue to determine if we should add product group to array
        // because we don't want to forward the customer directly to the product
        } else {
            // if this product group can appear in search results and it has not been added already, then add it
            if ((in_array($row['product_group_id'], $items_that_can_appear_in_search_results['product_groups']) == true) && (in_array($row['product_group_id'], $added_items['product_groups']) == false)) {
                $items[] = array(
                    'id' => $row['product_group_id'],
                    'name' => $row['product_group_name'],
                    'short_description' => $row['product_group_short_description'],
                    'image_name' => $row['product_group_image_name'],
                    'display_type' => 'select',
                    'type' => 'product group'
                );
                
                $item_names[] = mb_strtolower($row['product_group_name']);
                
                // remember that this product group has been added
                $added_items['product_groups'][] = $row['product_group_id'];
            }
        }
    }
    
    // sort the items by name
    array_multisort($item_names, $items);
    
    return array(
        'items' => $items,
        'added_items' => $added_items
    );
}

function require_cookies() {
    // if cookies are required and the visitor has cookies disabled, then output error to the visitor
    if ((isset($_POST['require_cookies']) == true) && ($_POST['require_cookies'] == 'true') && (isset($_COOKIE[session_name()]) == false)) {
        output_error('Please enable cookies in your web browser and then <a href="javascript:history.go(-1)">go back</a> and try again.');
    }
}

// This function is currently used when a comment is added
// by the product submit form feature.
// It is not currently used when a comment is added manually,
// because that area of the code is a little different,
// however it could be used for that also in the future.

function send_comment_email_to_administrators($comment_id)
{
    // Get comment info.
    $comment = db_item(
        "SELECT
            id,
            page_id,
            item_id,
            item_type,
            name,
            message
        FROM comments
        WHERE id = '" . escape($comment_id) . "'");

    // Get info for the page that the comment is connected to.
    $page = db_item(
        "SELECT
            page_id AS id,
            page_name AS name,
            page_type AS type,
            comments_label,
            comments_administrator_email_to_email_address,
            comments_administrator_email_subject,
            comments_administrator_email_conditional_administrators
        FROM page
        WHERE page_id = '" . $comment['page_id'] . "'");

    // Prepare to e-mail addresses for the administrator e-mail.
    $administrator_email_addresses = array();

    // If there is a to e-mail address, then add it to the array.
    if ($page['comments_administrator_email_to_email_address'] != '') {
        $administrator_email_addresses[] = $page['comments_administrator_email_to_email_address'];
    }

    // If the comment is connected to a submitted form,
    // then get additional administrator e-mail addresses.
    if ($comment['item_type'] == 'submitted_form') {
        // If e-mailing conditional administrators is enabled,
        // then check if there are any to add.
        if ($page['comments_administrator_email_conditional_administrators']) {
            $custom_form_page_id = db_value("SELECT page_id FROM forms WHERE id = '" . $comment['item_id'] . "'");

            // Get conditional administrators
            $conditional_administrators = db_values(
                "SELECT form_field_options.email_address
                FROM form_field_options
                LEFT JOIN form_data ON form_field_options.form_field_id = form_data.form_field_id
                WHERE
                    (form_field_options.page_id = '$custom_form_page_id')
                    AND (form_data.form_id = '" . $comment['item_id'] . "')
                    AND (form_field_options.email_address != '')
                    AND (form_data.data = form_field_options.value)");

            // Loop through the conditional administrators in order to add them.
            foreach ($conditional_administrators as $conditional_administrator) {
                // We support multiple conditional admin email addresses, separated by comma,
                // (e.g. ^^example1@example.com,example2@example.com^^), so deal with that.
                $conditional_email_addresses = explode(',', $conditional_administrator);

                foreach ($conditional_email_addresses as $conditional_email_address) {
                    $conditional_email_address = trim($conditional_email_address);

                    // If this e-mail address has not already been added, then add it.
                    if (in_array($conditional_email_address, $administrator_email_addresses) == false) {
                        $administrator_email_addresses[] = $conditional_email_address;
                    }
                }
            }
        }

        $form_editor_email_address = db_value(
            "SELECT user.user_email as form_editor_email_address
            FROM forms
            LEFT JOIN user ON forms.form_editor_user_id = user.user_id
            WHERE forms.id = '" . $comment['item_id'] . "'");

        // If a form editor e-mail address was found and it has not already been added, then add it.
        if (($form_editor_email_address != '') && (in_array($form_editor_email_address, $administrator_email_addresses) == false)) {
            $administrator_email_addresses[] = $form_editor_email_address;
        }
    }

    // If there is at least one administrator email address to e-mail,
    // then send emails to administrators.
    if (count($administrator_email_addresses) > 0) {
        $comment_label = get_comment_label(array('label' => $page['comments_label']));
        $comment_label_lowercase = mb_strtolower($comment_label);

        // If the administrator e-mail subject is not blank,
        // then prepare it for output and set it to the e-mail subject
        if ($page['comments_administrator_email_subject'] != '') {
            // If the item is a submitted form, then replace variables
            // in subject with submitted form data
            if ($comment['item_type'] == 'submitted_form') {
                $subject = get_variable_submitted_form_data_for_content($page['id'], $comment['item_id'], $page['comments_administrator_email_subject'], $prepare_for_html = false);
                
            // Otherwise the item is not a submitted form, so just use the subject.
            } else {
                $subject = $page['comments_administrator_email_subject'];
            }
        
        // Otherwise, use the default subject.
        } else {
            $subject = 'A ' . $comment_label_lowercase . ' has been added and published.';
        }

        $body =
            'The following ' . $comment_label_lowercase . ' has been added and published:' . "\n" .
            "\n" .
            'Display Name: ' . $comment['name'] . "\n" .
            "\n" .
            $comment_label . ': ' . $comment['message'] . "\n";

        $query_string = get_query_string_for_page_url($page['type'], $comment['item_id'], $comment['item_type']);

        // If this is the first item that is being added to the query string, then add question mark.
        if (mb_strpos($query_string, '?') === false) {
            $query_string .= '?';
            
        // Otherwise this is not the first item that is being added to the query string, so add ampersand.
        } else {
            $query_string .= '&';
        }

        $query_string .= 'comments=all';
        
        $body .=
            "\n" .
            'The ' . $comment_label_lowercase . ' appears at the link below.' . "\n" .
            "\n" .
            URL_SCHEME . HOSTNAME . PATH . encode_url_path($page['name']) . $query_string . '#c-' . $comment['id'];

        email(array(
            'to' => $administrator_email_addresses,
            'from_name' => ORGANIZATION_NAME,
            'from_email_address' => EMAIL_ADDRESS,
            'subject' => $subject,
            'body' => $body
        ));
    }
}

function send_comment_email_to_custom_form_submitter($comment_id) {
    // get comment data
    $query = 
        "SELECT 
            comments.page_id,
            comments.item_id,
            comments.name,
            comments.message,
            files.name as file_name,
            files.size as file_size,
            comments.created_timestamp
        FROM comments
        LEFT JOIN files ON comments.file_id = files.id
        WHERE comments.id = '" . escape($comment_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $page_id = $row['page_id'];
    $item_id = $row['item_id'];
    $name = $row['name'];
    $message = $row['message'];
    $file_name = $row['file_name'];
    $file_size = $row['file_size'];
    $created_timestamp = $row['created_timestamp'];
    
    // get page properties
    $query = 
        "SELECT
            page.page_type,
            page.comments_submitter_email_page_id,
            page.comments_submitter_email_subject,
            custom_form_pages.submitter_email_from_email_address
        FROM page
        LEFT JOIN form_item_view_pages ON
            (page.page_id = form_item_view_pages.page_id)
            AND (form_item_view_pages.collection = 'a')
        LEFT JOIN custom_form_pages ON form_item_view_pages.custom_form_page_id = custom_form_pages.page_id
        WHERE page.page_id = '$page_id'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $page_type = $row['page_type'];
    $submitter_email_from_email_address = $row['submitter_email_from_email_address'];
    $comments_submitter_email_page_id = $row['comments_submitter_email_page_id'];
    $comments_submitter_email_subject = $row['comments_submitter_email_subject'];
    
    // get forms reference code
    $query = "SELECT reference_code FROM forms WHERE id = '$item_id'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $reference_code = $row['reference_code'];

    $email_address = get_submitter_email_address($item_id);
    
    // if there is a page to send and an e-mail address to e-mail, e-mail page
    if ($comments_submitter_email_page_id && $email_address) {
        
        if ($submitter_email_from_email_address) {
            $from_email_address = $submitter_email_from_email_address;
        } else {
            $from_email_address = EMAIL_ADDRESS;
        }
        
        // check the subject line for variable data and replace any variables with content from the submitted form
        $comments_submitter_email_subject = get_variable_submitted_form_data_for_content($page_id, $item_id, $comments_submitter_email_subject, $prepare_for_html = FALSE);
        
        $body = '';
        $extra_system_content = '';
        
        // if name is blank then output anonymous
        if ($name == '') {
            $name = 'Anonymous';
        }
        
        $output_file_attachment = '';
        
        // if there is a file attachment, then output it
        if ($file_name != '') {
            // we are using a separate link for the image and the file name because we don't want an underline on the image and we don't want to have to update all themes with new CSS
            $output_file_attachment = '<div class="software_attachment" style="margin-top: 1.5em"><a href="' . OUTPUT_PATH . h(encode_url_path($file_name)) . '" target="_blank" style="background: none; padding: 0"><img src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/icon_attachment.png" width="16" height="16" alt="attachment" title="" border="0" style="padding-right: .5em; vertical-align: middle" /></a><a href="' . OUTPUT_PATH . h(encode_url_path($file_name)) . '" target="_blank">' . h($file_name) . '</a> (' . convert_bytes_to_string($file_size) . ')</div>';
        }
        
        // set body message
        $extra_system_content =
            '<div class="comment">
                <div class="name_line"><span class="added_by">Added by</span> <span class="name">' . h($name) . '</span></div>
                <div class="date_and_time">' . get_absolute_time(array('timestamp' => $created_timestamp, 'size' => 'long', 'timezone_type' => 'site')) . '</div>
                <br />
                <div class="message">' . convert_text_to_html($message) . '</div>
                ' . $output_file_attachment . '
            </div>
            <div style="margin-top: 1em;"><a class="software_input_submit_primary reply_button" href="' . h(URL_SCHEME . HOSTNAME . PATH . get_page_name($page_id) . '?r=' . $reference_code . '&comments=all#c-' . $comment_id) . '">View or Reply</a></div>';

        require_once(dirname(__FILE__) . '/get_page_content.php');
        
        $body = get_page_content($comments_submitter_email_page_id, $system_content = '', $extra_system_content, $mode = 'preview', $email = true);
        
        email(array(
            'to' => $email_address,
            'from_name' => ORGANIZATION_NAME,
            'from_email_address' => $from_email_address,
            'subject' => $comments_submitter_email_subject,
            'format' => 'html',
            'body' => $body));

    }
}

function send_comment_email_to_watchers($comment_id) {
    // get comment and page info (the email_page join is used to determine if the page still exists)
    $query = 
        "SELECT 
            comments.page_id,
            comments.item_id,
            comments.item_type,
            comments.name,
            comments.message,
            files.name as file_name,
            files.size as file_size,
            comments.created_timestamp,
            page.page_name,
            page.page_type,
            page.comments_watcher_email_page_id,
            page.comments_watcher_email_subject,
            email_page.page_id as email_page_id
        FROM comments
        LEFT JOIN files ON comments.file_id = files.id
        LEFT JOIN page ON comments.page_id = page.page_id
        LEFT JOIN page AS email_page ON page.comments_watcher_email_page_id = email_page.page_id
        WHERE comments.id = '" . escape($comment_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $page_id = $row['page_id'];
    $item_id = $row['item_id'];
    $item_type = $row['item_type'];
    $name = $row['name'];
    $message = $row['message'];
    $file_name = $row['file_name'];
    $file_size = $row['file_size'];
    $created_timestamp = $row['created_timestamp'];
    $page_name = $row['page_name'];
    $page_type = $row['page_type'];
    $comments_watcher_email_page_id = $row['comments_watcher_email_page_id'];
    $comments_watcher_email_subject = $row['comments_watcher_email_subject'];
    $email_page_id = $row['email_page_id'];
    
    // if the page to be e-mailed still exists, then continue to e-mail watchers
    if ($email_page_id != '') {
        // get all watchers for this page and etc.
        $query = 
            "SELECT
                watchers.email_address AS watcher_email_address,
                user.user_email AS user_email_address
            FROM watchers
            LEFT JOIN user ON watchers.user_id = user.user_id
            WHERE
                (watchers.page_id = '" . escape($page_id) . "')
                AND (watchers.item_id = '" . escape($item_id) . "')
                AND (watchers.item_type = '" . escape($item_type) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $watchers = array();
        
        // loop through all watchers in order to add them to array
        while ($row = mysqli_fetch_assoc($result)) {
            $watchers[] = $row;
        }
        
        // if the page that the comment was added to is a form item view, then check the subject line for variable data and replace any variables with content from the submitted form
        if ($page_type == 'form item view') {
            $comments_watcher_email_subject = get_variable_submitted_form_data_for_content($page_id, $item_id, $comments_watcher_email_subject, $prepare_for_html = FALSE);
        }
        
        // if name is blank then output anonymous
        if ($name == '') {
            $name = 'Anonymous';
        }
        
        $output_file_attachment = '';
        
        // if there is a file attachment, then output it
        if ($file_name != '') {
            // we are using a separate link for the image and the file name because we don't want an underline on the image and we don't want to have to update all themes with new CSS
            $output_file_attachment = '<div class="software_attachment" style="margin-top: 1.5em"><a href="' . OUTPUT_PATH . h(encode_url_path($file_name)) . '" target="_blank" style="background: none; padding: 0"><img src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/icon_attachment.png" width="16" height="16" alt="attachment" title="" border="0" style="padding-right: .5em; vertical-align: middle" /></a><a href="' . OUTPUT_PATH . h(encode_url_path($file_name)) . '" target="_blank">' . h($file_name) . '</a> (' . convert_bytes_to_string($file_size) . ')</div>';
        }

        $query_string = get_query_string_for_page_url($page_type, $item_id, $item_type);

        // If this is the first item that is being added to the query string, then add question mark.
        if (mb_strpos($query_string, '?') === false) {
            $query_string .= '?';
            
        // Otherwise this is not the first item that is being added to the query string, so add ampersand.
        } else {
            $query_string .= '&';
        }

        // Add comments parameter now.
        $query_string .= 'comments=all';
        
        // set body message
        $extra_system_content =
            '<div class="comment">
                <div class="name_line"><span class="added_by">Added by</span> <span class="name">' . h($name) . '</span></div>
                <div class="date_and_time">' . get_absolute_time(array('timestamp' => $created_timestamp, 'size' => 'long', 'timezone_type' => 'site')) . '</div>
                <br />
                <div class="message">' . convert_text_to_html($message) . '</div>
                ' . $output_file_attachment . '
            </div>
            <div style="margin-top: 1em;"><a class="software_input_submit_primary reply_button" href="' . h(URL_SCHEME . HOSTNAME . PATH . $page_name . $query_string . '#c-' . $comment_id) . '">View or Reply</a></div>';

        require_once(dirname(__FILE__) . '/get_page_content.php');
        
        $body = get_page_content($comments_watcher_email_page_id, $system_content = '', $extra_system_content, $mode = 'preview', $email = true);
        
        // loop through all watchers in order to e-mail each one
        foreach ($watchers as $watcher) {

            // If this watcher is connected to a user, then use the user's e-mail address.
            if ($watcher['user_email_address'] != '') {
                $to = $watcher['user_email_address'];

            // Otherwise the watcher is not connected to a user, so use the static e-mail address for the watcher.
            } else {
                $to = $watcher['watcher_email_address'];
            }
            
            email(array(
                'to' => $to,
                'from_name' => ORGANIZATION_NAME,
                'from_email_address' => EMAIL_ADDRESS,
                'subject' => $comments_watcher_email_subject,
                'format' => 'html',
                'body' => $body));

        }
    }
}

function get_query_string_for_page_url($page_type, $item_id, $item_type) {
    $query_string = '';

    // if there is an item id, then check certain page types to see if a query string needs to be added
    if ($item_id != '0') {
        // switch between the different item types and add any needed URL parameters
        switch($page_type) {
            case 'catalog':
            case 'catalog detail':
                $address_name = '';
                
                // get the address name
                $address_name = get_catalog_item_address_name_from_id($item_id, $item_type);
                
                // if the address name is not blank, then set the query string to it
                if ($address_name != '') {
                    $query_string = '/' . $address_name;
                    
                // else set the query string to blank
                } else {
                    $query_string = '';
                }
                
                break;
                
            case 'calendar event view':
                $query_string = '?id=' . $item_id;
                
                break;
                
            case 'form item view':
                // get the reference code for this item
                $query = "SELECT reference_code FROM forms WHERE id = '" . escape($item_id) . "'";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                $row = mysqli_fetch_assoc($result);
                
                $query_string = '?r=' . $row['reference_code'];
                
                break;
        }
    }
    
    return $query_string;
}

function get_duplicate_catalog_item_address_name_number($address_name, $number = 1)
{
    // assume that the address name does not already exist until we find out otherwise
    $address_name_exists = FALSE;
    
    // check products table to see if there is already a product with an address name that matches the one we want to use
    $query = "SELECT id FROM products where address_name = '" . escape($address_name) . "[" . $number . "]'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if there is a result, then there is already a product with that address name
    if (mysqli_num_rows($result) > 0) {
        $address_name_exists = TRUE;
    }
    
    // check product groups table to see if there is already a product group with an address name that matches the one we want to use
    $query = "SELECT id FROM product_groups where address_name = '" . escape($address_name) . "[" . $number . "]'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if there is a result, then there is already a product group with that address name
    if (mysqli_num_rows($result) > 0) {
        $address_name_exists = TRUE;
    }
    
    // if the address name already exists, then use recursion to try to get a different address name
    if ($address_name_exists == TRUE) {
        return get_duplicate_catalog_item_address_name_number($address_name, $number + 1);
    } else {
        return $number;
    }
}

function prepare_catalog_item_address_name($address_name, $item_id = '')
{
    // if item id is blank then set to zero.
    if ($item_id == '') {
        $item_id = '0';
    }
    
    // prepare the address name for the database
    $address_name = trim($address_name);
    $address_name = str_replace(' ', '_', $address_name);
    $address_name = str_replace('&', '_', $address_name);
    
    // assume that the address name does not already exist until we find out otherwise
    $address_name_exists = FALSE;
    
    // check products table to see if there is already a product with an address name that matches the one we want to use
    $query = "SELECT id FROM products where address_name = '" . escape($address_name) . "' AND id != '" . escape($item_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if there is a result, then there is already a product with that address name
    if (mysqli_num_rows($result) > 0) {
        $address_name_exists = TRUE;
    }
    
    // check product groups table to see if there is already a product group with an address name that matches the one we want to use
    $query = "SELECT id FROM product_groups where address_name = '" . escape($address_name) . "' AND id != '" . escape($item_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if there is a result, then there is already a product group with that address name
    if (mysqli_num_rows($result) > 0) {
        $address_name_exists = TRUE;
    }
    
    // if the address name already exists then create a unique address name
    if ($address_name_exists == TRUE) {
        $address_name .= '[' . get_duplicate_catalog_item_address_name_number($address_name) . ']';
    }
    
    return $address_name;
}

function get_catalog_item_from_url()
{
    // get the address name
    $address_name = mb_substr(mb_substr($_GET['page'], mb_strpos($_GET['page'], '/')), 1);
    
    $item_id = '';
    $item_type = '';
    
    // get the product group if there is one
    $query =
        "SELECT
            id,
            name,
            enabled,
            short_description,
            address_name
        FROM product_groups
        WHERE address_name = '" . e($address_name) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if a product group was found, then set its information
    if (mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        
        $item_id = $row['id'];
        $item_type = 'product group';
        $item_name = $row['name'];
        $item_enabled = $row['enabled'];
        $item_short_description = $row['short_description'];
        $item_address_name = $row['address_name'];
        
    // else a product group was not found, so check for a product
    } else {
        $query =
            "SELECT
                id,
                name,
                enabled,
                short_description,
                address_name
            FROM products
            WHERE address_name = '" . escape($address_name) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // if a product was found, then set its information
        if (mysqli_num_rows($result) > 0) {
            $row = mysqli_fetch_assoc($result);
            
            $item_id = $row['id'];
            $item_type = 'product';
            $item_name = $row['name'];
            $item_enabled = $row['enabled'];
            $item_short_description = $row['short_description'];
            $item_address_name = $row['address_name'];
        }
    }
    
    return
        array(
            'id' => $item_id,
            'type' => $item_type,
            'name' => $item_name,
            'enabled' => $item_enabled,
            'short_description' => $item_short_description,
            'address_name' => $item_address_name
        );
}

function get_catalog_item_address_name_from_id($item_id, $item_type)
{
    // if the item type is product group then set sql table accordingly
    if (($item_type == 'product group') || ($item_type == 'product_group')) {
        $sql_table = 'product_groups';
        
    // else the item type is a product so set the sql tabel accordingly
    } else {
        $sql_table = 'products';
    }
    
    // get the item id based off of the item type
    $query = "SELECT address_name FROM $sql_table WHERE id = '" . escape($item_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
    $address_name = $row['address_name'];
    
    // return the address name
    return $address_name;
}

function select_rss_field($rss_field = '')
{
    $rss_field_options = '';
    
    // put rss field types into an array
    $rss_field_options_in_array = array('', 'category', 'title', 'description');
    
    // loop through each rss field type and build options for select list
    foreach ($rss_field_options_in_array as $rss_field_option_in_array) {
        $selected = '';
        
        // if rss field type is equal to the rss field value from the database, then select the option
        if ($rss_field_option_in_array == $rss_field) {
            $selected = ' selected="selected"';
        }
        
        $rss_field_options .= '<option value="' . h($rss_field_option_in_array) . '"' . $selected . '>' . h($rss_field_option_in_array) . '</option>';
    }
    
    return $rss_field_options;
}

// create function that gets data for a calendar event and unique data for a recurring instance
function get_calendar_event($event_id, $recurrence_number)
{
    // get calendar event data
    $query =
        "SELECT
            calendar_events.id,
            calendar_events.name,
            calendar_events.published,
            calendar_events.unpublish_days,
            calendar_events.short_description,
            calendar_events.full_description,
            calendar_events.notes,
            calendar_events.all_day,
            calendar_events.start_time,
            calendar_events.end_time,
            calendar_events.show_start_time,
            calendar_events.show_end_time,
            calendar_events.recurrence,
            calendar_events.recurrence_number as total_recurrence_number,
            calendar_events.recurrence_type,
            calendar_events.recurrence_day_sun,
            calendar_events.recurrence_day_mon,
            calendar_events.recurrence_day_tue,
            calendar_events.recurrence_day_wed,
            calendar_events.recurrence_day_thu,
            calendar_events.recurrence_day_fri,
            calendar_events.recurrence_day_sat,
            calendar_events.recurrence_month_type,
            calendar_events.location,
            calendar_events.reservations,
            calendar_events.separate_reservations,
            calendar_events.limit_reservations,
            calendar_events.number_of_initial_spots,
            calendar_events.no_remaining_spots_message,
            calendar_events.reserve_button_label,
            calendar_events.next_page_id,
            products.id as product_id,
            products.enabled AS product_enabled,
            products.inventory,
            products.inventory_quantity,
            products.backorder,
            products.out_of_stock_message,
            calendar_events.created_timestamp
        FROM calendar_events
        LEFT JOIN products ON calendar_events.product_id = products.id
        WHERE calendar_events.id = '" . escape($event_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    // if calendar event could not be found, then return false
    if (mysqli_num_rows($result) == 0) {
        return FALSE;
    }
    
    $row = mysqli_fetch_assoc($result);
    
    $id = $row['id'];
    $name = $row['name'];
    $published = $row['published'];
    $unpublish_days = $row['unpublish_days'];
    $short_description = $row['short_description'];
    $full_description = $row['full_description'];
    $notes_content = $row['notes'];
    $all_day = $row['all_day'];
    $start_date_and_time = $row['start_time'];
    $end_date_and_time = $row['end_time'];
    $show_start_time = $row['show_start_time'];
    $show_end_time = $row['show_end_time'];
    $recurrence = $row['recurrence'];
    $total_recurrence_number = $row['total_recurrence_number'];
    $recurrence_type = $row['recurrence_type'];
    $recurrence_day_sun = $row['recurrence_day_sun'];
    $recurrence_day_mon = $row['recurrence_day_mon'];
    $recurrence_day_tue = $row['recurrence_day_tue'];
    $recurrence_day_wed = $row['recurrence_day_wed'];
    $recurrence_day_thu = $row['recurrence_day_thu'];
    $recurrence_day_fri = $row['recurrence_day_fri'];
    $recurrence_day_sat = $row['recurrence_day_sat'];
    $recurrence_month_type = $row['recurrence_month_type'];
    $location = $row['location'];
    $reservations = $row['reservations'];
    $separate_reservations = $row['separate_reservations'];
    $limit_reservations = $row['limit_reservations'];
    $number_of_initial_spots = $row['number_of_initial_spots'];
    $no_remaining_spots_message = $row['no_remaining_spots_message'];
    $reserve_button_label = $row['reserve_button_label'];
    $next_page_id = $row['next_page_id'];
    $product_id = $row['product_id'];
    $product_enabled = $row['product_enabled'];
    $inventory = $row['inventory'];
    $inventory_quantity = $row['inventory_quantity'];
    $backorder = $row['backorder'];
    $out_of_stock_message = $row['out_of_stock_message'];
    $created_timestamp = $row['created_timestamp'];
    
    $number_of_remaining_spots = 0;
    
    // if reservations is enabled and limit reservations is enabled, then prepare to get number of remaining spots
    if (($reservations == 1) && ($limit_reservations == 1)) {
        // if this is a recurring event and reservations are separated for each recurring instance
        // then prepare to look for remaining spots for this recurrence number
        if (
            ($recurrence == 1)
            && ($separate_reservations == 1)
        ) {
            $sql_recurrence_number = $recurrence_number;
            
        // else this is not a recurring event or reservations are not separated so look for remaining spots for the 0 recurrence number
        } else {
            $sql_recurrence_number = 0;
        }
        
        // get number of remaining spots for this calendar event and recurrence number
        $query =
            "SELECT number_of_remaining_spots
            FROM remaining_reservation_spots
            WHERE
                (calendar_event_id = '" . escape($event_id) . "')
                AND (recurrence_number = '" . escape($sql_recurrence_number) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // if a remaining spots record was found then use it to populate the remaining spots
        if (mysqli_num_rows($result) > 0) {
            $row = mysqli_fetch_assoc($result);
            $number_of_remaining_spots = $row['number_of_remaining_spots'];
            
        // else a remaining spots record was not found, so use initial spots
        } else {
            $number_of_remaining_spots = $number_of_initial_spots;
        }
    }
    
    // Get the locations that this event will use in alphabetical order.
    $query =
        "SELECT
            calendar_event_locations.name
        FROM calendar_events_calendar_event_locations_xref
        LEFT JOIN calendar_event_locations ON calendar_events_calendar_event_locations_xref.calendar_event_location_id = calendar_event_locations.id
        WHERE calendar_event_id = '" . escape($event_id) . "'
        ORDER BY calendar_event_locations.name";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $calendar_locations = "";
    
    // Add each location to the main location list that will be output to the front end soon
    while ($row = mysqli_fetch_assoc($result)) {
        if ($calendar_locations) {
            $calendar_locations .= ', ';
        }
        $calendar_locations .= $row['name'];
    }
    
    // if a calendar event location with an id was not selected, then use the custom location that was manually entered
    if ($calendar_locations == '') {
        $location_content = $location;
    // else a calendar event location with an id was selected
    } else {
        // If the user entered a custom location
        if ($location != '') {
            // Add it to the end, prefixed with a comman and space.
            $location_content = $calendar_locations . ', ' . $location;
            // Do not worry about it.
        } else {
            $location_content = $calendar_locations;
        }
    }
    
    // if there is location content, then prepare to output location
    if ($location_content) {
        $location = $location_content;
    }
    
    // split start date and time in order to get date and time
    $start_date_and_time_parts = explode(' ', $start_date_and_time);
    $start_date = $start_date_and_time_parts[0];
    $start_time = $start_date_and_time_parts[1];
    
    // split end date and time in order to get date and time
    $end_date_and_time_parts = explode(' ', $end_date_and_time);
    $end_date = $end_date_and_time_parts[0];
    $end_time = $end_date_and_time_parts[1];
    
    // if this is a recurring event and this is a recurring instance,
    // then adjust start date & time and end date & time for this recurring instance
    if (($recurrence == 1) && ($recurrence_number)) {
        // Split the date into more precise values
        $event_start_date_parts = explode('-', $start_date);
        $event_start_year = $event_start_date_parts[0];
        $event_start_month = $event_start_date_parts[1];
        $event_start_day = $event_start_date_parts[2];
        
        // Find the difference between the original start date and end date (So we do not need to calculate the end dates new date too)
        $event_start_end_difference = (strtotime($end_date_and_time) - strtotime($start_date_and_time));

        switch ($recurrence_type) {
            // Daily
            case 'day':
                $event_start_date = $start_date;

                // Loop through all recurrence numbers in order to
                // calculate the date for this recurrence number.
                for ($i = 1; $i <= $recurrence_number; $i++) {
                    $count = 0;

                    // Loop through days in the future until we find a date that is valid
                    // based on the valid days of the week that were selected.
                    while (true) {
                        $new_start_time = strtotime('+1 day', strtotime($event_start_date));
                        $event_start_date = date('Y-m-d', $new_start_time);
                        $day_of_the_week = strtolower(date('D', $new_start_time));

                        // If this day of the week is valid for this calendar event,
                        // then we have found a valid date, so break out of the loop.
                        if (${'recurrence_day_' . $day_of_the_week} == 1) {
                            break;
                        }

                        $count++;

                        // If we have already looped 7 times, then something is wrong,
                        // so break out of this loop and the recurrence loop above.
                        // This should never happen but is added just in case in order to
                        // prevent an endless loop.
                        if ($count == 7) {
                            break 2;
                        }
                    }
                }
                
                break;
                
            // Weekly
            case 'week':
                $new_start_time = mktime(0, 0, 0, $event_start_month, $event_start_day + (7 * $recurrence_number), $event_start_year);
                $event_start_date = date('Y', $new_start_time) . '-' . date('m', $new_start_time) . '-' . date('d', $new_start_time);
                
                break;
            
            // Monthly
            case 'month':
                switch ($recurrence_month_type) {
                    case 'day_of_the_month':
                        $new_start_time = mktime(0, 0, 0, $event_start_month + $recurrence_number, 1, $event_start_year);
                        $new_event_start_year = date('Y', $new_start_time);
                        $new_event_start_month = date('m', $new_start_time);
                        $new_event_start_day = $event_start_day;
                        
                        // if date is not valid, then get last date for month
                        if (checkdate($new_event_start_month, $new_event_start_day, $new_event_start_year) == false) {
                            $new_event_start_day = date('t', mktime(0, 0, 0, $new_event_start_month, 1, $new_event_start_year));
                        }
                        
                        $event_start_date = $new_event_start_year . '-' . $new_event_start_month . '-' . $new_event_start_day;

                        break;

                    case 'day_of_the_week':
                        // Determine which week in the month the event is on.
                        // If the week is 1-4 then we will use that, however if the week is 5,
                        // then we interpret that as the last week.

                        $day_of_the_week = date('l', strtotime($start_date));
                        $first_day_of_the_month_timestamp = strtotime($event_start_year . '-' . $event_start_month . '-01');

                        $week = '';

                        // Create a loop in order to determine which week event falls on.
                        // We only loop through 4 weeks, because we are going to set "last" below for 5th week.
                        for ($week_index = 0; $week_index <= 3; $week_index++) {
                            // If the event is in this week, then remember the week number and break out of this loop.
                            if ($start_date == date('Y-m-d', strtotime('+' . $week_index . ' week ' . $day_of_the_week, $first_day_of_the_month_timestamp))) {
                                $week = $week_index + 1;
                                break;
                            }
                        }

                        // If a week was not found, then that means it falls on the 5th week,
                        // so set it to be the last week.
                        if ($week == '') {
                            $week = 'last';
                        }

                        $first_day_of_the_month_timestamp = mktime(0, 0, 0, $event_start_month + $recurrence_number, 1, $event_start_year);

                        // If the week is 1-4 then find the date in a certain way.
                        if ($week != 'last') {
                            $week_index = $week - 1;

                            $new_start_time = strtotime('+' . $week_index . ' week ' . $day_of_the_week, $first_day_of_the_month_timestamp);

                        // Otherwise the week is last, so find the date in a different way.
                        } else {
                            $last_day_of_the_month_timestamp = strtotime(date('Y-m-t', $first_day_of_the_month_timestamp));

                            // If the last day of the month happens to be the right day of the week,
                            // then thats that day that we want.
                            if (date('l', $last_day_of_the_month_timestamp) == $day_of_the_week) {
                                $new_start_time = $last_day_of_the_month_timestamp;

                            // Otherwise find the day of the week that we want in the last week of the month.
                            } else {
                                $new_start_time = strtotime('last ' . $day_of_the_week, $last_day_of_the_month_timestamp);
                            }
                        }

                        $event_start_date = date('Y-m-d', $new_start_time);

                        break;
                }

                break;
            
            // Yearly
            case 'year':
                $new_event_start_year = $event_start_year + $recurrence_number;
                $new_event_start_month = $event_start_month;
                $new_event_start_day = $event_start_day;
                
                // if date is not valid, then get last date for month
                if (checkdate($new_event_start_month, $new_event_start_day, $new_event_start_year) == false) {
                    $new_event_start_day = date('t', mktime(0, 0, 0, $new_event_start_month, 1, $new_event_start_year));
                }
                
                $event_start_date = $new_event_start_year . '-' . $new_event_start_month . '-' . $new_event_start_day;
                
                break;
        }
        
        // Save the new start date and new end date to the variables we were using previously.
        $start_date_and_time = $event_start_date . ' ' . $start_time;
        $end_date_and_time = date('Y-m-d H:i:s', (strtotime($start_date_and_time) + $event_start_end_difference));
    }

    // If the date format is month and then day, then use that format.
    if (DATE_FORMAT == 'month_day') {
        $month_and_day_format = 'F j';

    // Otherwise the date format is day and then month, so use that format.
    } else {
        $month_and_day_format = 'j F';
    }
    
    $date_and_time_range = '';
    $time_range = '';
    
    // If this is not an all day event then deal with both dates & times.
    if ($all_day == 0) {
        // if start date is equal to end date, then date and time range and time range in a certain way
        if ($start_date == $end_date) {
            // If the start and end times should both be shown, then do that.
            if ($show_start_time && $show_end_time) {
                $date_and_time_range = date('l, ' . $month_and_day_format . ', Y g:i A', strtotime($start_date_and_time)) . ' - ' . date('g:i A', strtotime($end_date_and_time));
                $time_range = prepare_form_data_for_output($start_time, 'time') . ' - ' . prepare_form_data_for_output($end_time, 'time');

            // Otherwise if both the start and end times should not be shown, then do that.
            } else if (!$show_start_time && !$show_end_time) {
                $date_and_time_range = date('l, ' . $month_and_day_format . ', Y', strtotime($start_date_and_time));
                $time_range = '';

            // Otherwise if only the start time should be shown, then do that.
            } else if ($show_start_time && !$show_end_time) {
                $date_and_time_range = date('l, ' . $month_and_day_format . ', Y g:i A', strtotime($start_date_and_time));
                $time_range = prepare_form_data_for_output($start_time, 'time');

            // Otherwise if only the end time should be shown, then do that.
            } else if (!$show_start_time && $show_end_time) {
                $date_and_time_range = date('l, ' . $month_and_day_format . ', Y', strtotime($start_date_and_time)) . ' (ends at ' . date('g:i A', strtotime($end_date_and_time)) . ')';
                $time_range = 'Ends at ' . prepare_form_data_for_output($end_time, 'time');
            }
            
        // else start date is not equal to end date, so prepare date and time range and time range in a certain way
        } else {
            // If the start and end times should both be shown, then do that.
            if ($show_start_time && $show_end_time) {
                $date_and_time_range = date('l, ' . $month_and_day_format . ', Y g:i A', strtotime($start_date_and_time)) . ' - ' . date('l, ' . $month_and_day_format . ', Y g:i A', strtotime($end_date_and_time));
                $time_range = prepare_form_data_for_output($start_time, 'time');

            // Otherwise if both the start and end times should not be shown, then do that.
            } else if (!$show_start_time && !$show_end_time) {
                $date_and_time_range = date('l, ' . $month_and_day_format . ', Y', strtotime($start_date_and_time)) . ' - ' . date('l, ' . $month_and_day_format . ', Y', strtotime($end_date_and_time));
                $time_range = '';

            // Otherwise if only the start time should be shown, then do that.
            } else if ($show_start_time && !$show_end_time) {
                $date_and_time_range = date('l, ' . $month_and_day_format . ', Y g:i A', strtotime($start_date_and_time)) . ' - ' . date('l, ' . $month_and_day_format . ', Y', strtotime($end_date_and_time));
                $time_range = prepare_form_data_for_output($start_time, 'time');

            // Otherwise if only the end time should be shown, then do that.
            } else if (!$show_start_time && $show_end_time) {
                $date_and_time_range = date('l, ' . $month_and_day_format . ', Y', strtotime($start_date_and_time)) . ' - ' . date('l, ' . $month_and_day_format . ', Y g:i A', strtotime($end_date_and_time));
                $time_range = '';
            }
        }
        
    // Otherwise this is an all day event, so just deal with dates.
    } else {
        // if start date is equal to end date, then prepare date and time range with only one date
        if ($start_date == $end_date) {
            $date_and_time_range = date('l, ' . $month_and_day_format . ', Y', strtotime($start_date_and_time));
            $time_range = '';
                
        // else start date is not equal to end date, so prepare date and time range with two dates
        } else {
            $date_and_time_range = date('l, ' . $month_and_day_format . ', Y', strtotime($start_date_and_time)) . ' - ' . date('l, ' . $month_and_day_format . ', Y', strtotime($end_date_and_time));
            $time_range = '';
        }
    }
    
    // return array
    return
        array (
            'id' => $id,
            'name' => $name,
            'published' => $published,
            'unpublish_days' => $unpublish_days,
            'short_description' => $short_description,
            'full_description' => $full_description,
            'notes_content' => $notes_content,
            'all_day' => $all_day,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'start_date_and_time' => $start_date_and_time,
            'end_date_and_time' => $end_date_and_time,
            'date_and_time_range' => $date_and_time_range,
            'time_range' => $time_range,
            'recurrence' => $recurrence,
            'total_recurrence_number' => $total_recurrence_number,
            'recurrence_type' => $recurrence_type,
            'recurrence_month_type' => $recurrence_month_type,
            'location' => $location,
            'reservations' => $reservations,
            'separate_reservations' => $separate_reservations,
            'limit_reservations' => $limit_reservations,
            'number_of_initial_spots' => $number_of_initial_spots,
            'number_of_remaining_spots' => $number_of_remaining_spots,
            'no_remaining_spots_message' => $no_remaining_spots_message,
            'reserve_button_label' => $reserve_button_label,
            'next_page_id' => $next_page_id,
            'product_id' => $product_id,
            'product_enabled' => $product_enabled,
            'inventory' => $inventory,
            'inventory_quantity' => $inventory_quantity,
            'backorder' => $backorder,
            'out_of_stock_message' => $out_of_stock_message,
            'created_timestamp' => $created_timestamp
        );
}

// Create function that will be used to get a fake answer to store in a hidden form field
// in order to trick spammers.
function get_fake_captcha_answer($real_answer)
{
    // Get random fake answer between 0 and 18.
    $fake_answer = rand(0, 18);

    // If the fake answer happens to be the real answer, then get different fake answer.
    if ($fake_answer == $real_answer) {
        return get_fake_captcha_answer($real_answer);

    // Else the fake answer is not the real answer, so use it.
    } else {
        return $fake_answer;
    }
}

function get_captcha_fields($liveform) {

    if (!CAPTCHA) {
        return '';
    }

    $output_captcha_fields = '';
    
    // if the user is not logged in then show the captcha form
    if (isset($_SESSION['sessionusername']) == FALSE) {
        
        // randomly generate the numbers to add and the question to be asked
        $first_number = rand(0, 9);
        $second_number = rand(0, 9);
        
        // Prepare an encypted version of the correct answer.
        // The encrypted format contains a random digit at the beginning and the end with a value in the middle
        // that is 2 higher than the correct value, and all of that base64 encoded.
        $correct_answer_encrypted = base64_encode(rand(0, 9) . ($first_number + $second_number + 2) . rand(0, 9));

        $liveform->assign_field_value('captcha_validation', $correct_answer_encrypted);

        // store incorrect, fake answer in hidden form field
        $liveform->assign_field_value('captcha_correct_answer', get_fake_captcha_answer($first_number + $second_number));
        
        // set the value for the submitted answer field to blank
        $liveform->assign_field_value('captcha_submitted_answer', '');
        
        // The body field is a honeypot field which we will use to determine if the submitter is a bot.
        // The captcha_correct_answer field actually contains a fake, incorrect answer.
        $output_captcha_fields =
            '<div class="captcha">
                <div style="display: none">' . $liveform->output_field(array('type'=>'textarea', 'name'=>'body', 'cols'=>'30', 'rows'=>'4', 'class'=>'software_textarea')) . '</div>
                ' . $liveform->output_field(array('type'=>'hidden', 'name'=>'captcha_correct_answer')) . '
                ' . $liveform->output_field(array('type'=>'hidden', 'name'=>'captcha_validation')) . '
                <div class="software_captcha_label" style="font-weight: bold; margin-bottom: .5em">To prevent spam, please tell us:</div>
                <span class="software_captcha_question">
                    What is ' . $first_number . ' + ' . $second_number . ' ?
                </span> ' .
                $liveform->output_field(array(
                    'type' => 'number',
                    'name' => 'captcha_submitted_answer',
                    'maxlength' => '2',
                    'size' => '2',
                    'required' => 'true',
                    'min' => '0',
                    'max' => '18',
                    'class' => 'software_input_text software_captcha_answer')) . '
            </div>';

    }
    
    return $output_captcha_fields;
    
}

// This is a slightly different version from the function above
// that is used by custom layouts to display captcha info.
// It is different from the function above because it does not
// output the captcha heading or question, because the custom layout will contain that.

function get_captcha_info($form) {

    // If the CAPTCHA is disabled in the settings or the user is logged in,
    // then don't show a CAPTCHA.
    if (!CAPTCHA || USER_LOGGED_IN) {
        return false;
    }
    
    // randomly generate the numbers to add and the question to be asked
    $first_number = rand(0, 9);
    $second_number = rand(0, 9);
    
    // Prepare an encypted version of the correct answer.
    // The encrypted format contains a random digit at the beginning and the end with a value in the middle
    // that is 2 higher than the correct value, and all of that base64 encoded.
    $correct_answer_encrypted = base64_encode(rand(0, 9) . ($first_number + $second_number + 2) . rand(0, 9));

    $form->assign_field_value('captcha_validation', $correct_answer_encrypted);

    // store incorrect, fake answer in hidden form field
    $form->assign_field_value('captcha_correct_answer', get_fake_captcha_answer($first_number + $second_number));
    
    $question = 'What is ' . $first_number . ' + ' . $second_number . ' ?';
    
    // set the value for the submitted answer field to blank
    $form->assign_field_value('captcha_submitted_answer', '');

    $form->set('captcha_submitted_answer', 'maxlength', 2);
    $form->set('captcha_submitted_answer', 'required', true);
    
    // The body field is a honeypot field which we will use to determine if the submitter is a bot.
    // The captcha_correct_answer field actually contains a fake, incorrect answer.
    $system =
        '<div style="display: none">
            <textarea name="body" cols="30" rows="4"></textarea>
        </div>
        <input type="hidden" name="captcha_correct_answer">
        <input type="hidden" name="captcha_validation">';

    return array(
        'question' => $question,
        'system' => $system);

}

function validate_captcha_answer($liveform)
{
    // If the visitor is not logged in then make sure that he/she answered the CAPTCHA correctly.
    if (isset($_SESSION['sessionusername']) == FALSE) {
        // If the honeypot field was filled out, then that means the submitter is a bot,
        // because the field is hidden with CSS, so add error.
        if ($liveform->get_field_value('body') != '') {
            $liveform->mark_error('captcha_submitted_answer', 'Sorry, the answer that you entered for the question is not valid. Please try again.');

        // Else the honeypot field was not filled out, so determine if the answer is correct.
        } else {
            // check to see if there is a submitted answer
            $liveform->validate_required_field('captcha_submitted_answer', 'You must answer the question before continuing. Please try again.');

            // If there is not already an error for the answer field, then determine if answer is correct.
            if ($liveform->check_field_error('captcha_submitted_answer') == FALSE) {
                // Start off decrypting the correct answer by base64 decoding the encrypted answer.
                $correct_answer_decrypted = base64_decode($liveform->get_field_value('captcha_validation'));

                // Remove the first character which is garbage.
                $correct_answer_decrypted = mb_substr($correct_answer_decrypted, 1);

                // Remove the last character which is also garbage.
                $correct_answer_decrypted = mb_substr($correct_answer_decrypted, 0, -1);

                // Subtract the remaining value by 2 to get the correct answer.
                $correct_answer_decrypted = $correct_answer_decrypted - 2;

                // If the answer that the visitor submitted is incorrect, then add error
                if ($liveform->get_field_value('captcha_submitted_answer') != $correct_answer_decrypted) {
                    $liveform->mark_error('captcha_submitted_answer', 'The answer you entered for the question was incorrect. Please try again.');

                // Otherwise the answer is correct, so if a blacklist is enabled,
                // then determine if IP address is on a blacklist and should be rejected.
                // We do this check after verifying that the captcha answer is correct,
                // in order to minimize requests to stopforumspam.com.
                } else if (defined('BLACKLIST') and BLACKLIST) {
                    $response = decode_json(@file_get_contents('http://www.stopforumspam.com/api?ip=' . urlencode($_SERVER['REMOTE_ADDR']) . '&f=json'));

                    // If there was a communication error with the blacklist service, then log that.
                    if ((isset($response['success']) == false) || ($response['success'] == false)) {
                        $error = '';

                        // If blacklist service returned an error, then include that info in log message.
                        if ($response['error'] != '') {
                            $error = ' (' . $response['error'] . ')';
                        }

                        log_activity('an error occurred when communicating with the spam protection service' . $error . ', so visitor\'s request was accepted', $_SESSION['sessionusername']);

                    // Otherwise there was not a communication error with the blacklist service,
                    // so if the visitor is on the blacklist, then log activity and add error.
                    } else if ($response['ip']['appears'] == true) {
                        log_activity('visitor\'s request was denied because the spam protection service reported that visitor\'s IP address was used by spammers. (confidence: ' . round($response['ip']['confidence']) . '%, frequency: ' . number_format($response['ip']['frequency']) . ')', $_SESSION['sessionusername']);
                        $liveform->mark_error('', 'Sorry, we were not able to accept your request.');
                    }
                }
            }
        }
    }
}

function get_checkboxes_for_items_user_can_edit($item_type, $selected_ids = array())
{
    $sql_table = '';
    $sql_columns = '';
    $sql_where = '';
    
    // prepare tables and columns for sql
    switch ($item_type) {
        case 'ad_regions':
            $sql_table = 'ad_regions';
            $sql_columns = '
                id,
                name';
            break;
        
        case 'common_regions':
            $sql_table = 'cregion';
            $sql_columns = '
                cregion_id as id,
                cregion_name as name';
            $sql_where = '
                WHERE cregion_designer_type = "no"';
            break;
        
        case 'menus':
            $sql_table = 'menus';
            $sql_columns = '
                id,
                name';
            break;
    }
    
    $items = array();
    
    // get items from database
    $query = 
        "SELECT
            $sql_columns
        FROM $sql_table
        $sql_where
        ORDER BY name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while ($row = mysqli_fetch_assoc($result)) {
        $items[] = $row;
    }
    
    // format item type for checkboxes
    $item_type = mb_substr($item_type, 0, -1);
    
    // loop through each item and build checkbox options
    foreach ($items as $item) {
        $checked = '';
        
        // if the item id is in the selected items array, then check this checkbox
        if (in_array($item['id'], $selected_ids) == true) {
            $checked = ' checked="checked"';
        }
        
        $output .= '<div style="white-space: nowrap"><input type="checkbox" name="' . $item_type . '_' . $item['id'] . '" id="' . $item_type . '_' . $item['id'] . '" value="1" class="checkbox"' . $checked . ' /><label for="' . $item_type . '_' . $item['id'] . '"> ' . h($item['name']) . '</label></div>';
    }
    
    return $output;
}

function get_items_user_can_edit($item_type, $user_id)
{
    $sql_table = '';
    $sql_columns = '';
    
    // prepare tables and columns for sql
    switch ($item_type) {
        case 'ad_regions':
            $sql_table = 'users_ad_regions_xref';
            $sql_columns = 'ad_region_id';
            break;
        
        case 'common_regions':
            $sql_table = 'users_common_regions_xref';
            $sql_columns = 'common_region_id';
            break;
            
        case 'contact_groups':
            $sql_table = 'users_contact_groups_xref';
            $sql_columns = 'contact_group_id';
            break;
        
        case 'menus':
            $sql_table = 'users_menus_xref';
            $sql_columns = 'menu_id';
            break;
    }
    
    $items_user_can_edit = array();
    
    // get the items that this user can edit
    $query =
        "SELECT $sql_columns as id
        FROM $sql_table
        WHERE user_id = '" . escape($user_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    while ($row = mysqli_fetch_assoc($result)) {
        $items_user_can_edit[] = $row['id'];
    }
    
    return $items_user_can_edit;
}

function add_edit_button_for_images($object_type = '', $object_id = 0, $content = '', $column_to_update = '')
{
    // If this is for a form list view or form item view, then do not add buttons for GIF's,
    // because for GIF's we have to save a new copy as a PNG because of a PicMonkey limitation,
    // and we do not currently support saving a new copy for images in form list views and form item views.
    if (
        ($object_type == 'form_list_view')
        || ($object_type == 'form_item_view')
    ) {
        $allowed_file_extensions = 'jpg|jpeg|png';

    // Otherwise this is not for a form list view or form item view, so include GIF as an allowed format.
    } else {
        $allowed_file_extensions = 'gif|jpg|jpeg|png';
    }

    // if curl is installed, and if there are image tags in the content, then add edit container to the images
    if ((function_exists('curl_init') == true) && (preg_match_all('/(<img.*?src="(.*?(' . $allowed_file_extensions . '))".*?>)/i', $content, $matches, PREG_SET_ORDER) != 0)) {
        // loop through all images in order to add an edit button to each image
        foreach($matches as $match) {
            $image_source = trim($match[2]);

            // Remove {path} from the image source so that we can find the image name.
            $image_source = str_replace('{path}', '', $image_source);

            // If there is a slash in the image source,
            // then get image name by looking at content after last slash.
            if (mb_strpos($image_source, '/') !== false) {
                $position_of_last_slash = mb_strrpos($image_source, '/');

                $image_name = mb_substr($image_source, $position_of_last_slash + 1);

            // Otherwise, the image name is the whole source.
            } else {
                $image_name = $image_source;
            }

            $image_name = trim($image_name);

            $image_name = rawurldecode($image_name);
            
            // get the image content
            $image_content = $match[0];
            
            // if there is an ID for the image, then set image id to it so that it can be passed into javascript function
            if (preg_match('/id="(.*?)"/i', $image_content, $id) != 0) {
                $image_id = unhtmlspecialchars($id[1]);
                
            // else, add an ID to the image
            } else {
                // get a unique image id
                $image_id = 'software_image_' . get_unique_image_id($image_name);
                
                // update the image tag with the id
                $image_content = preg_replace('/<img(.*?)>/i', '<img$1 id="' . h($image_id) . '">', $image_content, 1);
            }
            
            // if there is a mouseover event for the image, then add our js function call to it
            if (preg_match('/onmouseover=".*?"/i', $image_content) != 0) {
                $image_content = preg_replace('/onmouseover="(.*?)"/i', 'onmouseover="software_show_or_hide_image_edit_button(\'' . h(escape_javascript($image_id)) . '\', event); $1"', $image_content, 1);
            
            // else, add onmouseover event to image tag
            } else {
                $image_content = preg_replace('/<img(.*?)>/i', '<img$1 onmouseover="software_show_or_hide_image_edit_button(\'' . h(escape_javascript($image_id)) . '\', event);">', $image_content, 1);
            }
            
            // if there is a onmouseout event for the image, then add our js function call to it
            if (preg_match('/onmouseout=".*?"/i', $image_content) != 0) {
                $image_content = preg_replace('/onmouseout="(.*?)"/i', 'onmouseout="software_show_or_hide_image_edit_button(\'' . h(escape_javascript($image_id)) . '\', event); $1"', $image_content, 1);
            
            // else, add onmouseover event to image tag
            } else {
                $image_content = preg_replace('/<img(.*?)>/i', '<img$1 onmouseout="software_show_or_hide_image_edit_button(\'' . h(escape_javascript($image_id)) . '\', event);">', $image_content, 1);
            }
            
            // remove any left over spaces
            $image_content = str_replace('  ', ' ', $image_content);
            
            // if there is a column specified to update, then prepare it for the link
            if ($column_to_update != '') {
                $column_to_update = '&amp;column_to_update=' . $column_to_update;
            }
            
            // add link to the image editor
            $image_content .= '<a id="software_edit_button_for_' . h($image_id) . '" href="' . URL_SCHEME . HOSTNAME . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/image_editor_edit.php?file_name=' . h(urlencode($image_name)) . '&amp;object_type=' . $object_type . '&amp;object_id=' . $object_id . $column_to_update . '&amp;send_to=' . h(urlencode(get_request_uri())) . '" style="position: absolute; left: 0px; top: 0px; display: none; padding: .5em; margin: 0em 0em 0em 1.5em; text-decoration: none; z-index: 9; border: 0" title="Edit Image (' . h($image_name) . ') with PicMonkey" onmouseover="software_show_or_hide_image_edit_button(\'' . h(escape_javascript($image_id)) . '\', event);" onmouseout="software_show_or_hide_image_edit_button(\'' . h(escape_javascript($image_id)) . '\', event);"><img src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/icon_picmonkey.png" width="86" height="20" alt="PicMonkey"></a>';
            
            // output the image we created above with a link around it
            $content = preg_replace('/' . escape_regex($match[0]) . '/i', $image_content, $content, 1);
        }
    }
    
    return $content;
}

// we should check sometime to see if we can use the built-in function preg_quote() instead of this
// PHP 4 and 5 both appear to support preg_quote()
// if we start using preg_quote(), make sure to also pass the delimiter also (e.g. preg_quote($string, '/'))
function escape_regex($string)
{
    // set patterns to replace
    $patterns = array('/\//', '/\^/', '/\./', '/\$/', '/\|/', '/\(/', '/\)/', '/\[/', '/\]/', '/\*/', '/\+/', '/\?/', '/\{/', '/\}/', '/\,/');
    
    // set the replacement characters
    $replace = array('\/', '\^', '\.', '\$', '\|', '\(', '\)', '\[', '\]', '\*', '\+', '\?', '\{', '\}', '\,');
    
    // return the modified string
    return preg_replace($patterns, $replace, $string);
}

// global array used to create unique image id
$image_ids_in_array = array();

// the following function is used in order to generate id's for images, so that the image edit button can be added
function get_unique_image_id($image_name, $number = 0)
{
    global $image_ids_in_array;
    
    $image_id = '';
    
    // if the number is greater than zero, then add an underscore with the number to the image name
    if ($number > 0) {
        $image_id = $image_name . '_' . $number;
    
    // else just use the image name as the id
    } else {
        $image_id = $image_name;
    }
    
    // if the current image name is already in the array, then call this function again until a unique id is found
    if (in_array($image_id, $image_ids_in_array) === TRUE) {
        return get_unique_image_id($image_name, $number + 1);
    }
    
    // add the image id to the array
    $image_ids_in_array[] = $image_id;
    
    return $image_id;
}

function update_image_in_content($content, $original_image_name, $new_image_name)
{
    // if there are image tags in the content that are in the files directory, then find the image that needs to be updated and update it's attributes
    if (preg_match_all('/<img.*?src="(.*?)".*?>/i', $content, $matches) != 0) {
        // split image tags and names into their own arrays
        $image_tags = $matches[0];
        $image_sources = $matches[1];
        
        // Loop through the image sources and see if they match the image we are needing to update.
        foreach($image_sources as $key => $image_source) {
            // Remove {path} from the image source so that we can find the image name.
            $image_source = str_replace('{path}', '', $image_source);

            // If there is a slash in the image source,
            // then get image name by looking at content after last slash.
            if (mb_strpos($image_source, '/') !== false) {
                $position_of_last_slash = mb_strrpos($image_source, '/');

                $image_name = mb_substr($image_source, $position_of_last_slash + 1);

            // Otherwise, the image name is the whole source.
            } else {
                $image_name = $image_source;
            }

            $image_name = trim($image_name);

            // If this is the image that we need to update, then update it.
            if (
                ($image_name == $original_image_name)
                || ($image_name == encode_url_path($original_image_name))
            ) {
                // set the image tag content in a variable
                $image_content = $image_tags[$key];
                
                // If the orignal file name is different than the current file name,
                // then update the src, title and alt attributes in the image tag.
                if ($original_image_name != $new_image_name) {
                    $image_content = preg_replace('/src=".*?"/i', 'src="{path}' . h($new_image_name) . '"', $image_content);
                    $image_content = preg_replace('/alt=".*?"/i', 'alt="' . h($new_image_name) . '"', $image_content);
                    $image_content = preg_replace('/title=".*?"/i', 'title="' . h($new_image_name) . '"', $image_content);
                }
                
                // Get the dimensions of the new image
                $image_size = getimagesize(FILE_DIRECTORY_PATH . '/' . $new_image_name);
                $image_width = $image_size[0];
                $image_height = $image_size[1];
                
                // Update the width and height attributes in the image tag.
                // The standard image plugin for CKEditor uses style="width: 100px; height: 100px;"
                // instead of width="100" height="100", so we have added support for that
                // format also.  We have left the code for the old width="100" height="100"
                // for images that were entered with old editor or if someone manually enters
                // that type of format.
                $image_content = preg_replace('/width=".*?"/i', 'width="' . $image_width . '"', $image_content);
                $image_content = preg_replace('/width:.*?px/i', 'width: ' . $image_width . 'px', $image_content);
                $image_content = preg_replace('/height=".*?"/i', 'height="' . $image_height . '"', $image_content);
                $image_content = preg_replace('/height:.*?px/i', 'height: ' . $image_height . 'px', $image_content);
                
                // return the updated image tag
                $content = preg_replace('/' . escape_regex($image_tags[$key]) . '/i', $image_content, $content);
            }
        }
    }
    
    // return the content
    return $content;
}

function send_givex_request($type, $gift_card_code, $amount = 0)
{
    $result = array();
    
    // if the request is a balance check, then prepare for request in a certain way
    if ($type == 'balance') {
        $method = 'balance';
        $params = array('0', 'G', ECOMMERCE_GIVEX_USER_ID, $gift_card_code);
    
    // else if the request is a redemption, so prepare for request in a different way
    } elseif ($type == 'redemption') {
        $method = 'secureRedemption';
        $params = array('0', 'G', ECOMMERCE_GIVEX_USER_ID, ECOMMERCE_GIVEX_PASSWORD, $gift_card_code, $amount);
    }
    
    $xml_params = '';
    
    // loop through the params in order to prepare XML for them
    foreach ($params as $param) {
        $xml_params .=
            '<param>
                <value>
                    <string>' . h($param) . '</string>
                </value>
            </param>';
    }
    
    $request =
        '<?xml version="1.0" ?>
        <methodCall>
            <methodName>' . $method . '</methodName>
            <params>
                ' . $xml_params . '
            </params>
        </methodCall>';
    
    // initialize cURL
    $ch = curl_init();

    curl_setopt($ch, CURLOPT_URL, 'https://' . ECOMMERCE_GIVEX_PRIMARY_HOSTNAME . ':50042');
    curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: text/xml'));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $request);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST,  0);
    curl_setopt($ch, CURLOPT_FORBID_REUSE, true);
    curl_setopt($ch, CURLOPT_POST, 1);
    
    // if there is a proxy address, then send cURL request through proxy
    if (PROXY_ADDRESS != '') {
        curl_setopt($ch, CURLOPT_HTTPPROXYTUNNEL, true);
        curl_setopt($ch, CURLOPT_PROXYTYPE, CURLPROXY_HTTP);
        curl_setopt($ch, CURLOPT_PROXY, PROXY_ADDRESS);
    }

    // get cURL response
    $response_data = curl_exec($ch);
    
    $curl_errno = curl_errno($ch);
    $curl_error = curl_error($ch);
    
    curl_close($ch);
    
    // if there was a cURL error, then send the same request to the secondary Givex server
    if ($curl_errno != 0) {
        // initialize cURL
        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, 'https://' . ECOMMERCE_GIVEX_SECONDARY_HOSTNAME . ':50042');
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: text/xml'));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $request);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST,  0);
        curl_setopt($ch, CURLOPT_FORBID_REUSE, true);
        curl_setopt($ch, CURLOPT_POST, 1);
        
        // if there is a proxy address, then send cURL request through proxy
        if (PROXY_ADDRESS != '') {
            curl_setopt($ch, CURLOPT_HTTPPROXYTUNNEL, true);
            curl_setopt($ch, CURLOPT_PROXYTYPE, CURLPROXY_HTTP);
            curl_setopt($ch, CURLOPT_PROXY, PROXY_ADDRESS);
        }

        // get cURL response
        $response_data = curl_exec($ch);
        
        $curl_errno = curl_errno($ch);
        $curl_error = curl_error($ch);
        
        curl_close($ch);
        
        // if there was also a cURL error for the secondary Givex server, then return error
        if ($curl_errno != 0) {
            $result['curl_errno'] = $curl_errno;
            $result['curl_error'] = $curl_error;
            return $result;
        }
    }
    
    // get all response values
    preg_match_all('/<string>(.*?)<\/string>/s', $response_data, $matches);
    
    // put response values in an array
    $response = $matches[1];
    
    // if there is an error, then return error
    if ((isset($response[1]) == FALSE) || ($response[1] != 0)) {
        // if there is not a standard error, then get error message from first value
        if (isset($response[1]) == FALSE) {
            $result['error_message'] = $response[0];
            
        // else there is a standard error, so get error message from third value
        } else {
            $result['error_message'] = $response[2];
        }
        
        return $result;
    }
    
    // if the request is a balance check, then prepare result in a certain way
    if ($type == 'balance') {
        $result['balance'] = $response[2];
        return $result;
    
    // else if the request is a redemption, so prepare result in a different way
    } elseif ($type == 'redemption') {
        $result['authorization_number'] = $response[2];
        $result['balance'] = $response[3];
        return $result;
    }
}

function protect_gift_card_code($gift_card_code)
{
    return '************' . mb_substr($gift_card_code, -4);
}

function protect_givex_gift_card_code($gift_card_code)
{
    // first 6 digits + next 5 digits hidden + next variable number of digits up until the last digit + last digit hidden
    return mb_substr($gift_card_code, 0, 6) . '*****' . mb_substr($gift_card_code, 11, -1) . '*';
}

// This function merges duplicate contacts together and then forwards the user back to the all duplicate contacts view
function merge_contacts($contacts)
{
    $child_contacts = array();
    $organized_contacts = array();
    
    /*** Start contacts array organization: ***/
    /* Organize the contacts array so that a child always comes
       before it's orphans. This is needed so that we can merge in 
       alphabetical order and always deal with the child before it's orphans.*/
    
    // loop through the contacts to put children contacts in their own array
    foreach ($contacts as $key => $contact) {
        // if the contact has a user id then it is a child so add it to the child contacts array
        if ($contact['user_id'] != '') {
            $child_contacts[$key] = $contact;
        }
    }
    
    // loop through all contacts to organize them
    foreach ($contacts as $contact) {
        // loop through all child contacts to see if there are any children for this contact
        foreach ($child_contacts as $key => $child_contact) {
            // if the child's email address is the same as the contact's email address, then add it to the organized contacts array, 
            // and remove it from the other arrays so that it isn't found again
            if (mb_strtolower($child_contact['email_address']) == mb_strtolower($contact['email_address'])) {
                $organized_contacts[] = $child_contact;
                
                // remove contact from arrays
                unset($contacts[$key]);
                unset($child_contacts[$key]);
            }
        }
        
        // if this contact is not a child then add it to the organized contacts array
        if ($contact['user_id'] == '') {
            $organized_contacts[] = $contact;
        }
    }
    
    // if there are organized contacts, then unset the contacts array and set it to the organized contacts array
    if (count($organized_contacts) > 0) {
        unset($contacts);
        $contacts = $organized_contacts;
    }
    
    /*** End array organization ***/
    
    $contacts_contact_groups_xref_include_where_statement == '';
    
    // loop through the contacts and build sql where statement to include the contacts, this will be used to get all of the contact groups to merge
    foreach ($contacts as $contact) {
        if ($contacts_contact_groups_xref_include_where_statement != '') {
            $contacts_contact_groups_xref_include_where_statement .= ' OR ';
        }
        
        $contacts_contact_groups_xref_include_where_statement .= "(contacts_contact_groups_xref.contact_id = '" . escape($contact['id']) . "')";
    }
    
    $where = '';
    
    // if there are contact groups to get then set the where statement to get them
    if ($contacts_contact_groups_xref_include_where_statement != '') {
        // If where is blank
        if ($where == '') {
            $where .= ' WHERE ';
        
        // else where is not blank, so add and
        } else {
            $where .= ' AND ';
        }
        
        $where .= "(" . $contacts_contact_groups_xref_include_where_statement . ")";
    }
    
    $contact_groups = array();
    
    // get the contact groups that will be merged
    $query = 
        "SELECT
            contacts_contact_groups_xref.contact_id,
            contacts_contact_groups_xref.contact_group_id,
            contact_groups.name,
            opt_in.opt_in,
            contact_groups.email_subscription
        FROM contacts_contact_groups_xref
        LEFT JOIN contact_groups ON contact_groups.id = contacts_contact_groups_xref.contact_group_id
        LEFT JOIN opt_in ON (contacts_contact_groups_xref.contact_id = opt_in.contact_id) AND (contacts_contact_groups_xref.contact_group_id = opt_in.contact_group_id)
        $where";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while($row = mysqli_fetch_assoc($result)) {
        $contact_groups[] = $row;
    }
    
    $processed_contact_ids = array();
    $merged_contacts_counter = 0;
    
    // loop through each contact and process a merge if able
    foreach ($contacts as $contact) {
        // if this contact is a child and if it has not already been processed, then process a merge for a child
        if (($contact['user_id'] != '') && (in_array($contact['id'], $processed_contact_ids) == FALSE)) {
            
            // define the contact as a child so that it is easier to read comparisons later on
            $child = $contact;
            
            // add this child to the list of processed contacts so that it isn't processed again
            $processed_contact_ids[] = $child['id'];
            
            $orphans_to_merge = array();
            
            // get all the orphans that match the child's e-mail address and put into an array
            foreach ($contacts as $orphan) {
                // if the orphan email adress is the same as the current child's, and if this orphan is not a child, 
                // and if this orphan has not already been processed, then add it to add it to arrays to be processed
                if ((mb_strtolower($orphan['email_address']) == mb_strtolower($child['email_address'])) && ($orphan['user_id'] == '') && (in_array($orphan['id'], $processed_contact_ids) == FALSE)) {
                    $orphans_to_merge[] = $orphan;
                    $processed_contact_ids[] = $orphan['id'];
                }
            }
            
            // if there are orphans to merge, then run the logic to merge them
            if (count($orphans_to_merge) > 0) {
                $orphans_to_merge_sort_order = array();
                
                // loop through each orphan to merge to build a sorting order array
                foreach ($orphans_to_merge as $key => $orphan_to_merge) {
                    $orphans_to_merge_sort_order[$key] = $orphan_to_merge['timestamp'];
                }
                
                // sort the orphans to merge array by last modified
                array_multisort($orphans_to_merge_sort_order, SORT_DESC, $orphans_to_merge);
                
                $contact_groups_to_be_merged = array();
                $delete_from_contacts_where_statement = '';
                $delete_from_contacts_contact_groups_xref_where_statement = '';
                
                // loop through orphans to merge to get the child's new opt in status, 
                // an array of all contact groups to be merged into the child,
                // build sql where statements to be used to clean up the contacts and contacts contact groups xref table,
                // and increment the merged contacts counter
                foreach ($orphans_to_merge as $orphan_to_merge) {
                    // if the child's new opt in status is not already set to opt out,
                    // and if this orphan's opt in status is set to opt out then set the child's new opt in status to be opt out
                    if (($child['opt_in'] != 0) && ($orphan_to_merge['opt_in'] == 0)) {
                        $child['opt_in'] = 0;
                    }
                    
                    // loop through contact groups to build a list of contact groups to merge into the child contact,
                    foreach ($contact_groups as $contact_group) {
                        // if the conact groups contact id is the same as this orphan's id, or if it is the same as the child's ids, then add the contact groups to the contact groups to be merged array
                        if (($contact_group['contact_id'] == $orphan_to_merge['id']) || ($contact_group['contact_id'] == $child['id'])) {
                            $contact_groups_to_be_merged[] = $contact_group;
                        }
                    }
                    
                    // build sql where statements to be used to clean up the contacts table
                    if ($delete_from_contacts_where_statement != '') {
                        $delete_from_contacts_where_statement .= ' OR ';
                    }
                    
                    $delete_from_contacts_where_statement .= "(id = '" . escape($orphan_to_merge['id']) . "')";
                    
                    // build sql where statements to be used to clean up the contacts contact groups xref table
                    if ($delete_from_contacts_contact_groups_xref_where_statement != '') {
                        $delete_from_contacts_contact_groups_xref_where_statement .= ' OR ';
                    }
                    
                    $delete_from_contacts_contact_groups_xref_where_statement .= "(contact_id = '" . escape($orphan_to_merge['id']) . "')";
                    
                    // increment the merged contacts counter, this number will be used when outputting a notice to the user
                    $merged_contacts_counter++;
                }
                
                // if the newest orphan is newer than the child then update the timestamp and last modified user
                if ($orphans_to_merge[0]['timestamp'] > $child['timestamp']) {
                    $child['timestamp'] = $orphans_to_merge[0]['timestamp'];
                    $child['user'] = $orphans_to_merge[0]['user'];
                }
                
                // update child's contact record with new opt in status, last modified date, and last modified user
                $query =
                    "UPDATE contacts
                    SET
                       opt_in = '" . escape($child['opt_in']) . "',
                       timestamp = '" . escape($child['timestamp']) . "',
                       user = '" . escape($child['user']) . "'
                    WHERE id = '" . escape($child['id']) . "'";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                
                $where = '';
                
                // if there are orphans to delete then set the where statement to delete them, and then delete them
                if ($delete_from_contacts_where_statement != '') {
                    // If where is blank
                    if ($where == '') {
                        $where .= ' WHERE ';
                    
                    // else where is not blank, so add and
                    } else {
                        $where .= ' AND ';
                    }
                    
                    $where .= "(" . $delete_from_contacts_where_statement . ")";
                    
                    $query = "DELETE FROM contacts $where";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                }
                
                $where = '';
                
                // if there are contact groups to delete, then set the where statement to delete them and then delete the contact group and their opt-in status
                if ($delete_from_contacts_contact_groups_xref_where_statement != '') {
                    // If where is blank
                    if ($where == '') {
                        $where .= ' WHERE ';
                    
                    // else where is not blank, so add and
                    } else {
                        $where .= ' AND ';
                    }
                    
                    $where .= "(" . $delete_from_contacts_contact_groups_xref_where_statement . ")";
                    
                    $query = "DELETE FROM contacts_contact_groups_xref $where";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                    
                    $query = "DELETE FROM opt_in $where";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                }
                
                $merged_contact_group_ids = array();
                
                // loop through the contact groups to be merged in order to update contact groups xref and opt in table to add references for the child
                foreach ($contact_groups_to_be_merged as $contact_group_to_be_merged) {
                    // if this contact group has not been merged yet, then process the merge
                    if (in_array($contact_group_to_be_merged['contact_group_id'], $merged_contact_group_ids) == FALSE) {
                        // add this contact group to the merged contact groups array so that it is not processed again
                        $merged_contact_group_ids[] = $contact_group_to_be_merged['contact_group_id'];
                        
                        // delete any records from the contacts contact groups xref table for this contact group and child
                        $query = "DELETE FROM contacts_contact_groups_xref WHERE (contact_id = '" . escape($child['id']) . "') AND (contact_group_id = '" . escape($contact_group_to_be_merged['contact_group_id']) . "')";
                        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                        
                        // insert a new xref record
                        $query =
                            "INSERT INTO contacts_contact_groups_xref (
                                contact_id,
                                contact_group_id)
                            VALUES (
                                '" . escape($child['id']) . "',
                                '" . escape($contact_group_to_be_merged['contact_group_id']) . "'
                                )";
                        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                        
                        $contact_groups_to_be_merged_opt_in = '1';
                        
                        // loop through the contact groups to be merged to get the new opt in value for this contact group,
                        // and add the contact group's id to the merged contact groups ids array so that they are not processed after this one is complete
                        foreach($contact_groups_to_be_merged as $contact_group) {
                            // if this contact group id is the same as the contact group id that we are merging in, then continue
                            if ($contact_group['contact_group_id'] == $contact_group_to_be_merged['contact_group_id']) {
                                // if the contact group's opt in is set to opt out, then use that value when merging the contact group
                                if ($contact_group['opt_in'] == '0') {
                                    $contact_groups_to_be_merged_opt_in = '0';
                                }
                                
                                // add this contact group to the meged contact groups array so that it is not processed
                                $merged_contact_group_ids[] = $contact_group['contact_group_id'];
                            }
                        }
                        
                        // if contact group has email subscription turned on, then get the opt in value and update database
                        if ($contact_group_to_be_merged['email_subscription'] == 1) {
                            // if the contact groups to be merged opt in is set to opt out, then opt out the contact group we are merging
                            if ($contact_groups_to_be_merged_opt_in == '0') {
                                $contact_group_to_be_merged['opt_in'] = '0';
                            
                            // else default to opt in
                            } else {
                                $contact_group_to_be_merged['opt_in'] = '1';
                            }
                            
                            // delete any records from the opt in table for this contact group and child
                            $query = "DELETE FROM opt_in WHERE (contact_id = '" . escape($child['id']) . "') AND (contact_group_id = '" . escape($contact_group_to_be_merged['contact_group_id']) . "')";
                            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                            
                            // insert a new opt in record
                            $query =
                                "INSERT INTO opt_in (
                                    contact_id,
                                    contact_group_id,
                                    opt_in)
                                VALUES (
                                    '" . escape($child['id']) . "',
                                    '" . escape($contact_group_to_be_merged['contact_group_id']) . "',
                                    '" . escape($contact_group_to_be_merged['opt_in']) . "')";
                            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                        }
                    }
                }
            }
        
        // else if this is an orphan that hasn't been processed yet then process a merge for an orphan
        } elseif (in_array($contact['id'], $processed_contact_ids) == FALSE) {
            // put contacts into current orphan to make it easier to read the conditions below
            $current_orphan = $contact;
            
            $orphans_to_merge = array();
            
            // get all the orphans that match the current orphan's e-mail address and put them into the orphans to merge and processed contact ids arrays
            foreach ($contacts as $orphan) {
                // if the orphan email adress is the same as the current orphan's, 
                // and if this orphan has not already been processed, then add it to the orphans to merge and processed contact ids array
                if ((mb_strtolower($orphan['email_address']) == mb_strtolower($current_orphan['email_address'])) && (in_array($orphan['id'], $processed_contact_ids) == FALSE)) {
                    $orphans_to_merge[] = $orphan;
                    $processed_contact_ids[] = $orphan['id'];
                }
            }
            
            // if there are orphans to merge, then run the logic to merge them
            if (count($orphans_to_merge) > 0) {
                $orphans_to_merge_sort_order = array();
                
                // loop through each orphan to merge to build a sorting order array
                foreach ($orphans_to_merge as $key => $orphan_to_merge) {
                    $orphans_to_merge_sort_order[$key] = $orphan_to_merge['timestamp'];
                }
                
                // sort the orphans to merge array by last modified
                array_multisort($orphans_to_merge_sort_order, SORT_DESC, $orphans_to_merge);
                
                // set the newest orphan as the remaining orphan
                $remaining_orphan = $orphans_to_merge[0];
                
                // remove the remaining orphan from the orphans to merge array
                unset($orphans_to_merge[0]);
                
                $remaining_orphans_fields_to_update = array();
                
                // get the blank fields from the remaining orphan's contact record
                foreach ($remaining_orphan as $key => $remaining_orphan_field_value) {
                    // if the field value is set to the default and if this is not the user id or opt in field, then add the field to the array
                    if (
                        (
                            ($remaining_orphan_field_value == '') 
                            || ($remaining_orphan_field_value == '0') 
                            || ($remaining_orphan_field_value == '0000-00-00')
                        ) 
                        && ($key != 'user_id') 
                        && ($key != 'opt_in')
                    ) {
                        $remaining_orphans_fields_to_update[$key] = $remaining_orphan_field_value;
                    }
                }
                
                $contact_groups_to_be_merged = array();
                $delete_from_contacts_where_statement = '';
                $delete_from_contacts_contact_groups_xref_where_statement = '';
                
                // loop through orphans to merge to get the remaining orphan's new opt in status, 
                // get data to be merged into the remaining orphan's blank field from the current orphan,
                // get an array of all contact groups to be merged into the remaining orphan,
                // build sql where statements to be used to clean up the contacts and contacts contact groups xref tables,
                // and increment the merged contacts counter
                foreach ($orphans_to_merge as $orphan_to_merge) {
                    // get the remaining orphans new opt in status
                    if (($remaining_orphan['opt_in'] != 0) && ($orphan_to_merge['opt_in'] == 0)) {
                        $remaining_orphan['opt_in'] = 0;
                    }
                    
                    // loop through the remaining orphan's fields to update the blank fields with data from the current orphan
                    foreach ($remaining_orphans_fields_to_update as $key => $remaining_orphans_field_to_update) {
                        // if this is the expiration date or if it is the warning expiration date and if it is set to the default, and if the orphan that will be merged has a value, then use that value
                        if ((($key == 'expiration_date') || ($key == 'warning_expiration_date')) && ($remaining_orphans_field_to_update == '0000-00-00') && ($orphan_to_merge[$key] != '0000-00-00')) {
                            $remaining_orphans_fields_to_update[$key] = $orphan_to_merge[$key];
                        
                        // else if this is the affiliate approved or affiliate commission rate and if it is set to the default, and if the orphan that will be merged has a value, then use that value
                        } elseif ((($key == 'affiliate_approved') || ($key == 'affiliate_commission_rate')) && ($remaining_orphans_field_to_update == '0') && ($orphan_to_merge[$key] != '0')) {
                            $remaining_orphans_fields_to_update[$key] = $orphan_to_merge[$key];
                        
                        // else if the remaining field to update is blank, and if the orphan that will be merged has a value, then use that value
                        } elseif (($remaining_orphans_field_to_update == '') && ($orphan_to_merge[$key] != '')) {
                            $remaining_orphans_fields_to_update[$key] = $orphan_to_merge[$key];
                        }
                    }
                    
                    // loop through contact groups to build a list of contact groups to merge into the remaining orphan contact
                    foreach ($contact_groups as $contact_group) {
                        // if the conact groups contact id is the same as this orphan's id, or if it is the same as the remaining orphan's id, then add the contact groups to the contact groups to be merged array
                        if (($contact_group['contact_id'] == $orphan_to_merge['id']) || ($contact_group['contact_id'] == $remaining_orphan['id'])) {
                            $contact_groups_to_be_merged[] = $contact_group;
                        }
                    }
                    
                    // build sql where statements to be used to clean up the contact table
                    if ($delete_from_contacts_where_statement != '') {
                        $delete_from_contacts_where_statement .= ' OR ';
                    }
                    
                    $delete_from_contacts_where_statement .= "(id = '" . escape($orphan_to_merge['id']) . "')";
                    
                    // build sql where statements to be used to clean up the contact groups xref table
                    if ($delete_from_contacts_contact_groups_xref_where_statement != '') {
                        $delete_from_contacts_contact_groups_xref_where_statement .= ' OR ';
                    }
                    
                    $delete_from_contacts_contact_groups_xref_where_statement .= "(contact_id = '" . escape($orphan_to_merge['id']) . "')";
                    
                    // increment the merged contacts counter, this number will be used when outputting a notice to the user
                    $merged_contacts_counter++;
                }
                
                $sql_columns_to_update = '';
                
                // build an sql update statement that updates the remaining contact's blank fields with data
                foreach ($remaining_orphans_fields_to_update as $key => $remaining_orphans_field_to_update) {
                    if ($remaining_orphans_field_to_update != '') {
                        $sql_columns_to_update .= $key . " = '" . escape($remaining_orphans_field_to_update) . "', ";
                    }
                }
                
                // update remaining orphan's contact record with data from orphans, new last mod date and opt status
                $query =
                    "UPDATE contacts
                    SET
                       $sql_columns_to_update
                       opt_in = '" . escape($remaining_orphan['opt_in']) . "',
                       timestamp = '" . escape($remaining_orphan['timestamp']) . "'
                    WHERE id = '" . escape($remaining_orphan['id']) . "'";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                
                $where = '';
                
                // if there are orphans to delete, then prepare the where statement and then delete them
                if ($delete_from_contacts_where_statement != '') {
                    // If where is blank
                    if ($where == '') {
                        $where .= ' WHERE ';
                    
                    // else where is not blank, so add and
                    } else {
                        $where .= ' AND ';
                    }
                    
                    $where .= "(" . $delete_from_contacts_where_statement . ")";
                    
                    $query = "DELETE FROM contacts $where";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                }
                
                $where = '';
                
                // if there are contact groups to delete, then prepare the where statement and then delete the contact group and their opt-in status
                if ($delete_from_contacts_contact_groups_xref_where_statement != '') {
                    // If where is blank
                    if ($where == '') {
                        $where .= ' WHERE ';
                    
                    // else where is not blank, so add and
                    } else {
                        $where .= ' AND ';
                    }
                    
                    $where .= "(" . $delete_from_contacts_contact_groups_xref_where_statement . ")";
                    
                    $query = "DELETE FROM contacts_contact_groups_xref $where";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                    
                    $query = "DELETE FROM opt_in $where";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                }
                
                $merged_contact_group_ids = array();
                
                // update contact groups xref and opt in table to add references for the remaining orphan
                foreach ($contact_groups_to_be_merged as $contact_group_to_be_merged) {
                    // if this contact group has not been merged yet, then process the merge
                    if (in_array($contact_group_to_be_merged['contact_group_id'], $merged_contact_group_ids) == FALSE) {
                        // add this contact group to the meged contact groups array so that it is not processed again
                        $merged_contact_group_ids[] = $contact_group_to_be_merged['contact_group_id'];
                        
                        // delete the any records from the contacts contact groups xref table for this contact group and orphan
                        $query = "DELETE FROM contacts_contact_groups_xref WHERE (contact_id = '" . escape($remaining_orphan['id']) . "') AND (contact_group_id = '" . escape($contact_group_to_be_merged['contact_group_id']) . "')";
                        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                        
                        // insert a new xref record
                        $query =
                            "INSERT INTO contacts_contact_groups_xref (
                                contact_id,
                                contact_group_id)
                            VALUES (
                                '" . escape($remaining_orphan['id']) . "',
                                '" . escape($contact_group_to_be_merged['contact_group_id']) . "'
                                )";
                        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                        
                        $contact_groups_to_be_merged_opt_in = 1;
                        
                        // loop through the contact groups to be merged to get the new opt in value,
                        // and add the contact groups to the merged contact groups ids array so that they are not processed again
                        foreach($contact_groups_to_be_merged as $contact_group) {
                            // if this contact group id is the same as the contact group id that we are merging in, then continue
                            if ($contact_group['contact_group_id'] == $contact_group_to_be_merged['contact_group_id']) {
                                // if the contact group's opt in is set to opt out, then use that value when merging the contact group
                                if ($contact_group['opt_in'] == 0) {
                                    $contact_groups_to_be_merged_opt_in = 0;
                                }
                                
                                // add this contact group to the meged contact groups array so that it is not processed again
                                $merged_contact_group_ids[] = $contact_group['contact_group_id'];
                            }
                        }
                        
                        // if contact group has email subscription turned on, then get the opt in value and update database
                        if ($contact_group_to_be_merged['email_subscription'] == 1) {
                            // if the contact groups to be merged opt in is set to opt out, then opt out the contact group we are merging
                            if ($contact_groups_to_be_merged_opt_in == 0) {
                                $contact_group_to_be_merged['opt_in'] = 0;
                            
                            // else default to opt in
                            } else {
                                $contact_group_to_be_merged['opt_in'] = 1;
                            }
                            
                            // delete the any records from the opt in table for this contact group and child
                            $query = "DELETE FROM opt_in WHERE (contact_id = '" . escape($remaining_orphan['id']) . "') AND (contact_group_id = '" . escape($contact_group_to_be_merged['contact_group_id']) . "')";
                            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                            
                            // insert a new opt in record
                            $query =
                                "INSERT INTO opt_in (
                                    contact_id,
                                    contact_group_id,
                                    opt_in)
                                VALUES (
                                    '" . escape($remaining_orphan['id']) . "',
                                    '" . escape($contact_group_to_be_merged['contact_group_id']) . "',
                                    '" . escape($contact_group_to_be_merged['opt_in']) . "')";
                            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                        }
                    }
                }
            }
        }
    }
    
    return $merged_contacts_counter;
}

function get_codemirror_includes()
{
    return
        '<link rel="stylesheet" href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/codemirror/codemirror.css">
        <link rel="stylesheet" href="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/codemirror/pastel-on-dark.css">
        <script type="text/javascript" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/codemirror/codemirror.js"></script>';
}

function get_codemirror_javascript($properties)
{
    switch ($properties['code_type']) {
        case 'mixed':
        default:
            $output_mode = 'htmlmixed';
            break;

        case 'css':
            $output_mode = 'css';
            break;
        
        case 'javascript':
            $output_mode = 'javascript';
            break;

        case 'php':
            $output_mode = 'php';
            break;
    }
    
    $output_readonly = '';
    
    if ($properties['readonly'] == true) {
        $output_readonly = 'readOnly: true,';
    }

    // Docs said that "viewportMargin: Infinity" is required for auto-grow to work.
    // Did not appear to be required, but we decided to leave it in, just in case.

    return
        '<script>
            var editor = CodeMirror.fromTextArea(document.getElementById("' . $properties['id'] . '"), {
                mode: "' . $output_mode . '",
                lineNumbers: true,
                indentUnit: 4,
                theme: "pastel-on-dark",
                lineWrapping: false,
                styleActiveLine: true,
                ' . $output_readonly . '
                viewportMargin: Infinity
            });
        </script>';
}

// this function will update the keywords in the tag cloud keywords table for a page
function update_tag_cloud_keywords_for_page($page_id, $new_page_search, $new_meta_keywords, $original_page_search, $original_meta_keywords)
{
    // if the new include in site search in on, and if there is data in the new web browser keywords field, then update the tag cloud table
    if (($new_page_search == 1) && ($new_meta_keywords != '')) {
        // break the keywords into an array
        $new_keywords = explode(',', $new_meta_keywords);
        
        // loop through the keywords to tag to only add keywords that are not blank and remove any extra spaces before and after the keyword
        foreach ($new_keywords as $key => $new_keyword) {
            if ($new_keyword != '') {
                $new_keywords[$key] = trim($new_keyword);
            }
        }
        
        // remove duplicate entries from the array
        $new_keywords = array_unique($new_keywords);
        
        // if the original page search is on and if there are original meta keywords, then there is going to be records in the database,
        // so remove the original keywords from the new keywords array and remove the orignal keywords from the database if they need to be removed
        if (($original_page_search == 1) && ($original_meta_keywords != '')) {
            // break the keywords into an array
            $original_keywords = explode(',', $original_meta_keywords);
            
            // loop through the keywords to tag to only add keywords that are not blank and remove any extra spaces before and after the keyword
            foreach ($original_keywords as $key => $original_keyword) {
                if ($original_keyword != '') {
                    $original_keywords[$key] = trim($original_keyword);
                }
            }
            
            // remove duplicate entries from the array
            $original_keywords = array_unique($original_keywords);
            
            // loop through the old and new keywords arrays to remove any keywords that are in both, and to remove old keywords that are not in the new keywords array
            foreach ($original_keywords as $original_keyword) {
                $found_keyword = FALSE;
                
                foreach ($new_keywords as $key => $new_keyword) {
                    // if the original keyword matches the new keyword, then remove it from the new keywords array and indicate that a keyword was found
                    if ($original_keyword == $new_keyword) {
                        unset($new_keywords[$key]);
                        $found_keyword = TRUE;
                    }
                }
                
                // if a keyword was not found, then remove it from the database
                if ($found_keyword == FALSE) {
                    $query = "DELETE FROM tag_cloud_keywords WHERE ((keyword = '" . escape($original_keyword) . "') AND (item_id = '" . escape($page_id) . "') AND (item_type = 'page'))";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                }
            }
        }
        
        // loop through the new keywords and add them to the database
        foreach ($new_keywords as $key => $new_keyword) {
            // if the new keyword is not blank, then insert the keyword
            if ($new_keyword != '') {
                $query = 
                    "INSERT INTO tag_cloud_keywords 
                    (
                        keyword, 
                        item_id, 
                        item_type
                    ) VALUES (
                        '" . escape($new_keyword) . "',
                        '" . escape($page_id) . "',
                        'page'
                    )";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            }
        }
    
    // else if the original page search is on and the original meta keywords are not blank, then remove any keywords for this page from the database
    } elseif (($original_page_search == 1) && ($original_meta_keywords != '')) {
        $query = "DELETE FROM tag_cloud_keywords WHERE item_id = '" . escape($page_id) . "' AND item_type = 'page'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    }
}

// this function gets all product groups within the parent product group that was passed into this function
function get_product_groups_in_product_group_tree($parent_product_group_id = 0, $all_product_groups = array())
{
    $product_groups = array();
    
    // if this is the first time this function has run, then get all product groups and put them in an array,
    // and add the first parent product group to the array
    if (count($all_product_groups) == 0) {
        // get all product groups
        $query =
            "SELECT
                id,
                parent_id,
                display_type
            FROM product_groups
            ORDER BY sort_order, name";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // add each product group to an array
        while ($row = mysqli_fetch_assoc($result)) {
            $all_product_groups[] = $row;
        }
        
        // loop through all product groups to add the parent product group to the array
        foreach ($all_product_groups as $product_group) {
            if ($product_group['id'] == $parent_product_group_id) {
                $product_groups[] = $product_group;
            }
        }
    }
    
    $child_product_groups = array();
    
    // loop through all product groups to get child product groups
    foreach ($all_product_groups as $product_group) {
        // if this product group's parent id is equal to the parent product group id then it's a child,
        // so add it to the product groups array, and then get it's children if it has any
        if ($product_group['parent_id'] == $parent_product_group_id) {
            $product_groups[] = $product_group;
            
            // if this product group is a browse product group, then get it's child product groups
            if ($product_group['display_type'] == 'browse') {
                $child_product_groups = get_product_groups_in_product_group_tree($product_group['id'], $all_product_groups);
            }
            
            // add the children to the product groups array
            $product_groups = array_merge($product_groups, $child_product_groups);
        }
    }
    
    return $product_groups;
}

// this function updates the tag cloud keywords that are in products and product groups whenever a search results page type is modified
function update_tag_cloud_keywords_for_search_results_page_type($page_id, $new_search_results_search_catalog_items, $new_search_results_product_group_id, $extra = '') {
    // get search catalog items and product group id for this page
    $query = "SELECT search_catalog_items, product_group_id FROM search_results_pages WHERE page_id = '" . escape($page_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $row = mysqli_fetch_assoc($result);
	$original_search_catalog_items = $row['search_catalog_items'];
	$original_product_group_id = $row['product_group_id'];
    
    if ($original_search_catalog_items == '') {
        $original_search_catalog_items = 0;
    }
    
    if ($new_search_results_product_group_id == '') {
        $new_search_results_product_group_id = 0;
    }
    
    // if "search products" has been enabled, or if "search products" is enabled and if the search results product group id is different than the current search results product group id value, 
    // then update the tag cloud tables accordingly
    if (
        (
            ($original_search_catalog_items == 0) 
            && ($new_search_results_search_catalog_items == 1)
        ) || (
            ($new_search_results_search_catalog_items == 1) 
            && ($new_search_results_product_group_id != $original_product_group_id)
        )
    ) {
        // if the search catalog items was already on, and if the selected product group has changed, then remove all of the xref records and keywords from the tag cloud tables for the items inside of the original product group
        if (($original_search_catalog_items == 1) && ($new_search_results_product_group_id != $original_product_group_id)) {
            delete_tag_cloud_keywords_for_search_results_page($page_id);
        }
        
        // call function that is responsible for updating the tag cloud keywords for the search results page's product group
        update_tag_cloud_keywords_for_search_results_page_product_group($page_id, $new_search_results_product_group_id);
    
    // else if "search products" is being turned off, or if the records need to be deleted,
    // then remove all of the xref records and keywords from the tag cloud tables for the items inside of the original product group
    } elseif (($original_search_catalog_items == 1) && ($new_search_results_search_catalog_items == 0)) {
        delete_tag_cloud_keywords_for_search_results_page($page_id);
    }
}

function delete_tag_cloud_keywords_for_search_results_page($page_id) {
    $all_items = array();
    
    // get items from xref table
    $query = 
        "SELECT 
            search_results_page_id,
            item_id,
            item_type
        FROM tag_cloud_keywords_xref";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while($row = mysqli_fetch_assoc($result)) {
        $all_items[] = $row;
    }
    
    $items_to_delete = array();
    
    // loop through items to find the ones we want to delete
    foreach($all_items as $key => $item) {
        // if the item's page id is equal to this page's id, then add it to the item to delete array and remove it from the items array
        if ($item['search_results_page_id'] == $page_id) {
            $items_to_delete[] = $item;
            unset($all_items[$key]);
        }
    }
    
    // loop through the items to remove their xref records and keywords
    foreach ($items_to_delete as $item_to_delete) {
        $delete_keywords = TRUE;
        
        // loop through all the items to see if this item has an xref record for a different search result page
        foreach($all_items as $item) {
            // if the item to delete's id and type is equal to the id and type of the current item, then this item is being used for a different search results page,
            // so indicate that it's keywords shouldn't be deleted
            if (($item_to_delete['item_id'] == $item['item_id']) && ($item_to_delete['item_type'] == $item['item_type'])) {
                $delete_keywords = FALSE;
            }
        }
        
        // if the keywords can be deleted, then delete them
        if ($delete_keywords == TRUE) {
            $query = "DELETE FROM tag_cloud_keywords WHERE (item_id = '" . escape($item_to_delete['item_id']) . "') AND (item_type = '" . escape($item_to_delete['item_type']) . "')";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        }
        
        // delete the xref record for this item
        $query = "DELETE FROM tag_cloud_keywords_xref WHERE ((search_results_page_id = '" . escape($page_id) . "') AND (item_id = '" . escape($item_to_delete['item_id']) . "') AND (item_type = '" . escape($item_to_delete['item_type']) . "'))";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    }
}

function update_tag_cloud_keywords_for_search_results_page_product_group($page_id, $new_search_results_product_group_id) {
    $product_groups = array();
    
    // get all product groups within the scope of the product group id
    $product_groups = get_product_groups_in_product_group_tree($new_search_results_product_group_id);
    
    $where = '';
    
    // loop through product groups to build sql where statment
    foreach ($product_groups as $product_group) {
        // if where isn't blank, then add or
        if ($where != '') {
            $where .= ' OR ';
        }
        
        $where .= "(id = '" . escape($product_group['id']) . "')";
    }
    
    if ($where != '') {
        $where = '(' . $where . ') AND ';
    }
    
    $items = array();
    
    // get product group data from the database
    $query = 
        "SELECT 
            product_groups.id,
            product_groups.name,
            product_groups.keywords,
            product_groups.display_type
        FROM product_groups
        WHERE $where(display_type = 'select')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while($row = mysqli_fetch_assoc($result)) {
        $items[] = 
            array(
                'type' => 'product_group', 
                'display_type' => 'display_type', 
                'id' => $row['id'], 
                'name' => $row['name'], 
                'keywords' => $row['keywords']
            );
    }
    
    $where = '';
    
    // loop through product groups to build sql where statment
    foreach ($product_groups as $product_group) {
        // if the product group's display type is set to browse, then add this product group id to the where statement
        if ($product_group['display_type'] == 'browse') {
            // if where isn't blank, then add or
            if ($where != '') {
                $where .= ' OR ';
            }
            
            $where .= "(products_groups_xref.product_group = '" . escape($product_group['id']) . "')";
        }
    }
    
    if ($where != '') {
        $where = 'WHERE (' . $where . ')';
    }
    
    // get product data from the database
    $query = 
        "SELECT 
            products.id,
            products.name,
            products.keywords 
        FROM products_groups_xref
        LEFT JOIN products ON products.id = products_groups_xref.product
        $where";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while($row = mysqli_fetch_assoc($result)) {
        $items[] = 
            array(
                'type' => 'product', 
                'id' => $row['id'], 
                'name' => $row['name'], 
                'keywords' => $row['keywords']
            );
    }
    
    $tag_cloud_keywords = array();
    
    // get data from tag cloud keywords table
    $query = 
        "SELECT 
            keyword,
            item_id,
            item_type
        FROM tag_cloud_keywords
        WHERE (item_type = 'product') OR (item_type = 'product_group')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while($row = mysqli_fetch_assoc($result)) {
        $tag_cloud_keywords[] = $row;
    }
    
    $inserted_items = array();
    $inserted_keywords = array();
    
    // loop through each item to add it's keywords to the tag cloud
    foreach ($items as $item) {
        $item_keywords = explode(',', $item['keywords']);
        
        // trim the keywords and remove any blank entries
        foreach ($item_keywords as $key => $item_keyword) {
            // if the keyword is not blank, then trim it
            if ($item_keyword != '') {
                $item_keywords[$key] = trim($item_keyword);
                
            // otherwise remove it from the array
            } else {
                unset($item_keywords[$key]);
            }
        }
        
        // remove duplicates from the array
        $item_keywords = array_unique($item_keywords);
        
        // loop through the keywords and add them to the tag cloud if able
        foreach ($item_keywords as $item_keyword) {
            $record_exits = FALSE;
            
            // loop through tag cloud keywords to see if there is already a record for this keyword
            foreach ($tag_cloud_keywords as $tag_cloud_keyword) {
                // if the item's keyword, id, and type matches the tag cloud keywords then a record for this keyword already exists
                if (($item_keyword == $tag_cloud_keyword['keyword']) && ($item['id'] == $tag_cloud_keyword['item_id']) && ($item['type'] == $tag_cloud_keyword['item_type'])) {
                    $record_exits = TRUE;
                }
            }
            
            // if a record doesn't exist in the database, then loop through the inserted keywords to see if this keyword has been added to the database
            if ($record_exits == FALSE) {
                // loop through the inserted keywords to see if there is already a record for this keyword
                foreach ($inserted_keywords as $inserted_keyword) {
                    // if the item's keyword, id, and type matches the inseted keywords then a record for this keyword already exists
                    if (($item_keyword == $inserted_keyword['keyword']) && ($item['id'] == $inserted_keyword['item_id']) && ($item['type'] == $inserted_keyword['item_type'])) {
                        $record_exits = TRUE;
                    }
                }
            }
            
            // if there is not a record for this item, then add one
            if ($record_exits == FALSE) {
                // if the new keyword is not blank, then insert the keyword
                if ($item_keyword != '') {
                    $query = 
                        "INSERT INTO tag_cloud_keywords 
                        (
                            keyword, 
                            item_id, 
                            item_type
                        ) VALUES (
                            '" . escape($item_keyword) . "',
                            '" . escape($item['id']) . "',
                            '" . escape($item['type']) . "'
                        )";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                    
                    // add keyword to the inserted keywords array so that it wont be added again
                    $inserted_keywords[] = array('keyword' => $item_keyword, 'item_id' => $item['id'], 'item_type' => $item['type']);
                }
            }
        }
        
        $record_exits = FALSE;
            
        // loop through inserted items to see if there is already a record for this item
        foreach ($inserted_items as $inserted_item) {
            // if the item's id and type matches the inserted items then a xref for this item already exists
            if (($item['id'] == $inserted_item['id']) && ($item['type'] == $inserted_item['type'])) {
                $record_exits = TRUE;
            }
        }
        
        // if there is not a record for this item, then add one
        if ($record_exits == FALSE) {
            $query = 
                "INSERT INTO tag_cloud_keywords_xref 
                (
                    search_results_page_id, 
                    item_id, 
                    item_type
                ) VALUES (
                    '" . escape($page_id) . "',
                    '" . escape($item['id']) . "',
                    '" . escape($item['type']) . "'
                )";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            // add item to array so that it is not inserted again
            $inserted_items[] = $item;
        }
    }
}

// this function is responsible for checking content for variables, and replacing any variables with the content from the submitted form
function get_variable_submitted_form_data_for_content($current_page_id, $submitted_form_id, $content, $prepare_for_html = TRUE)
{
    // if the content contains variables then replace them with data
    if (mb_strpos($content, '^^') !== FALSE) {
        // get standard fields
        $standard_fields = get_standard_fields_for_view();

        // loop through all standard fields in order to prepare filters
        foreach ($standard_fields as $standard_field) {
            if ($sql_field_selects) {
                $separator = ",\n";
            } else {
                $separator = '';
            }
            
            $sql_field_selects .= $separator . $standard_field['sql_name'] . " as " . $standard_field['value'];
        }

        // get submitted form
        $query =
            "SELECT
                forms.id,
                forms.user_id as submitter_id,
                forms.form_editor_user_id,
                forms.page_id as custom_form_page_id,
                submitter.user_badge AS submitter_badge,
                submitter.user_badge_label AS submitter_badge_label,
                last_modifier.user_badge AS last_modifier_badge,
                last_modifier.user_badge_label AS last_modifier_badge_label,
                newest_comment_submitter.user_badge AS newest_comment_submitter_badge,
                newest_comment_submitter.user_badge_label AS newest_comment_submitter_badge_label,
                $sql_field_selects
            FROM forms
            LEFT JOIN user as submitter ON forms.user_id = submitter.user_id
            LEFT JOIN user as last_modifier ON forms.last_modified_user_id = last_modifier.user_id
            LEFT JOIN submitted_form_info ON ((forms.id = submitted_form_info.submitted_form_id) AND (submitted_form_info.page_id = '" . e($current_page_id) . "'))
            LEFT JOIN comments AS newest_comment ON submitted_form_info.newest_comment_id = newest_comment.id
            LEFT JOIN user AS newest_comment_submitter ON newest_comment.created_user_id = newest_comment_submitter.user_id
            WHERE forms.id = '" . e($submitted_form_id) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $submitted_form = mysqli_fetch_assoc($result);

        // get form data for all custom fields
        $query =
            "SELECT
                form_data.form_field_id,
                form_data.data,
                count(*) as number_of_values,
                form_fields.name,
                form_fields.type,
                form_fields.office_use_only as office_use_only,
                files.name as file_name
            FROM form_data
            LEFT JOIN form_fields ON form_data.form_field_id = form_fields.id
            LEFT JOIN files on form_data.file_id = files.id
            WHERE form_data.form_id = '" . escape($submitted_form['id']) . "'
            GROUP BY form_data.form_field_id";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

        // initialize array to remember which fields have data, for use with conditionals later
        $custom_fields_with_data = array();

        $fields = array();

        while ($row = mysqli_fetch_assoc($result)) {
            $fields[] = $row;
        }
        
        // loop through all field data
        foreach ($fields as $field) {
            // if there is more than one value, get all values
            if ($field['number_of_values'] > 1) {
                $query =
                    "SELECT data
                    FROM form_data
                    WHERE (form_id = '" . escape($submitted_form['id']) . "') AND (form_field_id = '" . $field['form_field_id'] . "')
                    ORDER BY id";
                $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                
                $field['data'] = array();
                
                while ($row = mysqli_fetch_assoc($result)) {
                    $field['data'][] = $row['data'];
                }
            }
                
            $data = '';
            
            // if there are multiple data parts, prepare data string with commas for separation
            if (is_array($field['data']) == true) {
                foreach ($field['data'] as $data_part) {
                    if ($data != '') {
                        $data .= ', ';
                    }
                    
                    $data .= $data_part;
                }
            
            // else there are not multiple data parts
            } else {
                // if there is a file name, use file name for data
                if ($field['file_name']) {
                    $data = $field['file_name'];
                    
                // else there is not a file name, so just use data
                } else {
                    $data = $field['data'];
                }
            }
            
            $submitted_form['field_' . $field['form_field_id']] = $data;

            // if there is data for this field, remember that for conditionals later
            if ($data != '') {
                $custom_fields_with_data[$field['name']] = TRUE;
            }
        }

        // get custom fields
        $query =
            "SELECT
                id,
                name,
                type,
                wysiwyg
            FROM form_fields
            WHERE page_id = '" . escape($submitted_form['custom_form_page_id']) . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $custom_fields = array();
        
        while ($row = mysqli_fetch_assoc($result)) {
            $custom_fields[] = $row;
        }

        // get all conditionals
        preg_match_all('/\[\[(.*?\^\^(.*?)\^\^.*?)(\|\|(.*?))?\]\]/si', $content, $conditionals, PREG_SET_ORDER);
        
        // loop through all conditionals
        foreach ($conditionals as $conditional) {
            $whole_string = $conditional[0];
            $positive_string = $conditional[1];
            $negative_string = $conditional[4];
            $field_name = $conditional[2];
            
            // if field name is reference code and there is another field, use the other field,
            // because we don't want to use reference code for conditional checking
            if (($field_name == 'reference_code') && (preg_match('/\[\[(.*?\^\^reference_code\^\^.*?\^\^(.*?)\^\^.*?)(\|\|(.*?))?\]\]/si', $whole_string, $conditional))) {
                $field_name = $conditional[2];
            }
            
            // assume that the field name is not valid, until we find out otherwise
            // we don't want to replace the conditional if the field name is not valid, so that a ^^name^^ conditional will be left alone if used with an e-mail campaign
            $field_name_valid = FALSE;
            
            // loop through the standard fields in order to determine if the field name is a valid standard field name
            foreach ($standard_fields as $standard_field) {
                // if this standard field value matches the field name, then the field name is valid, so remember that and break out of the loop
                if ($standard_field['value'] == $field_name) {
                    $field_name_valid = TRUE;
                    break;
                }
            }
            
            // if we don't know if the field name is valid yet, then loop through the custom fields in order to check if the field name is a valid custom field
            if ($field_name_valid == FALSE) {
                foreach ($custom_fields as $custom_field) {
                    // if this custom field name matches the field name, then the field name is valid, so remember that and break out of the loop
                    if ($custom_field['name'] == $field_name) {
                        $field_name_valid = TRUE;
                        break;
                    }
                }
            }
            
            // if the field name is valid, then replace conditional
            if ($field_name_valid == TRUE) {
                // if there is data to output, use first part of conditional
                if (($submitted_form[$field_name] != '') || ($custom_fields_with_data[$field_name] == TRUE)) {
                    $content = str_replace($whole_string, $positive_string, $content);
                    
                // else there is no data to output, so use second part of conditional
                } else {
                    $content = str_replace($whole_string, $negative_string, $content);
                }
            }
        }

        // get all variables so they can be replaced with data
        preg_match_all('/\^\^(.*?)\^\^(%%(.*?)%%)?/i', $content, $variables, PREG_SET_ORDER);

        // loop through the variables in order to replace them with data
        foreach ($variables as $variable) {
            $whole_string = $variable[0];
            $field_name = $variable[1];

            $date_format = '';

            // if a date format was passed along with the variable, then store that
            if (isset($variable[3]) == TRUE) {
                $date_format = $variable[3];

                // if prepare for HTML is enabled, then that means
                // we have to use unhtmlspecialchars() because the date format was created
                // in the rich-text editor, so there might be HTML entities (e.g. &lt;)
                // and the date() function would convert those characters into date elements
                if ($prepare_for_html == TRUE) {
                    $date_format = unhtmlspecialchars($date_format);
                }
            }

            // assume that the field name is not valid, until we find out otherwise
            $field_name_valid = FALSE;

            $field_group = '';
            $field_type = '';
            $field_id = '';
            $field_wysiwyg = '';

            // loop through the standard fields in order to determine
            // if the field name is valid and to get field info
            foreach ($standard_fields as $standard_field) {
                // if this standard field value matches the field name, then the field name is valid,
                // so remember that, store field info, and break out of the loop
                if ($standard_field['value'] == $field_name) {
                    $field_name_valid = TRUE;
                    $field_group = 'standard';
                    $field_type = $standard_field['type'];

                    break;
                }
            }

            // if we don't know if the field name is valid yet, then loop through the custom fields
            // in order to check if the field name is a valid custom field and to get field info
            if ($field_name_valid == FALSE) {
                foreach ($custom_fields as $custom_field) {
                    // if this custom field name matches the field name, then the field name is valid,
                    // so remember that, store field info, and break out of loop
                    if ($custom_field['name'] == $field_name) {
                        $field_name_valid = TRUE;
                        $field_group = 'custom';
                        $field_id = $custom_field['id'];
                        $field_type = $custom_field['type'];
                        $field_wysiwyg = $custom_field['wysiwyg'];

                        break;
                    }
                }
            }

            // if the field name is valid, then continue to replace variable with data
            if ($field_name_valid == TRUE) {
                $data = '';

                // set prepare for html value to global value, until we find it should be set to something else
                $prepare_this_field_for_html = $prepare_for_html;

                // get values differently based on the field group
                switch ($field_group) {
                    case 'standard':
                        $data = $submitted_form[$field_name];
                        break;
                    
                    case 'custom':
                        $data = $submitted_form['field_' . $field_id];

                        // if this field is a WYSIWYG field, then do not prepare for HTML
                        if ($field_wysiwyg == 1) {
                            $prepare_this_field_for_html = FALSE;
                        }

                        break;
                }

                $data = prepare_form_data_for_output($data, $field_type, $prepare_this_field_for_html, $date_format);

                // if this is a standard field, then do some extra things for standard fields
                if ($field_group == 'standard') {
                    // If this is the number of views or number of comments field then do some things for the numeric value.
                    if (
                        ($field_name == 'number_of_views')
                        || ($field_name == 'number_of_comments')
                    ) {
                        // If the value is blank, then set it to 0.
                        if ($data == '') {
                            $data = 0;

                        // Otherwise the value is not blank, so format the number,
                        // so that it has commas in the thousands place.
                        } else {
                            $data = number_format($data);
                        }
                    }
                    
                    // if this is the newest comment field and the message is greater than 100 characters, then shorten message
                    if (($field_name == 'newest_comment') && (mb_strlen($data) > 100)) {
                        $data = mb_substr($data, 0, 100) . '...';
                    }
                    
                    // if this is the newest comment name field and the value is blank and there is a newest comment, then set name to "Anonymous"
                    if (($field_name == 'newest_comment_name') && ($data == '') && ($submitted_form['newest_comment_id'] != '')) {
                        $data = 'Anonymous';
                    }
                    
                    // If this is the submitter field and badge is enabled for the submitter,
                    // or if this is the last modifier field and badge is enabled for the last modifier,
                    // or if this is the newest comment name field and badge is enabled for the newest comment submitter,
                    // then add badge
                    if (
                        (($field_name == 'submitter') && ($submitted_form['submitter_badge'] == 1) && (($submitted_form['submitter_badge_label'] != '') || (BADGE_LABEL != '')))
                        || (($field_name == 'last_modifier') && ($submitted_form['last_modifier_badge'] == 1) && (($submitted_form['last_modifier_badge_label'] != '') || (BADGE_LABEL != '')))
                        || (($field_name == 'newest_comment_name') && ($submitted_form['newest_comment_submitter_badge'] == 1) && (($submitted_form['newest_comment_submitter_badge_label'] != '') || (BADGE_LABEL != '')))
                    ) {
                        $badge_label = '';

                        // Get the user's badge label differently based on the field.
                        switch ($field_name) {
                            case 'submitter':
                                $badge_label = $submitted_form['submitter_badge_label'];
                                break;

                            case 'last_modifier':
                                $badge_label = $submitted_form['last_modifier_badge_label'];
                                break;

                            case 'newest_comment_name':
                                $badge_label = $submitted_form['newest_comment_submitter_badge_label'];
                                break;
                        }

                        // If the user's badge label is blank, then use default label.
                        if ($badge_label == '') {
                            $badge_label = BADGE_LABEL;
                        }

                        // If this field is being prepared for HTML, then add HTML version of badge.
                        if ($prepare_this_field_for_html == TRUE) {
                            $data .= ' <span class="software_badge ' . h(get_class_name($badge_label)) . '">' . h($badge_label) . '</span>';

                        // Otherwise this field is being prepared for plain text, so add plain text version of badge.
                        } else {
                            $data .= ' [' . $badge_label . ']';
                        }
                    }

                    // If this is the comment attachments field and the data is not blank,
                    // then output the comment attachments as links.
                    if (($field_name == 'comment_attachments') && ($data != '')) {
                        $comment_attachments = explode('||', $data);

                        $output_comment_attachments = '';

                        foreach ($comment_attachments as $comment_attachment) {
                            if ($output_comment_attachments != '') {
                                $output_comment_attachments .= ', ';
                            }

                            // If this field is being prepared for HTML, then output links for comment attachments.
                            if ($prepare_this_field_for_html == true) {
                                $output_comment_attachments .= '<a href="' . OUTPUT_PATH . h(encode_url_path($comment_attachment)) . '" target="_blank">' . h($comment_attachment) . '</a>';

                            // Otherwise this field is being prepared for plain text, so just output attachment name.
                            } else {
                                $output_comment_attachments .= $comment_attachment;
                            }
                        }

                        $data = $output_comment_attachments;
                    }
                }
                
                // replace the variable with the data
                // we can't use str_replace() for this because we need to limit the number of replacements to 1
                // in order to prevent bugs where it will replace variables further below that might have date formats
                $content = preg_replace('/' . preg_quote($whole_string, '/') . '/', addcslashes($data, '\\$'), $content, 1);
            }
        }
    }
    
    // return the content
    return $content;
}

// outputs checkboxes with labels for page types
function get_page_type_checkboxes_and_labels($set_page_type_values = array())
{
    $output = '';
    
    $set_page_type_email_a_friend_checked = '';
    
    // if email a friend is set to blank or if it is turned on, then check it's checkbox
    if (($set_page_type_values['set_page_type_email_a_friend'] == '') || ($set_page_type_values['set_page_type_email_a_friend'] == '1')) {
        $set_page_type_email_a_friend_checked = ' checked="checked"';
    }

    $set_page_type_folder_view_checked = '';
    
    // If folder view is set to blank or if it is turned on, then check it's checkbox.
    if (($set_page_type_values['set_page_type_folder_view'] == '') || ($set_page_type_values['set_page_type_folder_view'] == '1')) {
        $set_page_type_folder_view_checked = ' checked="checked"';
    }
    
    $set_page_type_photo_gallery_checked = '';
    
    // if photo gallery is set to blank or if it is turned on, then check it's checkbox
    if (($set_page_type_values['set_page_type_photo_gallery'] == '') || ($set_page_type_values['set_page_type_photo_gallery'] == '1')) {
        $set_page_type_photo_gallery_checked = ' checked="checked"';
    }
    
    // output basic page types
    $output .= 
        '<div style="white-space: nowrap"><img src="images/check_mark.gif" width="7" height="7" alt="check mark" title="" style="margin: 0em 0.55em;" /> Standard</div>
        <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_email_a_friend" id="set_page_type_email_a_friend" value="1" class="checkbox"' . $set_page_type_email_a_friend_checked . ' /><label for="set_page_type_email_a_friend"> Email a Friend</label></div>
        <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_folder_view" id="set_page_type_folder_view" value="1" class="checkbox"' . $set_page_type_folder_view_checked . ' /><label for="set_page_type_folder_view"> Folder View</label></div>
        <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_photo_gallery" id="set_page_type_photo_gallery" value="1" class="checkbox"' . $set_page_type_photo_gallery_checked . ' /><label for="set_page_type_photo_gallery"> Photo Gallery</label></div>';
    
    // if forms module is active, then display forms page types
    if (FORMS == true) {
        $set_page_type_custom_form_checked = '';
        
        // if custom form is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_custom_form'] == '') || ($set_page_type_values['set_page_type_custom_form'] == '1')) {
            $set_page_type_custom_form_checked = ' checked="checked"';
        }
        
        $set_page_type_custom_form_confirmation_checked = '';
        
        // if custom form confirmation is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_custom_form_confirmation'] == '') || ($set_page_type_values['set_page_type_custom_form_confirmation'] == '1')) {
            $set_page_type_custom_form_confirmation_checked = ' checked="checked"';
        }
        
        $set_page_type_form_list_view_checked = '';
        
        // if form list view is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_form_list_view'] == '') || ($set_page_type_values['set_page_type_form_list_view'] == '1')) {
            $set_page_type_form_list_view_checked = ' checked="checked"';
        }
        
        $set_page_type_form_item_view_checked = '';
        
        // if form item view is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_form_item_view'] == '') || ($set_page_type_values['set_page_type_form_item_view'] == '1')) {
            $set_page_type_form_item_view_checked = ' checked="checked"';
        }
        
        $set_page_type_form_view_directory_checked = '';
        
        // if form view directory is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_form_view_directory'] == '') || ($set_page_type_values['set_page_type_form_view_directory'] == '1')) {
            $set_page_type_form_view_directory_checked = ' checked="checked"';
        }
        
        $output .= 
            '<div style="white-space: nowrap"><input type="checkbox" name="set_page_type_custom_form" id="set_page_type_custom_form" value="1" class="checkbox"' . $set_page_type_custom_form_checked . ' /><label for="set_page_type_custom_form"> Custom Form</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_custom_form_confirmation" id="set_page_type_custom_form_confirmation" value="1" class="checkbox"' . $set_page_type_custom_form_confirmation_checked . ' /><label for="set_page_type_custom_form_confirmation"> Custom Form Confirmation</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_form_list_view" id="set_page_type_form_list_view" value="1" class="checkbox"' . $set_page_type_form_list_view_checked . ' /><label for="set_page_type_form_list_view"> Form List View</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_form_item_view" id="set_page_type_form_item_view" value="1" class="checkbox"' . $set_page_type_form_item_view_checked . ' /><label for="set_page_type_form_item_view"> Form Item View</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_form_view_directory" id="set_page_type_form_view_directory" value="1" class="checkbox"' . $set_page_type_form_view_directory_checked . ' /><label for="set_page_type_form_view_directory"> Form View Directory</label></div>';
    }
    
    // if calendars module is active and user has access to calendars, then display calendars page types
    if (CALENDARS == true) {
        $set_page_type_calendar_view_checked = '';
        
        // if calendar view is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_calendar_view'] == '') || ($set_page_type_values['set_page_type_calendar_view'] == '1')) {
            $set_page_type_calendar_view_checked = ' checked="checked"';
        }
        
        $set_page_type_calendar_event_view_checked = '';
        
        // if calendar event view is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_calendar_event_view'] == '') || ($set_page_type_values['set_page_type_calendar_event_view'] == '1')) {
            $set_page_type_calendar_event_view_checked = ' checked="checked"';
        }
        
        $output .= 
            '<div style="white-space: nowrap;' . $set_page_type_calendar_style . '"><input type="checkbox" name="set_page_type_calendar_view" id="set_page_type_calendar_view" value="1" class="checkbox"' . $set_page_type_calendar_view_checked . ' /><label for="set_page_type_calendar_view"> Calendar View</label></div>
            <div style="white-space: nowrap;' . $set_page_type_calendar_style . '"><input type="checkbox" name="set_page_type_calendar_event_view" id="set_page_type_calendar_event_view" value="1" class="checkbox"' . $set_page_type_calendar_event_view_checked . ' /><label for="set_page_type_calendar_event_view"> Calendar Event View</label></div>';
    }
    
    // if e-commerce module is active, then display e-commerce page types
    if (ECOMMERCE == true) {
        $set_page_type_catalog_checked = '';
        
        // if catalog is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_catalog'] == '') || ($set_page_type_values['set_page_type_catalog'] == '1')) {
            $set_page_type_catalog_checked = ' checked="checked"';
        }
        
        $set_page_type_catalog_detail_checked = '';
        
        // if catalog detail is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_catalog_detail'] == '') || ($set_page_type_values['set_page_type_catalog_detail'] == '1')) {
            $set_page_type_catalog_detail_checked = ' checked="checked"';
        }
        
        $set_page_type_express_order_checked = '';
        
        // if express order is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_express_order'] == '') || ($set_page_type_values['set_page_type_express_order'] == '1')) {
            $set_page_type_express_order_checked = ' checked="checked"';
        }
        
        $set_page_type_order_form_checked = '';
        
        // if order form is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_order_form'] == '') || ($set_page_type_values['set_page_type_order_form'] == '1')) {
            $set_page_type_order_form_checked = ' checked="checked"';
        }
        
        $set_page_type_shopping_cart_checked = '';
        
        // if shopping cart is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_shopping_cart'] == '') || ($set_page_type_values['set_page_type_shopping_cart'] == '1')) {
            $set_page_type_shopping_cart_checked = ' checked="checked"';
        }
        
        $set_page_type_shipping_address_and_arrival_checked = '';
        
        // if shipping address and arrival is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_shipping_address_and_arrival'] == '') || ($set_page_type_values['set_page_type_shipping_address_and_arrival'] == '1')) {
            $set_page_type_shipping_address_and_arrival_checked = ' checked="checked"';
        }
        
        $set_page_type_shipping_method_checked = '';
        
        // if shipping method is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_shipping_method'] == '') || ($set_page_type_values['set_page_type_shipping_method'] == '1')) {
            $set_page_type_shipping_method_checked = ' checked="checked"';
        }
        
        $set_page_type_billing_information_checked = '';
        
        // if billing information is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_billing_information'] == '') || ($set_page_type_values['set_page_type_billing_information'] == '1')) {
            $set_page_type_billing_information_checked = ' checked="checked"';
        }
        
        $set_page_type_order_preview_checked = '';
        
        // if order preview is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_order_preview'] == '') || ($set_page_type_values['set_page_type_order_preview'] == '1')) {
            $set_page_type_order_preview_checked = ' checked="checked"';
        }
        
        $set_page_type_order_receipt_checked = '';
        
        // if order receipt is set to blank or if it is turned on, then check it's checkbox
        if (($set_page_type_values['set_page_type_order_receipt'] == '') || ($set_page_type_values['set_page_type_order_receipt'] == '1')) {
            $set_page_type_order_receipt_checked = ' checked="checked"';
        }
        
        $output .= 
            '<div style="white-space: nowrap"><input type="checkbox" name="set_page_type_catalog" id="set_page_type_catalog" value="1" class="checkbox"' . $set_page_type_catalog_checked . ' /><label for="set_page_type_catalog"> Catalog</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_catalog_detail" id="set_page_type_catalog_detail" value="1" class="checkbox"' . $set_page_type_catalog_detail_checked . ' /><label for="set_page_type_catalog_detail"> Catalog Detail</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_express_order" id="set_page_type_express_order" value="1" class="checkbox"' . $set_page_type_express_order_checked . ' /><label for="set_page_type_express_order"> Express Order</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_order_form" id="set_page_type_order_form" value="1" class="checkbox"' . $set_page_type_order_form_checked . ' /><label for="set_page_type_order_form"> Order Form</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_shopping_cart" id="set_page_type_shopping_cart" value="1" class="checkbox"' . $set_page_type_shopping_cart_checked . ' /><label for="set_page_type_shopping_cart"> Shopping Cart</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_shipping_address_and_arrival" id="set_page_type_shipping_address_and_arrival" value="1"' . $set_page_type_shipping_address_and_arrival_checked . ' class="checkbox" /><label for="set_page_type_shipping_address_and_arrival"> Shipping Address and Arrival</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_shipping_method" id="set_page_type_shipping_method" value="1" class="checkbox"' . $set_page_type_shipping_method_checked . ' /><label for="set_page_type_shipping_method"> Shipping Method</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_billing_information" id="set_page_type_billing_information" value="1" class="checkbox"' . $set_page_type_billing_information_checked . ' /><label for="set_page_type_billing_information"> Billing Information</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_order_preview" id="set_page_type_order_preview" value="1" class="checkbox"' . $set_page_type_order_preview_checked . ' /><label for="set_page_type_order_preview"> Order Preview</label></div>
            <div style="white-space: nowrap"><input type="checkbox" name="set_page_type_order_receipt" id="set_page_type_order_receipt" value="1" class="checkbox"' . $set_page_type_order_receipt_checked . ' /><label for="set_page_type_order_receipt"> Order Receipt</label></div>';
    }

    return $output;
}

// create function that will be used to get discounted product prices based on products that are discounted by offers
function get_discounted_product_prices()
{
    $discounted_product_prices = array();
    
    $special_offer_code = '';
    
    // if the visitor has an order, then get special offer code from order in database
    if (isset($_SESSION['ecommerce']['order_id']) == TRUE) {
        $query = "SELECT special_offer_code FROM orders WHERE id = '" . $_SESSION['ecommerce']['order_id'] . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_assoc($result);
        $special_offer_code = $row['special_offer_code'];
        
    // else the visitor does not have an order, so if there is a special offer code in the visitor's session (e.g. passed via the query string), then set special offer code
    } else if (isset($_SESSION['ecommerce']['special_offer_code']) == TRUE) {
        $special_offer_code = $_SESSION['ecommerce']['special_offer_code'];
    }
        
    // if there is a special offer code for this order, then get offer code because the special offer code might just be a key code
    if ($special_offer_code != '') {
        $offer_code = get_offer_code_for_special_offer_code($special_offer_code);
    }
    
    // get all active offers that have an order scope so we can get discounted product prices
    // we can't get offers that have a recipient scope, because when we show a discounted product, we don't know which recipient it might eventually be added to
    $query =
        "SELECT
            offers.id,
            offers.code,
            offers.only_apply_best_offer
        FROM offers
        WHERE
            (offers.status = 'enabled')
            AND (offers.start_date <= CURRENT_DATE())
            AND (CURRENT_DATE() <= offers.end_date)
            AND ((offers.require_code = 0) OR (offers.code = '" . escape($offer_code) . "'))
            AND (offers.scope = 'order')
        ORDER BY offers.code ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $offers = array();

    while ($row = mysqli_fetch_assoc($result)) {
        $offers[] = $row;
    }
    
    // prepare to get best offers (including non-competing offers and competing offers that are the best offer)
    // competing offers are offers that share the same code but where only the best offer will be applied
    $best_offers = array();
    
    // create array to store the best offer id's for offer codes, so we don't have to look up the best offer multiple times
    $best_offer_ids = array();
    
    // loop through the offers in order to get best offers
    foreach ($offers as $key => $offer) {
        $previous_offer_code = '';
    
        // if this is not the first offer, then get previous offer code
        if ($key != 0) {
            $previous_offer_code = $offers[$key - 1]['code'];
        }
        
        $next_offer_code = '';
    
        // if this is not the last offer, then get next offer code
        if ($key != (count($offers) - 1)) {
            $next_offer_code = $offers[$key + 1]['code'];
        }
    
        // if this is not the first offer and the previous offer had the same code,
        // or if this is not the last offer and the next offer had the same code,
        // and if this offer is set so only the best offer is applied,
        // then this is a competing offer, so determine if offer is the best offer
        if (
            (
                ($key != 0) && ($offer['code'] == $previous_offer_code)
                || ($key != (count($offers) - 1)) && ($offer['code'] == $next_offer_code)
            )
            && ($offer['only_apply_best_offer'] == 1)
        ) {
            $best_offer_id = '';
            
            // if the best offer id has already been found for this offer code, then get it
            if (isset($best_offer_ids[$offer['code']]) == TRUE) {
                $best_offer_id = $best_offer_ids[$offer['code']];
                
            // else the best offer id has not already been found for this offer code, so get it
            // and then remember it by adding to array
            } else {
                $best_offer_id = get_best_offer_id($offer['code']);
                $best_offer_ids[$offer['code']] = $best_offer_id;
            }
            
            // if this offer is the best offer, then add it to array
            if ($offer['id'] == $best_offer_id) {
                $best_offers[] = $offer;
            }
            
        // else this is not a competing offer, so it is a best offer, so add it to array
        } else {
            $best_offers[] = $offer;
        }
    }
    
    // set active offers to all the best offers
    $offers = $best_offers;
    
    // loop through all active offers so we can get discounted product prices
    foreach ($offers as $offer) {
        // if the offer is valid, then continue to get discounted product prices for this offer
        if (validate_offer($offer['id']) == TRUE) {
            // get offer actions for this offer that discount a product
            $query =
                "SELECT
                    offer_actions.discount_product_product_id,
                    offer_actions.discount_product_amount,
                    offer_actions.discount_product_percentage,
                    discount_products.price AS discount_product_price
                FROM offers_offer_actions_xref
                LEFT JOIN offer_actions ON offers_offer_actions_xref.offer_action_id = offer_actions.id
                LEFT JOIN products AS discount_products ON offer_actions.discount_product_product_id = discount_products.id
                WHERE
                    (offers_offer_actions_xref.offer_id = '" . $offer['id'] . "')
                    AND (offer_actions.type = 'discount product')
                    AND (discount_products.id IS NOT NULL)
                    AND (discount_products.price != '0')
                    AND ((offer_actions.discount_product_amount != 0) OR (offer_actions.discount_product_percentage != 0))";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            $offer_actions = array();

            while ($row = mysqli_fetch_assoc($result)) {
                $offer_actions[] = $row;
            }
            
            // loop through the offer actions in order to apply offers to order
            foreach ($offer_actions as $offer_action) {
                $discounted_price = 0;
                
                // if discount is by amount, then get discounted price
                if ($offer_action['discount_product_amount']) {
                    $discounted_price = $offer_action['discount_product_price'] - $offer_action['discount_product_amount'];
                
                // else discount is by percentage, so get discounted price
                } else {
                    $discounted_price = $offer_action['discount_product_price'] - ($offer_action['discount_product_price'] * ($offer_action['discount_product_percentage'] / 100));
                }
                
                // if the discounted price is less than 0, then set discounted price to 0
                if ($discounted_price < 0) {
                    $discounted_price = 0;
                }
                
                // if this offer and action provides the greatest discount so far, then store discount information in array
                if (
                    (isset($discounted_product_prices[$offer_action['discount_product_product_id']]) == FALSE)
                    || ($discounted_price < $discounted_product_prices[$offer_action['discount_product_product_id']])
                ) {
                    $discounted_product_prices[$offer_action['discount_product_product_id']] = $discounted_price;
                }
            }
        }
    }
    
    return $discounted_product_prices;
}

function prepare_price_for_output($original_price, $discounted, $discounted_price, $format, $show_code = TRUE, $show_html_entity_symbol = TRUE)
{

    $original_amount = get_currency_amount($original_price / 100, VISITOR_CURRENCY_EXCHANGE_RATE);

    $original_negative = '';

    // If the amount is negative, then prepare to show negative sign before price,
    // and convert amount to positive value.
    if ($original_amount < 0) {
        $original_negative = '-';
        $original_amount = abs($original_amount);
    }

    if ($discounted) {

        $discounted_amount = get_currency_amount($discounted_price / 100, VISITOR_CURRENCY_EXCHANGE_RATE);

        $discounted_negative = '';

        // If the amount is negative, then prepare to show negative sign before price,
        // and convert amount to positive value.
        if ($discounted_amount < 0) {
            $discounted_negative = '-';
            $discounted_amount = abs($discounted_amount);
        }

    }

    // we do not show an html entity symbol for plain text order receipts, because the entity does not appear correctly in the e-mails
    // so determine if we should show the symbol or not
    $output_symbol = '';
    
    // if the symbol should be shown, then show it
    if (
        ($show_html_entity_symbol == TRUE)
        || (mb_substr(VISITOR_CURRENCY_SYMBOL, 0, 1) != '&')
    ) {
        $output_symbol = VISITOR_CURRENCY_SYMBOL;
    }

    $output_code = '';
    
    // if the code should be shown, then prepare to show code
    if ($show_code == TRUE) {
        $output_code = h(VISITOR_CURRENCY_CODE_FOR_OUTPUT);
    }
    
    // prepare to output product price differently based on the format
    switch ($format) {
        // if the format is HTML then prepare to show price with detailed styling (e.g. strike-through)
        case 'html':
            // if the product is discounted by an offer, then prepare to show original price and discounted price
            if ($discounted == TRUE) {
                return '<span style="text-decoration: line-through; white-space: nowrap">' . $original_negative . $output_symbol . number_format($original_amount, 2, '.', ',') . $output_code . '</span> <span style="white-space: nowrap" class="software_discounted_price">' . $discounted_negative . $output_symbol . number_format($discounted_amount, 2, '.', ',') . $output_code . '</span>';
                
            // else the product is not discounted by an offer, so prepare to just show the product price
            } else {
                return '<span style="white-space: nowrap">' . $original_negative . $output_symbol . number_format($original_amount, 2, '.', ',') . $output_code . '</span>';
            }
            
            break;
            
        // if the format is plain text then prepare to show price with no styling
        case 'plain_text':
            // if the product is discounted by an offer, then prepare to show original price and discounted price
            if ($discounted == TRUE) {
                return $discounted_negative . $output_symbol . number_format($discounted_amount, 2, '.', ',') . $output_code . ' (was ' . $original_negative . $output_symbol . number_format($original_amount, 2, '.', ',') . $output_code . ')';
                
            // else the product is not discounted by an offer, so prepare to just show the product price
            } else {
                return $original_negative . $output_symbol . number_format($original_amount, 2, '.', ',') . $output_code;
            }
            
            break;
    }
}

// Takes an amount in dollars (not cents) for base currency, and returns negative symbol,
// if necessary, base currency symbol, and the amount (e.g. -$10.00).  This is used
// by various screens where currency conversion is not necessary and
// the amount is never discounted.  The purpose of this function is so that
// the negative symbol will appear before the currency symbol instead of after
// the currency symbol, which would look weird.

function prepare_amount($amount) {

    // Remove any commas that might exist in amount, because number_format()
    // below won't work correctly when commas exist.  number_format() below
    // will add the commas back properly.
    $amount = str_replace(',', '', $amount);

    $negative_symbol = '';

    // If the amount is negative, then prepare to show negative sign before amount,
    // and convert amount to positive value.
    if ($amount < 0) {
        $negative_symbol = '-';
        $amount = abs($amount);
    }

    return $negative_symbol . BASE_CURRENCY_SYMBOL . number_format($amount, 2);
}

// Create function that will allow us to get a time, like "5 days ago", from a timestamp.
// Properties:
// timestamp: The Unix timestamp for the time that you want to compare to now.
// format: "html" (default) or "plain_text"
// type: "date_and_time" (default) or "date".  This property is used in order to indicate
// if the timestamp that was passed was for a general date or for a specific date & time.
// This information is used in order to determine whether a date & time or just a date is
// outputted for the absolute time tooltip.
// timezone_type: "user" (default) or "site".  This property allows you output a time
// for the user's timezone (if user has one set) or the site's timezone.
function get_relative_time($properties)
{
    $timestamp = $properties['timestamp'];
    $format = $properties['format'];
    $type = $properties['type'];

    // If the timestamp is blank, then set it to 0.  We have to do this in order
    // to avoid PHP exceptions when we call new DateTime('@' . $timestamp) below.
    // This can happen when timestamp has a null value in the db for some reason.
    if ($timestamp == '') {
        $timestamp = 0;
    }

    // If the type is "date_and_time" and if the timezone type is not "site",
    // and the visitor is logged in, and the user has a specific timezone set,
    // then get time in user's timezone.
    if (
        ($type != 'date')
        && ($properties['timezone_type'] != 'site')
        && (USER_LOGGED_IN)
        && (USER_TIMEZONE != '')
    ) {
        $timezone_type = 'user';

    // Otherwise get the time for the site's timezone.
    } else {
        $timezone_type = 'site';
    }

    $current_timestamp = time();

    // if the timestamp is in the past or right now, then prepare values for the past
    if ($current_timestamp >= $timestamp) {
        $difference_in_seconds = $current_timestamp - $timestamp;
        $past_or_future_message = 'ago';

    // else the timestamp is in the future, so prepare values for the future
    } else {
        $difference_in_seconds = $timestamp - $current_timestamp;
        $past_or_future_message = 'from now';
    }
        
    // if the difference is less than 1 minute, then return "a moment ago/from now"
    if ($difference_in_seconds < 60) {
        $relative_time = 'A moment ' . $past_or_future_message;
        
    // else if the difference is less than 1 hour, then return minutes
    } else if ($difference_in_seconds < 3600) {
        $difference_in_minutes = round($difference_in_seconds / 60);
        
        $plural_suffix = '';
        
        // if the difference is more than 1 minute, then prepare plural suffix
        if ($difference_in_minutes > 1) {
            $plural_suffix = 's';
        }
        
        $relative_time = $difference_in_minutes . ' minute' . $plural_suffix . ' ' . $past_or_future_message;
        
    // else if the difference is less than 1 day, then return hours
    } else if ($difference_in_seconds < 86400) {
        $difference_in_hours = round($difference_in_seconds / 60 / 60);
        
        $plural_suffix = '';
        
        // if the difference is more than 1 hour, then prepare plural suffix
        if ($difference_in_hours > 1) {
            $plural_suffix = 's';
        }
        
        $relative_time = $difference_in_hours . ' hour' . $plural_suffix . ' ' . $past_or_future_message;
        
    // else if the difference is less than 1 month, then return days.
    } else if ($difference_in_seconds < 2592000) {
        // Please note that we calculate the difference in calendar days below,
        // based on the current date for the appropriate time zone.
        // We no longer calculate the difference in days by considering a day to be 24 hours.
        // For example, if the logged time is Mon @ 11:59 PM and the current time is Wed @ 12:00 AM,
        // that is considered 2 days, even though it technically about 1 day of time.

        // If we should use site's timezone, then do that.
        if ($timezone_type == 'site') {
            $current_date = date('Y-m-d', $current_timestamp);
            $date = date('Y-m-d', $timestamp);

            $current_date_timestamp = strtotime($current_date);
            $date_timestamp = strtotime($date);

        // Otherwise we should use user's timezone, so do that.
        } else {
            $current_date = new DateTime('@' . $current_timestamp);
            $current_date->setTimezone(new DateTimeZone(USER_TIMEZONE));

            $date = new DateTime('@' . $timestamp);
            $date->setTimezone(new DateTimeZone(USER_TIMEZONE));

            $current_date_timestamp = strtotime($current_date->format('Y-m-d'));
            $date_timestamp = strtotime($date->format('Y-m-d'));
        }

        $difference_in_days = round(abs($current_date_timestamp - $date_timestamp) / 86400);
        
        $plural_suffix = '';
        
        // if the difference is more than 1 day, then prepare plural suffix
        if ($difference_in_days > 1) {
            $plural_suffix = 's';
        }
        
        $relative_time = $difference_in_days . ' day' . $plural_suffix . ' ' . $past_or_future_message;
        
    // else the difference is at least 1 month, so return the actual date.
    } else {
        // If the date format is month and then day, then use that format.
        if (DATE_FORMAT == 'month_day') {
            $month_and_day_format = 'M j';

        // Otherwise the date format is day and then month, so use that format.
        } else {
            $month_and_day_format = 'j M';
        }

        // If we should output date for site's timezone, then do that.
        if ($timezone_type == 'site') {
            $relative_time = date($month_and_day_format . ' Y', $timestamp);

        // Otherwise we should output date for user's timezone, so do that.
        } else {
            $date = new DateTime('@' . $timestamp);
            $date->setTimezone(new DateTimeZone(USER_TIMEZONE));
            $relative_time = $date->format($month_and_day_format . ' Y');
        }
    }

    // If the format is plain text, then just return the relative time without a time tag around it.
    if ($format == 'plain_text') {
        return $relative_time;

    // Otherwise the format is html, so add time tag with tooltip for absolute time around relative time.
    } else {
        // If the date format is month and then day, then use that format.
        if (DATE_FORMAT == 'month_day') {
            $month_and_day_format = 'M j';

        // Otherwise the date format is day and then month, so use that format.
        } else {
            $month_and_day_format = 'j M';
        }
        
        // If the type is just date, then prepare datetime and title attributes with just the date.
        if ($type == 'date') {
            $datetime = date('Y-m-d', $timestamp);
            $title = date('D ' . $month_and_day_format . ' Y', $timestamp);

        // Otherwise the type is date & time, so prepare datetime and title attributes with both the date and time.
        } else {
            $datetime = gmdate('Y-m-d\TH:i:s\Z', $timestamp);

            // If we should output time for site's timezone, then do that.
            if ($timezone_type == 'site') {
                $title = date('D ' . $month_and_day_format . ' Y g:i A T', $timestamp);

            // Otherwise we should output time for user's timezone, so do that.
            } else {
                $date = new DateTime('@' . $timestamp);
                $date->setTimezone(new DateTimeZone(USER_TIMEZONE));
                $title = $date->format('D ' . $month_and_day_format . ' Y g:i A T');
            }
        }

        return '<time datetime="' . $datetime . '" title="' . $title . '">' . $relative_time . '</time>';
    }
}

// Create function that will allow us to get an absolute time (e.g. Mon Feb 3 2020 12:00 PM CST).
// Properties:
// timestamp: The Unix timestamp for the absolute time.
// format: "html" (default) or "plain_text".
// type: "date_and_time" (default) or "date".  This property is used in order to indicate
// if the timestamp that was passed was for a general date or for a specific date & time.
// This information is used in order to determine whether a date & time or just a date is outputted.
// size: "short" (default) or "long".
// timezone_type: "user" (default) or "site".  This property allows you output a time
// for the user's timezone (if user has one set) or the site's timezone.
// year: true (default) or false. Include year or not.  There are some types of dates, like
// estimated shipping delivery date, where the year is obvious and unnecessary.

function get_absolute_time($properties)
{
    $timestamp = $properties['timestamp'];
    $format = $properties['format'];
    $type = $properties['type'];
    $size = $properties['size'];

    if (isset($properties['year'])) {
        $year = $properties['year'];
    } else {
        $year = true;
    }

    // If the timestamp is blank, then set it to 0.  We have to do this in order
    // to avoid PHP exceptions when we call new DateTime('@' . $timestamp) below.
    // This can happen when timestamp has a null value in the db for some reason.
    if ($timestamp == '') {
        $timestamp = 0;
    }

    // If the type is just date, then just get a date.
    if ($type == 'date') {
        if ($size == 'long') {
            // If the date format is month and then day, then use that format.
            if (DATE_FORMAT == 'month_day') {
                $month_and_day_format = 'F j';

            // Otherwise the date format is day and then month, so use that format.
            } else {
                $month_and_day_format = 'j F';
            }

            $year_format = '';

            if ($year) {
                $year_format = ', Y';
            }

            $absolute_time = date('l, ' . $month_and_day_format . $year_format, $timestamp);

        } else {
            // If the date format is month and then day, then use that format.
            if (DATE_FORMAT == 'month_day') {
                $month_and_day_format = 'M j';

            // Otherwise the date format is day and then month, so use that format.
            } else {
                $month_and_day_format = 'j M';
            }

            $year_format = '';

            if ($year) {
                $year_format = ' Y';
            }

            $absolute_time = date('D ' . $month_and_day_format . $year_format, $timestamp);
        }

    // Otherwise the type is date & time, so get both of those.
    } else {
        // If the timezone type is not "site", and the visitor is logged in,
        // and the user has a specific timezone set,
        // then get time in user's timezone.
        if (
            ($properties['timezone_type'] != 'site')
            && (USER_LOGGED_IN)
            && (USER_TIMEZONE != '')
        ) {
            $timezone_type = 'user';

        // Otherwise get the time for the site's timezone.
        } else {
            $timezone_type = 'site';
        }

        if ($size == 'long') {

            // If the date format is month and then day, then use that format.
            if (DATE_FORMAT == 'month_day') {
                $month_and_day_format = 'F j';

            // Otherwise the date format is day and then month, so use that format.
            } else {
                $month_and_day_format = 'j F';
            }

            $year_format = '';

            if ($year) {
                $year_format = ', Y';
            }

            // If we should output time for site's timezone, then do that.
            if ($timezone_type == 'site') {
                $absolute_time = date('l, ' . $month_and_day_format . $year_format . ' \a\t g:i A T', $timestamp);

            // Otherwise we should output time for user's timezone, so do that.
            } else {
                $date = new DateTime('@' . $timestamp);
                $date->setTimezone(new DateTimeZone(USER_TIMEZONE));
                $absolute_time = $date->format('l, ' . $month_and_day_format . $year_format . ' \a\t g:i A T');
            }

        } else {

            // If the date format is month and then day, then use that format.
            if (DATE_FORMAT == 'month_day') {
                $month_and_day_format = 'M j';

            // Otherwise the date format is day and then month, so use that format.
            } else {
                $month_and_day_format = 'j M';
            }

            $year_format = '';

            if ($year) {
                $year_format = ' Y';
            }

            // If we should output time for site's timezone, then do that.
            if ($timezone_type == 'site') {
                $absolute_time = date('D ' . $month_and_day_format . $year_format . ' g:i A T', $timestamp);

            // Otherwise we should output time for user's timezone, so do that.
            } else {
                $date = new DateTime('@' . $timestamp);
                $date->setTimezone(new DateTimeZone(USER_TIMEZONE));
                $absolute_time = $date->format('D ' . $month_and_day_format . $year_format . ' g:i A T');
            }
        }
    }

    // If the format is plain text, then just return the absolute time without a time tag around it.
    if ($format == 'plain_text') {
        return $absolute_time;

    // Otherwise the format is html, so add time tag around absolute time.
    } else {
        // If the type is just date, then prepare datetime attribute with just the date.
        if ($type == 'date') {
            $datetime = date('Y-m-d', $timestamp);

        // Otherwise the type is date & time, so prepare datetime attribute with both the date and time.
        } else {
            $datetime = gmdate('Y-m-d\TH:i:s\Z', $timestamp);
        }

        return '<time datetime="' . $datetime . '">' . $absolute_time . '</time>';
    }
}

// this function searches the content for links, then disables them and returns the new content
function disable_links_in_content($content)
{
    // if there are links in the content, then loop through them all and disable their links
    if (preg_match_all('/<a.*?href=["|\'\'](.*?)["|\'\'].*?>.*?<\/a>/', $content, $matches) > 0) {
        foreach($matches[1] as $match) {
            // if the link is not an ad region link, then disable the link
            if (preg_match('/#software_ad_.*?/i', $match) == 0) {
                $content = preg_replace('/<a(.*?)href=["|\'\']' . escape_regex($match) . '["|\'\'](.*?)>(.*?)<\/a>/si', '<a$1href="javascript:void(0)"$2>$3</a>', $content, 1);
            }
        }
    }
    
    return $content;
}

// Create a function that allows us to get a valid class name for CSS
// from some general content.  For example, if a style name is "home page!",
// then this function will convert the name into "home_page".
function get_class_name($content)
{
    $class_name = $content;

    // convert spaces to underscores so that CSS does not treat it as separate classes
    $class_name = str_replace(' ', '_', $class_name);

    // remove any characters that are not valid for a class name
    $class_name = preg_replace('/[^A-Za-z0-9-_]/', '', $class_name);
    
    return $class_name;
}

// initialize function that can generate HTML for a system style based on the layout and cells that were submitted which are in the areas array
function get_system_style_code($layout, $areas, $style_name, $empty_cell_width_percentage, $additional_body_classes)
{
    $output_viewport_meta_tag = '';

    // if the layout is one column mobile, then add viewport meta tag,
    // so that mobile devices will know that a page has been optimized for mobile
    if ($layout == 'one_column_mobile') {
        $output_viewport_meta_tag = '<meta id="viewport" name="viewport" content="width=device-width, initial-scale=1.0" />';
    }

    // prepare HTML header
    $code =
        '<!DOCTYPE html>
        <html lang="en">
            <head>
                <meta charset="utf-8">
                <title></title>
                ' . $output_viewport_meta_tag . '
                <meta_tags></meta_tags>
                <stylesheet></stylesheet>
            </head>' . "\n";

    $output_additional_body_classes = '';

    // If there is an additional body class, then prepare value with space before it so it can be outputted.
    if ($additional_body_classes != '') {
        $output_additional_body_classes = ' ' . h($additional_body_classes);
    }
    
    // prepare body differently based on the layout
    switch ($layout) {
        case 'one_column':
            $code .=
                '            <body class="one_column ' . get_class_name($style_name) . $output_additional_body_classes . '">
                    <div id="site_border">
                        <div id="site_top">' . get_system_style_area_code($areas['site_top']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="site_header">' . get_system_style_area_code($areas['site_header']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="area_border">
                            <div id="area_header">' . get_system_style_area_code($areas['area_header']['rows'], $empty_cell_width_percentage) . '</div>
                            <div id="page_wrapper">
                                <div id="page_border">
                                    <div id="page_header">' . get_system_style_area_code($areas['page_header']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_content">' . get_system_style_area_code($areas['page_content']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_footer">' . get_system_style_area_code($areas['page_footer']['rows'], $empty_cell_width_percentage) . '</div>
                                </div>
                                <div id="area_footer">' . get_system_style_area_code($areas['area_footer']['rows'], $empty_cell_width_percentage) . '</div>
                            </div>
                        </div>
                        <div id="site_footer_border">
                            <div id="site_footer">' . get_system_style_area_code($areas['site_footer']['rows'], $empty_cell_width_percentage) . '</div>
                        </div>
                    </div>
                </body>' . "\n";
            
            break;
            
        case 'one_column_email':
            $code .=
                '            <body class="one_column_email ' . get_class_name($style_name) . $output_additional_body_classes . '">
                    <div id="email_border">
                        <div id="site_top">' . get_system_style_area_code($areas['site_top']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="site_header">' . get_system_style_area_code($areas['site_header']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="area_border">
                            <div id="area_header">' . get_system_style_area_code($areas['area_header']['rows'], $empty_cell_width_percentage) . '</div>
                            <div id="page_wrapper">
                                <div id="page_border">
                                    <div id="page_header">' . get_system_style_area_code($areas['page_header']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_content">' . get_system_style_area_code($areas['page_content']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_footer">' . get_system_style_area_code($areas['page_footer']['rows'], $empty_cell_width_percentage) . '</div>
                                </div>
                                <div id="area_footer">' . get_system_style_area_code($areas['area_footer']['rows'], $empty_cell_width_percentage) . '</div>
                            </div>
                        </div>
                        <div id="site_footer_border">
                            <div id="site_footer">' . get_system_style_area_code($areas['site_footer']['rows'], $empty_cell_width_percentage) . '</div>
                        </div>
                    </div>
                </body>' . "\n";
            
            break;

        case 'one_column_mobile':
            $code .=
                '            <body class="one_column_mobile ' . get_class_name($style_name) . $output_additional_body_classes . '">
                    <div id="mobile_border">
                        <div id="site_top">' . get_system_style_area_code($areas['site_top']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="site_header">' . get_system_style_area_code($areas['site_header']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="area_border">
                            <div id="area_header">' . get_system_style_area_code($areas['area_header']['rows'], $empty_cell_width_percentage) . '</div>
                            <div id="page_wrapper">
                                <div id="page_border">
                                    <div id="page_header">' . get_system_style_area_code($areas['page_header']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_content">' . get_system_style_area_code($areas['page_content']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_footer">' . get_system_style_area_code($areas['page_footer']['rows'], $empty_cell_width_percentage) . '</div>
                                </div>
                                <div id="area_footer">' . get_system_style_area_code($areas['area_footer']['rows'], $empty_cell_width_percentage) . '</div>
                            </div>
                        </div>
                        <div id="site_footer_border">
                            <div id="site_footer">' . get_system_style_area_code($areas['site_footer']['rows'], $empty_cell_width_percentage) . '</div>
                        </div>
                    </div>
                </body>' . "\n";
            
            break;


        case 'two_column_sidebar_left':
            $code .=
                '            <body class="two_column_sidebar_left ' . get_class_name($style_name) . $output_additional_body_classes . '">
                    <div id="site_border">
                        <div id="site_top">' . get_system_style_area_code($areas['site_top']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="site_header">' . get_system_style_area_code($areas['site_header']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="area_border">
                            <div id="area_header">' . get_system_style_area_code($areas['area_header']['rows'], $empty_cell_width_percentage) . '</div>
                            <div id="page_wrapper">
                                <div id="page_border">
                                    <div id="page_header">' . get_system_style_area_code($areas['page_header']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_content" style="float: right; width: 69%">' . get_system_style_area_code($areas['page_content']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="sidebar" style="float: left; width: 30%">' . get_system_style_area_code($areas['sidebar']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div style="clear: both"></div>
                                    <div id="page_footer">' . get_system_style_area_code($areas['page_footer']['rows'], $empty_cell_width_percentage) . '</div>
                                </div>
                                <div id="area_footer">' . get_system_style_area_code($areas['area_footer']['rows'], $empty_cell_width_percentage) . '</div>
                            </div>
                        </div>
                        <div id="site_footer_border">
                            <div id="site_footer">' . get_system_style_area_code($areas['site_footer']['rows'], $empty_cell_width_percentage) . '</div>
                        </div>
                    </div>
                </body>' . "\n";
            
            break;
        
        case 'two_column_sidebar_right':
            $code .=
                '            <body class="two_column_sidebar_right ' . get_class_name($style_name) . $output_additional_body_classes . '">
                    <div id="site_border">
                        <div id="site_top">' . get_system_style_area_code($areas['site_top']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="site_header">' . get_system_style_area_code($areas['site_header']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="area_border">
                            <div id="area_header">' . get_system_style_area_code($areas['area_header']['rows'], $empty_cell_width_percentage) . '</div>
                            <div id="page_wrapper">
                                <div id="page_border">
                                    <div id="page_header">' . get_system_style_area_code($areas['page_header']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_content" style="float: left; width: 69%">' . get_system_style_area_code($areas['page_content']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="sidebar" style="float: right; width: 30%">' . get_system_style_area_code($areas['sidebar']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div style="clear: both"></div>
                                    <div id="page_footer">' . get_system_style_area_code($areas['page_footer']['rows'], $empty_cell_width_percentage) . '</div>
                                </div>
                                <div id="area_footer">' . get_system_style_area_code($areas['area_footer']['rows'], $empty_cell_width_percentage) . '</div>
                            </div>
                        </div>
                        <div id="site_footer_border">
                            <div id="site_footer">' . get_system_style_area_code($areas['site_footer']['rows'], $empty_cell_width_percentage) . '</div>
                        </div>
                    </div>
                </body>' . "\n";
            
            break;
            
        case 'three_column_sidebar_left':
            $code .=
                '            <body class="three_column_sidebar_left ' . get_class_name($style_name) . $output_additional_body_classes . '">
                    <div id="site_border">
                        <div id="site_top">' . get_system_style_area_code($areas['site_top']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="site_header">' . get_system_style_area_code($areas['site_header']['rows'], $empty_cell_width_percentage) . '</div>
                        <div id="area_border">
                            <div id="area_header">' . get_system_style_area_code($areas['area_header']['rows'], $empty_cell_width_percentage) . '</div>
                            <div id="page_wrapper">
                                <div id="page_border">
                                    <div id="page_header">' . get_system_style_area_code($areas['page_header']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="sidebar" style="float: left; width: 22%">' . get_system_style_area_code($areas['sidebar']['rows'], $empty_cell_width_percentage) . '</div>
                                    <div id="page_content" style="float: right; width: 70%">
                                        <div id="page_content_left" style="float: left; width: 74%">' . get_system_style_area_code($areas['page_content_left']['rows'], $empty_cell_width_percentage) . '</div>
                                        <div id="page_content_right" style="float: right; width: 23%">' . get_system_style_area_code($areas['page_content_right']['rows'], $empty_cell_width_percentage) . '</div>
                                        <div style="clear: both"></div>
                                    </div>
                                    <div style="clear: both"></div>
                                    <div id="page_footer">' . get_system_style_area_code($areas['page_footer']['rows'], $empty_cell_width_percentage) . '</div>
                                </div>
                                <div id="area_footer">' . get_system_style_area_code($areas['area_footer']['rows'], $empty_cell_width_percentage) . '</div>
                            </div>
                        </div>
                        <div id="site_footer_border">
                            <div id="site_footer">' . get_system_style_area_code($areas['site_footer']['rows'], $empty_cell_width_percentage) . '</div>
                        </div>
                    </div>
                </body>' . "\n";
            
            break;
    }
    
    // prepare HTML footer
    $code .= '        </html>';
    
    return $code;
}

function get_system_style_area_code($rows, $empty_cell_width_percentage)
{
    $code = '';
    
    // if the empty cell width percentage is blank, then set it to 0
    if ($empty_cell_width_percentage == '') {
        $empty_cell_width_percentage = 0;
    }
    
    // loop through the rows in order to prepare code for each row
    foreach ($rows as $row_key => $row) {

        // Add wrapper divs so banded designs with Advanced Styling is possible within the Theme Designer
        $rk = $row_key + 1;
        $code .= '<div class="rband r' . $rk . ' clr"><div class="rwrap r' . $rk . '">';

        // loop through the cells in this row in order to prepare code for each one
        foreach ($row['cells'] as $cell_key => $cell) {
            $output_row_number = $row_key + 1;
            $output_column_number = $cell_key + 1;
            $output_style = '';
            
            // if there is more than 1 cell in this row, ONLY then prepare styling for this row
            if (count($row['cells']) > 1) {

                // if this is not the second cell or there is more than 2 cells in this row, then the float is left
                if (($cell_key != 1) || (count($row['cells']) > 2)) {
                    $output_style = ' style="float: left;';
                // else this is the second cell and there is not more than 2 cells in this row
                } else {
                    $output_style = ' style="float: right;';
                }
                
                // determine width and complete $output_style     
                                
                if ($empty_cell_width_percentage != 0) { // page style contains default empty cell width
                    $empty_cell_counter = 0;
                    // go through all cells in the current row to get the total number of empty cells 
                    foreach ($row['cells'] as $cell2_key => $cell2) {
                        if ($cell2['region_type'] == '') {
                         $empty_cell_counter++;
                        }
                    }               
                
                    // determine cell width based on whether it is empty
                    if ($cell['region_type'] != '') {
                        $output_width = ((100 - ($empty_cell_counter * $empty_cell_width_percentage)) / (count($row['cells']) - $empty_cell_counter)) - .6;
                    } else { // empty so fix the width
                        $output_width = $empty_cell_width_percentage - .7; // round down for v9 responsive cells
                    }
                } else {
                    // no default empty cell width was found in page style so set all cells to the same size
                    $output_width = (100 / count($row['cells'])) - .6;
                }
             
                // round down the final cell width so the cell doesn't wrap unintentionally within narrow areas
                $output_style .= ' width: ' . round($output_width, 2) . '%; padding: .1em;"';    
                  
            }
            
            $region_class = '';
                
            // prepare the region tag differently based on the region type
            switch ($cell['region_type']) {
                case 'ad':
                    $output_region = '<ad>' . $cell['region_name'] . '</ad>';
                    $region_class = ' ad_' . str_replace(' ', '_', $cell['region_name']);
                    break;
                
                case 'cart':
                    $output_region = '<cart></cart>';
                    $region_class = ' cart';
                    break;
                    
                case 'common':
                    $output_region = '<cregion>' . $cell['region_name'] . '</cregion>';
                    $region_class = ' cregion_' . str_replace(' ', '_', $cell['region_name']);
                    break;
                    
                case 'designer':
                    $output_region = '<cregion>' . $cell['region_name'] . '</cregion>';
                    $region_class = ' cregion_' . str_replace(' ', '_', $cell['region_name']);
                    break;
                
                case 'dynamic':
                    $output_region = '<dregion>' . $cell['region_name'] . '</dregion>';
                    $region_class = ' dregion_' . str_replace(' ', '_', $cell['region_name']);
                    break;
                
                case 'login':
                    $output_region = '<login>' . $cell['region_name'] . '</login>';
                    $region_class = ' login_' . str_replace(' ', '_', $cell['region_name']);
                    break;
                
                case 'menu':
                    $output_region = '<menu>' . $cell['region_name'] . '</menu><div style="clear: both"></div>';
                    $region_class = ' menu_' . str_replace(' ', '_', $cell['region_name']);
                    break;
                
                case 'menu_sequence':
                    $output_region = '<menu_sequence>' . $cell['region_name'] . '</menu_sequence>';
                    $region_class = ' menu_sequence_' . str_replace(' ', '_', $cell['region_name']);
                    break;

                case 'mobile_switch':
                    $output_region = '<mobile_switch></mobile_switch>';
                    $region_class = ' mobile_switch';
                    break;
                
                case 'page':
                    $output_region = '<pregion></pregion>';
                    $region_class = ' pregion';
                    break;
                    
                case 'pdf':
                    $output_region = '<pdf></pdf>';
                    $region_class = ' pdf';
                    break;
                
                case 'system':
                    // if there is a region name, then output it in the system tag
                    if ($cell['region_name'] != '') {
                        $output_region = '<system>' . $cell['region_name'] . '</system>';
                        $region_class = ' system_' . str_replace(' ', '_', $cell['region_name']);
                    
                    // else output the tag by itself
                    } else {
                        $output_region = '<system></system>';
                        $region_class = ' system';
                    }
                    break;
                
                case 'tag_cloud':
                    $output_region = '<tcloud>' . $cell['region_name'] . '</tcloud>';
                    $region_class = ' tcloud_' . str_replace(' ', '_', $cell['region_name']);
                    break;
                    
                default:
                    $output_region = '';
                    break;
            }
            
            $code .= '<div class="r' . $output_row_number . 'c' . $output_column_number . h($region_class) . ' c' . $output_column_number . '"' . $output_style . '>' . $output_region . '</div>';
        }
        
        // closed additional wrapper divs
        $code .= '</div></div>';
    }
    
    // If there are any content areas found, wrap them all in a div so Advanced Styling is possible within the Theme Designer
    if ($code != '') {
        return '<div class="wrap">' . $code . '</div>';
    } else {
        return $code;
    }
}

// initialize function for getting HTML for a layout for the style designer screen
// this function is used by the create/edit system style screens
function get_style_designer_content($layout)
{
    // prepare HTML for areas differently based on the selected layout
    switch ($layout) {
        case 'one_column':
            $output_areas =
                '<fieldset>
                    <legend>Site Border<span class="theme_fold_css" style="padding:0"><br />#site_border</span></legend>
                    ' . get_style_designer_area_content('site_top') . '
                    ' . get_style_designer_area_content('site_header') . '
                    <fieldset>
                        <legend>Area Border<span class="theme_fold_css" style="padding:0"><br />#area_border</span></legend>
                        ' . get_style_designer_area_content('area_header') . '
                        <fieldset>
                            <legend>Page Wrapper<span class="theme_fold_css" style="padding:0"><br />#page_wrapper</span></legend>
                            <fieldset>
                                <legend>Page Border<span class="theme_fold_css" style="padding:0"><br />#page_border</span></legend>
                                ' . get_style_designer_area_content('page_header') . '
                                ' . get_style_designer_area_content('page_content') . '
                                ' . get_style_designer_area_content('page_footer') . '
                            </fieldset>
                            ' . get_style_designer_area_content('area_footer') . '
                        </fieldset>
                    </fieldset>
                    <fieldset>
                        <legend>Site Footer Border<span class="theme_fold_css" style="padding:0"><br />#site_footer_border</span></legend>
                        ' . get_style_designer_area_content('site_footer') . '
                    </fieldset>
                </fieldset>';
            
            break;
            
        case 'one_column_email':
            $output_areas =
                '<fieldset>
                    <legend>E-mail Border<span class="theme_fold_css" style="padding:0"><br />#email_border</span></legend>
                    ' . get_style_designer_area_content('site_top') . '
                    ' . get_style_designer_area_content('site_header') . '
                    <fieldset>
                        <legend>Area Border<span class="theme_fold_css" style="padding:0"><br />#area_border</span></legend>
                        ' . get_style_designer_area_content('area_header') . '
                        <fieldset>
                            <legend>Page Wrapper<span class="theme_fold_css" style="padding:0"><br />#page_wrapper</span></legend>
                            <fieldset>
                                <legend>Page Border<span class="theme_fold_css" style="padding:0"><br />#page_border</span></legend>
                                ' . get_style_designer_area_content('page_header') . '
                                ' . get_style_designer_area_content('page_content') . '
                                ' . get_style_designer_area_content('page_footer') . '
                            </fieldset>
                            ' . get_style_designer_area_content('area_footer') . '
                        </fieldset>
                    </fieldset>
                    <fieldset>
                        <legend>Site Footer Border<span class="theme_fold_css" style="padding:0"><br />#site_footer_border</span></legend>
                        ' . get_style_designer_area_content('site_footer') . '
                    </fieldset>
                </fieldset>';
            
            break;

        case 'one_column_mobile':
            $output_areas =
                '<fieldset>
                    <legend><span style="border: 1px solid #99FF33;padding: 0.25em 0.75em 1em 0.75em;-moz-border-radius-topleft: 5px;-webkit-border-top-left-radius: 5px;border-top-left-radius: 5px;-moz-border-radius-topright: 5px;-webkit-border-top-right-radius: 5px;border-top-right-radius: 5px;-moz-border-radius-bottomleft: 5px;-webkit-border-bottom-left-radius: 5px;border-bottom-left-radius: 5px;-moz-border-radius-bottomright: 5px;-webkit-border-bottom-right-radius: 5px;border-bottom-right-radius: 5px;">Mobile Border</span><span class="theme_fold_css" style="padding: 0"><br />&nbsp;&nbsp;#mobile_border</span></legend>
                    ' . get_style_designer_area_content('site_top') . '
                    ' . get_style_designer_area_content('site_header') . '
                    <fieldset>
                        <legend>Area Border<span class="theme_fold_css" style="padding:0"><br />#area_border</span></legend>
                        ' . get_style_designer_area_content('area_header') . '
                        <fieldset>
                            <legend>Page Wrapper<span class="theme_fold_css" style="padding:0"><br />#page_wrapper</span></legend>
                            <fieldset>
                                <legend>Page Border<span class="theme_fold_css" style="padding:0"><br />#page_border</span></legend>
                                ' . get_style_designer_area_content('page_header') . '
                                ' . get_style_designer_area_content('page_content') . '
                                ' . get_style_designer_area_content('page_footer') . '
                            </fieldset>
                            ' . get_style_designer_area_content('area_footer') . '
                        </fieldset>
                    </fieldset>
                    <fieldset>
                        <legend>Site Footer Border<span class="theme_fold_css" style="padding:0"><br />#site_footer_border</span></legend>
                        ' . get_style_designer_area_content('site_footer') . '
                    </fieldset>
                </fieldset>';
            
            break;
            
        case 'two_column_sidebar_left':
            $output_areas =
                '<fieldset>
                    <legend>Site Border<span class="theme_fold_css" style="padding:0"><br />#site_border</span></legend>
                    ' . get_style_designer_area_content('site_top') . '
                    ' . get_style_designer_area_content('site_header') . '
                    <fieldset>
                        <legend>Area Border<span class="theme_fold_css" style="padding:0"><br />#area_border</span></legend>
                        ' . get_style_designer_area_content('area_header') . '
                        <fieldset>
                            <legend>Page Wrapper<span class="theme_fold_css" style="padding:0"><br />#page_wrapper</span></legend>
                            <fieldset>
                                <legend>Page Border<span class="theme_fold_css" style="padding:0"><br />#page_border</span></legend>
                                ' . get_style_designer_area_content('page_header') . '
                                ' . get_style_designer_area_content('page_content') . '
                                ' . get_style_designer_area_content('sidebar') . '
                                <div class="clear"></div>
                                ' . get_style_designer_area_content('page_footer') . '
                            </fieldset>
                            ' . get_style_designer_area_content('area_footer') . '
                        </fieldset>
                    </fieldset>
                    <fieldset>
                        <legend>Site Footer Border<span class="theme_fold_css" style="padding:0"><br />#site_footer_border</span></legend>
                        ' . get_style_designer_area_content('site_footer') . '
                    </fieldset>
                </fieldset>';
            
            break;
        
        case 'two_column_sidebar_right':
            $output_areas =
                '<fieldset>
                    <legend>Site Border<span class="theme_fold_css" style="padding:0"><br />#site_border</span></legend>
                    ' . get_style_designer_area_content('site_top') . '
                    ' . get_style_designer_area_content('site_header') . '
                    <fieldset>
                        <legend>Area Border<span class="theme_fold_css" style="padding:0"><br />#area_border</span></legend>
                        ' . get_style_designer_area_content('area_header') . '
                        <fieldset>
                            <legend>Page Wrapper<span class="theme_fold_css" style="padding:0"><br />#page_wrapper</span></legend>
                            <fieldset>
                                <legend>Page Border<span class="theme_fold_css" style="padding:0"><br />#page_border</span></legend>
                                ' . get_style_designer_area_content('page_header') . '
                                ' . get_style_designer_area_content('page_content') . '
                                ' . get_style_designer_area_content('sidebar') . '
                                <div class="clear"></div>
                                ' . get_style_designer_area_content('page_footer') . '
                            </fieldset>
                            ' . get_style_designer_area_content('area_footer') . '
                        </fieldset>
                    </fieldset>
                    <fieldset>
                        <legend>Site Footer Border<span class="theme_fold_css" style="padding:0"><br />#site_footer_border</span></legend>
                        ' . get_style_designer_area_content('site_footer') . '
                    </fieldset>
                </fieldset>';
            
            break;
            
        case 'three_column_sidebar_left':
            $output_areas =
                '<fieldset>
                    <legend>Site Border<span class="theme_fold_css" style="padding:0"><br />#site_border</span></legend>
                    ' . get_style_designer_area_content('site_top') . '
                    ' . get_style_designer_area_content('site_header') . '
                    <fieldset>
                        <legend>Area Border<span class="theme_fold_css" style="padding:0"><br />#area_border</span></legend>
                        ' . get_style_designer_area_content('area_header') . '
                        <fieldset>
                            <legend>Page Wrapper<span class="theme_fold_css" style="padding:0"><br />#page_wrapper</span></legend>
                            <fieldset>
                                <legend>Page Border<span class="theme_fold_css" style="padding:0"><br />#page_border</span></legend>
                                ' . get_style_designer_area_content('page_header') . '
                                ' . get_style_designer_area_content('sidebar') . '
                                <fieldset id="page_content">
                                    <legend>Page Content<span class="theme_fold_css" style="padding:0"><br />#page_content</span></legend>
                                    ' . get_style_designer_area_content('page_content_left') . '
                                    ' . get_style_designer_area_content('page_content_right') . '
                                    <div class="clear"></div>
                                </fieldset>
                                <div class="clear"></div>
                                ' . get_style_designer_area_content('page_footer') . '
                            </fieldset>
                            ' . get_style_designer_area_content('area_footer') . '
                        </fieldset>
                    </fieldset>
                    <fieldset>
                        <legend>Site Footer Border<span class="theme_fold_css" style="padding:0"><br />#site_footer_border</span></legend>
                        ' . get_style_designer_area_content('site_footer') . '
                    </fieldset>
                </fieldset>';
            
            break;
    }
    
    return
        '<div id="layout" class="' . $layout . '">
            ' . $output_areas . '
            <div id="edit_cell_properties">
                <table style="margin-bottom: 1.5em">
                    <tr>
                        <td>Region Type:</td>
                        <td>
                            <select id="region_type">
                                <option value=""></option>
                                <option value="ad">Ad</option>
                                <option value="cart">Cart</option>
                                <option value="common">Common</option>
                                <option value="designer">Designer</option>
                                <option value="dynamic">Dynamic</option>
                                <option value="login">Login</option>
                                <option value="menu">Menu</option>
                                <option value="menu_sequence">Menu Sequence</option>
                                <option value="mobile_switch">Mobile Switch</option>
                                <option value="page">Page</option>
                                <option value="pdf">PDF (beta)</option>
                                <option value="system">System</option>
                                <option value="tag_cloud">Tag Cloud</option>
                            </select>
                        </td>
                    </tr>
                    <tr id="region_name_row" style="display: none">
                        <td>Region Name:</td>
                        <td><select id="region_name"></select></td>
                    </tr>
                </table>
                <div style="text-align: center"><input type="button" id="update_cell_properties" value="Update" class="submit-primary" />&nbsp;&nbsp;&nbsp;<input type="button" id="cancel_cell_properties" value="Cancel" class="submit-secondary" /></div>
            </div>
        </div>';
}

// initialize function for getting HTML for a single area for the style designer
// this function is used by the create/edit system style screens
function get_style_designer_area_content($area)
{
    // get area label based on the area
    $output_area_label = ucwords(str_replace("_", " ", $area)) . '<span class="theme_fold_css"><br />#' . $area . '</span>';

    return
        '<div id="' . $area . '" class="area">
            <div class="area_label">' . $output_area_label . '</div>
            <div class="area_buttons"><img id="' . $area . '_add_row_before" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/style_designer_add_row_before.png" width="22" height="21" border="0" alt="Insert row before" title="Insert row before" class="area_button" /> <img id="' . $area . '_add_row_after" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/style_designer_add_row_after.png" width="22" height="21" border="0" alt="Insert row after" title="Insert row after" class="area_button" /> <img id="' . $area . '_add_column_before" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/style_designer_add_column_before.png" width="22" height="21" border="0" alt="Insert column before" title="Insert column before" class="area_button disabled" /> <img id="' . $area . '_add_column_after" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/style_designer_add_column_after.png" width="22" height="21" border="0" alt="Insert column after" title="Insert column after" class="area_button disabled" /> <img id="' . $area . '_edit_cell_properties" src="' . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/images/style_designer_edit_cell_properties.png" width="22" height="21" border="0" alt="Edit cell properties" title="Edit cell properties" class="area_button disabled" /></div>
            <div class="clear"></div>
            <div class="cells"></div>
        </div>';
}

// initialize function for getting JavaScript for all regions for style designer
// this function is used by the create/edit system style screens
function get_style_designer_regions_for_javascript()
{
    // get ad regions in order to prepare list for JavaScript
    $query = "SELECT name FROM ad_regions ORDER BY name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $ad_regions = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $ad_regions[] = $row;
    }
    
    $output_ad_regions_for_javascript = '';
    
    // loop through the ad regions in order to prepare JavaScript list
    foreach ($ad_regions as $ad_region) {
        // if this is not the first ad region in the list, then add a comma and space
        if ($output_ad_regions_for_javascript != '') {
            $output_ad_regions_for_javascript .= ', ';
        }
        
        // add ad region to list
        $output_ad_regions_for_javascript .= '"' . escape_javascript($ad_region['name']) . '"';
    }
    
    // get common regions in order to prepare list for JavaScript
    $query = "SELECT cregion_name as name FROM cregion WHERE cregion_designer_type = 'no' ORDER BY name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $common_regions = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $common_regions[] = $row;
    }
    
    $output_common_regions_for_javascript = '';
    
    // loop through the common regions in order to prepare JavaScript list
    foreach ($common_regions as $common_region) {
        // if this is not the first common region in the list, then add a comma and space
        if ($output_common_regions_for_javascript != '') {
            $output_common_regions_for_javascript .= ', ';
        }
        
        // add common region to list
        $output_common_regions_for_javascript .= '"' . escape_javascript($common_region['name']) . '"';
    }
    
    // get designer regions in order to prepare list for JavaScript
    $query = "SELECT cregion_name as name FROM cregion WHERE cregion_designer_type = 'yes' ORDER BY name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $designer_regions = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $designer_regions[] = $row;
    }
    
    $output_designer_regions_for_javascript = '';
    
    // loop through the designer regions in order to prepare JavaScript list
    foreach ($designer_regions as $designer_region) {
        // if this is not the first designer region in the list, then add a comma and space
        if ($output_designer_regions_for_javascript != '') {
            $output_designer_regions_for_javascript .= ', ';
        }
        
        // add designer region to list
        $output_designer_regions_for_javascript .= '"' . escape_javascript($designer_region['name']) . '"';
    }
    
    // get dynamic regions in order to prepare list for JavaScript
    $query = "SELECT dregion_name as name FROM dregion ORDER BY name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $dynamic_regions = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $dynamic_regions[] = $row;
    }
    
    $output_dynamic_regions_for_javascript = '';
    
    // loop through the dynamic regions in order to prepare JavaScript list
    foreach ($dynamic_regions as $dynamic_region) {
        // if this is not the first dynamic region in the list, then add a comma and space
        if ($output_dynamic_regions_for_javascript != '') {
            $output_dynamic_regions_for_javascript .= ', ';
        }
        
        // add dynamic region to list
        $output_dynamic_regions_for_javascript .= '"' . escape_javascript($dynamic_region['name']) . '"';
    }
    
    // get login regions in order to prepare list for JavaScript
    $query = "SELECT name FROM login_regions ORDER BY name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $login_regions = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $login_regions[] = $row;
    }
    
    $output_login_regions_for_javascript = '';
    
    // loop through the login regions in order to prepare JavaScript list
    foreach ($login_regions as $login_region) {
        // if this is not the first login region in the list, then add a comma and space
        if ($output_login_regions_for_javascript != '') {
            $output_login_regions_for_javascript .= ', ';
        }
        
        // add login region to list
        $output_login_regions_for_javascript .= '"' . escape_javascript($login_region['name']) . '"';
    }
    
    // get menu regions in order to prepare list for JavaScript
    $query = "SELECT name FROM menus ORDER BY name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $menu_regions = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $menu_regions[] = $row;
    }
    
    $output_menu_regions_for_javascript = '';
    
    // loop through the menu regions in order to prepare JavaScript list
    foreach ($menu_regions as $menu_region) {
        // if this is not the first menu region in the list, then add a comma and space
        if ($output_menu_regions_for_javascript != '') {
            $output_menu_regions_for_javascript .= ', ';
        }
        
        // add menu region to list
        $output_menu_regions_for_javascript .= '"' . escape_javascript($menu_region['name']) . '"';
    }
    
    $output_menu_sequence_for_javascript = '';
    
    // loop through the menu regions in order to prepare JavaScript list
    foreach ($menu_regions as $menu_region) {
        // if this is not the first menu region in the list, then add a comma and space
        if ($output_menu_sequence_for_javascript != '') {
            $output_menu_sequence_for_javascript .= ', ';
        }
        
        // add menu region to list
        $output_menu_sequence_for_javascript .= '"' . escape_javascript($menu_region['name']) . '"';
    }
    
    // get tag cloud regions in order to prepare list for JavaScript
    $query = "SELECT page_name as name FROM page WHERE page_type = 'search results' ORDER BY page_name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $tag_cloud_regions = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $tag_cloud_regions[] = $row;
    }
    
    $output_tag_cloud_regions_for_javascript = '';
    
    // loop through the tag cloud regions in order to prepare JavaScript list
    foreach ($tag_cloud_regions as $tag_cloud_region) {
        // if this is not the first tag cloud region in the list, then add a comma and space
        if ($output_tag_cloud_regions_for_javascript != '') {
            $output_tag_cloud_regions_for_javascript .= ', ';
        }
        
        // add tag cloud region to list
        $output_tag_cloud_regions_for_javascript .= '"' . escape_javascript($tag_cloud_region['name']) . '"';
    }
    
    // get pages with certain page types in order to prepare the system region list for JavaScript
    // we don't want to include pages which will create an error (e.g. form item view)
    $query =
        "SELECT page_name as name
        FROM page
        WHERE
            (page_type = 'folder view')
            || (page_type = 'photo gallery')
            || (page_type = 'custom form')
            || (page_type = 'form list view')
            || (page_type = 'form view directory')
            || (page_type = 'calendar view')
            || (page_type = 'catalog')
            || (page_type = 'express order')
            || (page_type = 'order form')
            || (page_type = 'shopping cart')
            || (page_type = 'search results')
        ORDER BY page_name ASC";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $system_region_pages = array();
    
    while ($row = mysqli_fetch_assoc($result)) {
        $system_region_pages[] = $row;
    }
    
    $output_system_region_pages_for_javascript = '';
    
    // loop through the system region pages in order to prepare JavaScript list
    foreach ($system_region_pages as $system_region_page) {
        // if this is not the first page in the list, then add a comma and space
        if ($output_system_region_pages_for_javascript != '') {
            $output_system_region_pages_for_javascript .= ', ';
        }
        
        // add tag cloud region to list
        $output_system_region_pages_for_javascript .= '"' . escape_javascript($system_region_page['name']) . '"';
    }
    
    return
        'var ad_regions = [' . $output_ad_regions_for_javascript . '];
        var common_regions = [' . $output_common_regions_for_javascript . '];
        var designer_regions = [' . $output_designer_regions_for_javascript . '];
        var dynamic_regions = [' . $output_dynamic_regions_for_javascript . '];
        var login_regions = [' . $output_login_regions_for_javascript . '];
        var menu_regions = [' . $output_menu_regions_for_javascript . '];
        var menu_sequence_regions = [' . $output_menu_sequence_for_javascript . '];
        var tag_cloud_regions = [' . $output_tag_cloud_regions_for_javascript . '];
        var system_region_pages = [' . $output_system_region_pages_for_javascript . '];';
}

// create function that will look through $css_properties recursively and find Google font families
function get_google_font_families($array)
    {
       $google_font_families = array();

       // loop through the array in order to find Google font families
       foreach ($array as $key => $value) {
           // if this item is a font_family item and the value is a Google font family,
           // then add it to the array
           if (
               ($key == 'font_family')
               && (mb_substr($value, 0, 2) == '\'-')
           ) {
               $google_font_families[] = $value;
               
           // else if the value is an array, then run this same function (i.e. by using recursion)
           // in order to look deeper into the array
           } else if (is_array($value) == TRUE) {
               $google_font_families = array_merge($google_font_families, get_google_font_families($value));
           }
       }

       // remove duplicates
       $google_font_families = array_unique($google_font_families);

       return $google_font_families;
}

//this function returns the tint color (tint to white) from the color and degree passed to it
function color_tint ($color, $degree)
{
    // gradient calc example: $color = 48823A $degree = 30 (for 30% percent shading)
    //      R = (48 * .3) + (FF * (1-.3)) = C8  //  G = (83 * .3) + (FF * (1-.3)) = D9  // B = (2A * .3) + (FF * (1-.3)) = BF
    //      darker bottom = #48832A; lighter top (30% white tint) = #C8D9BF
    $rgb = hexdec($color);
    $r = ($rgb >> 16) & 0xFF;
    $g = ($rgb >> 8) & 0xFF;
    $b = $rgb & 0xFF;
    $mixcolor1 = ($degree / 100);
    $mixcolor2 = (1 - $mixcolor1);
    $r_tint = ($r * $mixcolor1) + (0xFF * $mixcolor2);
    $g_tint = ($g * $mixcolor1) + (0xFF * $mixcolor2);
    $b_tint = ($b * $mixcolor1) + (0xFF * $mixcolor2);
    $tint_color = mb_strtoupper(dechex(($r_tint << 16) + ($g_tint << 8) + ($b_tint)));
    return $tint_color;
}

// this function prepares the module selector
function prepare_module_selector($module, $selector) 
{
    // if the module is not one of the base modules, then add the module selector to the selector
    if (
        ($module != 'base_module') 
        && ($module != 'background_borders_and_spacing') 
        && ($module != 'text') 
        && ($module != 'layout')
    ) {
        switch ($module) {
            case 'headings_general':
                $selector = $selector . ' h1,' . $selector . ' h2,' . $selector . ' h3,' . $selector . ' h4,' . $selector . ' h5,' . $selector . ' h6';
                break;
            
            case 'heading_1':
                $selector .= ' h1';
                break;
            
            case 'heading_2':
                $selector .= ' h2';
                break;
            
            case 'heading_3':
                $selector .= ' h3';
                break;
            
            case 'heading_4':
                $selector .= ' h4';
                break;
            
            case 'heading_5':
                $selector .= ' h5';
                break;
            
            case 'heading_6':
                $selector .= ' h6';
                break;
            
            case 'image_primary':
                $selector = $selector . ' img.image-primary,' . $selector . ' img.image-left-primary,' . $selector . ' img.image-right-primary';
                break;
            
            case 'image_secondary':
                $selector = $selector . ' img.image-secondary,' . $selector . ' img.image-left-secondary,' . $selector . ' img.image-right-secondary';
                break;
            
            case 'links':
                $selector = $selector . ' a:link,' . $selector . ' a:active,' . $selector . ' a:visited';
                break;
            
            case 'links_hover':
                $selector = $selector . ' a:hover,' . $selector . ' a:focus';
                break;
            
            case 'paragraph':
                $selector = $selector . ' p';
                break;
                
            case 'table':
                $selector = $selector . ' table';
                break;
        }
    }
    
    return $selector;
}

// this function prepares and returns the css properties
function output_css_properties($properties, $object = '')
{
    $output = '';
    
    // if there are properties, then continue
    if (empty($properties) == FALSE) {
        foreach ($properties as $property => $value) {

            // force importance for button styling so, for example, the theme designer that sets #site_footer a:link {...}
            // doesn't override the custom format buttons.
            if (($object == 'primary_buttons') || ($object == 'secondary_buttons') || ($object == 'primary_buttons_hover') || ($object == 'secondary_buttons_hover')) {
                $make_important = ' !important';
            } else {
                $make_important = '';
            }

            switch($property) {
                case 'background_type':
                case 'background_color':
                case 'background_image':
                case 'background_horizontal_position':
                case 'background_vertical_position':
                case 'background_repeat':
                    $output_background = '';
                    
                    // if there is a background color, then output the background css property and the add the color to it
                    if ($properties['background_color'] != '') {
                        // add a space so that there is one between the property and value
                        $output_background .= ' ';
                        
                        // if the background color is not transparent, then add a pound sign to the background
                        if ($properties['background_color'] != 'transparent') {
                            $output_background .= '#';
                        }
                        
                        // add the background color value to the background property
                        $output_background .= $properties['background_color'];
                        
                        // unset the property so that we do not loop throug it again
                        unset($properties['background_color']);
                    }
                    
                    // if there is a background image then add it to the background property
                    if ($properties['background_image'] != '') {
                        $output_background .= ' url({path}' . $properties['background_image'] . ')';
                        
                        // unset the property so that we do not loop throug it again
                        unset($properties['background_image']);
                    }
                    
                    // if there is a background horizontal position then add it to the background property
                    if ($properties['background_horizontal_position'] != '') {
                        $output_background .= ' ' . $properties['background_horizontal_position'];
                        
                        // unset the property so that we do not loop throug it again
                        unset($properties['background_horizontal_position']);
                    }
                    
                    // if there is a background horizontal position then add it to the background property
                    if ($properties['background_vertical_position'] != '') {
                        $output_background .= ' ' . $properties['background_vertical_position'];
                        
                        // unset the property so that we do not loop throug it again
                        unset($properties['background_vertical_position']);
                    }
                    
                    // if there is a background repeat then add it to the background property
                    if ($properties['background_repeat'] != '') {
                        $output_background .= ' ' . str_replace('_', '-', $properties['background_repeat']);
                        
                        // unset the property so that we do not loop throug it again
                        unset($properties['background_repeat']);
                    }
                    
                    // if there are properties, then output them
                    if ($output_background != '') {
                      //  if (($object == 'primary_buttons') || ($object == 'secondary_buttons')) {
                      //      $output .= 'background:' . $output_background . ' !important;' . "\r\n";
                      //  } else {
                            $output .= 'background:' . $output_background . $make_important . ';' . "\r\n";
                      //  }
                    }
                    break;
                
                case 'borders_toggle':
                case 'border_size':
                case 'border_style':
                case 'border_color':
                case 'border_position':

                    $output_border = '';
                    
                    // if the border size is not blank, then add it and remove the property from the array
                    if ($properties['border_size'] != '') {
                        $output_border .= ' ' . $properties['border_size'] . 'px';
                        unset($properties['border_size']);
                    }
                    
                    // if the border color is not blank, then add it and remove the property from the array
                    if ($properties['border_color'] != '') {
                        $output_border .= ' #' . $properties['border_color'];
                        unset($properties['border_color']);
                    }

                    // if there are properties then determine and add position
                    if ($output_border != '') {

                        // add border style
                        $output_border .= ' ' . $properties['border_style'];

                        // if the position is not all, then output the appropriate border property
                        if ($properties['border_position'] != 'all') {
                            switch($properties['border_position']) {
                                case 'top':
                                    $output_border = 'border-top:' . $output_border;
                                    break;
                                
                                case 'right':
                                    $output_border = 'border-right:' . $output_border;
                                    break;
                                
                                case 'bottom':
                                    $output_border = 'border-bottom:' . $output_border;
                                    break;
                                
                                case 'left':
                                    $output_border = 'border-left:' . $output_border;
                                    break;
                            }

                        // else output the general border property
                        } else {
                            $output_border = 'border:' . $output_border;
                        }

                        // output borders if enabled
                        if ($properties['borders_toggle'] == 1) {
                            $output .= $output_border . $make_important . ';' . "\r\n";
                        }
                    }
                    // unset properties
                    unset($properties['border_style']);
                    unset($properties['border_position']);

                    break;
                
                case 'font_color':
                    if ($value != '') {
                    //    if (($object == 'primary_buttons') || ($object == 'secondary_buttons')) {
                     //       $output .= 'color: #' . $value . ' !important;' . "\r\n";
                     //   } else {
                            $output .= 'color: #' . $value . $make_important . ';' . "\r\n";
                     //   }
                    }
                    break;

                case 'font_family':
                    if ($value != '') {
                        // check for google font and remove leading font (used only to identify a google font)
                        if (mb_substr($value, 0, 2) == '\'-') {
                            $output .= 'font-family: ' . mb_substr($value,mb_strpos($value,',',2)+1) . $make_important . ';' . "\r\n";
                        } else {
                            // else use total font stack
                           $output .= 'font-family: ' . $value . $make_important . ';' . "\r\n";
                        }
                    }
                    break;

                case 'rounded_corners_toggle':
                case 'rounded_corner_top_left':
                case 'rounded_corner_top_right':
                case 'rounded_corner_bottom_left':
                case 'rounded_corner_bottom_right':
                    // Do nothing. We will handle these below.
                    break;
                    
                case 'shadows_toggle':
                case 'shadow_horizontal_offset':
                case 'shadow_vertical_offset':
                case 'shadow_blur_radius':
                case 'shadow_color':
                    // Do nothing. We will handle these below.
                    break;
                    
                case 'margin_top':
                case 'margin_right':
                case 'margin_bottom':
                case 'margin_left':
                    // if the margin top is not blank, then add it and remove the property from the array
                    if ($properties['margin_top'] != '') {
                        $output .= 'margin-top: ' . $properties['margin_top'] . $make_important . ';' . "\r\n";
                        unset($properties['margin_top']);
                    }
                    
                    // if there is a right margin, and if the position is not the center, then output the margin right property
                    if (($property == 'margin_right') && ($properties['margin_right'] != '') && ($properties['position'] != 'center')) {
                        $output .= 'margin-right: ' . $properties['margin_right'] . $make_important . ';' . "\r\n";
                        unset($properties['margin_right']);
                    }
                    
                    // if the margin bottom is not blank, then add it and remove the property from the array
                    if ($properties['margin_bottom'] != '') {
                        $output .= 'margin-bottom: ' . $properties['margin_bottom'] . $make_important . ';' . "\r\n";
                        unset($properties['margin_bottom']);
                    }
                    
                    // if there is a left margin, and if the position is not the center, then output the margin left property
                    if (($property == 'margin_left') && ($properties['margin_left'] != '') && ($properties['position'] != 'center')) {
                        $output .= 'margin-left: ' . $properties['margin_left'] . $make_important . ';' . "\r\n";
                        unset($properties['margin_left']);
                    }
                    break;
                
                case 'padding_top':
                case 'padding_right':
                case 'padding_bottom':
                case 'padding_left':
                    // if the padding is not blank, output and add !important to override style designer default padding, then remove from array
                    if ($properties['padding_top'] != '') {
                        $output .= 'padding-top: ' . $properties['padding_top'] . ' !important;' . "\r\n";
                        unset($properties['padding_top']);
                    }
                    
                    if ($properties['padding_right'] != '') {
                        $output .= 'padding-right: ' . $properties['padding_right'] . ' !important;' . "\r\n";
                        unset($properties['padding_right']);
                    }
                    
                    if ($properties['padding_bottom'] != '') {
                        $output .= 'padding-bottom: ' . $properties['padding_bottom'] . ' !important;' . "\r\n";
                        unset($properties['padding_bottom']);
                    }
                    
                    if ($properties['padding_left'] != '') {
                        $output .= 'padding-left: ' . $properties['padding_left'] . ' !important;' . "\r\n";
                        unset($properties['padding_left']);
                    }
                    break;

                case 'position':
                    // if the postion has a value then set a float property or margins
                    if ($value != '') {
                        if ($value != 'center') {
                            $output .= 'float: ' . $value . ';' . "\r\n";

                        // else the position is set to center
                        } else {
                            $output .= 
                                'margin-left: auto;' . "\r\n" . 
                                'margin-right: auto;' . "\r\n";
                        }
                    }
                    break;

                // if width has a value, output and add !important to override style designer default padding
                case 'width':
                    if ($value != '') {
                        $output .= 'width: ' . $value . ' !important;' . "\r\n";
                    }
                    break;
                
                default:
                    if ($value != '') {
                        $output .= str_replace('_', '-', $property) . ': ' . $value . $make_important . ';' . "\r\n";
                    }
                    break;
            }
        }

        // if this is the primary buttons or secondary buttons object or their hover equivalents, then force values for rounded corners,
        // even if there are no properties because we need to override the rounded corners for form fields
        if (
            ($object == 'primary_buttons')
            || ($object == 'primary_buttons_hover')
            || ($object == 'secondary_buttons')
            || ($object == 'secondary_buttons_hover')
        ) {
            $properties['rounded_corners_toggle'] = 1;

            // if the top left property is blank, then set it to 0
            if ($properties['rounded_corner_top_left'] == '') {
                $properties['rounded_corner_top_left'] = 0;
            }

            // if the top right property is blank, then set it to 0
            if ($properties['rounded_corner_top_right'] == '') {
                $properties['rounded_corner_top_right'] = 0;
            }

            // if the bottom left property is blank, then set it to 0
            if ($properties['rounded_corner_bottom_left'] == '') {
                $properties['rounded_corner_bottom_left'] = 0;
            }

            // if the bottom right property is blank, then set it to 0
            if ($properties['rounded_corner_bottom_right'] == '') {
                $properties['rounded_corner_bottom_right'] = 0;
            }
        }

        // if rounded corners are enabled then output CSS for them
        if ($properties['rounded_corners_toggle'] == 1) {
            // if there is a top left corner radius then output CSS for it
            if ($properties['rounded_corner_top_left'] !== '') {
                $output .= 
                    '-moz-border-radius-topleft: ' . $properties['rounded_corner_top_left'] . 'px' . $important .';' . "\r\n" . 
                    '-webkit-border-top-left-radius: ' . $properties['rounded_corner_top_left'] . 'px' . $important . ';' . "\r\n" . 
                    'border-top-left-radius: ' . $properties['rounded_corner_top_left'] . 'px' . $important . ';' . "\r\n";
            }

            // if there is a top right corner radius, then output CSS for it
            if ($properties['rounded_corner_top_right'] !== '') {
                $output .= 
                    '-moz-border-radius-topright: ' . $properties['rounded_corner_top_right'] . 'px' . $important . ';' . "\r\n" . 
                    '-webkit-border-top-right-radius: ' . $properties['rounded_corner_top_right'] . 'px' . $important . ';' . "\r\n" . 
                    'border-top-right-radius: ' . $properties['rounded_corner_top_right'] . 'px' . $important . ';' . "\r\n";
            }

            // if there is a bottom left corner radius, then output CSS for it
            if ($properties['rounded_corner_bottom_left'] !== '') {
                $output .= 
                    '-moz-border-radius-bottomleft: ' . $properties['rounded_corner_bottom_left'] . 'px' . $important . ';' . "\r\n" . 
                    '-webkit-border-bottom-left-radius: ' . $properties['rounded_corner_bottom_left'] . 'px' . $important . ';' . "\r\n" . 
                    'border-bottom-left-radius: ' . $properties['rounded_corner_bottom_left'] . 'px' . $important . ';' . "\r\n";
            }
            
            // if there is a bottom right corner radius, then output CSS for it
            if ($properties['rounded_corner_bottom_right'] !== '') {
                $output .= 
                    '-moz-border-radius-bottomright: ' . $properties['rounded_corner_bottom_right'] . 'px' . $important . ';' . "\r\n" . 
                    '-webkit-border-bottom-right-radius: ' . $properties['rounded_corner_bottom_right'] . 'px' . $important . ';' . "\r\n" . 
                    'border-bottom-right-radius: ' . $properties['rounded_corner_bottom_right'] . 'px' . $important . ';' . "\r\n";
            }
        }

        // If shadows are enabled then output CSS for them.
        if ($properties['shadows_toggle'] == 1) {
            // If the horizontal offset is blank, then set it to 0.
            if ($properties['shadow_horizontal_offset'] == '') {
                $properties['shadow_horizontal_offset'] = 0;
            }

            // If the veritical offset is blank, then set it to 0.
            if ($properties['shadow_vertical_offset'] == '') {
                $properties['shadow_vertical_offset'] = 0;
            }

            // Add horizontal and vertical offset values regardless of whether there is a value because those values are required.
            $output_shadow = $properties['shadow_horizontal_offset'] . 'px ' . $properties['shadow_vertical_offset'] . 'px';

            // If the shadow blur radius is not blank then add it.
            if ($properties['shadow_blur_radius'] != '') {
                $output_shadow .= ' ' . $properties['shadow_blur_radius'] . 'px';
            }
            
            // If the shadow color is not blank then add it.
            if ($properties['shadow_color'] != '') {
                $output_shadow .= ' #' . $properties['shadow_color'];
            }

            // Output shadow CSS properties.
            $output .= 
                '-moz-box-shadow: ' . $output_shadow . $important . ';' . "\r\n" . 
                '-webkit-box-shadow: ' . $output_shadow . $important . ';' . "\r\n" . 
                'box-shadow: ' . $output_shadow . $important . ';' . "\r\n";
        }
    }
    
    return $output;
}

// Gets the sequence of the given menu
function get_menu_sequence($menu_id, $parent_id = 0, $menu_sequence = Array())
{
    global $page_names_in_menu_sequence;
    
    // if this is the first time we are running this function, then set the page names in menu sequence array
    if (is_array($page_names_in_menu_sequence) == FALSE) {
        $page_names_in_menu_sequence = Array();
    }
    
    $menu_items = array();
    
    // get menu items
    $query = 
        "SELECT 
            menu_items.id,
            menu_items.name,
            page.page_name as link_page_name
        FROM menu_items
        LEFT JOIN page ON page.page_id = menu_items.link_page_id
        WHERE 
           (menu_items.parent_id = '" . escape($parent_id) . "') 
           AND (menu_items.menu_id = '" . escape($menu_id) . "') 
        ORDER BY menu_items.sort_order";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    while ($row = mysqli_fetch_assoc($result)) {
        $menu_items[] = $row;
    }
    
    // loop through all menu items in order to get child menu items, and add to the array
    foreach ($menu_items as $menu_item) {
        // if this item has a link page name, and if it does not link to a page that is already in the sequence, then add menu item to array
        if (($menu_item['link_page_name'] != '') && (in_array($menu_item['link_page_name'], $page_names_in_menu_sequence) == FALSE)) {
            // add the item's link page name to the array
            $page_names_in_menu_sequence[] = $menu_item['link_page_name'];
            
            // add the menu item to the array
            $menu_sequence[] = array('name'=>$menu_item['name'], 'link_page_name'=>$menu_item['link_page_name']);
        }
        
        // find out if sub menu exists
        $query = "SELECT COUNT(*) FROM menu_items WHERE parent_id = '" . $menu_item['id'] . "'";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        $row = mysqli_fetch_row($result);
        
        // if sub menu exists, then add sub menu content to the array
        if ($row[0] > 0) {
            $menu_sequence = get_menu_sequence($menu_id, $menu_item['id'], $menu_sequence);
        }
    }
    
    return $menu_sequence;
}

// this function is responsible for building options for the filters picklist
function get_filter_options($filters_in_array, $current_filter)
{
    $output = '';

    // loop through each of the filters in the array, and build the picklist options
    foreach($filters_in_array as $filter_value => $filter_name) {
        $selected = '';
        
        // if the filter value is equal to the current filter, then select the option
        if ($filter_value == $current_filter) {
            $selected = ' selected="selected"';
        }
        
        // output the option
        $output .= '<option value="' . $filter_value . '"' . $selected . '>' . $filter_name . '</option>';
    }
    
    return $output;
}

function check_if_request_is_secure()
{
    // if request is secure, then return TRUE
    if (
        ((isset($_SERVER['HTTPS']) == TRUE) && ($_SERVER['HTTPS'] == 'on'))
        || ((isset($_SERVER['SERVER_PORT']) == TRUE) && ($_SERVER['SERVER_PORT'] == '443'))
    ) {
        return TRUE;
        
    // else request is not secure, so return FALSE
    } else {
        return FALSE;
    }
}

// Create function that is responsible for checking minimum and maximum quantity for products
// and adjusting order items as necessary.
function check_quantity($liveform)
{
    // Get order items that have a minimum or maximum quantity.
    $order_items = db_items(
        "SELECT
            order_items.id,
            order_items.quantity,
            products.minimum_quantity,
            products.maximum_quantity,
            products.name,
            products.short_description
        FROM order_items
        LEFT JOIN products ON order_items.product_id = products.id
        WHERE 
            (order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
            AND
            (
                (products.minimum_quantity != '0')
                OR (products.maximum_quantity != '0')
            )");

    // Loop through order items in order to check minimum and maximum quantity.
    foreach ($order_items as $order_item) {
        // If there is both a minimum and maximum quantity for the product
        // and they are the same, and the order item quantity is not the required amount,
        // then adjust the quantity and add notice.
        if (
            ($order_item['minimum_quantity'] != 0)
            && ($order_item['maximum_quantity'] != 0)
            && ($order_item['minimum_quantity'] == $order_item['maximum_quantity'])
            && ($order_item['quantity'] != $order_item['minimum_quantity'])
        ) {
            db("UPDATE order_items SET quantity = '" . $order_item['minimum_quantity'] . "' WHERE id = '" . $order_item['id'] . "'");

            // prepare product description for notice
            $product_description = '';
            
            // if there is a name, then add it to the description
            if ($order_item['name'] != '') {
                $product_description .= $order_item['name'];
            }
            
            // if there is a short description, then add it to the description
            if ($order_item['short_description'] != '') {
                // if the description is not blank, then add separator
                if ($product_description != '') {
                    $product_description .= ' - ';
                }
                
                $product_description .= $order_item['short_description'];
            }
            
            // if the description is blank, then set default description
            if ($product_description == '') {
                $product_description = 'an item';
            }

            $liveform->add_notice('We\'re sorry, we require a quantity of ' . number_format($order_item['minimum_quantity']) . ' for ' . h($product_description) . ', so we have updated the quantity for you.');

        // Otherwise, if there is a minimum quantity and the quantity is below that minimum,
        // then update quantity and add notice.
        } else if (
            ($order_item['minimum_quantity'] != 0)
            && ($order_item['quantity'] < $order_item['minimum_quantity'])
        ) {
            db("UPDATE order_items SET quantity = '" . $order_item['minimum_quantity'] . "' WHERE id = '" . $order_item['id'] . "'");

            // prepare product description for notice
            $product_description = '';
            
            // if there is a name, then add it to the description
            if ($order_item['name'] != '') {
                $product_description .= $order_item['name'];
            }
            
            // if there is a short description, then add it to the description
            if ($order_item['short_description'] != '') {
                // if the description is not blank, then add separator
                if ($product_description != '') {
                    $product_description .= ' - ';
                }
                
                $product_description .= $order_item['short_description'];
            }
            
            // if the description is blank, then set default description
            if ($product_description == '') {
                $product_description = 'an item';
            }

            $liveform->add_notice('We\'re sorry, we require a minimum quantity of ' . number_format($order_item['minimum_quantity']) . ' for ' . h($product_description) . ', so we have increased the quantity for you.');

        // Otherwise, if there is a maximum quantity and the quantity is above that maximum,
        // then update quantity and add notice.
        } else if (
            ($order_item['maximum_quantity'] != 0)
            && ($order_item['quantity'] > $order_item['maximum_quantity'])
        ) {
            db("UPDATE order_items SET quantity = '" . $order_item['maximum_quantity'] . "' WHERE id = '" . $order_item['id'] . "'");

            // prepare product description for notice
            $product_description = '';
            
            // if there is a name, then add it to the description
            if ($order_item['name'] != '') {
                $product_description .= $order_item['name'];
            }
            
            // if there is a short description, then add it to the description
            if ($order_item['short_description'] != '') {
                // if the description is not blank, then add separator
                if ($product_description != '') {
                    $product_description .= ' - ';
                }
                
                $product_description .= $order_item['short_description'];
            }
            
            // if the description is blank, then set default description
            if ($product_description == '') {
                $product_description = 'an item';
            }

            $liveform->add_notice('We\'re sorry, we allow a maximum quantity of ' . number_format($order_item['maximum_quantity']) . ' for ' . h($product_description) . ', so we have decreased the quantity for you.');
        }
    }
}

// create function for checking if order items are still valid based on inventory levels
function check_inventory($liveform)
{
    // get all of the products that are in the order that have inventory enabled
    $query =
        "SELECT
            DISTINCT(order_items.product_id),
            products.id,
            products.name,
            products.short_description,
            products.inventory_quantity,
            products.out_of_stock_message
        FROM order_items
        LEFT JOIN products ON order_items.product_id = products.id
        WHERE
            (order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
            AND (products.inventory = '1')
            AND (products.backorder = '0')";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $products = array();
    
    // loop through products in order to add them to array
    while ($row = mysqli_fetch_assoc($result)) {
        $products[] = $row;
    }
    
    // loop through products in order to check inventory for order items
    foreach ($products as $product) {
        // set the starting inventory quantity
        // so later we know if the product was completely out of stock to begin with before we started looking at order items
        $product['starting_inventory_quantity'] = $product['inventory_quantity'];
        
        // get all order items in order to check if there is enough inventory quantity for them
        $query =
            "SELECT
                order_items.id,
                order_items.quantity,
                ship_tos.id as ship_to_id,
                ship_tos.complete as ship_to_complete
            FROM order_items
            LEFT JOIN ship_tos ON order_items.ship_to_id = ship_tos.id
            WHERE
                (order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                AND (order_items.product_id = '" . $product['id'] . "')
            ORDER BY order_items.id ASC";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $order_items = array();
        
        // loop through order items in order to add them to array
        while ($row = mysqli_fetch_assoc($result)) {
            $order_items[] = $row;
        }
        
        // loop through order items in order to check if there is enough inventory quantity for them
        foreach ($order_items as $order_item) {
            // prepare product description for notice
            $product_description = '';
            
            // if there is a name, then add it to the description
            if ($product['name'] != '') {
                $product_description .= $product['name'];
            }
            
            // if there is a short description, then add it to the description
            if ($product['short_description'] != '') {
                // if the description is not blank, then add separator
                if ($product_description != '') {
                    $product_description .= ' - ';
                }
                
                $product_description .= $product['short_description'];
            }
            
            // if the description is blank, then set default description
            if ($product_description == '') {
                $product_description = 'an item';
            }
            
            // if there is no more inventory, then remove order item and add notice
            if ($product['inventory_quantity'] == 0) {
                remove_order_item($order_item['id']);
                
                // if the product is completely out of stock, then add a certain notice
                if ($product['starting_inventory_quantity'] == 0) {
                    $liveform->add_notice('We\'re sorry, ' . h($product_description) . ' has been removed from your order because it is not currently available. ' . $product['out_of_stock_message']);
                    
                // else the product is only out of stock between there are other order items in the order that are using up inventory quantity, so add different notice
                } else {
                    $liveform->add_notice('We\'re sorry, ' .h($product_description) . ' has been removed from your order because the availabilty of the item is currently limited.');
                }
                
            // else there is remaining inventory, so check if quantity needs to be reduced
            } else {
                // if the order item quantity is greater than the inventory quantity, then reduce it by setting it to the inventory quantity
                // Eventually we need to deal with the fact that we might be reducing the quantity below the
                // minimum quantity property below.  This is a current bug when using minimum quantity with inventory tracking enabled.
                if ($order_item['quantity'] > $product['inventory_quantity']) {
                    $order_item['quantity'] = $product['inventory_quantity'];
                    
                    $query = "UPDATE order_items SET quantity = '" . $order_item['quantity'] . "' WHERE id = '" . $order_item['id'] . "'";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                    
                    // if there is a ship to and it is complete, then update shipping cost for ship to
                    if (($order_item['ship_to_id'] != '') && ($order_item['ship_to_complete'] == 1)) {
                        require_once(dirname(__FILE__) . '/shipping.php');
                        update_shipping_cost_for_ship_to($order_item['ship_to_id']);
                    }
                    
                    $liveform->add_notice('We\'re sorry, the quantity you entered for ' . h($product_description) . ' has been reduced because the availabilty of the item is currently limited.');
                }
                
                // decrement the inventory quantity for the product,
                // so as we loop through these order items we can dynamically know how much inventory quantity is left
                // based on order items that we have already looped through
                $product['inventory_quantity'] = $product['inventory_quantity'] - $order_item['quantity'];
            }
        }
    }
}

// create function for checking if calendar event reservations are still valid and available for all order items in order
function check_reservations($liveform)
{
    // if calendars is enabled in the site settings, then proceed with checking reservations
    if (CALENDARS == TRUE) {
        // get all calendar event reservation order items in order to verify that they are still valid
        // (e.g. check if there are remaining reservation spots)
        $query =
            "SELECT
                order_items.id,
                order_items.quantity,
                order_items.calendar_event_id,
                order_items.recurrence_number,
                products.id as product_id,
                products.enabled AS product_enabled,
                products.name,
                products.short_description,
                ship_tos.id as ship_to_id,
                ship_tos.complete as ship_to_complete
            FROM order_items
            LEFT JOIN products ON order_items.product_id = products.id
            LEFT JOIN ship_tos ON order_items.ship_to_id = ship_tos.id
            WHERE
                (order_items.order_id = '" . $_SESSION['ecommerce']['order_id'] . "')
                AND (order_items.calendar_event_id != '0')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $reservation_order_items = array();
        
        // loop through reservation order items in order to add them to array
        while ($row = mysqli_fetch_assoc($result)) {
            $reservation_order_items[] = $row;
        }
        
        // loop through all reservation order items in order to perform maintenance on them
        foreach ($reservation_order_items as $reservation_order_item) {
            $calendar_event = get_calendar_event($reservation_order_item['calendar_event_id'], $reservation_order_item['recurrence_number']);
            
            // prepare product description in case we need to add a notice
            $product_description = '';
            
            // if there is a name, then add it to the description
            if ($reservation_order_item['name'] != '') {
                $product_description .= $reservation_order_item['name'];
            }
            
            // if there is a short description, then add it to the description
            if ($reservation_order_item['short_description'] != '') {
                // if the description is not blank, then add separator
                if ($product_description != '') {
                    $product_description .= ' - ';
                }
                
                $product_description .= $reservation_order_item['short_description'];
            }
            
            // if the description is blank, then set default description
            if ($product_description == '') {
                $product_description = 'an item';
            }
            
            // if the calendar event no longer exists,
            // or if the product no longer exists,
            // or if the product is disabled,
            // or if the calendar event is no longer published,
            // or if reservations are no longer enabled for the calendar event,
            // or if this event is not a recurring event or it is a recurring event but it separates reservations and the event/instance is in the past,
            // or if this event is a recurring event and it does not separate reservations and the initial instance is in the past and it does not have any recurring instances in the future,
            // then remove order item and add notice because reservations should not be allowed for this product anymore
            if (
                ($calendar_event == FALSE)
                || ($reservation_order_item['product_id'] == '')
                || ($reservation_order_item['product_enabled'] == 0)
                || ($calendar_event['published'] == 0)
                || ($calendar_event['reservations'] == 0)
                ||
                (
                    (
                        ($calendar_event['total_recurrence_number'] == 0)
                        || ($calendar_event['separate_reservations'] == 1)
                    )
                    && (strtotime($calendar_event['end_date_and_time']) < time())
                )
                ||
                (
                    ($calendar_event['total_recurrence_number'] > 0)
                    && ($calendar_event['separate_reservations'] == 0)
                    && (strtotime($calendar_event['end_date_and_time']) < time())
                    && (check_for_recurring_instance_in_the_future($calendar_event['id']) == FALSE)
                )
            ) {
                remove_order_item($reservation_order_item['id']);
                $liveform->add_notice('We\'re sorry, ' . h($product_description) . ' has been removed from your order because it is no longer available.');

            // else reservations are enabled so if reservations are limited, then check if there are available spots
            } else if ($calendar_event['limit_reservations'] == 1) {
                // if there are no remaining spots, then remove order item and add notice
                if ($calendar_event['number_of_remaining_spots'] == 0) {
                    remove_order_item($reservation_order_item['id']);
                    $liveform->add_notice('We\'re sorry, ' . h($product_description) . ' has been removed from your order because it is no longer available. ' . $calendar_event['no_remaining_spots_message']);
                    
                // else there are remaining spots, so if the quantity is greater than the number of remaining spots,
                // then adjust quantity for order item and add notice
                } else if ($reservation_order_item['quantity'] > $calendar_event['number_of_remaining_spots']) {
                    $query = "UPDATE order_items SET quantity = '" . $calendar_event['number_of_remaining_spots'] . "' WHERE id = '" . $reservation_order_item['id'] . "'";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                    
                    // if there is a ship to and it is complete, then update shipping cost for ship to
                    if (($reservation_order_item['ship_to_id'] != '') && ($reservation_order_item['ship_to_complete'] == 1)) {
                        require_once(dirname(__FILE__) . '/shipping.php');
                        update_shipping_cost_for_ship_to($reservation_order_item['ship_to_id']);
                    }
                    
                    $liveform->add_notice('We\'re sorry, ' . h($product_description) . ' is limited, so we were not able to add your requested quantity.');
                }
            }
        }
    }
}

// create function that will check to see if a recurring instance exists in the future for a calendar event
// we use this in order to determine if a calendar event is allowed to be reserved.
// For example it might be a calendar event that does not separate reservations and the first instance is the past
// but the ordering checks need to make sure there is an instance in the future in order to know if the event should be
// allowed to be reserved.
function check_for_recurring_instance_in_the_future($calendar_event_id)
{
    // get calendar event info
    $query =
        "SELECT
            id,
            recurrence_number as total_recurrence_number,
            recurrence_type,
            recurrence_day_sun,
            recurrence_day_mon,
            recurrence_day_tue,
            recurrence_day_wed,
            recurrence_day_thu,
            recurrence_day_fri,
            recurrence_day_sat,
            recurrence_month_type,
            start_time as start_date_and_time,
            end_time as end_date_and_time
        FROM calendar_events
        WHERE id = '" . escape($calendar_event_id) . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    $calendar_event = mysqli_fetch_assoc($result);

    $recurrence_day_sun = $calendar_event['recurrence_day_sun'];
    $recurrence_day_mon = $calendar_event['recurrence_day_mon'];
    $recurrence_day_tue = $calendar_event['recurrence_day_tue'];
    $recurrence_day_wed = $calendar_event['recurrence_day_wed'];
    $recurrence_day_thu = $calendar_event['recurrence_day_thu'];
    $recurrence_day_fri = $calendar_event['recurrence_day_fri'];
    $recurrence_day_sat = $calendar_event['recurrence_day_sat'];
    $recurrence_month_type = $calendar_event['recurrence_month_type'];
    
    // get exceptions so we don't look at those instances
    $query = "SELECT recurrence_number FROM calendar_event_exceptions WHERE calendar_event_id = '" . $calendar_event['id'] . "'";
    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
    
    $exceptions = array();
    
    // loop through exceptions in order to add them to array
    while ($row = mysqli_fetch_assoc($result)) {
        $exceptions[] = $row['recurrence_number'];
    }
    
    // split event start date and time into parts
    $event_start_date_and_time_parts = explode(' ', $calendar_event['start_date_and_time']);
    $event_start_date = $event_start_date_and_time_parts[0];
    $event_start_time = $event_start_date_and_time_parts[1];
    $event_start_date_parts = explode('-', $event_start_date);
    $event_start_year = $event_start_date_parts[0];
    $event_start_month = $event_start_date_parts[1];
    $event_start_day = $event_start_date_parts[2];
    
    // Find the difference between the original start date and end date (So we do not need to calculate the end dates new date too)
    $event_start_end_difference = strtotime($calendar_event['end_date_and_time']) - strtotime($calendar_event['start_date_and_time']);

    // If this is a monthly event and the month type is "day of the week",
    // then determine which week in the month the event is on.
    // If the week is 1-4 then we will use that, however if the week is 5,
    // then we interpret that as the last week.
    if (
        ($calendar_event['recurrence_type'] == 'month')
        && ($calendar_event['recurrence_month_type'] == 'day_of_the_week')
    ) {
        $day_of_the_week = date('l', strtotime($event_start_date));
        $first_day_of_the_month_timestamp = strtotime($event_start_year . '-' . $event_start_month . '-01');

        $week = '';

        // Create a loop in order to determine which week event falls on.
        // We only loop through 4 weeks, because we are going to set "last" below for 5th week.
        for ($week_index = 0; $week_index <= 3; $week_index++) {
            // If the event is in this week, then remember the week number and break out of this loop.
            if ($event_start_date == date('Y-m-d', strtotime('+' . $week_index . ' week ' . $day_of_the_week, $first_day_of_the_month_timestamp))) {
                $week = $week_index + 1;
                break;
            }
        }

        // If a week was not found, then that means it falls on the 5th week,
        // so set it to be the last week.
        if ($week == '') {
            $week = 'last';
        }
    }
    
    // loop through all recurring instances of this event in order to determine if one exists in the future
    for ($recurrence_number = 1; $recurrence_number <= $calendar_event['total_recurrence_number']; $recurrence_number++) {
        // adjust event start date depending on recurrence type
        switch ($calendar_event['recurrence_type']) {
            // Daily
            case 'day':
                $count = 0;

                // Loop through days in the future until we find a date that is valid
                // based on the valid days of the week that were selected.
                while (true) {
                    $new_time = strtotime('+1 day', strtotime($event_start_date));
                    $event_start_date = date('Y-m-d', $new_time);
                    $instance_start_date = $event_start_date;
                    $day_of_the_week = strtolower(date('D', $new_time));

                    // If this day of the week is valid for this calendar event,
                    // then we have found a valid date, so break out of the loop.
                    if (${'recurrence_day_' . $day_of_the_week} == 1) {
                        break;
                    }

                    $count++;

                    // If we have already looped 7 times, then something is wrong,
                    // so break out of this loop and the recurrence loop above.
                    // This should never happen but is added just in case in order to
                    // prevent an endless loop.
                    if ($count == 7) {
                        break 3;
                    }
                }

                break;
                
            // Weekly
            case 'week':
                $new_time = mktime(0, 0, 0, $event_start_month, $event_start_day + (7 * $recurrence_number), $event_start_year);
                $instance_start_date = date('Y', $new_time) . '-' . date('m', $new_time) . '-' . date('d', $new_time);
                break;

            // Monthly
            case 'month':
                switch ($recurrence_month_type) {
                    case 'day_of_the_month':
                        $new_time = mktime(0, 0, 0, $event_start_month + $recurrence_number, 1, $event_start_year);
                        $new_event_start_year = date('Y', $new_time);
                        $new_event_start_month = date('m', $new_time);
                        $new_event_start_day = $event_start_day;

                        // if date is not valid, then get last date for month
                        if (checkdate($new_event_start_month, $new_event_start_day, $new_event_start_year) == FALSE) {
                            $new_event_start_day = date('t', mktime(0, 0, 0, $new_event_start_month, 1, $new_event_start_year));
                        }

                        $instance_start_date = $new_event_start_year . '-' . $new_event_start_month . '-' . $new_event_start_day;

                        break;

                    case 'day_of_the_week':
                        $first_day_of_the_month_timestamp = mktime(0, 0, 0, $event_start_month + $recurrence_number, 1, $event_start_year);

                        // If the week is 1-4 then find the date in a certain way.
                        if ($week != 'last') {
                            $week_index = $week - 1;

                            $new_time = strtotime('+' . $week_index . ' week ' . $day_of_the_week, $first_day_of_the_month_timestamp);

                        // Otherwise the week is last, so find the date in a different way.
                        } else {
                            $last_day_of_the_month_timestamp = strtotime(date('Y-m-t', $first_day_of_the_month_timestamp));

                            // If the last day of the month happens to be the right day of the week,
                            // then thats that day that we want.
                            if (date('l', $last_day_of_the_month_timestamp) == $day_of_the_week) {
                                $new_time = $last_day_of_the_month_timestamp;

                            // Otherwise find the day of the week that we want in the last week of the month.
                            } else {
                                $new_time = strtotime('last ' . $day_of_the_week, $last_day_of_the_month_timestamp);
                            }
                        }

                        $instance_start_date = date('Y-m-d', $new_time);

                        break;
                }

                break;
            
            // Yearly
            case 'year':
                $new_event_start_year = $event_start_year + $recurrence_number;
                $new_event_start_month = $event_start_month;
                $new_event_start_day = $event_start_day;
                
                // if date is not valid, then get last date for month
                if (checkdate($new_event_start_month, $new_event_start_day, $new_event_start_year) == FALSE) {
                    $new_event_start_day = date('t', mktime(0, 0, 0, $new_event_start_month, 1, $new_event_start_year));
                }
                
                $instance_start_date = $new_event_start_year . '-' . $new_event_start_month . '-' . $new_event_start_day;
                break;
        }

        // if this instance is not an exception then continue to check it
        if (in_array($recurrence_number, $exceptions) == FALSE) {        
            // set the start date and time for this instance
            $instance_start_date_and_time = $instance_start_date . ' ' . $event_start_time;
            
            // set the end date and time for this instance by adding the difference in time to this instance's start date and time
            $instance_end_date_and_time = date('Y-m-d H:i:s', strtotime($instance_start_date_and_time) + $event_start_end_difference);
            
            // if this recurring instance is in the future, then return true
            if (strtotime($instance_end_date_and_time) >= time()) {
                return TRUE;
            }
        }
    }
    
    // if we have gotten here then there are no recurring instances in the future, so return false
    return FALSE;
}

// create function that will be responsible for checking if commissions need to be created from recurring profiles
function update_recurring_commissions()
{
    // if it is currently at least 24 hours after the last check, then complete check
    if (time() >= (LAST_RECURRING_COMMISSION_CHECK_TIMESTAMP + 86400)) {
        // update the config table to remember that the check has been completed today
        $query = "UPDATE config SET last_recurring_commission_check_timestamp = UNIX_TIMESTAMP()";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // get all enabled recurring commission profiles where the start date is today or in the past
        $query =
            "SELECT
                id,
                affiliate_code,
                order_id,
                amount,
                start_date,
                period,
                number_of_commissions
            FROM recurring_commission_profiles
            WHERE
                (enabled = '1')
                AND (start_date <= '" . date('Y-m-d') . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        $recurring_commission_profiles = array();
        
        // loop through profiles in order to add them to array
        while ($row = mysqli_fetch_assoc($result)) {
            $recurring_commission_profiles[] = $row;
        }
        
        // loop through the profiles in order to create commissions
        foreach ($recurring_commission_profiles as $recurring_commission_profile) {
            // get existing commissions for this profile, so we can determine if a commission already exists
            $query = "SELECT created_timestamp FROM commissions WHERE recurring_commission_profile_id = '" . $recurring_commission_profile['id'] . "'";
            $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
            
            $existing_commission_dates = array();
            
            // loop through commissions in order to add dates to array
            while ($row = mysqli_fetch_assoc($result)) {
                $existing_commission_dates[] = date('Y-m-d', $row['created_timestamp']);
            }
            
            // if the number of commissions is 0, then it is unlimited, so set it to a very high number
            if ($recurring_commission_profile['number_of_commissions'] == 0) {
                $recurring_commission_profile['number_of_commissions'] = 999999;
            }
            
            $start_date_parts = explode('-', $recurring_commission_profile['start_date']);
            $start_year = $start_date_parts[0];
            $start_month = $start_date_parts[1];
            $start_day = $start_date_parts[2];
            
            $commission_number = 1;
            
            // create a loop for the number of commissions in order to check if a commission needs to be created
            for ($commission_number = 1; $commission_number <= $recurring_commission_profile['number_of_commissions']; $commission_number++) {
                $instance_start_date = '';
                
                // if the commission number is 1, then set the instance start date to the start date for the profile
                if ($commission_number == 1) {
                    $instance_start_date = $recurring_commission_profile['start_date'];
                    
                // else the commission number is greater than 1, so calculate the start date based on the commission number
                } else {
                    $recurrence_number = $commission_number - 1;
                    
                    // get the date for the commission differently based on the period
                    switch ($recurring_commission_profile['period']) {
                        case 'monthly':
                            $instance_time = mktime(0, 0, 0, $start_month + $recurrence_number, 1, $start_year);
                            $instance_start_year = date('Y', $instance_time);
                            $instance_start_month = date('m', $instance_time);
                            $instance_start_day = $start_day;

                            // if date is not valid, then get last date for month
                            if (checkdate($instance_start_month, $instance_start_day, $instance_start_year) == false) {
                                $instance_start_day = date('t', mktime(0, 0, 0, $instance_start_month, 1, $instance_start_year));
                            }

                            $instance_start_date = $instance_start_year . '-' . $instance_start_month . '-' . $instance_start_day;
                            break;
                        
                        case 'yearly':
                            $instance_start_year = $start_year + $recurrence_number;
                            $instance_start_month = $start_month;
                            $instance_start_day = $start_day;
                            
                            // if date is not valid, then get last date for month
                            if (checkdate($instance_start_month, $instance_start_day, $instance_start_year) == false) {
                                $instance_start_day = date('t', mktime(0, 0, 0, $instance_start_month, 1, $instance_start_year));
                            }
                            
                            $instance_start_date = $instance_start_year . '-' . $instance_start_month . '-' . $instance_start_day;
                            break;
                    }
                    
                    // if the instance start date is greater than today, then we are in the future, so we can continue to the next profile
                    if ($instance_start_date > date('Y-m-d')) {
                        continue 2;
                    }
                }
                
                // if there is not already an existing commission for this instance, then create commission record
                if (in_array($instance_start_date, $existing_commission_dates) == FALSE) {
                    $query =
                        "INSERT INTO commissions (
                            reference_code,
                            affiliate_code,
                            order_id,
                            amount,
                            status,
                            recurring_commission_profile_id,
                            created_timestamp,
                            last_modified_timestamp)
                        VALUES (
                            '" . generate_commission_reference_code() . "',
                            '" . escape($recurring_commission_profile['affiliate_code']) . "',
                            '" . $recurring_commission_profile['order_id'] . "',
                            '" . $recurring_commission_profile['amount'] . "',
                            'pending',
                            '" . $recurring_commission_profile['id'] . "',
                            '" . strtotime($instance_start_date) . "',
                            UNIX_TIMESTAMP())";
                    $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
                }
            }
        }
    }
}

// create function that replaces the actual path in the content from a rich-text editor with the {path} placeholder,
// so that the actual path is not stored in the database
function prepare_rich_text_editor_content_for_input($content)
{
    $content = str_replace('href="' . PATH, 'href="{path}', $content);
    $content = str_replace('src="' . PATH, 'src="{path}', $content);
    $content = str_replace('value="' . PATH, 'value="{path}', $content);
    
    return $content;
}

// create function that will replace the {path} placeholder with the actual path so that the content works correctly
// in the rich-text editor
function prepare_rich_text_editor_content_for_output($content)
{
    $content = str_replace('href="{path}', 'href="' . PATH, $content);
    $content = str_replace('src="{path}', 'src="' . PATH, $content);
    $content = str_replace('value="{path}', 'value="' . PATH, $content);
    
    return $content;
}

function get_shipping_tracking_url($number)
{
    // if the shipping carrier is UPS, then return tracking URL for it
    if (preg_match('/\b(1Z ?[0-9A-Z]{3} ?[0-9A-Z]{3} ?[0-9A-Z]{2} ?[0-9A-Z]{4} ?[0-9A-Z]{3} ?[0-9A-Z]|[\dT]\d\d\d ?\d\d\d\d ?\d\d\d)\b/', $number) == 1) {
        return 'https://wwwapps.ups.com/WebTracking/processInputRequest?HTMLVersion=5.0&error_carried=true&tracknums_displayed=5&TypeOfInquiryNumber=T&loc=en_US&InquiryNumber1=' . urlencode($number);
        
    // Otherwise if the shipping carrier is FedEx, then return tracking URL for it.
    // We include the FedEx SmartPost support below, because if we don't then they will match USPS,
    // and the FedEx tracking for SmartPost is much better than the USPS tracking.  For example,
    // the USPS tracking won't contain any info until USPS gets the package from FedEx.
    } else if (
        (preg_match('/(\b96\d{20}\b)|(\b\d{15}\b)|(\b\d{12}\b)/', $number) == 1)
        || (preg_match('/\b((98\d\d\d\d\d?\d\d\d\d|98\d\d) ?\d\d\d\d ?\d\d\d\d( ?\d\d\d)?)\b/', $number) == 1)
        || (preg_match('/^[0-9]{15}$/', $number) == 1)
        || (preg_match('/^927489\d{16}$/', $number) == 1) // FedEx SmartPost
        || (preg_match('/^926129\d{16}$/', $number) == 1) // FedEx SmartPost
    ) {
        return 'https://www.fedex.com/Tracking?tracknumbers=' . urlencode($number) . '&action=track';
        
    // else if the shipping carrier is USPS, then return tracking URL for it
    } else if (
        (preg_match('/(\b\d{30}\b)|(\b91\d+\b)|(\b\d{20}\b)/', $number) == 1)
        || (preg_match('/^E\D{1}\d{9}\D{2}$|^9\d{15,21}$/', $number) == 1)
        || (preg_match('/^91[0-9]+$/', $number) == 1)
        || (preg_match('/^[A-Za-z]{2}[0-9]+US$/', $number) == 1)
    ) {
        return 'https://tools.usps.com/go/TrackConfirmAction?qtc_tLabels1=' . urlencode($number);
        
    // else a shipping carrier was not found, so return empty string
    } else {
        return '';
    }
}

// create function to detect and set device type (i.e. desktop or mobile) in session and cookie if it has not already been done
function initialize_device_type()
{
    // if the device type (desktop or mobile) for this visitor's session is not known yet,
    // then get device type
    if (isset($_SESSION['software']['device_type']) == FALSE) {
        // If mobile site setting is disabled,
        // or if this script is being run by the update search index process,
        // then don't detect device type and just set it to desktop.
        if (
            (MOBILE == false)
            || (defined('UPDATE_SEARCH_INDEX') and UPDATE_SEARCH_INDEX)
        ) {
            $_SESSION['software']['device_type'] = 'desktop';

        // Otherwise detect device type.
        } else {
            // if the visitor's device type is stored in a cookie, then use that
            if (isset($_COOKIE['software']['device_type']) == TRUE) {
                $_SESSION['software']['device_type'] = $_COOKIE['software']['device_type'];
            
            // else the visitor's device type is not stored in a cookie, so determine if we should detect it
            } else {
                // if the device type is mobile (based on the user agent), then set that in the session
                // we got the following code from detectmobilebrowsers.com
                if (
                    (isset($_SERVER['HTTP_USER_AGENT']) == TRUE)
                    &&
                    (
                        (preg_match('/android.+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|symbian|treo|up\.(browser|link)|vodafone|wap|windows (ce|phone)|xda|xiino/i',$_SERVER['HTTP_USER_AGENT']))
                        || (preg_match('/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|e\-|e\/|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(di|rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|xda(\-|2|g)|yas\-|your|zeto|zte\-/i',mb_substr($_SERVER['HTTP_USER_AGENT'],0,4)))
                    )
                ) {
                    $_SESSION['software']['device_type'] = 'mobile';

                // else the device type is desktop so set that in the session
                } else {
                    $_SESSION['software']['device_type'] = 'desktop';
                }

                // store the device type in a cookie for 10 years
                // so that we can remember the device type for the next visit
                setcookie('software[device_type]', $_SESSION['software']['device_type'], time() + 315360000, '/');
            }
        }
    }
}

// Create function that will generate a token and add it to a visitor's session if it has not already been done,
// in order to prevent CSRF attacks.
function initialize_token()
{
    // If a token has not been generated for this visitor yet,
    // then generate token and set it in session.
    if (isset($_SESSION['software']['token']) == FALSE) {
        $_SESSION['software']['token'] = md5(uniqid(mt_rand(), TRUE));
    }
}

// Create function in order to get hidden form field with a token in order to prevent CSRF attacks.
// This is used in almost all post requests.
function get_token_field() {
    return '<input type="hidden" name="token" value="' . $_SESSION['software']['token'] . '">';
}

// Create function in order to get query string field with a token in order to prevent CSRF attacks.
// This is used in get requests that change things.
function get_token_query_string_field()
{
    return '&amp;token=' . $_SESSION['software']['token'];
}

// Create function in order to validate the passed token in order to prevent CSRF attacks.
// This is used for almost all post requests and some get requests that change things.
function validate_token_field()
{

    // If the token does not exist in the session,
    // or the passed token does not match the token from the session,
    // then this might be a CSRF attack so output error.
    // We don't log activity anymore for this because the log was getting filled up with these messages
    // when we tried to log it in the past.  Innocent activities (e.g. session expired, web crawlers, spammers)
    // and not CSRF attacks generate most of these errors, so it is not important to log them.
    if (
        ($_SESSION['software']['token'] == '')
        ||
        (
            ($_POST['token'] != $_SESSION['software']['token'])
            && ($_GET['token'] != $_SESSION['software']['token'])
        )
    ) {

        // If the visitor has cookies disabled, then output unique error for that condition.
        if (!isset($_COOKIE[session_name()])) {

            output_error('Sorry, we could not accept your request, because it appears that cookies are disabled in your web browser.  Please enable cookies and then <a href="javascript:history.go(-1)">go back</a> and try again. If the problem persists then you might need to refresh the page, after you go back.');

        // Otherwise cookies are enabled, but the token just does not match
        // so output error for that condition.
        } else {

            output_error('Sorry, we could not accept your request because it appears that your session expired or you logged out. We recommend that you <a href="javascript:history.go(-1)">go back</a> and try again. If the problem persists then you might need to refresh the page, after you go back. This issue might happen if you logged out in a different browser tab or window, or cleared your browser cookies.');

        }

    }

}

// Create function to handle remember me and set user information if user is logged in.
function initialize_user() {

    // if remember me is on and there is cookie login information,
    // and there is not session login information,
    // and this is not the update_search_index.php script running
    // (we don't want to populate session login info from the cookie if this
    // script is updating the search index because we don't want it to run as the user that called it)
    // then add cookie login information to session if login info is valid
    if (
        (REMEMBER_ME == TRUE)
        && (isset($_COOKIE['software']['username']) == TRUE)
        && (isset($_COOKIE['software']['password']) == TRUE)
        && (isset($_SESSION['sessionusername']) == FALSE)
        && (!defined('UPDATE_SEARCH_INDEX') or UPDATE_SEARCH_INDEX !== true)
    ) {
        // check to see if the login information is valid by trying to find a user
        $query =
            "SELECT
                user.user_id AS id,
                user.user_username AS username,
                user.user_email AS email_address,
                user.user_role AS role,
                user.user_home AS start_page_id,
                user.user_manage_ecommerce AS manage_ecommerce,
                user.manage_ecommerce_reports,
                user.user_manage_forms AS manage_forms,
                user.timezone,
                contacts.id AS contact_id,
                contacts.member_id AS member_id,
                contacts.expiration_date AS expiration_date
            FROM user
            LEFT JOIN contacts ON user.user_contact = contacts.id
            WHERE
                (user.user_username = '" . escape($_COOKIE['software']['username']) . "')
                AND (user.user_password = '" . escape($_COOKIE['software']['password']) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');
        
        // if the login information in the cookie is valid,
        // then add login information to session, log activity, and store user info
        if (mysqli_num_rows($result) > 0) {
            $_SESSION['sessionusername'] = $_COOKIE['software']['username'];
            $_SESSION['sessionpassword'] = $_COOKIE['software']['password'];
            
            // create log entry to note that user logged in
            log_activity('user logged in', $_SESSION['sessionusername']);

            $user = mysqli_fetch_assoc($result);
            
        // else the login information in the cookie is not valid, so delete cookie
        } else {
            $current_timestamp = time();

            setcookie('software[username]', '', $current_timestamp - 1000, '/');
            setcookie('software[password]', '', $current_timestamp - 1000, '/');
        }

    // Otherwise if there is login info from an API request or the session then check if login
    // info is valid by trying to find a user.
    } else if (defined('API_USERNAME') or isset($_SESSION['sessionusername'])) {

        if (defined('API_USERNAME')) {
            $username = API_USERNAME;
            $password = API_PASSWORD;

        } else {
            $username = $_SESSION['sessionusername'];
            $password = $_SESSION['sessionpassword'];
        }

        $query =
            "SELECT
                user.user_id AS id,
                user.user_username AS username,
                user.user_email AS email_address,
                user.user_role AS role,
                user.user_home AS start_page_id,
                user.user_manage_ecommerce AS manage_ecommerce,
                user.manage_ecommerce_reports,
                user.user_manage_forms AS manage_forms,
                user.timezone,
                contacts.id AS contact_id,
                contacts.member_id AS member_id,
                contacts.expiration_date AS expiration_date
            FROM user
            LEFT JOIN contacts ON user.user_contact = contacts.id
            WHERE
                (user.user_username = '" . e($username) . "')
                AND (user.user_password = '" . e($password) . "')";
        $result = mysqli_query(db::$con, $query) or output_error('Query failed.');

        // if the login is valid, then store user info
        if (mysqli_num_rows($result) > 0) {
            $user = mysqli_fetch_assoc($result);
        }
    }

    // if the user is logged in, then store user info in global constants to be used later
    if (isset($user) == TRUE) {
        define('USER_LOGGED_IN', TRUE);
        define('USER_ID', $user['id']);
        define('USER_USERNAME', $user['username']);
        define('USER_EMAIL_ADDRESS', $user['email_address']);
        define('USER_ROLE', $user['role']);
        define('USER_CONTACT_ID', $user['contact_id']);
        define('USER_MEMBER_ID', $user['member_id']);
        define('USER_EXPIRATION_DATE', $user['expiration_date']);

        // If the user is an active member then store that.
        if (
            (USER_MEMBER_ID != '')
            &&
            (
                (USER_EXPIRATION_DATE == '')
                || (USER_EXPIRATION_DATE == '0000-00-00')
                || (USER_EXPIRATION_DATE >= date('Y-m-d'))
            )
        ) {
            define('USER_MEMBER', true);

        // Otherwise the user is not an active member, so store that.
        } else {
            define('USER_MEMBER', false);
        }

        define('USER_START_PAGE_ID', $user['start_page_id']);

        if ((USER_ROLE < 3) || ($user['manage_ecommerce'] == 'yes')) {
            define('USER_MANAGE_ECOMMERCE', true);
        } else {
            define('USER_MANAGE_ECOMMERCE', false);
        }

        if ((USER_ROLE < 3) or $user['manage_ecommerce_reports']) {
            define('USER_MANAGE_ECOMMERCE_REPORTS', true);
        } else {
            define('USER_MANAGE_ECOMMERCE_REPORTS', false);
        }

        if ((USER_ROLE < 3) || ($user['manage_forms'] == 'yes')) {
            define('USER_MANAGE_FORMS', true);
        } else {
            define('USER_MANAGE_FORMS', false);
        }

        // If the user has a timezone set that is not the default,
        // and the PHP version is high enough to support a user timezone,
        // then set user timezone.
        if (($user['timezone'] != '') && (version_compare(PHP_VERSION, '5.2.0', '>=') == true)) {
            define('USER_TIMEZONE', $user['timezone']);

        } else {
            define('USER_TIMEZONE', '');
        }

    // else the user is not logged in, so store that
    } else {
        define('USER_LOGGED_IN', FALSE);
        define('USER_ID', '');
        define('USER_CONTACT_ID', '');
        define('USER_USERNAME', '');
        define('USER_EMAIL_ADDRESS', '');
    }
}

// Create function that will check if a visitor has view access to a folder.
// This does not just check private access.  It checks for all types of access control.
// Since any visitor can get access to registration or guest content by registering or choosing to be a guest,
// you can set $always_grant_access_for_registration_and_guest to true which will grant access
// regardless of whether the visitor is logged in or not.
function check_view_access($folder_id, $always_grant_access_for_registration_and_guest = false) {

    // If the user is logged in and the user is an administrator, designer, or manager,
    // then they have view access to all folders, so just grant access.
    if (USER_LOGGED_IN && (USER_ROLE < 3)) {
        return true;
    }

    // Assume that visitor does not have access until we find out otherwise.
    $access = false;

    // Check if visitor has access differently based on the access control type of the folder.
    switch (get_access_control_type($folder_id)) {
        case 'public':
            $access = true;
            break;
        
        case 'private':
            $access_check = check_private_access($folder_id);

            // If the visitor has private access to this folder, then visitor has access.
            if ($access_check['access'] == true) {
                $access = true;
            }
            
            break;
            
        case 'guest':
            // If the visitor should always be granted access for guest access control,
            // or if the visitor has logged in, or if the visitor has selected to be a guest,
            // then the visitor has access.
            if (
                ($always_grant_access_for_registration_and_guest == true)
                || (USER_LOGGED_IN == true)
                || ($_SESSION['software']['guest'] == true)
            ) {
                $access = true;
            }
            
            break;

        case 'registration':
            // If the visitor should always be granted access for registration access control,
            // or if the visitor has logged in, then the visitor has access.
            if (
                ($always_grant_access_for_registration_and_guest == true)
                || (USER_LOGGED_IN == true)
            ) {
                $access = true;
            }
            
            break;

        case 'membership':
            // If the visitor is logged in and is a member
            // or has edit access, then the visitor has access.
            if (
                (USER_LOGGED_IN == true)
                &&
                (
                    (USER_MEMBER == true)
                    || (check_edit_access($folder_id) == true)
                )
            ) {
                $access = true;
            }
            
            break;
    }

    return $access;
}

// Create function in order to check if a visitor has edit access to a folder.
// It returns true or false.
function check_edit_access($folder_id) {

    // Assume that the visitor does not have edit access until we find out otherwise.
    $access = false;

    // If the visitor is logged in, then continue to check if the visitor has edit access.
    if (USER_LOGGED_IN == true) {
        // If the user is a manager or above, then the user has edit access.
        if (USER_ROLE < 3) {
            $access = true;

        // Otherwise the user has a user role, so continue to check if user has edit access.
        } else {
            // Determine what type of access user has to folder.
            $row = db_item(
                "SELECT
                    aclfolder.aclfolder_rights AS rights,
                    folder.folder_parent AS parent_folder_id
                FROM aclfolder
                LEFT JOIN folder ON aclfolder.aclfolder_folder = folder.folder_id
                WHERE
                    (aclfolder.aclfolder_user = '" . USER_ID . "')
                    AND (aclfolder.aclfolder_folder = '" . escape($folder_id) . "')");

            $rights = $row['rights'];
            $parent_folder_id = $row['parent_folder_id'];

            // If this user has edit rights to this folder, then remember that.
            if ($rights == 2) {
                $access = true;

            // Otherwise we do not know if access has been granted, so if this is not the root folder
            // then use recursion to check for access in parent folder.
            } else {
                // If the parent folder has not been found yet, then get it.
                if ($parent_folder_id == '') {
                    $parent_folder_id = db_value("SELECT folder_parent AS parent_folder_id FROM folder WHERE folder_id = '" . escape($folder_id) . "'");
                }

                // If this is not the root folder, then use recursion to check parent folder for access.
                if ($parent_folder_id != 0) {
                    $access = check_edit_access($parent_folder_id);
                }
            }
        }
    }

    return $access;
}

// Create function in order to check if a visitor has access to a private folder.
// This function returns an array with two properties: "access" (set to true
// if the user has access and false if user does not have access) and "expired"
// (set to true if the access has expired and false otherwise).
// Visitors with edit rights to a folder also have private access to that folder.
function check_private_access($folder_id)
{
    $result = array();

    // Assume that the visitor does not have access and access has not expired until we find out otherwise.
    $result['access'] = false;
    $result['expired'] = false;

    // If the visitor is logged in, then continue to check if the visitor has private access.
    if (USER_LOGGED_IN == true) {
        // If the user is a manager or above, then the user has private access.
        if (USER_ROLE < 3) {
            $result['access'] = true;

        // Otherwise the user has a user role, so continue to check if user has private access.
        } else {
            // Determine what type of access user has to folder.
            $row = db_item(
                "SELECT
                    aclfolder.aclfolder_rights AS rights,
                    aclfolder.expiration_date,
                    folder.folder_parent AS parent_folder_id
                FROM aclfolder
                LEFT JOIN folder ON aclfolder.aclfolder_folder = folder.folder_id
                WHERE
                    (aclfolder.aclfolder_user = '" . USER_ID . "')
                    AND (aclfolder.aclfolder_folder = '" . escape($folder_id) . "')");

            $rights = $row['rights'];
            $expiration_date = $row['expiration_date'];
            $parent_folder_id = $row['parent_folder_id'];

            // If this user has edit rights to this folder, then they also have private access.
            if ($rights == 2) {
                $result['access'] = true;

            // Otherwise if this user has private access, then determine if access has expired.
            } else if ($rights == 1) {
                // If there is an expiration date and it has expired, then remember that.
                if (
                    ($expiration_date != '0000-00-00')
                    && ($expiration_date < date('Y-m-d'))
                ) {
                    $result['expired'] = true;

                // Otherwise the private access has not expired, so user has access.
                } else {
                    $result['access'] = true;
                }

            // Otherwise we do not know if access has been granted, so if this is not the root folder
            // then use recursion to check for access in parent folder.
            } else {
                // If the parent folder has not been found yet, then get it.
                if ($parent_folder_id == '') {
                    $parent_folder_id = db_value("SELECT folder_parent AS parent_folder_id FROM folder WHERE folder_id = '" . escape($folder_id) . "'");
                }

                // If this is not the root folder, then use recursion to check parent folder for access.
                if ($parent_folder_id != 0) {
                    $result = check_private_access($parent_folder_id);
                }
            }
        }
    }

    return $result;
}

// Create function that can be used to build the options for a pick list of themes.
function get_theme_options()
{
    $theme_options = array();
    $theme_options[''] = '';
    
    // Get all themes.
    $themes = db_items(
        "SELECT
            id,
            name
        FROM files
        WHERE
            (type = 'css')
            AND (design = '1')
        ORDER BY name ASC");

    // Loop through the themes in order to prepare options.
    foreach ($themes as $theme) {
        $theme_options[h($theme['name'])] = $theme['id'];
    }
    
    return $theme_options;
}

// Create function that can be used to build a URL.  You can pass in properties for each part of the URL
// and/or pass in a URL that you want this function to pull default parts from.
// Properties:
// url: an optional URL string that you want to use as a default.  If specific properties are not passed
// then the parts of this URL will be used.  You can pass all types of URL's for this property.
// (e.g. http://www.example.com, /example, example.html).
// scheme: http:// or https://
// hostname: example.com
// path: /example
// parameters: an array of name/value pairs.  If you include a url property that contains query string parameters
// then these parameters will be added to the passed url parameters.  If the same parameter appears in both then
// the parameter for this property will be used.  Don't urlencode the names or values (that is done for you).
// fragment: a string for the bookmark that will appear as "#example" (don't include "#").
function build_url($properties)
{
    $url = $properties['url'];
    $scheme = $properties['scheme'];
    $hostname = $properties['hostname'];
    $path = $properties['path'];
    $parameters = $properties['parameters'];
    $fragment = $properties['fragment'];

    // Get url parts in order to prepare URL.
    $url_parts = parse_url($url);

    $url_scheme = '';

    // If a scheme was passed, then use it.
    if ($scheme != '') {
        $url_scheme = $scheme;

    // Otherwise if the passed URL contains a scheme, then use it.
    } else if (isset($url_parts['scheme']) == true) {
        $url_scheme = $url_parts['scheme'] . '://';
    }

    $url_hostname = '';

    // If a hostname was passed, then use it.
    if ($hostname != '') {
        $url_hostname = $hostname;

    // Otherwise if the passed URL contains a hostname, then use it.
    } else if (isset($url_parts['host']) == true) {
        $url_hostname = $url_parts['host'];
    }

    $url_path = '';

    // If a path was passed, then use it.
    if ($path != '') {
        $url_path = $path;

    // Otherwise if the passed URL contains a path, then use it.
    } else if (isset($url_parts['path']) == true) {
        $url_path = $url_parts['path'];
    }

    $url_query_string = '';

    // If the URL contains a query string, then process it.
    if (isset($url_parts['query']) == true) {
        // Put query string parameters into an array in order to prepare new query string.
        parse_str($url_parts['query'], $query_string_parameters);

        // Loop through the query string parameters in order to build query string.
        foreach ($query_string_parameters as $name => $value) {
            // If this parameter is not one of the parameters that was passed into this function, then add it.
            if (
                (is_array($parameters) == false)
                || (array_key_exists($name, $parameters) == false)
            ) {
                // If this is the first item that is being added to the query string, then add question mark.
                if ($url_query_string == '') {
                    $url_query_string = '?';
                    
                // Otherwise this is not the first item that is being added to the query string, so add ampersand.
                } else {
                    $url_query_string .= '&';
                }
                
                $url_query_string .= urlencode($name) . '=' . urlencode($value);
            }
        }
    }

    // If parameters were passed into this function then loop through them in order to add them to query string.
    if (is_array($parameters) == true) {
        foreach ($parameters as $name => $value) {
            // If this is the first item that is being added to the query string, then add question mark.
            if ($url_query_string == '') {
                $url_query_string = '?';
                
            // Otherwise this is not the first item that is being added to the query string, so add ampersand.
            } else {
                $url_query_string .= '&';
            }
            
            $url_query_string .= urlencode($name) . '=' . urlencode($value);
        }
    }

    $url_fragment = '';

    // If a fragment was passed, then use it.
    if ($fragment != '') {
        $url_fragment = '#' . urlencode($fragment);

    // Otherwise if the passed URL contains a fragment, then use it.
    } else if (isset($url_parts['fragment']) == true) {
        $url_fragment = '#' . $url_parts['fragment'];
    }

    // If the path is blank and there is a query string or a fragment, then set path to slash.
    if (
        ($url_path == '')
        &&
        (
            ($url_query_string != '')
            || ($url_fragment != '')
        )
    ) {
        $url_path = '/';
    }

    return $url_scheme . $url_hostname . $url_path . $url_query_string . $url_fragment;
}

// Create function that can be used to build the options for a pick list of calendar events.
// Properties:
// reservations: true/false. Used to get events that have reservations enabled or disabled.
// Don't pass property in order to get all calendar events.
function get_calendar_event_options($properties)
{
    $reservations = $properties['reservations'];

    $calendar_event_options = array();

    // Add first blank option.
    $calendar_event_options[] = array(
        'label' => '',
        'value' => ''
    );

    $sql_reservations = "";

    // If a reservations property was passed, then filter calendar events.
    if (isset($properties['reservations']) == true) {
        if ($reservations == true) {
            $sql_reservations = "WHERE reservations = '1'";
        } else {
            $sql_reservations = "WHERE reservations = '0'";
        }
    }
    
    // Get calendar events in order to prepare options.
    $calendar_events = db_items(
        "SELECT
            id,
            name,
            short_description
        FROM calendar_events
        $sql_reservations
        ORDER BY name ASC");

    // Loop through the events in order to prepare options.
    foreach ($calendar_events as $calendar_event) {
        // If this user has access to this calendar event, then include it.
        if (validate_calendar_event_access($calendar_event['id']) == true) {
            $output_label = h($calendar_event['name']);

            // If there is a short description and it is not the same as the name, then add it to the label.
            if (
                ($calendar_event['short_description'] != '')
                && ($calendar_event['short_description'] != $calendar_event['name'])
            ) {
                $output_label .= ' - ' . h($calendar_event['short_description']);
            }

            $calendar_event_options[] = array(
                'label' => $output_label,
                'value' =>  $calendar_event['id']
            );
        }
    }
    
    return $calendar_event_options;
}

// Create function that can be used to build the options for a pick list of email campaign profiles.
function get_email_campaign_profile_options()
{
    $email_campaign_profile_options = array();
    $email_campaign_profile_options[''] = '';
    
    // Get email campaign profiles in order to prepare options.
    $email_campaign_profiles = db_items(
        "SELECT
            id,
            name,
            created_user_id
        FROM email_campaign_profiles
        ORDER BY name ASC");

    // Loop through the profiles in order to prepare options.
    foreach ($email_campaign_profiles as $email_campaign_profile) {
        // If this user has a role greater than a user role or if this user is the creator,
        // then add this profile as an option.
        if (
            (USER_ROLE < 3)
            || (USER_ID == $email_campaign_profile['created_user_id'])
        ) {
            $email_campaign_profile_options[h($email_campaign_profile['name'])] = $email_campaign_profile['id'];
        }
    }
    
    return $email_campaign_profile_options;
}

// Create function that is used to see if auto email campaigns need to be created after a specific action has occurred
// and then create them if necessary.
// Properties:
// action: "calendar_event_reserved", "custom_form_submitted", "email_campaign_sent", "order_abandoned", "order_completed", "order_shipped", "product_ordered"
// action_item_id: The id of item that is related to the action (e.g. the calendar event id)
// calendar_event_recurrence_number
// order_id: Used by order_abandoned action.
// contact_id: The ID of the contact that generated the action.
// email_address: You can pass an email address for recipient, instead of the contact above.
// If you pass both, then the contact email address will be used.
// fields: An array of mail-merge fields (e.g. so ^^order_number^^ can be replaced in subject/body).
// data: An array of misc data that will be passed to get_page_content(), so custom PHP on page can 
// use the data.

function create_auto_email_campaigns($properties) {

    $action = $properties['action'];
    $action_item_id = $properties['action_item_id'];
    $calendar_event_recurrence_number = $properties['calendar_event_recurrence_number'];
    $order_id = $properties['order_id'];
    $contact_id = $properties['contact_id'];
    $email_address = $properties['email_address'];
    $fields = $properties['fields'];
    $data = $properties['data'];

    // If a contact id was not passed and there is no email address, then abort.
    if (!$contact_id and !$email_address) {
        return;
    }

    $contact = array();

    // If a contact id was passed, then get contact info in order to determine if contact is
    // opted-out and to get e-mail address.
    if ($contact_id) {
        $contact = db_item(
            "SELECT
                id,
                opt_in,
                email_address
            FROM contacts
            WHERE id = '" . e($contact_id) . "'");
    }

    // If a contact email was found, then use that email address for recipient.
    if ($contact['email_address']) {
        $email_address = $contact['email_address'];

    // Otherwise if no email address was passed either, then return, because we have no one to email.
    } else if (!$email_address) {
        return;
    }

    // Get all enabled email campaign profiles for this action and action item.
    $email_campaign_profiles = db_items(
        "SELECT
            email_campaign_profiles.id,
            email_campaign_profiles.name,
            email_campaign_profiles.subject,
            email_campaign_profiles.format,
            email_campaign_profiles.body,
            email_campaign_profiles.page_id,
            page.page_name,
            email_campaign_profiles.from_name,
            email_campaign_profiles.from_email_address,
            email_campaign_profiles.reply_email_address,
            email_campaign_profiles.bcc_email_address,
            email_campaign_profiles.schedule_time,
            email_campaign_profiles.schedule_length,
            email_campaign_profiles.schedule_unit,
            email_campaign_profiles.schedule_period,
            email_campaign_profiles.schedule_base,
            email_campaign_profiles.purpose,
            email_campaign_profiles.created_user_id
        FROM email_campaign_profiles
        LEFT JOIN page ON email_campaign_profiles.page_id = page.page_id
        WHERE
            (email_campaign_profiles.enabled = '1')
            AND (email_campaign_profiles.action = '$action')
            AND (email_campaign_profiles.action_item_id = '" . e($action_item_id) . "')
        ORDER BY email_campaign_profiles.name ASC");

    // Loop through the profiles in order to create auto email campaigns.
    foreach ($email_campaign_profiles as $email_campaign_profile) {

        // If the purpose is commercial and there is a contact email and the contact is opted-out,
        // then we don't want to send email to recipient, so skip to the next profile.
        if (
            $email_campaign_profile['purpose'] == 'commercial'
            and $contact['email_address']
            and $contact['opt_in'] != 1
        ) {
            continue;
        }

        $subject = $email_campaign_profile['subject'];

        // If there are mail-merge fields, then replace in subject.
        if ($fields) {
            $subject = replace_variables(array(
                'content' => $subject,
                'fields' => $fields,
                'format' => 'plain_text'));
        }

        // If plain text was selected for the format, then store body in variable
        // and clear page id so that we don't store it with the e-mail campaign.
        if ($email_campaign_profile['format'] == 'plain_text') {
            
            $body = $email_campaign_profile['body'];
            $email_campaign_profile['page_id'] = '';

        // Otherwise HTML was selected for the format, so prepare body for that format.
        } else {

            require_once(dirname(__FILE__) . '/get_page_content.php');

            // Get html for page.
            $body = get_page_content($email_campaign_profile['page_id'], $system_content = '', $extra_system_content = '', $mode = 'preview', $email = true, $data);
            
            // Find if there is a base tag in the HTML.
            $base_in_html = preg_match('/<\s*base\s+[^>]*href\s*=\s*["\'](?:http:\/\/|https:\/\/|ftp:\/\/).*?["\']/is', $body);

            // If there is not a base tag in the HTML, add base tag and convert relative links to absolute links.
            if (!$base_in_html) {
                $base = '<head>' . "\n" . '<base href="' . URL_SCHEME . HOSTNAME_SETTING . '/" />';
                $body = preg_replace('/<head>/i', $base, $body);

                // Change relative URLs to absolute URLs for links.
                $body = preg_replace('/(<\s*a\s+[^>]*href\s*=\s*["\'])(?!ftp:\/\/|https:\/\/|mailto:|http:\/\/)(?:\/|\.\.\/|\.\/|)(.*?["\'].*?>)/is', "$1" . URL_SCHEME . HOSTNAME_SETTING . "/$2", $body);

                // Change relative URLs to absolute URLs for images.
                $body = preg_replace('/(<\s*img\s+[^>]*src\s*=\s*["\'])(?!http:\/\/|https:\/\/)(?:\/|\.\.\/|\.\/|)(.*?["\'].*?>)/is', "$1" . URL_SCHEME . HOSTNAME_SETTING . "/$2", $body);

                // Change relative URLs to absolute URLs for CSS background images.
                $body = preg_replace('/(background-image\s*:\s*url\s*\(\s*(?:"|\'|))(?!http:\/\/|https:\/\/)(?:\/|\.\.\/|\.\/|)(.*?(?:"|\'|).*?\))/is', "$1" . URL_SCHEME . HOSTNAME_SETTING . "/$2", $body);

                // Change relative URLs to absolute URLs for HTML background images.
                $body = preg_replace('/(background\s*=\s*["\'])(?!http:\/\/|https:\/\/)(?:\/|\.\.\/|\.\/|)(.*?["\'])/is', "$1" . URL_SCHEME . HOSTNAME_SETTING . "/$2", $body);
            }

            $commercial_content = '';

            // If the campaign is commercial, then add tracking codes to links and prepare
            // commercial content for footer.
            if ($email_campaign_profile['purpose'] == 'commercial') {
            
                // Get all links in order to add tracking codes to links.
                preg_match_all('/(<\s*a\s+[^>]*href\s*=\s*["\']\s*)(.*?)(\s*["\'].*?>)/is', $body, $links);

                // If the date format is month and then day, then use that format.
                if (DATE_FORMAT == 'month_day') {
                    $month_and_day_format = 'm/d';

                // Otherwise the date format is day and then month, so use that format.
                } else {
                    $month_and_day_format = 'd/m';
                }

                // Set tracking code to contain the page name and date and time.
                $tracking_code = $email_campaign_profile['page_name'] . '_' . date($month_and_day_format . '/Y_h:i_A');

                // loop through all links in order to add tracking codes to links
                foreach ($links[0] as $key => $link) {
                    // set the URL that was found in the link
                    $url = $links[2][$key];
                    
                    // remove new lines from the link
                    $url = str_replace("\r\n", '', $url);
                    $url = str_replace("\n", '', $url);
                    
                    $url_parts = @parse_url($url);
                    
                    // if the URL is valid
                    // and if there is not a scheme or the scheme is http or https
                    // and if there is not a hostname or the hostname is this site's hostname
                    // and if there is not already a tracking code in the URL
                    // then continue with adding tracking code to URL
                    if (
                        ($url_parts != false)
                        && ((isset($url_parts['scheme']) == false) || ($url_parts['scheme'] == '') || (mb_strtolower($url_parts['scheme']) == 'http') || (mb_strtolower($url_parts['scheme']) == 'https'))
                        && ((isset($url_parts['host']) == false) || ($url_parts['host'] == '') || (mb_strtolower(str_replace('www.', '', $url_parts['host'])) == mb_strtolower(str_replace('www.', '', HOSTNAME_SETTING))))
                        && ((isset($url_parts['query']) == false) || ($url_parts['query'] == '') || mb_strpos($url_parts['query'], 't=') === false)
                    ) {
                        $new_url = '';
                        
                        // if there is a scheme, then add scheme to new URL
                        if ((isset($url_parts['scheme']) == true) && ($url_parts['scheme'] != '')) {
                            $new_url .= $url_parts['scheme'] . '://';
                        }
                        
                        // if there is a hostname, then add hostname to new URL
                        if ((isset($url_parts['host']) == true) && ($url_parts['host'] != '')) {
                            $new_url .= $url_parts['host'];
                        }
                        
                        // if there is a path, then add path to new URL
                        if ((isset($url_parts['path']) == true) && ($url_parts['path'] != '')) {
                            $new_url .= $url_parts['path'];
                        }
                        
                        $new_url .= '?';
                        
                        // if there is a query string, then add query string and ampersand to new URL
                        if ((isset($url_parts['query']) == true) && ($url_parts['query'] != '')) {
                            $new_url .= $url_parts['query'] . '&amp;';
                        }
                        
                        $new_url .= 't=' . h(urlencode($tracking_code));
                        
                        // if there is a bookmark, then add bookmark to the new URL
                        if ((isset($url_parts['fragment']) == true) && ($url_parts['fragment'] != '')) {
                            $new_url .= '#' . $url_parts['fragment'];
                        }
                        
                        $entire_link = $links[0][$key];
                        $link_start = $links[1][$key];
                        $link_end = $links[3][$key];        
                        
                        // replace the link with the new link
                        $body = str_replace($entire_link, $link_start . $new_url . $link_end, $body);
                    }
                }

                $commercial_content =
                    h(ORGANIZATION_NAME) . '
                    ' . h(ORGANIZATION_ADDRESS_1) . '
                    ' . h(ORGANIZATION_ADDRESS_2) . '
                    ' . h(ORGANIZATION_CITY) . ' ' . h(ORGANIZATION_STATE) . ' ' . h(ORGANIZATION_ZIP_CODE) . ' ' . h(ORGANIZATION_COUNTRY) . '<br>';

                // If there is a contact, then include email preferences info.
                if ($contact) {

                    $email_preferences_url = URL_SCHEME . HOSTNAME_SETTING . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/email_preferences.php?id=<email_address_id></email_address_id>';

                    $commercial_content .= '<a href="' . $email_preferences_url . '" style="color: #666666">Update email preferences</a> or <a href="' . $email_preferences_url . '" style="color: #666666">unsubscribe</a><br>';
                }
            }
            
            // get URL for page
            $page_url = URL_SCHEME . HOSTNAME_SETTING . OUTPUT_PATH . OUTPUT_SOFTWARE_DIRECTORY . '/view_email_campaign.php?r=<reference_code></reference_code>';
            
            $footer =
                '<div class="software_email_footer" style="font-family: arial; font-size: 11px; color: #666666; text-align: center; background-color: #ffffff; padding: 5px; margin-top: 15px">
                    <a href="' . $page_url . '" style="color: #666666">View this email at our site</a><br>
                    ' . $commercial_content . '
                </div>
                </body>';
            $body = preg_replace('/<\/body>/i', $footer, $body);
        }

        // Wrap long lines (RFC 821).
        $body = wordwrap($body, 900, "\n", 1);

        // If there are mail-merge fields, then replace in body.
        if ($fields) {
            $body = replace_variables(array(
                'content' => $body,
                'fields' => $fields,
                'format' => $email_campaign_profile['format']));
        }

        // Prepare start date and time for email campaign.
        // We start with the schedule base and then use the length of the schedule to calculate the start timestamp.
        // The calculated start timestamp goes back or forward at least the amount of the schedule length (e.g. 5 days)
        // and then might go back more (for "before") or forward more (for "after") so that the time matches the schedule time override (e.g. 12:00 PM).

        $current_timestamp = time();

        // Set base timestamp differently based on the selected base.
        switch ($email_campaign_profile['schedule_base']) {
            // If the schedule base is the action, then set the base timestamp to the current timestamp.
            case 'action':
                $base_timestamp = $current_timestamp;
                break;
            
            // If the schedule base is calendar_event_start_time then set the base timestamp to that.
            case 'calendar_event_start_time':
                // Get calendar event info in order to get start date and time.
                $calendar_event = get_calendar_event($action_item_id, $calendar_event_recurrence_number);

                $base_timestamp = strtotime($calendar_event['start_date_and_time']);
                break;
        }

        // Set start timestamp differently based on the period.
        switch ($email_campaign_profile['schedule_period']) {
            case 'before':
                // If there is a schedule length, then calculate the start timestamp.
                if ($email_campaign_profile['schedule_length'] != 0) {
                    $start_timestamp = strtotime('-' . $email_campaign_profile['schedule_length'] . ' ' . $email_campaign_profile['schedule_unit'], $base_timestamp);

                // Otherwise there is not a schedule length, so just set the start timestamp to the base timestamp.
                } else {
                    $start_timestamp = $base_timestamp;
                }

                // If there is a schedule time, and the calculated start time does not happen to equal that time,
                // then adjust the start timestamp to be at the schedule time.
                if (
                    ($email_campaign_profile['schedule_time'] != '00:00:00')
                    && (date('H:i:s', $start_timestamp) != $email_campaign_profile['schedule_time'])
                ) {
                    // If the schedule time comes before the calculated start time, then set the start timestamp
                    // to the schedule time on the same calculated date.
                    if ($email_campaign_profile['schedule_time'] < date('H:i:s', $start_timestamp)) {
                        $start_timestamp = strtotime(date('Y-m-d', $start_timestamp) . ' ' . $email_campaign_profile['schedule_time'], $start_timestamp);

                    // Otherwise the schedule time comes after the calculated start time, so set the start timestamp
                    // to the schedule time on the day before the calculated date.
                    } else {
                        $start_timestamp = strtotime('-1 day', $start_timestamp);
                        $start_timestamp = strtotime(date('Y-m-d', $start_timestamp) . ' ' . $email_campaign_profile['schedule_time'], $start_timestamp);
                    }
                }

                break;
            
            case 'after':
                // If there is a schedule length, then calculate the start timestamp.
                if ($email_campaign_profile['schedule_length'] != 0) {
                    $start_timestamp = strtotime('+' . $email_campaign_profile['schedule_length'] . ' ' . $email_campaign_profile['schedule_unit'], $base_timestamp);

                // Otherwise there is not a schedule length, so just set the start timestamp to the base timestamp.
                } else {
                    $start_timestamp = $base_timestamp;
                }

                // If there is a schedule time, and the calculated start time does not happen to equal that time,
                // then adjust the start timestamp to be at the schedule time.
                if (
                    ($email_campaign_profile['schedule_time'] != '00:00:00')
                    && (date('H:i:s', $start_timestamp) != $email_campaign_profile['schedule_time'])
                ) {
                    // If the schedule time comes after the calculated start time, then set the start timestamp
                    // to the schedule time on the same calculated date.
                    if ($email_campaign_profile['schedule_time'] > date('H:i:s', $start_timestamp)) {
                        $start_timestamp = strtotime(date('Y-m-d', $start_timestamp) . ' ' . $email_campaign_profile['schedule_time'], $start_timestamp);

                    // Otherwise the schedule time comes before the calculated start time, so set the start timestamp
                    // to the schedule time on the day after the calculated date.
                    } else {
                        $start_timestamp = strtotime('+1 day', $start_timestamp);
                        $start_timestamp = strtotime(date('Y-m-d', $start_timestamp) . ' ' . $email_campaign_profile['schedule_time'], $start_timestamp);
                    }
                }

                break;
        }

        // If the campaign is set to be scheduled for the present or future, then create campaign.
        // We don't create campaigns if the scheduled date and time is in the past.
        if ($start_timestamp >= $current_timestamp) {
            $start_date_and_time = date('Y-m-d H:i:s', $start_timestamp);
            
            // Create e-mail campaign.
            db(
                "INSERT INTO email_campaigns (
                    type,
                    email_campaign_profile_id,
                    action,
                    action_item_id,
                    calendar_event_recurrence_number,
                    order_id,
                    from_name,
                    from_email_address,
                    reply_email_address,
                    bcc_email_address,
                    subject,
                    format,
                    body,
                    page_id,
                    start_time,
                    status,
                    purpose,
                    created_user_id,
                    created_timestamp,
                    last_modified_user_id,
                    last_modified_timestamp)
                VALUES (
                    'automatic',
                    '" . $email_campaign_profile['id'] . "',
                    '" . $action . "',
                    '" . $action_item_id . "',
                    '" . $calendar_event_recurrence_number . "',
                    '" . e($order_id) . "',
                    '" . e($email_campaign_profile['from_name']) . "',
                    '" . e($email_campaign_profile['from_email_address']) . "',
                    '" . e($email_campaign_profile['reply_email_address']) . "',
                    '" . e($email_campaign_profile['bcc_email_address']) . "',
                    '" . e($subject) . "',
                    '" . $email_campaign_profile['format'] . "',
                    '" . e($body) . "',
                    '" . $email_campaign_profile['page_id'] . "',
                    '$start_date_and_time',
                    '$status',
                    '" . e($email_campaign_profile['purpose']) . "',
                    '" . $email_campaign_profile['created_user_id'] . "',
                    UNIX_TIMESTAMP(),
                    '" . $email_campaign_profile['created_user_id'] . "',
                    UNIX_TIMESTAMP())");
            
            $email_campaign_id = mysqli_insert_id(db::$con);

            // Create record for e-mail recipient.
            db(
                "INSERT INTO email_recipients (
                    email_campaign_id,
                    type,
                    email_address,
                    contact_id)
                VALUES (
                    '$email_campaign_id',
                    'automatic',
                    '" . e($email_address) . "',
                    '" . e($contact['id']) . "')");

            // Get user-friendly action name for log message.
            switch ($action) {
                case 'calendar_event_reserved':
                    $log_action = 'calendar event reserved';
                    break;

                case 'custom_form_submitted':
                    $log_action = 'custom form submitted';
                    break;

                case 'email_campaign_sent':
                    $log_action = 'auto campaign sent';
                    break;

                case 'order_abandoned':
                    $log_action = 'order abandoned';
                    break;

                case 'order_completed':
                    $log_action = 'order completed';
                    break;

                case 'order_shipped':
                    $log_action = 'order shipped';
                    break;

                case 'product_ordered':
                    $log_action = 'product ordered';
                    break;
            }
            
            log_activity('auto campaign for profile (' . $email_campaign_profile['name'] . ') was created because of an action (' . $log_action . ')', $_SESSION['sessionusername']);
        }
    }
}

// Create function that is used to replace variables with data (e.g. ^^name^^ in email campaigns)
// Properties:
// content: The content that you want to replace variables in.
// fields: An array of fields
//     name: The system name of the field (e.g. "name")
//     data: The content/data that the variable should be replaced with.
//     type: The type of field (e.g. "standard")
// format: "plain_text" or "html"
function replace_variables($properties)
{
    $content = $properties['content'];
    $fields = $properties['fields'];
    $format = $properties['format'];

    // if the content contains variables then replace them with data
    if (mb_strpos($content, '^^') !== false) {
        // Set prepare for html value based on the format.
        switch ($format) {
            case 'plain_text':
                $prepare_for_html = false;
                break;
            
            case 'html':
                $prepare_for_html = true;
                break;
        }

        // Get all conditionals.
        preg_match_all('/\[\[(.*?\^\^(.*?)\^\^.*?)(\|\|(.*?))?\]\]/si', $content, $conditionals, PREG_SET_ORDER);
        
        // Loop through all conditionals in order to replace them with either the positive or negative version
        // depending on whether there is data for a field or not.
        foreach ($conditionals as $conditional) {
            $whole_string = $conditional[0];
            $positive_string = $conditional[1];
            $negative_string = $conditional[4];
            $field_name = $conditional[2];
            
            // Assume that the field name is not valid, until we find out otherwise.
            $field_name_valid = false;
            
            // Loop through the fields in order to determine if the field name is valid
            // and store field data in $field variable for use below.
            foreach ($fields as $field) {
                // If this field name matches the field name from the variable, then the field name from the variable is valid,
                // so remember that and break out of the loop.
                if ($field['name'] == $field_name) {
                    $field_name_valid = true;
                    break;
                }
            }
            
            // If the field name is valid, then replace conditional.
            if ($field_name_valid == true) {
                // If there is data to output, use first part of conditional.
                if ($field['data'] != '') {
                    $content = str_replace($whole_string, $positive_string, $content);
                    
                // Otherwise there is no data to output, so use second part of conditional.
                } else {
                    $content = str_replace($whole_string, $negative_string, $content);
                }
            }
        }

        // Get all variables so they can be replaced with data.
        preg_match_all('/\^\^(.*?)\^\^(%%(.*?)%%)?/i', $content, $variables, PREG_SET_ORDER);

        // Loop through the variables in order to replace them with data
        foreach ($variables as $variable) {
            $whole_string = $variable[0];
            $field_name = $variable[1];

            $date_format = '';

            // If a date format was passed along with the variable, then store that.
            if (isset($variable[3]) == true) {
                $date_format = $variable[3];

                // If the format is html, then that means we have to use unhtmlspecialchars()
                // because the date format might contain HTML entities (e.g. &lt;)
                // and the date() function would convert those characters into date elements.
                if ($format == 'html') {
                    $date_format = unhtmlspecialchars($date_format);
                }
            }

            // Assume that the field name is not valid, until we find out otherwise.
            $field_name_valid = false;

            // Loop through the fields in order to determine if the field name is valid
            // and store field data in $field variable for use below.
            foreach ($fields as $field) {
                // If this field name matches the field name from the variable, then the field name from the variable is valid,
                // so remember that and break out of the loop.
                if ($field['name'] == $field_name) {
                    $field_name_valid = true;
                    break;
                }
            }

            // If the field name is valid, then continue to replace variable with data.
            if ($field_name_valid == true) {
                // Start data off with the field data.
                $data = $field['data'];

                // Update the data so that it is ready for output (e.g. convert dates to fiendly formats).
                $data = prepare_form_data_for_output($data, $field['type'], $prepare_for_html, $date_format);
                
                // Replace the variable with the data.
                // We can't use str_replace() for this because we need to limit the number of replacements to 1
                // in order to prevent bugs where it will replace variables further below that might have date formats.
                $content = preg_replace('/' . preg_quote($whole_string, '/') . '/', addcslashes($data, '\\$'), $content, 1);
            }
        }
    }
    
    return $content;
}

// Create function to take a PHP array and create JSON content. This function will use the built-in
// function if it is available in PHP (normally v5.2+) or include the JSON libary if not.
// The built-in function is much faster, which is why we do this check.
function encode_json($value)
{
    if (function_exists('json_encode') == true) {
        return json_encode($value);

    } else {
        include_once('JSON.php');
        $json = new Services_JSON(SERVICES_JSON_LOOSE_TYPE);
        return $json->encode($value);
    }
}

// Create function to take JSON content and convert it to a PHP array. This function will use the built-in
// function if it is available in PHP (normally v5.2+) or include the JSON libary if not.
// The built-in function is much faster, which is why we do this check.
function decode_json($content)
{
    if (function_exists('json_decode') == true) {
        return json_decode($content, true);
        
    } else {
        include_once('JSON.php');
        $json = new Services_JSON(SERVICES_JSON_LOOSE_TYPE);
        return $json->decode($content);
    }
}

// Create function that will allow us to get a unique name for an item
// in order to prevent multiple items from having the same name.
// Properties:
// name: the name of the item that you are attempting to use.
// type: the type of item (e.g. common region)

function get_unique_name($properties) {

    $name = $properties['name'];
    $type = $properties['type'];

    // Determine if name is available in different ways depending on the type.
    switch ($type) {
        // For pages and files, we use a special function because
        // we need to make sure there are no name space conflicts in web root.
        case 'page':
        case 'file':
            if (check_name_availability(array('name' => $name)) == true) {
                return $name;
            }

            break;

        case 'common_region':
        case 'designer_region':
            if (db_value("SELECT COUNT(*) FROM cregion WHERE cregion_name = '" . escape($name) . "'") == 0) {
                return $name;
            }

            break;

        case 'folder':
            if (db_value("SELECT COUNT(*) FROM folder WHERE folder_name = '" . escape($name) . "'") == 0) {
                return $name;
            }

            break;

        case 'menu':
            if (db_value("SELECT COUNT(*) FROM menus WHERE name = '" . e($name) . "'") == 0) {
                return $name;
            }

            break;

        case 'style':
            if (db_value("SELECT COUNT(*) FROM style WHERE style_name = '" . escape($name) . "'") == 0) {
                return $name;
            }

            break;

        case 'product':
            if (db_value("SELECT COUNT(*) FROM products WHERE name = '" . escape($name) . "'") == 0) {
                return $name;
            }

            break;

        case 'product_group':
            if (db_value("SELECT COUNT(*) FROM product_groups WHERE name = '" . escape($name) . "'") == 0) {
                return $name;
            }

            break;

        case 'email_campaign_profile':
            if (db_value("SELECT COUNT(*) FROM email_campaign_profiles WHERE name = '" . e($name) . "'") == 0) {
                return $name;
            }

            break;
    }

    // If we have gotten here, then the name is already in use,
    // so prepare new name differently based on the item type.
    switch ($type) {
        default:
            // If there is already a bracket area on the end of the name,
            // then just increase number in bracket.
            if (preg_match('/(.*?)\[(\d+)\]$/', $name, $matches) == 1) {
                $new_name = $matches[1] . '[' . ($matches[2] + 1) . ']';

            // Otherwise there is not already a bracket area on the end of the name,
            // so add bracket area with a 1.
            } else {
                $new_name = $name . '[1]';
            }

            break;

       