<?php
/*-------------------------------------------------------+
| PHPFusion Content Management System
| Copyright (C) PHP Fusion Inc
| https://phpfusion.com/
+--------------------------------------------------------+
| Filename: weblinks_submissions.php
| Author: Core Development Team
+--------------------------------------------------------+
| This program is released as free software under the
| Affero GPL license. You can redistribute it and/or
| modify it under the terms of this license which you
| can read by viewing the included agpl.txt or online
| at www.gnu.org/licenses/agpl.html. Removal of this
| copyright header is strictly prohibited without
| written permission from the original author(s).
+--------------------------------------------------------*/
namespace PHPFusion\Weblinks;

class WeblinksSubmissionsAdmin extends WeblinksAdminModel {
    private static $instance = NULL;
    private $inputArray = [];
    private $locale = [];
    private $dataUser = [];
    private $submit_id = 0;

    public static function getInstance() {
        if (self::$instance == NULL) {
            self::$instance = new static();
        }
        return self::$instance;
    }

    /**
     * Handle Preview and Publish of a Weblink Submission
     */
    private function postSubmission() {

        if (check_post('publish_submission')) {

            // Check posted information
            $this->inputArray = [
                'weblink_name'        => sanitizer('weblink_name', '', 'weblink_name'),
                'weblink_description' => sanitizer('weblink_description', "", "weblink_description"),
                'weblink_url'         => sanitizer('weblink_url', "", 'weblink_url'),
                'weblink_cat'         => sanitizer('weblink_cat', 0, 'weblink_cat'),
                'weblink_datestamp'   => sanitizer('weblink_datestamp', time(), 'weblink_datestamp'),
                'weblink_visibility'  => sanitizer('weblink_visibility', 0, 'weblink_visibility'),
                'weblink_status'      => sanitizer('weblink_status', 0, 'weblink_status'),
                'weblink_count'       => "0",
                'weblink_language'    => sanitizer('weblink_language', LANGUAGE, 'weblink_language'),
                'weblink_user_name'   => sanitizer('weblink_user_name', '', 'weblink_user_name'),
            ];
            // Handle
            if (fusion_safe()) {
                // Publish Submission
                dbquery("DELETE FROM ".DB_SUBMISSIONS." WHERE submit_id=:submitid AND submit_type=:submittype", [':submitid' => $this->submit_id, ':submittype' => 'l']);
                dbquery_insert(DB_WEBLINKS, $this->inputArray, 'save');
                addnotice('success', $this->locale['WLS_0060']);
                redirect(clean_request('', ['submit_id'], FALSE));
            }
        }
    }

    /**
     * Delete a Weblink Submission
     */
    private function deleteSubmission() {
        if (check_post('delete_submission')) {
            $submit_id = get('submit_id', FILTER_VALIDATE_INT);
            dbquery("DELETE FROM ".DB_SUBMISSIONS." WHERE submit_id = :submitid AND submit_type = :submittype", [':submitid' => (int)$submit_id, ':submittype' => 'l']);
            addnotice('success', $this->locale['WLS_0061']);
            redirect(clean_request('', ['submit_id'], FALSE));
        }
    }

    /**
     * Get unserialize Datas for a Submission
     */
    private function unserializeData() {
        $result = dbquery("SELECT s.*
            FROM ".DB_SUBMISSIONS." AS s
            WHERE s.submit_id = :submitid
            LIMIT 0,1
        ", [':submitid' => $this->submit_id]);

        if (dbrows($result) > 0) {
            $data = dbarray($result);
            $submit_criteria = \Defender::decode($data['submit_criteria']);
            return [
                'weblink_user_name'   => $data['submit_user'],
                'weblink_name'        => $submit_criteria['weblink_name'],
                'weblink_cat'         => $submit_criteria['weblink_cat'],
                'weblink_description' => phpentities(stripslashes($submit_criteria['weblink_description'])),
                'weblink_url'         => $submit_criteria['weblink_url'],
                'weblink_visibility'  => 0,
                'weblink_language'    => $submit_criteria['weblink_language'],
                'weblink_datestamp'   => $data['submit_datestamp'],
                'weblink_user'        => $data['submit_user'],
                'weblink_status'      => 0
            ];
        } else {
            redirect(clean_request("", [], FALSE));
        }

        return NULL;
    }

    /**
     * Display Form
     */
    private function displayForm() {
        // Textarea Settings
        if (!fusion_get_settings("tinymce_enabled")) {
            $weblinkSnippetSettings = [
                'required'    => TRUE,
                'preview'     => TRUE,
                'html'        => TRUE,
                'autosize'    => TRUE,
                'placeholder' => $this->locale['WLS_0255'],
                'error_text'  => $this->locale['WLS_0270']
            ];
        } else {
            $weblinkSnippetSettings = [
                'required'   => TRUE,
                'type'       => "tinymce",
                'tinymce'    => "advanced",
                'error_text' => $this->locale['WLS_0270']
            ];
        }

        // Start Form
        echo openform('submissionform', 'post', FUSION_REQUEST);
        echo form_hidden('weblink_status', '', 1);
        echo form_hidden('weblink_user_name', '', $this->inputArray['weblink_user_name']);
        ?>
        <div class="well clearfix">
            <div class="pull-left">
                <?php echo display_avatar($this->dataUser, "30px", "", FALSE, "img-rounded m-t-5 m-r-5"); ?>
            </div>
            <div class="overflow-hide">
                <?php
                $submissionUser = ($this->dataUser['user_name'] != $this->locale['user_na'] ? profile_link($this->dataUser['user_id'], $this->dataUser['user_name'], $this->dataUser['user_status']) : $this->locale['user_na']);
                $submissionDate = showdate("shortdate", $this->inputArray['weblink_datestamp']);
                $submissionTime = timer($this->inputArray['weblink_datestamp']);

                $replacements = ["{%SUBMISSION_AUTHOR%}" => $submissionUser, "{%SUBMISSION_DATE%}" => $submissionDate, "{%SUBMISSION_TIME%}" => $submissionTime];
                $submissionInfo = strtr($this->locale['WLS_0350']."<br />".$this->locale['WLS_0351'], $replacements);

                echo $submissionInfo;
                ?>
            </div>
        </div>
        <?php ?>

        <!-- Display Form -->
        <div class="row">

            <!-- Display Left Column -->
            <div class="col-xs-12 col-sm-12 col-md-7 col-lg-8">
                <?php

                echo form_text('weblink_name', $this->locale['WLS_0201'], $this->inputArray['weblink_name'], [
                    'required'    => TRUE,
                    'placeholder' => $this->locale['WLS_0201'],
                    'error_text'  => $this->locale['WLS_0252']
                ]);

                echo form_text('weblink_url', $this->locale['WLS_0253'], $this->inputArray['weblink_url'], [
                    'required'    => TRUE,
                    'type'        => "url",
                    'placeholder' => "http://"
                ]);

                echo form_textarea('weblink_description', $this->locale['WLS_0254'], $this->inputArray['weblink_description'], $weblinkSnippetSettings);

                ?>
            </div>

            <!-- Display Right Column -->
            <div class="col-xs-12 col-sm-12 col-md-5 col-lg-4">
                <?php

                openside($this->locale['WLS_0260']);

                echo form_select_tree('weblink_cat', $this->locale['WLS_0101'], $this->inputArray['weblink_cat'], [
                    'no_root'     => TRUE,
                    'inner_width' => '100%',
                    'placeholder' => $this->locale['choose'],
                    'query'       => (multilang_table("WL") ? "WHERE ".in_group('weblink_cat_language', LANGUAGE) : "")
                ], DB_WEBLINK_CATS, "weblink_cat_name", "weblink_cat_id", "weblink_cat_parent");

                echo form_select('weblink_visibility', $this->locale['WLS_0103'], $this->inputArray['weblink_visibility'], [
                    'options'     => fusion_get_groups(),
                    'placeholder' => $this->locale['choose'],
                    'inner_width' => "100%"
                ]);

                if (multilang_table("WL")) {
                    echo form_select('weblink_language[]', $this->locale['language'], $this->inputArray['weblink_language'], [
                        'options'     => fusion_get_enabled_languages(),
                        'placeholder' => $this->locale['choose'],
                        'inner_width' => '100%',
                        'multiple'    => TRUE
                    ]);
                } else {
                    echo form_hidden('weblink_language', '', $this->inputArray['weblink_language']);
                }

                echo form_datepicker('weblink_datestamp', $this->locale['WLS_0203'], $this->inputArray['weblink_datestamp']);
                closeside();

                ?>

            </div>
        </div>
        <?php
        echo "<div class='m-t-20'>";
        echo form_button('publish_submission', $this->locale['publish'], $this->locale['publish'], [
            'class' => "btn-success m-r-10",
            'icon'  => "fa fa-fw fa-hdd-o"
        ]);
        echo form_button('delete_submission', $this->locale['delete'], $this->locale['delete'], [
            'class' => "btn-danger m-r-10",
            'icon'  => "fa fa-fw fa-trash"
        ]);
        echo "</div>";
        echo closeform();
    }

    /**
     * Display List with Submissions
     */
    private function displaySubmissionList() {
        $result = dbquery("
            SELECT s.submit_id, s.submit_criteria, s.submit_datestamp, u.user_id, u.user_name, u.user_status, u.user_avatar
            FROM ".DB_SUBMISSIONS." AS s
            LEFT JOIN ".DB_USERS." AS u ON u.user_id=s.submit_user
            WHERE s.submit_type='l'
            ORDER BY submit_datestamp DESC
        ");
        ?>

        <!-- Display Table -->
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                <tr>
                    <td class="strong"><?php echo $this->locale['WLS_0200']; ?></td>
                    <td class="strong col-xs-5"><?php echo $this->locale['WLS_0201'] ?></td>
                    <td class="strong"><?php echo $this->locale['WLS_0202'] ?></td>
                    <td class="strong"><?php echo $this->locale['WLS_0203'] ?></td>
                    <td class="strong"><?php echo $this->locale['WLS_0204'] ?></td>
                </tr>
                </thead>
                <tbody>
                <?php if (dbrows($result) > 0) :
                    while ($data = dbarray($result)) : ?>
                        <?php
                        $submitData = \Defender::decode($data['submit_criteria']);

                        $submitUser = $this->locale['user_na'];
                        if ($data['user_name']) {
                            $submitUser = display_avatar($data, "20px", "", FALSE, "img-rounded m-r-5");
                            $submitUser .= profile_link($data['user_id'], $data['user_name'], $data['user_status']);
                        }

                        $reviewLink = clean_request("section=submissions&submit_id=".$data['submit_id'], ["section", "ref", "action", "submit_id"], FALSE);
                        ?>
                        <tr>
                            <td>#<?php echo $data['submit_id']; ?></td>
                            <td><span class="text-dark"><?php echo $submitData['weblink_name']; ?></span></td>
                            <td><?php echo $submitUser; ?></td>
                            <td><?php echo timer($data['submit_datestamp']); ?></td>
                            <td>
                                <a href="<?php echo $reviewLink; ?>" title="<?php echo $this->locale['WLS_0205']; ?>"
                                   class="btn btn-default btn-sm"><i
                                            class="fa fa-fw fa-eye"></i> <?php echo $this->locale['WLS_0205']; ?></a>
                            </td>
                        </tr>
                    <?php
                    endwhile;
                else: ?>
                    <tr>
                        <td colspan="5" class="text-center"><?php echo $this->locale['WLS_0062']; ?></td>
                    </tr>
                <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * Display Admin Area
     */
    public function displayWeblinksAdmin() {
        pageaccess("W");

        $this->locale = self::getWeblinkAdminLocale();

        // Handle a Submission
        $this->submit_id = get('submit_id', FILTER_VALIDATE_INT);

        if (!empty($this->submit_id) && dbcount("(submit_id)", DB_SUBMISSIONS, "submit_id=:submitid AND submit_type=:submittype", [':submitid' => (int)$this->submit_id, ':submittype' => 'l'])) {
            $this->inputArray = self::unserializeData();

            // Get Infos about Submissioner
            $resultUser = dbquery("SELECT user_id, user_name, user_status, user_avatar FROM ".DB_USERS." WHERE user_id=:userid LIMIT 0,1", [':userid' => $this->inputArray['weblink_user']]);
            if (dbrows($resultUser) > 0) {
                $this->dataUser = dbarray($resultUser);
            } else {
                $this->dataUser = ['user_id' => $this->inputArray['weblink_name'], 'user_name' => $this->locale['user_na'], 'user_status' => 0, 'user_avatar' => FALSE];
            }

            // Delete, Publish, Preview
            self::deleteSubmission();
            self::postSubmission();

            // Display Form with Buttons
            self::displayForm();

            // Display List
        } else {
            self::displaySubmissionList();
        }
    }
}
