/**
 * Internal dependencies
 */
import {
	extractResourceNameFromRoute,
	getRouteIds,
	simplifyRouteWithId,
} from '../utils';

describe( 'extractResourceNameFromRoute', () => {
	it.each`
		namespace        | route                                                                            | expected
		${ 'kkart/blocks' } | ${ 'kkart/blocks/products' }                                                        | ${ 'products' }
		${ 'kkart/other' }  | ${ 'kkart/blocks/product' }                                                         | ${ 'kkart/blocks/product' }
		${ 'kkart/blocks' } | ${ 'kkart/blocks/products/attributes/(?P<attribute_id>[\\d]+)' }                    | ${ 'products/attributes' }
		${ 'kkart/blocks' } | ${ 'kkart/blocks/products/attributes/(?P<attribute_id>[\\d]+)/terms' }              | ${ 'products/attributes/terms' }
		${ 'kkart/blocks' } | ${ 'kkart/blocks/products/attributes/(?P<attribute_id>[\\d]+)/terms/(?P<id>[d]+)' } | ${ 'products/attributes/terms' }
	`(
		'returns "$expected" when namespace is "$namespace" and route is "$route"',
		( { namespace, route, expected } ) => {
			expect( extractResourceNameFromRoute( namespace, route ) ).toBe(
				expected
			);
		}
	);
} );

describe( 'getRouteIds', () => {
	it.each`
		route                                                                              | expected
		${ 'kkart/blocks/products' }                                                          | ${ [] }
		${ 'kkart/blocks/products/(?P<id>[\\d]+)' }                                           | ${ [ 'id' ] }
		${ 'kkart/blocks/products/attributes/(?P<attribute_id>[\\d]+)/terms/(?P<id>[\\d]+)' } | ${ [ 'attribute_id', 'id' ] }
	`(
		'returns "$expected" when route is "$route"',
		( { route, expected } ) => {
			expect( getRouteIds( route ) ).toEqual( expected );
		}
	);
} );

describe( 'simplifyRouteWithId', () => {
	it.each`
		route                                                                              | matchIds                      | expected
		${ 'kkart/blocks/products' }                                                          | ${ [] }                       | ${ 'kkart/blocks/products' }
		${ 'kkart/blocks/products/attributes/(?P<attribute_id>[\\d]+)' }                      | ${ [ 'attribute_id' ] }       | ${ 'kkart/blocks/products/attributes/{attribute_id}' }
		${ 'kkart/blocks/products/attributes/(?P<attribute_id>[\\d]+)/terms' }                | ${ [ 'attribute_id' ] }       | ${ 'kkart/blocks/products/attributes/{attribute_id}/terms' }
		${ 'kkart/blocks/products/attributes/(?P<attribute_id>[\\d]+)/terms/(?P<id>[\\d]+)' } | ${ [ 'attribute_id', 'id' ] } | ${ 'kkart/blocks/products/attributes/{attribute_id}/terms/{id}' }
		${ 'kkart/blocks/products/attributes/(?P<attribute_id>[\\d]+)/terms/(?P<id>[\\d]+)' } | ${ [ 'id', 'attribute_id' ] } | ${ 'kkart/blocks/products/attributes/{attribute_id}/terms/{id}' }
	`(
		'returns "$expected" when route is "$route" and matchIds is "$matchIds"',
		( { route, matchIds, expected } ) => {
			expect( simplifyRouteWithId( route, matchIds ) ).toBe( expected );
		}
	);
} );
