<?php
/**
 * Personal data exporters.
 *
 * @since 3.4.0
 * @package Kkart\Classes
 */

defined( 'ABSPATH' ) || exit;

/**
 * KKART_Privacy_Exporters Class.
 */
class KKART_Privacy_Exporters {
	/**
	 * Finds and exports customer data by email address.
	 *
	 * @since 3.4.0
	 * @param string $email_address The user email address.
	 * @return array An array of personal data in name value pairs
	 */
	public static function customer_data_exporter( $email_address ) {
		$user           = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
		$data_to_export = array();

		if ( $user instanceof WP_User ) {
			$customer_personal_data = self::get_customer_personal_data( $user );
			if ( ! empty( $customer_personal_data ) ) {
				$data_to_export[] = array(
					'group_id'          => 'kkart_customer',
					'group_label'       => __( 'Customer Data', 'kkart' ),
					'group_description' => __( 'User&#8217;s Kkart customer data.', 'kkart' ),
					'item_id'           => 'user',
					'data'              => $customer_personal_data,
				);
			}
		}

		return array(
			'data' => $data_to_export,
			'done' => true,
		);
	}

	/**
	 * Finds and exports data which could be used to identify a person from Kkart data associated with an email address.
	 *
	 * Orders are exported in blocks of 10 to avoid timeouts.
	 *
	 * @since 3.4.0
	 * @param string $email_address The user email address.
	 * @param int    $page  Page.
	 * @return array An array of personal data in name value pairs
	 */
	public static function order_data_exporter( $email_address, $page ) {
		$done           = true;
		$page           = (int) $page;
		$user           = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
		$data_to_export = array();
		$order_query    = array(
			'limit'    => 10,
			'page'     => $page,
			'customer' => array( $email_address ),
		);

		if ( $user instanceof WP_User ) {
			$order_query['customer'][] = (int) $user->ID;
		}

		$orders = kkart_get_orders( $order_query );

		if ( 0 < count( $orders ) ) {
			foreach ( $orders as $order ) {
				$data_to_export[] = array(
					'group_id'          => 'kkart_orders',
					'group_label'       => __( 'Orders', 'kkart' ),
					'group_description' => __( 'User&#8217;s Kkart orders data.', 'kkart' ),
					'item_id'           => 'order-' . $order->get_id(),
					'data'              => self::get_order_personal_data( $order ),
				);
			}
			$done = 10 > count( $orders );
		}

		return array(
			'data' => $data_to_export,
			'done' => $done,
		);
	}

	/**
	 * Finds and exports customer download logs by email address.
	 *
	 * @since 3.4.0
	 * @param string $email_address The user email address.
	 * @param int    $page  Page.
	 * @throws Exception When KKART_Data_Store validation fails.
	 * @return array An array of personal data in name value pairs
	 */
	public static function download_data_exporter( $email_address, $page ) {
		$done            = true;
		$page            = (int) $page;
		$user            = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
		$data_to_export  = array();
		$downloads_query = array(
			'limit' => 10,
			'page'  => $page,
		);

		if ( $user instanceof WP_User ) {
			$downloads_query['user_id'] = (int) $user->ID;
		} else {
			$downloads_query['user_email'] = $email_address;
		}

		$customer_download_data_store     = KKART_Data_Store::load( 'customer-download' );
		$customer_download_log_data_store = KKART_Data_Store::load( 'customer-download-log' );
		$downloads                        = $customer_download_data_store->get_downloads( $downloads_query );

		if ( 0 < count( $downloads ) ) {
			foreach ( $downloads as $download ) {
				$data_to_export[] = array(
					'group_id'          => 'kkart_downloads',
					/* translators: This is the headline for a list of downloads purchased from the store for a given user. */
					'group_label'       => __( 'Purchased Downloads', 'kkart' ),
					'group_description' => __( 'User&#8217;s Kkart purchased downloads data.', 'kkart' ),
					'item_id'           => 'download-' . $download->get_id(),
					'data'              => self::get_download_personal_data( $download ),
				);

				$download_logs = $customer_download_log_data_store->get_download_logs_for_permission( $download->get_id() );

				foreach ( $download_logs as $download_log ) {
					$data_to_export[] = array(
						'group_id'          => 'kkart_download_logs',
						/* translators: This is the headline for a list of access logs for downloads purchased from the store for a given user. */
						'group_label'       => __( 'Access to Purchased Downloads', 'kkart' ),
						'group_description' => __( 'User&#8217;s Kkart access to purchased downloads data.', 'kkart' ),
						'item_id'           => 'download-log-' . $download_log->get_id(),
						'data'              => array(
							array(
								'name'  => __( 'Download ID', 'kkart' ),
								'value' => $download_log->get_permission_id(),
							),
							array(
								'name'  => __( 'Timestamp', 'kkart' ),
								'value' => $download_log->get_timestamp(),
							),
							array(
								'name'  => __( 'IP Address', 'kkart' ),
								'value' => $download_log->get_user_ip_address(),
							),
						),
					);
				}
			}
			$done = 10 > count( $downloads );
		}

		return array(
			'data' => $data_to_export,
			'done' => $done,
		);
	}

	/**
	 * Get personal data (key/value pairs) for a user object.
	 *
	 * @since 3.4.0
	 * @param WP_User $user user object.
	 * @throws Exception If customer cannot be read/found and $data is set to KKART_Customer class.
	 * @return array
	 */
	protected static function get_customer_personal_data( $user ) {
		$personal_data = array();
		$customer      = new KKART_Customer( $user->ID );

		if ( ! $customer ) {
			return array();
		}

		$props_to_export = apply_filters(
			'kkart_privacy_export_customer_personal_data_props',
			array(
				'billing_first_name'  => __( 'Billing First Name', 'kkart' ),
				'billing_last_name'   => __( 'Billing Last Name', 'kkart' ),
				'billing_company'     => __( 'Billing Company', 'kkart' ),
				'billing_address_1'   => __( 'Billing Address 1', 'kkart' ),
				'billing_address_2'   => __( 'Billing Address 2', 'kkart' ),
				'billing_city'        => __( 'Billing City', 'kkart' ),
				'billing_postcode'    => __( 'Billing Postal/Zip Code', 'kkart' ),
				'billing_state'       => __( 'Billing State', 'kkart' ),
				'billing_country'     => __( 'Billing Country / Region', 'kkart' ),
				'billing_phone'       => __( 'Phone Number', 'kkart' ),
				'billing_email'       => __( 'Email Address', 'kkart' ),
				'shipping_first_name' => __( 'Shipping First Name', 'kkart' ),
				'shipping_last_name'  => __( 'Shipping Last Name', 'kkart' ),
				'shipping_company'    => __( 'Shipping Company', 'kkart' ),
				'shipping_address_1'  => __( 'Shipping Address 1', 'kkart' ),
				'shipping_address_2'  => __( 'Shipping Address 2', 'kkart' ),
				'shipping_city'       => __( 'Shipping City', 'kkart' ),
				'shipping_postcode'   => __( 'Shipping Postal/Zip Code', 'kkart' ),
				'shipping_state'      => __( 'Shipping State', 'kkart' ),
				'shipping_country'    => __( 'Shipping Country / Region', 'kkart' ),
			),
			$customer
		);

		foreach ( $props_to_export as $prop => $description ) {
			$value = '';

			if ( is_callable( array( $customer, 'get_' . $prop ) ) ) {
				$value = $customer->{"get_$prop"}( 'edit' );
			}

			$value = apply_filters( 'kkart_privacy_export_customer_personal_data_prop_value', $value, $prop, $customer );

			if ( $value ) {
				$personal_data[] = array(
					'name'  => $description,
					'value' => $value,
				);
			}
		}

		/**
		 * Allow extensions to register their own personal data for this customer for the export.
		 *
		 * @since 3.4.0
		 * @param array    $personal_data Array of name value pairs.
		 * @param KKART_Order $order A customer object.
		 */
		$personal_data = apply_filters( 'kkart_privacy_export_customer_personal_data', $personal_data, $customer );

		return $personal_data;
	}

	/**
	 * Get personal data (key/value pairs) for an order object.
	 *
	 * @since 3.4.0
	 * @param KKART_Order $order Order object.
	 * @return array
	 */
	protected static function get_order_personal_data( $order ) {
		$personal_data   = array();
		$props_to_export = apply_filters(
			'kkart_privacy_export_order_personal_data_props',
			array(
				'order_number'               => __( 'Order Number', 'kkart' ),
				'date_created'               => __( 'Order Date', 'kkart' ),
				'total'                      => __( 'Order Total', 'kkart' ),
				'items'                      => __( 'Items Purchased', 'kkart' ),
				'customer_ip_address'        => __( 'IP Address', 'kkart' ),
				'customer_user_agent'        => __( 'Browser User Agent', 'kkart' ),
				'formatted_billing_address'  => __( 'Billing Address', 'kkart' ),
				'formatted_shipping_address' => __( 'Shipping Address', 'kkart' ),
				'billing_phone'              => __( 'Phone Number', 'kkart' ),
				'billing_email'              => __( 'Email Address', 'kkart' ),
			),
			$order
		);

		foreach ( $props_to_export as $prop => $name ) {
			$value = '';

			switch ( $prop ) {
				case 'items':
					$item_names = array();
					foreach ( $order->get_items() as $item ) {
						$item_names[] = $item->get_name() . ' x ' . $item->get_quantity();
					}
					$value = implode( ', ', $item_names );
					break;
				case 'date_created':
					$value = kkart_format_datetime( $order->get_date_created(), get_option( 'date_format' ) . ', ' . get_option( 'time_format' ) );
					break;
				case 'formatted_billing_address':
				case 'formatted_shipping_address':
					$value = preg_replace( '#<br\s*/?>#i', ', ', $order->{"get_$prop"}() );
					break;
				default:
					if ( is_callable( array( $order, 'get_' . $prop ) ) ) {
						$value = $order->{"get_$prop"}();
					}
					break;
			}

			$value = apply_filters( 'kkart_privacy_export_order_personal_data_prop', $value, $prop, $order );

			if ( $value ) {
				$personal_data[] = array(
					'name'  => $name,
					'value' => $value,
				);
			}
		}

		// Export meta data.
		$meta_to_export = apply_filters(
			'kkart_privacy_export_order_personal_data_meta',
			array(
				'Payer first name'     => __( 'Payer first name', 'kkart' ),
				'Payer last name'      => __( 'Payer last name', 'kkart' ),
				'Payer PayPal address' => __( 'Payer PayPal address', 'kkart' ),
				'Transaction ID'       => __( 'Transaction ID', 'kkart' ),
			)
		);

		if ( ! empty( $meta_to_export ) && is_array( $meta_to_export ) ) {
			foreach ( $meta_to_export as $meta_key => $name ) {
				$value = apply_filters( 'kkart_privacy_export_order_personal_data_meta_value', $order->get_meta( $meta_key ), $meta_key, $order );

				if ( $value ) {
					$personal_data[] = array(
						'name'  => $name,
						'value' => $value,
					);
				}
			}
		}

		/**
		 * Allow extensions to register their own personal data for this order for the export.
		 *
		 * @since 3.4.0
		 * @param array    $personal_data Array of name value pairs to expose in the export.
		 * @param KKART_Order $order An order object.
		 */
		$personal_data = apply_filters( 'kkart_privacy_export_order_personal_data', $personal_data, $order );

		return $personal_data;
	}

	/**
	 * Get personal data (key/value pairs) for a download object.
	 *
	 * @since 3.4.0
	 * @param KKART_Order $download Download object.
	 * @return array
	 */
	protected static function get_download_personal_data( $download ) {
		$personal_data = array(
			array(
				'name'  => __( 'Download ID', 'kkart' ),
				'value' => $download->get_id(),
			),
			array(
				'name'  => __( 'Order ID', 'kkart' ),
				'value' => $download->get_order_id(),
			),
			array(
				'name'  => __( 'Product', 'kkart' ),
				'value' => get_the_title( $download->get_product_id() ),
			),
			array(
				'name'  => __( 'User email', 'kkart' ),
				'value' => $download->get_user_email(),
			),
			array(
				'name'  => __( 'Downloads remaining', 'kkart' ),
				'value' => $download->get_downloads_remaining(),
			),
			array(
				'name'  => __( 'Download count', 'kkart' ),
				'value' => $download->get_download_count(),
			),
			array(
				'name'  => __( 'Access granted', 'kkart' ),
				'value' => date( 'Y-m-d', $download->get_access_granted( 'edit' )->getTimestamp() ),
			),
			array(
				'name'  => __( 'Access expires', 'kkart' ),
				'value' => ! is_null( $download->get_access_expires( 'edit' ) ) ? date( 'Y-m-d', $download->get_access_expires( 'edit' )->getTimestamp() ) : null,
			),
		);

		/**
		 * Allow extensions to register their own personal data for this download for the export.
		 *
		 * @since 3.4.0
		 * @param array    $personal_data Array of name value pairs to expose in the export.
		 * @param KKART_Order $order An order object.
		 */
		$personal_data = apply_filters( 'kkart_privacy_export_download_personal_data', $personal_data, $download );

		return $personal_data;
	}

	/**
	 * Finds and exports payment tokens by email address for a customer.
	 *
	 * @since 3.4.0
	 * @param string $email_address The user email address.
	 * @param int    $page  Page.
	 * @return array An array of personal data in name value pairs
	 */
	public static function customer_tokens_exporter( $email_address, $page ) {
		$user           = get_user_by( 'email', $email_address ); // Check if user has an ID in the DB to load stored personal data.
		$data_to_export = array();

		if ( ! $user instanceof WP_User ) {
			return array(
				'data' => $data_to_export,
				'done' => true,
			);
		}

		$tokens = KKART_Payment_Tokens::get_tokens(
			array(
				'user_id' => $user->ID,
				'limit'   => 10,
				'page'    => $page,
			)
		);

		if ( 0 < count( $tokens ) ) {
			foreach ( $tokens as $token ) {
				$data_to_export[] = array(
					'group_id'          => 'kkart_tokens',
					'group_label'       => __( 'Payment Tokens', 'kkart' ),
					'group_description' => __( 'User&#8217;s Kkart payment tokens data.', 'kkart' ),
					'item_id'           => 'token-' . $token->get_id(),
					'data'              => array(
						array(
							'name'  => __( 'Token', 'kkart' ),
							'value' => $token->get_display_name(),
						),
					),
				);
			}
			$done = 10 > count( $tokens );
		} else {
			$done = true;
		}

		return array(
			'data' => $data_to_export,
			'done' => $done,
		);
	}
}
