<?php
/**
 * Installation related functions and actions.
 *
 * @package Kkart\Classes
 * @version 3.0.0
 */

use Automattic\Jetpack\Constants;
use Automattic\Kkart\Internal\KKARTCom\ConnectionHelper as KKARTConnectionHelper;

defined( 'ABSPATH' ) || exit;

/**
 * KKART_Install Class.
 */
class KKART_Install {

	/**
	 * DB updates and callbacks that need to be run per version.
	 *
	 * @var array
	 */
	private static $db_updates = array(
		'2.0.0' => array(
			'kkart_update_200_file_paths',
			'kkart_update_200_permalinks',
			'kkart_update_200_subcat_display',
			'kkart_update_200_taxrates',
			'kkart_update_200_line_items',
			'kkart_update_200_images',
			'kkart_update_200_db_version',
		),
		'2.0.9' => array(
			'kkart_update_209_brazillian_state',
			'kkart_update_209_db_version',
		),
		'2.1.0' => array(
			'kkart_update_210_remove_pages',
			'kkart_update_210_file_paths',
			'kkart_update_210_db_version',
		),
		'2.2.0' => array(
			'kkart_update_220_shipping',
			'kkart_update_220_order_status',
			'kkart_update_220_variations',
			'kkart_update_220_attributes',
			'kkart_update_220_db_version',
		),
		'2.3.0' => array(
			'kkart_update_230_options',
			'kkart_update_230_db_version',
		),
		'2.4.0' => array(
			'kkart_update_240_options',
			'kkart_update_240_shipping_methods',
			'kkart_update_240_api_keys',
			'kkart_update_240_refunds',
			'kkart_update_240_db_version',
		),
		'2.4.1' => array(
			'kkart_update_241_variations',
			'kkart_update_241_db_version',
		),
		'2.5.0' => array(
			'kkart_update_250_currency',
			'kkart_update_250_db_version',
		),
		'2.6.0' => array(
			'kkart_update_260_options',
			'kkart_update_260_termmeta',
			'kkart_update_260_zones',
			'kkart_update_260_zone_methods',
			'kkart_update_260_refunds',
			'kkart_update_260_db_version',
		),
		'3.0.0' => array(
			'kkart_update_300_grouped_products',
			'kkart_update_300_settings',
			'kkart_update_300_product_visibility',
			'kkart_update_300_db_version',
		),
		'3.1.0' => array(
			'kkart_update_310_downloadable_products',
			'kkart_update_310_old_comments',
			'kkart_update_310_db_version',
		),
		'3.1.2' => array(
			'kkart_update_312_shop_manager_capabilities',
			'kkart_update_312_db_version',
		),
		'3.2.0' => array(
			'kkart_update_320_mexican_states',
			'kkart_update_320_db_version',
		),
		'3.3.0' => array(
			'kkart_update_330_image_options',
			'kkart_update_330_webhooks',
			'kkart_update_330_product_stock_status',
			'kkart_update_330_set_default_product_cat',
			'kkart_update_330_clear_transients',
			'kkart_update_330_set_paypal_sandbox_credentials',
			'kkart_update_330_db_version',
		),
		'3.4.0' => array(
			'kkart_update_340_states',
			'kkart_update_340_state',
			'kkart_update_340_last_active',
			'kkart_update_340_db_version',
		),
		'3.4.3' => array(
			'kkart_update_343_cleanup_foreign_keys',
			'kkart_update_343_db_version',
		),
		'3.4.4' => array(
			'kkart_update_344_recreate_roles',
			'kkart_update_344_db_version',
		),
		'3.5.0' => array(
			'kkart_update_350_reviews_comment_type',
			'kkart_update_350_db_version',
		),
		'3.5.2' => array(
			'kkart_update_352_drop_download_log_fk',
		),
		'3.5.4' => array(
			'kkart_update_354_modify_shop_manager_caps',
			'kkart_update_354_db_version',
		),
		'3.6.0' => array(
			'kkart_update_360_product_lookup_tables',
			'kkart_update_360_term_meta',
			'kkart_update_360_downloadable_product_permissions_index',
			'kkart_update_360_db_version',
		),
		'3.7.0' => array(
			'kkart_update_370_tax_rate_classes',
			'kkart_update_370_mro_std_currency',
			'kkart_update_370_db_version',
		),
		'3.9.0' => array(
			'kkart_update_390_move_maxmind_database',
			'kkart_update_390_change_geolocation_database_update_cron',
			'kkart_update_390_db_version',
		),
		'4.0.0' => array(
			'kkart_update_product_lookup_tables',
			'kkart_update_400_increase_size_of_column',
			'kkart_update_400_reset_action_scheduler_migration_status',
			'kkart_update_400_db_version',
		),
		'4.4.0' => array(
			'kkart_update_440_insert_attribute_terms_for_variable_products',
			'kkart_update_440_db_version',
		),
		'4.5.0' => array(
			'kkart_update_450_sanitize_coupons_code',
			'kkart_update_450_db_version',
		),
	);

	/**
	 * Hook in tabs.
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'check_version' ), 5 );
		add_action( 'init', array( __CLASS__, 'manual_database_update' ), 20 );
		add_action( 'admin_init', array( __CLASS__, 'kkart_admin_db_update_notice' ) );
		add_action( 'kkart_run_update_callback', array( __CLASS__, 'run_update_callback' ) );
		add_action( 'admin_init', array( __CLASS__, 'install_actions' ) );
		add_filter( 'plugin_action_links_' . KKART_PLUGIN_BASENAME, array( __CLASS__, 'plugin_action_links' ) );
		add_filter( 'plugin_row_meta', array( __CLASS__, 'plugin_row_meta' ), 10, 2 );
		add_filter( 'wpmu_drop_tables', array( __CLASS__, 'wpmu_drop_tables' ) );
		add_filter( 'cron_schedules', array( __CLASS__, 'cron_schedules' ) );
	}

	/**
	 * Check Kkart version and run the updater is required.
	 *
	 * This check is done on all requests and runs if the versions do not match.
	 */
	public static function check_version() {
		if ( ! Constants::is_defined( 'IFRAME_REQUEST' ) && version_compare( get_option( 'kkart_version' ), KKART()->version, '<' ) ) {
			self::install();
			do_action( 'kkart_updated' );
		}
	}

	/**
	 * Performan manual database update when triggered by Kkart System Tools.
	 *
	 * @since 3.6.5
	 */
	public static function manual_database_update() {
		$blog_id = get_current_blog_id();

		add_action( 'wp_' . $blog_id . '_kkart_updater_cron', array( __CLASS__, 'run_manual_database_update' ) );
	}

	/**
	 * Add KKART Admin based db update notice.
	 *
	 * @since 4.0.0
	 */
	public static function kkart_admin_db_update_notice() {
		if (
			KKART()->is_kkart_admin_active() &&
			false !== get_option( 'kkart_admin_install_timestamp' )
		) {
			new KKART_Notes_Run_Db_Update();
		}
	}

	/**
	 * Run manual database update.
	 */
	public static function run_manual_database_update() {
		self::update();
	}

	/**
	 * Run an update callback when triggered by ActionScheduler.
	 *
	 * @since 3.6.0
	 * @param string $callback Callback name.
	 */
	public static function run_update_callback( $callback ) {
		include_once dirname( __FILE__ ) . '/kkart-update-functions.php';

		if ( is_callable( $callback ) ) {
			self::run_update_callback_start( $callback );
			$result = (bool) call_user_func( $callback );
			self::run_update_callback_end( $callback, $result );
		}
	}

	/**
	 * Triggered when a callback will run.
	 *
	 * @since 3.6.0
	 * @param string $callback Callback name.
	 */
	protected static function run_update_callback_start( $callback ) {
		kkart_maybe_define_constant( 'KKART_UPDATING', true );
	}

	/**
	 * Triggered when a callback has ran.
	 *
	 * @since 3.6.0
	 * @param string $callback Callback name.
	 * @param bool   $result Return value from callback. Non-false need to run again.
	 */
	protected static function run_update_callback_end( $callback, $result ) {
		if ( $result ) {
			KKART()->queue()->add(
				'kkart_run_update_callback',
				array(
					'update_callback' => $callback,
				),
				'kkart-db-updates'
			);
		}
	}

	/**
	 * Install actions when a update button is clicked within the admin area.
	 *
	 * This function is hooked into admin_init to affect admin only.
	 */
	public static function install_actions() {
		if ( ! empty( $_GET['do_update_kkart'] ) ) { // WPCS: input var ok.
			check_admin_referer( 'kkart_db_update', 'kkart_db_update_nonce' );
			self::update();
			KKART_Admin_Notices::add_notice( 'update', true );
		}
	}

	/**
	 * Install KKART.
	 */
	public static function install() {
		if ( ! is_blog_installed() ) {
			return;
		}

		// Check if we are not already running this routine.
		if ( 'yes' === get_transient( 'kkart_installing' ) ) {
			return;
		}

		// If we made it till here nothing is running yet, lets set the transient now.
		set_transient( 'kkart_installing', 'yes', MINUTE_IN_SECONDS * 10 );
		kkart_maybe_define_constant( 'KKART_INSTALLING', true );

		KKART()->wpdb_table_fix();
		self::remove_admin_notices();
		self::create_tables();
		self::verify_base_tables();
		self::create_options();
		self::create_roles();
		self::setup_environment();
		self::create_terms();
		self::create_cron_jobs();
		self::create_files();
		self::maybe_create_pages();
		self::maybe_set_activation_transients();
		self::update_kkart_version();
		self::maybe_update_db_version();

		delete_transient( 'kkart_installing' );

		do_action( 'kkart_flush_rewrite_rules' );
		do_action( 'kkart_installed' );
	}

	/**
	 * Check if all the base tables are present.
	 *
	 * @param bool $modify_notice Whether to modify notice based on if all tables are present.
	 * @param bool $execute       Whether to execute get_schema queries as well.
	 *
	 * @return array List of querues.
	 */
	public static function verify_base_tables( $modify_notice = true, $execute = false ) {
		require_once KKART_ADMIN_DIR . 'includes/upgrade.php';

		if ( $execute ) {
			self::create_tables();
		}
		$queries        = dbDelta( self::get_schema(), false );
		$missing_tables = array();
		foreach ( $queries as $table_name => $result ) {
			if ( "Created table $table_name" === $result ) {
				$missing_tables[] = $table_name;
			}
		}

		if ( 0 < count( $missing_tables ) ) {
			if ( $modify_notice ) {
				KKART_Admin_Notices::add_notice( 'base_tables_missing' );
			}
			update_option( 'kkart_schema_missing_tables', $missing_tables );
		} else {
			if ( $modify_notice ) {
				KKART_Admin_Notices::remove_notice( 'base_tables_missing' );
			}
			update_option( 'kkart_schema_version', KKART()->db_version );
			delete_option( 'kkart_schema_missing_tables' );
		}
		return $missing_tables;
	}

	/**
	 * Reset any notices added to admin.
	 *
	 * @since 3.2.0
	 */
	private static function remove_admin_notices() {
		include_once dirname( __FILE__ ) . '/admin/class-kkart-admin-notices.php';
		KKART_Admin_Notices::remove_all_notices();
	}

	/**
	 * Setup KKART environment - post types, taxonomies, endpoints.
	 *
	 * @since 3.2.0
	 */
	private static function setup_environment() {
		KKART_Post_types::register_post_types();
		KKART_Post_types::register_taxonomies();
		KKART()->query->init_query_vars();
		KKART()->query->add_endpoints();
		KKART_API::add_endpoint();
		KKART_Auth::add_endpoint();
	}

	/**
	 * Is this a brand new KKART install?
	 *
	 * A brand new install has no version yet. Also treat empty installs as 'new'.
	 *
	 * @since  3.2.0
	 * @return boolean
	 */
	public static function is_new_install() {
		$product_count = array_sum( (array) wp_count_posts( 'product' ) );

		return is_null( get_option( 'kkart_version', null ) ) || ( 0 === $product_count && -1 === kkart_get_page_id( 'shop' ) );
	}

	/**
	 * Is a DB update needed?
	 *
	 * @since  3.2.0
	 * @return boolean
	 */
	public static function needs_db_update() {
		$current_db_version = get_option( 'kkart_db_version', null );
		$updates            = self::get_db_update_callbacks();
		$update_versions    = array_keys( $updates );
		usort( $update_versions, 'version_compare' );

		return ! is_null( $current_db_version ) && version_compare( $current_db_version, end( $update_versions ), '<' );
	}

	/**
	 * See if we need to set redirect transients for activation or not.
	 *
	 * @since 4.6.0
	 */
	private static function maybe_set_activation_transients() {
		if ( self::is_new_install() ) {
			set_transient( '_kkart_activation_redirect', 1, 30 );
		}
	}

	/**
	 * See if we need to show or run database updates during install.
	 *
	 * @since 3.2.0
	 */
	private static function maybe_update_db_version() {
		if ( self::needs_db_update() ) {
			if ( apply_filters( 'kkart_enable_auto_update_db', false ) ) {
				self::update();
			} else {
				KKART_Admin_Notices::add_notice( 'update', true );
			}
		} else {
			self::update_db_version();
		}
	}

	/**
	 * Update KKART version to current.
	 */
	private static function update_kkart_version() {
		update_option( 'kkart_version', KKART()->version );
	}

	/**
	 * Get list of DB update callbacks.
	 *
	 * @since  3.0.0
	 * @return array
	 */
	public static function get_db_update_callbacks() {
		return self::$db_updates;
	}

	/**
	 * Push all needed DB updates to the queue for processing.
	 */
	private static function update() {
		$current_db_version = get_option( 'kkart_db_version' );
		$loop               = 0;

		foreach ( self::get_db_update_callbacks() as $version => $update_callbacks ) {
			if ( version_compare( $current_db_version, $version, '<' ) ) {
				foreach ( $update_callbacks as $update_callback ) {
					KKART()->queue()->schedule_single(
						time() + $loop,
						'kkart_run_update_callback',
						array(
							'update_callback' => $update_callback,
						),
						'kkart-db-updates'
					);
					$loop++;
				}
			}
		}
	}

	/**
	 * Update DB version to current.
	 *
	 * @param string|null $version New Kkart DB version or null.
	 */
	public static function update_db_version( $version = null ) {
		update_option( 'kkart_db_version', is_null( $version ) ? KKART()->version : $version );
	}

	/**
	 * Add more cron schedules.
	 *
	 * @param array $schedules List of WP scheduled cron jobs.
	 *
	 * @return array
	 */
	public static function cron_schedules( $schedules ) {
		$schedules['monthly']     = array(
			'interval' => 2635200,
			'display'  => __( 'Monthly', 'kkart' ),
		);
		$schedules['fifteendays'] = array(
			'interval' => 1296000,
			'display'  => __( 'Every 15 Days', 'kkart' ),
		);
		return $schedules;
	}

	/**
	 * Create cron jobs (clear them first).
	 */
	private static function create_cron_jobs() {
		wp_clear_scheduled_hook( 'kkart_scheduled_sales' );
		wp_clear_scheduled_hook( 'kkart_cancel_unpaid_orders' );
		wp_clear_scheduled_hook( 'kkart_cleanup_sessions' );
		wp_clear_scheduled_hook( 'kkart_cleanup_personal_data' );
		wp_clear_scheduled_hook( 'kkart_cleanup_logs' );
		wp_clear_scheduled_hook( 'kkart_geoip_updater' );
		wp_clear_scheduled_hook( 'kkart_tracker_send_event' );

		$ve = get_option( 'gmt_offset' ) > 0 ? '-' : '+';

		wp_schedule_event( strtotime( '00:00 tomorrow ' . $ve . absint( get_option( 'gmt_offset' ) ) . ' HOURS' ), 'daily', 'kkart_scheduled_sales' );

		$held_duration = get_option( 'kkart_hold_stock_minutes', '60' );

		if ( '' !== $held_duration ) {
			wp_schedule_single_event( time() + ( absint( $held_duration ) * 60 ), 'kkart_cancel_unpaid_orders' );
		}

		// Delay the first run of `kkart_cleanup_personal_data` by 10 seconds
		// so it doesn't occur in the same request. Kkart Admin also schedules
		// a daily cron that gets lost due to a race condition. KKART_Privacy's background
		// processing instance updates the cron schedule from within a cron job.
		wp_schedule_event( time() + 10, 'daily', 'kkart_cleanup_personal_data' );
		wp_schedule_event( time() + ( 3 * HOUR_IN_SECONDS ), 'daily', 'kkart_cleanup_logs' );
		wp_schedule_event( time() + ( 6 * HOUR_IN_SECONDS ), 'twicedaily', 'kkart_cleanup_sessions' );
		wp_schedule_event( time() + MINUTE_IN_SECONDS, 'fifteendays', 'kkart_geoip_updater' );
		wp_schedule_event( time() + 10, apply_filters( 'kkart_tracker_event_recurrence', 'daily' ), 'kkart_tracker_send_event' );
	}

	/**
	 * Create pages on installation.
	 */
	public static function maybe_create_pages() {
		if ( empty( get_option( 'kkart_db_version' ) ) ) {
			self::create_pages();
		}
	}

	/**
	 * Create pages that the plugin relies on, storing page IDs in variables.
	 */
	public static function create_pages() {
		include_once dirname( __FILE__ ) . '/admin/kkart-admin-functions.php';

		$pages = apply_filters(
			'kkart_create_pages',
			array(
				'shop'      => array(
					'name'    => _x( 'shop', 'Page slug', 'kkart' ),
					'title'   => _x( 'Shop', 'Page title', 'kkart' ),
					'content' => '',
				),
				'cart'      => array(
					'name'    => _x( 'cart', 'Page slug', 'kkart' ),
					'title'   => _x( 'Cart', 'Page title', 'kkart' ),
					'content' => '<!-- wp:shortcode -->[' . apply_filters( 'kkart_cart_shortcode_tag', 'kkart_cart' ) . ']<!-- /wp:shortcode -->',
				),
				'checkout'  => array(
					'name'    => _x( 'checkout', 'Page slug', 'kkart' ),
					'title'   => _x( 'Checkout', 'Page title', 'kkart' ),
					'content' => '<!-- wp:shortcode -->[' . apply_filters( 'kkart_checkout_shortcode_tag', 'kkart_checkout' ) . ']<!-- /wp:shortcode -->',
				),
				'order_pay'  => array(
					'name'    => _x( 'order-pay', 'Page slug', 'kkart' ),
					'title'   => _x( 'Order Pay', 'Page title', 'kkart' ),
					'content' => '',
					'parent' => 'checkout',
				),
				'order_received'  => array(
					'name'    => _x( 'order-received', 'Page slug', 'kkart' ),
					'title'   => _x( 'Order Received', 'Page title', 'kkart' ),
					'content' => '',
					'parent' => 'checkout',
				),
				'myaccount' => array(
					'name'    => _x( 'my-account', 'Page slug', 'kkart' ),
					'title'   => _x( 'My account', 'Page title', 'kkart' ),
					'content' => '<!-- wp:shortcode -->[' . apply_filters( 'kkart_my_account_shortcode_tag', 'kkart_my_account' ) . ']<!-- /wp:shortcode -->',
				),
			)
		);

		foreach ( $pages as $key => $page ) {
			kkart_create_page( esc_sql( $page['name'] ), 'kkart_' . $key . '_page_id', $page['title'], $page['content'], ! empty( $page['parent'] ) ? kkart_get_page_id( $page['parent'] ) : '' );
		}
		
		if( class_exists('KKART_Admin_Import_Template') ){
			$fields = json_decode(KKART_Admin_Import_Template::get_conf(), true);
			$content = KKART_Admin_Import_Template::get_content();			
			kkart_import_pages_default_content($fields, $content);
		}
	}

	/**
	 * Default options.
	 *
	 * Sets up the default options used on the settings page.
	 */
	private static function create_options() {
		// Include settings so that we can run through defaults.
		include_once dirname( __FILE__ ) . '/admin/class-kkart-admin-settings.php';

		$settings = KKART_Admin_Settings::get_settings_pages();

		foreach ( $settings as $section ) {
			if ( ! method_exists( $section, 'get_settings' ) ) {
				continue;
			}
			$subsections = array_unique( array_merge( array( '' ), array_keys( $section->get_sections() ) ) );

			foreach ( $subsections as $subsection ) {
				foreach ( $section->get_settings( $subsection ) as $value ) {
					if ( isset( $value['default'] ) && isset( $value['id'] ) ) {
						$autoload = isset( $value['autoload'] ) ? (bool) $value['autoload'] : true;
						add_option( $value['id'], $value['default'], '', ( $autoload ? 'yes' : 'no' ) );
					}
				}
			}
		}

		// Define other defaults if not in setting screens.
		add_option( 'kkart_single_image_width', '600', '', 'yes' );
		add_option( 'kkart_thumbnail_image_width', '300', '', 'yes' );
		add_option( 'kkart_checkout_highlight_required_fields', 'yes', '', 'yes' );
		add_option( 'kkart_demo_store', 'no', '', 'no' );

		// Define initial tax classes.
		KKART_Tax::create_tax_class( __( 'Reduced rate', 'kkart' ) );
		KKART_Tax::create_tax_class( __( 'Zero rate', 'kkart' ) );
		
		// Set Default payment method
		// Load gateways.
		$payment_gateways = KKART()->payment_gateways->payment_gateways();

		// Get posted gateway.
		$gateway_id = 'cod';

		foreach( $payment_gateways as $gateway ){
			if(! in_array( $gateway_id, array( $gateway->id, sanitize_title( get_class( $gateway ) ) ), true )){
				continue;
			}
			
			$enabled = $gateway->get_option( 'enabled', 'no' );

			if(! kkart_string_to_bool( $enabled ) && ! $gateway->needs_setup()){
				$gateway->update_option( 'enabled', 'yes' );
			}
		}
	}

	/**
	 * Add the default terms for KKART taxonomies - product types and order statuses. Modify this at your own risk.
	 */
	public static function create_terms() {
		$taxonomies = array(
			'product_type'       => array(
				'simple',
				'grouped',
				'variable',
				'external',
			),
			'product_visibility' => array(
				'exclude-from-search',
				'exclude-from-catalog',
				'featured',
				'outofstock',
				'rated-1',
				'rated-2',
				'rated-3',
				'rated-4',
				'rated-5',
			),
		);

		foreach ( $taxonomies as $taxonomy => $terms ) {
			foreach ( $terms as $term ) {
				if ( ! get_term_by( 'name', $term, $taxonomy ) ) { // @codingStandardsIgnoreLine.
					wp_insert_term( $term, $taxonomy );
				}
			}
		}

		$kkart_default_category = (int) get_option( 'default_product_cat', 0 );

		if ( ! $kkart_default_category || ! term_exists( $kkart_default_category, 'product_cat' ) ) {
			$default_product_cat_id   = 0;
			$default_product_cat_slug = sanitize_title( _x( 'Uncategorized', 'Default category slug', 'kkart' ) );
			$default_product_cat      = get_term_by( 'slug', $default_product_cat_slug, 'product_cat' ); // @codingStandardsIgnoreLine.

			if ( $default_product_cat ) {
				$default_product_cat_id = absint( $default_product_cat->term_taxonomy_id );
			} else {
				$result = wp_insert_term( _x( 'Uncategorized', 'Default category slug', 'kkart' ), 'product_cat', array( 'slug' => $default_product_cat_slug ) );

				if ( ! is_wp_error( $result ) && ! empty( $result['term_taxonomy_id'] ) ) {
					$default_product_cat_id = absint( $result['term_taxonomy_id'] );
				}
			}

			if ( $default_product_cat_id ) {
				update_option( 'default_product_cat', $default_product_cat_id );
			}
		}
	}

	/**
	 * Set up the database tables which the plugin needs to function.
	 * WARNING: If you are modifying this method, make sure that its safe to call regardless of the state of database.
	 *
	 * This is called from `install` method and is executed in-sync when KKART is installed or updated. This can also be called optionally from `verify_base_tables`.
	 *
	 * TODO: Add all crucial tables that we have created from workers in the past.
	 *
	 * Tables:
	 *      kkart_attribute_taxonomies - Table for storing attribute taxonomies - these are user defined
	 *      kkart_downloadable_product_permissions - Table for storing user and guest download permissions.
	 *          KEY(order_id, product_id, download_id) used for organizing downloads on the My Account page
	 *      kkart_order_items - Order line items are stored in a table to make them easily queryable for reports
	 *      kkart_order_itemmeta - Order line item meta is stored in a table for storing extra data.
	 *      kkart_tax_rates - Tax Rates are stored inside 2 tables making tax queries simple and efficient.
	 *      kkart_tax_rate_locations - Each rate can be applied to more than one postcode/city hence the second table.
	 */
	private static function create_tables() {
		global $wpdb;

		$wpdb->hide_errors();

		require_once KKART_ADMIN_DIR . 'includes/upgrade.php';

		/**
		 * Before updating with DBDELTA, remove any primary keys which could be
		 * modified due to schema updates.
		 */
		if ( $wpdb->get_var( "SHOW TABLES LIKE '{$wpdb->prefix}kkart_downloadable_product_permissions';" ) ) {
			if ( ! $wpdb->get_var( "SHOW COLUMNS FROM `{$wpdb->prefix}kkart_downloadable_product_permissions` LIKE 'permission_id';" ) ) {
				$wpdb->query( "ALTER TABLE {$wpdb->prefix}kkart_downloadable_product_permissions DROP PRIMARY KEY, ADD `permission_id` BIGINT UNSIGNED NOT NULL PRIMARY KEY AUTO_INCREMENT;" );
			}
		}

		/**
		 * Change wp_kkart_sessions schema to use a bigint auto increment field instead of char(32) field as
		 * the primary key as it is not a good practice to use a char(32) field as the primary key of a table and as
		 * there were reports of issues with this table (see https://github.com/kkart/kkart/issues/20912).
		 *
		 * This query needs to run before dbDelta() as this WP function is not able to handle primary key changes
		 * (see https://github.com/kkart/kkart/issues/21534 and https://core.trac.wordpress.org/ticket/40357).
		 */
		if ( $wpdb->get_var( "SHOW TABLES LIKE '{$wpdb->prefix}kkart_sessions'" ) ) {
			if ( ! $wpdb->get_var( "SHOW KEYS FROM {$wpdb->prefix}kkart_sessions WHERE Key_name = 'PRIMARY' AND Column_name = 'session_id'" ) ) {
				$wpdb->query(
					"ALTER TABLE `{$wpdb->prefix}kkart_sessions` DROP PRIMARY KEY, DROP KEY `session_id`, ADD PRIMARY KEY(`session_id`), ADD UNIQUE KEY(`session_key`)"
				);
			}
		}

		dbDelta( self::get_schema() );

		$index_exists = $wpdb->get_row( "SHOW INDEX FROM {$wpdb->comments} WHERE column_name = 'comment_type' and key_name = 'woo_idx_comment_type'" );

		if ( is_null( $index_exists ) ) {
			// Add an index to the field comment_type to improve the response time of the query
			// used by KKART_Comments::wp_count_comments() to get the number of comments by type.
			$wpdb->query( "ALTER TABLE {$wpdb->comments} ADD INDEX woo_idx_comment_type (comment_type)" );
		}

		// Get tables data types and check it matches before adding constraint.
		$download_log_columns     = $wpdb->get_results( "SHOW COLUMNS FROM {$wpdb->prefix}kkart_download_log WHERE Field = 'permission_id'", ARRAY_A );
		$download_log_column_type = '';
		if ( isset( $download_log_columns[0]['Type'] ) ) {
			$download_log_column_type = $download_log_columns[0]['Type'];
		}

		$download_permissions_columns     = $wpdb->get_results( "SHOW COLUMNS FROM {$wpdb->prefix}kkart_downloadable_product_permissions WHERE Field = 'permission_id'", ARRAY_A );
		$download_permissions_column_type = '';
		if ( isset( $download_permissions_columns[0]['Type'] ) ) {
			$download_permissions_column_type = $download_permissions_columns[0]['Type'];
		}

		// Add constraint to download logs if the columns matches.
		if ( ! empty( $download_permissions_column_type ) && ! empty( $download_log_column_type ) && $download_permissions_column_type === $download_log_column_type ) {
			$fk_result = $wpdb->get_row( "SHOW CREATE TABLE {$wpdb->prefix}kkart_download_log" );
			if ( false === strpos( $fk_result->{'Create Table'}, "fk_{$wpdb->prefix}kkart_download_log_permission_id" ) ) {
				$wpdb->query(
					"ALTER TABLE `{$wpdb->prefix}kkart_download_log`
					ADD CONSTRAINT `fk_{$wpdb->prefix}kkart_download_log_permission_id`
					FOREIGN KEY (`permission_id`)
					REFERENCES `{$wpdb->prefix}kkart_downloadable_product_permissions` (`permission_id`) ON DELETE CASCADE;"
				);
			}
		}

		// Clear table caches.
		delete_transient( 'kkart_attribute_taxonomies' );
	}

	/**
	 * Get Table schema.
	 *
	 * See https://github.com/kkart/kkart/wiki/Database-Description/
	 *
	 * A note on indexes; Indexes have a maximum size of 767 bytes. Historically, we haven't need to be concerned about that.
	 * As of WordPress 4.2, however, we moved to utf8mb4, which uses 4 bytes per character. This means that an index which
	 * used to have room for floor(767/3) = 255 characters, now only has room for floor(767/4) = 191 characters.
	 *
	 * Changing indexes may cause duplicate index notices in logs due to https://core.trac.wordpress.org/ticket/34870 but dropping
	 * indexes first causes too much load on some servers/larger DB.
	 *
	 * When adding or removing a table, make sure to update the list of tables in KKART_Install::get_tables().
	 *
	 * @return string
	 */
	private static function get_schema() {
		global $wpdb;

		$collate = '';

		if ( $wpdb->has_cap( 'collation' ) ) {
			$collate = $wpdb->get_charset_collate();
		}

		/*
		 * Indexes have a maximum size of 767 bytes. Historically, we haven't need to be concerned about that.
		 * As of WP 4.2, however, they moved to utf8mb4, which uses 4 bytes per character. This means that an index which
		 * used to have room for floor(767/3) = 255 characters, now only has room for floor(767/4) = 191 characters.
		 */
		$max_index_length = 191;

		$tables = "
CREATE TABLE {$wpdb->prefix}kkart_sessions (
  session_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  session_key char(32) NOT NULL,
  session_value longtext NOT NULL,
  session_expiry BIGINT UNSIGNED NOT NULL,
  PRIMARY KEY  (session_id),
  UNIQUE KEY session_key (session_key)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_api_keys (
  key_id BIGINT UNSIGNED NOT NULL auto_increment,
  user_id BIGINT UNSIGNED NOT NULL,
  description varchar(200) NULL,
  permissions varchar(10) NOT NULL,
  consumer_key char(64) NOT NULL,
  consumer_secret char(43) NOT NULL,
  nonces longtext NULL,
  truncated_key char(7) NOT NULL,
  last_access datetime NULL default null,
  PRIMARY KEY  (key_id),
  KEY consumer_key (consumer_key),
  KEY consumer_secret (consumer_secret)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_attribute_taxonomies (
  attribute_id BIGINT UNSIGNED NOT NULL auto_increment,
  attribute_name varchar(200) NOT NULL,
  attribute_label varchar(200) NULL,
  attribute_type varchar(20) NOT NULL,
  attribute_orderby varchar(20) NOT NULL,
  attribute_public int(1) NOT NULL DEFAULT 1,
  PRIMARY KEY  (attribute_id),
  KEY attribute_name (attribute_name(20))
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_downloadable_product_permissions (
  permission_id BIGINT UNSIGNED NOT NULL auto_increment,
  download_id varchar(36) NOT NULL,
  product_id BIGINT UNSIGNED NOT NULL,
  order_id BIGINT UNSIGNED NOT NULL DEFAULT 0,
  order_key varchar(200) NOT NULL,
  user_email varchar(200) NOT NULL,
  user_id BIGINT UNSIGNED NULL,
  downloads_remaining varchar(9) NULL,
  access_granted datetime NOT NULL default '0000-00-00 00:00:00',
  access_expires datetime NULL default null,
  download_count BIGINT UNSIGNED NOT NULL DEFAULT 0,
  PRIMARY KEY  (permission_id),
  KEY download_order_key_product (product_id,order_id,order_key(16),download_id),
  KEY download_order_product (download_id,order_id,product_id),
  KEY order_id (order_id),
  KEY user_order_remaining_expires (user_id,order_id,downloads_remaining,access_expires)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_order_items (
  order_item_id BIGINT UNSIGNED NOT NULL auto_increment,
  order_item_name TEXT NOT NULL,
  order_item_type varchar(200) NOT NULL DEFAULT '',
  order_id BIGINT UNSIGNED NOT NULL,
  PRIMARY KEY  (order_item_id),
  KEY order_id (order_id)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_order_itemmeta (
  meta_id BIGINT UNSIGNED NOT NULL auto_increment,
  order_item_id BIGINT UNSIGNED NOT NULL,
  meta_key varchar(255) default NULL,
  meta_value longtext NULL,
  PRIMARY KEY  (meta_id),
  KEY order_item_id (order_item_id),
  KEY meta_key (meta_key(32))
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_tax_rates (
  tax_rate_id BIGINT UNSIGNED NOT NULL auto_increment,
  tax_rate_country varchar(2) NOT NULL DEFAULT '',
  tax_rate_state varchar(200) NOT NULL DEFAULT '',
  tax_rate varchar(8) NOT NULL DEFAULT '',
  tax_rate_name varchar(200) NOT NULL DEFAULT '',
  tax_rate_priority BIGINT UNSIGNED NOT NULL,
  tax_rate_compound int(1) NOT NULL DEFAULT 0,
  tax_rate_shipping int(1) NOT NULL DEFAULT 1,
  tax_rate_order BIGINT UNSIGNED NOT NULL,
  tax_rate_class varchar(200) NOT NULL DEFAULT '',
  PRIMARY KEY  (tax_rate_id),
  KEY tax_rate_country (tax_rate_country),
  KEY tax_rate_state (tax_rate_state(2)),
  KEY tax_rate_class (tax_rate_class(10)),
  KEY tax_rate_priority (tax_rate_priority)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_tax_rate_locations (
  location_id BIGINT UNSIGNED NOT NULL auto_increment,
  location_code varchar(200) NOT NULL,
  tax_rate_id BIGINT UNSIGNED NOT NULL,
  location_type varchar(40) NOT NULL,
  PRIMARY KEY  (location_id),
  KEY tax_rate_id (tax_rate_id),
  KEY location_type_code (location_type(10),location_code(20))
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_shipping_zones (
  zone_id BIGINT UNSIGNED NOT NULL auto_increment,
  zone_name varchar(200) NOT NULL,
  zone_order BIGINT UNSIGNED NOT NULL,
  PRIMARY KEY  (zone_id)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_shipping_zone_locations (
  location_id BIGINT UNSIGNED NOT NULL auto_increment,
  zone_id BIGINT UNSIGNED NOT NULL,
  location_code varchar(200) NOT NULL,
  location_type varchar(40) NOT NULL,
  PRIMARY KEY  (location_id),
  KEY location_id (location_id),
  KEY location_type_code (location_type(10),location_code(20))
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_shipping_zone_methods (
  zone_id BIGINT UNSIGNED NOT NULL,
  instance_id BIGINT UNSIGNED NOT NULL auto_increment,
  method_id varchar(200) NOT NULL,
  method_order BIGINT UNSIGNED NOT NULL,
  is_enabled tinyint(1) NOT NULL DEFAULT '1',
  PRIMARY KEY  (instance_id)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_payment_tokens (
  token_id BIGINT UNSIGNED NOT NULL auto_increment,
  gateway_id varchar(200) NOT NULL,
  token text NOT NULL,
  user_id BIGINT UNSIGNED NOT NULL DEFAULT '0',
  type varchar(200) NOT NULL,
  is_default tinyint(1) NOT NULL DEFAULT '0',
  PRIMARY KEY  (token_id),
  KEY user_id (user_id)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_payment_tokenmeta (
  meta_id BIGINT UNSIGNED NOT NULL auto_increment,
  payment_token_id BIGINT UNSIGNED NOT NULL,
  meta_key varchar(255) NULL,
  meta_value longtext NULL,
  PRIMARY KEY  (meta_id),
  KEY payment_token_id (payment_token_id),
  KEY meta_key (meta_key(32))
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_log (
  log_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  timestamp datetime NOT NULL,
  level smallint(4) NOT NULL,
  source varchar(200) NOT NULL,
  message longtext NOT NULL,
  context longtext NULL,
  PRIMARY KEY (log_id),
  KEY level (level)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_webhooks (
  webhook_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  status varchar(200) NOT NULL,
  name text NOT NULL,
  user_id BIGINT UNSIGNED NOT NULL,
  delivery_url text NOT NULL,
  secret text NOT NULL,
  topic varchar(200) NOT NULL,
  date_created datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
  date_created_gmt datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
  date_modified datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
  date_modified_gmt datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
  api_version smallint(4) NOT NULL,
  failure_count smallint(10) NOT NULL DEFAULT '0',
  pending_delivery tinyint(1) NOT NULL DEFAULT '0',
  PRIMARY KEY  (webhook_id),
  KEY user_id (user_id)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_download_log (
  download_log_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  timestamp datetime NOT NULL,
  permission_id BIGINT UNSIGNED NOT NULL,
  user_id BIGINT UNSIGNED NULL,
  user_ip_address VARCHAR(100) NULL DEFAULT '',
  PRIMARY KEY  (download_log_id),
  KEY permission_id (permission_id),
  KEY timestamp (timestamp)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_product_meta_lookup (
  `product_id` bigint(20) NOT NULL,
  `sku` varchar(100) NULL default '',
  `virtual` tinyint(1) NULL default 0,
  `downloadable` tinyint(1) NULL default 0,
  `min_price` decimal(19,4) NULL default NULL,
  `max_price` decimal(19,4) NULL default NULL,
  `onsale` tinyint(1) NULL default 0,
  `stock_quantity` double NULL default NULL,
  `stock_status` varchar(100) NULL default 'instock',
  `rating_count` bigint(20) NULL default 0,
  `average_rating` decimal(3,2) NULL default 0.00,
  `total_sales` bigint(20) NULL default 0,
  `tax_status` varchar(100) NULL default 'taxable',
  `tax_class` varchar(100) NULL default '',
  PRIMARY KEY  (`product_id`),
  KEY `virtual` (`virtual`),
  KEY `downloadable` (`downloadable`),
  KEY `stock_status` (`stock_status`),
  KEY `stock_quantity` (`stock_quantity`),
  KEY `onsale` (`onsale`),
  KEY min_max_price (`min_price`, `max_price`)
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_tax_rate_classes (
  tax_rate_class_id BIGINT UNSIGNED NOT NULL auto_increment,
  name varchar(200) NOT NULL DEFAULT '',
  slug varchar(200) NOT NULL DEFAULT '',
  PRIMARY KEY  (tax_rate_class_id),
  UNIQUE KEY slug (slug($max_index_length))
) $collate;
CREATE TABLE {$wpdb->prefix}kkart_reserved_stock (
	`order_id` bigint(20) NOT NULL,
	`product_id` bigint(20) NOT NULL,
	`stock_quantity` double NOT NULL DEFAULT 0,
	`timestamp` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
	`expires` datetime NOT NULL DEFAULT '0000-00-00 00:00:00',
	PRIMARY KEY  (`order_id`, `product_id`)
) $collate;
		";

		return $tables;
	}

	/**
	 * Return a list of Kkart tables. Used to make sure all KKART tables are dropped when uninstalling the plugin
	 * in a single site or multi site environment.
	 *
	 * @return array KKART tables.
	 */
	public static function get_tables() {
		global $wpdb;

		$tables = array(
			"{$wpdb->prefix}kkart_download_log",
			"{$wpdb->prefix}kkart_product_meta_lookup",
			"{$wpdb->prefix}kkart_tax_rate_classes",
			"{$wpdb->prefix}kkart_webhooks",
			"{$wpdb->prefix}kkart_api_keys",
			"{$wpdb->prefix}kkart_attribute_taxonomies",
			"{$wpdb->prefix}kkart_downloadable_product_permissions",
			"{$wpdb->prefix}kkart_log",
			"{$wpdb->prefix}kkart_order_itemmeta",
			"{$wpdb->prefix}kkart_order_items",
			"{$wpdb->prefix}kkart_payment_tokenmeta",
			"{$wpdb->prefix}kkart_payment_tokens",
			"{$wpdb->prefix}kkart_sessions",
			"{$wpdb->prefix}kkart_shipping_zone_locations",
			"{$wpdb->prefix}kkart_shipping_zone_methods",
			"{$wpdb->prefix}kkart_shipping_zones",
			"{$wpdb->prefix}kkart_tax_rate_locations",
			"{$wpdb->prefix}kkart_tax_rates",
			"{$wpdb->prefix}kkart_reserved_stock",
		);

		/**
		 * Filter the list of known Kkart tables.
		 *
		 * If Kkart plugins need to add new tables, they can inject them here.
		 *
		 * @param array $tables An array of Kkart-specific database table names.
		 */
		$tables = apply_filters( 'kkart_install_get_tables', $tables );

		return $tables;
	}

	/**
	 * Drop Kkart tables.
	 *
	 * @return void
	 */
	public static function drop_tables() {
		global $wpdb;

		$tables = self::get_tables();

		foreach ( $tables as $table ) {
			$wpdb->query( "DROP TABLE IF EXISTS {$table}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	/**
	 * Uninstall tables when MU blog is deleted.
	 *
	 * @param array $tables List of tables that will be deleted by WP.
	 *
	 * @return string[]
	 */
	public static function wpmu_drop_tables( $tables ) {
		return array_merge( $tables, self::get_tables() );
	}

	/**
	 * Create roles and capabilities.
	 */
	public static function create_roles() {
		global $wp_roles;

		if ( ! class_exists( 'WP_Roles' ) ) {
			return;
		}

		if ( ! isset( $wp_roles ) ) {
			$wp_roles = new WP_Roles(); // @codingStandardsIgnoreLine
		}

		// Dummy gettext calls to get strings in the catalog.
		/* translators: user role */
		_x( 'Customer', 'User role', 'kkart' );
		/* translators: user role */
		_x( 'Shop manager', 'User role', 'kkart' );

		// Customer role.
		add_role(
			'customer',
			'Customer',
			array(
				'read' => true,
			)
		);

		// Shop manager role.
		add_role(
			'shop_manager',
			'Shop manager',
			array(
				'level_9'                => true,
				'level_8'                => true,
				'level_7'                => true,
				'level_6'                => true,
				'level_5'                => true,
				'level_4'                => true,
				'level_3'                => true,
				'level_2'                => true,
				'level_1'                => true,
				'level_0'                => true,
				'read'                   => true,
				'read_private_pages'     => true,
				'read_private_posts'     => true,
				'edit_posts'             => true,
				'edit_pages'             => true,
				'edit_published_posts'   => true,
				'edit_published_pages'   => true,
				'edit_private_pages'     => true,
				'edit_private_posts'     => true,
				'edit_others_posts'      => true,
				'edit_others_pages'      => true,
				'publish_posts'          => true,
				'publish_pages'          => true,
				'delete_posts'           => true,
				'delete_pages'           => true,
				'delete_private_pages'   => true,
				'delete_private_posts'   => true,
				'delete_published_pages' => true,
				'delete_published_posts' => true,
				'delete_others_posts'    => true,
				'delete_others_pages'    => true,
				'manage_categories'      => true,
				'manage_links'           => true,
				'moderate_comments'      => true,
				'upload_files'           => true,
				'export'                 => true,
				'import'                 => true,
				'list_users'             => true,
				'edit_theme_options'     => true,
			)
		);

		$capabilities = self::get_core_capabilities();

		foreach ( $capabilities as $cap_group ) {
			foreach ( $cap_group as $cap ) {
				$wp_roles->add_cap( 'shop_manager', $cap );
				$wp_roles->add_cap( 'administrator', $cap );
			}
		}
	}

	/**
	 * Get capabilities for Kkart - these are assigned to admin/shop manager during installation or reset.
	 *
	 * @return array
	 */
	public static function get_core_capabilities() {
		$capabilities = array();

		$capabilities['core'] = array(
			'manage_kkart',
			'view_kkart_reports',
		);

		$capability_types = array( 'product', 'shop_order', 'shop_coupon' );

		foreach ( $capability_types as $capability_type ) {

			$capabilities[ $capability_type ] = array(
				// Post type.
				"edit_{$capability_type}",
				"read_{$capability_type}",
				"delete_{$capability_type}",
				"edit_{$capability_type}s",
				"edit_others_{$capability_type}s",
				"publish_{$capability_type}s",
				"read_private_{$capability_type}s",
				"delete_{$capability_type}s",
				"delete_private_{$capability_type}s",
				"delete_published_{$capability_type}s",
				"delete_others_{$capability_type}s",
				"edit_private_{$capability_type}s",
				"edit_published_{$capability_type}s",

				// Terms.
				"manage_{$capability_type}_terms",
				"edit_{$capability_type}_terms",
				"delete_{$capability_type}_terms",
				"assign_{$capability_type}_terms",
			);
		}

		return $capabilities;
	}

	/**
	 * Remove Kkart roles.
	 */
	public static function remove_roles() {
		global $wp_roles;

		if ( ! class_exists( 'WP_Roles' ) ) {
			return;
		}

		if ( ! isset( $wp_roles ) ) {
			$wp_roles = new WP_Roles(); // @codingStandardsIgnoreLine
		}

		$capabilities = self::get_core_capabilities();

		foreach ( $capabilities as $cap_group ) {
			foreach ( $cap_group as $cap ) {
				$wp_roles->remove_cap( 'shop_manager', $cap );
				$wp_roles->remove_cap( 'administrator', $cap );
			}
		}

		remove_role( 'customer' );
		remove_role( 'shop_manager' );
	}

	/**
	 * Create files/directories.
	 */
	private static function create_files() {
		// Bypass if filesystem is read-only and/or non-standard upload system is used.
		if ( apply_filters( 'kkart_install_skip_create_files', false ) ) {
			return;
		}

		// Install files and folders for uploading files and prevent hotlinking.
		$upload_dir      = wp_get_upload_dir();
		$download_method = get_option( 'kkart_file_download_method', 'force' );

		$files = array(
			array(
				'base'    => $upload_dir['basedir'] . '/kkart_uploads',
				'file'    => 'index.html',
				'content' => '',
			),
			array(
				'base'    => KKART_LOG_DIR,
				'file'    => '.htaccess',
				'content' => 'deny from all',
			),
			array(
				'base'    => KKART_LOG_DIR,
				'file'    => 'index.html',
				'content' => '',
			),
			array(
				'base'    => $upload_dir['basedir'] . '/kkart_uploads',
				'file'    => '.htaccess',
				'content' => 'redirect' === $download_method ? 'Options -Indexes' : 'deny from all',
			),
		);

		foreach ( $files as $file ) {
			if ( wp_mkdir_p( $file['base'] ) && ! file_exists( trailingslashit( $file['base'] ) . $file['file'] ) ) {
				$file_handle = @fopen( trailingslashit( $file['base'] ) . $file['file'], 'wb' ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged, WordPress.WP.AlternativeFunctions.file_system_read_fopen
				if ( $file_handle ) {
					fwrite( $file_handle, $file['content'] ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fwrite
					fclose( $file_handle ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fclose
				}
			}
		}

		// Create attachment for placeholders.
		self::create_placeholder_image();
	}

	/**
	 * Create a placeholder image in the media library.
	 *
	 * @since 3.5.0
	 */
	private static function create_placeholder_image() {
		$placeholder_image = get_option( 'kkart_placeholder_image', 0 );

		// Validate current setting if set. If set, return.
		if ( ! empty( $placeholder_image ) ) {
			if ( ! is_numeric( $placeholder_image ) ) {
				return;
			} elseif ( $placeholder_image && wp_attachment_is_image( $placeholder_image ) ) {
				return;
			}
		}

		$upload_dir = wp_upload_dir();
		$source     = KKART()->plugin_path() . '/assets/images/placeholder-attachment.png';
		$filename   = $upload_dir['basedir'] . '/kkart-placeholder.png';

		if ( ! file_exists( $filename ) ) {
			copy( $source, $filename ); // @codingStandardsIgnoreLine.
		}

		if ( ! file_exists( $filename ) ) {
			update_option( 'kkart_placeholder_image', 0 );
			return;
		}

		$filetype   = wp_check_filetype( basename( $filename ), null );
		$attachment = array(
			'guid'           => $upload_dir['url'] . '/' . basename( $filename ),
			'post_mime_type' => $filetype['type'],
			'post_title'     => preg_replace( '/\.[^.]+$/', '', basename( $filename ) ),
			'post_content'   => '',
			'post_status'    => 'inherit',
		);
		$attach_id  = wp_insert_attachment( $attachment, $filename );

		update_option( 'kkart_placeholder_image', $attach_id );

		// Make sure that this file is included, as wp_generate_attachment_metadata() depends on it.
		require_once KKART_ADMIN_DIR . 'includes/image.php';

		// Generate the metadata for the attachment, and update the database record.
		$attach_data = wp_generate_attachment_metadata( $attach_id, $filename );
		wp_update_attachment_metadata( $attach_id, $attach_data );
	}

	/**
	 * Show action links on the plugin screen.
	 *
	 * @param mixed $links Plugin Action links.
	 *
	 * @return array
	 */
	public static function plugin_action_links( $links ) {
		$action_links = array(
			'settings' => '<a href="' . admin_url( 'admin.php?page=kkart-settings' ) . '" aria-label="' . esc_attr__( 'View Kkart settings', 'kkart' ) . '">' . esc_html__( 'Settings', 'kkart' ) . '</a>',
		);

		return array_merge( $action_links, $links );
	}

	/**
	 * Show row meta on the plugin screen.
	 *
	 * @param mixed $links Plugin Row Meta.
	 * @param mixed $file  Plugin Base file.
	 *
	 * @return array
	 */
	public static function plugin_row_meta( $links, $file ) {
		if ( KKART_PLUGIN_BASENAME !== $file ) {
			return $links;
		}

		$row_meta = array(
			'docs'    => '<a href="' . esc_url( apply_filters( 'kkart_docs_url', 'https://docs.kkart.com/documentation/plugins/kkart/' ) ) . '" aria-label="' . esc_attr__( 'View Kkart documentation', 'kkart' ) . '">' . esc_html__( 'Docs', 'kkart' ) . '</a>',
			'apidocs' => '<a href="' . esc_url( apply_filters( 'kkart_apidocs_url', 'https://docs.kkart.com/kkart-apidocs/' ) ) . '" aria-label="' . esc_attr__( 'View Kkart API docs', 'kkart' ) . '">' . esc_html__( 'API docs', 'kkart' ) . '</a>',
			'support' => '<a href="' . esc_url( apply_filters( 'kkart_community_support_url', 'https://wordpress.org/support/plugin/kkart/' ) ) . '" aria-label="' . esc_attr__( 'Visit community forums', 'kkart' ) . '">' . esc_html__( 'Community support', 'kkart' ) . '</a>',
		);

		if ( KKARTConnectionHelper::is_connected() ) {
			$row_meta['premium_support'] = '<a href="' . esc_url( apply_filters( 'kkart_support_url', 'https://kkart.com/my-account/tickets/' ) ) . '" aria-label="' . esc_attr__( 'Visit premium customer support', 'kkart' ) . '">' . esc_html__( 'Premium support', 'kkart' ) . '</a>';
		}

		return array_merge( $links, $row_meta );
	}

	/**
	 * Get slug from path and associate it with the path.
	 *
	 * @param array  $plugins Associative array of plugin files to paths.
	 * @param string $key Plugin relative path. Example: kkart/kkart.php.
	 */
	private static function associate_plugin_file( $plugins, $key ) {
		$path                 = explode( '/', $key );
		$filename             = end( $path );
		$plugins[ $filename ] = $key;
		return $plugins;
	}

	/**
	 * Install a plugin from .org in the background via a cron job (used by
	 * installer - opt in).
	 *
	 * @param string $plugin_to_install_id Plugin ID.
	 * @param array  $plugin_to_install Plugin information.
	 *
	 * @throws Exception If unable to proceed with plugin installation.
	 * @since  2.6.0
	 */
	public static function background_installer( $plugin_to_install_id, $plugin_to_install ) {
		// Explicitly clear the event.
		$args = func_get_args();

		if ( ! empty( $plugin_to_install['repo-slug'] ) ) {
			require_once KKART_ADMIN_DIR . 'includes/file.php';
			require_once KKART_ADMIN_DIR . 'includes/plugin-install.php';
			require_once KKART_ADMIN_DIR . 'includes/class-wp-upgrader.php';
			require_once KKART_ADMIN_DIR . 'includes/plugin.php';

			WP_Filesystem();

			$skin              = new Automatic_Upgrader_Skin();
			$upgrader          = new WP_Upgrader( $skin );
			$installed_plugins = array_reduce( array_keys( get_plugins() ), array( __CLASS__, 'associate_plugin_file' ) );
			if ( empty( $installed_plugins ) ) {
				$installed_plugins = array();
			}
			$plugin_slug = $plugin_to_install['repo-slug'];
			$plugin_file = isset( $plugin_to_install['file'] ) ? $plugin_to_install['file'] : $plugin_slug . '.php';
			$installed   = false;
			$activate    = false;

			// See if the plugin is installed already.
			if ( isset( $installed_plugins[ $plugin_file ] ) ) {
				$installed = true;
				$activate  = ! is_plugin_active( $installed_plugins[ $plugin_file ] );
			}

			// Install this thing!
			if ( ! $installed ) {
				// Suppress feedback.
				ob_start();

				try {
					$plugin_information = plugins_api(
						'plugin_information',
						array(
							'slug'   => $plugin_slug,
							'fields' => array(
								'short_description' => false,
								'sections'          => false,
								'requires'          => false,
								'rating'            => false,
								'ratings'           => false,
								'downloaded'        => false,
								'last_updated'      => false,
								'added'             => false,
								'tags'              => false,
								'homepage'          => false,
								'donate_link'       => false,
								'author_profile'    => false,
								'author'            => false,
							),
						)
					);

					if ( is_wp_error( $plugin_information ) ) {
						throw new Exception( $plugin_information->get_error_message() );
					}

					$package  = $plugin_information->download_link;
					$download = $upgrader->download_package( $package );

					if ( is_wp_error( $download ) ) {
						throw new Exception( $download->get_error_message() );
					}

					$working_dir = $upgrader->unpack_package( $download, true );

					if ( is_wp_error( $working_dir ) ) {
						throw new Exception( $working_dir->get_error_message() );
					}

					$result = $upgrader->install_package(
						array(
							'source'                      => $working_dir,
							'destination'                 => KKART_PLUGIN_DIR,
							'clear_destination'           => false,
							'abort_if_destination_exists' => false,
							'clear_working'               => true,
							'hook_extra'                  => array(
								'type'   => 'plugin',
								'action' => 'install',
							),
						)
					);

					if ( is_wp_error( $result ) ) {
						throw new Exception( $result->get_error_message() );
					}

					$activate = true;

				} catch ( Exception $e ) {
					KKART_Admin_Notices::add_custom_notice(
						$plugin_to_install_id . '_install_error',
						sprintf(
							// translators: 1: plugin name, 2: error message, 3: URL to install plugin manually.
							__( '%1$s could not be installed (%2$s). <a href="%3$s">Please install it manually by clicking here.</a>', 'kkart' ),
							$plugin_to_install['name'],
							$e->getMessage(),
							esc_url( admin_url( 'index.php?kkart-install-plugin-redirect=' . $plugin_slug ) )
						)
					);
				}

				// Discard feedback.
				ob_end_clean();
			}

			wp_clean_plugins_cache();

			// Activate this thing.
			if ( $activate ) {
				try {
					add_action( 'add_option_mailchimp_kkart_plugin_do_activation_redirect', array( __CLASS__, 'remove_mailchimps_redirect' ), 10, 2 );
					$result = activate_plugin( $installed ? $installed_plugins[ $plugin_file ] : $plugin_slug . '/' . $plugin_file );

					if ( is_wp_error( $result ) ) {
						throw new Exception( $result->get_error_message() );
					}
				} catch ( Exception $e ) {
					KKART_Admin_Notices::add_custom_notice(
						$plugin_to_install_id . '_install_error',
						sprintf(
							// translators: 1: plugin name, 2: URL to WP plugin page.
							__( '%1$s was installed but could not be activated. <a href="%2$s">Please activate it manually by clicking here.</a>', 'kkart' ),
							$plugin_to_install['name'],
							admin_url( 'plugins.php' )
						)
					);
				}
			}
		}
	}

	/**
	 * Removes redirect added during MailChimp plugin's activation.
	 *
	 * @param string $option Option name.
	 * @param string $value  Option value.
	 */
	public static function remove_mailchimps_redirect( $option, $value ) {
		// Remove this action to prevent infinite looping.
		remove_action( 'add_option_mailchimp_kkart_plugin_do_activation_redirect', array( __CLASS__, 'remove_mailchimps_redirect' ) );

		// Update redirect back to false.
		update_option( 'mailchimp_kkart_plugin_do_activation_redirect', false );
	}

	/**
	 * Install a theme from .org in the background via a cron job (used by installer - opt in).
	 *
	 * @param string $theme_slug Theme slug.
	 *
	 * @throws Exception If unable to proceed with theme installation.
	 * @since  3.1.0
	 */
	public static function theme_background_installer( $theme_slug ) {
		// Explicitly clear the event.
		$args = func_get_args();

		if ( ! empty( $theme_slug ) ) {
			// Suppress feedback.
			ob_start();

			try {
				$theme = wp_get_theme( $theme_slug );

				if ( ! $theme->exists() ) {
					require_once KKART_ADMIN_DIR . 'includes/file.php';
					include_once KKART_ADMIN_DIR . 'includes/class-wp-upgrader.php';
					include_once KKART_ADMIN_DIR . 'includes/theme.php';

					WP_Filesystem();

					$skin     = new Automatic_Upgrader_Skin();
					$upgrader = new Theme_Upgrader( $skin );
					$api      = themes_api(
						'theme_information',
						array(
							'slug'   => $theme_slug,
							'fields' => array( 'sections' => false ),
						)
					);
					$result   = $upgrader->install( $api->download_link );

					if ( is_wp_error( $result ) ) {
						throw new Exception( $result->get_error_message() );
					} elseif ( is_wp_error( $skin->result ) ) {
						throw new Exception( $skin->result->get_error_message() );
					} elseif ( is_null( $result ) ) {
						throw new Exception( 'Unable to connect to the filesystem. Please confirm your credentials.' );
					}
				}

				switch_theme( $theme_slug );
			} catch ( Exception $e ) {
				KKART_Admin_Notices::add_custom_notice(
					$theme_slug . '_install_error',
					sprintf(
						// translators: 1: theme slug, 2: error message, 3: URL to install theme manually.
						__( '%1$s could not be installed (%2$s). <a href="%3$s">Please install it manually by clicking here.</a>', 'kkart' ),
						$theme_slug,
						$e->getMessage(),
						esc_url( admin_url( 'update.php?action=install-theme&theme=' . $theme_slug . '&_wpnonce=' . wp_create_nonce( 'install-theme_' . $theme_slug ) ) )
					)
				);
			}

			// Discard feedback.
			ob_end_clean();
		}
	}
}

KKART_Install::init();
