<?php
namespace Blesta\Core\Util\Captcha\Captchas;

use Blesta\Core\Util\Captcha\Common\AbstractCaptcha;
use Gregwar\Captcha\CaptchaBuilder;
use RuntimeException;
use Configure;
use Language;
use Loader;

/**
 * Internal captcha integration
 *
 * @package blesta
 * @subpackage blesta.core.Util.Captcha.Captchas
 * @copyright Copyright (c) 2020, Phillips Data, Inc.
 * @license http://www.blesta.com/license/ The Blesta License Agreement
 * @link http://www.blesta.com/ Blesta
 */
class InternalCaptcha extends AbstractCaptcha
{
    /**
     * @var array An array of options
     */
    private $options = [];

    /**
     * Builds the HTML content to render the reCaptcha
     *
     * @return string The HTML
     */
    public function buildHtml()
    {
        Loader::loadComponents($this, ['Session']);

        // Autoload the language file
        Language::loadLang(
            'internal_captcha',
            $this->Html->safe(
                $this->Html->ifSet($this->options['lang']),
                Configure::get('Blesta.language')
            ),
            COREDIR . 'Util' . DS . 'Captcha' . DS . 'Captchas' . DS . 'language' . DS
        );

        // Build the captcha
        $builder = new CaptchaBuilder();
        $builder->build();

        // Store the generated phrase in the current session
        $phrase = $builder->getPhrase();
        $this->Session->write('phrase', $phrase);

        $html = '
<div class="panel-body" style="max-width: 300px; border-radius: 4px;">
    <img src="' . $builder->inline() . '" style="height: 50px; border-radius: 4px; margin-bottom: 10px;" />
    <input type="text" name="phrase" value="" class="form-control" placeholder="' .
                Language::_('Util.captchas.internal_captcha.type_the_text', true) . '">
</div>';

        return $html;
    }

    /**
     * Sets internal captcha options
     *
     * @param array $options An array of options including:
     *
     *  - lang The user's language (e.g. "en_us" for English)
     *  - ip_address The user's IP address (optional)
     */
    public function setOptions(array $options)
    {
        $this->options = $options;
    }

    /**
     * Verifies whether or not the captcha is valid
     *
     * @param array $data An array of data to validate against, including:
     *
     *  - response The value of 'phrase' in the submitted form
     * @return bool Whether or not the captcha is valid
     */
    public function verify(array $data)
    {
        Loader::loadComponents($this, ['Session']);

        $success = false;

        // Attempt to verify the captcha was accepted
        try {
            $phrase = $this->Session->read('phrase');
            $success = $this->Html->ifSet($data['response']) === $phrase;
        } catch (RuntimeException $e) {
            // Internal captcha could not process the request due to missing data
            $this->setErrors(['internalcaptcha' => ['error' => $e->getMessage()]]);
        }

        return $success;
    }
}
