<?php
/**
 * The admin-specific utilities methods for the plugin
 *
 * @link http://www.boldgrid.com
 * @since 1.0
 *
 * @package Boldgrid_Backup
 * @subpackage Boldgrid_Backup/admin
 * @copyright BoldGrid.com
 * @version $Id$
 * @author BoldGrid.com <wpb@boldgrid.com>
 */

/**
 * BoldGrid Backup admin settings class.
 *
 * @since 1.0
 */
class Boldgrid_Backup_Admin_Settings {
	/**
	 * The core class object.
	 *
	 * @since 1.0
	 * @access private
	 * @var Boldgrid_Backup_Admin_Core
	 */
	private $core;

	/**
	 * Whether or not we're in the middle of saving settings from $_POST.
	 *
	 * @since  1.5.4
	 * @access public
	 * @var    bool
	 */
	public $is_saving_settings = false;

	/**
	 * Constructor.
	 *
	 * @since 1.0
	 *
	 * @param Boldgrid_Backup_Admin_Core $core Core class object.
	 */
	public function __construct( $core ) {
		// Save the Boldgrid_Backup_Admin_Core object as a class property.
		$this->core = $core;

		$this->is_saving_settings = isset( $_POST['save_time'] );
	}

	/**
	 * How many days of the week are being saved?
	 *
	 * This method counts the number of $_POST keys that begin with "dow_". The count returned by
	 * this method is used to help enforce restrictions on the free version of the plugin.
	 *
	 * @since 1.3.1
	 *
	 * @return int
	 */
	public function get_dow_count() {
		$count = 0;

		if ( ! isset( $_POST ) || ! is_array( $_POST ) ) {
			return 0;
		}

		// Loop through each $_POST value and check if the key begins with dow_.
		foreach ( $_POST as $k => $v ) {
			if ( substr( $k, 0, 4 ) === 'dow_' ) {
				$count++;
			}
		}

		return $count;
	}

	/**
	 * Get settings using defaults.
	 *
	 * @since 1.0
	 *
	 * @param  bool $raw Return the raw settings. This would happen very early
	 *                   on in this method, before all of the other checks
	 *                   happen.
	 * @return array     An array of settings.
	 */
	public function get_settings( $raw = false ) {
		// Get settings.
		$settings = get_site_option( 'boldgrid_backup_settings', array() );

		if ( $raw ) {
			return $settings;
		}

		// Configure a random minute. 5:4am will fail, but 5:04am will pass.
		$random_minute = mt_rand( 1, 59 );
		$random_minute = 1 === strlen( $random_minute ) ? '0' . $random_minute : $random_minute;

		// Parse settings.
		if ( ! empty( $settings['schedule'] ) ) {
			// Update schedule format.
			// Days of the week.
			$settings['schedule']['dow_sunday'] = (
				! empty( $settings['schedule']['dow_sunday'] ) ? 1 : 0
			);
			$settings['schedule']['dow_monday'] = (
				! empty( $settings['schedule']['dow_monday'] ) ? 1 : 0
			);
			$settings['schedule']['dow_tuesday'] = (
				! empty( $settings['schedule']['dow_tuesday'] ) ? 1 : 0
			);
			$settings['schedule']['dow_wednesday'] = (
				! empty( $settings['schedule']['dow_wednesday'] ) ? 1 : 0
			);
			$settings['schedule']['dow_thursday'] = (
				! empty( $settings['schedule']['dow_thursday'] ) ? 1 : 0
			);
			$settings['schedule']['dow_friday'] = (
				! empty( $settings['schedule']['dow_friday'] ) ? 1 : 0
			);
			$settings['schedule']['dow_saturday'] = (
				! empty( $settings['schedule']['dow_saturday'] ) ? 1 : 0
			);

			// Time of day.
			$settings['schedule']['tod_h'] = (
				! empty( $settings['schedule']['tod_h'] ) ?
				$settings['schedule']['tod_h'] : mt_rand( 1, 5 )
			);
			$settings['schedule']['tod_m'] = (
				! empty( $settings['schedule']['tod_m'] ) ?
				$settings['schedule']['tod_m'] : $random_minute
			);
			$settings['schedule']['tod_a'] = (
				! empty( $settings['schedule']['tod_a'] ) ?
				$settings['schedule']['tod_a'] : 'AM'
			);

			// Notification settings.
			$settings['notifications']['backup'] = (
				! isset( $settings['notifications']['backup'] ) ||
				! empty( $settings['notifications']['backup'] ) ? 1 : 0
			);
			$settings['notifications']['restore'] = (
				! isset( $settings['notifications']['restore'] ) ||
				! empty( $settings['notifications']['restore'] ) ? 1 : 0
			);

			// Notification email address.
			if ( empty( $settings['notification_email'] ) ) {
				$settings['notification_email'] = $this->core->config->get_admin_email();
			}

			// Other settings.
			$settings['auto_backup'] = (
				! isset( $settings['auto_backup'] ) || ! empty( $settings['auto_backup'] ) ? 1 : 0
			);
			$settings['auto_rollback'] = (
				! isset( $settings['auto_rollback'] ) || ! empty( $settings['auto_rollback'] ) ?
				1 : 0
			);
		} else {
			// Define defaults.
			// Days of the week.
			$settings['schedule']['dow_sunday'] = 0;
			$settings['schedule']['dow_monday'] = 0;
			$settings['schedule']['dow_tuesday'] = 0;
			$settings['schedule']['dow_wednesday'] = 0;
			$settings['schedule']['dow_thursday'] = 0;
			$settings['schedule']['dow_friday'] = 0;
			$settings['schedule']['dow_saturday'] = 0;

			// Time of day.
			$settings['schedule']['tod_h'] = mt_rand( 1, 5 );
			$settings['schedule']['tod_m'] = $random_minute;
			$settings['schedule']['tod_a'] = 'AM';

			// Other settings.
			$settings['retention_count'] = 5;
			$settings['notification_email'] = $this->core->config->get_admin_email();
			$settings['notifications']['backup'] = 1;
			$settings['notifications']['restore'] = 1;
			$settings['auto_backup'] = 1;
			$settings['auto_rollback'] = 1;
		}

		/*
		 * If a cron schedule was found, then merge the settings.
		 *
		 * @todo As of 1.6.0, this feature is on hold. We need to take into
		 * account timezones, and possibly tell the user their settings don't
		 * actually match what's in the crontab.
		 */
		/*
		// If not updating the settings, then check cron for schedule.
		if ( ! isset( $_POST['save_time'] ) ) {
			$cron_schedule = $this->core->cron->read_cron_entry();
		}

		// If a cron schedule was found, then merge the settings.
		if ( ! empty( $cron_schedule ) ) {
			$settings['schedule'] = array_merge( $settings['schedule'], $cron_schedule );
		}
		*/

		$boldgrid_settings = get_site_option( 'boldgrid_settings' );

		$settings['plugin_autoupdate'] = (
			! empty( $boldgrid_settings['plugin_autoupdate'] ) ? 1 : 0
		);

		$settings['theme_autoupdate'] = (
			! empty( $boldgrid_settings['theme_autoupdate'] ) ? 1 : 0
		);

		if ( empty( $settings['remote'] ) ) {
			$settings['remote'] = array();
		}

		// For consistency, untrailingslashit the backup dir.
		if ( isset( $settings['backup_directory'] ) ) {
			$settings['backup_directory'] = untrailingslashit( $settings['backup_directory'] );
		}

		if ( empty( $settings['exclude_tables'] ) ) {
			$settings['exclude_tables'] = array();
		}

		// Configure default folder_exclusion settings.
		$settings['folder_exclusion_include'] = $this->core->folder_exclusion->from_settings( 'include', $settings );
		$settings['folder_exclusion_exclude'] = $this->core->folder_exclusion->from_settings( 'exclude', $settings );

		// Return the settings array.
		return $settings;
	}

	/**
	 * Move backups from one directory to another.
	 *
	 * @since 1.3.2
	 *
	 * @param string $old_dir
	 * @param string $new_dir
	 * @return bool True on success / no backups needed to be moved.
	 */
	private function move_backups( $old_dir, $new_dir ) {
		$fail_count = 0;

		$old_dir = Boldgrid_Backup_Admin_Utility::trailingslashit( $old_dir );
		$new_dir = Boldgrid_Backup_Admin_Utility::trailingslashit( $new_dir );

		$archives = $this->core->get_archive_list( null, $old_dir );

		ignore_user_abort( true );

		// Loop through each archive and move it.
		foreach ( $archives as $archive ) {
			$source = $archive['filepath'];
			$destination = $new_dir . $archive['filename'];

			$success = @$this->core->wp_filesystem->move( $source, $destination );

			if ( ! $success ) {
				$fail_count++;
			}
		}

		return 0 === $fail_count;
	}

	/**
	 * Update settings.
	 *
	 * @since 1.0
	 * @access private
	 *
	 * @see Boldgrid_Backup_Admin_Cron::add_cron_entry()
	 * @see Boldgrid_Backup_Admin_Cron::get_cron_secret()
	 *
	 * @return bool Update success.
	 */
	private function update_settings() {
		$update_errors = array();

		// Verify nonce.
		check_admin_referer( 'boldgrid-backup-settings', 'settings_auth' );

		// Get the retention count.
		if ( isset( $_POST['retention_count'] ) ) {
			$retention_count = intval( $_POST['retention_count'] );
		} else {
			$retention_count = $this->core->config->get_default_retention();
		}

		// Check for settings update.
		if ( ! empty( $_POST['save_time'] ) ) {
			// Get settings.
			$settings = $this->get_settings();

			// Initialize $update_error.
			$update_error = false;

			// Initialize $days_scheduled.
			$days_scheduled = array();

			// Validate input for schedule.
			$indices = array(
				'dow_sunday',
				'dow_monday',
				'dow_tuesday',
				'dow_wednesday',
				'dow_thursday',
				'dow_friday',
				'dow_saturday',
				'tod_h',
				'tod_m',
				'tod_a',
			);

			foreach ( $indices as $index ) {
				// Determine input type.
				if ( 0 === strpos( $index, 'dow_' ) ) {
					$type = 'day';
				} elseif ( 'tod_h' === $index ) {
					$type = 'h';
				} elseif ( 'tod_m' === $index ) {
					$type = 'm';
				} elseif ( 'tod_a' === $index ) {
					$type = 'a';
				} else {
					// Unknown type.
					$type = '?';
				}

				if ( ! empty( $_POST[ $index ] ) ) {
					// Validate by type.
					switch ( $type ) {
						case 'day' :
							// Convert to integer.
							$_POST[ $index ] = (int) $_POST[ $index ];

							// If day was scheduled, then track it.
							if ( 1 === $_POST[ $index ] ) {
								$days_scheduled[] = date( 'w', strtotime( str_replace( 'dow_', '', $index ) ) );
							}

							break;
						case 'h' :
							if ( $_POST[ $index ] < 1 || $_POST[ $index ] > 12 ) {
								// Error in input.
								$update_error = true;
								break 2;
							}

							// Convert to integer.
							$_POST[ $index ] = (int) $_POST[ $index ];

							break;
						case 'm' :
							if ( $_POST[ $index ] < 0 || $_POST[ $index ] > 59 ) {
								// Error in input.
								$update_error = true;
								break 2;
							}

							// Convert to integer.
							$_POST[ $index ] = (int) $_POST[ $index ];

							// Pad left with 0.
							$_POST[ $index ] = str_pad( $_POST[ $index ], 2, '0', STR_PAD_LEFT );

							break;
						case 'a' :
							if ( 'AM' !== $_POST[ $index ] && 'PM' !== $_POST[ $index ] ) {
								// Error in input; unknown type.
								$update_error = true;
								break 2;
							}

							break;
						default :
							// Error in input; unknown type.
							$update_error = true;
							break 2;
					}

					// Update the setting value provided.
					$settings['schedule'][ $index ] = $_POST[ $index ];
				} elseif ( 'day' === $type ) {
					// Unassigned days.
					$settings['schedule'][ $index ] = 0;
				} else {
					// Error in input.
					$update_error = true;

					break;
				}
			}

			// Validate input for other settings.
			$settings['retention_count'] = (
				isset( $_POST['retention_count'] ) ? (int) $_POST['retention_count'] : 5
			);

			$settings['notifications']['backup'] = (
				( isset( $_POST['notify_backup'] ) && '1' === $_POST['notify_backup'] ) ? 1 : 0
			);

			$settings['notifications']['restore'] = (
				( isset( $_POST['notify_restore'] ) && '1' === $_POST['notify_restore'] ) ? 1 : 0
			);

			$settings['auto_backup'] = (
				( ! isset( $_POST['auto_backup'] ) || '1' === $_POST['auto_backup'] ) ? 1 : 0
			);

			$settings['auto_rollback'] = (
				( ! isset( $_POST['auto_rollback'] ) || '1' === $_POST['auto_rollback'] ) ? 1 : 0
			);

			// Update notification email address, if changed.
			if ( isset( $settings['notification_email'] ) &&
			sanitize_email( $_POST['notification_email'] ) !== $settings['notification_email'] ) {
				$settings['notification_email'] = sanitize_email( $_POST['notification_email'] );
			}

			$boldgrid_settings['plugin_autoupdate'] = (
				( isset( $_POST['plugin_autoupdate'] ) && '1' === $_POST['plugin_autoupdate'] ) ?
				1 : 0
			);

			$boldgrid_settings['theme_autoupdate'] = (
				( isset( $_POST['theme_autoupdate'] ) && '1' === $_POST['theme_autoupdate'] ) ?
				1 : 0
			);

			unset( $settings['plugin_autoupdate'], $settings['theme_autoupdate'] );

			// Get the current backup directory path.
			$backup_dir_changed = false;
			$original_backup_directory = ! empty( $settings['backup_directory'] ) ? $settings['backup_directory'] : false;

			if ( ! empty( $_POST['backup_directory'] ) ) {
				$post_backup_directory = trim( $_POST['backup_directory'] );
				$post_backup_directory = untrailingslashit( $post_backup_directory );
				$post_backup_directory = str_replace( '\\\\', '\\', $post_backup_directory );
			}

			/*
			 * Create the backup directory.
			 *
			 * Allow the user to submit a blank backup directory if they want
			 * to set the backup directory to the default.
			 */
			if ( empty( $_POST['backup_directory'] ) ) {
				// The get method validates and creates the backup directory.
				$backup_directory = $this->core->backup_dir->guess_and_set();

				$backup_dir_changed = $original_backup_directory !== $backup_directory;
			} elseif ( $post_backup_directory !== $original_backup_directory ) {
				$backup_directory = $post_backup_directory;

				/*
				 * Create the backup directory.
				 *
				 * Even if the backup directory already exists, we still want to
				 * run the create method so that the necessary .htaccess and other
				 * files are created to protect the directory.
				 */
				$backup_directory = $this->core->backup_dir->create( $backup_directory );

				// Make sure that the backup directory has proper permissions.
				$valid = $this->core->backup_dir->is_valid( $backup_directory );
				if ( ! $valid ) {
					$backup_directory = false;
				}

				$backup_dir_changed = true;
			}

			if ( $backup_dir_changed ) {
				if ( false === $backup_directory ) {
					$update_error = true;
					$backup_dir_changed = false;
					$update_errors = array_merge( $update_errors, $this->core->backup_dir->errors );
				} else {
					$settings['backup_directory'] = $backup_directory;
				}
			}

			// Move backups to the new directory.
			if ( $backup_dir_changed && isset( $_POST['move-backups'] ) && 'on' === $_POST['move-backups'] ) {
				$backups_moved = $this->move_backups( $original_backup_directory, $backup_directory );

				if ( ! $backups_moved ) {
					$update_error = true;
					$update_errors[] = sprintf( __( 'Unable to move backups from %1$s to %2$s', 'boldgrid-backup' ), $original_backup_directory, $backup_directory );
				}
			}

			/*
			 * Save compressor settings.
			 *
			 * @since 1.5.1
			 */
			if ( ! empty( $_POST['compressor'] ) ) {
				$available_compressors = $this->core->compressors->get_available();
				$selected_compressor = $_POST['compressor'];
				if ( in_array( $selected_compressor, $available_compressors, true ) ) {
					$settings['compressor'] = $selected_compressor;
				} else {
					$update_error = true;
					$update_errors[] = __( 'The compressor you seleted is unavailable. Please select another.', 'boldgrid-backup' );
				}
			}

			/*
			 * Save extractor settings.
			 *
			 * @since 1.5.1
			 */
			if ( ! empty( $_POST['extractor'] ) ) {
				$selected_extractor = $_POST['extractor'];
				if ( in_array( $selected_extractor, $available_compressors, true ) ) {
					$settings['extractor'] = $selected_extractor;
				} else {
					$update_error = true;
					$update_errors[] = __( 'The extractor you seleted is unavailable. Please select another.', 'boldgrid-backup' );
				}
			}

			/*
			 * Change the scheduler.
			 *
			 * If the scheduler is indeed changed, clear all prior backup
			 * schedules.
			 *
			 * @since 1.5.1
			 */
			$original_scheduler = ! empty( $settings['scheduler'] ) ? $settings['scheduler'] : false;
			$schedulers_available = $this->core->scheduler->get_available();
			$scheduler_changed = ! empty( $_POST['scheduler'] ) && $original_scheduler !== $_POST['scheduler'];
			if ( $scheduler_changed && array_key_exists( $_POST['scheduler'], $schedulers_available ) ) {
				$settings['scheduler'] = $_POST['scheduler'];
			}

			/*
			 * Save WP Cron / Crons.
			 *
			 * @since 1.5.1
			 */
			$scheduler = ! empty( $settings['scheduler'] ) ? $settings['scheduler'] : null;
			if ( 'wp-cron' === $scheduler ) {
				$crons_added = $this->core->wp_cron->add_all_crons( $settings );
			} elseif ( 'cron' === $scheduler ) {
				$crons_added = $this->core->cron->add_all_crons( $settings );
				$settings['crontab_version'] = $this->core->cron->crontab_version;
				$settings['cron_secret'] = $this->core->cron->get_cron_secret();
			}
			// Take action if we tried and failed to add crons.
			if ( isset( $crons_added ) && ! $crons_added ) {
				$update_error = true;
				$update_errors[] = esc_html__( 'An error occurred when modifying cron jobs. Please try again.', 'boldgrid-backup' );
			}

			/*
			 * Save storage locations.
			 *
			 * @since 1.5.2
			 */
			$storage_locations = ! empty( $settings['remote'] ) ? $settings['remote'] : array();

			// Start off by disabling each storage location.
			foreach ( $storage_locations as $remote_key => $storage_location ) {
				$settings['remote'][ $remote_key ]['enabled'] = false;
			}

			// Get the storage location array from POST, then sanitize below.
			$storage_locations_save = ! empty( $_POST['storage_location'] ) ?
				$_POST['storage_location'] : array();

			// Then enable it only if submitted.  Values are not used, only key/index.
			foreach ( $storage_locations_save as $storage_location => $storage_location_enabled ) {
				$storage_location = sanitize_key( $storage_location );

				/*
				 * Unless this is the local environment, don't enable a storage location if it has
				 * not yet been setup. Why enable FTP if FTP hasn't been setup yet.
				 */
				if ( 'local' === $storage_location || isset( $settings['remote'][ $storage_location ] ) ) {
					$settings['remote'][ $storage_location ]['enabled'] = true;
				}
			}

			/*
			 * Save tables to include.
			 *
			 * @since 1.5.3
			 */
			$settings['exclude_tables'] = $this->core->db_omit->get_from_post();
			$settings['exclude_tables_type'] = $this->core->db_omit->get_post_type();

			/*
			 * Save folder exclusion settings.
			 *
			 * @since 1.5.4
			 */
			$settings['folder_exclusion_include'] = $this->core->folder_exclusion->from_post( 'include' );
			$settings['folder_exclusion_exclude'] = $this->core->folder_exclusion->from_post( 'exclude' );
			$settings['folder_exclusion_type'] = $this->core->folder_exclusion->from_post( 'type' );

			// If no errors, then save the settings.
			if ( ! $update_error ) {
				$settings['updated'] = time();
				update_site_option( 'boldgrid_backup_settings', $settings );
				$this->update_boldgrid_settings( $boldgrid_settings );
			}
		}

		// If there was no error, then show success notice.
		if ( ! $update_error ) {
			// Success.
			do_action(
				'boldgrid_backup_notice',
				esc_html__( 'Settings saved.', 'boldgrid-backup' ),
				'updated settings-error notice is-dismissible'
			);
		} elseif ( empty( $update_errors ) ) {
			$failure_message = esc_html__( 'Invalid settings submitted.  Please try again.', 'boldgrid-backup' );
		} else {
			$failure_message = sprintf( '<strong>%1$s</strong><br /><br />%2$s', __( 'We were unable to save your settings for the following reason(s):', 'boldgrid-backup' ), implode( '<br /><br />', $update_errors ) );
		}

		if ( isset( $failure_message ) ) {
			do_action( 'boldgrid_backup_notice', $failure_message );
		}

		if ( ! $update_error ) {
			/**
			 * Take action when settings have been updated.
			 *
			 * @since 1.5.3
			 */
			do_action( 'boldgrid_backup_settings_updated' );
		}

		// Return success.
		return ! $update_error;
	}

	/**
	 * Delete the boldgrid_backup_pending_rollback option.
	 *
	 * @since 1.0.1
	 */
	public function delete_rollback_option() {
		delete_site_option( 'boldgrid_backup_pending_rollback' );
	}

	/**
	 * Menu callback to display the Backup schedule page.
	 *
	 * @since 1.0
	 *
	 * @return null
	 */
	public function page_backup_settings() {
		add_thickbox();
		wp_enqueue_style( 'boldgrid-backup-admin-new-thickbox-style' );

		wp_enqueue_style( 'bglib-ui-css' );
		wp_enqueue_script( 'bglib-ui-js' );
		wp_enqueue_script( 'bglib-sticky' );

		wp_enqueue_script( 'bglib-license' );

		if ( ! $this->is_saving_settings ) {
			$is_functional = $this->core->test->run_functionality_tests();
		}

		// If tests fail, then show an admin notice and abort.
		if ( isset( $is_functional ) && ! $is_functional ) {
			do_action(
				'boldgrid_backup_notice',
				sprintf(
					esc_html__(
						'Functionality test has failed.  You can go to %1$sFunctionality Test%1$s to view a report.',
						'boldgrid-backup'
					),
					'<a href="' . admin_url( 'admin.php?page=boldgrid-backup-test' ) . '">',
					'</a>'
				),
				'notice notice-error is-dismissible'
			);
		}

		// Display warning on resource usage and backups.
		do_action(
			'boldgrid_backup_notice',
			esc_html__(
				'Warning: Making backups uses resources. When the system is backing up, it will slow down your site for visitors. Furthermore, when the database itself is being copied, your site must “pause” temporarily to preserve data integrity. For most sites, the pause is typically a few seconds and is not noticed by visitors. Large sites take longer though. Please keep the number of backups you have stored and how often you make those backups to a minimum.',
				'boldgrid-backup'
			),
			'notice notice-warning is-dismissible'
		);

		// Get BoldGrid reseller settings.
		$boldgrid_reseller = get_option( 'boldgrid_reseller' );

		// If not part of a reseller, then show the unofficial host notice.
		if ( empty( $boldgrid_reseller ) ) {
			do_action(
				'boldgrid_backup_notice',
				esc_html__(
					'Please note that your web hosting provider may have a policy against these types of backups. Please verify with your provider or choose a BoldGrid Official Host.',
					'boldgrid-backup'
				),
				'notice notice-warning is-dismissible'
			);
		}

		// Check for settings update.
		if ( $this->is_saving_settings ) {
			// Verify nonce.
			check_admin_referer( 'boldgrid-backup-settings', 'settings_auth' );

			$this->update_settings();
		}

		// Enqueue CSS for the settings page.
		wp_enqueue_style( 'boldgrid-backup-admin-settings',
			plugin_dir_url( __FILE__ ) . 'css/boldgrid-backup-admin-settings.css', array(),
			BOLDGRID_BACKUP_VERSION, 'all'
		);

		// Enqueue the JS for the settings page.
		wp_enqueue_script( 'boldgrid-backup-admin-settings',
			plugin_dir_url( __FILE__ ) . 'js/boldgrid-backup-admin-settings.js',
			array( 'jquery' ),
			BOLDGRID_BACKUP_VERSION,
			false
		);

		$this->core->folder_exclusion->enqueue_scripts();
		$this->core->db_omit->enqueue_scripts();

		$settings = $this->get_settings();

		// If the directory path is not in the settings, then add it for the form.
		if ( empty( $settings['backup_directory'] ) ) {
			$settings['backup_directory'] = $this->core->backup_dir->get();
		}

		$available_compressors = $this->core->compressors->get_available();

		// Include the page template.
		include BOLDGRID_BACKUP_PATH . '/admin/partials/boldgrid-backup-admin-settings.php';

		return;
	}

	/**
	 * Save our settings.
	 *
	 * @since 1.5.2
	 *
	 * @param  array $settings
	 * @return bool True on success.
	 */
	public function save( $settings ) {

		// For consistency, untrailingslashit the backup dir.
		if ( isset( $settings['backup_directory'] ) ) {
			$settings['backup_directory'] = untrailingslashit( $settings['backup_directory'] );
		}

		return update_site_option( 'boldgrid_backup_settings', $settings );
	}

	/**
	 * Update BoldGrid general settings.
	 *
	 * @since 1.3.11
	 *
	 * @param array $settings Array of BoldGrid settings.
	 * @return bool
	 */
	public function update_boldgrid_settings( array $settings ) {
		$boldgrid_settings = get_site_option( 'boldgrid_settings' );

		$boldgrid_settings['plugin_autoupdate'] = (
			( isset( $settings['plugin_autoupdate'] ) && 1 === $settings['plugin_autoupdate'] ) ?
			1 : 0
		);

		$boldgrid_settings['theme_autoupdate'] = (
			( isset( $settings['theme_autoupdate'] ) && 1 === $settings['theme_autoupdate'] ) ?
			1 : 0
		);

		return update_site_option( 'boldgrid_settings', $boldgrid_settings );
	}
}
