<?php
/**
 * History class.
 *
 * @link  http://www.boldgrid.com
 * @since 1.5.3
 *
 * @package    Boldgrid_Backup
 * @subpackage Boldgrid_Backup/admin
 * @copyright  BoldGrid.com
 * @version    $Id$
 * @author     BoldGrid.com <wpb@boldgrid.com>
 */

/**
 * History class.
 *
 * @since 1.5.3
 */
class Boldgrid_Backup_Premium_Admin_History {

	/**
	 * The core class object.
	 *
	 * @since 1.5.3
	 * @var Boldgrid_Backup_Admin_Core
	 */
	private $core;

	/**
	 * Max number of entries to keep in the history.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	public $max_entries = 100;

	/**
	 * An instance of Boldgrid_Backup_Premium_Admin_Core.
	 *
	 * @since 1.5.3
	 * @var   Boldgrid_Backup_Premium_Admin_Core
	 */
	private $premium_core;

	/**
	 * Constructor.
	 *
	 * @since 1.5.3
	 *
	 * @param Boldgrid_Backup_Admin_Core $core Core class object.
	 */
	public function __construct( $core, $premium_core ) {
		$this->core = $core;
		$this->premium_core = $premium_core;
	}

	/**
	 * Add a message to the history log.
	 *
	 * @since 1.5.3
	 *
	 * @param string $message
	 */
	public function add( $message ) {
		if( empty( $message ) ) {
			return;
		}

		$history = $this->get();

		// Determine our user_id.
		$sapi_type = php_sapi_name();
		if( $this->core->doing_cron && substr( $sapi_type, 0, 3 ) === 'cli' ) {
			$user_id = 'Cron';
		} elseif( $this->core->doing_cron ) {
			$user_id = 'WP Cron';
		} else {
			$user_id = get_current_user_id();
		}

		$history[] = array(
			'user_id' => $user_id,
			'timestamp' => time(),
			'message' => $message,
		);

		$this->save( $history );
	}

	/**
	 * Enqueue scripts for the history page.
	 *
	 * @since 1.5.3
	 */
	public function admin_enqueue_scripts( $hook ) {
		if( 'admin_page_boldgrid-backup-history' !== $hook ) {
			return;
		}
	}

	/**
	 *
	 */
	public function filter_tools_section( $sections ) {
		$history = $this->get();

		$sections['sections'][] = array(
			'id' => 'section_history',
			'title' => __( 'History', 'boldgrid-backup' ),
			'content' => include BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/partials/history.php',
		);

		return $sections;
	}

	/**
	 * Get history.
	 *
	 * @since 1.5.3
	 *
	 * @return array
	 */
	public function get() {
		$history = get_site_option( 'boldgrid_backup_history', array() );

		return $history;
	}

	/**
	 * Take action after a backup has been generated.
	 *
	 * @since 1.5.3
	 *
	 * @param array $info
	 */
	public function post_archive_files( $info ) {
		if( empty( $info['filepath'] ) ) {
			return;
		}

		$this->add( sprintf(
			__( 'Backup file created: %1$s.', 'boldgrid-backup' ),
			$info['filepath']
		));
	}

	/**
	 * Take action when a remove backup has been deleted due to retention
	 * settings.
	 *
	 * @since 1.5.3
	 *
	 * @param string $title
	 * @param string $message
	 */
	public function remote_retention_deleted( $title, $message ) {
		if( empty( $title ) || empty( $message ) ) {
			return;
		}

		$this->add( sprintf(
			__( 'Due to your retention settings with %1$s, the following was deleted remotely: %2$s.', 'boldgrid-backup' ),
			$title,
			$message
		));
	}

	/**
	 * Take action when a file has been uploaded to a remove storage provider.
	 *
	 * @since 1.5.3
	 *
	 * @param string $title
	 * @param string $filepath
	 */
	public function remote_uploaded( $title, $filepath ) {
		if( empty( $title ) || empty( $filepath ) ) {
			return;
		}

		$this->add( sprintf(
			__( 'Backup file %1$s was uploaded to %2$s.', 'boldgrid-backup' ),
			$filepath,
			$title
		));
	}

	/**
	 * Take action when a local backup has been deleted due to retention
	 * settings.
	 *
	 * @since 1.5.3
	 *
	 * @param string $filepath
	 */
	public function retention_deleted( $filepath ) {
		if( empty( $filepath ) ) {
			return;
		}

		$this->add( sprintf(
			__( 'Due to retention settings, the following backup was deleted: %1$s', 'boldgrid-backup' ),
			$filepath
		));
	}

	/**
	 * Take action when a plugin has been deleted.
	 *
	 * @since 1.5.3
	 *
	 * @param string $plugin_file Plugin file name.
	 */
	public function delete_plugin( $plugin_file ) {
		$data = $this->core->utility->get_plugin_data( $plugin_file );
		$this->add( sprintf( __( '%1$s plugin (version %2$s) deleted.', 'boldgrid-backup' ), $data['Name'], $data['Version'] ) );
	}

	/**
	 * Save our history.
	 *
	 * @since 1.5.3
	 *
	 * @param  array $history
	 * @return bool
	 */
	public function save( $history ) {
		if( ! is_array( $history ) ) {
			return false;
		}

		$number_to_delete = count( $history ) - $this->max_entries;

		if( $number_to_delete > 0 ) {
			for( $x = 1; $x <= $number_to_delete; $x++ ) {
				array_shift( $history );
			}
		}

		$updated = update_site_option( 'boldgrid_backup_history', $history );

		return $updated;
	}

	/**
	 * Take action when the BoldGrid Backup settings have been updated.
	 *
	 * @since 1.5.3
	 */
	public function settings_updated() {
		$this->add( __( 'BoldGrid Backup settings updated.', 'boldgrid-backup' ) );
	}

	/**
	 * Take action when a theme has been switched.
	 *
	 * @since 1.5.3
	 *
	 * @param string   $new_name  Name of the new theme.
     * @param WP_Theme $new_theme WP_Theme instance of the new theme.
     * @param WP_Theme $old_theme WP_Theme instance of the old theme.
	 */
	public function switch_theme( $new_name, $new_theme, $old_theme ) {
		$this->add( sprintf( __( '%1$s theme (version %2$s) deactivated.', 'boldgrid-backup' ), $old_theme->get( 'Name' ), $old_theme->get( 'Version' ) ) );
		$this->add( sprintf( __( '%1$s theme (version %2$s) activated.', 'boldgrid-backup' ), $new_theme->get( 'Name' ), $new_theme->get( 'Version' ) ) );
	}

	/**
	 * Take option when the active_plugins option is updated.
	 *
	 * Read the values, determine which plugins were activated and which were
	 * deactivated.
	 *
	 * @since 1.5.3
	 *
	 * @param array  old_value
	 * @param array  $value
	 * @param string $option
	 */
	public function update_option_active_plugins( $old_value, $value, $option ) {
		$old_value = ! is_array( $old_value ) ? array() : $old_value;
		$value = ! is_array( $value ) ? array() : $value;

		$activated = array_diff( $value, $old_value );
		$deactivated = array_diff( $old_value, $value );

		foreach( $activated as $key => $plugin ) {
			$data = $this->core->utility->get_plugin_data( $plugin );
			$this->add( sprintf( __( '%1$s plugin (version %2$s) activated.', 'boldgrid-backup' ), $data['Name'], $data['Version'] ));
		}

		foreach( $deactivated as $key => $plugin ) {
			$data = $this->core->utility->get_plugin_data( $plugin );
			$this->add( sprintf( __( '%1$s plugin (version %2$s) deactivated.', 'boldgrid-backup' ), $data['Name'], $data['Version'] ));
		}
	}

	/**
	 * Log whenever core, a plugin, or a theme are upgraded.
	 *
	 * @since 1.5.3
	 *
	 * @param object $upgrader_object
	 * @param array $options Example: https://pastebin.com/ah4E048B
	 */
	public function upgrader_process_complete( $upgrader_object, $options ) {
		$action = ! empty( $options['action'] ) ? $options['action'] : null;
		$type = ! empty( $options['type'] ) ? $options['type'] : null;

		if( 'update' !== $action ) {
			return;
		}

		switch( $type ) {
			case 'core':
				$wordpress_version = get_bloginfo( 'version' );
				$this->add( sprintf(
					__( 'WordPress updated to version %1$s.', 'boldgrid-backup' ),
					get_bloginfo( 'version' )
				));
				break;
			case 'theme':
				foreach( $options['themes'] as $theme ) {
					$data = wp_get_theme( $theme );
					$this->add( sprintf(
						__( '%1$s theme upgraded to version %2$s.', 'boldgrid-backup' ),
						$data->get( 'Name' ),
						$data->get( 'Version' )
					));
				}
				break;
			case 'plugin':
				foreach( $options['plugins'] as $plugin ) {
					$data = $this->core->utility->get_plugin_data( $plugin );
					$this->add( sprintf(
						__( '%1$s plugin upgraded to version %2$s.', 'boldgrid-backup' ),
						$data['Name'],
						$data['Version']
					));
				}
				break;
		}
	}

	/**
	 * Take action when a user has deleted a backup.
	 *
	 * @since 1.5.3
	 *
	 * @param string $filepath
	 * @param bool   $deleted
	 */
	public function user_deleted_backup( $filepath, $deleted ) {
		if( empty( $filepath ) || ! $deleted ) {
			return;
		}

		$this->add( sprintf(
			__( 'The following backup file was deleted: %1$s', 'boldgrid-backup' ),
			$filepath
		));
	}
}
